#!/bin/bash

# Define colors using ANSI escape codes
RED='\033[0;31m'
GREEN='\033[0;32m'
NC='\033[0m' # No Color

# we don't include data-pipeline/setup.py because that causes the pytype cache
# to get confused with ./setup.py (IDK why)
SOURCES=(src/ setup.py manage.py locustfile.py
         deploy/replace_cloud_run_service.py
         data-pipeline/src)
declare -A tool_status

set -uo pipefail

# Define a function for printing a border
print_border() {
    echo -e "\n====================================\n"
}

print_border
echo -e "${GREEN}Type checking with pytype...${NC}"
pytype -j auto -k "${SOURCES[@]}"
pytype_status=$?
tool_status["pytype"]=$pytype_status
print_border

echo -e "${GREEN}Checking import order with isort...${NC}"
isort --check-only "${SOURCES[@]}"
isort_status=$?
tool_status["isort"]=$isort_status
print_border

echo -e "${GREEN}Checking code formatting with black...${NC}"
black --check "${SOURCES[@]}"
black_status=$?
tool_status["black"]=$black_status
print_border

echo -e "${GREEN}Checking if Jupyter notebook outputs are cleared...${NC}"
jupyter-nbconvert --clear-output --inplace notebooks/*.ipynb
git diff --exit-code notebooks/*.ipynb
notebook_status=$?
tool_status["notebooks"]=$notebook_status
if [[ $notebook_status -ne 0 ]]; then
    echo -e "${RED}Notebooks have outputs that were not cleared.${NC}"
    echo -e "${RED}Run ./clear_notebook.sh to clear. First 50 lines of diff.${NC}"
    git diff notebooks/*.ipynb | head -n 50
fi
print_border

echo -e "${GREEN}Status report:${NC}"
all_success=true
for tool in "${!tool_status[@]}"; do
  if [[ ${tool_status[$tool]} -ne 0 ]]; then
    echo -e "${RED}$tool failed${NC}"
    all_success=false
  else
    echo -e "${GREEN}$tool succeeded${NC}"
  fi
done
print_border

if $all_success; then
  echo -e "${GREEN}All checks succeeded, you're good to go!${NC}"
  exit 0
else
  echo -e "${RED}There were some tool failures. Please check the errors above${NC}"
  echo -e "${RED}and re-commit once the ./check.sh script is happy.${NC}"
  echo
  echo -e "${RED}You can usually fix formatting errors with ./format.sh, but${NC}"
  echo -e "${RED}type errors might require some manual effort.${NC}"
  exit 1
fi
