import _ from 'lodash'
import { nanoid } from 'nanoid'
import { UserState } from 'twitch-js'

import LogType from 'constants/logType'
import Theme from 'constants/theme'
import ChatterColor from 'libs/ChatterColor'
import base, { TwitchUserColorMap } from 'styled/base'

/**
 * Chatter class.
 */
export default class Chatter implements Serializable<SerializedChatter> {
  /**
   * Creates a potential chatter which represents a user with only its username known.
   * @param username - The chatter username.
   * @return The potential chatter.
   */
  public static createPotentialChatter(username: string) {
    return new Chatter(
      {
        badges: null,
        'badges-raw': '',
        color: null,
        'display-name': username,
        emotes: null,
        id: nanoid(),
        'message-type': LogType.Chat,
        mod: false,
        subscriber: false,
        'tmi-sent-ts': Date.now().toString(),
        'user-id': nanoid(),
        'user-type': null,
        username: username.toLowerCase(),
      },
      Theme.Dark
    )
  }

  public id: string
  public color: string | null
  public userName: string
  public isMod: boolean
  public banned: boolean = false
  private displayName: string
  private showUserName: boolean
  private blocked: boolean = false
  private isSelf: boolean
  private isBroadcaster: boolean

  /**
   * Creates a new chatter instance.
   * @class
   * @param userstate - The associated user state.
   * @param theme - The current theme.
   */
  constructor(userstate: UserState, theme: Theme) {
    this.displayName = userstate['display-name'].replace(/\\s/g, ' ').trim()
    this.id = userstate['user-id']
    this.userName = userstate.username
    this.color = this.sanitizeColor(userstate.color, theme)
    this.isBroadcaster = _.has(userstate.badges, 'broadcaster')
    this.isMod = userstate.mod || this.isBroadcaster
    this.showUserName = this.displayName.toLocaleLowerCase() !== this.userName.toLocaleLowerCase()
    this.isSelf = userstate['user-id'] === 'self'
  }

  /**
   * Generates a color for the chatter.
   * @return The new generated color.
   */
  public generateColor() {
    const firstLetterCharCode = this.userName.charCodeAt(0)
    const lastLetterCharCode = this.userName.charCodeAt(this.userName.length - 1)

    const colorIndex = (firstLetterCharCode + lastLetterCharCode) % base.chatters.length

    const color = _.get(base.chatters, colorIndex, _.sample(base.chatters)) || null

    this.color = color

    return color
  }

  /**
   * Serializes a chatter.
   * @return The serialized chatter.
   */
  public serialize() {
    return {
      banned: this.banned,
      blocked: this.blocked,
      color: this.color,
      displayName: this.displayName,
      id: this.id,
      isBroadcaster: this.isBroadcaster,
      isMod: this.isMod,
      isSelf: this.isSelf,
      showUserName: this.showUserName,
      userName: this.userName,
    }
  }

  /**
   * Sanitizes colors if necessary.
   * @param  color - The color to sanitize.
   * @param theme - The current theme.
   * @return The sanitized color.
   */
  private sanitizeColor(color: string | null, theme: Theme) {
    if (_.isNil(color)) {
      return color
    }

    let sanitizedColor = _.get(TwitchUserColorMap, color, color)

    if (sanitizedColor === color) {
      const chatterColor = new ChatterColor(color, base.background[theme])

      if (!chatterColor.isReadable()) {
        sanitizedColor = chatterColor.getReadableColor()
      }
    }

    return sanitizedColor
  }
}

/**
 * Serialized chat message.
 */
export type SerializedChatter = {
  banned: boolean
  blocked: boolean
  color: string | null
  displayName: string
  id: string
  isBroadcaster: boolean
  isMod: boolean
  isSelf: boolean
  showUserName: boolean
  userName: string
}

/**
 * Name color interface.
 */
export interface WithNameColorProps {
  color: string
}
