﻿unit TGC.Entity.Files;

interface

type
  /// <summary>
  /// Represents a remote file.
  /// </summary>
  TTgRemoteFile = class
  private
    FId: string;
    FIs_uploading_active: Boolean;
    FIs_uploading_completed: Boolean;
    FUnique_id: string;
    FUploaded_size: Int64;
  public
    /// <summary>
    /// Remote file identifier; may be empty. Can be used by the current user across application restarts or
    /// even from other devices. Uniquely identifies a file, but a file can have a lot of different valid identifiers.
    /// If the ID starts with "http://" or "https://", it represents the HTTP URL of the file.
    /// TDLib is currently unable to download files if only their URL is known. If downloadFile is called on such a
    /// file or if it is sent to a secret chat, TDLib starts a file generation process by sending updateFileGenerationStart
    /// to the application with the HTTP URL in the original_path and "#url#" as the conversion string. Application must
    /// generate the file by downloading it to the specified location.
    /// </summary>
    property Id: string read FId write FId;
    /// <summary>
    /// True, if the file is currently being uploaded (or a remote copy is being generated by some other means).
    /// </summary>
    property IsUploadingActive: Boolean read FIs_uploading_active write FIs_uploading_active;
    /// <summary>
    /// True, if a remote copy is fully available.
    /// </summary>
    property IsUploadingCompleted: Boolean read FIs_uploading_completed write FIs_uploading_completed;
    /// <summary>
    /// Unique file identifier; may be empty if unknown. The unique file identifier which is the same for the same file even for different users and is persistent over time.
    /// </summary>
    property UniqueId: string read FUnique_id write FUnique_id;
    /// <summary>
    /// Size of the remote available part of the file, in bytes; 0 if unknown.
    /// </summary>
    property UploadedSize: Int64 read FUploaded_size write FUploaded_size;
  end;

  /// <summary>
  /// Represents a local file.
  /// </summary>
  TTgLocalFile = class
  private
    FCan_be_deleted: Boolean;
    FCan_be_downloaded: Boolean;
    FDownload_offset: Int64;
    FDownloaded_prefix_size: Int64;
    FDownloaded_size: Int64;
    FIs_downloading_active: Boolean;
    FIs_downloading_completed: Boolean;
    FPath: string;
  public
    /// <summary>
    /// True, if the file can be deleted.
    /// </summary>
    property CanBeDeleted: Boolean read FCan_be_deleted write FCan_be_deleted;
    /// <summary>
    /// True, if it is possible to download or generate the file.
    /// </summary>
    property CanBeDownloaded: Boolean read FCan_be_downloaded write FCan_be_downloaded;
    /// <summary>
    /// Download will be started from this offset. downloaded_prefix_size is calculated from this offset.
    /// </summary>
    property DownloadOffset: Int64 read FDownload_offset write FDownload_offset;
    /// <summary>
    /// If is_downloading_completed is false, then only some prefix of the file starting from download_offset is ready to be read. downloaded_prefix_size is the size of that prefix in bytes.
    /// </summary>
    property DownloadedPrefixSize: Int64 read FDownloaded_prefix_size write FDownloaded_prefix_size;
    /// <summary>
    /// Total downloaded file size, in bytes. Can be used only for calculating download progress. The actual file size may be bigger, and some parts of it may contain garbage.
    /// </summary>
    property DownloadedSize: Int64 read FDownloaded_size write FDownloaded_size;
    /// <summary>
    /// True, if the file is currently being downloaded (or a local copy is being generated by some other means).
    /// </summary>
    property IsDownloadingActive: Boolean read FIs_downloading_active write FIs_downloading_active;
    /// <summary>
    /// True, if the local copy is fully available.
    /// </summary>
    property IsDownloadingCompleted: Boolean read FIs_downloading_completed write FIs_downloading_completed;
    /// <summary>
    /// Local path to the locally available file part; may be empty.
    /// </summary>
    property Path: string read FPath write FPath;
  end;

  /// <summary>
  /// Represents a file.
  /// </summary>
  TtgFile = class
  private
    FExpected_size: Int64;
    FId: Int64;
    FLocal: TTgLocalFile;
    FRemote: TTgRemoteFile;
    FSize: Int64;
  public
    /// <summary>
    /// Approximate file size in bytes in case the exact file size is unknown. Can be used to show download/upload progress.
    /// </summary>
    property ExpectedSize: Int64 read FExpected_size write FExpected_size;
    /// <summary>
    /// Unique file identifier.
    /// </summary>
    property Id: Int64 read FId write FId;
    /// <summary>
    /// Information about the local copy of the file.
    /// </summary>
    property Local: TTgLocalFile read FLocal write FLocal;
    /// <summary>
    /// Information about the remote copy of the file.
    /// </summary>
    property Remote: TTgRemoteFile read FRemote write FRemote;
    /// <summary>
    /// File size, in bytes; 0 if unknown.
    /// </summary>
    property Size: Int64 read FSize write FSize;
    destructor Destroy; override;
  end;

implementation

{ TtgFile }

destructor TtgFile.Destroy;
begin
  if Assigned(FRemote) then
    FRemote.Free;
  if Assigned(FLocal) then
    FLocal.Free;
  inherited;
end;

end.

