<?php
/**
 * Handles Comment Post to GeChiUI and prevents duplicate comment posting.
 *
 * @package GeChiUI
 */

if ( 'POST' !== $_SERVER['REQUEST_METHOD'] ) {
	$protocol = $_SERVER['SERVER_PROTOCOL'];
	if ( ! in_array( $protocol, array( 'HTTP/1.1', 'HTTP/2', 'HTTP/2.0', 'HTTP/3' ), true ) ) {
		$protocol = 'HTTP/1.0';
	}

	header( 'Allow: POST' );
	header( "$protocol 405 Method Not Allowed" );
	header( 'Content-Type: text/plain' );
	exit;
}

/** Sets up the GeChiUI Environment. */
require __DIR__ . '/gc-load.php';

nocache_headers();

$comment = gc_handle_comment_submission( gc_unslash( $_POST ) );
if ( is_gc_error( $comment ) ) {
	$data = (int) $comment->get_error_data();
	if ( ! empty( $data ) ) {
		gc_die(
			'<p>' . $comment->get_error_message() . '</p>',
			__( '评论提交失败' ),
			array(
				'response'  => $data,
				'back_link' => true,
			)
		);
	} else {
		exit;
	}
}

$user            = gc_get_current_user();
$cookies_consent = ( isset( $_POST['gc-comment-cookies-consent'] ) );

/**
 * Fires after comment cookies are set.
 * The `$cookies_consent` parameter was added.
 *
 * @param GC_Comment $comment         Comment object.
 * @param GC_User    $user            Comment author's user object. The user may not exist.
 * @param bool       $cookies_consent Comment author's consent to store cookies.
 */
do_action( 'set_comment_cookies', $comment, $user, $cookies_consent );

$location = empty( $_POST['redirect_to'] ) ? get_comment_link( $comment ) : $_POST['redirect_to'] . '#comment-' . $comment->comment_ID;

// If user didn't consent to cookies, add specific query arguments to display the awaiting moderation message.
if ( ! $cookies_consent && 'unapproved' === gc_get_comment_status( $comment ) && ! empty( $comment->comment_author_email ) ) {
	$location = add_query_arg(
		array(
			'unapproved'      => $comment->comment_ID,
			'moderation-hash' => gc_hash( $comment->comment_date_gmt ),
		),
		$location
	);
}

/**
 * Filters the location URI to send the commenter after posting.
 *
 * @since 2.0.5
 *
 * @param string     $location The 'redirect_to' URI sent via $_POST.
 * @param GC_Comment $comment  Comment object.
 */
$location = apply_filters( 'comment_post_redirect', $location, $comment );

gc_safe_redirect( $location );
exit;
