package migrations

import (
	"bytes"
	"fmt"
	"go/format"
	"html/template"
	"os"
	"sort"

	"github.com/rotisserie/eris"
	"github.com/spf13/cobra"
)

var migrationsTemplate = `// Code generated by 'make migrations-rebuild'; DO NOT EDIT.
package database

import (
	"fmt"

	log "github.com/sirupsen/logrus"
	"gorm.io/gorm"

	{{ range packages }}"github.com/G-Research/fasttrackml/pkg/database/migrations/{{ . }}"
        {{ end }}
)

func currentVersion() string {
	return {{ maxPackage }}.Version
}

func generatedMigrations(db *gorm.DB, schemaVersion string) error {
	switch schemaVersion {
	{{- range $i, $package := packages }}
        {{- if eq $i 0 }}
        case "":
	{{- else if le $i maxIndex }}
	case {{ package (sub $i 1)}}.Version:
	{{- end }}{{ template "migration" $i }}{{- end }}
	default:
		return fmt.Errorf("unsupported database FastTrackML schema version %s", schemaVersion)
        }
        return nil
}
{{ define "migration" }}
log.Infof("Migrating database to FastTrackML schema %s", {{ package . }}.Version)
if err := {{ package . }}.Migrate(db); err != nil {
	return fmt.Errorf("error migrating database to FastTrackML schema %s: %w", {{ package . }}.Version, err)
}
{{ if ne (package .) (maxPackage) }}fallthrough{{ end }}
{{ end }}
`

var RebuildCmd = &cobra.Command{
	Use:   "rebuild",
	Short: "Rebuilds the migration_generated.go file with the current state of pkg/database/migrations",
	Long: `The rebuild command generates the pkg/database/migration_generated.go file, using the
               current contents of the pkg/database/migrations folder.`,
	RunE: rebuildMigrationsCmd,
}

func rebuildMigrationsCmd(cmd *cobra.Command, args []string) error {
	if err := rebuildMigrations(MigrationsSources, DatabaseSources); err != nil {
		return eris.Wrap(err, "error creating migrate_generated.go file")
	}

	return nil
}

func rebuildMigrations(migrationsSources, databaseSources string) error {
	// find next migration number
	files, err := os.ReadDir(migrationsSources)
	if err != nil {
		return eris.Wrap(err, "error reading migrations source folder")
	}

	packages := []string{}
	for _, file := range files {
		if file.IsDir() {
			packages = append(packages, file.Name())
		}
	}

	sort.Slice(packages, func(i, j int) bool {
		return packages[i] < packages[j]
	})

	funcs := template.FuncMap{
		"sub": func(n1, n2 int) int {
			return n1 - n2
		},
		"package": func(index int) string {
			return packages[index]
		},
		"packages": func() []string {
			return packages
		},
		"maxPackage": func() string {
			return packages[len(packages)-1]
		},
		"maxIndex": func() int {
			return len(packages) - 1
		},
	}

	tmpl, err := template.New("migrations").Funcs(funcs).Parse(migrationsTemplate)
	if err != nil {
		return eris.Wrap(err, "error parsing template")
	}

	var buf bytes.Buffer
	if err := tmpl.Execute(&buf, nil); err != nil {
		return eris.Wrap(err, "error executing template")
	}

	src, err := format.Source(buf.Bytes())
	if err != nil {
		return eris.Wrap(err, "error formatting generated source file")
	}
	//nolint:gosec
	if err := os.WriteFile(fmt.Sprintf("%s/migrate_generated.go", databaseSources), src, 0o644); err != nil {
		return eris.Wrap(err, "error writing generated source file")
	}
	return nil
}
