<?php declare(strict_types = 1);
/**
 * This file is part of the SqlFtw library (https://github.com/sqlftw)
 *
 * Copyright (c) 2017 Vlasta Neubauer (@paranoiq)
 *
 * For the full copyright and license information read the file 'license.md', distributed with this source code
 */

namespace SqlFtw\Sql\Dml\Error;

use SqlFtw\Formatter\Formatter;
use SqlFtw\Sql\InvalidDefinitionException;
use SqlFtw\Sql\SqlSerializable;
use function preg_match;
use function substr;

class SqlState implements SqlSerializable
{

    // as per SQL:2011 - https://en.wikipedia.org/wiki/SQLSTATE
    public const S_00000_SUCCESS = '00000';
    public const S_01000_WARNING = '01000';
    public const S_01001_CURSOR_OPERATION_CONFLICT = '01001';
    public const S_01002_DISCONNECT_ERROR = '01002';
    public const S_01003_NULL_VALUE_ELIMINATED_IN_SET_FUNCTION = '01003';
    public const S_01004_STRING_DATA_RIGHT_TRUNCATION = '01004';
    public const S_01005_INSUFFICIENT_ITEM_DESCRIPTOR_AREAS = '01005';
    public const S_01006_PRIVILEGE_NOT_REVOKED = '01006';
    public const S_01007_PRIVILEGE_NOT_GRANTED = '01007';
    public const S_01009_SEARCH_CONDITION_TOO_LONG_FOR_INFORMATION_SCHEMA = '01009';
    public const S_0100A_QUERY_EXPRESSION_TOO_LONG_FOR_INFORMATION_SCHEMA = '0100A';
    public const S_0100B_DEFAULT_VALUE_TOO_LONG_FOR_INFORMATION_SCHEMA = '0100B';
    public const S_0100C_RESULT_SETS_RETURNED = '0100C';
    public const S_0100D_ADDITIONAL_RESULT_SETS_RETURNED = '0100D';
    public const S_0100E_ATTEMPT_TO_RETURN_TOO_MANY_RESULT_SETS = '0100E';
    public const S_0100F_STATEMENT_TOO_LONG_FOR_INFORMATION_SCHEMA = '0100F';
    public const S_01010_COLUMN_CANNOT_BE_MAPPED = '01010';
    public const S_01011_SQL_JAVA_PATH_TOO_LONG_FOR_INFORMATION_SCHEMA = '01011';
    public const S_01012_INVALID_NUMBER_OF_CONDITIONS = '01012';
    public const S_0102F_ARRAY_DATA_RIGHT_TRUNCATION = '0102F';
    public const S_02000_NO_DATA = '02000';
    public const S_02001_NO_ADDITIONAL_RESULT_SETS_RETURNED = '02001';
    public const S_07000_SQL_ERROR = '07000';
    public const S_07001_USING_CLAUSE_DOES_NOT_MATCH_DYNAMIC_PARAMETER_SPECIFICATIONS = '07001';
    public const S_07002_USING_CLAUSE_DOES_NOT_MATCH_TARGET_SPECIFICATIONS = '07002';
    public const S_07003_CURSOR_SPECIFICATION_CANNOT_BE_EXECUTED = '07003';
    public const S_07004_USING_CLAUSE_REQUIRED_FOR_DYNAMIC_PARAMETERS = '07004';
    public const S_07005_PREPARED_STATEMENT_NOT_A_CURSOR_SPECIFICATION = '07005';
    public const S_07006_RESTRICTED_DATA_TYPE_ATTRIBUTE_VIOLATION = '07006';
    public const S_07007_USING_CLAUSE_REQUIRED_FOR_RESULT_FIELDS = '07007';
    public const S_07008_INVALID_DESCRIPTOR_COUNT = '07008';
    public const S_07009_INVALID_DESCRIPTOR_INDEX = '07009';
    public const S_0700B_DATA_TYPE_TRANSFORM_FUNCTION_VIOLATION = '0700B';
    public const S_0700C_UNDEFINED_DATA_VALUE = '0700C';
    public const S_0700D_INVALID_DATA_TARGET = '0700D';
    public const S_0700E_INVALID_LEVEL_VALUE = '0700E';
    public const S_0700F_INVALID_DATETIME_INTERVAL_CODE = '0700F';
    public const S_08000_CONNECTION_EXCEPTION = '08000';
    public const S_08001_SQL_CLIENT_UNABLE_TO_ESTABLISH_SQL_CONNECTION = '08001';
    public const S_08002_CONNECTION_NAME_IN_USE = '08002';
    public const S_08003_CONNECTION_DOES_NOT_EXIST = '08003';
    public const S_08004_SQL_SERVER_REJECTED_ESTABLISHMENT_OF_SQL_CONNECTION = '08004';
    public const S_08006_CONNECTION_FAILURE = '08006';
    public const S_08007_TRANSACTION_RESOLUTION_UNKNOWN = '08007';
    public const S_09000_TRIGGERED_ACTION_EXCEPTION = '09000';
    public const S_0A000_FEATURE_NOT_SUPPORTED = '0A000';
    public const S_0A001_MULTIPLE_SERVER_TRANSACTIONS = '0A001';
    public const S_0D000_INVALID_TARGET_TYPE_SPECIFICATION = '0D000';
    public const S_0E000_INVALID_SCHEMA_NAME_LIST_SPECIFICATION = '0E000';
    public const S_0F000_LOCATOR_EXCEPTION = '0F000';
    public const S_0F001_INVALID_SPECIFICATION = '0F001';
    public const S_0K000_RESIGNAL_WHEN_HANDLER_NOT_ACTIVE = '0K000';
    public const S_0L000_INVALID_GRANTOR = '0L000';
    public const S_0M000_INVALID_SQL_INVOKED_PROCEDURE_REFERENCE = '0M000';
    public const S_0N000_SQL_XML_MAPPING_ERROR = '0N000';
    public const S_0N001_UNMAPPABLE_XML_NAME = '0N001';
    public const S_0N002_INVALID_XML_CHARACTER = '0N002';
    public const S_0P000_INVALID_ROLE_SPECIFICATION = '0P000';
    public const S_0S000_INVALID_TRANSFORM_GROUP_NAME_SPECIFICATION = '0S000';
    public const S_0T000_TARGET_TABLE_DISAGREES_WITH_CURSOR_SPECIFICATION = '0T000';
    public const S_0U000_ATTEMPT_TO_ASSIGN_TO_NON_UPDATABLE_COLUMN = '0U000';
    public const S_0V000_ATTEMPT_TO_ASSIGN_TO_ORDERING_COLUMN = '0V000';
    public const S_0W000_PROHIBITED_STATEMENT_ENCOUNTERED_DURING_TRIGGER_EXECUTION = '0W000';
    public const S_0W001_MODIFY_TABLE_MODIFIED_BY_DATA_CHANGE_DELTA_TABLE = '0W001';
    public const S_0X000_INVALID_FOREIGN_SERVER_SPECIFICATION = '0X000';
    public const S_0Y000_PASS_THROUGH_SPECIFIC_CONDITION = '0Y000';
    public const S_0Y001_INVALID_CURSOR_OPTION = '0Y001';
    public const S_0Y002_INVALID_CURSOR_ALLOCATION = '0Y002';
    public const S_0Z000_DIAGNOSTICS_EXCEPTION = '0Z000';
    public const S_0Z001_MAXIMUM_NUMBER_OF_STACKED_DIAGNOSTICS_AREAS_EXCEEDED = '0Z001';
    public const S_0Z002_STACKED_DIAGNOSTICS_ACCESSED_WITHOUT_ACTIVE_HANDLER = '0Z002';
    public const S_10000_XQUERY_ERROR = '10000';
    public const S_20000_CASE_NOT_FOUND_FOR_CASE_STATEMENT = '20000';
    public const S_21000_CARDINALITY_VIOLATION = '21000';
    public const S_22000_DATA_EXCEPTION = '22000';
    public const S_22001_STRING_DATA_RIGHT_TRUNCATION = '22001';
    public const S_22002_NULL_VALUE_NO_INDICATOR_PARAMETER = '22002';
    public const S_22003_NUMERIC_VALUE_OUT_OF_RANGE = '22003';
    public const S_22004_NULL_VALUE_NOT_ALLOWED = '22004';
    public const S_22005_ERROR_IN_ASSIGNMENT = '22005';
    public const S_22006_INVALID_INTERVAL_FORMAT = '22006';
    public const S_22007_INVALID_DATETIME_FORMAT = '22007';
    public const S_22008_DATETIME_FIELD_OVERFLOW = '22008';
    public const S_22009_INVALID_TIME_ZONE_DISPLACEMENT_VALUE = '22009';
    public const S_2200B_ESCAPE_CHARACTER_CONFLICT = '2200B';
    public const S_2200C_INVALID_USE_OF_ESCAPE_CHARACTER = '2200C';
    public const S_2200D_INVALID_ESCAPE_OCTET = '2200D';
    public const S_2200E_NULL_VALUE_IN_ARRAY_TARGET = '2200E';
    public const S_2200F_ZERO_LENGTH_CHARACTER_STRING = '2200F';
    public const S_2200G_MOST_SPECIFIC_TYPE_MISMATCH = '2200G';
    public const S_2200H_SEQUENCE_GENERATOR_LIMIT_EXCEEDED = '2200H';
    public const S_2200J_NONIDENTICAL_NOTATIONS_WITH_THE_SAME_NAME = '2200J';
    public const S_2200K_NONIDENTICAL_UNPARSED_ENTITIES_WITH_THE_SAME_NAME = '2200K';
    public const S_2200L_NOT_AN_XML_DOCUMENT = '2200L';
    public const S_2200M_INVALID_XML_DOCUMENT = '2200M';
    public const S_2200N_INVALID_XML_CONTENT = '2200N';
    public const S_2200P_INTERVAL_VALUE_OUT_OF_RANGE = '2200P';
    public const S_2200Q_MULTISET_VALUE_OVERFLOW = '2200Q';
    public const S_2200R_XML_VALUE_OVERFLOW = '2200R';
    public const S_2200S_INVALID_COMMENT = '2200S';
    public const S_2200T_INVALID_PROCESSING_INSTRUCTION = '2200T';
    public const S_2200U_NOT_AN_XQUERY_DOCUMENT_NODE = '2200U';
    public const S_2200V_INVALID_XQUERY_CONTEXT_ITEM = '2200V';
    public const S_2200W_XQUERY_SERIALIZATION_ERROR = '2200W';
    public const S_22010_INVALID_INDICATOR_PARAMETER_VALUE = '22010';
    public const S_22011_SUBSTRING_ERROR = '22011';
    public const S_22012_DIVISION_BY_ZERO = '22012';
    public const S_22013_INVALID_PRECEDING_OR_FOLLOWING_SIZE_IN_WINDOW_FUNCTION = '22013';
    public const S_22014_INVALID_ARGUMENT_FOR_NTILE_FUNCTION = '22014';
    public const S_22015_INTERVAL_FIELD_OVERFLOW = '22015';
    public const S_22016_INVALID_ARGUMENT_FOR_NTH_VALUE_FUNCTION = '22016';
    public const S_22017_INVALID_DATA_SPECIFIED_FOR_DATALINK = '22017';
    public const S_22018_INVALID_CHARACTER_VALUE_FOR_CAST = '22018';
    public const S_22019_INVALID_ESCAPE_CHARACTER = '22019';
    public const S_2201A_NULL_ARGUMENT_PASSED_TO_DATALINK_CONSTRUCTOR = '2201A';
    public const S_2201B_INVALID_REGULAR_EXPRESSION = '2201B';
    public const S_2201C_NULL_ROW_NOT_PERMITTED_IN_TABLE = '2201C';
    public const S_2201D_DATALINK_VALUE_EXCEEDS_MAXIMUM_LENGTH = '2201D';
    public const S_2201E_INVALID_ARGUMENT_FOR_NATURAL_LOGARITHM = '2201E';
    public const S_2201F_INVALID_ARGUMENT_FOR_POWER_FUNCTION = '2201F';
    public const S_2201G_INVALID_ARGUMENT_FOR_WIDTH_BUCKET_FUNCTION = '2201G';
    public const S_2201H_INVALID_ROW_VERSION = '2201H';
    public const S_2201J_XQUERY_SEQUENCE_CANNOT_BE_VALIDATED = '2201J';
    public const S_2201K_XQUERY_DOCUMENT_NODE_CANNOT_BE_VALIDATED = '2201K';
    public const S_2201L_NO_XML_SCHEMA_FOUND = '2201L';
    public const S_2201M_ELEMENT_NAMESPACE_NOT_DECLARED = '2201M';
    public const S_2201N_GLOBAL_ELEMENT_NOT_DECLARED = '2201N';
    public const S_2201P_NO_XML_ELEMENT_WITH_THE_SPECIFIED_QNAME = '2201P';
    public const S_2201Q_NO_XML_ELEMENT_WITH_THE_SPECIFIED_NAMESPACE = '2201Q';
    public const S_2201R_VALIDATION_FAILURE = '2201R';
    public const S_2201S_INVALID_QUERY_REGULAR_EXPRESSION = '2201S';
    public const S_2201T_INVALID_QUERY_OPTION_FLAG = '2201T';
    public const S_2201U_ATTEMPT_TO_REPLACE_A_ZERO_LENGTH_STRING = '2201U';
    public const S_2201V_INVALID_QUERY_REPLACEMENT_STRING = '2201V';
    public const S_2201W_INVALID_ROW_COUNT_IN_FETCH_FIRST_CLAUSE = '2201W';
    public const S_2201X_INVALID_ROW_COUNT_IN_RESULT_OFFSET_CLAUSE = '2201X';
    public const S_22021_CHARACTER_NOT_IN_REPERTOIRE = '22021';
    public const S_22022_INDICATOR_OVERFLOW = '22022';
    public const S_22023_INVALID_PARAMETER_VALUE = '22023';
    public const S_22024_UNTERMINATED_C_STRING = '22024';
    public const S_22025_INVALID_ESCAPE_SEQUENCE = '22025';
    public const S_22026_STRING_DATA_LENGTH_MISMATCH = '22026';
    public const S_22027_TRIM_ERROR = '22027';
    public const S_22029_NONCHARACTER_IN_UCS_STRING = '22029';
    public const S_2202A_NULL_VALUE_IN_FIELD_REFERENCE = '2202A';
    public const S_2202D_NULL_VALUE_SUBSTITUTED_FOR_MUTATOR_SUBJECT_PARAMETER = '2202D';
    public const S_2202E_ARRAY_ELEMENT_ERROR = '2202E';
    public const S_2202F_ARRAY_DATA_RIGHT_TRUNCATION = '2202F';
    public const S_2202G_INVALID_REPEAT_ARGUMENT_IN_A_SAMPLE_CLAUSE = '2202G';
    public const S_2202H_INVALID_SAMPLE_SIZE = '2202H';
    public const S_23000_INTEGRITY_CONSTRAINT_VIOLATION = '23000';
    public const S_23001_RESTRICT_VIOLATION = '23001';
    public const S_24000_INVALID_CURSOR_STATE = '24000';
    public const S_25000_INVALID_TRANSACTION_STATE = '25000';
    public const S_25001_ACTIVE_SQL_TRANSACTION = '25001';
    public const S_25002_BRANCH_TRANSACTION_ALREADY_ACTIVE = '25002';
    public const S_25003_INAPPROPRIATE_ACCESS_MODE_FOR_BRANCH_TRANSACTION = '25003';
    public const S_25004_INAPPROPRIATE_ISOLATION_LEVEL_FOR_BRANCH_TRANSACTION = '25004';
    public const S_25005_NO_ACTIVE_SQL_TRANSACTION_FOR_BRANCH_TRANSACTION = '25005';
    public const S_25006_READ_ONLY_SQL_TRANSACTION = '25006';
    public const S_25007_SCHEMA_AND_DATA_STATEMENT_MIXING_NOT_SUPPORTED = '25007';
    public const S_25008_HELD_CURSOR_REQUIRES_SAME_ISOLATION_LEVEL = '25008';
    public const S_26000_INVALID_SQL_STATEMENT_NAME = '26000';
    public const S_27000_TRIGGERED_DATA_CHANGE_VIOLATION = '27000';
    public const S_27001_MODIFY_TABLE_MODIFIED_BY_DATA_CHANGE_DELTA_TABLE = '27001';
    public const S_28000_INVALID_AUTHORIZATION_SPECIFICATION = '28000';
    public const S_2B000_DEPENDENT_PRIVILEGE_DESCRIPTORS_STILL_EXIST = '2B000';
    public const S_2C000_INVALID_CHARACTER_SET_NAME = '2C000';
    public const S_2D000_INVALID_TRANSACTION_TERMINATION = '2D000';
    public const S_2E000_INVALID_CONNECTION_NAME = '2E000';
    public const S_2F000_SQL_ROUTINE_EXCEPTION = '2F000';
    public const S_2F002_MODIFYING_SQL_DATA_NOT_PERMITTED = '2F002';
    public const S_2F003_PROHIBITED_SQL_STATEMENT_ATTEMPTED = '2F003';
    public const S_2F004_READING_SQL_DATA_NOT_PERMITTED = '2F004';
    public const S_2F005_FUNCTION_EXECUTED_NO_RETURN_STATEMENT = '2F005';
    public const S_2H000_INVALID_COLLATION_NAME = '2H000';
    public const S_30000_INVALID_SQL_STATEMENT_IDENTIFIER = '30000';
    public const S_33000_INVALID_SQL_DESCRIPTOR_NAME = '33000';
    public const S_34000_INVALID_CURSOR_NAME = '34000';
    public const S_35000_INVALID_CONDITION_NUMBER = '35000';
    public const S_36000_CURSOR_SENSITIVITY_EXCEPTION = '36000';
    public const S_36001_REQUEST_REJECTED = '36001';
    public const S_36002_REQUEST_FAILED = '36002';
    public const S_38000_EXTERNAL_ROUTINE_EXCEPTION = '38000';
    public const S_38001_CONTAINING_SQL_NOT_PERMITTED = '38001';
    public const S_38002_MODIFYING_SQL_DATA_NOT_PERMITTED = '38002';
    public const S_38003_PROHIBITED_SQL_STATEMENT_ATTEMPTED = '38003';
    public const S_38004_READING_SQL_DATA_NOT_PERMITTED = '38004';
    public const S_39000_EXTERNAL_ROUTINE_INVOCATION_EXCEPTION = '39000';
    public const S_39004_NULL_VALUE_NOT_ALLOWED = '39004';
    public const S_3B000_SAVEPOINT_EXCEPTION = '3B000';
    public const S_3B001_INVALID_SPECIFICATION = '3B001';
    public const S_3B002_TOO_MANY = '3B002';
    public const S_3C000_AMBIGUOUS_CURSOR_NAME = '3C000';
    public const S_3D000_INVALID_CATALOG_NAME = '3D000';
    public const S_3F000_INVALID_SCHEMA_NAME = '3F000';
    public const S_40000_TRANSACTION_ROLLBACK = '40000';
    public const S_40001_SERIALIZATION_FAILURE = '40001';
    public const S_40002_INTEGRITY_CONSTRAINT_VIOLATION = '40002';
    public const S_40003_STATEMENT_COMPLETION_UNKNOWN = '40003';
    public const S_40004_TRIGGERED_ACTION_EXCEPTION = '40004';
    public const S_42000_SYNTAX_ERROR_OR_ACCESS_RULE_VIOLATION = '42000';
    public const S_44000_WITH_CHECK_OPTION_VIOLATION = '44000';
    public const S_45000_UNHANDLED_USER_DEFINED_EXCEPTION = '45000';
    public const S_46000_OLB_SPECIFIC_ERROR = '46000';
    public const S_46001_INVALID_URL = '46001';
    public const S_46002_INVALID_JAR_NAME = '46002';
    public const S_46003_INVALID_CLASS_DELETION = '46003';
    public const S_46005_INVALID_REPLACEMENT = '46005';
    public const S_4600A_ATTEMPT_TO_REPLACE_UNINSTALLED_JAR = '4600A';
    public const S_4600B_ATTEMPT_TO_REMOVE_UNINSTALLED_JAR = '4600B';
    public const S_4600C_INVALID_JAR_REMOVAL = '4600C';
    public const S_4600D_INVALID_PATH = '4600D';
    public const S_4600E_SELF_REFERENCING_PATH = '4600E';
    public const S_46102_INVALID_JAR_NAME_IN_PATH = '46102';
    public const S_46103_UNRESOLVED_CLASS_NAME = '46103';
    public const S_46110_UNSUPPORTED_FEATURE = '46110';
    public const S_46120_INVALID_CLASS_DECLARATION = '46120';
    public const S_46121_INVALID_COLUMN_NAME = '46121';
    public const S_46122_INVALID_NUMBER_OF_COLUMNS = '46122';
    public const S_46130_INVALID_PROFILE_STATE = '46130';
    public const S_HW000_DATALINK_EXCEPTION = 'HW000';
    public const S_HW001_EXTERNAL_FILE_NOT_LINKED = 'HW001';
    public const S_HW002_EXTERNAL_FILE_ALREADY_LINKED = 'HW002';
    public const S_HW003_REFERENCED_FILE_DOES_NOT_EXIST = 'HW003';
    public const S_HW004_INVALID_WRITE_TOKEN = 'HW004';
    public const S_HW005_INVALID_DATALINK_CONSTRUCTION = 'HW005';
    public const S_HW006_INVALID_WRITE_PERMISSION_FOR_UPDATE = 'HW006';
    public const S_HW007_REFERENCED_FILE_NOT_VALID = 'HW007';
    public const S_HV000_FDW_SPECIFIC_CONDITION = 'HV000';
    public const S_HV001_MEMORY_ALLOCATION_ERROR = 'HV001';
    public const S_HV002_DYNAMIC_PARAMETER_VALUE_NEEDED = 'HV002';
    public const S_HV004_INVALID_DATA_TYPE = 'HV004';
    public const S_HV005_COLUMN_NAME_NOT_FOUND = 'HV005';
    public const S_HV006_INVALID_DATA_TYPE_DESCRIPTORS = 'HV006';
    public const S_HV007_INVALID_COLUMN_NAME = 'HV007';
    public const S_HV008_INVALID_COLUMN_NUMBER = 'HV008';
    public const S_HV009_INVALID_USE_OF_NULL_POINTER = 'HV009';
    public const S_HV00A_INVALID_STRING_FORMAT = 'HV00A';
    public const S_HV00B_INVALID_HANDLE = 'HV00B';
    public const S_HV00C_INVALID_OPTION_INDEX = 'HV00C';
    public const S_HV00D_INVALID_OPTION_NAME = 'HV00D';
    public const S_HV00J_OPTION_NAME_NOT_FOUND = 'HV00J';
    public const S_HV00K_REPLY_HANDLE = 'HV00K';
    public const S_HV00L_UNABLE_TO_CREATE_EXECUTION = 'HV00L';
    public const S_HV00M_UNABLE_TO_CREATE_REPLY = 'HV00M';
    public const S_HV00N_UNABLE_TO_ESTABLISH_CONNECTION = 'HV00N';
    public const S_HV00P_NO_SCHEMAS = 'HV00P';
    public const S_HV00Q_SCHEMA_NOT_FOUND = 'HV00Q';
    public const S_HV00R_TABLE_NOT_FOUND = 'HV00R';
    public const S_HV010_FUNCTION_SEQUENCE_ERROR = 'HV010';
    public const S_HV014_LIMIT_ON_NUMBER_OF_HANDLES_EXCEEDED = 'HV014';
    public const S_HV021_INCONSISTENT_DESCRIPTOR_INFORMATION = 'HV021';
    public const S_HV024_INVALID_ATTRIBUTE_VALUE = 'HV024';
    public const S_HV090_INVALID_STRING_LENGTH_OR_BUFFER_LENGTH = 'HV090';
    public const S_HV091_INVALID_DESCRIPTOR_FIELD_IDENTIFIER = 'HV091';
    public const S_HY000_CLI_SPECIFIC_CONDITION = 'HY000';
    public const S_HY001_MEMORY_ALLOCATION_ERROR = 'HY001';
    public const S_HY003_INVALID_DATA_TYPE_IN_APPLICATION_DESCRIPTOR = 'HY003';
    public const S_HY004_INVALID_DATA_TYPE = 'HY004';
    public const S_HY007_ASSOCIATED_STATEMENT_IS_NOT_PREPARED = 'HY007';
    public const S_HY008_OPERATION_CANCELED = 'HY008';
    public const S_HY009_INVALID_USE_OF_NULL_POINTER = 'HY009';
    public const S_HY010_FUNCTION_SEQUENCE_ERROR = 'HY010';
    public const S_HY011_ATTRIBUTE_CANNOT_BE_SET_NOW = 'HY011';
    public const S_HY012_INVALID_TRANSACTION_OPERATION_CODE = 'HY012';
    public const S_HY013_MEMORY_MANAGEMENT_ERROR = 'HY013';
    public const S_HY014_LIMIT_ON_NUMBER_OF_HANDLES_EXCEEDED = 'HY014';
    public const S_HY017_INVALID_USE_OF_AUTOMATICALLY_ALLOCATED_DESCRIPTOR_HANDLE = 'HY017';
    public const S_HY018_SERVER_DECLINED_THE_CANCELLATION_REQUEST = 'HY018';
    public const S_HY019_NON_STRING_DATA_CANNOT_BE_SENT_IN_PIECES = 'HY019';
    public const S_HY020_ATTEMPT_TO_CONCATENATE_A_NULL_VALUE = 'HY020';
    public const S_HY021_INCONSISTENT_DESCRIPTOR_INFORMATION = 'HY021';
    public const S_HY024_INVALID_ATTRIBUTE_VALUE = 'HY024';
    public const S_HY055_NON_STRING_DATA_CANNOT_BE_USED_WITH_STRING_ROUTINE = 'HY055';
    public const S_HY090_INVALID_STRING_LENGTH_OR_BUFFER_LENGTH = 'HY090';
    public const S_HY091_INVALID_DESCRIPTOR_FIELD_IDENTIFIER = 'HY091';
    public const S_HY092_INVALID_ATTRIBUTE_IDENTIFIER = 'HY092';
    public const S_HY093_INVALID_DATALINK_VALUE = 'HY093';
    public const S_HY095_INVALID_FUNCTIONID_SPECIFIED = 'HY095';
    public const S_HY096_INVALID_INFORMATION_TYPE = 'HY096';
    public const S_HY097_COLUMN_TYPE_OUT_OF_RANGE = 'HY097';
    public const S_HY098_SCOPE_OUT_OF_RANGE = 'HY098';
    public const S_HY099_NULLABLE_TYPE_OUT_OF_RANGE = 'HY099';
    public const S_HY103_INVALID_RETRIEVAL_CODE = 'HY103';
    public const S_HY104_INVALID_LENGTHPRECISION_VALUE = 'HY104';
    public const S_HY105_INVALID_PARAMETER_MODE = 'HY105';
    public const S_HY106_INVALID_FETCH_ORIENTATION = 'HY106';
    public const S_HY107_ROW_VALUE_OUT_OF_RANGE = 'HY107';
    public const S_HY108_INVALID_CURSOR_POSITION = 'HY108';
    public const S_HYC00_OPTIONAL_FEATURE_NOT_IMPLEMENTED = 'HYC00';

    private string $value;

    public function __construct(string $value)
    {
        if (preg_match('~^[\dA-Z]{5}$~', $value) === 0) {
            throw new InvalidDefinitionException('Invalid SQLSTATE value.');
        }
        $this->value = $value;
    }

    public function getValue(): string
    {
        return $this->value;
    }

    public function getClass(): SqlStateClass
    {
        $class = substr($this->getValue(), 0, 2);
        if (!SqlStateClass::isValidValue($class)) {
            $class = SqlStateClass::CUSTOM;
        }

        return new SqlStateClass($class);
    }

    public function getCategory(): SqlStateCategory
    {
        return $this->getClass()->getCategory();
    }

    public function serialize(Formatter $formatter): string
    {
        return "'$this->value'";
    }

}
