<?php declare(strict_types = 1);
/**
 * This file is part of the Dogma library (https://github.com/paranoiq/dogma)
 *
 * Copyright (c) 2012 Vlasta Neubauer (@paranoiq)
 *
 * For the full copyright and license information read the file 'license.md', distributed with this source code
 */

namespace Dogma\Language;

use Dogma\Enum\StringEnum;
use function array_search;
use function strtolower;

/**
 * 2-letter language codes by ISO-639
 */
class Language extends StringEnum
{

    public const ABKHAZ = 'ab';
    public const AFAR = 'aa';
    public const AFRIKAANS = 'af';
    public const AKAN = 'ak';
    public const ALBANIAN = 'sq';
    public const AMHARIC = 'am';
    public const ARABIC = 'ar';
    public const ARAGONESE = 'an';
    public const ARMENIAN = 'hy';
    public const ASSAMESE = 'as';
    public const AVARIC = 'av';
    public const AVESTAN = 'ae';
    public const AYMARA = 'ay';
    public const AZERBAIJANI = 'az';
    public const BAMBARA = 'bm';
    public const BASHKIR = 'ba';
    public const BASQUE = 'eu';
    public const BELARUSIAN = 'be';
    public const BENGALI = 'bn';
    public const BIHARI = 'bh';
    public const BISLAMA = 'bi';
    public const BOSNIAN = 'bs';
    public const BRETON = 'br';
    public const BULGARIAN = 'bg';
    public const BURMESE = 'my';
    public const CATALAN = 'ca';
    public const CHAMORRO = 'ch';
    public const CHECHEN = 'ce';
    public const CHICHEWA = 'ny';
    public const CHINESE = 'zh';
    public const CHURCH_SLAVIC = 'cu';
    public const CHUVASH = 'cv';
    public const CORNISH = 'kw';
    public const CORSICAN = 'co';
    public const CREE = 'cr';
    public const CROATIAN = 'hr';
    public const CZECH = 'cs';
    public const DANISH = 'da';
    public const DIVEHI = 'dv';
    public const DUTCH = 'nl';
    public const DZONGKHA = 'dz';
    public const ENGLISH = 'en';
    public const ESPERANTO = 'eo';
    public const ESTONIAN = 'et';
    public const EWE = 'ee';
    public const FAROESE = 'fo';
    public const FIJIAN = 'fj';
    public const FINNISH = 'fi';
    public const FRENCH = 'fr';
    public const FULAH = 'ff';
    public const GAELIC = 'gd';
    public const GALICIAN = 'gl';
    public const GANDA = 'lg';
    public const GEORGIAN = 'ka';
    public const GERMAN = 'de';
    public const GREEK = 'el';
    public const GUARANI = 'gn';
    public const GUJARATI = 'gu';
    public const HAITIAN = 'ht';
    public const HAUSA = 'ha';
    public const HEBREW = 'he';
    public const HERERO = 'hz';
    public const HINDI = 'hi';
    public const HIRI_MOTU = 'ho';
    public const HUNGARIAN = 'hu';
    public const ICELANDIC = 'is';
    public const IDO = 'io';
    public const IGBO = 'ig';
    public const INDONESIAN = 'id';
    public const INTERLINGUA = 'ia';
    public const INTERLINGUE = 'ie';
    public const INUIT = 'iu';
    public const INUPIAQ = 'ik';
    public const IRISH = 'ga';
    public const ITALIAN = 'it';
    public const JAPANESE = 'ja';
    public const JAVANESE = 'jv';
    public const KALAALLISUT = 'kl';
    public const KANNADA = 'kn';
    public const KANURI = 'kr';
    public const KASHMIRI = 'ks';
    public const KAZAKH = 'kk';
    public const KHMER = 'km';
    public const KIKUYU = 'ki';
    public const KINYARWANDA = 'rw';
    public const KIRGHIZ = 'ky';
    public const KIRUNDI = 'rn';
    public const KOMI = 'kv';
    public const KONGO = 'kg';
    public const KOREAN = 'ko';
    public const KUANYAMA = 'kj';
    public const KURDISH = 'ku';
    public const LAO = 'lo';
    public const LATIN = 'la';
    public const LATVIAN = 'lv';
    public const LIMBURGISH = 'li';
    public const LINGALA = 'ln';
    public const LITHUANIAN = 'lt';
    public const LUBA_KATANGA = 'lu';
    public const LUXEMBOURGISH = 'lb';
    public const MACEDONIAN = 'mk';
    public const MALAGASY = 'mg';
    public const MALAY = 'ms';
    public const MALAYALAM = 'ml';
    public const MALTESE = 'mt';
    public const MANX = 'gv';
    public const MAORI = 'mi';
    public const MARATHI = 'mr';
    public const MARSHALLESE = 'mh';
    public const MOLDAVIAN = 'mo';
    public const MONGOLIAN = 'mn';
    public const NAURU = 'na';
    public const NAVAJO = 'nv';
    public const NDONGA = 'ng';
    public const NEPALI = 'ne';
    public const NORTHERN_SAMI = 'se';
    public const NORTH_NDEBELE = 'nd';
    public const NORWEGIAN = 'no';
    public const NORWEGIAN_BOKMAL = 'nb';
    public const NORWEGIAN_NYNORSK = 'nn';
    public const OCCITAN = 'oc';
    public const OJIBWA = 'oj';
    public const ORIYA = 'or';
    public const OROMO = 'om';
    public const OSSETIAN = 'os';
    public const PALI = 'pi';
    public const PANJABI = 'pa';
    public const PASHTO = 'ps';
    public const PERSIAN = 'fa';
    public const POLISH = 'pl';
    public const PORTUGUESE = 'pt';
    public const QUECHUA = 'qu';
    public const RAETO_ROMANCE = 'rm';
    public const ROMANIAN = 'ro';
    public const RUSSIAN = 'ru';
    public const RUSYN = 'ry';
    public const SAMOAN = 'sm';
    public const SANGO = 'sg';
    public const SANSKRIT = 'sa';
    public const SARDINIAN = 'sc';
    public const SERBIAN = 'sr';
    public const SERBO_CROATIAN = 'sh';
    public const SHONA = 'sn';
    public const SICHUAN_YI = 'ii';
    public const SINDHI = 'sd';
    public const SINHALA = 'si';
    public const SLOVAK = 'sk';
    public const SLOVENIAN = 'sl';
    public const SOMALI = 'so';
    public const SOUTHERN_SOTHO = 'st';
    public const SOUTH_NDEBELE = 'nr';
    public const SPANISH = 'es';
    public const SUNDANESE = 'su';
    public const SWAHILI = 'sw';
    public const SWATI = 'ss';
    public const SWEDISH = 'sv';
    public const TAGALOG = 'tl';
    public const TAHITIAN = 'ty';
    public const TAJIK = 'tg';
    public const TAMIL = 'ta';
    public const TATAR = 'tt';
    public const TELUGU = 'te';
    public const THAI = 'th';
    public const TIBETAN = 'bo';
    public const TIGRINYA = 'ti';
    public const TONGA = 'to';
    public const TSONGA = 'ts';
    public const TSWANA = 'tn';
    public const TURKISH = 'tr';
    public const TURKMEN = 'tk';
    public const TWI = 'tw';
    public const UIGHUR = 'ug';
    public const UKRAINIAN = 'uk';
    public const URDU = 'ur';
    public const UZBEK = 'uz';
    public const VENDA = 've';
    public const VIETNAMESE = 'vi';
    public const VOLAPUK = 'vo';
    public const WALLOON = 'wa';
    public const WELSH = 'cy';
    public const WESTERN_FRISIAN = 'fy';
    public const WOLOF = 'wo';
    public const XHOSA = 'xh';
    public const YIDDISH = 'yi';
    public const YORUBA = 'yo';
    public const ZHUANG = 'za';
    public const ZULU = 'zu';

    /** @var string[] */
    private static $names = [
        self::ABKHAZ => 'abkhaz',
        self::AFAR => 'afar',
        self::AFRIKAANS => 'afrikaans',
        self::AKAN => 'akan',
        self::ALBANIAN => 'albanian',
        self::AMHARIC => 'amharic',
        self::ARABIC => 'arabic',
        self::ARAGONESE => 'aragonese',
        self::ARMENIAN => 'armenian',
        self::ASSAMESE => 'assamese',
        self::AVARIC => 'avaric',
        self::AVESTAN => 'avestan',
        self::AYMARA => 'aymara',
        self::AZERBAIJANI => 'azerbaijani',
        self::BAMBARA => 'bambara',
        self::BASHKIR => 'bashkir',
        self::BASQUE => 'basque',
        self::BELARUSIAN => 'belarusian',
        self::BENGALI => 'bengali',
        self::BIHARI => 'bihari',
        self::BISLAMA => 'bislama',
        self::BOSNIAN => 'bosnian',
        self::BRETON => 'breton',
        self::BULGARIAN => 'bulgarian',
        self::BURMESE => 'burmese',
        self::CATALAN => 'catalan',
        self::CHAMORRO => 'chamorro',
        self::CHECHEN => 'chechen',
        self::CHICHEWA => 'chichewa',
        self::CHINESE => 'chinese',
        self::CHURCH_SLAVIC => 'church slavic',
        self::CHUVASH => 'chuvash',
        self::CORNISH => 'cornish',
        self::CORSICAN => 'corsican',
        self::CREE => 'cree',
        self::CROATIAN => 'croatian',
        self::CZECH => 'czech',
        self::DANISH => 'danish',
        self::DIVEHI => 'divehi',
        self::DUTCH => 'dutch',
        self::DZONGKHA => 'dzongkha',
        self::ENGLISH => 'english',
        self::ESPERANTO => 'esperanto',
        self::ESTONIAN => 'estonian',
        self::EWE => 'ewe',
        self::FAROESE => 'faroese',
        self::FIJIAN => 'fijian',
        self::FINNISH => 'finnish',
        self::FRENCH => 'french',
        self::FULAH => 'fulah',
        self::GAELIC => 'gaelic',
        self::GALICIAN => 'galician',
        self::GANDA => 'ganda',
        self::GEORGIAN => 'georgian',
        self::GERMAN => 'german',
        self::GREEK => 'greek',
        self::GUARANI => 'guaraní',
        self::GUJARATI => 'gujarati',
        self::HAITIAN => 'haitian',
        self::HAUSA => 'hausa',
        self::HEBREW => 'hebrew',
        self::HERERO => 'herero',
        self::HINDI => 'hindi',
        self::HIRI_MOTU => 'hiri motu',
        self::HUNGARIAN => 'hungarian',
        self::ICELANDIC => 'icelandic',
        self::IDO => 'ido',
        self::IGBO => 'igbo',
        self::INDONESIAN => 'indonesian',
        self::INTERLINGUA => 'interlingua',
        self::INTERLINGUE => 'interlingue',
        self::INUIT => 'inuit',
        self::INUPIAQ => 'inupiaq',
        self::IRISH => 'irish',
        self::ITALIAN => 'italian',
        self::JAPANESE => 'japanese',
        self::JAVANESE => 'javanese',
        self::KALAALLISUT => 'kalaallisut',
        self::KANNADA => 'kannada',
        self::KANURI => 'kanuri',
        self::KASHMIRI => 'kashmiri',
        self::KAZAKH => 'kazakh',
        self::KHMER => 'khmer',
        self::KIKUYU => 'kikuyu',
        self::KINYARWANDA => 'kinyarwanda',
        self::KIRGHIZ => 'kirghiz',
        self::KIRUNDI => 'kirundi',
        self::KOMI => 'komi',
        self::KONGO => 'kongo',
        self::KOREAN => 'korean',
        self::KUANYAMA => 'kuanyama',
        self::KURDISH => 'kurdish',
        self::LAO => 'lao',
        self::LATIN => 'latin',
        self::LATVIAN => 'latvian',
        self::LIMBURGISH => 'limburgish',
        self::LINGALA => 'lingala',
        self::LITHUANIAN => 'lithuanian',
        self::LUBA_KATANGA => 'luba-katanga',
        self::LUXEMBOURGISH => 'luxembourgish',
        self::MACEDONIAN => 'macedonian',
        self::MALAGASY => 'malagasy',
        self::MALAY => 'malay',
        self::MALAYALAM => 'malayalam',
        self::MALTESE => 'maltese',
        self::MANX => 'manx',
        self::MAORI => 'māori',
        self::MARATHI => 'marathi',
        self::MARSHALLESE => 'marshallese',
        self::MOLDAVIAN => 'moldavian',
        self::MONGOLIAN => 'mongolian',
        self::NAURU => 'nauru',
        self::NAVAJO => 'navajo',
        self::NDONGA => 'ndonga',
        self::NEPALI => 'nepali',
        self::NORTHERN_SAMI => 'northern sami',
        self::NORTH_NDEBELE => 'north ndebele',
        self::NORWEGIAN => 'norwegian',
        self::NORWEGIAN_BOKMAL => 'norwegian bokmål',
        self::NORWEGIAN_NYNORSK => 'norwegian nynorsk',
        self::OCCITAN => 'occitan',
        self::OJIBWA => 'ojibwa',
        self::ORIYA => 'oriya',
        self::OROMO => 'oromo',
        self::OSSETIAN => 'ossetian',
        self::PALI => 'pāli',
        self::PANJABI => 'panjabi',
        self::PASHTO => 'pashto',
        self::PERSIAN => 'persian',
        self::POLISH => 'polish',
        self::PORTUGUESE => 'portuguese',
        self::QUECHUA => 'quechua',
        self::RAETO_ROMANCE => 'raeto-romance',
        self::ROMANIAN => 'romanian',
        self::RUSSIAN => 'russian',
        self::RUSYN => 'rusyn',
        self::SAMOAN => 'samoan',
        self::SANGO => 'sango',
        self::SANSKRIT => 'sanskrit',
        self::SARDINIAN => 'sardinian',
        self::SERBIAN => 'serbian',
        self::SERBO_CROATIAN => 'serbo-croatian',
        self::SHONA => 'shona',
        self::SICHUAN_YI => 'sichuan yi',
        self::SINDHI => 'sindhi',
        self::SINHALA => 'sinhala',
        self::SLOVAK => 'slovak',
        self::SLOVENIAN => 'slovenian',
        self::SOMALI => 'somali',
        self::SOUTHERN_SOTHO => 'southern sotho',
        self::SOUTH_NDEBELE => 'south ndebele',
        self::SPANISH => 'spanish',
        self::SUNDANESE => 'sundanese',
        self::SWAHILI => 'swahili',
        self::SWATI => 'swati',
        self::SWEDISH => 'swedish',
        self::TAGALOG => 'tagalog',
        self::TAHITIAN => 'tahitian',
        self::TAJIK => 'tajik',
        self::TAMIL => 'tamil',
        self::TATAR => 'tatar',
        self::TELUGU => 'telugu',
        self::THAI => 'thai',
        self::TIBETAN => 'tibetan',
        self::TIGRINYA => 'tigrinya',
        self::TONGA => 'tonga',
        self::TSONGA => 'tsonga',
        self::TSWANA => 'tswana',
        self::TURKISH => 'turkish',
        self::TURKMEN => 'turkmen',
        self::TWI => 'twi',
        self::UIGHUR => 'uighur',
        self::UKRAINIAN => 'ukrainian',
        self::URDU => 'urdu',
        self::UZBEK => 'uzbek',
        self::VENDA => 'venda',
        self::VIETNAMESE => 'vietnamese',
        self::VOLAPUK => 'volapük',
        self::WALLOON => 'walloon',
        self::WELSH => 'welsh',
        self::WESTERN_FRISIAN => 'western frisian',
        self::WOLOF => 'wolof',
        self::XHOSA => 'xhosa',
        self::YIDDISH => 'yiddish',
        self::YORUBA => 'yoruba',
        self::ZHUANG => 'zhuang',
        self::ZULU => 'zulu',
    ];

    /** @var string[] */
    private static $native = [
        self::ABKHAZ => 'аҧсуа',
        self::AFAR => 'afaraf',
        self::AFRIKAANS => 'afrikaans',
        self::AKAN => 'akan',
        self::ALBANIAN => 'shqip',
        self::AMHARIC => 'አማርኛ',
        self::ARABIC => '‫العربية‬',
        self::ARAGONESE => 'aragonés',
        self::ARMENIAN => 'հայերեն',
        self::ASSAMESE => 'অসমীয়া',
        self::AVARIC => 'авар мацӏ',
        self::AVESTAN => 'avesta',
        self::AYMARA => 'aymar aru',
        self::AZERBAIJANI => 'azərbaycan dili',
        self::BAMBARA => 'bamanankan',
        self::BASHKIR => 'башҡорт теле',
        self::BASQUE => 'euskara',
        self::BELARUSIAN => 'беларуская',
        self::BENGALI => 'বাংলা',
        self::BIHARI => 'भोजपुरी',
        self::BISLAMA => 'bislama',
        self::BOSNIAN => 'bosanski jezik',
        self::BRETON => 'brezhoneg',
        self::BULGARIAN => 'български език',
        self::BURMESE => 'burmese', ///
        self::CATALAN => 'català',
        self::CHAMORRO => 'chamoru',
        self::CHECHEN => 'нохчийн мотт',
        self::CHICHEWA => 'chicheŵa; chinyanja',
        self::CHINESE => '中文',
        self::CHURCH_SLAVIC => 'staroslověnština',
        self::CHUVASH => 'чӑваш чӗлхи',
        self::CORNISH => 'kernewek',
        self::CORSICAN => 'corsu',
        self::CREE => 'ᓀᐦᐃᔭᐍᐏᐣ',
        self::CROATIAN => 'hrvatski',
        self::CZECH => 'čeština',
        self::DANISH => 'dansk',
        self::DIVEHI => '‫ދިވެހި‬',
        self::DUTCH => 'nederlands',
        self::DZONGKHA => 'རྫོང་ཁ',
        self::ENGLISH => 'english',
        self::ESPERANTO => 'esperanto',
        self::ESTONIAN => 'eesti keel',
        self::EWE => 'ɛʋɛgbɛ',
        self::FAROESE => 'føroyskt',
        self::FIJIAN => 'vosa vakaviti',
        self::FINNISH => 'suomen kieli',
        self::FRENCH => 'français',
        self::FULAH => 'fulfulde',
        self::GAELIC => 'gàidhlig',
        self::GALICIAN => 'galego',
        self::GANDA => 'luganda',
        self::GEORGIAN => 'ქართული',
        self::GERMAN => 'deutsch',
        self::GREEK => 'ελληνικά',
        self::GUARANI => 'avañe\'ẽ',
        self::GUJARATI => 'ગુજરાતી',
        self::HAITIAN => 'kreyòl ayisyen',
        self::HAUSA => '‫هَوُسَ‬',
        self::HEBREW => '‫עברית‬',
        self::HERERO => 'otjiherero',
        self::HINDI => 'हिन्दी',
        self::HIRI_MOTU => 'hiri motu',
        self::HUNGARIAN => 'magyar',
        self::ICELANDIC => 'íslenska',
        self::IDO => 'ido',
        self::IGBO => 'igbo',
        self::INDONESIAN => 'bahasa indonesia',
        self::INTERLINGUA => 'interlingua',
        self::INTERLINGUE => 'interlingue',
        self::INUIT => 'ᐃᓄᒃᑎᑐᑦ',
        self::INUPIAQ => 'iñupiaq',
        self::IRISH => 'gaeilge',
        self::ITALIAN => 'italiano',
        self::JAPANESE => '日本語',
        self::JAVANESE => 'basa jawa',
        self::KALAALLISUT => 'kalaallisut',
        self::KANNADA => 'ಕನ್ನಡ',
        self::KANURI => 'kanuri',
        self::KASHMIRI => 'कश्मीरी; ‫كشميري‬',
        self::KAZAKH => 'қазақ тілі',
        self::KHMER => 'ភាសាខ្មែរ',
        self::KIKUYU => 'gĩkũyũ',
        self::KINYARWANDA => 'kinyarwanda',
        self::KIRGHIZ => 'кыргыз тили',
        self::KIRUNDI => 'kirundi',
        self::KOMI => 'коми кыв',
        self::KONGO => 'kikongo',
        self::KOREAN => '한국어',
        self::KUANYAMA => 'kuanyama',
        self::KURDISH => 'kurdî; ‫كوردی‬',
        self::LAO => 'ພາສາລາວ',
        self::LATIN => 'latine',
        self::LATVIAN => 'latviešu valoda',
        self::LIMBURGISH => 'limburgs',
        self::LINGALA => 'lingála',
        self::LITHUANIAN => 'lietuvių kalba',
        self::LUBA_KATANGA => 'luba-katanga',
        self::LUXEMBOURGISH => 'lëtzebuergesch',
        self::MACEDONIAN => 'македонски јазик',
        self::MALAGASY => 'malagasy fiteny',
        self::MALAY => 'bahasa melayu; ‫بهاس ملايو‬',
        self::MALAYALAM => 'മലയാളം',
        self::MALTESE => 'malti',
        self::MANX => 'ghaelg',
        self::MAORI => 'te reo māori',
        self::MARATHI => 'मराठी',
        self::MARSHALLESE => 'kajin m̧ajeļ',
        self::MOLDAVIAN => 'лимба молдовеняскэ',
        self::MONGOLIAN => 'монгол',
        self::NAURU => 'ekakairũ naoero',
        self::NAVAJO => 'diné bizaad; dinékʼehǰí',
        self::NDONGA => 'owambo',
        self::NEPALI => 'नेपाली',
        self::NORTHERN_SAMI => 'davvisámegiella',
        self::NORTH_NDEBELE => 'isindebele',
        self::NORWEGIAN => 'norsk',
        self::NORWEGIAN_BOKMAL => 'norsk bokmål',
        self::NORWEGIAN_NYNORSK => 'norsk nynorsk',
        self::OCCITAN => 'occitan',
        self::OJIBWA => 'ᐊᓂᔑᓈᐯᒧᐎᓐ',
        self::ORIYA => 'ଓଡ଼ିଆ',
        self::OROMO => 'afaan oromoo',
        self::OSSETIAN => 'ирон æвзаг',
        self::PALI => 'पािऴ',
        self::PANJABI => 'ਪੰਜਾਬੀ; ‫پنجابی‬',
        self::PASHTO => '‫پښتو‬',
        self::PERSIAN => '‫فارسی‬',
        self::POLISH => 'polski',
        self::PORTUGUESE => 'português',
        self::QUECHUA => 'runa simi; kichwa',
        self::RAETO_ROMANCE => 'rumantsch grischun',
        self::ROMANIAN => 'română',
        self::RUSSIAN => 'русский язык',
        self::RUSYN => 'русинськый язык',
        self::SAMOAN => 'gagana fa\'a samoa',
        self::SANGO => 'yângâ tî sängö',
        self::SANSKRIT => 'संस्कृतम्',
        self::SARDINIAN => 'sardu',
        self::SERBIAN => 'српски језик',
        self::SERBO_CROATIAN => 'српскохрватски',
        self::SHONA => 'chishona',
        self::SICHUAN_YI => 'ꆇꉙ',
        self::SINDHI => 'सिन्धी; ‫سنڌي، سندھی‬',
        self::SINHALA => 'සිංහල',
        self::SLOVAK => 'slovenčina',
        self::SLOVENIAN => 'slovenščina',
        self::SOMALI => 'soomaaliga; af soomaali',
        self::SOUTHERN_SOTHO => 'sesotho',
        self::SOUTH_NDEBELE => 'ndébélé',
        self::SPANISH => 'español',
        self::SUNDANESE => 'basa sunda',
        self::SWAHILI => 'kiswahili',
        self::SWATI => 'siswati',
        self::SWEDISH => 'svenska',
        self::TAGALOG => 'tagalog',
        self::TAHITIAN => 'reo mā`ohi',
        self::TAJIK => 'тоҷикӣ; toğikī; ‫تاجیکی‬',
        self::TAMIL => 'தமிழ்',
        self::TATAR => 'татарча; tatarça; ‫تاتارچا‬',
        self::TELUGU => 'తెలుగు',
        self::THAI => 'ไทย',
        self::TIBETAN => 'བོད་ཡིག',
        self::TIGRINYA => 'ትግርኛ',
        self::TONGA => 'faka tonga',
        self::TSONGA => 'xitsonga',
        self::TSWANA => 'setswana',
        self::TURKISH => 'türkçe',
        self::TURKMEN => 'türkmen; түркмен',
        self::TWI => 'twi',
        self::UIGHUR => 'uyƣurqə; ‫ئۇيغۇرچ ‬',
        self::UKRAINIAN => 'українська мова',
        self::URDU => '‫اردو‬',
        self::UZBEK => 'o\'zbek; ўзбек; ‫أۇزبېك‬',
        self::VENDA => 'tshivenḓa',
        self::VIETNAMESE => 'tiếng việt',
        self::VOLAPUK => 'volapük',
        self::WALLOON => 'walon',
        self::WELSH => 'cymraeg',
        self::WESTERN_FRISIAN => 'frysk',
        self::WOLOF => 'wollof',
        self::XHOSA => 'isixhosa',
        self::YIDDISH => '‫ייִדיש‬',
        self::YORUBA => 'yorùbá',
        self::ZHUANG => 'saɯ cueŋƅ',
        self::ZULU => 'isizulu',
    ];

    /** @var string[] */
    private static $idents = [
        self::ABKHAZ => 'abkhaz',
        self::AFAR => 'afar',
        self::AFRIKAANS => 'afrikaans',
        self::AKAN => 'akan',
        self::ALBANIAN => 'albanian',
        self::AMHARIC => 'amharic',
        self::ARABIC => 'arabic',
        self::ARAGONESE => 'aragonese',
        self::ARMENIAN => 'armenian',
        self::ASSAMESE => 'assamese',
        self::AVARIC => 'avaric',
        self::AVESTAN => 'avestan',
        self::AYMARA => 'aymara',
        self::AZERBAIJANI => 'azerbaijani',
        self::BAMBARA => 'bambara',
        self::BASHKIR => 'bashkir',
        self::BASQUE => 'basque',
        self::BELARUSIAN => 'belarusian',
        self::BENGALI => 'bengali',
        self::BIHARI => 'bihari',
        self::BISLAMA => 'bislama',
        self::BOSNIAN => 'bosnian',
        self::BRETON => 'breton',
        self::BULGARIAN => 'bulgarian',
        self::BURMESE => 'burmese',
        self::CATALAN => 'catalan',
        self::CHAMORRO => 'chamorro',
        self::CHECHEN => 'chechen',
        self::CHICHEWA => 'chichewa',
        self::CHINESE => 'chinese',
        self::CHURCH_SLAVIC => 'church-slavic',
        self::CHUVASH => 'chuvash',
        self::CORNISH => 'cornish',
        self::CORSICAN => 'corsican',
        self::CREE => 'cree',
        self::CROATIAN => 'croatian',
        self::CZECH => 'czech',
        self::DANISH => 'danish',
        self::DIVEHI => 'divehi',
        self::DUTCH => 'dutch',
        self::DZONGKHA => 'dzongkha',
        self::ENGLISH => 'english',
        self::ESPERANTO => 'esperanto',
        self::ESTONIAN => 'estonian',
        self::EWE => 'ewe',
        self::FAROESE => 'faroese',
        self::FIJIAN => 'fijian',
        self::FINNISH => 'finnish',
        self::FRENCH => 'french',
        self::FULAH => 'fulah',
        self::GAELIC => 'gaelic',
        self::GALICIAN => 'galician',
        self::GANDA => 'ganda',
        self::GEORGIAN => 'georgian',
        self::GERMAN => 'german',
        self::GREEK => 'greek',
        self::GUARANI => 'guarani',
        self::GUJARATI => 'gujarati',
        self::HAITIAN => 'haitian',
        self::HAUSA => 'hausa',
        self::HEBREW => 'hebrew',
        self::HERERO => 'herero',
        self::HINDI => 'hindi',
        self::HIRI_MOTU => 'hiri-motu',
        self::HUNGARIAN => 'hungarian',
        self::ICELANDIC => 'icelandic',
        self::IDO => 'ido',
        self::IGBO => 'igbo',
        self::INDONESIAN => 'indonesian',
        self::INTERLINGUA => 'interlingua',
        self::INTERLINGUE => 'interlingue',
        self::INUIT => 'inuit',
        self::INUPIAQ => 'inupiaq',
        self::IRISH => 'irish',
        self::ITALIAN => 'italian',
        self::JAPANESE => 'japanese',
        self::JAVANESE => 'javanese',
        self::KALAALLISUT => 'kalaallisut',
        self::KANNADA => 'kannada',
        self::KANURI => 'kanuri',
        self::KASHMIRI => 'kashmiri',
        self::KAZAKH => 'kazakh',
        self::KHMER => 'khmer',
        self::KIKUYU => 'kikuyu',
        self::KINYARWANDA => 'kinyarwanda',
        self::KIRGHIZ => 'kirghiz',
        self::KIRUNDI => 'kirundi',
        self::KOMI => 'komi',
        self::KONGO => 'kongo',
        self::KOREAN => 'korean',
        self::KUANYAMA => 'kuanyama',
        self::KURDISH => 'kurdish',
        self::LAO => 'lao',
        self::LATIN => 'latin',
        self::LATVIAN => 'latvian',
        self::LIMBURGISH => 'limburgish',
        self::LINGALA => 'lingala',
        self::LITHUANIAN => 'lithuanian',
        self::LUBA_KATANGA => 'luba-katanga',
        self::LUXEMBOURGISH => 'luxembourgish',
        self::MACEDONIAN => 'macedonian',
        self::MALAGASY => 'malagasy',
        self::MALAY => 'malay',
        self::MALAYALAM => 'malayalam',
        self::MALTESE => 'maltese',
        self::MANX => 'manx',
        self::MAORI => 'aori',
        self::MARATHI => 'marathi',
        self::MARSHALLESE => 'marshallese',
        self::MOLDAVIAN => 'moldavian',
        self::MONGOLIAN => 'mongolian',
        self::NAURU => 'nauru',
        self::NAVAJO => 'navajo',
        self::NDONGA => 'ndonga',
        self::NEPALI => 'nepali',
        self::NORTHERN_SAMI => 'northern-sami',
        self::NORTH_NDEBELE => 'north-ndebele',
        self::NORWEGIAN => 'norwegian',
        self::NORWEGIAN_BOKMAL => 'norwegian-bokmal',
        self::NORWEGIAN_NYNORSK => 'norwegian-nynorsk',
        self::OCCITAN => 'occitan',
        self::OJIBWA => 'ojibwa',
        self::ORIYA => 'oriya',
        self::OROMO => 'oromo',
        self::OSSETIAN => 'ossetian',
        self::PALI => 'pali',
        self::PANJABI => 'panjabi',
        self::PASHTO => 'pashto',
        self::PERSIAN => 'persian',
        self::POLISH => 'polish',
        self::PORTUGUESE => 'portuguese',
        self::QUECHUA => 'quechua',
        self::RAETO_ROMANCE => 'raeto-romance',
        self::ROMANIAN => 'romanian',
        self::RUSSIAN => 'russian',
        self::RUSYN => 'rusyn',
        self::SAMOAN => 'samoan',
        self::SANGO => 'sango',
        self::SANSKRIT => 'sanskrit',
        self::SARDINIAN => 'sardinian',
        self::SERBIAN => 'serbian',
        self::SERBO_CROATIAN => 'serbo-croatian',
        self::SHONA => 'shona',
        self::SICHUAN_YI => 'sichuan-yi',
        self::SINDHI => 'sindhi',
        self::SINHALA => 'sinhala',
        self::SLOVAK => 'slovak',
        self::SLOVENIAN => 'slovenian',
        self::SOMALI => 'somali',
        self::SOUTHERN_SOTHO => 'southern-sotho',
        self::SOUTH_NDEBELE => 'south-ndebele',
        self::SPANISH => 'spanish',
        self::SUNDANESE => 'sundanese',
        self::SWAHILI => 'swahili',
        self::SWATI => 'swati',
        self::SWEDISH => 'swedish',
        self::TAGALOG => 'tagalog',
        self::TAHITIAN => 'tahitian',
        self::TAJIK => 'tajik',
        self::TAMIL => 'tamil',
        self::TATAR => 'tatar',
        self::TELUGU => 'telugu',
        self::THAI => 'thai',
        self::TIBETAN => 'tibetan',
        self::TIGRINYA => 'tigrinya',
        self::TONGA => 'tonga',
        self::TSONGA => 'tsonga',
        self::TSWANA => 'tswana',
        self::TURKISH => 'turkish',
        self::TURKMEN => 'turkmen',
        self::TWI => 'twi',
        self::UIGHUR => 'uighur',
        self::UKRAINIAN => 'ukrainian',
        self::URDU => 'urdu',
        self::UZBEK => 'uzbek',
        self::VENDA => 'venda',
        self::VIETNAMESE => 'vietnamese',
        self::VOLAPUK => 'volapuk',
        self::WALLOON => 'walloon',
        self::WELSH => 'welsh',
        self::WESTERN_FRISIAN => 'western-frisian',
        self::WOLOF => 'wolof',
        self::XHOSA => 'xhosa',
        self::YIDDISH => 'yiddish',
        self::YORUBA => 'yoruba',
        self::ZHUANG => 'zhuang',
        self::ZULU => 'zulu',
    ];

    public function getName(): string
    {
        return self::$names[$this->getValue()];
    }

    public function getNativeName(): string
    {
        return self::$native[$this->getValue()];
    }

    public function getIdent(): string
    {
        return self::$idents[$this->getValue()];
    }

    public static function getByIdent(string $ident): self
    {
        /** @var string $code */
        $code = array_search($ident, self::$idents, true);

        return self::get($code);
    }

    public static function validateValue(string &$value): bool
    {
        $value = strtolower($value);

        return parent::validateValue($value);
    }

}
