/* eslint-env node */

const fs = require('fs');
const path = require('path');

const yargs = require('yargs/yargs');
const { hideBin } = require('yargs/helpers');
const inquirer = require('inquirer');
const prettier = require('prettier');
const toCamelCase = require('../templates/toCamelCase.js');

const prettierConfig = prettier.resolveConfig.sync(process.cwd());

const createComponent = async () => {
  const checkDirs = ['packages/banana/src', 'packages/banana-react/src', 'docs/example', 'public'];

  for (const dir of checkDirs) {
    if (!fs.existsSync(path.resolve(process.cwd(), dir))) {
      console.error(
        `❌ Directory '${dir}' does not exist, please run the command in the root directory of the project.`,
      );
      process.exit(1);
    }
  }

  const { name, dirs, chineseName, description, formField } = await inquirer.prompt([
    {
      type: 'input',
      name: 'name',
      validate: (input) => {
        if (!input) return 'Component name is required.';
        const reg = /^[a-zA-Z][a-zA-Z0-9-]*$/;
        if (reg.test(input)) {
          return true;
        }
        return 'Component name must start with a letter and contain only letters, numbers and dashes.';
      },
      filter: (input) => input.toLowerCase(),
      message: 'Component name:',
    },
    {
      type: 'input',
      name: 'chineseName',
      validate: (input) => {
        if (!input) return 'A chinese name of the component is required.';
        return true;
      },
      message: 'Chinese name of the component:',
    },
    {
      type: 'input',
      name: 'description',
      validate: (input) => {
        if (!input) return 'A description of the component is required.';
        return true;
      },
      message: 'Description of the component:',
    },
    {
      type: 'confirm',
      name: 'formField',
      message: 'Is this a field of the form?',
      default: false,
    },
    {
      type: 'checkbox',
      name: 'dirs',
      choices: [
        {
          name: 'banana',
          checked: true,
        },
        {
          name: 'banana-react',
          checked: true,
        },
      ],
      validate: (input) => {
        if (input.length < 1) {
          return 'At least one directory must be selected.';
        }
        return true;
      },
      message: 'Select the directory to create:',
    },
  ]);

  for (const dir of checkDirs) {
    const dirPath = path.resolve(process.cwd(), dir, name);
    if (fs.existsSync(dirPath)) {
      console.error(`❌ Directory '${dirPath}' already exists.`);
      process.exit(1);
    }
  }

  if (dirs.includes('banana')) {
    console.log('----------------------------------------');
    console.log('🍌 Creating banana component...');
    console.log('----------------------------------------');

    fs.mkdirSync(path.resolve(process.cwd(), 'packages/banana/src', name));

    const index = require('../templates/banana/index.js')(name, formField);
    const formatted = prettier.format(index, {
      ...prettierConfig,
      parser: 'typescript',
    });
    fs.writeFileSync(path.resolve(process.cwd(), 'packages/banana/src', name, 'index.ts'), formatted);
    console.log(`✅ Created 'packages/banana/src/${name}/index.ts'`);

    const styles = require('../templates/banana/index.styles.js')(name);
    const formattedStyles = prettier.format(styles, {
      ...prettierConfig,
      parser: 'typescript',
    });
    fs.writeFileSync(path.resolve(process.cwd(), 'packages/banana/src', name, 'index.styles.ts'), formattedStyles);
    console.log(`✅ Created 'packages/banana/src/${name}/index.styles.ts'`);

    const test = require('../templates/banana/index.test.js')(name, formField);
    const formattedTest = prettier.format(test, {
      ...prettierConfig,
      parser: 'typescript',
    });
    fs.writeFileSync(path.resolve(process.cwd(), 'packages/banana/src', name, 'index.test.ts'), formattedTest);
    console.log(`✅ Created 'packages/banana/src/${name}/index.test.ts'`);

    const componentNames = fs
      .readdirSync(path.resolve(process.cwd(), 'packages/banana/src'), { withFileTypes: true })
      // filter out which is not a directory or does not have index.ts
      .filter(
        (dirent) =>
          dirent.isDirectory() &&
          fs.existsSync(path.resolve(process.cwd(), 'packages/banana/src', dirent.name, 'index.ts')),
      )
      .map((dirent) => dirent.name);

    const bananaIndex = require('../templates/banana/banana-index.js')(componentNames);
    const formattedBananaIndex = prettier.format(bananaIndex, {
      ...prettierConfig,
      parser: 'typescript',
    });
    fs.writeFileSync(path.resolve(process.cwd(), 'packages/banana/src/index.ts'), formattedBananaIndex);
    console.log(`✅ Updated 'packages/banana/src/index.ts'`);

    // build after create
    console.log('🚀 Building...');
    require('child_process').execSync('pnpm build', {
      stdio: 'inherit',
      cwd: path.resolve(process.cwd(), 'packages/banana'),
    });
    console.log('✅ Build completed.');

    console.log('----------------------------------------');
    console.log('🍌 Banana component created successfully.');
    console.log('----------------------------------------');
  }

  if (dirs.includes('banana-react')) {
    console.log('----------------------------------------');
    console.log('🍌 Creating banana-react component...');
    console.log('----------------------------------------');

    fs.mkdirSync(path.resolve(process.cwd(), 'packages/banana-react/src', name));

    if (dirs.includes('banana')) {
      const index = require('../templates/banana-react/index.js')(name, formField);
      const formatted = prettier.format(index, {
        ...prettierConfig,
        parser: 'typescript',
      });
      fs.writeFileSync(path.resolve(process.cwd(), 'packages/banana-react/src', name, 'index.ts'), formatted);
      console.log(`✅ Created 'packages/banana-react/src/${name}/index.ts'`);
    } else {
      // react-only
      const reactOnly = require('../templates/banana-react/index.react-only.js')(name);
      const formatted = prettier.format(reactOnly, {
        ...prettierConfig,
        parser: 'typescript',
      });
      fs.writeFileSync(path.resolve(process.cwd(), 'packages/banana-react/src', name, 'index.ts'), formatted);
      console.log(`✅ Created 'packages/banana-react/src/${name}/index.ts'`);
    }

    const componentNames = fs
      .readdirSync(path.resolve(process.cwd(), 'packages/banana-react/src'), { withFileTypes: true })
      // filter out which is not a directory or does not have index.ts
      .filter(
        (dirent) =>
          dirent.isDirectory() &&
          fs.existsSync(path.resolve(process.cwd(), 'packages/banana/src', dirent.name, 'index.ts')),
      )
      .map((dirent) => dirent.name);

    const bananaReactIndex = require('../templates/banana-react/banana-react-index.js')(componentNames);
    const formatted = prettier.format(bananaReactIndex, {
      ...prettierConfig,
      parser: 'typescript',
    });
    fs.writeFileSync(path.resolve(process.cwd(), 'packages/banana-react/src/index.ts'), formatted);
    console.log(`✅ Updated 'packages/banana-react/src/index.ts'`);

    // build after create
    console.log('🚀 Building...');
    require('child_process').execSync('pnpm build', {
      stdio: 'inherit',
      cwd: path.resolve(process.cwd(), 'packages/banana-react'),
    });
    console.log('✅ Build completed.');

    console.log('----------------------------------------');
    console.log('🍌 Banana-react component created successfully.');
    console.log('----------------------------------------');
  }

  console.log('----------------------------------------');
  console.log('📝 Creating docs...');
  console.log('----------------------------------------');

  fs.mkdirSync(path.resolve(process.cwd(), 'docs/example', toCamelCase(name)));
  fs.mkdirSync(path.resolve(process.cwd(), 'docs/example', toCamelCase(name), 'demos'));
  fs.mkdirSync(path.resolve(process.cwd(), 'public', toCamelCase(name)));
  const basicUsage = require('../templates/docs/basicUsage.js')(name);
  const formattedBasicUsage = prettier.format(basicUsage, {
    ...prettierConfig,
    parser: 'typescript',
  });
  fs.writeFileSync(
    path.resolve(process.cwd(), 'docs/example', toCamelCase(name), 'demos', 'basicUsage.tsx'),
    formattedBasicUsage,
  );
  console.log(`✅ Created 'docs/example/${toCamelCase(name)}/basicUsage.tsx'`);

  const index = require('../templates/docs/index.js')(name, chineseName, description);
  const formattedIndex = prettier.format(index, {
    ...prettierConfig,
    parser: 'markdown',
  });
  fs.writeFileSync(path.resolve(process.cwd(), 'docs/example', toCamelCase(name), 'index.md'), formattedIndex);
  console.log(`✅ Created 'docs/example/${toCamelCase(name)}/index.md'`);

  const htmlSourceCode = require('../templates/docs/htmlSource.js')(name);
  const formattedHtmlSourceCode = prettier.format(htmlSourceCode, {
    ...prettierConfig,
    parser: 'html',
  });
  fs.writeFileSync(
    path.resolve(process.cwd(), `public`, toCamelCase(name), 'basicUsage.html'),
    formattedHtmlSourceCode,
  );

  console.log(`✅ Created 'public/${toCamelCase(name)}/basicUsage.html`);

  console.log('----------------------------------------');
  console.log('📝 Docs created successfully.');
  console.log('----------------------------------------');

  console.log('🎉 All done!');
};

const cli = yargs(hideBin(process.argv));

cli
  .strict()
  .usage('Usage: $0 <command> [options]')
  .command('new', 'Create a new component', () => {}, createComponent).argv;
