// -*-Mode: C++; tab-width: 4; c-basic-offset: 4;-*-
// vi:set ts=4 sw=4:
//
// Partition.h -- ルール関連のクラス定義、ルール宣言
// 
// Copyright (c) 2012, 2023 Ricoh Company, Ltd.
// 
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// 
//     http://www.apache.org/licenses/LICENSE-2.0
// 
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// 

#ifndef	__SYDNEY_SCHEMA_PARTITION_H
#define	__SYDNEY_SCHEMA_PARTITION_H

#include "Schema/Module.h"
#include "Schema/Object.h"

#include "Common/Hasher.h"
#include "Common/UnicodeString.h"

#include "Statement/PartitionDefinition.h"

#include "ModCharString.h"
#include "ModHashMap.h"
#include "ModUnicodeString.h"
#include "ModVector.h"

_SYDNEY_BEGIN

namespace Statement
{
	class AlterPartitionStatement;
	class DropPartitionStatement;
	class ColumnNameList;
	class Object;
}
namespace Trans
{
	class Transaction;
}

_SYDNEY_SCHEMA_BEGIN

class Database;
class LogData;
namespace SystemTable
{
	class Partition;
}

//	CLASS
//	Schema::Partition -- ルールオブジェクトを表すクラス
//
//	NOTES
//		ルールの親オブジェクトは存在しない

class Partition
	: public Schema::Object
{
public:
	friend class SystemTable::Partition;

	typedef Statement::PartitionDefinition::Category Category;

	//	TYPEDEF public
	//	Schema::Partition::Pointer -- Partitionを保持するObjectPointer
	//
	//	NOTES
	typedef PartitionPointer Pointer;

	struct Log {


		//	STRUCT
		//		Schema::Log::Value -- ログの共通要素位置
		//
		//	NOTES
		enum Value {
			Name = 0,						// 名前
			ID,								// ID
			TableID,						// 対象の表ID
			ValueNum
		};

		//	STRUCT
		//		Schema::Log::Create -- Create ログの要素位置
		//
		//	NOTES
		struct Create {
			enum Value {
				Category = ValueNum,
				FunctionName,
				ColumnIDs,
				Num
			};
		};

		//	STRUCT
		//		Schema::Log::Drop -- Drop ログの要素位置
		//
		//	NOTES
		struct Drop {
			enum Value {
				Num = ValueNum
			};
		};

		//	STRUCT
		//		Schema::Log::Alter -- Alter ログの要素位置
		//
		//	NOTES
		struct Alter {
			enum Value {
				PrevFunctionName = ValueNum,
				PrevColumnIDs,
				PostFunctionName,
				PostColumnIDs,
				Num
			};
		};
	};


	// STRUCT
	//	Partition::Target -- ルールを指定する情報
	//
	// NOTES

	struct Target
	{
		Name					m_cFunctionName;
		ModVector<ID::Value>	m_vecColumn;

		Target() : m_cFunctionName(), m_vecColumn() {}
		Target(const Target& cOther_)
			: m_cFunctionName(cOther_.m_cFunctionName),
			  m_vecColumn(cOther_.m_vecColumn)
		{}
	};

	////////////////////
	// Partitionのメソッド //
	////////////////////

	// コンストラクター
	Partition();
	// デストラクター
	SYD_SCHEMA_FUNCTION
	~Partition();

	// DataArrayDataを元にインスタンスを生成する
	static Partition*		getNewInstance(const Common::DataArrayData& cData_);

	void					clear();			// メンバーをすべて初期化する

	static Partition*		create(Database& cDatabase_,
								   const Statement::PartitionDefinition& cStatement_,
								   LogData& cLogData_,
								   Trans::Transaction& cTrans_);
	static Partition*		create(Trans::Transaction& cTrans_,
								   const Database& cDatabase_,
								   const LogData& cLogData_);
												// ルールを生成する

	static Name				getName(const Statement::DropPartitionStatement& cStatement_);
	static Name				getName(const Statement::AlterPartitionStatement& cStatement_);
												// SQL文からルールが対象とする表名を得る
	static Name				getFunctionName(const Statement::AlterPartitionStatement& cStatement_);
												// SQL文からルールの関数名を得る
	static ModVector<ID::Value> getColumnIDs(Trans::Transaction& cTrans_,
											 const Table* pTable_,
											 const Statement::ColumnNameList& cColumnList_);
												// SQL文からルールに渡す列IDリストを得る

	static void				drop(Trans::Transaction& cTrans_,
								 Partition& cPartition_, LogData& cLogData_);

	void					drop(Trans::Transaction& cTrans_, bool bRecovery_ = false);

	static bool				alter(Trans::Transaction& cTrans_,
								  Partition& cPartition_,
								  const Statement::AlterPartitionStatement& cStatement_,
  								  Target& cPrevTarget_,
  								  Target& cPostTarget_,
								  LogData& cLogData_);
	static bool				alter(Trans::Transaction& cTrans_,
								  const LogData& cLogData_,
  								  Target& cPrevTarget_,
  								  Target& cPostTarget_);
												// ルール変更の準備をする

	SYD_SCHEMA_FUNCTION
	static Partition*		get(ID::Value id, Database* pDatabase_,
								Trans::Transaction& cTrans_);
	SYD_SCHEMA_FUNCTION
	static Partition*		get(const Name& cName_, Database* pDatabase_,
								Trans::Transaction& cTrans_);
												// ルールを表すクラスを得る

	const Target&			getTarget() const;
												// 対象を指定する情報を得る
	void					setTarget(const Target& cTarget_);
												// 対象を指定する情報を設定する

	Category::Value			getCategory() const; // カテゴリーを得る
	void					setCategory(Category::Value eCategory_);
												// カテゴリーを設定する

	static void				doBeforePersist(const Pointer& pPartition_,
											Status::Value eStatus_,
											bool bNeedToErase_,
											Trans::Transaction& cTrans_);
												// 永続化前に行う処理

	static void				doAfterPersist(const Pointer& pPartition_,
										   Status::Value eStatus_,
										   bool bNeedToErase_,
										   Trans::Transaction& cTrans_);
												// 永続化後に行う処理

	// システム表からの読み込み前に行う処理
	static void				doAfterLoad(const Pointer& pPartition_,
										Database& cDatabase_,
										Trans::Transaction& cTrans_);

//	Object::
//	Timestamp				getTimestamp() const; // タイムスタンプを得る
//	ID::Value				getID() const;		// オブジェクト ID を得る
//	ID::Value				getParentID() const;
//												// 親オブジェクトの
//												// オブジェクト ID を得る
	SYD_SCHEMA_FUNCTION
	const Name&				getName() const;	// オブジェクトの名前を得る
//	Category::Value			getCategory() const;
//												// オブジェクトの種別を得る

	void					reset(Database& cDatabase_);
	void					reset();
												// 下位オブジェクトを抹消する

	SYD_SCHEMA_FUNCTION
	Table*					getTable(Trans::Transaction& cTrans_) const;
												// ルールが対象とする表を得る
	SYD_SCHEMA_FUNCTION
	ID::Value				getTableID() const;	// ルールが対象とする表の
												// オブジェクトIDを得る
	SYD_SCHEMA_FUNCTION
	void					setTableID(ID::Value id_);
												// ルールが対象とする表の
												// オブジェクトIDを設定する

	SYD_SCHEMA_FUNCTION
	virtual void			serialize(ModArchive& archiver);
												// このクラスをシリアル化する
	SYD_SCHEMA_FUNCTION
	virtual int				getClassID() const;	// このクラスのクラス ID を得る

	// 再構成用のメソッド

	// 論理ログ出力用のメソッド
	void					makeLogData(LogData& cLogData_) const;
												// ログデータを作る
	// ログデータからさまざまな情報を取得する
	static ID::Value		getObjectID(const LogData& log_);
												// ログデータより PartitionID を取得する
	static ID::Value		getTableID(const LogData& log_);
												// ログデータより TableID を取得する
	static Name				getName(const LogData& cLogData_);
												// ログデータからルールが対象とする表名を得る
	static Name				getFunctionName(const LogData& cLogData_);
												// ログデータからルールの関数名を得る

	// 論理ログ出力、REDOのためのメソッド
	// pack、unpackの下請けとしても使用される

	virtual int				getMetaFieldNumber() const;
	virtual Meta::MemberType::Value
							getMetaMemberType(int iMemberID_) const;

	virtual Common::Data::Pointer packMetaField(int iMemberID_) const;
	virtual bool			unpackMetaField(const Common::Data* pData_, int iMemberID_);

protected:
private:
	// コンストラクター
	Partition(const Database& cDatabase_, const Table& cTable_,
			  const Statement::PartitionDefinition& cStatement_);
	Partition(const Database& cDatabase_, const LogData& cLogData_);

	void					setTarget(Trans::Transaction& cTrans_,
									  const Statement::PartitionDefinition& cStatement_,
									  Target& cTarget_);
	void					setTarget(Trans::Transaction& cTrans_,
									  const Statement::AlterPartitionStatement& cStatement_,
									  Target& cTarget_);

//	Object::
//	void					addTimestamp();		// タイムスタンプを進める

	void					destruct();			// デストラクター下位ルール

	// 以下のメンバーは、「ルール」表を検索して得られる

//	Object::
//	ID::Value				_id;				// オブジェクト ID
//	ID::Value				_parent;			// 親オブジェクトの
//												// オブジェクト ID
//	Name					_name;				// オブジェクトの名前
//	Category::Value			_category;			// オブジェクトの種別

	mutable Category::Value m_eCategory;
	mutable ID::Value m_iTableID;
	mutable Table* m_pTable;
	mutable Target m_cTarget;
};

_SYDNEY_SCHEMA_END
_SYDNEY_END

#endif	// __SYDNEY_SCHEMA_PARTITION_H

//
// Copyright (c) 2012, 2023 Ricoh Company, Ltd.
// All rights reserved.
//
