// -*-Mode: Java; tab-width: 4; c-basic-offset: 4;-*-
// vi:set ts=4 sw=4:
//
// TRMeisterDialect.java -- Dialect class for Hibernate 3.5
// 
// Copyright (c) 2006, 2007, 2010, 2011, 2023 Ricoh Company, Ltd.
// 
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// 
//     http://www.apache.org/licenses/LICENSE-2.0
// 
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// 

package jp.co.ricoh.doquedb.hibernate.dialect;

import java.sql.Types;

import org.hibernate.type.DoubleType;
import org.hibernate.type.IntegerType;
import org.hibernate.type.StringType;
import org.hibernate.internal.util.StringHelper;
import org.hibernate.dialect.Dialect;
import org.hibernate.dialect.function.SQLFunctionTemplate;
import org.hibernate.dialect.function.StandardSQLFunction;

/**
 * Dialect class for Hibernate3.6 of TRMeister
 *
 * @author Xie Xuansong
 */
public class TRMeisterDialect extends Dialect
{
	/**
	 * Constructor
	 */
	public TRMeisterDialect() {
		super();	// The constructor of Hibernate is executed.

		/* *
		   データ型
		 */
		registerColumnType(Types.BIT, "int");
		registerColumnType(Types.BOOLEAN, "int");
		registerColumnType(Types.BIGINT, "bigint");
		registerColumnType(Types.VARBINARY, "image");
		registerColumnType(Types.BINARY, "binary");
		registerColumnType(Types.TINYINT, "int");
		registerColumnType(Types.CHAR, "char(1)");
		registerColumnType(Types.SMALLINT, "int");
		registerColumnType(Types.INTEGER, "int");
		registerColumnType(Types.FLOAT, "float");
		registerColumnType(Types.NUMERIC, "decimal");
		registerColumnType(Types.DOUBLE, "float");
		registerColumnType(Types.DECIMAL, "decimal");
		registerColumnType(Types.VARCHAR, 65535, "varchar($l)");
		registerColumnType(Types.VARCHAR, 16777215, "ntext hint heap");

		registerColumnType(Types.DATE, "datetime");
		registerColumnType(Types.TIME, "datetime");
		registerColumnType(Types.TIMESTAMP, "datetime");
		registerColumnType(Types.BLOB, "blob");
		registerColumnType(Types.CLOB, "nclob");
//		registerColumnType(Types.ARRAY, "array");

		/**
		   関数
		 */

		registerFunction("length", new StandardSQLFunction("char_length", new IntegerType()));
		registerFunction("score", new StandardSQLFunction("score", new DoubleType()));

		registerFunction("substring", new SQLFunctionTemplate(new StringType(), "substring(?1 from ?2 for ?3)"));
		registerFunction("overlay", new SQLFunctionTemplate(new StringType(), "overlay(?1 placing ?2 from ?3)"));

		/**
		   キーワード
		 */
		registerKeyword("AREA");
		registerKeyword("AVERAGE");
		registerKeyword("BACKUP");
		registerKeyword("BITMAP");
		registerKeyword("CALCULATOR");
		registerKeyword("CATEGORY");
		registerKeyword("CHECKPOINT");
		registerKeyword("CLUSTERED");
		registerKeyword("COMBINER");
		registerKeyword("CONST");
		registerKeyword("CONSTANT");
		registerKeyword("CORRECT");
		registerKeyword("DATABASE");
		registerKeyword("DATETIME");
		registerKeyword("DF");
		registerKeyword("DISCARD");
		registerKeyword("EXACTWORD");
		registerKeyword("EXPAND");
		registerKeyword("EXTRACTOR");
		registerKeyword("FILE");
		registerKeyword("FREETEXT");
		registerKeyword("FULLTEXT");
		registerKeyword("HEAD");
		registerKeyword("HEAP");
		registerKeyword("HINT");
		registerKeyword("IMAGE");
		registerKeyword("INDEX");
		registerKeyword("LIMIT");
		registerKeyword("LOGICALLOG");
		registerKeyword("MASTER");
		registerKeyword("MODIFY");
		registerKeyword("MOUNT");
		registerKeyword("NONCLUSTERED");
		registerKeyword("NONTRUNCATE");
		registerKeyword("NTEXT");
		registerKeyword("NVARCHAR");
		registerKeyword("OFFLINE");
		registerKeyword("OFFSET");
		registerKeyword("ONLINE");
		registerKeyword("PHYSICALLOG");
		registerKeyword("RECOVERY");
		registerKeyword("RENAME");
		registerKeyword("SCORE");
		registerKeyword("SECTIONIZED");
		registerKeyword("SIMPLEWORD");
		registerKeyword("SNAPSHOT");
		registerKeyword("SYNC");
		registerKeyword("TAIL");
		registerKeyword("TEMPORARY");
		registerKeyword("UNIQUEIDENTIFIER");
		registerKeyword("UPDATABLE");
		registerKeyword("UNMOUNT");
		registerKeyword("VERBOSE");
		registerKeyword("VERIFY");
		registerKeyword("WEIGHT");
		registerKeyword("WORD");
		registerKeyword("WORDLIST");
	}

	/**
	 * Does this dialect support identity column key generation?
	 *
	 * @return boolean
	 */
	public boolean supportsIdentityColumns() {
		return true;
	}

	/*
	 *
	*/
	protected String getIdentityColumnString() {
		return " generated by default as identity ";
	}

	/**
	 * The keyword used to specify an identity column, if identity
	 * column key generation is supported.
	 *
	 * @param type the SQL column type, as defined by <tt>java.sql.Types</tt>
	 * @throws MappingException if no native key generation
	 */
	public String getIdentityColumnString(int type) {
		if(type == 0) return getIdentityColumnString();
		else return " generated always as identity ";
	}

	/**
	 * The keyword used to insert a row without specifying any column values.
	 * This is not possible on some databases.
	 */
	public String getNoColumnsInsertString() {
		return " default values ";
	}

	/*
	 * not support PreparedStatement.execute();
	 * should consider about it
	 * Now return true for testing
	*/
	public boolean supportsInsertSelectIdentity() {
		return true;
	}

	/*
	//here is temporary.
	public String getIdentitySelectString() {
		return "select last_insert_id()";
	}
	*/

	/**
	 * Retrieves the <tt>FOR UPDATE</tt> syntax specific to this dialect.
	 *
	 * @return The appropriate <tt>FOR UPDATE</tt> clause string.
	 *
	 * Now TRMeister not supported
	 */
	public String getForUpdateString() {
		return " ";
	}

	/**
	 * The syntax used to add a column to a table (optional).
	 */
	public String getAddColumnString() {
		return " add column ";
	}

	/**
	 * Do we need to qualify index names with the schema name?
	 *
	 * @return boolean
	 */
	public boolean qualifyIndexName() {
		return false;
	}

    //can create but not alter.
    /**
     * Does this dialect support adding Unique constraints via create and alter table ?
     * @return boolean
     */
	public boolean supportsUniqueConstraintInCreateAlterTable() {
	    return false;
	}

	/**
	 * Do we need to drop constraints before dropping tables in this dialect?
	 *
	 * @return boolean
	*/
	public boolean dropConstraints() {
		return false;		//not support drop constraints.
	}

	/**
	 * The syntax used to add a primary key constraint to a table.
	 * @return String
	 */
	public String getAddPrimaryKeyConstraintString(String constraintName) {
		return " add primary key " + (constraintName);
	}

	/**
	 * The syntax used to add a foreign key constraint to a table.
	 *
	 * @param referencesPrimaryKey if false, constraint should be
	 * explicit about which column names the constraint refers to
	 *
	 * @return String
	 */
	public String getAddForeignKeyConstraintString(
			String constraintName,
			String[] foreignKey,
			String referencedTable,
			String[] primaryKey,
			boolean referencesPrimaryKey
	) {
		StringBuilder res = new StringBuilder(30);

		res.append(" add foreign key (")
		   .append(StringHelper.join(", ", foreignKey))
		   .append(") references ")
		   .append(referencedTable)
		   .append(" (")
		   .append(StringHelper.join(", ", primaryKey))
		   .append(')');

		return res.toString();
	}

	/*
	 * Now TRMeister not supported, the swith is controled by dropConstraints
	*/
	public String getDropForeignKeyString() {
		return " drop foreign key ";
	}

	/*
	 * Now TRMeister not supported
	*/
	public boolean supportsCascadeDelete() {
		return false;
	}

	/*
	 * Now TRMeister not supported
	*/
	public boolean hasSelfReferentialForeignKeyBug() {
		return true;
	}

	/**
	 * The keyword used to specify a nullable column.
	 *
	 * @return String
	 */
	//public String getNullColumnString() {
	//	return "";
	//}

	/**
	 * Does this <tt>Dialect</tt> have some kind of <tt>LIMIT</tt> syntax?
	 */
	public boolean supportsLimit() {
		return true;
	}

	/**
	 * Add a <tt>LIMIT</tt> clause to the given SQL <tt>SELECT</tt>
	 * @return the modified SQL
	*/

	public String getLimitString(String querySelect, boolean hasOffset) {
		return new StringBuilder(querySelect.length() + 20)
			.append(querySelect)
			.append(hasOffset ? " limit ? offset ? + 1" : " limit ?")
			.toString();
	}

	/**
	 * @return the modified SQL
	*/
	public String getLimitString(String querySelect, int offset, int limit) {
		StringBuilder buf = new StringBuilder(querySelect.length() + 20)
			.append(querySelect);
		if (offset>0) {
			buf.append(" limit ")
				.append(limit)
				.append(" offset ")
				.append(offset + 1);
		}
		else {
			buf.append(" limit ")
				.append(limit);
		}
		return buf.toString();
	}

	/**
	 * The opening quote for a quoted identifier
	 */
	public char openQuote() {
		return '"';
	}

	/**
	 * The closing quote for a quoted identifier
	 */
	public char closeQuote() {
		return '"';
	}

	/**
	 * Support after, not before
	 */
	public boolean supportsIfExistsAfterTableName() {
		return true;
	}

	/**
	 * Does this dialect support column-level check constraints?
	 */
	public boolean supportsColumnCheck() {
		return false;
	}

	/**
	 * Does this dialect support table-level check constraints?
	 */
	public boolean supportsTableCheck() {
		return false;
	}

	/**
	 * "for update " is used for line lock just as updating.
	 */
	public boolean supportsOuterJoinForUpdate() {
		return false;
	}

	/**
	 * Does this dialect support the <tt>UNIQUE</tt> column syntax?
	 *
	 * @return boolean
	 * I think we should support in future
	 */
	public boolean supportsUnique() {
		return false;
	}

	public boolean supportsNotNullUnique() {
		return false;
	}

	//For now, TRMeister can not use input stream for blob.
	public boolean useInputStreamToInsertBlob() {
		return false;
	}

	//It returns false
	public boolean supportsCommentOn() {
		return false;
	}

	public String getTableComment(String comment) {
		return " --" + comment + "\n";
	}

	public String getColumnComment(String comment) {
		return " --" + comment + "\n";
	}

	public boolean supportsTemporaryTables() {
		return true;
	}

	public String generateTemporaryTableName(String baseTableName) {
		return " #" + baseTableName;
	}

	public String getCreateTemporaryTableString() {
		return "create table ";
	}

 	// is temporary
	public int getMaxAliasLength() {
		return 100;
	}
}

//
// Copyright (c) 2006, 2007, 2010, 2011, 2023 Ricoh Company, Ltd.
// All rights reserved.
//
