/*
 * Copyright (c) 2015-2020, www.dibo.ltd (service@dibo.ltd).
 * <p>
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * <p>
 * https://www.apache.org/licenses/LICENSE-2.0
 * <p>
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.diboot.iam.init;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.diboot.core.entity.Dictionary;
import com.diboot.core.entity.I18nConfig;
import com.diboot.core.exception.BusinessException;
import com.diboot.core.mapper.I18nConfigMapper;
import com.diboot.core.service.DictionaryService;
import com.diboot.core.util.ContextHolder;
import com.diboot.core.util.JSON;
import com.diboot.core.util.SqlFileInitializer;
import com.diboot.core.vo.DictionaryVO;
import com.diboot.iam.config.Cons;
import com.diboot.iam.entity.*;
import com.diboot.iam.service.*;
import com.diboot.iam.vo.IamResourceListVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * IAM组件相关的初始化
 *
 * @author mazc@dibo.ltd
 * @version v2.0
 * @date 2019/12/23
 */
@Order(912)
@Slf4j
@Component
@ConditionalOnProperty(prefix = "diboot", name = "init-sql", havingValue = "true")
public class IamPluginInitializer implements ApplicationRunner {

    @Override
    public void run(ApplicationArguments args) throws Exception {
        // 检查数据库表是否已存在
        String initDetectSql = "SELECT id FROM dbt_iam_role WHERE id='0'";
        if (!SqlFileInitializer.checkSqlExecutable(initDetectSql)) {
            log.info("diboot-IAM 初始化SQL ...");
            // 执行初始化SQL
            SqlFileInitializer.initBootstrapSql(this.getClass(), "iam");
            // 插入相关数据：Dict，Role等
            insertInitData();
            log.info("diboot-IAM 初始化SQL完成.");
        }
    }

    /**
     * 插入初始化数据
     */
    private synchronized void insertInitData() {
        // 插入iam组件所需的数据字典
        DictionaryService dictionaryService = ContextHolder.getBean(DictionaryService.class);
        if (dictionaryService != null && !dictionaryService.exists(Dictionary::getType, "AUTH_TYPE")) {
            String[] DICT_INIT_DATA = {
                    "{\"type\":\"AUTH_TYPE\", \"itemName\":\"登录认证方式\", \"description\":\"IAM用户登录认证方式\", \"children\":[{\"itemName\":\"用户名密码\", \"itemNameI18n\":\"Dictionary.AUTH_TYPE.PWD\", \"itemValue\":\"PWD\", \"sortId\":1},{\"itemName\":\"单点登录\", \"itemNameI18n\":\"Dictionary.AUTH_TYPE.SSO\", \"itemValue\":\"SSO\", \"sortId\":2},{\"itemName\":\"公众号\", \"itemNameI18n\":\"Dictionary.AUTH_TYPE.WX_MP\", \"itemValue\":\"WX_MP\", \"sortId\":3},{\"itemName\":\"企业微信\", \"itemNameI18n\":\"Dictionary.AUTH_TYPE.WX_CP\", \"itemValue\":\"WX_CP\", \"sortId\":4},{\"itemName\":\"其他\", \"itemNameI18n\":\"Dictionary.AUTH_TYPE.Other\", \"itemValue\":\"OTHER\", \"sortId\":5}]}",
                    "{\"type\":\"ACCOUNT_STATUS\",\"itemName\":\"账号状态\", \"description\":\"IAM登录账号状态\",\"children\":[{\"itemName\":\"有效\",\"itemNameI18n\":\"Dictionary.ACCOUNT_STATUS.A\", \"itemValue\":\"A\",\"extension\":{\"color\":\"#2ECC71\"},\"sortId\":1},{\"itemName\":\"无效\",\"itemNameI18n\":\"Dictionary.ACCOUNT_STATUS.I\", \"itemValue\":\"I\",\"sortId\":2},{\"itemName\":\"锁定\",\"itemNameI18n\":\"Dictionary.ACCOUNT_STATUS.L\", \"itemValue\":\"L\",\"extension\":{\"color\":\"#FF6F00\"},\"sortId\":3}]}",
                    "{\"type\":\"USER_STATUS\",\"itemName\":\"用户状态\", \"description\":\"IAM用户状态\",\"isEditable\":true,\"children\":[{\"itemName\":\"在职\",\"itemNameI18n\":\"Dictionary.USER_STATUS.A\", \"itemValue\":\"A\",\"extension\":{\"color\":\"#2ECC71\"},\"sortId\":1},{\"itemName\":\"离职\",\"itemNameI18n\":\"Dictionary.USER_STATUS.I\", \"itemValue\":\"I\",\"sortId\":2}]}",
                    "{\"itemName\":\"用户性别\", \"type\":\"GENDER\",\"description\":\"用户性别数据字典\",\"children\":[{\"itemValue\":\"F\",\"itemName\":\"女\",\"itemNameI18n\":\"Dictionary.GENDER.F\",\"extension\":{\"color\":\"#FD8BB8\"}},{\"itemValue\":\"M\",\"itemName\":\"男\",\"itemNameI18n\":\"Dictionary.GENDER.M\",\"extension\":{\"color\":\"#55B0EE\"}}]}",
                    "{\"type\":\"RESOURCE_TYPE\", \"itemName\":\"资源类型\", \"description\":\"IAM资源类型\", \"children\":[{\"itemName\":\"菜单\", \"itemNameI18n\":\"Dictionary.RESOURCE_TYPE.MENU\", \"itemValue\":\"MENU\", \"sortId\":1},{\"itemName\":\"按钮/操作\", \"itemNameI18n\":\"Dictionary.RESOURCE_TYPE.OPERATION\", \"itemValue\":\"OPERATION\", \"sortId\":2}]}",
                    "{\"itemName\":\"前端按钮/权限编码\", \"type\":\"RESOURCE_CODE\",\"description\":\"前端按钮/权限编码 常用选项\",\"children\":[{\"sortId\":1,\"itemName\":\"详情\",\"itemNameI18n\":\"Dictionary.RESOURCE_CODE.detail\", \"itemValue\":\"detail\"},{\"sortId\":2,\"itemName\":\"新建\",\"itemNameI18n\":\"Dictionary.RESOURCE_CODE.create\", \"itemValue\":\"create\"},{\"sortId\":3,\"itemName\":\"更新\",\"itemNameI18n\":\"Dictionary.RESOURCE_CODE.update\", \"itemValue\":\"update\"},{\"sortId\":4,\"itemName\":\"删除\",\"itemNameI18n\":\"Dictionary.RESOURCE_CODE.delete\", \"itemValue\":\"delete\"},{\"sortId\":5,\"itemName\":\"导出\",\"itemNameI18n\":\"Dictionary.RESOURCE_CODE.export\", \"itemValue\":\"export\"},{\"sortId\":6,\"itemName\":\"导入\",\"itemNameI18n\":\"Dictionary.RESOURCE_CODE.import\", \"itemValue\":\"import\"}]}",
                    "{\"type\":\"ORG_TYPE\", \"itemName\":\"组织类型\", \"description\":\"组织节点类型\", \"isEditable\":false, \"children\":[{\"itemName\":\"部门\", \"itemNameI18n\":\"Dictionary.ORG_TYPE.DEPT\", \"itemValue\":\"DEPT\", \"sortId\":1},{\"itemName\":\"公司\", \"itemNameI18n\":\"Dictionary.ORG_TYPE.COMP\", \"itemValue\":\"COMP\", \"sortId\":2}]}",
                    "{\"type\":\"DATA_PERMISSION_TYPE\", \"itemName\":\"IAM数据权限类型\", \"description\":\"IAM数据权限类型定义\", \"isEditable\":true, \"children\":[{\"itemName\":\"本人\", \"itemNameI18n\":\"Dictionary.DATA_PERMISSION_TYPE.SELF\", \"itemValue\":\"SELF\", \"sortId\":1},{\"itemName\":\"本人及下属\", \"itemNameI18n\":\"Dictionary.DATA_PERMISSION_TYPE.SELF_AND_SUB\", \"itemValue\":\"SELF_AND_SUB\", \"sortId\":2},{\"itemName\":\"本部门\", \"itemNameI18n\":\"Dictionary.DATA_PERMISSION_TYPE.DEPT\", \"itemValue\":\"DEPT\", \"sortId\":3},{\"itemName\":\"本部门及下属部门\", \"itemNameI18n\":\"Dictionary.DATA_PERMISSION_TYPE.DEPT_AND_SUB\", \"itemValue\":\"DEPT_AND_SUB\", \"sortId\":4},{\"itemName\":\"全部\", \"itemNameI18n\":\"Dictionary.DATA_PERMISSION_TYPE.ALL\", \"itemValue\":\"ALL\", \"sortId\":5}]}",
                    "{\"type\":\"POSITION_GRADE\", \"itemName\":\"职级定义\", \"description\":\"职务级别定义\", \"isEditable\":true, \"children\":[{\"itemName\":\"初级\", \"itemNameI18n\":\"Dictionary.POSITION_GRADE.E1\", \"itemValue\":\"E1\", \"sortId\":1},{\"itemName\":\"中级\", \"itemNameI18n\":\"Dictionary.POSITION_GRADE.E2\", \"itemValue\":\"E2\", \"sortId\":2},{\"itemName\":\"高级\", \"itemNameI18n\":\"Dictionary.POSITION_GRADE.E3\", \"itemValue\":\"E3\", \"sortId\":3},{\"itemName\":\"专家\", \"itemNameI18n\":\"Dictionary.POSITION_GRADE.E4\", \"itemValue\":\"E4\", \"sortId\":4}]}"
            };
            // 插入数据字典
            for (String dictJson : DICT_INIT_DATA) {
                DictionaryVO dictVo = JSON.toJavaObject(dictJson, DictionaryVO.class);
                dictVo.setParentId(Cons.ID_PREVENT_NULL);
                dictionaryService.createDictAndChildren(dictVo);
            }
        }

        // 插入iam组件所需的初始权限数据
        IamResourceService resourcePermissionService = ContextHolder.getBean(IamResourceService.class);
        if (resourcePermissionService != null && !resourcePermissionService.exists(IamResource::getResourceCode, "system")) {
            String[] RESOURCE_PERMISSION_DATA = {
                    "{\"displayType\":\"CATALOGUE\",\"displayName\":\"组织人员\",\"displayNameI18n\":\"Resource.OrgStructure\",\"routePath\":\"org-structure\",\"resourceCode\":\"OrgStructure\",\"meta\":\"{\\\"icon\\\":\\\"Element:UserFilled\\\"}\",\"sortId\":\"90\",\"children\":[{\"displayType\":\"MENU\",\"displayName\":\"人员管理\",\"displayNameI18n\":\"Resource.User\",\"routePath\":\"user\",\"resourceCode\":\"User\",\"permissionCode\":\"IamOrg:read,IamUser:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:User\\\",\\\"componentPath\\\":\\\"@/views/org-structure/user/index.vue\\\"}\",\"sortId\":\"1\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"新建\",\"resourceCode\":\"create\",\"displayNameI18n\":\"Operation.create\",\"permissionCode\":\"IamUser:write\",\"sortId\":\"40\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"更新\",\"resourceCode\":\"update\",\"displayNameI18n\":\"Operation.update\",\"permissionCode\":\"IamUser:write,IamUser:read\",\"sortId\":\"39\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"删除\",\"resourceCode\":\"delete\",\"displayNameI18n\":\"Operation.delete\",\"permissionCode\":\"IamUser:write\",\"sortId\":\"38\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"详情\",\"resourceCode\":\"detail\",\"displayNameI18n\":\"Operation.detail\",\"permissionCode\":\"IamUser:read\",\"sortId\":\"37\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"导入\",\"resourceCode\":\"import\",\"displayNameI18n\":\"Operation.import\",\"permissionCode\":\"IamUserExcel:import\",\"sortId\":\"36\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"导出\",\"resourceCode\":\"export\",\"displayNameI18n\":\"Operation.export\",\"permissionCode\":\"IamUserExcel:export\",\"sortId\":\"35\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"人员岗位设置\",\"resourceCode\":\"position\",\"displayNameI18n\":\"Resource.User.PositionSetting\",\"permissionCode\":\"IamPosition:write,IamPosition:read\",\"sortId\":\"34\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"添加岗位\",\"resourceCode\":\"addPosition\",\"displayNameI18n\":\"Resource.User.AddPosition\",\"permissionCode\":\"IamPosition:write,IamPosition:read\",\"sortId\":\"33\"}]},{\"displayType\":\"MENU\",\"displayName\":\"组织部门\",\"displayNameI18n\":\"Resource.Org\",\"routePath\":\"org\",\"resourceCode\":\"Org\",\"permissionCode\":\"IamOrg:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:Folder\\\",\\\"componentPath\\\":\\\"@/views/org-structure/org/index.vue\\\"}\",\"sortId\":\"2\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"排序\",\"resourceCode\":\"sort\",\"displayNameI18n\":\"Operation.sort\",\"permissionCode\":\"IamOrg:write\",\"sortId\":\"106\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"删除\",\"resourceCode\":\"delete\",\"displayNameI18n\":\"Operation.delete\",\"permissionCode\":\"IamOrg:write\",\"sortId\":\"105\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"更新\",\"resourceCode\":\"update\",\"displayNameI18n\":\"Operation.update\",\"permissionCode\":\"IamOrg:write,IamOrg:read\",\"sortId\":\"104\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"新建\",\"resourceCode\":\"create\",\"displayNameI18n\":\"Operation.create\",\"permissionCode\":\"IamOrg:write\",\"sortId\":\"103\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"详情\",\"resourceCode\":\"detail\",\"displayNameI18n\":\"Operation.detail\",\"permissionCode\":\"IamOrg:read\",\"sortId\":\"102\"}]},{\"displayType\":\"MENU\",\"displayName\":\"岗位管理\",\"displayNameI18n\":\"Resource.Position\",\"routePath\":\"position\",\"resourceCode\":\"Position\",\"permissionCode\":\"IamPosition:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:Postcard\\\",\\\"componentPath\\\":\\\"@/views/org-structure/position/List.vue\\\"}\",\"sortId\":\"3\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"删除\",\"resourceCode\":\"delete\",\"displayNameI18n\":\"Operation.delete\",\"permissionCode\":\"IamPosition:write\",\"sortId\":\"112\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"详情\",\"resourceCode\":\"detail\",\"displayNameI18n\":\"Operation.detail\",\"permissionCode\":\"IamPosition:read\",\"sortId\":\"111\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"更新\",\"resourceCode\":\"update\",\"displayNameI18n\":\"Operation.update\",\"permissionCode\":\"IamPosition:write,IamPosition:read\",\"sortId\":\"110\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"新建\",\"resourceCode\":\"create\",\"displayNameI18n\":\"Operation.create\",\"permissionCode\":\"IamPosition:write\",\"sortId\":\"108\"}]}]}",
                    "{\"displayType\":\"CATALOGUE\",\"displayName\":\"系统管理\",\"displayNameI18n\":\"Resource.System\",\"routePath\":\"system\",\"resourceCode\":\"System\",\"meta\":\"{\\\"icon\\\":\\\"Element:Tools\\\"}\",\"sortId\":\"95\",\"children\":[{\"displayType\":\"MENU\",\"displayName\":\"数据字典\",\"displayNameI18n\":\"Resource.Dictionary\",\"routePath\":\"dictionary\",\"resourceCode\":\"Dictionary\",\"permissionCode\":\"Dictionary:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:Collection\\\",\\\"componentPath\\\":\\\"@/views/system/dictionary/List.vue\\\"}\",\"sortId\":\"10\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"详情\",\"resourceCode\":\"detail\",\"displayNameI18n\":\"Operation.detail\",\"permissionCode\":\"Dictionary:read\",\"sortId\":\"6\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"新建\",\"resourceCode\":\"create\",\"displayNameI18n\":\"Operation.create\",\"permissionCode\":\"Dictionary:write\",\"sortId\":\"5\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"更新\",\"resourceCode\":\"update\",\"displayNameI18n\":\"Operation.update\",\"permissionCode\":\"Dictionary:write,Dictionary:read\",\"sortId\":\"4\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"删除\",\"resourceCode\":\"delete\",\"displayNameI18n\":\"Operation.delete\",\"permissionCode\":\"Dictionary:write\",\"sortId\":\"3\"}]},{\"displayType\":\"MENU\",\"displayName\":\"菜单资源\",\"displayNameI18n\":\"Resource.Resource\",\"routePath\":\"resource\",\"resourceCode\":\"Resource\",\"permissionCode\":\"IamResource:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:Menu\\\",\\\"componentPath\\\":\\\"@/views/system/resource/index.vue\\\"}\",\"sortId\":\"20\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"详情\",\"resourceCode\":\"detail\",\"displayNameI18n\":\"Operation.detail\",\"permissionCode\":\"IamResource:read\",\"sortId\":\"23\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"新建\",\"resourceCode\":\"create\",\"displayNameI18n\":\"Operation.create\",\"permissionCode\":\"IamResource:write\",\"sortId\":\"21\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"更新\",\"resourceCode\":\"update\",\"displayNameI18n\":\"Operation.update\",\"permissionCode\":\"IamResource:write,IamResource:read\",\"sortId\":\"20\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"删除\",\"resourceCode\":\"delete\",\"displayNameI18n\":\"Operation.delete\",\"permissionCode\":\"IamResource:write\",\"sortId\":\"19\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"排序\",\"resourceCode\":\"sort\",\"displayNameI18n\":\"Operation.sort\",\"permissionCode\":\"IamResource:write\",\"sortId\":\"18\"}]},{\"displayType\":\"MENU\",\"displayName\":\"角色权限\",\"displayNameI18n\":\"Resource.Role\",\"routePath\":\"role\",\"resourceCode\":\"Role\",\"permissionCode\":\"IamRole:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:Avatar\\\",\\\"componentPath\\\":\\\"@/views/system/role/List.vue\\\"}\",\"sortId\":\"30\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"详情\",\"resourceCode\":\"detail\",\"displayNameI18n\":\"Operation.detail\",\"permissionCode\":\"IamRole:read\",\"sortId\":\"16\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"新建\",\"resourceCode\":\"create\",\"displayNameI18n\":\"Operation.create\",\"permissionCode\":\"IamRole:write\",\"sortId\":\"15\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"更新\",\"resourceCode\":\"update\",\"displayNameI18n\":\"Operation.update\",\"permissionCode\":\"IamRole:write,IamRole:read,IamResource:read\",\"sortId\":\"14\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"删除\",\"resourceCode\":\"delete\",\"displayNameI18n\":\"Operation.delete\",\"permissionCode\":\"IamRole:write\",\"sortId\":\"13\"}]},{\"displayType\":\"MENU\",\"displayName\":\"定时任务\",\"displayNameI18n\":\"Resource.ScheduledJob\",\"routePath\":\"schedule-job\",\"resourceCode\":\"ScheduleJob\",\"permissionCode\":\"ScheduleJob:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:AlarmClock\\\",\\\"componentPath\\\":\\\"@/views/system/schedule-job/List.vue\\\"}\",\"sortId\":\"40\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"删除\",\"resourceCode\":\"delete\",\"displayNameI18n\":\"Operation.delete\",\"permissionCode\":\"ScheduleJob:write\",\"sortId\":\"7\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"更新\",\"resourceCode\":\"update\",\"displayNameI18n\":\"Operation.update\",\"permissionCode\":\"ScheduleJob:write,ScheduleJob:read\",\"sortId\":\"6\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"新建\",\"resourceCode\":\"create\",\"displayNameI18n\":\"Operation.create\",\"permissionCode\":\"ScheduleJob:write\",\"sortId\":\"5\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"详情\",\"resourceCode\":\"detail\",\"displayNameI18n\":\"Operation.detail\",\"permissionCode\":\"ScheduleJob:read\",\"sortId\":\"4\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"运行一次\",\"resourceCode\":\"executeOnce\",\"displayNameI18n\":\"Resource.ScheduledJob.ExecuteOnce\",\"permissionCode\":\"ScheduleJob:write\",\"sortId\":\"3\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"日志记录\",\"resourceCode\":\"logList\",\"displayNameI18n\":\"Resource.ScheduledJob.LogList\",\"permissionCode\":\"ScheduleJob:read\",\"sortId\":\"2\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"日志删除\",\"resourceCode\":\"logDelete\",\"displayNameI18n\":\"Resource.ScheduledJob.LogDelete\",\"permissionCode\":\"ScheduleJob:write\",\"sortId\":\"1\"}]},{\"displayType\":\"MENU\",\"displayName\":\"消息模板\",\"displayNameI18n\":\"Resource.MessageTemplate\",\"routePath\":\"message-template\",\"resourceCode\":\"MessageTemplate\",\"permissionCode\":\"MessageTemplate:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:ChatLineSquare\\\",\\\"componentPath\\\":\\\"@/views/system/message-template/List.vue\\\"}\",\"sortId\":\"50\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"详情\",\"resourceCode\":\"detail\",\"displayNameI18n\":\"Operation.detail\",\"permissionCode\":\"MessageTemplate:read\",\"sortId\":\"16\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"新建\",\"resourceCode\":\"create\",\"displayNameI18n\":\"Operation.create\",\"permissionCode\":\"MessageTemplate:write\",\"sortId\":\"15\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"更新\",\"resourceCode\":\"update\",\"displayNameI18n\":\"Operation.update\",\"permissionCode\":\"MessageTemplate:write,MessageTemplate:read\",\"sortId\":\"14\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"删除\",\"resourceCode\":\"delete\",\"displayNameI18n\":\"Operation.delete\",\"permissionCode\":\"MessageTemplate:write\",\"sortId\":\"13\"}]},{\"displayType\":\"MENU\",\"displayName\":\"消息记录\",\"displayNameI18n\":\"Resource.Message\",\"routePath\":\"message\",\"resourceCode\":\"Message\",\"permissionCode\":\"Message:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:ChatDotRound\\\",\\\"componentPath\\\":\\\"@/views/system/message/List.vue\\\"}\",\"sortId\":\"60\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"详情\",\"resourceCode\":\"detail\",\"displayNameI18n\":\"Operation.detail\",\"permissionCode\":\"Message:read\",\"sortId\":\"16\"}]},{\"displayType\":\"MENU\",\"displayName\":\"文件记录\",\"displayNameI18n\":\"Resource.FileRecord\",\"routePath\":\"file-record\",\"resourceCode\":\"FileRecord\",\"permissionCode\":\"FileRecord:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:FolderOpened\\\",\\\"componentPath\\\":\\\"@/views/system/file-record/List.vue\\\"}\",\"sortId\":\"70\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"详情\",\"resourceCode\":\"detail\",\"displayNameI18n\":\"Operation.detail\",\"permissionCode\":\"FileRecord:read\",\"sortId\":\"16\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"更新\",\"resourceCode\":\"update\",\"displayNameI18n\":\"Operation.update\",\"permissionCode\":\"FileRecord:write,FileRecord:read\",\"sortId\":\"14\"}]},{\"displayType\":\"MENU\",\"displayName\":\"国际化管理\",\"displayNameI18n\":\"Resource.I18n\",\"routePath\":\"i18n-config\",\"resourceCode\":\"I18nConfig\",\"permissionCode\":\"\",\"meta\":\"{\\\"icon\\\":\\\"Local:Language\\\",\\\"componentPath\\\":\\\"@/views/system/i18n-config/index.vue\\\"}\",\"sortId\":\"85\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"新建\",\"resourceCode\":\"create\",\"displayNameI18n\":\"Operation.create\",\"permissionCode\":\"I18nConfig:write\",\"sortId\":\"1\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"更新\",\"resourceCode\":\"update\",\"displayNameI18n\":\"Operation.update\",\"permissionCode\":\"I18nConfig:write\",\"sortId\":\"2\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"删除\",\"resourceCode\":\"delete\",\"displayNameI18n\":\"Operation.delete\",\"permissionCode\":\"I18nConfig:write\",\"sortId\":\"3\"}]},{\"displayType\":\"MENU\",\"displayName\":\"系统配置\",\"displayNameI18n\":\"Resource.Configuration\",\"routePath\":\"config\",\"resourceCode\":\"SystemConfig\",\"permissionCode\":\"SystemConfig:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:Setting\\\",\\\"componentPath\\\":\\\"@/views/system/config/index.vue\\\"}\",\"sortId\":\"80\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"新建\",\"resourceCode\":\"create\",\"displayNameI18n\":\"Operation.create\",\"permissionCode\":\"SystemConfig:write\",\"sortId\":\"1\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"更新\",\"resourceCode\":\"update\",\"displayNameI18n\":\"Operation.update\",\"permissionCode\":\"SystemConfig:write\",\"sortId\":\"2\"},{\"displayType\":\"PERMISSION\",\"displayName\":\"删除\",\"resourceCode\":\"delete\",\"displayNameI18n\":\"Operation.delete\",\"permissionCode\":\"SystemConfig:write\",\"sortId\":\"3\"}]},{\"displayType\":\"MENU\",\"displayName\":\"操作日志\",\"displayNameI18n\":\"Resource.OperationLog\",\"routePath\":\"operation-log\",\"resourceCode\":\"OperationLog\",\"permissionCode\":\"IamOperationLog:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:Pointer\\\",\\\"componentPath\\\":\\\"@/views/system/operation-log/List.vue\\\"}\",\"sortId\":\"90\",\"children\":[{\"displayType\":\"PERMISSION\",\"displayName\":\"详情\",\"resourceCode\":\"detail\",\"displayNameI18n\":\"Operation.detail\",\"permissionCode\":\"IamOperationLog:read\",\"sortId\":\"1\"}]},{\"displayType\":\"MENU\",\"displayName\":\"登录日志\",\"displayNameI18n\":\"Resource.LoginLog\",\"routePath\":\"login-trace\",\"resourceCode\":\"LoginTrace\",\"permissionCode\":\"IamLoginTrace:read\",\"meta\":\"{\\\"icon\\\":\\\"Element:Finished\\\",\\\"componentPath\\\":\\\"@/views/system/login-trace/List.vue\\\"}\",\"sortId\":\"100\",\"children\":[]}]}"
            };
            // 插入多层级资源权限初始数据
            try {
                for (String resourcePermissionJson : RESOURCE_PERMISSION_DATA) {
                    IamResourceListVO permissionListVO = JSON.toJavaObject(resourcePermissionJson, IamResourceListVO.class);
                    resourcePermissionService.deepCreateResourceAndChildren(permissionListVO);
                }
            } catch (BusinessException e) {
                log.error("初始化资源权限数据出错: {}，请手动配置前端资源初始的权限数据", e.getMessage());
            }
        }

        I18nConfigMapper i18nConfigMapper = ContextHolder.getBean(I18nConfigMapper.class);
        // 插入国际化配置
        if (!i18nConfigMapper.exists(Wrappers.<I18nConfig>lambdaQuery().select(I18nConfig::getType).eq(I18nConfig::getType, "SYSTEM"))) {
            String i18nData = "[{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.I18N_TYPE.System\",\"content\":\"系统\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.I18N_TYPE.System\",\"content\":\"System\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.I18N_TYPE.Custom\",\"content\":\"自定义\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.I18N_TYPE.Custom\",\"content\":\"Custom\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.AUTH_TYPE.PWD\",\"content\":\"用户名密码\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.AUTH_TYPE.PWD\",\"content\":\"Password\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.AUTH_TYPE.SSO\",\"content\":\"单点登录\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.AUTH_TYPE.SSO\",\"content\":\"Single Sign-On\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.AUTH_TYPE.WX_MP\",\"content\":\"公众号\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.AUTH_TYPE.WX_MP\",\"content\":\"WeChat\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.AUTH_TYPE.WX_CP\",\"content\":\"企业微信\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.AUTH_TYPE.WX_CP\",\"content\":\"Enterprise WeChat\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.AUTH_TYPE.Other\",\"content\":\"其他\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.AUTH_TYPE.Other\",\"content\":\"Other\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.ACCOUNT_STATUS.A\",\"content\":\"有效\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.ACCOUNT_STATUS.A\",\"content\":\"Active\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.ACCOUNT_STATUS.I\",\"content\":\"无效\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.ACCOUNT_STATUS.I\",\"content\":\"Invalid\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.ACCOUNT_STATUS.L\",\"content\":\"锁定\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.ACCOUNT_STATUS.L\",\"content\":\"Locked\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.USER_STATUS.A\",\"content\":\"在职\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.USER_STATUS.A\",\"content\":\"Active\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.MEMBER_STATUS.A\",\"content\":\"有效\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.MEMBER_STATUS.A\",\"content\":\"Active\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.TENANT_STATUS.A\",\"content\":\"有效\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.TENANT_STATUS.A\",\"content\":\"Active\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.MEMBER_STATUS.I\",\"content\":\"无效\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.USER_STATUS.I\",\"content\":\"离职\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.TENANT_STATUS.I\",\"content\":\"无效\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.MEMBER_STATUS.I\",\"content\":\"Invalid\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.USER_STATUS.I\",\"content\":\"Invalid\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.TENANT_STATUS.I\",\"content\":\"Invalid\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.GENDER.F\",\"content\":\"女\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.GENDER.F\",\"content\":\"Female\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.GENDER.M\",\"content\":\"男\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.GENDER.M\",\"content\":\"Male\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.RESOURCE_TYPE.MENU\",\"content\":\"菜单\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.RESOURCE_TYPE.MENU\",\"content\":\"Menu\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.RESOURCE_TYPE.OPERATION\",\"content\":\"按钮/操作\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.RESOURCE_TYPE.OPERATION\",\"content\":\"Operation\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.RESOURCE_CODE.detail\",\"content\":\"详情\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.RESOURCE_CODE.detail\",\"content\":\"Detail\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.RESOURCE_CODE.import\",\"content\":\"导入\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.RESOURCE_CODE.import\",\"content\":\"Import\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.RESOURCE_CODE.create\",\"content\":\"新建\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.RESOURCE_CODE.create\",\"content\":\"Create\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.RESOURCE_CODE.update\",\"content\":\"新建\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.RESOURCE_CODE.update\",\"content\":\"Update\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.RESOURCE_CODE.delete\",\"content\":\"删除\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.RESOURCE_CODE.delete\",\"content\":\"Delete\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.RESOURCE_CODE.export\",\"content\":\"导出\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.RESOURCE_CODE.export\",\"content\":\"Export\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.ORG_TYPE.DEPT\",\"content\":\"部门\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.ORG_TYPE.DEPT\",\"content\":\"Department\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.ORG_TYPE.COMP\",\"content\":\"公司\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.ORG_TYPE.COMP\",\"content\":\"Company\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.DATA_PERMISSION_TYPE.SELF\",\"content\":\"本人\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.DATA_PERMISSION_TYPE.SELF\",\"content\":\"Self\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.DATA_PERMISSION_TYPE.SELF_AND_SUB\",\"content\":\"本人及下属\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.DATA_PERMISSION_TYPE.SELF_AND_SUB\",\"content\":\"Self And Sub\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.DATA_PERMISSION_TYPE.DEPT\",\"content\":\"本部门\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.DATA_PERMISSION_TYPE.DEPT\",\"content\":\"Current Department\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.DATA_PERMISSION_TYPE.DEPT_AND_SUB\",\"content\":\"本部门及下属部门\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.DATA_PERMISSION_TYPE.DEPT_AND_SUB\",\"content\":\"Current Department and Sub-departments\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.DATA_PERMISSION_TYPE.ALL\",\"content\":\"全部\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.DATA_PERMISSION_TYPE.ALL\",\"content\":\"All\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.POSITION_GRADE.E1\",\"content\":\"初级\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.POSITION_GRADE.E1\",\"content\":\"E1\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.POSITION_GRADE.E2\",\"content\":\"中级\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.POSITION_GRADE.E2\",\"content\":\"E2\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.POSITION_GRADE.E3\",\"content\":\"高级\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.POSITION_GRADE.E3\",\"content\":\"E3\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.POSITION_GRADE.E4\",\"content\":\"专家\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.POSITION_GRADE.E4\",\"content\":\"E4\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.MESSAGE_STATUS.PENDING\",\"content\":\"发送中\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.MESSAGE_STATUS.PENDING\",\"content\":\"Pening\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.MESSAGE_STATUS.FAILED\",\"content\":\"发送失败\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.MESSAGE_STATUS.FAILED\",\"content\":\"Failed\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.MESSAGE_STATUS.DELIVERY\",\"content\":\"已送达\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.MESSAGE_STATUS.DELIVERY\",\"content\":\"Delivery\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.MESSAGE_STATUS.READ\",\"content\":\"已读\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.MESSAGE_STATUS.READ\",\"content\":\"Read\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.MESSAGE_CHANNEL.SMS\",\"content\":\"短信\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.MESSAGE_CHANNEL.SMS\",\"content\":\"SMS\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.MESSAGE_CHANNEL.SYS_MSG\",\"content\":\"系统消息\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.MESSAGE_CHANNEL.SYS_MSG\",\"content\":\"System Message\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.MESSAGE_CHANNEL.WEBSOCKET\",\"content\":\"站内信\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.MESSAGE_CHANNEL.WEBSOCKET\",\"content\":\"Websocket\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.MESSAGE_CHANNEL.EMAIL\",\"content\":\"邮件\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.MESSAGE_CHANNEL.EMAIL\",\"content\":\"Email\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.INIT_STRATEGY.DO_NOTHING\",\"content\":\"周期执行\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.INIT_STRATEGY.DO_NOTHING\",\"content\":\"Cycle Execution\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.INIT_STRATEGY.EFIRE_AND_PROCEED2\",\"content\":\"立即执行一次，并周期执行\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.INIT_STRATEGY.EFIRE_AND_PROCEED2\",\"content\":\"Execute Once Immediately and Cycle\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Dictionary.INIT_STRATEGY.IGNORE_MISFIRES\",\"content\":\"超期立即执行，并周期执行\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Dictionary.INIT_STRATEGY.IGNORE_MISFIRES\",\"content\":\"Execute Immediately on Overdue and Cycle\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.OrgStructure\",\"content\":\"组织人员\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.OrgStructure\",\"content\":\"Org Structure\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.User\",\"content\":\"人员管理\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.User\",\"content\":\"User\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.Org\",\"content\":\"组织部门\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.Org\",\"content\":\"Department\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.Position\",\"content\":\"岗位管理\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.Position\",\"content\":\"Position\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.System\",\"content\":\"系统管理\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.System\",\"content\":\"System\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.Dictionary\",\"content\":\"数据字典\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.Dictionary\",\"content\":\"Dictionary\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.Resource\",\"content\":\"菜单资源\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.Resource\",\"content\":\"Menu Resource\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.Role\",\"content\":\"角色权限\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.Role\",\"content\":\"Role\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.ScheduledJob\",\"content\":\"定时任务\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.ScheduledJob\",\"content\":\"Scheduled Job\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.MessageTemplate\",\"content\":\"消息模板\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.MessageTemplate\",\"content\":\"Message Template\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.Message\",\"content\":\"消息记录\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.Message\",\"content\":\"Message\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.FileRecord\",\"content\":\"文件记录\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.FileRecord\",\"content\":\"File Record\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.I18n\",\"content\":\"国际化管理\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.I18n\",\"content\":\"International\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.Configuration\",\"content\":\"系统配置\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.Configuration\",\"content\":\"System Config\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.OperationLog\",\"content\":\"操作日志\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.OperationLog\",\"content\":\"Operation Log\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.LoginLog\",\"content\":\"登录日志\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.LoginLog\",\"content\":\"Login Log\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.ChatAI\",\"content\":\"Chat AI\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.ChatAI\",\"content\":\"Chat AI\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.Tenant\",\"content\":\"租户管理\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.Tenant\",\"content\":\"Tenant\"},{\"type\":\"SYSTEM\",\"language\":\"zh_CN\",\"code\":\"Resource.TenantInformation\",\"content\":\"租户信息管理\"},{\"type\":\"SYSTEM\",\"language\":\"en\",\"code\":\"Resource.TenantInformation\",\"content\":\"Tenant Info\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Operation.create\",\"content\":\"新建\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Operation.create\",\"content\":\"Create\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Operation.update\",\"content\":\"更新\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Operation.update\",\"content\":\"Update\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Operation.detail\",\"content\":\"详情\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Operation.detail\",\"content\":\"Detail\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Operation.delete\",\"content\":\"删除\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Operation.delete\",\"content\":\"Delete\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Operation.batchDelete\",\"content\":\"批量删除\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Operation.batchDelete\",\"content\":\"Batch Delete\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Operation.import\",\"content\":\"导入\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Operation.import\",\"content\":\"Import\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Operation.export\",\"content\":\"导出\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Operation.export\",\"content\":\"Export\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Resource.User.AddPosition\",\"content\":\"添加岗位\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Resource.User.AddPosition\",\"content\":\"Add Position\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Resource.User.PositionSetting\",\"content\":\"人员岗位设置\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Resource.User.PositionSetting\",\"content\":\"Position Setting\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Operation.sort\",\"content\":\"排序\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Operation.sort\",\"content\":\"Sort\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Resource.ScheduledJob.ExecuteOnce\",\"content\":\"运行一次\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Resource.ScheduledJob.ExecuteOnce\",\"content\":\"Execute Once\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Resource.ScheduledJob.LogList\",\"content\":\"日志记录\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Resource.ScheduledJob.LogList\",\"content\":\"Log Records\"},{\"type\":\"CUSTOM\",\"language\":\"zh_CN\",\"code\":\"Resource.ScheduledJob.LogDelete\",\"content\":\"日志删除\"},{\"type\":\"CUSTOM\",\"language\":\"en\",\"code\":\"Resource.ScheduledJob.LogDelete\",\"content\":\"Delete Log\"}]";
            try {
                List<I18nConfig> i18nConfigs = JSON.parseArray(i18nData, I18nConfig.class);
                i18nConfigMapper.insert(i18nConfigs);
            } catch (BusinessException e) {
                log.error("初始化国际化配置数据出错: {}，请手动配置国际化配置数据", e.getMessage());
            }
        }
        // 插入公司根节点
        IamOrgService iamOrgService = ContextHolder.getBean(IamOrgService.class);
        IamOrg iamOrg = new IamOrg();
        if (iamOrgService != null && iamOrgService.getEntityListCount(null) == 0) {
            iamOrg.setParentId(Cons.ID_PREVENT_NULL);
            iamOrg.setCode("ROOT").setRootOrgId("1").setName("我的公司")
                    .setType(Cons.DICTCODE_ORG_TYPE.COMP.name()).setOrgComment("初始根节点，请按需修改").setId("1");
            iamOrgService.createEntity(iamOrg);
        }

        // 插入超级管理员用户及角色
        IamRoleService iamRoleService = ContextHolder.getBean(IamRoleService.class);
        if (iamRoleService != null && iamRoleService.getEntityListCount(null) == 0) {
            IamRole iamRole = new IamRole();
            iamRole.setName("超级管理员").setCode(Cons.ROLE_SUPER_ADMIN);
            iamRoleService.createEntity(iamRole);

            IamUser iamUser = new IamUser();
            iamUser.setOrgId(iamOrg.getId()).setRealname("DIBOOT").setUserNum("0000").setGender("M").setMobilePhone("10000000000");
            ContextHolder.getBean(IamUserService.class).createEntity(iamUser);

            // 插入对象
            IamUserRole iamUserRole = new IamUserRole(IamUser.class.getSimpleName(), iamUser.getId(), iamRole.getId());
            ContextHolder.getBean(IamUserRoleService.class).getMapper().insert(iamUserRole);

            // 创建账号
            IamAccount iamAccount = new IamAccount();
            iamAccount.setUserType(IamUser.class.getSimpleName()).setUserId(iamUser.getId())
                    .setAuthType(Cons.DICTCODE_AUTH_TYPE.PWD.name())
                    .setAuthAccount("admin").setAuthSecret("123456");
            ContextHolder.getBean(IamAccountService.class).createEntity(iamAccount);
        }

    }
}
