# Copyright 2015 The TensorFlow Authors. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ==============================================================================

"""GradientDescent for TensorFlow."""
from __future__ import absolute_import
from __future__ import division
from __future__ import print_function

from tensorflow.python.framework import ops
from tensorflow.python.framework import dtypes
from tensorflow.python.ops import gen_hash_training_ops
from tensorflow.python.ops import kv_variable_ops
from tensorflow.python.ops import math_ops
from tensorflow.python.ops import array_ops
from tensorflow.python.ops import gen_array_ops
from tensorflow.python.ops import resource_variable_ops
from tensorflow.python.training import optimizer
from tensorflow.python.training import training_ops
from tensorflow.python.training import training_util
from tensorflow.python.util.tf_export import tf_export


@tf_export(v1=["train.GradientDescentOptimizer"])
class GradientDescentOptimizer(optimizer.Optimizer):
  """Optimizer that implements the gradient descent algorithm.
  """

  def __init__(self, learning_rate, use_locking=False, name="GradientDescent"):
    """Construct a new gradient descent optimizer.

    Args:
      learning_rate: A Tensor or a floating point value.  The learning
        rate to use.
      use_locking: If True use locks for update operations.
      name: Optional name prefix for the operations created when applying
        gradients. Defaults to "GradientDescent".

    @compatibility(eager)
    When eager execution is enabled, `learning_rate` can be a callable that
    takes no arguments and returns the actual value to use. This can be useful
    for changing these values across different invocations of optimizer
    functions.
    @end_compatibility
    """
    super(GradientDescentOptimizer, self).__init__(use_locking, name)
    self._learning_rate = learning_rate
    self._learning_rate_tensor = None

  def _apply_dense(self, grad, var):
    return training_ops.apply_gradient_descent(
        var,
        math_ops.cast(self._learning_rate_tensor, var.dtype.base_dtype),
        grad,
        use_locking=self._use_locking).op

  def _resource_apply_dense(self, grad, handle):
    return training_ops.resource_apply_gradient_descent(
        handle.handle, math_ops.cast(self._learning_rate_tensor,
                                     grad.dtype.base_dtype),
        grad, use_locking=self._use_locking)

  def _resource_apply_sparse_duplicate_indices(self, grad, handle, indices):
    if isinstance(handle, kv_variable_ops.EmbeddingVariable):
      global_step = training_util.get_or_create_global_step()
      if handle.need_counts() and len(handle._counts_tensor.keys()) != 0:
        extra_counts, extra_indices = [], []
        if indices.op.type == "ConcatV2":
          for tensor in indices.op.inputs:
            if tensor.op.type == "Reshape":
              indices_tensor = tensor.op.inputs[0]
              if indices_tensor in handle._counts_tensor:
                extra_counts.append(handle._counts_tensor[indices_tensor])
                extra_indices.append(indices_tensor)
        elif indices.op.type == "Reshape":
          indices_tensor = indices.op.inputs[0]
          if indices_tensor in handle._counts_tensor:
            extra_counts.append(handle._counts_tensor[indices_tensor])
            extra_indices.append(indices_tensor)
        unique_indices, new_index_positions, indices_counts = \
            gen_array_ops._unique_with_extra_counts(indices, extra_indices, extra_counts)
        summed_grads = math_ops.unsorted_segment_sum(
            grad, new_index_positions, array_ops.shape(unique_indices)[0])
        return training_ops.kv_resource_sparse_apply_gradient_descent_with_counts(
            handle.handle, math_ops.cast(self._learning_rate_tensor,
                                         grad.dtype.base_dtype),
            summed_grads, unique_indices, global_step,
            indices_counts, use_locking=self._use_locking)
      else:
        return training_ops.kv_resource_sparse_apply_gradient_descent(
            handle.handle, math_ops.cast(self._learning_rate_tensor,
                                         grad.dtype.base_dtype),
            grad, indices, global_step, use_locking=self._use_locking)
    else:
      return resource_variable_ops.resource_scatter_add(
          handle.handle, indices, -grad * self._learning_rate)

  def _apply_sparse_duplicate_indices(self, grad, var):
    delta = ops.IndexedSlices(
        grad.values *
        math_ops.cast(self._learning_rate_tensor, var.dtype.base_dtype),
        grad.indices, grad.dense_shape)
    return var.scatter_sub(delta, use_locking=self._use_locking)

  def _hash_table_apply_sparse(self, grad, var, indices):
    return gen_hash_training_ops.tensible_variable_apply_gradient_descent(
        var.handle,
        math_ops.cast(self._learning_rate_tensor, grad.dtype.base_dtype),
        grad, indices, use_locking=self._use_locking)

  def _prepare(self):
    learning_rate = self._call_if_callable(self._learning_rate)
    self._learning_rate_tensor = ops.convert_to_tensor(
        learning_rate, name="learning_rate")
