import { __decorate } from '../../../../tslib/tslib.es6.mjs.js';
import '../../focus/md-focus-ring.js';
import '../../ripple/ripple.js';
import '../../../../@lit/reactive-element/reactive-element.js';
import { html as x } from '../../../../lit-html/lit-html.js';
import { LitElement as s } from '../../../../lit-element/lit-element.js';
import { property as n } from '../../../../@lit/reactive-element/decorators/property.js';
import { classMap as e } from '../../../../lit-html/directives/class-map.js';
import { requestUpdateOnAriaChange } from '../../internal/aria/delegate.js';

/**
 * @license
 * Copyright 2023 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * A chip component.
 *
 * @fires update-focus {Event} Dispatched when `disabled` is toggled. --bubbles
 */
class Chip extends s {
    constructor() {
        super(...arguments);
        /**
         * Whether or not the chip is disabled.
         *
         * Disabled chips are not focusable, unless `always-focusable` is set.
         */
        this.disabled = false;
        /**
         * When true, allow disabled chips to be focused with arrow keys.
         *
         * Add this when a chip needs increased visibility when disabled. See
         * https://www.w3.org/WAI/ARIA/apg/practices/keyboard-interface/#kbd_disabled_controls
         * for more guidance on when this is needed.
         */
        this.alwaysFocusable = false;
        /**
         * The label of the chip.
         */
        this.label = '';
        /**
         * Only needed for SSR.
         *
         * Add this attribute when a chip has a `slot="icon"` to avoid a Flash Of
         * Unstyled Content.
         */
        this.hasIcon = false;
    }
    /**
     * Whether or not the primary ripple is disabled (defaults to `disabled`).
     * Some chip actions such as links cannot be disabled.
     */
    get rippleDisabled() {
        return this.disabled;
    }
    focus(options) {
        if (this.disabled && !this.alwaysFocusable) {
            return;
        }
        super.focus(options);
    }
    render() {
        return x `
      <div class="container ${e(this.getContainerClasses())}">
        ${this.renderContainerContent()}
      </div>
    `;
    }
    updated(changed) {
        if (changed.has('disabled') && changed.get('disabled') !== undefined) {
            this.dispatchEvent(new Event('update-focus', { bubbles: true }));
        }
    }
    getContainerClasses() {
        return {
            'disabled': this.disabled,
            'has-icon': this.hasIcon,
        };
    }
    renderContainerContent() {
        return x `
      ${this.renderOutline()}
      <md-focus-ring part="focus-ring" for=${this.primaryId}></md-focus-ring>
      <md-ripple
        for=${this.primaryId}
        ?disabled=${this.rippleDisabled}></md-ripple>
      ${this.renderPrimaryAction(this.renderPrimaryContent())}
    `;
    }
    renderOutline() {
        return x `<span class="outline"></span>`;
    }
    renderLeadingIcon() {
        return x `<slot name="icon" @slotchange=${this.handleIconChange}></slot>`;
    }
    renderPrimaryContent() {
        return x `
      <span class="leading icon" aria-hidden="true">
        ${this.renderLeadingIcon()}
      </span>
      <span class="label">${this.label}</span>
      <span class="touch"></span>
    `;
    }
    handleIconChange(event) {
        const slot = event.target;
        this.hasIcon = slot.assignedElements({ flatten: true }).length > 0;
    }
}
(() => {
    requestUpdateOnAriaChange(Chip);
})();
/** @nocollapse */
Chip.shadowRootOptions = {
    ...s.shadowRootOptions,
    delegatesFocus: true,
};
__decorate([
    n({ type: Boolean, reflect: true })
], Chip.prototype, "disabled", void 0);
__decorate([
    n({ type: Boolean, attribute: 'always-focusable' })
], Chip.prototype, "alwaysFocusable", void 0);
__decorate([
    n()
], Chip.prototype, "label", void 0);
__decorate([
    n({ type: Boolean, reflect: true, attribute: 'has-icon' })
], Chip.prototype, "hasIcon", void 0);

export { Chip };
