// ******************************************************************
// *
// *  This file is part of the Cxbx project.
// *
// *  Cxbx and Cxbe are free software; you can redistribute them
// *  and/or modify them under the terms of the GNU General Public
// *  License as published by the Free Software Foundation; either
// *  version 2 of the license, or (at your option) any later version.
// *
// *  This program is distributed in the hope that it will be useful,
// *  but WITHOUT ANY WARRANTY; without even the implied warranty of
// *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// *  GNU General Public License for more details.
// *
// *  You should have recieved a copy of the GNU General Public License
// *  along with this program; see the file COPYING.
// *  If not, write to the Free Software Foundation, Inc.,
// *  59 Temple Place - Suite 330, Bostom, MA 02111-1307, USA.
// *
// *  (c) 2002-2003 Aaron Robinson <caustik@caustik.com>
// *
// *  All rights reserved
// *
// ******************************************************************
#ifndef XBCONVERT_H
#define XBCONVERT_H

#include "common/cxbxr.hpp"
#include "core\kernel\init\CxbxKrnl.h"

#include "core\hle\D3D8\XbD3D8Types.h"

#define VERTICES_PER_DOT 1
#define VERTICES_PER_LINE 2
#define VERTICES_PER_TRIANGLE 3
#define VERTICES_PER_QUAD 4
#define TRIANGLES_PER_QUAD 2

#define LOG_PREFIX_D3DCVT CXBXR_MODULE::D3DCVT

// simple render state encoding lookup table
#define X_D3DRSSE_UNK 0x7fffffff

typedef void(*FormatToARGBRow)(const uint8_t* src, uint8_t* dst_argb, int width);

extern const FormatToARGBRow EmuXBFormatComponentConverter(xbox::X_D3DFORMAT Format);

bool EmuXBFormatCanBeConvertedToARGB(xbox::X_D3DFORMAT Format);

bool EmuXBFormatRequiresConversionToARGB(xbox::X_D3DFORMAT Format);

// how many bits does this format use per pixel?
extern DWORD EmuXBFormatBitsPerPixel(xbox::X_D3DFORMAT Format);

// how many bytes does this format use per pixel?
extern DWORD EmuXBFormatBytesPerPixel(xbox::X_D3DFORMAT Format);

// is this format compressed?
extern BOOL EmuXBFormatIsCompressed(xbox::X_D3DFORMAT Format);

// is this format linear?
extern BOOL EmuXBFormatIsLinear(xbox::X_D3DFORMAT Format);

// is this format swizzled?
extern BOOL EmuXBFormatIsSwizzled(xbox::X_D3DFORMAT Format);

// is this format a valid render target?
extern BOOL EmuXBFormatIsRenderTarget(xbox::X_D3DFORMAT Format);

// is this format a valid depth buffer?
extern BOOL EmuXBFormatIsDepthBuffer(xbox::X_D3DFORMAT Format);

// convert from xbox to pc color formats
extern D3DFORMAT EmuXB2PC_D3DFormat(xbox::X_D3DFORMAT Format);

// convert from pc to xbox color formats
extern xbox::X_D3DFORMAT EmuPC2XB_D3DFormat(D3DFORMAT Format, bool bPreferLinear = true);

// convert from xbox to pc d3d lock flags
extern DWORD EmuXB2PC_D3DLock(DWORD Flags);

// convert from xbox to pc multisample formats
extern D3DMULTISAMPLE_TYPE EmuXB2PC_D3DMultiSampleFormat(DWORD Type);

/**
// convert from pc to xbox texture transform state types (unnecessary so far)
if((uint32)State < 4)
    State = (D3DTRANSFORMSTATETYPE)(State - 2);
else if((uint32)State < 20)
    State = (D3DTRANSFORMSTATETYPE)(State - 14);
else if((uint32)State > 255)
    State = (D3DTRANSFORMSTATETYPE)(State - 250);
else
    CxbxrAbortEx(LOG_PREFIX_D3DCVT, "Unknown Transform State Type (%d)", State);
//*/

// convert from xbox to pc texture transform state types
inline D3DTRANSFORMSTATETYPE EmuXB2PC_D3DTS(xbox::X_D3DTRANSFORMSTATETYPE State)
{
    // Handle Xbox -> D3D State mapping
    switch (State) {
	case xbox::X_D3DTS_VIEW: return D3DTS_VIEW;
    case xbox::X_D3DTS_PROJECTION: return D3DTS_PROJECTION;
    case xbox::X_D3DTS_TEXTURE0: return D3DTS_TEXTURE0;
    case xbox::X_D3DTS_TEXTURE1: return D3DTS_TEXTURE1;
    case xbox::X_D3DTS_TEXTURE2: return D3DTS_TEXTURE2;
    case xbox::X_D3DTS_TEXTURE3: return D3DTS_TEXTURE3;
    case xbox::X_D3DTS_WORLD: return D3DTS_WORLD;
    case xbox::X_D3DTS_WORLD1: return D3DTS_WORLD1;
    case xbox::X_D3DTS_WORLD2: return D3DTS_WORLD2;
    case xbox::X_D3DTS_WORLD3: return D3DTS_WORLD3;
    }

    // Handle World Matrix offsets
    if (State >= 256 && State <= 511) {
		return D3DTS_WORLDMATRIX(State - 256);
    }

    CxbxrAbortEx(LOG_PREFIX_D3DCVT, "Unknown Transform State Type (%d)", State);
    return (D3DTRANSFORMSTATETYPE)0;
}

// convert from xbox to pc blend ops
inline D3DBLENDOP EmuXB2PC_D3DBLENDOP(xbox::X_D3DBLENDOP Value)
{
	switch(Value)
    {
        case 0x8006: return D3DBLENDOP_ADD;
        case 0x800a: return D3DBLENDOP_SUBTRACT;
        case 0x800b: return D3DBLENDOP_REVSUBTRACT;
        case 0x8007: return D3DBLENDOP_MIN;
        case 0x8008: return D3DBLENDOP_MAX;
        case 0xF006: {
            EmuLogEx(LOG_PREFIX_D3DCVT, LOG_LEVEL::WARNING, "D3DBLENDOP_ADDSIGNED is not supported!");
            return D3DBLENDOP_ADD;
        };
        case 0xF005: {
            EmuLogEx(LOG_PREFIX_D3DCVT, LOG_LEVEL::WARNING, "D3DBLENDOP_REVSUBTRACTSIGNED is not supported!");
            return D3DBLENDOP_REVSUBTRACT;
        }
    }

    EmuLogEx(LOG_PREFIX_D3DCVT, LOG_LEVEL::WARNING, "Unknown D3DBLENDOP (0x%.08X)", Value);

    return (D3DBLENDOP)D3DBLENDOP_ADD;
}

// convert from xbox to pc blend types 
inline D3DBLEND EmuXB2PC_D3DBLEND(xbox::X_D3DBLEND Value)
{
	switch (Value) {
        case 0x000: return D3DBLEND_ZERO;
        case 0x001: return D3DBLEND_ONE;
        case 0x300: return D3DBLEND_SRCCOLOR;
        case 0x301: return D3DBLEND_INVSRCCOLOR;
        case 0x302: return D3DBLEND_SRCALPHA;
        case 0x303: return D3DBLEND_INVSRCALPHA;
        case 0x304: return D3DBLEND_DESTALPHA;
        case 0x305: return D3DBLEND_INVDESTALPHA;
        case 0x306: return D3DBLEND_DESTCOLOR;
        case 0x307: return D3DBLEND_INVDESTCOLOR;
        case 0x308: return D3DBLEND_SRCALPHASAT;
        case 0x8001:return D3DBLEND_BLENDFACTOR; // Maps Xbox D3DBLEND_CONSTANTCOLOR
        case 0x8002:return D3DBLEND_INVBLENDFACTOR; // Maps Xbox D3DBLEND_INVCONSTANTCOLOR
        case 0x8003: EmuLogEx(LOG_PREFIX_D3DCVT, LOG_LEVEL::WARNING, "Unsupported Xbox D3DBlend Extension: D3DBLEND_CONSTANTALPHA"); return D3DBLEND_ONE;
        case 0x8004: EmuLogEx(LOG_PREFIX_D3DCVT, LOG_LEVEL::WARNING, "Unsupported Xbox D3DBlend Extension: D3DBLEND_INVCONSTANTALPHA"); return D3DBLEND_ONE;
    }

    EmuLogEx(LOG_PREFIX_D3DCVT, LOG_LEVEL::WARNING, "Unknown Xbox D3DBLEND Extension (0x%.08X)", Value);
	return D3DBLEND_ONE;
}

// convert from xbox to pc comparison functions
inline D3DCMPFUNC EmuXB2PC_D3DCMPFUNC(xbox::X_D3DCMPFUNC Value)
{
	switch (Value) {
        case 0x200: return D3DCMP_NEVER;
        case 0x201: return D3DCMP_LESS;
        case 0x202: return D3DCMP_EQUAL;
        case 0x203: return D3DCMP_LESSEQUAL;
        case 0x204: return D3DCMP_GREATER;
        case 0x205: return D3DCMP_NOTEQUAL;
        case 0x206: return D3DCMP_GREATEREQUAL;
        case 0x207: return D3DCMP_ALWAYS;
    }

    EmuLogEx(LOG_PREFIX_D3DCVT, LOG_LEVEL::WARNING, "Unknown Xbox D3DCMPFUNC Extension (0x%.08X)", Value);
    return D3DCMP_NEVER;
}

// convert from xbox to pc fill modes
inline D3DFILLMODE EmuXB2PC_D3DFILLMODE(xbox::X_D3DFILLMODE Value)
{
	switch (Value) {
        case 0x1B00: return D3DFILL_POINT;
        case 0x1B01: return D3DFILL_WIREFRAME;
        case 0x1B02: return D3DFILL_SOLID;
    }

    EmuLogEx(LOG_PREFIX_D3DCVT, LOG_LEVEL::WARNING, "Unknown Xbox D3DFILLMODE Extension (0x%.08X)", Value);
    return D3DFILL_SOLID;
}

// convert from xbox to pc shade modes
inline D3DSHADEMODE EmuXB2PC_D3DSHADEMODE(xbox::X_D3DSHADEMODE Value)
{
	switch (Value) {
        case 0x1D00: return D3DSHADE_FLAT;
        case 0x1D01: return D3DSHADE_GOURAUD;
    }

    EmuLogEx(LOG_PREFIX_D3DCVT, LOG_LEVEL::WARNING, "Unknown Xbox D3DSHADEMODE Extension (0x%.08X)", Value);
    return D3DSHADE_GOURAUD;
}

// convert from xbox to pc stencilop modes
inline D3DSTENCILOP EmuXB2PC_D3DSTENCILOP(xbox::X_D3DSTENCILOP Value)
{
    switch(Value)
    {
        case 0x1e00: return D3DSTENCILOP_KEEP;
        case 0x0000: return D3DSTENCILOP_ZERO;
        case 0x1e01: return D3DSTENCILOP_REPLACE;
        case 0x1e02: return D3DSTENCILOP_INCRSAT;
        case 0x1e03: return D3DSTENCILOP_DECRSAT;
        case 0x150a: return D3DSTENCILOP_INVERT;
        case 0x8507: return D3DSTENCILOP_INCR;
        case 0x8508: return D3DSTENCILOP_DECR;
        default:
            EmuLogEx(LOG_PREFIX_D3DCVT, LOG_LEVEL::WARNING, "Unknown D3DSTENCILOP (0x%.08X)", Value);
    }

    return (D3DSTENCILOP) Value;
}

// table used for vertex->primitive count conversion
extern const UINT g_XboxPrimitiveTypeInfo[xbox::X_D3DPT_POLYGON + 1][2];

inline bool IsValidXboxVertexCount(xbox::X_D3DPRIMITIVETYPE XboxPrimitiveType, UINT VertexCount)
{
	assert(XboxPrimitiveType < xbox::X_D3DPT_MAX);

	// Are there more vertices than required for setup?
	if (VertexCount > g_XboxPrimitiveTypeInfo[XboxPrimitiveType][0])
		// Are the additional vertices exact multiples of the required additional vertices per primitive?
		if (0 == ((VertexCount - g_XboxPrimitiveTypeInfo[XboxPrimitiveType][0]) % g_XboxPrimitiveTypeInfo[XboxPrimitiveType][1]))
			return true;

	return false;
}

// convert from vertex count to primitive count (Xbox)
inline unsigned ConvertXboxVertexCountToPrimitiveCount(xbox::X_D3DPRIMITIVETYPE XboxPrimitiveType, unsigned VertexCount)
{
	assert(XboxPrimitiveType < xbox::X_D3DPT_MAX);

	return (VertexCount - g_XboxPrimitiveTypeInfo[XboxPrimitiveType][0]) / g_XboxPrimitiveTypeInfo[XboxPrimitiveType][1];
}

// conversion table for xbox->pc primitive types
extern const D3DPRIMITIVETYPE g_XboxPrimitiveTypeToHost[];

// convert xbox->pc primitive type
inline D3DPRIMITIVETYPE EmuXB2PC_D3DPrimitiveType(xbox::X_D3DPRIMITIVETYPE XboxPrimitiveType)
{
	if (XboxPrimitiveType >= xbox::X_D3DPT_MAX) {
		LOG_TEST_CASE("XboxPrimitiveType too large");
		return D3DPT_FORCE_DWORD;
	}

    return g_XboxPrimitiveTypeToHost[XboxPrimitiveType];
}

extern void EmuUnswizzleBox
(
	CONST PVOID pSrcBuff,
	CONST DWORD dwWidth,
	CONST DWORD dwHeight,
	CONST DWORD dwDepth,
	CONST DWORD dwBytesPerPixel,
	CONST PVOID pDstBuff,
	CONST DWORD dwDstRowPitch,
	CONST DWORD dwDstSlicePitch
); // NOPATCH

// From : https://www.virtualbox.org/svn/vbox/trunk/src/VBox/Additions/x11/x11include/libdrm-2.4.13/nouveau_class.h
// Cxbx notes (like Dxbx) :
// - Changed NV20TCL_ into NV2A_
// - Added specific NV2A constants
// TODO : Add more (especially when they conflict with NV20)

#define NV2A										0x00000097

#define ENUM_NV2A(METHOD, RANGED, BITFLD, VALUE) \
	METHOD(NV2A_NOP,											0x00000100) \
	METHOD(NV2A_NOTIFY,											0x00000104) \
	METHOD(NV2A_SET_WARNING_ENABLE,								0x00000108) \
	METHOD(NV2A_GET_STATE,										0x0000010C) \
	METHOD(NV2A_WAIT_FOR_IDLE,									0x00000110) \
	METHOD(NV2A_FLIP_READ,										0x00000120) \
	METHOD(NV2A_FLIP_WRITE,										0x00000124) \
	METHOD(NV2A_FLIP_MODULO,									0x00000128) \
	METHOD(NV2A_FLIP_INCREMENT_WRITE,							0x0000012C) \
	METHOD(NV2A_FLIP_STALL,										0x00000130) \
	METHOD(NV2A_PM_TRIGGER,										0x00000140) \
	METHOD(NV2A_DMA_NOTIFY,										0x00000180) \
	METHOD(NV2A_DMA_TEXTURE0,									0x00000184) \
	METHOD(NV2A_DMA_TEXTURE1,									0x00000188) \
	METHOD(NV2A_DMA_STATE,										0x00000190) \
	METHOD(NV2A_DMA_COLOR,										0x00000194) \
	METHOD(NV2A_DMA_ZETA,										0x00000198) \
	METHOD(NV2A_DMA_VTXBUF0,									0x0000019c) \
	METHOD(NV2A_DMA_VTXBUF1,									0x000001a0) \
	METHOD(NV2A_DMA_FENCE,										0x000001a4) \
	METHOD(NV2A_DMA_QUERY,										0x000001a8) \
	METHOD(NV2A_RT_HORIZ,										0x00000200) \
	BITFLD(NV2A_RT_HORIZ_X_SHIFT,									0) \
	BITFLD(NV2A_RT_HORIZ_X_MASK,										0x0000ffff) \
	BITFLD(NV2A_RT_HORIZ_W_SHIFT,									16) \
	BITFLD(NV2A_RT_HORIZ_W_MASK,										0xffff0000) \
	METHOD(NV2A_RT_VERT,										0x00000204) \
	BITFLD(NV2A_RT_VERT_Y_SHIFT,									0) \
	BITFLD(NV2A_RT_VERT_Y_MASK,											0x0000ffff) \
	BITFLD(NV2A_RT_VERT_H_SHIFT,									16) \
	BITFLD(NV2A_RT_VERT_H_MASK,											0xffff0000) \
	METHOD(NV2A_RT_FORMAT,										0x00000208) \
	BITFLD(NV2A_RT_FORMAT_TYPE_SHIFT,								8) \
	BITFLD(NV2A_RT_FORMAT_TYPE_MASK,									0x00000f00) \
	 VALUE(NV2A_RT_FORMAT_TYPE_LINEAR,									0x00000100) \
	 VALUE(NV2A_RT_FORMAT_TYPE_SWIZZLED,								0x00000200) \
	BITFLD(NV2A_RT_FORMAT_COLOR_SHIFT,								0) \
	BITFLD(NV2A_RT_FORMAT_COLOR_MASK,									0x0000001f) \
	 VALUE(NV2A_RT_FORMAT_COLOR_R5G6B5,									0x00000003) \
	 VALUE(NV2A_RT_FORMAT_COLOR_X8R8G8B8,								0x00000005) \
	 VALUE(NV2A_RT_FORMAT_COLOR_A8R8G8B8,								0x00000008) \
	 VALUE(NV2A_RT_FORMAT_COLOR_B8,										0x00000009) \
	 VALUE(NV2A_RT_FORMAT_COLOR_UNKNOWN,								0x0000000d) \
	 VALUE(NV2A_RT_FORMAT_COLOR_X8B8G8R8,								0x0000000f) \
	 VALUE(NV2A_RT_FORMAT_COLOR_A8B8G8R8,								0x00000010) \
	METHOD(NV2A_RT_PITCH,										0x0000020c) \
	BITFLD(NV2A_RT_PITCH_COLOR_PITCH_SHIFT,							0) \
	BITFLD(NV2A_RT_PITCH_COLOR_PITCH_MASK,								0x0000ffff) \
	BITFLD(NV2A_RT_PITCH_ZETA_PITCH_SHIFT,							16) \
	BITFLD(NV2A_RT_PITCH_ZETA_PITCH_MASK,								0xffff0000) \
	METHOD(NV2A_COLOR_OFFSET,									0x00000210) \
	METHOD(NV2A_ZETA_OFFSET,									0x00000214) \
	RANGED(NV2A_RC_IN_ALPHA,									0x00000260, 0x04, 8) \
	BITFLD(NV2A_RC_IN_ALPHA_D_INPUT_SHIFT,							0) \
	BITFLD(NV2A_RC_IN_ALPHA_D_INPUT_MASK,								0x0000000f) \
	 VALUE(NV2A_RC_IN_ALPHA_D_INPUT_ZERO,								0x00000000) \
	 VALUE(NV2A_RC_IN_ALPHA_D_INPUT_CONSTANT_COLOR0_NV,					0x00000001) \
	 VALUE(NV2A_RC_IN_ALPHA_D_INPUT_CONSTANT_COLOR1_NV,					0x00000002) \
	 VALUE(NV2A_RC_IN_ALPHA_D_INPUT_FOG,								0x00000003) \
	 VALUE(NV2A_RC_IN_ALPHA_D_INPUT_PRIMARY_COLOR_NV,					0x00000004) \
	 VALUE(NV2A_RC_IN_ALPHA_D_INPUT_SECONDARY_COLOR_NV,					0x00000005) \
	 VALUE(NV2A_RC_IN_ALPHA_D_INPUT_TEXTURE0_ARB,						0x00000008) \
	 VALUE(NV2A_RC_IN_ALPHA_D_INPUT_TEXTURE1_ARB,						0x00000009) \
	 VALUE(NV2A_RC_IN_ALPHA_D_INPUT_SPARE0_NV,							0x0000000c) \
	 VALUE(NV2A_RC_IN_ALPHA_D_INPUT_SPARE1_NV,							0x0000000d) \
	 VALUE(NV2A_RC_IN_ALPHA_D_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x0000000e) \
	 VALUE(NV2A_RC_IN_ALPHA_D_INPUT_E_TIMES_F_NV,						0x0000000f) \
	BITFLD(NV2A_RC_IN_ALPHA_D_COMPONENT_USAGE,						(1 <<  4)) \
	 VALUE(NV2A_RC_IN_ALPHA_D_COMPONENT_USAGE_BLUE,						0x00000000) \
	 VALUE(NV2A_RC_IN_ALPHA_D_COMPONENT_USAGE_ALPHA,					0x00000010) \
	BITFLD(NV2A_RC_IN_ALPHA_D_MAPPING_SHIFT,						5) \
	BITFLD(NV2A_RC_IN_ALPHA_D_MAPPING_MASK,								0x000000e0) \
	 VALUE(NV2A_RC_IN_ALPHA_D_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_IN_ALPHA_D_MAPPING_UNSIGNED_INVERT_NV,				0x00000020) \
	 VALUE(NV2A_RC_IN_ALPHA_D_MAPPING_EXPAND_NORMAL_NV,					0x00000040) \
	 VALUE(NV2A_RC_IN_ALPHA_D_MAPPING_EXPAND_NEGATE_NV,					0x00000060) \
	 VALUE(NV2A_RC_IN_ALPHA_D_MAPPING_HALF_BIAS_NORMAL_NV,				0x00000080) \
	 VALUE(NV2A_RC_IN_ALPHA_D_MAPPING_HALF_BIAS_NEGATE_NV,				0x000000a0) \
	 VALUE(NV2A_RC_IN_ALPHA_D_MAPPING_SIGNED_IDENTITY_NV,				0x000000c0) \
	 VALUE(NV2A_RC_IN_ALPHA_D_MAPPING_SIGNED_NEGATE_NV,					0x000000e0) \
	BITFLD(NV2A_RC_IN_ALPHA_C_INPUT_SHIFT,							8) \
	BITFLD(NV2A_RC_IN_ALPHA_C_INPUT_MASK,								0x00000f00) \
	 VALUE(NV2A_RC_IN_ALPHA_C_INPUT_ZERO,								0x00000000) \
	 VALUE(NV2A_RC_IN_ALPHA_C_INPUT_CONSTANT_COLOR0_NV,					0x00000100) \
	 VALUE(NV2A_RC_IN_ALPHA_C_INPUT_CONSTANT_COLOR1_NV,					0x00000200) \
	 VALUE(NV2A_RC_IN_ALPHA_C_INPUT_FOG,								0x00000300) \
	 VALUE(NV2A_RC_IN_ALPHA_C_INPUT_PRIMARY_COLOR_NV,					0x00000400) \
	 VALUE(NV2A_RC_IN_ALPHA_C_INPUT_SECONDARY_COLOR_NV,					0x00000500) \
	 VALUE(NV2A_RC_IN_ALPHA_C_INPUT_TEXTURE0_ARB,						0x00000800) \
	 VALUE(NV2A_RC_IN_ALPHA_C_INPUT_TEXTURE1_ARB,						0x00000900) \
	 VALUE(NV2A_RC_IN_ALPHA_C_INPUT_SPARE0_NV,							0x00000c00) \
	 VALUE(NV2A_RC_IN_ALPHA_C_INPUT_SPARE1_NV,							0x00000d00) \
	 VALUE(NV2A_RC_IN_ALPHA_C_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x00000e00) \
	 VALUE(NV2A_RC_IN_ALPHA_C_INPUT_E_TIMES_F_NV,						0x00000f00) \
	BITFLD(NV2A_RC_IN_ALPHA_C_COMPONENT_USAGE,						(1 << 12)) \
	 VALUE(NV2A_RC_IN_ALPHA_C_COMPONENT_USAGE_BLUE,						0x00000000) \
	 VALUE(NV2A_RC_IN_ALPHA_C_COMPONENT_USAGE_ALPHA,					0x00001000) \
	BITFLD(NV2A_RC_IN_ALPHA_C_MAPPING_SHIFT,						13) \
	BITFLD(NV2A_RC_IN_ALPHA_C_MAPPING_MASK,								0x0000e000) \
	 VALUE(NV2A_RC_IN_ALPHA_C_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_IN_ALPHA_C_MAPPING_UNSIGNED_INVERT_NV,				0x00002000) \
	 VALUE(NV2A_RC_IN_ALPHA_C_MAPPING_EXPAND_NORMAL_NV,					0x00004000) \
	 VALUE(NV2A_RC_IN_ALPHA_C_MAPPING_EXPAND_NEGATE_NV,					0x00006000) \
	 VALUE(NV2A_RC_IN_ALPHA_C_MAPPING_HALF_BIAS_NORMAL_NV,				0x00008000) \
	 VALUE(NV2A_RC_IN_ALPHA_C_MAPPING_HALF_BIAS_NEGATE_NV,				0x0000a000) \
	 VALUE(NV2A_RC_IN_ALPHA_C_MAPPING_SIGNED_IDENTITY_NV,				0x0000c000) \
	 VALUE(NV2A_RC_IN_ALPHA_C_MAPPING_SIGNED_NEGATE_NV,					0x0000e000) \
	BITFLD(NV2A_RC_IN_ALPHA_B_INPUT_SHIFT,							16) \
	BITFLD(NV2A_RC_IN_ALPHA_B_INPUT_MASK,								0x000f0000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_INPUT_ZERO,								0x00000000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_INPUT_CONSTANT_COLOR0_NV,					0x00010000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_INPUT_CONSTANT_COLOR1_NV,					0x00020000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_INPUT_FOG,								0x00030000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_INPUT_PRIMARY_COLOR_NV,					0x00040000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_INPUT_SECONDARY_COLOR_NV,					0x00050000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_INPUT_TEXTURE0_ARB,						0x00080000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_INPUT_TEXTURE1_ARB,						0x00090000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_INPUT_SPARE0_NV,							0x000c0000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_INPUT_SPARE1_NV,							0x000d0000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x000e0000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_INPUT_E_TIMES_F_NV,						0x000f0000) \
	BITFLD(NV2A_RC_IN_ALPHA_B_COMPONENT_USAGE,						(1 << 20)) \
	 VALUE(NV2A_RC_IN_ALPHA_B_COMPONENT_USAGE_BLUE,						0x00000000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_COMPONENT_USAGE_ALPHA,					0x00100000) \
	BITFLD(NV2A_RC_IN_ALPHA_B_MAPPING_SHIFT,						21) \
	BITFLD(NV2A_RC_IN_ALPHA_B_MAPPING_MASK,								0x00e00000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_MAPPING_UNSIGNED_INVERT_NV,				0x00200000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_MAPPING_EXPAND_NORMAL_NV,					0x00400000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_MAPPING_EXPAND_NEGATE_NV,					0x00600000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_MAPPING_HALF_BIAS_NORMAL_NV,				0x00800000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_MAPPING_HALF_BIAS_NEGATE_NV,				0x00a00000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_MAPPING_SIGNED_IDENTITY_NV,				0x00c00000) \
	 VALUE(NV2A_RC_IN_ALPHA_B_MAPPING_SIGNED_NEGATE_NV,					0x00e00000) \
	BITFLD(NV2A_RC_IN_ALPHA_A_INPUT_SHIFT,							24) \
	BITFLD(NV2A_RC_IN_ALPHA_A_INPUT_MASK,								0x0f000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_INPUT_ZERO,								0x00000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_INPUT_CONSTANT_COLOR0_NV,					0x01000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_INPUT_CONSTANT_COLOR1_NV,					0x02000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_INPUT_FOG,								0x03000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_INPUT_PRIMARY_COLOR_NV,					0x04000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_INPUT_SECONDARY_COLOR_NV,					0x05000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_INPUT_TEXTURE0_ARB,						0x08000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_INPUT_TEXTURE1_ARB,						0x09000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_INPUT_SPARE0_NV,							0x0c000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_INPUT_SPARE1_NV,							0x0d000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x0e000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_INPUT_E_TIMES_F_NV,						0x0f000000) \
	BITFLD(NV2A_RC_IN_ALPHA_A_COMPONENT_USAGE,						(1 << 28)) \
	 VALUE(NV2A_RC_IN_ALPHA_A_COMPONENT_USAGE_BLUE,						0x00000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_COMPONENT_USAGE_ALPHA,					0x10000000) \
	BITFLD(NV2A_RC_IN_ALPHA_A_MAPPING_SHIFT,						29) \
	BITFLD(NV2A_RC_IN_ALPHA_A_MAPPING_MASK,								0xe0000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_MAPPING_UNSIGNED_INVERT_NV,				0x20000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_MAPPING_EXPAND_NORMAL_NV,					0x40000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_MAPPING_EXPAND_NEGATE_NV,					0x60000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_MAPPING_HALF_BIAS_NORMAL_NV,				0x80000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_MAPPING_HALF_BIAS_NEGATE_NV,				0xa0000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_MAPPING_SIGNED_IDENTITY_NV,				0xc0000000) \
	 VALUE(NV2A_RC_IN_ALPHA_A_MAPPING_SIGNED_NEGATE_NV,					0xe0000000) \
	METHOD(NV2A_RC_FINAL0,										0x00000288) \
	BITFLD(NV2A_RC_FINAL0_D_INPUT_SHIFT,							0) \
	BITFLD(NV2A_RC_FINAL0_D_INPUT_MASK,									0x0000000f) \
	 VALUE(NV2A_RC_FINAL0_D_INPUT_ZERO,									0x00000000) \
	 VALUE(NV2A_RC_FINAL0_D_INPUT_CONSTANT_COLOR0_NV,					0x00000001) \
	 VALUE(NV2A_RC_FINAL0_D_INPUT_CONSTANT_COLOR1_NV,					0x00000002) \
	 VALUE(NV2A_RC_FINAL0_D_INPUT_FOG,									0x00000003) \
	 VALUE(NV2A_RC_FINAL0_D_INPUT_PRIMARY_COLOR_NV,						0x00000004) \
	 VALUE(NV2A_RC_FINAL0_D_INPUT_SECONDARY_COLOR_NV,					0x00000005) \
	 VALUE(NV2A_RC_FINAL0_D_INPUT_TEXTURE0_ARB,							0x00000008) \
	 VALUE(NV2A_RC_FINAL0_D_INPUT_TEXTURE1_ARB,							0x00000009) \
	 VALUE(NV2A_RC_FINAL0_D_INPUT_SPARE0_NV,							0x0000000c) \
	 VALUE(NV2A_RC_FINAL0_D_INPUT_SPARE1_NV,							0x0000000d) \
	 VALUE(NV2A_RC_FINAL0_D_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x0000000e) \
	 VALUE(NV2A_RC_FINAL0_D_INPUT_E_TIMES_F_NV,							0x0000000f) \
	BITFLD(NV2A_RC_FINAL0_D_COMPONENT_USAGE,						(1 <<  4)) \
	 VALUE(NV2A_RC_FINAL0_D_COMPONENT_USAGE_RGB,						0x00000000) \
	 VALUE(NV2A_RC_FINAL0_D_COMPONENT_USAGE_ALPHA,						0x00000010) \
	BITFLD(NV2A_RC_FINAL0_D_MAPPING_SHIFT,							5) \
	BITFLD(NV2A_RC_FINAL0_D_MAPPING_MASK,								0x000000e0) \
	 VALUE(NV2A_RC_FINAL0_D_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_FINAL0_D_MAPPING_UNSIGNED_INVERT_NV,					0x00000020) \
	 VALUE(NV2A_RC_FINAL0_D_MAPPING_EXPAND_NORMAL_NV,					0x00000040) \
	 VALUE(NV2A_RC_FINAL0_D_MAPPING_EXPAND_NEGATE_NV,					0x00000060) \
	 VALUE(NV2A_RC_FINAL0_D_MAPPING_HALF_BIAS_NORMAL_NV,				0x00000080) \
	 VALUE(NV2A_RC_FINAL0_D_MAPPING_HALF_BIAS_NEGATE_NV,				0x000000a0) \
	 VALUE(NV2A_RC_FINAL0_D_MAPPING_SIGNED_IDENTITY_NV,					0x000000c0) \
	 VALUE(NV2A_RC_FINAL0_D_MAPPING_SIGNED_NEGATE_NV,					0x000000e0) \
	BITFLD(NV2A_RC_FINAL0_C_INPUT_SHIFT,							8) \
	BITFLD(NV2A_RC_FINAL0_C_INPUT_MASK,									0x00000f00) \
	 VALUE(NV2A_RC_FINAL0_C_INPUT_ZERO,									0x00000000) \
	 VALUE(NV2A_RC_FINAL0_C_INPUT_CONSTANT_COLOR0_NV,					0x00000100) \
	 VALUE(NV2A_RC_FINAL0_C_INPUT_CONSTANT_COLOR1_NV,					0x00000200) \
	 VALUE(NV2A_RC_FINAL0_C_INPUT_FOG,									0x00000300) \
	 VALUE(NV2A_RC_FINAL0_C_INPUT_PRIMARY_COLOR_NV,						0x00000400) \
	 VALUE(NV2A_RC_FINAL0_C_INPUT_SECONDARY_COLOR_NV,					0x00000500) \
	 VALUE(NV2A_RC_FINAL0_C_INPUT_TEXTURE0_ARB,							0x00000800) \
	 VALUE(NV2A_RC_FINAL0_C_INPUT_TEXTURE1_ARB,							0x00000900) \
	 VALUE(NV2A_RC_FINAL0_C_INPUT_SPARE0_NV,							0x00000c00) \
	 VALUE(NV2A_RC_FINAL0_C_INPUT_SPARE1_NV,							0x00000d00) \
	 VALUE(NV2A_RC_FINAL0_C_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x00000e00) \
	 VALUE(NV2A_RC_FINAL0_C_INPUT_E_TIMES_F_NV,							0x00000f00) \
	BITFLD(NV2A_RC_FINAL0_C_COMPONENT_USAGE,						(1 << 12)) \
	 VALUE(NV2A_RC_FINAL0_C_COMPONENT_USAGE_RGB,						0x00000000) \
	 VALUE(NV2A_RC_FINAL0_C_COMPONENT_USAGE_ALPHA,						0x00001000) \
	BITFLD(NV2A_RC_FINAL0_C_MAPPING_SHIFT,							13) \
	BITFLD(NV2A_RC_FINAL0_C_MAPPING_MASK,								0x0000e000) \
	 VALUE(NV2A_RC_FINAL0_C_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_FINAL0_C_MAPPING_UNSIGNED_INVERT_NV,					0x00002000) \
	 VALUE(NV2A_RC_FINAL0_C_MAPPING_EXPAND_NORMAL_NV,					0x00004000) \
	 VALUE(NV2A_RC_FINAL0_C_MAPPING_EXPAND_NEGATE_NV,					0x00006000) \
	 VALUE(NV2A_RC_FINAL0_C_MAPPING_HALF_BIAS_NORMAL_NV,				0x00008000) \
	 VALUE(NV2A_RC_FINAL0_C_MAPPING_HALF_BIAS_NEGATE_NV,				0x0000a000) \
	 VALUE(NV2A_RC_FINAL0_C_MAPPING_SIGNED_IDENTITY_NV,					0x0000c000) \
	 VALUE(NV2A_RC_FINAL0_C_MAPPING_SIGNED_NEGATE_NV,					0x0000e000) \
	BITFLD(NV2A_RC_FINAL0_B_INPUT_SHIFT,							16) \
	BITFLD(NV2A_RC_FINAL0_B_INPUT_MASK,									0x000f0000) \
	 VALUE(NV2A_RC_FINAL0_B_INPUT_ZERO,									0x00000000) \
	 VALUE(NV2A_RC_FINAL0_B_INPUT_CONSTANT_COLOR0_NV,					0x00010000) \
	 VALUE(NV2A_RC_FINAL0_B_INPUT_CONSTANT_COLOR1_NV,					0x00020000) \
	 VALUE(NV2A_RC_FINAL0_B_INPUT_FOG,									0x00030000) \
	 VALUE(NV2A_RC_FINAL0_B_INPUT_PRIMARY_COLOR_NV,						0x00040000) \
	 VALUE(NV2A_RC_FINAL0_B_INPUT_SECONDARY_COLOR_NV,					0x00050000) \
	 VALUE(NV2A_RC_FINAL0_B_INPUT_TEXTURE0_ARB,							0x00080000) \
	 VALUE(NV2A_RC_FINAL0_B_INPUT_TEXTURE1_ARB,							0x00090000) \
	 VALUE(NV2A_RC_FINAL0_B_INPUT_SPARE0_NV,							0x000c0000) \
	 VALUE(NV2A_RC_FINAL0_B_INPUT_SPARE1_NV,							0x000d0000) \
	 VALUE(NV2A_RC_FINAL0_B_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x000e0000) \
	 VALUE(NV2A_RC_FINAL0_B_INPUT_E_TIMES_F_NV,							0x000f0000) \
	BITFLD(NV2A_RC_FINAL0_B_COMPONENT_USAGE,						(1 << 20)) \
	 VALUE(NV2A_RC_FINAL0_B_COMPONENT_USAGE_RGB,						0x00000000) \
	 VALUE(NV2A_RC_FINAL0_B_COMPONENT_USAGE_ALPHA,						0x00100000) \
	BITFLD(NV2A_RC_FINAL0_B_MAPPING_SHIFT,							21) \
	BITFLD(NV2A_RC_FINAL0_B_MAPPING_MASK,								0x00e00000) \
	 VALUE(NV2A_RC_FINAL0_B_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_FINAL0_B_MAPPING_UNSIGNED_INVERT_NV,					0x00200000) \
	 VALUE(NV2A_RC_FINAL0_B_MAPPING_EXPAND_NORMAL_NV,					0x00400000) \
	 VALUE(NV2A_RC_FINAL0_B_MAPPING_EXPAND_NEGATE_NV,					0x00600000) \
	 VALUE(NV2A_RC_FINAL0_B_MAPPING_HALF_BIAS_NORMAL_NV,				0x00800000) \
	 VALUE(NV2A_RC_FINAL0_B_MAPPING_HALF_BIAS_NEGATE_NV,				0x00a00000) \
	 VALUE(NV2A_RC_FINAL0_B_MAPPING_SIGNED_IDENTITY_NV,					0x00c00000) \
	 VALUE(NV2A_RC_FINAL0_B_MAPPING_SIGNED_NEGATE_NV,					0x00e00000) \
	BITFLD(NV2A_RC_FINAL0_A_INPUT_SHIFT,							24) \
	BITFLD(NV2A_RC_FINAL0_A_INPUT_MASK,									0x0f000000) \
	 VALUE(NV2A_RC_FINAL0_A_INPUT_ZERO,									0x00000000) \
	 VALUE(NV2A_RC_FINAL0_A_INPUT_CONSTANT_COLOR0_NV,					0x01000000) \
	 VALUE(NV2A_RC_FINAL0_A_INPUT_CONSTANT_COLOR1_NV,					0x02000000) \
	 VALUE(NV2A_RC_FINAL0_A_INPUT_FOG,									0x03000000) \
	 VALUE(NV2A_RC_FINAL0_A_INPUT_PRIMARY_COLOR_NV,						0x04000000) \
	 VALUE(NV2A_RC_FINAL0_A_INPUT_SECONDARY_COLOR_NV,					0x05000000) \
	 VALUE(NV2A_RC_FINAL0_A_INPUT_TEXTURE0_ARB,							0x08000000) \
	 VALUE(NV2A_RC_FINAL0_A_INPUT_TEXTURE1_ARB,							0x09000000) \
	 VALUE(NV2A_RC_FINAL0_A_INPUT_SPARE0_NV,							0x0c000000) \
	 VALUE(NV2A_RC_FINAL0_A_INPUT_SPARE1_NV,							0x0d000000) \
	 VALUE(NV2A_RC_FINAL0_A_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x0e000000) \
	 VALUE(NV2A_RC_FINAL0_A_INPUT_E_TIMES_F_NV,							0x0f000000) \
	BITFLD(NV2A_RC_FINAL0_A_COMPONENT_USAGE,						(1 << 28)) \
	 VALUE(NV2A_RC_FINAL0_A_COMPONENT_USAGE_RGB,						0x00000000) \
	 VALUE(NV2A_RC_FINAL0_A_COMPONENT_USAGE_ALPHA,						0x10000000) \
	BITFLD(NV2A_RC_FINAL0_A_MAPPING_SHIFT,							29) \
	BITFLD(NV2A_RC_FINAL0_A_MAPPING_MASK,								0xe0000000) \
	 VALUE(NV2A_RC_FINAL0_A_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_FINAL0_A_MAPPING_UNSIGNED_INVERT_NV,					0x20000000) \
	 VALUE(NV2A_RC_FINAL0_A_MAPPING_EXPAND_NORMAL_NV,					0x40000000) \
	 VALUE(NV2A_RC_FINAL0_A_MAPPING_EXPAND_NEGATE_NV,					0x60000000) \
	 VALUE(NV2A_RC_FINAL0_A_MAPPING_HALF_BIAS_NORMAL_NV,				0x80000000) \
	 VALUE(NV2A_RC_FINAL0_A_MAPPING_HALF_BIAS_NEGATE_NV,				0xa0000000) \
	 VALUE(NV2A_RC_FINAL0_A_MAPPING_SIGNED_IDENTITY_NV,					0xc0000000) \
	 VALUE(NV2A_RC_FINAL0_A_MAPPING_SIGNED_NEGATE_NV,					0xe0000000) \
	METHOD(NV2A_RC_FINAL1,										0x0000028c) \
	BITFLD(NV2A_RC_FINAL1_COLOR_SUM_CLAMP,							(1 <<  7)) \
	BITFLD(NV2A_RC_FINAL1_G_INPUT_SHIFT,							8) \
	BITFLD(NV2A_RC_FINAL1_G_INPUT_MASK,									0x00000f00) \
	 VALUE(NV2A_RC_FINAL1_G_INPUT_ZERO,									0x00000000) \
	 VALUE(NV2A_RC_FINAL1_G_INPUT_CONSTANT_COLOR0_NV,					0x00000100) \
	 VALUE(NV2A_RC_FINAL1_G_INPUT_CONSTANT_COLOR1_NV,					0x00000200) \
	 VALUE(NV2A_RC_FINAL1_G_INPUT_FOG,									0x00000300) \
	 VALUE(NV2A_RC_FINAL1_G_INPUT_PRIMARY_COLOR_NV,						0x00000400) \
	 VALUE(NV2A_RC_FINAL1_G_INPUT_SECONDARY_COLOR_NV,					0x00000500) \
	 VALUE(NV2A_RC_FINAL1_G_INPUT_TEXTURE0_ARB,							0x00000800) \
	 VALUE(NV2A_RC_FINAL1_G_INPUT_TEXTURE1_ARB,							0x00000900) \
	 VALUE(NV2A_RC_FINAL1_G_INPUT_SPARE0_NV,							0x00000c00) \
	 VALUE(NV2A_RC_FINAL1_G_INPUT_SPARE1_NV,							0x00000d00) \
	 VALUE(NV2A_RC_FINAL1_G_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x00000e00) \
	 VALUE(NV2A_RC_FINAL1_G_INPUT_E_TIMES_F_NV,							0x00000f00) \
	BITFLD(NV2A_RC_FINAL1_G_COMPONENT_USAGE,						(1 << 12)) \
	 VALUE(NV2A_RC_FINAL1_G_COMPONENT_USAGE_RGB,						0x00000000) \
	 VALUE(NV2A_RC_FINAL1_G_COMPONENT_USAGE_ALPHA,						0x00001000) \
	BITFLD(NV2A_RC_FINAL1_G_MAPPING_SHIFT,							13) \
	BITFLD(NV2A_RC_FINAL1_G_MAPPING_MASK,								0x0000e000) \
	 VALUE(NV2A_RC_FINAL1_G_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_FINAL1_G_MAPPING_UNSIGNED_INVERT_NV,					0x00002000) \
	 VALUE(NV2A_RC_FINAL1_G_MAPPING_EXPAND_NORMAL_NV,					0x00004000) \
	 VALUE(NV2A_RC_FINAL1_G_MAPPING_EXPAND_NEGATE_NV,					0x00006000) \
	 VALUE(NV2A_RC_FINAL1_G_MAPPING_HALF_BIAS_NORMAL_NV,				0x00008000) \
	 VALUE(NV2A_RC_FINAL1_G_MAPPING_HALF_BIAS_NEGATE_NV,				0x0000a000) \
	 VALUE(NV2A_RC_FINAL1_G_MAPPING_SIGNED_IDENTITY_NV,					0x0000c000) \
	 VALUE(NV2A_RC_FINAL1_G_MAPPING_SIGNED_NEGATE_NV,					0x0000e000) \
	BITFLD(NV2A_RC_FINAL1_F_INPUT_SHIFT,							16) \
	BITFLD(NV2A_RC_FINAL1_F_INPUT_MASK,									0x000f0000) \
	 VALUE(NV2A_RC_FINAL1_F_INPUT_ZERO,									0x00000000) \
	 VALUE(NV2A_RC_FINAL1_F_INPUT_CONSTANT_COLOR0_NV,					0x00010000) \
	 VALUE(NV2A_RC_FINAL1_F_INPUT_CONSTANT_COLOR1_NV,					0x00020000) \
	 VALUE(NV2A_RC_FINAL1_F_INPUT_FOG,									0x00030000) \
	 VALUE(NV2A_RC_FINAL1_F_INPUT_PRIMARY_COLOR_NV,						0x00040000) \
	 VALUE(NV2A_RC_FINAL1_F_INPUT_SECONDARY_COLOR_NV,					0x00050000) \
	 VALUE(NV2A_RC_FINAL1_F_INPUT_TEXTURE0_ARB,							0x00080000) \
	 VALUE(NV2A_RC_FINAL1_F_INPUT_TEXTURE1_ARB,							0x00090000) \
	 VALUE(NV2A_RC_FINAL1_F_INPUT_SPARE0_NV,							0x000c0000) \
	 VALUE(NV2A_RC_FINAL1_F_INPUT_SPARE1_NV,							0x000d0000) \
	 VALUE(NV2A_RC_FINAL1_F_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x000e0000) \
	 VALUE(NV2A_RC_FINAL1_F_INPUT_E_TIMES_F_NV,							0x000f0000) \
	BITFLD(NV2A_RC_FINAL1_F_COMPONENT_USAGE,						(1 << 20)) \
	 VALUE(NV2A_RC_FINAL1_F_COMPONENT_USAGE_RGB,						0x00000000) \
	 VALUE(NV2A_RC_FINAL1_F_COMPONENT_USAGE_ALPHA,						0x00100000) \
	BITFLD(NV2A_RC_FINAL1_F_MAPPING_SHIFT,							21) \
	BITFLD(NV2A_RC_FINAL1_F_MAPPING_MASK,								0x00e00000) \
	 VALUE(NV2A_RC_FINAL1_F_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_FINAL1_F_MAPPING_UNSIGNED_INVERT_NV,					0x00200000) \
	 VALUE(NV2A_RC_FINAL1_F_MAPPING_EXPAND_NORMAL_NV,					0x00400000) \
	 VALUE(NV2A_RC_FINAL1_F_MAPPING_EXPAND_NEGATE_NV,					0x00600000) \
	 VALUE(NV2A_RC_FINAL1_F_MAPPING_HALF_BIAS_NORMAL_NV,				0x00800000) \
	 VALUE(NV2A_RC_FINAL1_F_MAPPING_HALF_BIAS_NEGATE_NV,				0x00a00000) \
	 VALUE(NV2A_RC_FINAL1_F_MAPPING_SIGNED_IDENTITY_NV,					0x00c00000) \
	 VALUE(NV2A_RC_FINAL1_F_MAPPING_SIGNED_NEGATE_NV,					0x00e00000) \
	BITFLD(NV2A_RC_FINAL1_E_INPUT_SHIFT,							24) \
	BITFLD(NV2A_RC_FINAL1_E_INPUT_MASK,									0x0f000000) \
	 VALUE(NV2A_RC_FINAL1_E_INPUT_ZERO,									0x00000000) \
	 VALUE(NV2A_RC_FINAL1_E_INPUT_CONSTANT_COLOR0_NV,					0x01000000) \
	 VALUE(NV2A_RC_FINAL1_E_INPUT_CONSTANT_COLOR1_NV,					0x02000000) \
	 VALUE(NV2A_RC_FINAL1_E_INPUT_FOG,									0x03000000) \
	 VALUE(NV2A_RC_FINAL1_E_INPUT_PRIMARY_COLOR_NV,						0x04000000) \
	 VALUE(NV2A_RC_FINAL1_E_INPUT_SECONDARY_COLOR_NV,					0x05000000) \
	 VALUE(NV2A_RC_FINAL1_E_INPUT_TEXTURE0_ARB,							0x08000000) \
	 VALUE(NV2A_RC_FINAL1_E_INPUT_TEXTURE1_ARB,							0x09000000) \
	 VALUE(NV2A_RC_FINAL1_E_INPUT_SPARE0_NV,							0x0c000000) \
	 VALUE(NV2A_RC_FINAL1_E_INPUT_SPARE1_NV,							0x0d000000) \
	 VALUE(NV2A_RC_FINAL1_E_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x0e000000) \
	 VALUE(NV2A_RC_FINAL1_E_INPUT_E_TIMES_F_NV,							0x0f000000) \
	BITFLD(NV2A_RC_FINAL1_E_COMPONENT_USAGE,						(1 << 28)) \
	 VALUE(NV2A_RC_FINAL1_E_COMPONENT_USAGE_RGB,						0x00000000) \
	 VALUE(NV2A_RC_FINAL1_E_COMPONENT_USAGE_ALPHA,						0x10000000) \
	BITFLD(NV2A_RC_FINAL1_E_MAPPING_SHIFT,							29) \
	BITFLD(NV2A_RC_FINAL1_E_MAPPING_MASK,								0xe0000000) \
	 VALUE(NV2A_RC_FINAL1_E_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_FINAL1_E_MAPPING_UNSIGNED_INVERT_NV,					0x20000000) \
	 VALUE(NV2A_RC_FINAL1_E_MAPPING_EXPAND_NORMAL_NV,					0x40000000) \
	 VALUE(NV2A_RC_FINAL1_E_MAPPING_EXPAND_NEGATE_NV,					0x60000000) \
	 VALUE(NV2A_RC_FINAL1_E_MAPPING_HALF_BIAS_NORMAL_NV,				0x80000000) \
	 VALUE(NV2A_RC_FINAL1_E_MAPPING_HALF_BIAS_NEGATE_NV,				0xa0000000) \
	 VALUE(NV2A_RC_FINAL1_E_MAPPING_SIGNED_IDENTITY_NV,					0xc0000000) \
	 VALUE(NV2A_RC_FINAL1_E_MAPPING_SIGNED_NEGATE_NV,					0xe0000000) \
	METHOD(NV2A_CONTROL0,										0x00000290) \
	METHOD(NV2A_LIGHT_MODEL,									0x00000294) \
	METHOD(NV2A_COLOR_MATERIAL,									0x00000298) \
	METHOD(NV2A_FOG_MODE,										0x0000029c) \
	 VALUE(NV2A_FOG_MODE_EXP,											0x00000800) \
	 VALUE(NV2A_FOG_MODE_EXP_2,											0x00000802) \
	 VALUE(NV2A_FOG_MODE_EXP2,											0x00000803) \
	 VALUE(NV2A_FOG_MODE_LINEAR,										0x00000804) \
	 VALUE(NV2A_FOG_MODE_LINEAR_2,										0x00002601) \
	METHOD(NV2A_FOG_COORD_DIST,									0x000002a0) \
	 VALUE(NV2A_FOG_COORD_DIST_COORD_FALSE,								0x00000000) \
	 VALUE(NV2A_FOG_COORD_DIST_COORD_FRAGMENT_DEPTH_DISTANCE_EYE_RADIAL_NV,	0x00000001) \
	 VALUE(NV2A_FOG_COORD_DIST_COORD_FRAGMENT_DEPTH_DISTANCE_EYE_PLANE_ABSOLUTE_NV,	0x00000002) \
	 VALUE(NV2A_FOG_COORD_DIST_COORD_FOG,								0x00000003) \
	METHOD(NV2A_FOG_ENABLE,										0x000002a4) \
	METHOD(NV2A_FOG_COLOR,										0x000002a8) \
	BITFLD(NV2A_FOG_COLOR_R_SHIFT,									0) \
	BITFLD(NV2A_FOG_COLOR_R_MASK,										0x000000ff) \
	BITFLD(NV2A_FOG_COLOR_G_SHIFT,									8) \
	BITFLD(NV2A_FOG_COLOR_G_MASK,										0x0000ff00) \
	BITFLD(NV2A_FOG_COLOR_B_SHIFT,									16) \
	BITFLD(NV2A_FOG_COLOR_B_MASK,										0x00ff0000) \
	BITFLD(NV2A_FOG_COLOR_A_SHIFT,									24) \
	BITFLD(NV2A_FOG_COLOR_A_MASK,										0xff000000) \
	METHOD(NV2A_VIEWPORT_CLIP_MODE,								0x000002b4) \
	RANGED(NV2A_VIEWPORT_CLIP_HORIZ,							0x000002c0, 0x04, 8) \
	RANGED(NV2A_VIEWPORT_CLIP_VERT,								0x000002e0, 0x04, 8) \
	METHOD(NV2A_ALPHA_FUNC_ENABLE,								0x00000300) \
	METHOD(NV2A_BLEND_FUNC_ENABLE,								0x00000304) \
	METHOD(NV2A_CULL_FACE_ENABLE,								0x00000308) \
	METHOD(NV2A_DEPTH_TEST_ENABLE,								0x0000030c) \
	METHOD(NV2A_DITHER_ENABLE,									0x00000310) \
	METHOD(NV2A_LIGHTING_ENABLE,								0x00000314) \
	METHOD(NV2A_POINT_PARAMETERS_ENABLE,						0x00000318) \
	METHOD(NV2A_POINT_SMOOTH_ENABLE,							0x0000031c) \
	METHOD(NV2A_LINE_SMOOTH_ENABLE,								0x00000320) \
	METHOD(NV2A_POLYGON_SMOOTH_ENABLE,							0x00000324) \
	METHOD(NV2A_SKIN_MODE,										0x00000328) \
	METHOD(NV2A_STENCIL_ENABLE,									0x0000032c) \
	METHOD(NV2A_POLYGON_OFFSET_POINT_ENABLE,					0x00000330) \
	METHOD(NV2A_POLYGON_OFFSET_LINE_ENABLE,						0x00000334) \
	METHOD(NV2A_POLYGON_OFFSET_FILL_ENABLE,						0x00000338) \
	METHOD(NV2A_ALPHA_FUNC_FUNC,								0x0000033c) \
	 VALUE(NV2A_ALPHA_FUNC_FUNC_NEVER,									0x00000200) \
	 VALUE(NV2A_ALPHA_FUNC_FUNC_LESS,									0x00000201) \
	 VALUE(NV2A_ALPHA_FUNC_FUNC_EQUAL,									0x00000202) \
	 VALUE(NV2A_ALPHA_FUNC_FUNC_LEQUAL,									0x00000203) \
	 VALUE(NV2A_ALPHA_FUNC_FUNC_GREATER,								0x00000204) \
	 VALUE(NV2A_ALPHA_FUNC_FUNC_NOTEQUAL,								0x00000205) \
	 VALUE(NV2A_ALPHA_FUNC_FUNC_GEQUAL,									0x00000206) \
	 VALUE(NV2A_ALPHA_FUNC_FUNC_ALWAYS,									0x00000207) \
	METHOD(NV2A_ALPHA_FUNC_REF,									0x00000340) \
	METHOD(NV2A_BLEND_FUNC_SRC,									0x00000344) \
	 VALUE(NV2A_BLEND_FUNC_SRC_ZERO,									0x00000000) \
	 VALUE(NV2A_BLEND_FUNC_SRC_ONE,										0x00000001) \
	 VALUE(NV2A_BLEND_FUNC_SRC_SRC_COLOR,								0x00000300) \
	 VALUE(NV2A_BLEND_FUNC_SRC_ONE_MINUS_SRC_COLOR,						0x00000301) \
	 VALUE(NV2A_BLEND_FUNC_SRC_SRC_ALPHA,								0x00000302) \
	 VALUE(NV2A_BLEND_FUNC_SRC_ONE_MINUS_SRC_ALPHA,						0x00000303) \
	 VALUE(NV2A_BLEND_FUNC_SRC_DST_ALPHA,								0x00000304) \
	 VALUE(NV2A_BLEND_FUNC_SRC_ONE_MINUS_DST_ALPHA,						0x00000305) \
	 VALUE(NV2A_BLEND_FUNC_SRC_DST_COLOR,								0x00000306) \
	 VALUE(NV2A_BLEND_FUNC_SRC_ONE_MINUS_DST_COLOR,						0x00000307) \
	 VALUE(NV2A_BLEND_FUNC_SRC_SRC_ALPHA_SATURATE,						0x00000308) \
	 VALUE(NV2A_BLEND_FUNC_SRC_CONSTANT_COLOR,							0x00008001) \
	 VALUE(NV2A_BLEND_FUNC_SRC_ONE_MINUS_CONSTANT_COLOR,				0x00008002) \
	 VALUE(NV2A_BLEND_FUNC_SRC_CONSTANT_ALPHA,							0x00008003) \
	 VALUE(NV2A_BLEND_FUNC_SRC_ONE_MINUS_CONSTANT_ALPHA,				0x00008004) \
	METHOD(NV2A_BLEND_FUNC_DST,									0x00000348) \
	 VALUE(NV2A_BLEND_FUNC_DST_ZERO,									0x00000000) \
	 VALUE(NV2A_BLEND_FUNC_DST_ONE,										0x00000001) \
	 VALUE(NV2A_BLEND_FUNC_DST_SRC_COLOR,								0x00000300) \
	 VALUE(NV2A_BLEND_FUNC_DST_ONE_MINUS_SRC_COLOR,						0x00000301) \
	 VALUE(NV2A_BLEND_FUNC_DST_SRC_ALPHA,								0x00000302) \
	 VALUE(NV2A_BLEND_FUNC_DST_ONE_MINUS_SRC_ALPHA,						0x00000303) \
	 VALUE(NV2A_BLEND_FUNC_DST_DST_ALPHA,								0x00000304) \
	 VALUE(NV2A_BLEND_FUNC_DST_ONE_MINUS_DST_ALPHA,						0x00000305) \
	 VALUE(NV2A_BLEND_FUNC_DST_DST_COLOR,								0x00000306) \
	 VALUE(NV2A_BLEND_FUNC_DST_ONE_MINUS_DST_COLOR,						0x00000307) \
	 VALUE(NV2A_BLEND_FUNC_DST_SRC_ALPHA_SATURATE,						0x00000308) \
	 VALUE(NV2A_BLEND_FUNC_DST_CONSTANT_COLOR,							0x00008001) \
	 VALUE(NV2A_BLEND_FUNC_DST_ONE_MINUS_CONSTANT_COLOR,				0x00008002) \
	 VALUE(NV2A_BLEND_FUNC_DST_CONSTANT_ALPHA,							0x00008003) \
	 VALUE(NV2A_BLEND_FUNC_DST_ONE_MINUS_CONSTANT_ALPHA,				0x00008004) \
	METHOD(NV2A_BLEND_COLOR,									0x0000034c) \
	BITFLD(NV2A_BLEND_COLOR_B_SHIFT,								0) \
	BITFLD(NV2A_BLEND_COLOR_B_MASK,										0x000000ff) \
	BITFLD(NV2A_BLEND_COLOR_G_SHIFT,								8) \
	BITFLD(NV2A_BLEND_COLOR_G_MASK,										0x0000ff00) \
	BITFLD(NV2A_BLEND_COLOR_R_SHIFT,								16) \
	BITFLD(NV2A_BLEND_COLOR_R_MASK,										0x00ff0000) \
	BITFLD(NV2A_BLEND_COLOR_A_SHIFT,								24) \
	BITFLD(NV2A_BLEND_COLOR_A_MASK,										0xff000000) \
	METHOD(NV2A_BLEND_EQUATION,									0x00000350) \
	 VALUE(NV2A_BLEND_EQUATION_FUNC_ADD,								0x00008006) \
	 VALUE(NV2A_BLEND_EQUATION_MIN,										0x00008007) \
	 VALUE(NV2A_BLEND_EQUATION_MAX,										0x00008008) \
	 VALUE(NV2A_BLEND_EQUATION_FUNC_SUBTRACT,							0x0000800a) \
	 VALUE(NV2A_BLEND_EQUATION_FUNC_REVERSE_SUBTRACT,					0x0000800b) \
	METHOD(NV2A_DEPTH_FUNC,										0x00000354) \
	 VALUE(NV2A_DEPTH_FUNC_NEVER,										0x00000200) \
	 VALUE(NV2A_DEPTH_FUNC_LESS,										0x00000201) \
	 VALUE(NV2A_DEPTH_FUNC_EQUAL,										0x00000202) \
	 VALUE(NV2A_DEPTH_FUNC_LEQUAL,										0x00000203) \
	 VALUE(NV2A_DEPTH_FUNC_GREATER,										0x00000204) \
	 VALUE(NV2A_DEPTH_FUNC_NOTEQUAL,									0x00000205) \
	 VALUE(NV2A_DEPTH_FUNC_GEQUAL,										0x00000206) \
	 VALUE(NV2A_DEPTH_FUNC_ALWAYS,										0x00000207) \
	METHOD(NV2A_COLOR_MASK,										0x00000358) \
	BITFLD(NV2A_COLOR_MASK_B,										(1 <<  0)) \
	BITFLD(NV2A_COLOR_MASK_G,										(1 <<  8)) \
	BITFLD(NV2A_COLOR_MASK_R,										(1 << 16)) \
	BITFLD(NV2A_COLOR_MASK_A,										(1 << 24)) \
	METHOD(NV2A_DEPTH_WRITE_ENABLE,								0x0000035c) \
	METHOD(NV2A_STENCIL_MASK,									0x00000360) \
	METHOD(NV2A_STENCIL_FUNC_FUNC,								0x00000364) \
	 VALUE(NV2A_STENCIL_FUNC_FUNC_NEVER,								0x00000200) \
	 VALUE(NV2A_STENCIL_FUNC_FUNC_LESS,									0x00000201) \
	 VALUE(NV2A_STENCIL_FUNC_FUNC_EQUAL,								0x00000202) \
	 VALUE(NV2A_STENCIL_FUNC_FUNC_LEQUAL,								0x00000203) \
	 VALUE(NV2A_STENCIL_FUNC_FUNC_GREATER,								0x00000204) \
	 VALUE(NV2A_STENCIL_FUNC_FUNC_NOTEQUAL,								0x00000205) \
	 VALUE(NV2A_STENCIL_FUNC_FUNC_GEQUAL,								0x00000206) \
	 VALUE(NV2A_STENCIL_FUNC_FUNC_ALWAYS,								0x00000207) \
	METHOD(NV2A_STENCIL_FUNC_REF,								0x00000368) \
	METHOD(NV2A_STENCIL_FUNC_MASK,								0x0000036c) \
	METHOD(NV2A_STENCIL_OP_FAIL,								0x00000370) \
	 VALUE(NV2A_STENCIL_OP_FAIL_ZERO,									0x00000000) \
	 VALUE(NV2A_STENCIL_OP_FAIL_INVERT,									0x0000150a) \
	 VALUE(NV2A_STENCIL_OP_FAIL_KEEP,									0x00001e00) \
	 VALUE(NV2A_STENCIL_OP_FAIL_REPLACE,								0x00001e01) \
	 VALUE(NV2A_STENCIL_OP_FAIL_INCR,									0x00001e02) \
	 VALUE(NV2A_STENCIL_OP_FAIL_DECR,									0x00001e03) \
	 VALUE(NV2A_STENCIL_OP_FAIL_INCR_WRAP,								0x00008507) \
	 VALUE(NV2A_STENCIL_OP_FAIL_DECR_WRAP,								0x00008508) \
	METHOD(NV2A_STENCIL_OP_ZFAIL,								0x00000374) \
	 VALUE(NV2A_STENCIL_OP_ZFAIL_ZERO,									0x00000000) \
	 VALUE(NV2A_STENCIL_OP_ZFAIL_INVERT,								0x0000150a) \
	 VALUE(NV2A_STENCIL_OP_ZFAIL_KEEP,									0x00001e00) \
	 VALUE(NV2A_STENCIL_OP_ZFAIL_REPLACE,								0x00001e01) \
	 VALUE(NV2A_STENCIL_OP_ZFAIL_INCR,									0x00001e02) \
	 VALUE(NV2A_STENCIL_OP_ZFAIL_DECR,									0x00001e03) \
	 VALUE(NV2A_STENCIL_OP_ZFAIL_INCR_WRAP,								0x00008507) \
	 VALUE(NV2A_STENCIL_OP_ZFAIL_DECR_WRAP,								0x00008508) \
	METHOD(NV2A_STENCIL_OP_ZPASS,								0x00000378) \
	 VALUE(NV2A_STENCIL_OP_ZPASS_ZERO,									0x00000000) \
	 VALUE(NV2A_STENCIL_OP_ZPASS_INVERT,								0x0000150a) \
	 VALUE(NV2A_STENCIL_OP_ZPASS_KEEP,									0x00001e00) \
	 VALUE(NV2A_STENCIL_OP_ZPASS_REPLACE,								0x00001e01) \
	 VALUE(NV2A_STENCIL_OP_ZPASS_INCR,									0x00001e02) \
	 VALUE(NV2A_STENCIL_OP_ZPASS_DECR,									0x00001e03) \
	 VALUE(NV2A_STENCIL_OP_ZPASS_INCR_WRAP,								0x00008507) \
	 VALUE(NV2A_STENCIL_OP_ZPASS_DECR_WRAP,								0x00008508) \
	METHOD(NV2A_SHADE_MODEL,									0x0000037c) \
	 VALUE(NV2A_SHADE_MODEL_FLAT,										0x00001d00) \
	 VALUE(NV2A_SHADE_MODEL_SMOOTH,										0x00001d01) \
	METHOD(NV2A_LINE_WIDTH,										0x00000380) \
	METHOD(NV2A_POLYGON_OFFSET_FACTOR,							0x00000384) \
	METHOD(NV2A_POLYGON_OFFSET_UNITS,							0x00000388) \
	METHOD(NV2A_POLYGON_MODE_FRONT	,							0x0000038c) \
	 VALUE(NV2A_POLYGON_MODE_FRONT_POINT,								0x00001b00) \
	 VALUE(NV2A_POLYGON_MODE_FRONT_LINE,								0x00001b01) \
	 VALUE(NV2A_POLYGON_MODE_FRONT_FILL,								0x00001b02) \
	METHOD(NV2A_POLYGON_MODE_BACK,								0x00000390) \
	 VALUE(NV2A_POLYGON_MODE_BACK_POINT,								0x00001b00) \
	 VALUE(NV2A_POLYGON_MODE_BACK_LINE,									0x00001b01) \
	 VALUE(NV2A_POLYGON_MODE_BACK_FILL,									0x00001b02) \
	METHOD(NV2A_DEPTH_RANGE_NEAR,								0x00000394) \
	METHOD(NV2A_DEPTH_RANGE_FAR,								0x00000398) \
	METHOD(NV2A_CULL_FACE,										0x0000039c) \
	 VALUE(NV2A_CULL_FACE_FRONT,										0x00000404) \
	 VALUE(NV2A_CULL_FACE_BACK,											0x00000405) \
	 VALUE(NV2A_CULL_FACE_FRONT_AND_BACK,								0x00000408) \
	METHOD(NV2A_FRONT_FACE,										0x000003a0) \
	 VALUE(NV2A_FRONT_FACE_CW,											0x00000900) \
	 VALUE(NV2A_FRONT_FACE_CCW,											0x00000901) \
	METHOD(NV2A_NORMALIZE_ENABLE,								0x000003a4) \
	METHOD(NV2A_COLOR_MATERIAL_FRONT_R,							0x000003a8) \
	METHOD(NV2A_COLOR_MATERIAL_FRONT_G,							0x000003ac) \
	METHOD(NV2A_COLOR_MATERIAL_FRONT_B,							0x000003b0) \
	METHOD(NV2A_COLOR_MATERIAL_FRONT_A,							0x000003b4) \
	METHOD(NV2A_SEPARATE_SPECULAR_ENABLE,						0x000003b8) \
	METHOD(NV2A_ENABLED_LIGHTS,									0x000003bc) \
	RANGED(NV2A_TX_GEN_S,										0x000003c0, 0x10, 4) \
	 VALUE(NV2A_TX_GEN_S_FALSE,											0x00000000) \
	 VALUE(NV2A_TX_GEN_S_EYE_LINEAR,									0x00002400) \
	 VALUE(NV2A_TX_GEN_S_OBJECT_LINEAR,									0x00002401) \
	 VALUE(NV2A_TX_GEN_S_SPHERE_MAP,									0x00002402) \
	 VALUE(NV2A_TX_GEN_S_NORMAL_MAP,									0x00008511) \
	 VALUE(NV2A_TX_GEN_S_REFLECTION_MAP,								0x00008512) \
	RANGED(NV2A_TX_GEN_T,										0x000003c4, 0x10, 4) \
	 VALUE(NV2A_TX_GEN_T_FALSE,											0x00000000) \
	 VALUE(NV2A_TX_GEN_T_EYE_LINEAR,									0x00002400) \
	 VALUE(NV2A_TX_GEN_T_OBJECT_LINEAR,									0x00002401) \
	 VALUE(NV2A_TX_GEN_T_SPHERE_MAP,									0x00002402) \
	 VALUE(NV2A_TX_GEN_T_NORMAL_MAP,									0x00008511) \
	 VALUE(NV2A_TX_GEN_T_REFLECTION_MAP,								0x00008512) \
	RANGED(NV2A_TX_GEN_R,										0x000003c8, 0x10, 4) \
	 VALUE(NV2A_TX_GEN_R_FALSE,											0x00000000) \
	 VALUE(NV2A_TX_GEN_R_EYE_LINEAR,									0x00002400) \
	 VALUE(NV2A_TX_GEN_R_OBJECT_LINEAR,									0x00002401) \
	 VALUE(NV2A_TX_GEN_R_SPHERE_MAP,									0x00002402) \
	 VALUE(NV2A_TX_GEN_R_NORMAL_MAP,									0x00008511) \
	 VALUE(NV2A_TX_GEN_R_REFLECTION_MAP,								0x00008512) \
	RANGED(NV2A_TX_GEN_Q,										0x000003cc, 0x10, 4) \
	 VALUE(NV2A_TX_GEN_Q_FALSE,											0x00000000) \
	 VALUE(NV2A_TX_GEN_Q_EYE_LINEAR,									0x00002400) \
	 VALUE(NV2A_TX_GEN_Q_OBJECT_LINEAR,									0x00002401) \
	 VALUE(NV2A_TX_GEN_Q_SPHERE_MAP,									0x00002402) \
	 VALUE(NV2A_TX_GEN_Q_NORMAL_MAP,									0x00008511) \
	 VALUE(NV2A_TX_GEN_Q_REFLECTION_MAP,								0x00008512) \
	RANGED(NV2A_TX_MATRIX_ENABLE,								0x00000420, 0x04, 4) \
	METHOD(NV2A_POINT_SIZE,										0x0000043c) \
	RANGED(NV2A_PROJECTION_MATRIX,								0x00000440, 0x04, 16) \
	RANGED(NV2A_MODELVIEW0_MATRIX,								0x00000480, 0x04, 16) \
	RANGED(NV2A_MODELVIEW1_MATRIX,								0x000004c0, 0x04, 16) \
	RANGED(NV2A_MODELVIEW2_MATRIX,								0x00000500, 0x04, 16) \
	RANGED(NV2A_MODELVIEW3_MATRIX,								0x00000540, 0x04, 16) \
	RANGED(NV2A_INVERSE_MODELVIEW0_MATRIX,						0x00000580, 0x04, 16) \
	RANGED(NV2A_INVERSE_MODELVIEW1_MATRIX,						0x000005c0, 0x04, 16) \
	RANGED(NV2A_INVERSE_MODELVIEW2_MATRIX,						0x00000600, 0x04, 16) \
	RANGED(NV2A_INVERSE_MODELVIEW3_MATRIX,						0x00000640, 0x04, 16) \
	RANGED(NV2A_COMPOSITE_MATRIX,								0x00000680, 0x04, 16) \
	RANGED(NV2A_TX0_MATRIX,										0x000006c0, 0x04, 16) \
	RANGED(NV2A_TX1_MATRIX,										0x00000700, 0x04, 16) \
	RANGED(NV2A_TX2_MATRIX,										0x00000740, 0x04, 16) \
	RANGED(NV2A_TX3_MATRIX,										0x00000780, 0x04, 16) \
	RANGED(NV2A_TX0_CLIP_PLANE_A,								0x00000840, 0x10, 4) \
	RANGED(NV2A_TX0_CLIP_PLANE_B,								0x00000844, 0x10, 4) \
	RANGED(NV2A_TX0_CLIP_PLANE_C,								0x00000848, 0x10, 4) \
	RANGED(NV2A_TX0_CLIP_PLANE_D,								0x0000084c, 0x10, 4) \
	RANGED(NV2A_TX1_CLIP_PLANE_A,								0x00000880, 0x10, 4) \
	RANGED(NV2A_TX1_CLIP_PLANE_B,								0x00000884, 0x10, 4) \
	RANGED(NV2A_TX1_CLIP_PLANE_C,								0x00000888, 0x10, 4) \
	RANGED(NV2A_TX1_CLIP_PLANE_D,								0x0000088c, 0x10, 4) \
	RANGED(NV2A_TX2_CLIP_PLANE_A,								0x000008c0, 0x10, 4) \
	RANGED(NV2A_TX2_CLIP_PLANE_B,								0x000008c4, 0x10, 4) \
	RANGED(NV2A_TX2_CLIP_PLANE_C,								0x000008c8, 0x10, 4) \
	RANGED(NV2A_TX2_CLIP_PLANE_D,								0x000008cc, 0x10, 4) \
	RANGED(NV2A_TX3_CLIP_PLANE_A,								0x00000900, 0x10, 4) \
	RANGED(NV2A_TX3_CLIP_PLANE_B,								0x00000904, 0x10, 4) \
	RANGED(NV2A_TX3_CLIP_PLANE_C,								0x00000908, 0x10, 4) \
	RANGED(NV2A_TX3_CLIP_PLANE_D,								0x0000090c, 0x10, 4) \
	METHOD(NV2A_FOG_EQUATION_CONSTANT,							0x000009c0) \
	METHOD(NV2A_FOG_EQUATION_LINEAR,							0x000009c4) \
	METHOD(NV2A_FOG_EQUATION_QUADRATIC,							0x000009c8) \
	METHOD(NV2A_SET_TEXGEN_VIEW_MODEL,							0X000009cc) \
	RANGED(NV2A_FOG_PLANE,										0x000009d0, 0x04, 4) \
	RANGED(NV2A_FRONT_MATERIAL_SHININESS,						0x000009e0, 0x04, 6) \
	METHOD(NV2A_SWATH_WIDTH,									0x000009f8) \
	METHOD(NV2A_FLAT_SHADE_OP,									0x000009fc) \
	METHOD(NV2A_LIGHT_MODEL_FRONT_SIDE_PRODUCT_AMBIENT_PLUS_EMISSION_R,	0x00000a10) \
	METHOD(NV2A_LIGHT_MODEL_FRONT_SIDE_PRODUCT_AMBIENT_PLUS_EMISSION_G,	0x00000a14) \
	METHOD(NV2A_LIGHT_MODEL_FRONT_SIDE_PRODUCT_AMBIENT_PLUS_EMISSION_B,	0x00000a18) \
	METHOD(NV2A_VIEWPORT_SCALE0_X,								0x00000a20) \
	METHOD(NV2A_VIEWPORT_SCALE0_Y,								0x00000a24) \
	METHOD(NV2A_VIEWPORT_SCALE0_Z,								0x00000a28) \
	METHOD(NV2A_VIEWPORT_SCALE0_W,								0x00000a2c) \
	RANGED(NV2A_POINT_PARAMETER,								0x00000a30, 0x04, 8) \
	RANGED(NV2A_EYE_POSITION,									0x00000a50, 0x04, 4) \
	RANGED(NV2A_RC_CONSTANT_COLOR0,								0x00000a60, 0x04, 8) \
	BITFLD(NV2A_RC_CONSTANT_COLOR0_B_SHIFT,							0) \
	BITFLD(NV2A_RC_CONSTANT_COLOR0_B_MASK,								0x000000ff) \
	BITFLD(NV2A_RC_CONSTANT_COLOR0_G_SHIFT,							8) \
	BITFLD(NV2A_RC_CONSTANT_COLOR0_G_MASK,								0x0000ff00) \
	BITFLD(NV2A_RC_CONSTANT_COLOR0_R_SHIFT,							16) \
	BITFLD(NV2A_RC_CONSTANT_COLOR0_R_MASK,								0x00ff0000) \
	BITFLD(NV2A_RC_CONSTANT_COLOR0_A_SHIFT,							24) \
	BITFLD(NV2A_RC_CONSTANT_COLOR0_A_MASK,								0xff000000) \
	RANGED(NV2A_RC_CONSTANT_COLOR1,								0x00000a80, 0x04, 8) \
	BITFLD(NV2A_RC_CONSTANT_COLOR1_B_SHIFT,							0) \
	BITFLD(NV2A_RC_CONSTANT_COLOR1_B_MASK,								0x000000ff) \
	BITFLD(NV2A_RC_CONSTANT_COLOR1_G_SHIFT,							8) \
	BITFLD(NV2A_RC_CONSTANT_COLOR1_G_MASK,								0x0000ff00) \
	BITFLD(NV2A_RC_CONSTANT_COLOR1_R_SHIFT,							16) \
	BITFLD(NV2A_RC_CONSTANT_COLOR1_R_MASK,								0x00ff0000) \
	BITFLD(NV2A_RC_CONSTANT_COLOR1_A_SHIFT,							24) \
	BITFLD(NV2A_RC_CONSTANT_COLOR1_A_MASK,								0xff000000) \
	RANGED(NV2A_RC_OUT_ALPHA,									0x00000aa0, 0x04, 8) \
	BITFLD(NV2A_RC_OUT_ALPHA_CD_OUTPUT_SHIFT,						0) \
	BITFLD(NV2A_RC_OUT_ALPHA_CD_OUTPUT_MASK,							0x0000000f) \
	 VALUE(NV2A_RC_OUT_ALPHA_CD_OUTPUT_ZERO,							0x00000000) \
	 VALUE(NV2A_RC_OUT_ALPHA_CD_OUTPUT_CONSTANT_COLOR0_NV,				0x00000001) \
	 VALUE(NV2A_RC_OUT_ALPHA_CD_OUTPUT_CONSTANT_COLOR1_NV,				0x00000002) \
	 VALUE(NV2A_RC_OUT_ALPHA_CD_OUTPUT_FOG,								0x00000003) \
	 VALUE(NV2A_RC_OUT_ALPHA_CD_OUTPUT_PRIMARY_COLOR_NV,				0x00000004) \
	 VALUE(NV2A_RC_OUT_ALPHA_CD_OUTPUT_SECONDARY_COLOR_NV,				0x00000005) \
	 VALUE(NV2A_RC_OUT_ALPHA_CD_OUTPUT_TEXTURE0_ARB,					0x00000008) \
	 VALUE(NV2A_RC_OUT_ALPHA_CD_OUTPUT_TEXTURE1_ARB,					0x00000009) \
	 VALUE(NV2A_RC_OUT_ALPHA_CD_OUTPUT_SPARE0_NV,						0x0000000c) \
	 VALUE(NV2A_RC_OUT_ALPHA_CD_OUTPUT_SPARE1_NV,						0x0000000d) \
	 VALUE(NV2A_RC_OUT_ALPHA_CD_OUTPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,	0x0000000e) \
	 VALUE(NV2A_RC_OUT_ALPHA_CD_OUTPUT_E_TIMES_F_NV,					0x0000000f) \
	BITFLD(NV2A_RC_OUT_ALPHA_AB_OUTPUT_SHIFT,						4) \
	BITFLD(NV2A_RC_OUT_ALPHA_AB_OUTPUT_MASK,							0x000000f0) \
	 VALUE(NV2A_RC_OUT_ALPHA_AB_OUTPUT_ZERO,							0x00000000) \
	 VALUE(NV2A_RC_OUT_ALPHA_AB_OUTPUT_CONSTANT_COLOR0_NV,				0x00000010) \
	 VALUE(NV2A_RC_OUT_ALPHA_AB_OUTPUT_CONSTANT_COLOR1_NV,				0x00000020) \
	 VALUE(NV2A_RC_OUT_ALPHA_AB_OUTPUT_FOG,								0x00000030) \
	 VALUE(NV2A_RC_OUT_ALPHA_AB_OUTPUT_PRIMARY_COLOR_NV,				0x00000040) \
	 VALUE(NV2A_RC_OUT_ALPHA_AB_OUTPUT_SECONDARY_COLOR_NV,				0x00000050) \
	 VALUE(NV2A_RC_OUT_ALPHA_AB_OUTPUT_TEXTURE0_ARB,					0x00000080) \
	 VALUE(NV2A_RC_OUT_ALPHA_AB_OUTPUT_TEXTURE1_ARB,					0x00000090) \
	 VALUE(NV2A_RC_OUT_ALPHA_AB_OUTPUT_SPARE0_NV,						0x000000c0) \
	 VALUE(NV2A_RC_OUT_ALPHA_AB_OUTPUT_SPARE1_NV,						0x000000d0) \
	 VALUE(NV2A_RC_OUT_ALPHA_AB_OUTPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,	0x000000e0) \
	 VALUE(NV2A_RC_OUT_ALPHA_AB_OUTPUT_E_TIMES_F_NV,					0x000000f0) \
	BITFLD(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_SHIFT,						8) \
	BITFLD(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_MASK,							0x00000f00) \
	 VALUE(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_ZERO,							0x00000000) \
	 VALUE(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_CONSTANT_COLOR0_NV,				0x00000100) \
	 VALUE(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_CONSTANT_COLOR1_NV,				0x00000200) \
	 VALUE(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_FOG,							0x00000300) \
	 VALUE(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_PRIMARY_COLOR_NV,				0x00000400) \
	 VALUE(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_SECONDARY_COLOR_NV,				0x00000500) \
	 VALUE(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_TEXTURE0_ARB,					0x00000800) \
	 VALUE(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_TEXTURE1_ARB,					0x00000900) \
	 VALUE(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_SPARE0_NV,						0x00000c00) \
	 VALUE(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_SPARE1_NV,						0x00000d00) \
	 VALUE(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,	0x00000e00) \
	 VALUE(NV2A_RC_OUT_ALPHA_SUM_OUTPUT_E_TIMES_F_NV,					0x00000f00) \
	BITFLD(NV2A_RC_OUT_ALPHA_CD_DOT_PRODUCT,						(1 << 12)) \
	BITFLD(NV2A_RC_OUT_ALPHA_AB_DOT_PRODUCT,						(1 << 13)) \
	BITFLD(NV2A_RC_OUT_ALPHA_MUX_SUM,								(1 << 14)) \
	BITFLD(NV2A_RC_OUT_ALPHA_BIAS,									(1 << 15)) \
	 VALUE(NV2A_RC_OUT_ALPHA_BIAS_NONE,									0x00000000) \
	 VALUE(NV2A_RC_OUT_ALPHA_BIAS_BIAS_BY_NEGATIVE_ONE_HALF_NV,			0x00008000) \
	BITFLD(NV2A_RC_OUT_ALPHA_SCALE_SHIFT,							17) \
	BITFLD(NV2A_RC_OUT_ALPHA_SCALE_MASK,								0x000C0000) \
	 VALUE(NV2A_RC_OUT_ALPHA_SCALE_NONE,								0x00000000) \
	 VALUE(NV2A_RC_OUT_ALPHA_SCALE_SCALE_BY_TWO_NV,						0x00020000) \
	 VALUE(NV2A_RC_OUT_ALPHA_SCALE_SCALE_BY_FOUR_NV,					0x00040000) \
	 VALUE(NV2A_RC_OUT_ALPHA_SCALE_SCALE_BY_ONE_HALF_NV,				0x00060000) \
	RANGED(NV2A_RC_IN_RGB,										0x00000ac0, 0x04, 8) \
	BITFLD(NV2A_RC_IN_RGB_D_INPUT_SHIFT,							0) \
	BITFLD(NV2A_RC_IN_RGB_D_INPUT_MASK,									0x0000000f) \
	 VALUE(NV2A_RC_IN_RGB_D_INPUT_ZERO,									0x00000000) \
	 VALUE(NV2A_RC_IN_RGB_D_INPUT_CONSTANT_COLOR0_NV,					0x00000001) \
	 VALUE(NV2A_RC_IN_RGB_D_INPUT_CONSTANT_COLOR1_NV,					0x00000002) \
	 VALUE(NV2A_RC_IN_RGB_D_INPUT_FOG,									0x00000003) \
	 VALUE(NV2A_RC_IN_RGB_D_INPUT_PRIMARY_COLOR_NV,						0x00000004) \
	 VALUE(NV2A_RC_IN_RGB_D_INPUT_SECONDARY_COLOR_NV,					0x00000005) \
	 VALUE(NV2A_RC_IN_RGB_D_INPUT_TEXTURE0_ARB,							0x00000008) \
	 VALUE(NV2A_RC_IN_RGB_D_INPUT_TEXTURE1_ARB,							0x00000009) \
	 VALUE(NV2A_RC_IN_RGB_D_INPUT_SPARE0_NV,							0x0000000c) \
	 VALUE(NV2A_RC_IN_RGB_D_INPUT_SPARE1_NV,							0x0000000d) \
	 VALUE(NV2A_RC_IN_RGB_D_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x0000000e) \
	 VALUE(NV2A_RC_IN_RGB_D_INPUT_E_TIMES_F_NV,							0x0000000f) \
	BITFLD(NV2A_RC_IN_RGB_D_COMPONENT_USAGE,						(1 <<  4)) \
	 VALUE(NV2A_RC_IN_RGB_D_COMPONENT_USAGE_RGB,						0x00000000) \
	 VALUE(NV2A_RC_IN_RGB_D_COMPONENT_USAGE_ALPHA,						0x00000010) \
	BITFLD(NV2A_RC_IN_RGB_D_MAPPING_SHIFT,							5) \
	BITFLD(NV2A_RC_IN_RGB_D_MAPPING_MASK,								0x000000e0) \
	 VALUE(NV2A_RC_IN_RGB_D_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_IN_RGB_D_MAPPING_UNSIGNED_INVERT_NV,					0x00000020) \
	 VALUE(NV2A_RC_IN_RGB_D_MAPPING_EXPAND_NORMAL_NV,					0x00000040) \
	 VALUE(NV2A_RC_IN_RGB_D_MAPPING_EXPAND_NEGATE_NV,					0x00000060) \
	 VALUE(NV2A_RC_IN_RGB_D_MAPPING_HALF_BIAS_NORMAL_NV,				0x00000080) \
	 VALUE(NV2A_RC_IN_RGB_D_MAPPING_HALF_BIAS_NEGATE_NV,				0x000000a0) \
	 VALUE(NV2A_RC_IN_RGB_D_MAPPING_SIGNED_IDENTITY_NV,					0x000000c0) \
	 VALUE(NV2A_RC_IN_RGB_D_MAPPING_SIGNED_NEGATE_NV,					0x000000e0) \
	BITFLD(NV2A_RC_IN_RGB_C_INPUT_SHIFT,							8) \
	BITFLD(NV2A_RC_IN_RGB_C_INPUT_MASK,									0x00000f00) \
	 VALUE(NV2A_RC_IN_RGB_C_INPUT_ZERO,									0x00000000) \
	 VALUE(NV2A_RC_IN_RGB_C_INPUT_CONSTANT_COLOR0_NV,					0x00000100) \
	 VALUE(NV2A_RC_IN_RGB_C_INPUT_CONSTANT_COLOR1_NV,					0x00000200) \
	 VALUE(NV2A_RC_IN_RGB_C_INPUT_FOG,									0x00000300) \
	 VALUE(NV2A_RC_IN_RGB_C_INPUT_PRIMARY_COLOR_NV,						0x00000400) \
	 VALUE(NV2A_RC_IN_RGB_C_INPUT_SECONDARY_COLOR_NV,					0x00000500) \
	 VALUE(NV2A_RC_IN_RGB_C_INPUT_TEXTURE0_ARB,							0x00000800) \
	 VALUE(NV2A_RC_IN_RGB_C_INPUT_TEXTURE1_ARB,							0x00000900) \
	 VALUE(NV2A_RC_IN_RGB_C_INPUT_SPARE0_NV,							0x00000c00) \
	 VALUE(NV2A_RC_IN_RGB_C_INPUT_SPARE1_NV,							0x00000d00) \
	 VALUE(NV2A_RC_IN_RGB_C_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x00000e00) \
	 VALUE(NV2A_RC_IN_RGB_C_INPUT_E_TIMES_F_NV,							0x00000f00) \
	BITFLD(NV2A_RC_IN_RGB_C_COMPONENT_USAGE,						(1 << 12)) \
	 VALUE(NV2A_RC_IN_RGB_C_COMPONENT_USAGE_RGB,						0x00000000) \
	 VALUE(NV2A_RC_IN_RGB_C_COMPONENT_USAGE_ALPHA,						0x00001000) \
	BITFLD(NV2A_RC_IN_RGB_C_MAPPING_SHIFT,							13) \
	BITFLD(NV2A_RC_IN_RGB_C_MAPPING_MASK,								0x0000e000) \
	 VALUE(NV2A_RC_IN_RGB_C_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_IN_RGB_C_MAPPING_UNSIGNED_INVERT_NV,					0x00002000) \
	 VALUE(NV2A_RC_IN_RGB_C_MAPPING_EXPAND_NORMAL_NV,					0x00004000) \
	 VALUE(NV2A_RC_IN_RGB_C_MAPPING_EXPAND_NEGATE_NV,					0x00006000) \
	 VALUE(NV2A_RC_IN_RGB_C_MAPPING_HALF_BIAS_NORMAL_NV,				0x00008000) \
	 VALUE(NV2A_RC_IN_RGB_C_MAPPING_HALF_BIAS_NEGATE_NV,				0x0000a000) \
	 VALUE(NV2A_RC_IN_RGB_C_MAPPING_SIGNED_IDENTITY_NV,					0x0000c000) \
	 VALUE(NV2A_RC_IN_RGB_C_MAPPING_SIGNED_NEGATE_NV,					0x0000e000) \
	BITFLD(NV2A_RC_IN_RGB_B_INPUT_SHIFT,							16) \
	BITFLD(NV2A_RC_IN_RGB_B_INPUT_MASK,									0x000f0000) \
	 VALUE(NV2A_RC_IN_RGB_B_INPUT_ZERO,									0x00000000) \
	 VALUE(NV2A_RC_IN_RGB_B_INPUT_CONSTANT_COLOR0_NV,					0x00010000) \
	 VALUE(NV2A_RC_IN_RGB_B_INPUT_CONSTANT_COLOR1_NV,					0x00020000) \
	 VALUE(NV2A_RC_IN_RGB_B_INPUT_FOG,									0x00030000) \
	 VALUE(NV2A_RC_IN_RGB_B_INPUT_PRIMARY_COLOR_NV,						0x00040000) \
	 VALUE(NV2A_RC_IN_RGB_B_INPUT_SECONDARY_COLOR_NV,					0x00050000) \
	 VALUE(NV2A_RC_IN_RGB_B_INPUT_TEXTURE0_ARB,							0x00080000) \
	 VALUE(NV2A_RC_IN_RGB_B_INPUT_TEXTURE1_ARB,							0x00090000) \
	 VALUE(NV2A_RC_IN_RGB_B_INPUT_SPARE0_NV,							0x000c0000) \
	 VALUE(NV2A_RC_IN_RGB_B_INPUT_SPARE1_NV,							0x000d0000) \
	 VALUE(NV2A_RC_IN_RGB_B_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x000e0000) \
	 VALUE(NV2A_RC_IN_RGB_B_INPUT_E_TIMES_F_NV,							0x000f0000) \
	BITFLD(NV2A_RC_IN_RGB_B_COMPONENT_USAGE,						(1 << 20)) \
	 VALUE(NV2A_RC_IN_RGB_B_COMPONENT_USAGE_RGB,						0x00000000) \
	 VALUE(NV2A_RC_IN_RGB_B_COMPONENT_USAGE_ALPHA,						0x00100000) \
	BITFLD(NV2A_RC_IN_RGB_B_MAPPING_SHIFT,							21) \
	BITFLD(NV2A_RC_IN_RGB_B_MAPPING_MASK,								0x00e00000) \
	 VALUE(NV2A_RC_IN_RGB_B_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_IN_RGB_B_MAPPING_UNSIGNED_INVERT_NV,					0x00200000) \
	 VALUE(NV2A_RC_IN_RGB_B_MAPPING_EXPAND_NORMAL_NV,					0x00400000) \
	 VALUE(NV2A_RC_IN_RGB_B_MAPPING_EXPAND_NEGATE_NV,					0x00600000) \
	 VALUE(NV2A_RC_IN_RGB_B_MAPPING_HALF_BIAS_NORMAL_NV,				0x00800000) \
	 VALUE(NV2A_RC_IN_RGB_B_MAPPING_HALF_BIAS_NEGATE_NV,				0x00a00000) \
	 VALUE(NV2A_RC_IN_RGB_B_MAPPING_SIGNED_IDENTITY_NV,					0x00c00000) \
	 VALUE(NV2A_RC_IN_RGB_B_MAPPING_SIGNED_NEGATE_NV,					0x00e00000) \
	BITFLD(NV2A_RC_IN_RGB_A_INPUT_SHIFT,							24) \
	BITFLD(NV2A_RC_IN_RGB_A_INPUT_MASK,									0x0f000000) \
	 VALUE(NV2A_RC_IN_RGB_A_INPUT_ZERO,									0x00000000) \
	 VALUE(NV2A_RC_IN_RGB_A_INPUT_CONSTANT_COLOR0_NV,					0x01000000) \
	 VALUE(NV2A_RC_IN_RGB_A_INPUT_CONSTANT_COLOR1_NV,					0x02000000) \
	 VALUE(NV2A_RC_IN_RGB_A_INPUT_FOG,									0x03000000) \
	 VALUE(NV2A_RC_IN_RGB_A_INPUT_PRIMARY_COLOR_NV,						0x04000000) \
	 VALUE(NV2A_RC_IN_RGB_A_INPUT_SECONDARY_COLOR_NV,					0x05000000) \
	 VALUE(NV2A_RC_IN_RGB_A_INPUT_TEXTURE0_ARB,							0x08000000) \
	 VALUE(NV2A_RC_IN_RGB_A_INPUT_TEXTURE1_ARB,							0x09000000) \
	 VALUE(NV2A_RC_IN_RGB_A_INPUT_SPARE0_NV,							0x0c000000) \
	 VALUE(NV2A_RC_IN_RGB_A_INPUT_SPARE1_NV,							0x0d000000) \
	 VALUE(NV2A_RC_IN_RGB_A_INPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,		0x0e000000) \
	 VALUE(NV2A_RC_IN_RGB_A_INPUT_E_TIMES_F_NV,							0x0f000000) \
	BITFLD(NV2A_RC_IN_RGB_A_COMPONENT_USAGE,						(1 << 28)) \
	 VALUE(NV2A_RC_IN_RGB_A_COMPONENT_USAGE_RGB,						0x00000000) \
	 VALUE(NV2A_RC_IN_RGB_A_COMPONENT_USAGE_ALPHA,						0x10000000) \
	BITFLD(NV2A_RC_IN_RGB_A_MAPPING_SHIFT,							29) \
	BITFLD(NV2A_RC_IN_RGB_A_MAPPING_MASK,								0xe0000000) \
	 VALUE(NV2A_RC_IN_RGB_A_MAPPING_UNSIGNED_IDENTITY_NV,				0x00000000) \
	 VALUE(NV2A_RC_IN_RGB_A_MAPPING_UNSIGNED_INVERT_NV,					0x20000000) \
	 VALUE(NV2A_RC_IN_RGB_A_MAPPING_EXPAND_NORMAL_NV,					0x40000000) \
	 VALUE(NV2A_RC_IN_RGB_A_MAPPING_EXPAND_NEGATE_NV,					0x60000000) \
	 VALUE(NV2A_RC_IN_RGB_A_MAPPING_HALF_BIAS_NORMAL_NV,				0x80000000) \
	 VALUE(NV2A_RC_IN_RGB_A_MAPPING_HALF_BIAS_NEGATE_NV,				0xa0000000) \
	 VALUE(NV2A_RC_IN_RGB_A_MAPPING_SIGNED_IDENTITY_NV,					0xc0000000) \
	 VALUE(NV2A_RC_IN_RGB_A_MAPPING_SIGNED_NEGATE_NV,					0xe0000000) \
	METHOD(NV2A_VIEWPORT_SCALE1_X,								0x00000af0) \
	METHOD(NV2A_VIEWPORT_SCALE1_Y,								0x00000af4) \
	METHOD(NV2A_VIEWPORT_SCALE1_Z,								0x00000af8) \
	METHOD(NV2A_VIEWPORT_SCALE1_W,								0x00000afc) \
	RANGED(NV2A_VP_UPLOAD_INST,									0x00000b00, 0x04, 4) \
	RANGED(NV2A_VP_UPLOAD_CONST,								0x00000b80, 0x04, 4) \
	RANGED(NV2A_LIGHT_BACK_SIDE_PRODUCT_AMBIENT_R,				0x00000c00, 0x40, 8) \
	RANGED(NV2A_LIGHT_BACK_SIDE_PRODUCT_AMBIENT_G,				0x00000c04, 0x40, 8) \
	RANGED(NV2A_LIGHT_BACK_SIDE_PRODUCT_AMBIENT_B,				0x00000c08, 0x40, 8) \
	RANGED(NV2A_LIGHT_BACK_SIDE_PRODUCT_DIFFUSE_R,				0x00000c0c, 0x40, 8) \
	RANGED(NV2A_LIGHT_BACK_SIDE_PRODUCT_DIFFUSE_G,				0x00000c10, 0x40, 8) \
	RANGED(NV2A_LIGHT_BACK_SIDE_PRODUCT_DIFFUSE_B,				0x00000c14, 0x40, 8) \
	RANGED(NV2A_LIGHT_BACK_SIDE_PRODUCT_SPECULAR_R,				0x00000c18, 0x40, 8) \
	RANGED(NV2A_LIGHT_BACK_SIDE_PRODUCT_SPECULAR_G,				0x00000c1c, 0x40, 8) \
	RANGED(NV2A_LIGHT_BACK_SIDE_PRODUCT_SPECULAR_B,				0x00000c20, 0x40, 8) \
	RANGED(NV2A_LIGHT_FRONT_SIDE_PRODUCT_AMBIENT_R,				0x00001000, 0x80, 8) \
	RANGED(NV2A_LIGHT_FRONT_SIDE_PRODUCT_AMBIENT_G,				0x00001004, 0x80, 8) \
	RANGED(NV2A_LIGHT_FRONT_SIDE_PRODUCT_AMBIENT_B,				0x00001008, 0x80, 8) \
	RANGED(NV2A_LIGHT_FRONT_SIDE_PRODUCT_DIFFUSE_R,				0x0000100c, 0x80, 8) \
	RANGED(NV2A_LIGHT_FRONT_SIDE_PRODUCT_DIFFUSE_G,				0x00001010, 0x80, 8) \
	RANGED(NV2A_LIGHT_FRONT_SIDE_PRODUCT_DIFFUSE_B,				0x00001014, 0x80, 8) \
	RANGED(NV2A_LIGHT_FRONT_SIDE_PRODUCT_SPECULAR_R,			0x00001018, 0x80, 8) \
	RANGED(NV2A_LIGHT_FRONT_SIDE_PRODUCT_SPECULAR_G,			0x0000101c, 0x80, 8) \
	RANGED(NV2A_LIGHT_FRONT_SIDE_PRODUCT_SPECULAR_B,			0x00001020, 0x80, 8) \
	RANGED(NV2A_LIGHT_LOCAL_RANGE,			                    0x00001024, 0x80, 8) \
	RANGED(NV2A_LIGHT_HALF_VECTOR_X,							0x00001028, 0x80, 8) \
	RANGED(NV2A_LIGHT_HALF_VECTOR_Y,							0x0000102c, 0x80, 8) \
	RANGED(NV2A_LIGHT_HALF_VECTOR_Z,							0x00001030, 0x80, 8) \
	RANGED(NV2A_LIGHT_DIRECTION_X,								0x00001034, 0x80, 8) \
	RANGED(NV2A_LIGHT_DIRECTION_Y,								0x00001038, 0x80, 8) \
	RANGED(NV2A_LIGHT_DIRECTION_Z,								0x0000103c, 0x80, 8) \
	RANGED(NV2A_LIGHT_SPOT_FALLOFF_X,							0x00001040, 0x80, 8) \
	RANGED(NV2A_LIGHT_SPOT_FALLOFF_Y,							0x00001044, 0x80, 8) \
	RANGED(NV2A_LIGHT_SPOT_FALLOFF_Z,							0x00001048, 0x80, 8) \
	RANGED(NV2A_LIGHT_SPOT_DIRECTION_X,							0x0000104c, 0x80, 8) \
	RANGED(NV2A_LIGHT_SPOT_DIRECTION_Y,							0x00001050, 0x80, 8) \
	RANGED(NV2A_LIGHT_SPOT_DIRECTION_Z,							0x00001054, 0x80, 8) \
	RANGED(NV2A_LIGHT_POSITION_X,								0x0000105c, 0x80, 8) \
	RANGED(NV2A_LIGHT_POSITION_Y,								0x00001060, 0x80, 8) \
	RANGED(NV2A_LIGHT_POSITION_Z,								0x00001064, 0x80, 8) \
	RANGED(NV2A_LIGHT_CONSTANT_ATTENUATION,						0x00001068, 0x80, 8) \
	RANGED(NV2A_LIGHT_LINEAR_ATTENUATION,						0x0000106c, 0x80, 8) \
	RANGED(NV2A_LIGHT_QUADRATIC_ATTENUATION,					0x00001070, 0x80, 8) \
	METHOD(NV2A_POLYGON_STIPPLE_ENABLE,							0x0000147c) \
	RANGED(NV2A_POLYGON_STIPPLE_PATTERN,						0x00001480, 0x04, 32) \
	METHOD(NV2A_VERTEX_POS_3F_X,								0x00001500) \
	METHOD(NV2A_VERTEX_POS_3F_Y,								0x00001504) \
	METHOD(NV2A_VERTEX_POS_3F_Z,								0x00001508) \
	METHOD(NV2A_VERTEX_POS_4F_X,								0x00001518) \
	METHOD(NV2A_VERTEX_POS_4F_Y,								0x0000151c) \
	METHOD(NV2A_VERTEX_POS_4F_Z,								0x00001520) \
	METHOD(NV2A_VERTEX_POS_3I_XY,								0x00001528) \
	BITFLD(NV2A_VERTEX_POS_3I_XY_X_SHIFT,							0) \
	BITFLD(NV2A_VERTEX_POS_3I_XY_X_MASK,								0x0000ffff) \
	BITFLD(NV2A_VERTEX_POS_3I_XY_Y_SHIFT,							16) \
	BITFLD(NV2A_VERTEX_POS_3I_XY_Y_MASK,								0xffff0000) \
	METHOD(NV2A_VERTEX_POS_3I_Z,								0x0000152c) \
	BITFLD(NV2A_VERTEX_POS_3I_Z_Z_SHIFT,							0) \
	BITFLD(NV2A_VERTEX_POS_3I_Z_Z_MASK,									0x0000ffff) \
	METHOD(NV2A_VERTEX_NOR_3F_X,								0x00001530) \
	METHOD(NV2A_VERTEX_NOR_3F_Y,								0x00001534) \
	METHOD(NV2A_VERTEX_NOR_3F_Z,								0x00001538) \
	METHOD(NV2A_VERTEX_NOR_3I_XY,								0x00001540) \
	BITFLD(NV2A_VERTEX_NOR_3I_XY_X_SHIFT,							0) \
	BITFLD(NV2A_VERTEX_NOR_3I_XY_X_MASK,								0x0000ffff) \
	BITFLD(NV2A_VERTEX_NOR_3I_XY_Y_SHIFT,							16) \
	BITFLD(NV2A_VERTEX_NOR_3I_XY_Y_MASK,								0xffff0000) \
	METHOD(NV2A_VERTEX_NOR_3I_Z,								0x00001544) \
	BITFLD(NV2A_VERTEX_NOR_3I_Z_Z_SHIFT,							0) \
	BITFLD(NV2A_VERTEX_NOR_3I_Z_Z_MASK,									0x0000ffff) \
	METHOD(NV2A_VERTEX_COL_4F_X,								0x00001550) \
	METHOD(NV2A_VERTEX_COL_4F_Y,								0x00001554) \
	METHOD(NV2A_VERTEX_COL_4F_Z,								0x00001558) \
	METHOD(NV2A_VERTEX_COL_4F_W,								0x0000155c) \
	METHOD(NV2A_VERTEX_COL_3F_X,								0x00001560) \
	METHOD(NV2A_VERTEX_COL_3F_Y,								0x00001564) \
	METHOD(NV2A_VERTEX_COL_3F_Z,								0x00001568) \
	METHOD(NV2A_VERTEX_COL_4I,									0x0000156c) \
	BITFLD(NV2A_VERTEX_COL_4I_R_SHIFT,								0) \
	BITFLD(NV2A_VERTEX_COL_4I_R_MASK,									0x000000ff) \
	BITFLD(NV2A_VERTEX_COL_4I_G_SHIFT,								8) \
	BITFLD(NV2A_VERTEX_COL_4I_G_MASK,									0x0000ff00) \
	BITFLD(NV2A_VERTEX_COL_4I_B_SHIFT,								16) \
	BITFLD(NV2A_VERTEX_COL_4I_B_MASK,									0x00ff0000) \
	BITFLD(NV2A_VERTEX_COL_4I_A_SHIFT,								24) \
	BITFLD(NV2A_VERTEX_COL_4I_A_MASK,									0xff000000) \
	METHOD(NV2A_VERTEX_COL2_4F_X,								0x00001570) \
	METHOD(NV2A_VERTEX_COL2_4F_Y,								0x00001574) \
	METHOD(NV2A_VERTEX_COL2_4F_Z,								0x00001578) \
	METHOD(NV2A_VERTEX_COL2_4F_W,								0x0000157c) \
	METHOD(NV2A_VERTEX_COL2_3F_X,								0x00001580) \
	METHOD(NV2A_VERTEX_COL2_3F_Y,								0x00001584) \
	METHOD(NV2A_VERTEX_COL2_3F_Z,								0x00001588) \
	METHOD(NV2A_VERTEX_COL2_4I,									0x0000158c) \
	BITFLD(NV2A_VERTEX_COL2_4I_R_SHIFT,								0) \
	BITFLD(NV2A_VERTEX_COL2_4I_R_MASK,									0x000000ff) \
	BITFLD(NV2A_VERTEX_COL2_4I_G_SHIFT,								8) \
	BITFLD(NV2A_VERTEX_COL2_4I_G_MASK,									0x0000ff00) \
	BITFLD(NV2A_VERTEX_COL2_4I_B_SHIFT,								16) \
	BITFLD(NV2A_VERTEX_COL2_4I_B_MASK,									0x00ff0000) \
	BITFLD(NV2A_VERTEX_COL2_4I_A_SHIFT,								24) \
	BITFLD(NV2A_VERTEX_COL2_4I_A_MASK,									0xff000000) \
	METHOD(NV2A_VERTEX_TX0_2F_S,								0x00001590) \
	METHOD(NV2A_VERTEX_TX0_2F_T,								0x00001594) \
	METHOD(NV2A_VERTEX_TX0_2I,									0x00001598) \
	BITFLD(NV2A_VERTEX_TX0_2I_S_SHIFT,								0) \
	BITFLD(NV2A_VERTEX_TX0_2I_S_MASK,									0x0000ffff) \
	BITFLD(NV2A_VERTEX_TX0_2I_T_SHIFT,								16) \
	BITFLD(NV2A_VERTEX_TX0_2I_T_MASK,									0xffff0000) \
	METHOD(NV2A_VERTEX_TX0_4F_S,								0x000015a0) \
	METHOD(NV2A_VERTEX_TX0_4F_T,								0x000015a4) \
	METHOD(NV2A_VERTEX_TX0_4F_R,								0x000015a8) \
	METHOD(NV2A_VERTEX_TX0_4F_Q,								0x000015ac) \
	METHOD(NV2A_VERTEX_TX0_4I_ST,								0x000015b0) \
	BITFLD(NV2A_VERTEX_TX0_4I_ST_S_SHIFT,							0) \
	BITFLD(NV2A_VERTEX_TX0_4I_ST_S_MASK,								0x0000ffff) \
	BITFLD(NV2A_VERTEX_TX0_4I_ST_T_SHIFT,							16) \
	BITFLD(NV2A_VERTEX_TX0_4I_ST_T_MASK,								0xffff0000) \
	METHOD(NV2A_VERTEX_TX0_4I_RQ,								0x000015b4) \
	BITFLD(NV2A_VERTEX_TX0_4I_RQ_R_SHIFT,							0) \
	BITFLD(NV2A_VERTEX_TX0_4I_RQ_R_MASK,								0x0000ffff) \
	BITFLD(NV2A_VERTEX_TX0_4I_RQ_Q_SHIFT,							16) \
	BITFLD(NV2A_VERTEX_TX0_4I_RQ_Q_MASK,								0xffff0000) \
	METHOD(NV2A_VERTEX_TX1_2F_S,								0x000015b8) \
	METHOD(NV2A_VERTEX_TX1_2F_T,								0x000015bc) \
	METHOD(NV2A_VERTEX_TX1_2I,									0x000015c0) \
	BITFLD(NV2A_VERTEX_TX1_2I_S_SHIFT,								0) \
	BITFLD(NV2A_VERTEX_TX1_2I_S_MASK,									0x0000ffff) \
	BITFLD(NV2A_VERTEX_TX1_2I_T_SHIFT,								16) \
	BITFLD(NV2A_VERTEX_TX1_2I_T_MASK,									0xffff0000) \
	METHOD(NV2A_VERTEX_TX1_4F_S,								0x000015c8) \
	METHOD(NV2A_VERTEX_TX1_4F_T,								0x000015cc) \
	METHOD(NV2A_VERTEX_TX1_4F_R,								0x000015d0) \
	METHOD(NV2A_VERTEX_TX1_4F_Q,								0x000015d4) \
	METHOD(NV2A_VERTEX_TX1_4I_ST,								0x000015d8) \
	BITFLD(NV2A_VERTEX_TX1_4I_ST_S_SHIFT,							0) \
	BITFLD(NV2A_VERTEX_TX1_4I_ST_S_MASK,								0x0000ffff) \
	BITFLD(NV2A_VERTEX_TX1_4I_ST_T_SHIFT,							16) \
	BITFLD(NV2A_VERTEX_TX1_4I_ST_T_MASK,								0xffff0000) \
	METHOD(NV2A_VERTEX_TX1_4I_RQ,								0x000015dc) \
	BITFLD(NV2A_VERTEX_TX1_4I_RQ_R_SHIFT,							0) \
	BITFLD(NV2A_VERTEX_TX1_4I_RQ_R_MASK,								0x0000ffff) \
	BITFLD(NV2A_VERTEX_TX1_4I_RQ_Q_SHIFT,							16) \
	BITFLD(NV2A_VERTEX_TX1_4I_RQ_Q_MASK,								0xffff0000) \
	METHOD(NV2A_VERTEX_TX2_2F_S,								0x000015e0) \
	METHOD(NV2A_VERTEX_TX2_2F_T,								0x000015e4) \
	METHOD(NV2A_VERTEX_TX2_2I,									0x000015e8) \
	BITFLD(NV2A_VERTEX_TX2_2I_S_SHIFT,								0) \
	BITFLD(NV2A_VERTEX_TX2_2I_S_MASK,									0x0000ffff) \
	BITFLD(NV2A_VERTEX_TX2_2I_T_SHIFT,								16) \
	BITFLD(NV2A_VERTEX_TX2_2I_T_MASK,									0xffff0000) \
	METHOD(NV2A_VERTEX_TX2_4F_S,								0x000015f0) \
	METHOD(NV2A_VERTEX_TX2_4F_T,								0x000015f4) \
	METHOD(NV2A_VERTEX_TX2_4F_R,								0x000015f8) \
	METHOD(NV2A_VERTEX_TX2_4F_Q,								0x000015fc) \
	METHOD(NV2A_VERTEX_TX2_4I_ST,								0x00001600) \
	BITFLD(NV2A_VERTEX_TX2_4I_ST_S_SHIFT,							0) \
	BITFLD(NV2A_VERTEX_TX2_4I_ST_S_MASK,								0x0000ffff) \
	BITFLD(NV2A_VERTEX_TX2_4I_ST_T_SHIFT,							16) \
	BITFLD(NV2A_VERTEX_TX2_4I_ST_T_MASK,								0xffff0000) \
	METHOD(NV2A_VERTEX_TX2_4I_RQ,								0x00001604) \
	BITFLD(NV2A_VERTEX_TX2_4I_RQ_R_SHIFT,							0) \
	BITFLD(NV2A_VERTEX_TX2_4I_RQ_R_MASK,								0x0000ffff) \
	BITFLD(NV2A_VERTEX_TX2_4I_RQ_Q_SHIFT,							16) \
	BITFLD(NV2A_VERTEX_TX2_4I_RQ_Q_MASK,								0xffff0000) \
	METHOD(NV2A_VERTEX_TX3_2F_S,								0x00001608) \
	METHOD(NV2A_VERTEX_TX3_2F_T,								0x0000160c) \
	METHOD(NV2A_VERTEX_TX3_2I,									0x00001610) \
	BITFLD(NV2A_VERTEX_TX3_2I_S_SHIFT,								0) \
	BITFLD(NV2A_VERTEX_TX3_2I_S_MASK,									0x0000ffff) \
	BITFLD(NV2A_VERTEX_TX3_2I_T_SHIFT,								16) \
	BITFLD(NV2A_VERTEX_TX3_2I_T_MASK,									0xffff0000) \
	METHOD(NV2A_VERTEX_TX3_4F_S,								0x00001620) \
	METHOD(NV2A_VERTEX_TX3_4F_T,								0x00001624) \
	METHOD(NV2A_VERTEX_TX3_4F_R,								0x00001628) \
	METHOD(NV2A_VERTEX_TX3_4F_Q,								0x0000162c) \
	METHOD(NV2A_VERTEX_TX3_4I_ST,								0x00001630) \
	BITFLD(NV2A_VERTEX_TX3_4I_ST_S_SHIFT,							0) \
	BITFLD(NV2A_VERTEX_TX3_4I_ST_S_MASK,								0x0000ffff) \
	BITFLD(NV2A_VERTEX_TX3_4I_ST_T_SHIFT,							16) \
	BITFLD(NV2A_VERTEX_TX3_4I_ST_T_MASK,								0xffff0000) \
	METHOD(NV2A_VERTEX_TX3_4I_RQ,								0x00001634) \
	BITFLD(NV2A_VERTEX_TX3_4I_RQ_R_SHIFT,							0) \
	BITFLD(NV2A_VERTEX_TX3_4I_RQ_R_MASK,								0x0000ffff) \
	BITFLD(NV2A_VERTEX_TX3_4I_RQ_Q_SHIFT,							16) \
	BITFLD(NV2A_VERTEX_TX3_4I_RQ_Q_MASK,								0xffff0000) \
	METHOD(NV2A_VERTEX_FOG_1F,									0x00001698) \
	METHOD(NV2A_SET_WEIGHT1F,									0x0000169c) \
	RANGED(NV2A_SET_WEIGHT2F,									0x000016a0, 0x04, 2) \
	RANGED(NV2A_SET_WEIGHT3F,									0x000016b0, 0x04, 3) \
	METHOD(NV2A_EDGEFLAG_ENABLE,								0x000016bc) \
	RANGED(NV2A_SET_WEIGHT4F,									0x000016c0, 0x04, 4) \
	RANGED(NV2A_TRANSFORM_FIXED_CONST3,							0x000016d0, 0x04, 4) \
	RANGED(NV2A_TRANSFORM_FIXED_CONST0,							0x000016e0, 0x04, 4) \
	RANGED(NV2A_TRANSFORM_FIXED_CONST1,							0x000016f0, 0x04, 4) \
	RANGED(NV2A_TRANSFORM_FIXED_CONST2,							0x00001700, 0x04, 4) \
	METHOD(NV2A_INVALIDATE_VERTEX_CACHE_FILE,					0x00001710) \
	METHOD(NV2A_INVALIDATE_VERTEX_FILE,							0x00001714) \
	METHOD(NV2A_TL_NOP,											0x00001718) \
	METHOD(NV2A_TL_SYNC,										0x0000171c) \
	RANGED(NV2A_VTXBUF_ADDRESS,									0x00001720, 0x04, 16) \
	BITFLD(NV2A_VTXBUF_ADDRESS_DMA1,								(1 << 31)) \
	BITFLD(NV2A_VTXBUF_ADDRESS_OFFSET_SHIFT,						0) \
	BITFLD(NV2A_VTXBUF_ADDRESS_OFFSET_MASK,								0x0fffffff) \
	RANGED(NV2A_VTXFMT,											0x00001760, 0x04, 16) \
	BITFLD(NV2A_VTXFMT_TYPE_SHIFT,									0) \
	BITFLD(NV2A_VTXFMT_TYPE_MASK,										0x0000000f) \
	 VALUE(NV2A_VTXFMT_TYPE_FLOAT,										0x00000002) \
	 VALUE(NV2A_VTXFMT_TYPE_UBYTE,										0x00000004) \
	 VALUE(NV2A_VTXFMT_TYPE_USHORT,										0x00000005) \
	BITFLD(NV2A_VTXFMT_SIZE_SHIFT,									4) \
	BITFLD(NV2A_VTXFMT_SIZE_MASK,										0x000000f0) \
	BITFLD(NV2A_VTXFMT_STRIDE_SHIFT,								8) \
	BITFLD(NV2A_VTXFMT_STRIDE_MASK,										0x0000ff00) \
	METHOD(NV2A_LIGHT_MODEL_BACK_SIDE_PRODUCT_AMBIENT_PLUS_EMISSION_R,	0x000017a0) \
	METHOD(NV2A_LIGHT_MODEL_BACK_SIDE_PRODUCT_AMBIENT_PLUS_EMISSION_G,	0x000017a4) \
	METHOD(NV2A_LIGHT_MODEL_BACK_SIDE_PRODUCT_AMBIENT_PLUS_EMISSION_B,	0x000017a8) \
	METHOD(NV2A_COLOR_MATERIAL_BACK_A,							0x000017ac) \
	METHOD(NV2A_COLOR_MATERIAL_BACK_R,							0x000017b0) \
	METHOD(NV2A_COLOR_MATERIAL_BACK_G,							0x000017b4) \
	METHOD(NV2A_COLOR_MATERIAL_BACK_B,							0x000017b8) \
	METHOD(NV2A_COLOR_LOGIC_OP_ENABLE,							0x000017bc) \
	METHOD(NV2A_COLOR_LOGIC_OP_OP,								0x000017c0) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_CLEAR,								0x00001500) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_AND,									0x00001501) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_AND_REVERSE,							0x00001502) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_COPY,									0x00001503) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_AND_INVERTED,							0x00001504) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_NOOP,									0x00001505) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_XOR,									0x00001506) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_OR,									0x00001507) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_NOR,									0x00001508) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_EQUIV,								0x00001509) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_INVERT,								0x0000150a) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_OR_REVERSE,							0x0000150b) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_COPY_INVERTED,						0x0000150c) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_OR_INVERTED,							0x0000150d) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_NAND,									0x0000150e) \
	 VALUE(NV2A_COLOR_LOGIC_OP_OP_SET,									0x0000150f) \
	METHOD(NV2A_LIGHT_MODEL_TWO_SIDE_ENABLE,					0x000017c4) \
	METHOD(NV2A_CLEAR_REPORT_VALUE,								0x000017c8) \
	METHOD(NV2A_SET_ZPASS_PIXEL_COUNT_ENABLE,					0x000017cc) \
	METHOD(NV2A_GET_REPORT,										0x000017d0) \
	RANGED(NV2A_SET_TLCONST_ZERO,								0x000017d4, 0x04, 3) \
	RANGED(NV2A_SET_EYE_DIRECTION,								0x000017e0, 0x04, 3) \
	RANGED(NV2A_SET_LINEAR_FOG_CONST,							0x000017ec, 0x04, 3) \
	METHOD(NV2A_TX_SHADER_CULL_MODE,							0x000017f8) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX0_S,							(1 <<  0)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX0_S_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX0_S_LESS,							0x00000001) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX0_T,							(1 <<  1)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX0_T_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX0_T_LESS,							0x00000002) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX0_R,							(1 <<  2)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX0_R_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX0_R_LESS,							0x00000004) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX0_Q,							(1 <<  3)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX0_Q_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX0_Q_LESS,							0x00000008) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX1_S,							(1 <<  4)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX1_S_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX1_S_LESS,							0x00000010) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX1_T,							(1 <<  5)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX1_T_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX1_T_LESS,							0x00000020) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX1_R,							(1 <<  6)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX1_R_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX1_R_LESS,							0x00000040) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX1_Q,							(1 <<  7)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX1_Q_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX1_Q_LESS,							0x00000080) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX2_S,							(1 <<  8)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX2_S_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX2_S_LESS,							0x00000100) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX2_T,							(1 <<  9)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX2_T_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX2_T_LESS,							0x00000200) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX2_R,							(1 << 10)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX2_R_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX2_R_LESS,							0x00000400) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX2_Q,							(1 << 11)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX2_Q_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX2_Q_LESS,							0x00000800) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX3_S,							(1 << 12)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX3_S_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX3_S_LESS,							0x00001000) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX3_T,							(1 << 13)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX3_T_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX3_T_LESS,							0x00002000) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX3_R,							(1 << 14)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX3_R_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX3_R_LESS,							0x00004000) \
	BITFLD(NV2A_TX_SHADER_CULL_MODE_TX3_Q,							(1 << 15)) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX3_Q_GEQUAL,						0x00000000) \
	 VALUE(NV2A_TX_SHADER_CULL_MODE_TX3_Q_LESS,							0x00008000) \
	METHOD(NV2A_VERTEX_BEGIN_END,								0x000017fc) \
	 VALUE(NV2A_VERTEX_BEGIN_END_STOP,									0x00000000) \
	 VALUE(NV2A_VERTEX_BEGIN_END_POINTS,								0x00000001) \
	 VALUE(NV2A_VERTEX_BEGIN_END_LINES,									0x00000002) \
	 VALUE(NV2A_VERTEX_BEGIN_END_LINE_LOOP,								0x00000003) \
	 VALUE(NV2A_VERTEX_BEGIN_END_LINE_STRIP,							0x00000004) \
	 VALUE(NV2A_VERTEX_BEGIN_END_TRIANGLES,								0x00000005) \
	 VALUE(NV2A_VERTEX_BEGIN_END_TRIANGLE_STRIP,						0x00000006) \
	 VALUE(NV2A_VERTEX_BEGIN_END_TRIANGLE_FAN,							0x00000007) \
	 VALUE(NV2A_VERTEX_BEGIN_END_QUADS,									0x00000008) \
	 VALUE(NV2A_VERTEX_BEGIN_END_QUAD_STRIP,							0x00000009) \
	 VALUE(NV2A_VERTEX_BEGIN_END_POLYGON,								0x0000000a) \
	METHOD(NV2A_VB_ELEMENT_U16,									0x00001800) \
	BITFLD(NV2A_VB_ELEMENT_U16_I0_SHIFT,							0) \
	BITFLD(NV2A_VB_ELEMENT_U16_I0_MASK,									0x0000ffff) \
	BITFLD(NV2A_VB_ELEMENT_U16_I1_SHIFT,							16) \
	BITFLD(NV2A_VB_ELEMENT_U16_I1_MASK,									0xffff0000) \
	METHOD(NV2A_VB_ELEMENT_U32,									0x00001808) \
	METHOD(NV2A_VB_VERTEX_BATCH,								0x00001810) \
	BITFLD(NV2A_VB_VERTEX_BATCH_OFFSET_SHIFT,						0) \
	BITFLD(NV2A_VB_VERTEX_BATCH_OFFSET_MASK,							0x00ffffff) \
	BITFLD(NV2A_VB_VERTEX_BATCH_COUNT_SHIFT,						24) \
	BITFLD(NV2A_VB_VERTEX_BATCH_COUNT_MASK,								0xff000000) \
	METHOD(NV2A_VERTEX_DATA,									0x00001818) \
	METHOD(NV2A_TX_SHADER_CONST_EYE_X,							0x0000181c) \
	METHOD(NV2A_TX_SHADER_CONST_EYE_Y,							0x00001820) \
	METHOD(NV2A_TX_SHADER_CONST_EYE_Z,							0x00001824) \
	METHOD(NV2A_INLINE_VERTEX_REUSE,							0x00001828) \
	RANGED(NV2A_SET_VERTEX_DATA2F_M,							0x00001880, 0x08, 16) \
	RANGED(NV2A_SET_VERTEX_DATA2S,								0x00001900, 0x04, 16) \
	RANGED(NV2A_SET_VERTEX_DATA4UB,								0x00001940, 0x04, 16) \
	RANGED(NV2A_SET_VERTEX_DATA4S_M,							0x00001980, 0x08, 16) \
	RANGED(NV2A_VTX_ATTR_4F_X,									0x00001a00, 0x10, 16) \
	RANGED(NV2A_VTX_ATTR_4F_Y,									0x00001a04, 0x10, 16) \
	RANGED(NV2A_VTX_ATTR_4F_Z,									0x00001a08, 0x10, 16) \
	RANGED(NV2A_VTX_ATTR_4F_W,									0x00001a0c, 0x10, 16) \
	RANGED(NV2A_TX_OFFSET,										0x00001b00, 0x40, 4) \
	RANGED(NV2A_TX_FORMAT,										0x00001b04, 0x40, 4) \
	BITFLD(NV2A_TX_FORMAT_DMA0,										(1 <<  0)) \
	BITFLD(NV2A_TX_FORMAT_DMA1,										(1 <<  1)) \
	BITFLD(NV2A_TX_FORMAT_CUBIC,									(1 <<  2)) \
	BITFLD(NV2A_TX_FORMAT_NO_BORDER,								(1 <<  3)) \
	BITFLD(NV2A_TX_FORMAT_DIMS_SHIFT,								4) \
	BITFLD(NV2A_TX_FORMAT_DIMS_MASK,									0x000000f0) \
	 VALUE(NV2A_TX_FORMAT_DIMS_1D,										0x00000010) \
	 VALUE(NV2A_TX_FORMAT_DIMS_2D,										0x00000020) \
	 VALUE(NV2A_TX_FORMAT_DIMS_3D,										0x00000030) \
	BITFLD(NV2A_TX_FORMAT_FORMAT_SHIFT,								8) \
	BITFLD(NV2A_TX_FORMAT_FORMAT_MASK,									0x0000ff00) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_L8,									0x00000000) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_A8,									0x00000100) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_A1R5G5B5,								0x00000200) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_A8_RECT,								0x00000300) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_A4R4G4B4,								0x00000400) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_R5G6B5,								0x00000500) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_A8R8G8B8,								0x00000600) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_X8R8G8B8,								0x00000700) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_INDEX8,								0x00000b00) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_DXT1,									0x00000c00) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_DXT3,									0x00000e00) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_DXT5,									0x00000f00) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_A1R5G5B5_RECT,							0x00001000) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_R5G6B5_RECT,							0x00001100) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_A8R8G8B8_RECT,							0x00001200) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_L8_RECT,								0x00001300) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_A8L8,									0x00001a00) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_A8_RECT2,								0x00001b00) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_A4R4G4B4_RECT,							0x00001d00) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_R8G8B8_RECT,							0x00001e00) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_L8A8_RECT,								0x00002000) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_DSDT,									0x00002800) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_A16,									0x00003200) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_HILO16,								0x00003300) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_A16_RECT,								0x00003500) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_HILO16_RECT,							0x00003600) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_HILO8,									0x00004400) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_SIGNED_HILO8,							0x00004500) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_HILO8_RECT,							0x00004600) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_SIGNED_HILO8_RECT,						0x00004700) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_FLOAT_RGBA16_NV,						0x00004a00) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_FLOAT_RGBA32_NV,						0x00004b00) \
	 VALUE(NV2A_TX_FORMAT_FORMAT_FLOAT_R32_NV,							0x00004c00) \
	BITFLD(NV2A_TX_FORMAT_MIPMAP,									(1 << 19)) \
	BITFLD(NV2A_TX_FORMAT_BASE_SIZE_U_SHIFT,						20) \
	BITFLD(NV2A_TX_FORMAT_BASE_SIZE_U_MASK,								0x00f00000) \
	BITFLD(NV2A_TX_FORMAT_BASE_SIZE_V_SHIFT,						24) \
	BITFLD(NV2A_TX_FORMAT_BASE_SIZE_V_MASK,								0x0f000000) \
	BITFLD(NV2A_TX_FORMAT_BASE_SIZE_W_SHIFT,						28) \
	BITFLD(NV2A_TX_FORMAT_BASE_SIZE_W_MASK,								0xf0000000) \
	RANGED(NV2A_TX_WRAP,										0x00001b08, 0x40, 4) \
	BITFLD(NV2A_TX_WRAP_S_SHIFT,									0) \
	BITFLD(NV2A_TX_WRAP_S_MASK,											0x000000ff) \
	 VALUE(NV2A_TX_WRAP_S_REPEAT,										0x00000001) \
	 VALUE(NV2A_TX_WRAP_S_MIRRORED_REPEAT,								0x00000002) \
	 VALUE(NV2A_TX_WRAP_S_CLAMP_TO_EDGE,								0x00000003) \
	 VALUE(NV2A_TX_WRAP_S_CLAMP_TO_BORDER,								0x00000004) \
	 VALUE(NV2A_TX_WRAP_S_CLAMP,										0x00000005) \
	BITFLD(NV2A_TX_WRAP_T_SHIFT,									8) \
	BITFLD(NV2A_TX_WRAP_T_MASK,											0x00000f00) \
	 VALUE(NV2A_TX_WRAP_T_REPEAT,										0x00000100) \
	 VALUE(NV2A_TX_WRAP_T_MIRRORED_REPEAT,								0x00000200) \
	 VALUE(NV2A_TX_WRAP_T_CLAMP_TO_EDGE,								0x00000300) \
	 VALUE(NV2A_TX_WRAP_T_CLAMP_TO_BORDER,								0x00000400) \
	 VALUE(NV2A_TX_WRAP_T_CLAMP,										0x00000500) \
	BITFLD(NV2A_TX_WRAP_R_SHIFT,									16) \
	BITFLD(NV2A_TX_WRAP_R_MASK,											0x000f0000) \
	 VALUE(NV2A_TX_WRAP_R_REPEAT,										0x00010000) \
	 VALUE(NV2A_TX_WRAP_R_MIRRORED_REPEAT,								0x00020000) \
	 VALUE(NV2A_TX_WRAP_R_CLAMP_TO_EDGE,								0x00030000) \
	 VALUE(NV2A_TX_WRAP_R_CLAMP_TO_BORDER,								0x00040000) \
	 VALUE(NV2A_TX_WRAP_R_CLAMP,										0x00050000) \
	RANGED(NV2A_TX_ENABLE,										0x00001b0c, 0x40, 4) \
	BITFLD(NV2A_TX_ENABLE_ANISO_SHIFT,								4) \
	BITFLD(NV2A_TX_ENABLE_ANISO_MASK,									0x00000030) \
	 VALUE(NV2A_TX_ENABLE_ANISO_NONE,									0x00000000) \
	 VALUE(NV2A_TX_ENABLE_ANISO_2X,										0x00000010) \
	 VALUE(NV2A_TX_ENABLE_ANISO_4X,										0x00000020) \
	 VALUE(NV2A_TX_ENABLE_ANISO_8X,										0x00000030) \
	BITFLD(NV2A_TX_ENABLE_MIPMAP_MAX_LOD_SHIFT,						14) \
	BITFLD(NV2A_TX_ENABLE_MIPMAP_MAX_LOD_MASK,							0x0003c000) \
	BITFLD(NV2A_TX_ENABLE_MIPMAP_MIN_LOD_SHIFT,						26) \
	BITFLD(NV2A_TX_ENABLE_MIPMAP_MIN_LOD_MASK,							0x3c000000) \
	BITFLD(NV2A_TX_ENABLE_ENABLE,									(1 << 30)) \
	RANGED(NV2A_TX_SWIZZLE,										0x00001b10, 0x40, 4) \
	BITFLD(NV2A_TX_SWIZZLE_RECT_PITCH_SHIFT,						16) \
	BITFLD(NV2A_TX_SWIZZLE_RECT_PITCH_MASK,								0xffff0000) \
	RANGED(NV2A_TX_FILTER,										0x00001b14, 0x40, 4) \
	BITFLD(NV2A_TX_FILTER_LOD_BIAS_SHIFT,							8) \
	BITFLD(NV2A_TX_FILTER_LOD_BIAS_MASK,								0x00000f00) \
	BITFLD(NV2A_TX_FILTER_MINIFY_SHIFT,								16) \
	BITFLD(NV2A_TX_FILTER_MINIFY_MASK,									0x000f0000) \
	 VALUE(NV2A_TX_FILTER_MINIFY_NEAREST,								0x00010000) \
	 VALUE(NV2A_TX_FILTER_MINIFY_LINEAR,								0x00020000) \
	 VALUE(NV2A_TX_FILTER_MINIFY_NEAREST_MIPMAP_NEAREST,				0x00030000) \
	 VALUE(NV2A_TX_FILTER_MINIFY_LINEAR_MIPMAP_NEAREST,					0x00040000) \
	 VALUE(NV2A_TX_FILTER_MINIFY_NEAREST_MIPMAP_LINEAR,					0x00050000) \
	 VALUE(NV2A_TX_FILTER_MINIFY_LINEAR_MIPMAP_LINEAR,					0x00060000) \
	BITFLD(NV2A_TX_FILTER_MAGNIFY_SHIFT,							24) \
	BITFLD(NV2A_TX_FILTER_MAGNIFY_MASK,									0x0f000000) \
	 VALUE(NV2A_TX_FILTER_MAGNIFY_NEAREST,								0x01000000) \
	 VALUE(NV2A_TX_FILTER_MAGNIFY_LINEAR,								0x02000000) \
	RANGED(NV2A_TX_NPOT_SIZE,									0x00001b1c, 0x40, 4) \
	BITFLD(NV2A_TX_NPOT_SIZE_H_SHIFT,								0) \
	BITFLD(NV2A_TX_NPOT_SIZE_H_MASK,									0x0000ffff) \
	BITFLD(NV2A_TX_NPOT_SIZE_W_SHIFT,								16) \
	BITFLD(NV2A_TX_NPOT_SIZE_W_MASK,									0xffff0000) \
	RANGED(NV2A_TX_PALETTE_OFFSET,								0x00001b20, 0x40, 4) \
	RANGED(NV2A_TX_BORDER_COLOR,								0x00001b24, 0x40, 4) \
	BITFLD(NV2A_TX_BORDER_COLOR_B_SHIFT,							0) \
	BITFLD(NV2A_TX_BORDER_COLOR_B_MASK,									0x000000ff) \
	BITFLD(NV2A_TX_BORDER_COLOR_G_SHIFT,							8) \
	BITFLD(NV2A_TX_BORDER_COLOR_G_MASK,									0x0000ff00) \
	BITFLD(NV2A_TX_BORDER_COLOR_R_SHIFT,							16) \
	BITFLD(NV2A_TX_BORDER_COLOR_R_MASK,									0x00ff0000) \
	BITFLD(NV2A_TX_BORDER_COLOR_A_SHIFT,							24) \
	BITFLD(NV2A_TX_BORDER_COLOR_A_MASK,									0xff000000) \
	RANGED(NV2A_TX_SHADER_OFFSET_MATRIX00,						0x00001b28, 0x40, 4) \
	RANGED(NV2A_TX_SHADER_OFFSET_MATRIX01,						0x00001b2c, 0x40, 4) \
	RANGED(NV2A_TX_SHADER_OFFSET_MATRIX11,						0x00001b30, 0x40, 4) \
	RANGED(NV2A_TX_SHADER_OFFSET_MATRIX10,						0x00001b34, 0x40, 4) \
	RANGED(NV2A_SET_TEXTURE_SET_BUMP_ENV_SCALE,					0x00001b38, 0x40, 4) \
	RANGED(NV2A_SET_TEXTURE_SET_BUMP_ENV_OFFSET,				0x00001b3c, 0x40, 4) \
	METHOD(NV2A_PARK_ATTRIBUTE,									0x00001d64) \
	METHOD(NV2A_UNPARK_ATTRIBUTE,								0x00001d68) \
	METHOD(NV2A_SET_SEMAPHORE_OFFSET,							0x00001d6c) \
	METHOD(NV2A_BACK_END_WRITE_SEMAPHORE_RELEASE,				0x00001d70) \
	METHOD(NV2A_TEXTURE_READ_SEMAPHORE_RELEASE,				    0x00001d74) \
	METHOD(NV2A_DEPTHCLIPCONTROL,								0x00001d78) \
	BITFLD(NV2A_DEPTHCLIPCONTROL_CLAMP_SHIFT,						4) \
	BITFLD(NV2A_DEPTHCLIPCONTROL_CLAMP_MASK,							0x000000f0) \
	METHOD(NV2A_MULTISAMPLE_CONTROL,							0x00001d7c) \
	METHOD(NV2A_COMPRESS_ZBUFFER_EN,							0x00001d80) \
	METHOD(NV2A_OCCLUDE_ZSTENCIL_EN,						    0x00001d84) \
	METHOD(NV2A_SET_DXT_DITHER_ENABLE_SW,						0x00001d88) \
	METHOD(NV2A_CLEAR_DEPTH_VALUE,								0x00001d8c) \
	METHOD(NV2A_CLEAR_VALUE,									0x00001d90) \
	METHOD(NV2A_CLEAR_BUFFERS,									0x00001d94) \
	BITFLD(NV2A_CLEAR_BUFFERS_COLOR_A,								(1 <<  7)) \
	BITFLD(NV2A_CLEAR_BUFFERS_COLOR_B,								(1 <<  6)) \
	BITFLD(NV2A_CLEAR_BUFFERS_COLOR_G,								(1 <<  5)) \
	BITFLD(NV2A_CLEAR_BUFFERS_COLOR_R,								(1 <<  4)) \
	BITFLD(NV2A_CLEAR_BUFFERS_STENCIL,								(1 <<  1)) \
	BITFLD(NV2A_CLEAR_BUFFERS_DEPTH,								(1 <<  0)) \
	METHOD(NV2A_CLEAR_X,										0x00001d98) \
	METHOD(NV2A_CLEAR_Y,										0x00001d9c) \
	METHOD(NV2A_SET_BEGIN_PATCH0,								0x00001de0) \
	METHOD(NV2A_SET_BEGIN_PATCH1,								0x00001de4) \
	METHOD(NV2A_SET_BEGIN_PATCH2,								0x00001de8) \
	METHOD(NV2A_SET_BEGIN_PATCH3,								0x00001dec) \
	METHOD(NV2A_SET_END_PATCH,									0x00001df0) \
	METHOD(NV2A_SET_BEGIN_END_SWATCH,							0x00001df4) \
	METHOD(NV2A_SET_BEGIN_END_CURVE,							0x00001df8) \
	RANGED(NV2A_SET_CURVE_COEFFICIENTS,							0x00001e00, 0x04, 4) \
	METHOD(NV2A_SET_BEGIN_TRANSITION0,							0x00001e10) \
	METHOD(NV2A_SET_BEGIN_TRANSITION1,							0x00001e14) \
	METHOD(NV2A_SET_BEGIN_TRANSITION2,							0x00001e18) \
	METHOD(NV2A_SET_END_TRANSITION,								0x00001e1c) \
	METHOD(NV2A_RC_COLOR0,										0x00001e20) \
	BITFLD(NV2A_RC_COLOR0_B_SHIFT,									0) \
	BITFLD(NV2A_RC_COLOR0_B_MASK,										0x000000ff) \
	BITFLD(NV2A_RC_COLOR0_G_SHIFT,									8) \
	BITFLD(NV2A_RC_COLOR0_G_MASK,										0x0000ff00) \
	BITFLD(NV2A_RC_COLOR0_R_SHIFT,									16) \
	BITFLD(NV2A_RC_COLOR0_R_MASK,										0x00ff0000) \
	BITFLD(NV2A_RC_COLOR0_A_SHIFT,									24) \
	BITFLD(NV2A_RC_COLOR0_A_MASK,										0xff000000) \
	METHOD(NV2A_RC_COLOR1,										0x00001e24) \
	BITFLD(NV2A_RC_COLOR1_B_SHIFT,									0) \
	BITFLD(NV2A_RC_COLOR1_B_MASK,										0x000000ff) \
	BITFLD(NV2A_RC_COLOR1_G_SHIFT,									8) \
	BITFLD(NV2A_RC_COLOR1_G_MASK,										0x0000ff00) \
	BITFLD(NV2A_RC_COLOR1_R_SHIFT,									16) \
	BITFLD(NV2A_RC_COLOR1_R_MASK,										0x00ff0000) \
	BITFLD(NV2A_RC_COLOR1_A_SHIFT,									24) \
	BITFLD(NV2A_RC_COLOR1_A_MASK,										0xff000000) \
	RANGED(NV2A_BACK_MATERIAL_SHININESS,						0x00001e28, 0x04, 6) \
	RANGED(NV2A_RC_OUT_RGB,										0x00001e40, 0x04, 8) \
	BITFLD(NV2A_RC_OUT_RGB_CD_OUTPUT_SHIFT,							0) \
	BITFLD(NV2A_RC_OUT_RGB_CD_OUTPUT_MASK,								0x0000000f) \
	 VALUE(NV2A_RC_OUT_RGB_CD_OUTPUT_ZERO,								0x00000000) \
	 VALUE(NV2A_RC_OUT_RGB_CD_OUTPUT_CONSTANT_COLOR0_NV,				0x00000001) \
	 VALUE(NV2A_RC_OUT_RGB_CD_OUTPUT_CONSTANT_COLOR1_NV,				0x00000002) \
	 VALUE(NV2A_RC_OUT_RGB_CD_OUTPUT_FOG,								0x00000003) \
	 VALUE(NV2A_RC_OUT_RGB_CD_OUTPUT_PRIMARY_COLOR_NV,					0x00000004) \
	 VALUE(NV2A_RC_OUT_RGB_CD_OUTPUT_SECONDARY_COLOR_NV,				0x00000005) \
	 VALUE(NV2A_RC_OUT_RGB_CD_OUTPUT_TEXTURE0_ARB,						0x00000008) \
	 VALUE(NV2A_RC_OUT_RGB_CD_OUTPUT_TEXTURE1_ARB,						0x00000009) \
	 VALUE(NV2A_RC_OUT_RGB_CD_OUTPUT_SPARE0_NV,							0x0000000c) \
	 VALUE(NV2A_RC_OUT_RGB_CD_OUTPUT_SPARE1_NV,							0x0000000d) \
	 VALUE(NV2A_RC_OUT_RGB_CD_OUTPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,	0x0000000e) \
	 VALUE(NV2A_RC_OUT_RGB_CD_OUTPUT_E_TIMES_F_NV,						0x0000000f) \
	BITFLD(NV2A_RC_OUT_RGB_AB_OUTPUT_SHIFT,							4) \
	BITFLD(NV2A_RC_OUT_RGB_AB_OUTPUT_MASK,								0x000000f0) \
	 VALUE(NV2A_RC_OUT_RGB_AB_OUTPUT_ZERO,								0x00000000) \
	 VALUE(NV2A_RC_OUT_RGB_AB_OUTPUT_CONSTANT_COLOR0_NV,				0x00000010) \
	 VALUE(NV2A_RC_OUT_RGB_AB_OUTPUT_CONSTANT_COLOR1_NV,				0x00000020) \
	 VALUE(NV2A_RC_OUT_RGB_AB_OUTPUT_FOG,								0x00000030) \
	 VALUE(NV2A_RC_OUT_RGB_AB_OUTPUT_PRIMARY_COLOR_NV,					0x00000040) \
	 VALUE(NV2A_RC_OUT_RGB_AB_OUTPUT_SECONDARY_COLOR_NV,				0x00000050) \
	 VALUE(NV2A_RC_OUT_RGB_AB_OUTPUT_TEXTURE0_ARB,						0x00000080) \
	 VALUE(NV2A_RC_OUT_RGB_AB_OUTPUT_TEXTURE1_ARB,						0x00000090) \
	 VALUE(NV2A_RC_OUT_RGB_AB_OUTPUT_SPARE0_NV,							0x000000c0) \
	 VALUE(NV2A_RC_OUT_RGB_AB_OUTPUT_SPARE1_NV,							0x000000d0) \
	 VALUE(NV2A_RC_OUT_RGB_AB_OUTPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,	0x000000e0) \
	 VALUE(NV2A_RC_OUT_RGB_AB_OUTPUT_E_TIMES_F_NV,						0x000000f0) \
	BITFLD(NV2A_RC_OUT_RGB_SUM_OUTPUT_SHIFT,						8) \
	BITFLD(NV2A_RC_OUT_RGB_SUM_OUTPUT_MASK,								0x00000f00) \
	 VALUE(NV2A_RC_OUT_RGB_SUM_OUTPUT_ZERO,								0x00000000) \
	 VALUE(NV2A_RC_OUT_RGB_SUM_OUTPUT_CONSTANT_COLOR0_NV,				0x00000100) \
	 VALUE(NV2A_RC_OUT_RGB_SUM_OUTPUT_CONSTANT_COLOR1_NV,				0x00000200) \
	 VALUE(NV2A_RC_OUT_RGB_SUM_OUTPUT_FOG,								0x00000300) \
	 VALUE(NV2A_RC_OUT_RGB_SUM_OUTPUT_PRIMARY_COLOR_NV,					0x00000400) \
	 VALUE(NV2A_RC_OUT_RGB_SUM_OUTPUT_SECONDARY_COLOR_NV,				0x00000500) \
	 VALUE(NV2A_RC_OUT_RGB_SUM_OUTPUT_TEXTURE0_ARB,						0x00000800) \
	 VALUE(NV2A_RC_OUT_RGB_SUM_OUTPUT_TEXTURE1_ARB,						0x00000900) \
	 VALUE(NV2A_RC_OUT_RGB_SUM_OUTPUT_SPARE0_NV,						0x00000c00) \
	 VALUE(NV2A_RC_OUT_RGB_SUM_OUTPUT_SPARE1_NV,						0x00000d00) \
	 VALUE(NV2A_RC_OUT_RGB_SUM_OUTPUT_SPARE0_PLUS_SECONDARY_COLOR_NV,	0x00000e00) \
	 VALUE(NV2A_RC_OUT_RGB_SUM_OUTPUT_E_TIMES_F_NV,						0x00000f00) \
	BITFLD(NV2A_RC_OUT_RGB_CD_DOT_PRODUCT,							(1 << 12)) \
	BITFLD(NV2A_RC_OUT_RGB_AB_DOT_PRODUCT,							(1 << 13)) \
	BITFLD(NV2A_RC_OUT_RGB_MUX_SUM,									(1 << 14)) \
	BITFLD(NV2A_RC_OUT_RGB_BIAS,									(1 << 15)) \
	 VALUE(NV2A_RC_OUT_RGB_BIAS_NONE,									0x00000000) \
	 VALUE(NV2A_RC_OUT_RGB_BIAS_BIAS_BY_NEGATIVE_ONE_HALF_NV,			0x00008000) \
	BITFLD(NV2A_RC_OUT_RGB_SCALE_SHIFT,								17) \
	BITFLD(NV2A_RC_OUT_RGB_SCALE_MASK,									0x000C0000) \
	 VALUE(NV2A_RC_OUT_RGB_SCALE_NONE,									0x00000000) \
	 VALUE(NV2A_RC_OUT_RGB_SCALE_SCALE_BY_TWO_NV,						0x00020000) \
	 VALUE(NV2A_RC_OUT_RGB_SCALE_SCALE_BY_FOUR_NV,						0x00040000) \
	 VALUE(NV2A_RC_OUT_RGB_SCALE_SCALE_BY_ONE_HALF_NV,					0x00060000) \
	METHOD(NV2A_RC_ENABLE,										0x00001e60) \
	BITFLD(NV2A_RC_ENABLE_NUM_COMBINERS_SHIFT,						0) \
	BITFLD(NV2A_RC_ENABLE_NUM_COMBINERS_MASK,							0x0000000f) \
	METHOD(NV2A_SHADOW_ZSLOPE_THRESHOLD,						0x00001e68) \
	METHOD(NV2A_TX_RCOMP,										0x00001e6c) \
	 VALUE(NV2A_TX_RCOMP_NEVER,											0x00000000) \
	 VALUE(NV2A_TX_RCOMP_GREATER,										0x00000001) \
	 VALUE(NV2A_TX_RCOMP_EQUAL,											0x00000002) \
	 VALUE(NV2A_TX_RCOMP_GEQUAL,										0x00000003) \
	 VALUE(NV2A_TX_RCOMP_LESS,											0x00000004) \
	 VALUE(NV2A_TX_RCOMP_NOTEQUAL,										0x00000005) \
	 VALUE(NV2A_TX_RCOMP_LEQUAL,										0x00000006) \
	 VALUE(NV2A_TX_RCOMP_ALWAYS,										0x00000007) \
	METHOD(NV2A_TX_SHADER_OP,									0x00001e70) \
	BITFLD(NV2A_TX_SHADER_OP_TX0_SHIFT,								0) \
	BITFLD(NV2A_TX_SHADER_OP_TX0_MASK,									0x0000001f) \
	 VALUE(NV2A_TX_SHADER_OP_TX0_NONE,									0x00000000) \
	 VALUE(NV2A_TX_SHADER_OP_TX0_TEXTURE_2D,							0x00000001) \
	 VALUE(NV2A_TX_SHADER_OP_TX0_PASS_THROUGH,							0x00000004) \
	 VALUE(NV2A_TX_SHADER_OP_TX0_CULL_FRAGMENT,							0x00000005) \
	 VALUE(NV2A_TX_SHADER_OP_TX0_OFFSET_TEXTURE_2D,						0x00000006) \
	 VALUE(NV2A_TX_SHADER_OP_TX0_DOT_PRODUCT_TEXTURE_2D,				0x00000009) \
	 VALUE(NV2A_TX_SHADER_OP_TX0_DOT_PRODUCT_DEPTH_REPLACE,				0x0000000a) \
	 VALUE(NV2A_TX_SHADER_OP_TX0_DEPENDANT_AR_TEXTURE_2D,				0x0000000f) \
	 VALUE(NV2A_TX_SHADER_OP_TX0_DEPENDANT_GB_TEXTURE_2D,				0x00000010) \
	 VALUE(NV2A_TX_SHADER_OP_TX0_DOT_PRODUCT,							0x00000011) \
	BITFLD(NV2A_TX_SHADER_OP_TX1_SHIFT,								5) \
	BITFLD(NV2A_TX_SHADER_OP_TX1_MASK,									0x000003e0) \
	 VALUE(NV2A_TX_SHADER_OP_TX1_NONE,									0x00000000) \
	 VALUE(NV2A_TX_SHADER_OP_TX1_TEXTURE_2D,							0x00000020) \
	 VALUE(NV2A_TX_SHADER_OP_TX1_PASS_THROUGH,							0x00000080) \
	 VALUE(NV2A_TX_SHADER_OP_TX1_CULL_FRAGMENT,							0x000000a0) \
	 VALUE(NV2A_TX_SHADER_OP_TX1_OFFSET_TEXTURE_2D,						0x000000c0) \
	 VALUE(NV2A_TX_SHADER_OP_TX1_DOT_PRODUCT_TEXTURE_2D,				0x00000120) \
	 VALUE(NV2A_TX_SHADER_OP_TX1_DOT_PRODUCT_DEPTH_REPLACE,				0x00000140) \
	 VALUE(NV2A_TX_SHADER_OP_TX1_DEPENDANT_AR_TEXTURE_2D,				0x000001e0) \
	 VALUE(NV2A_TX_SHADER_OP_TX1_DEPENDANT_GB_TEXTURE_2D,				0x00000200) \
	 VALUE(NV2A_TX_SHADER_OP_TX1_DOT_PRODUCT,							0x00000220) \
	BITFLD(NV2A_TX_SHADER_OP_TX2_SHIFT,								10) \
	BITFLD(NV2A_TX_SHADER_OP_TX2_MASK,									0x00007c00) \
	 VALUE(NV2A_TX_SHADER_OP_TX2_NONE,									0x00000000) \
	 VALUE(NV2A_TX_SHADER_OP_TX2_TEXTURE_2D,							0x00000400) \
	 VALUE(NV2A_TX_SHADER_OP_TX2_PASS_THROUGH,							0x00001000) \
	 VALUE(NV2A_TX_SHADER_OP_TX2_CULL_FRAGMENT,							0x00001400) \
	 VALUE(NV2A_TX_SHADER_OP_TX2_OFFSET_TEXTURE_2D,						0x00001800) \
	 VALUE(NV2A_TX_SHADER_OP_TX2_DOT_PRODUCT_TEXTURE_2D,				0x00002400) \
	 VALUE(NV2A_TX_SHADER_OP_TX2_DOT_PRODUCT_DEPTH_REPLACE,				0x00002800) \
	 VALUE(NV2A_TX_SHADER_OP_TX2_DEPENDANT_AR_TEXTURE_2D,				0x00003c00) \
	 VALUE(NV2A_TX_SHADER_OP_TX2_DEPENDANT_GB_TEXTURE_2D,				0x00004000) \
	 VALUE(NV2A_TX_SHADER_OP_TX2_DOT_PRODUCT,							0x00004400) \
	BITFLD(NV2A_TX_SHADER_OP_TX3_SHIFT,								15) \
	BITFLD(NV2A_TX_SHADER_OP_TX3_MASK,									0x000f8000) \
	 VALUE(NV2A_TX_SHADER_OP_TX3_NONE,									0x00000000) \
	 VALUE(NV2A_TX_SHADER_OP_TX3_TEXTURE_2D,							0x00008000) \
	 VALUE(NV2A_TX_SHADER_OP_TX3_PASS_THROUGH,							0x00020000) \
	 VALUE(NV2A_TX_SHADER_OP_TX3_CULL_FRAGMENT,							0x00028000) \
	 VALUE(NV2A_TX_SHADER_OP_TX3_OFFSET_TEXTURE_2D,						0x00030000) \
	 VALUE(NV2A_TX_SHADER_OP_TX3_DOT_PRODUCT_TEXTURE_2D,				0x00048000) \
	 VALUE(NV2A_TX_SHADER_OP_TX3_DOT_PRODUCT_DEPTH_REPLACE,				0x00050000) \
	 VALUE(NV2A_TX_SHADER_OP_TX3_DEPENDANT_AR_TEXTURE_2D,				0x00078000) \
	 VALUE(NV2A_TX_SHADER_OP_TX3_DEPENDANT_GB_TEXTURE_2D,				0x00080000) \
	 VALUE(NV2A_TX_SHADER_OP_TX3_DOT_PRODUCT,							0x00088000) \
	METHOD(NV2A_TX_SHADER_DOTMAPPING,							0x00001e74) \
	BITFLD(NV2A_TX_SHADER_DOTMAPPING_TX0_SHIFT,						0) \
	BITFLD(NV2A_TX_SHADER_DOTMAPPING_TX0_MASK,							0x0000000f) \
	BITFLD(NV2A_TX_SHADER_DOTMAPPING_TX1_SHIFT,						4) \
	BITFLD(NV2A_TX_SHADER_DOTMAPPING_TX1_MASK,							0x000000f0) \
	BITFLD(NV2A_TX_SHADER_DOTMAPPING_TX2_SHIFT,						8) \
	BITFLD(NV2A_TX_SHADER_DOTMAPPING_TX2_MASK,							0x00000f00) \
	BITFLD(NV2A_TX_SHADER_DOTMAPPING_TX3_SHIFT,						12) \
	BITFLD(NV2A_TX_SHADER_DOTMAPPING_TX3_MASK,							0x0000f000) \
	METHOD(NV2A_TX_SHADER_PREVIOUS,								0x00001e78) \
	BITFLD(NV2A_TX_SHADER_PREVIOUS_TX0_SHIFT,						8) \
	BITFLD(NV2A_TX_SHADER_PREVIOUS_TX0_MASK,							0x00000f00) \
	BITFLD(NV2A_TX_SHADER_PREVIOUS_TX1_SHIFT,						12) \
	BITFLD(NV2A_TX_SHADER_PREVIOUS_TX1_MASK,							0x0000f000) \
	BITFLD(NV2A_TX_SHADER_PREVIOUS_TX2_SHIFT,						16) \
	BITFLD(NV2A_TX_SHADER_PREVIOUS_TX2_MASK,							0x00030000) \
	BITFLD(NV2A_TX_SHADER_PREVIOUS_TX3_SHIFT,						20) \
	BITFLD(NV2A_TX_SHADER_PREVIOUS_TX3_MASK,							0x00300000) \
	RANGED(NV2A_SET_TRANSFORM_DATA,								0x00001e80, 0x04, 4) \
	METHOD(NV2A_LAUNCH_TRANSFORM_PROGRAM,						0x00001e90) \
	METHOD(NV2A_ENGINE,											0x00001e94) \
	BITFLD(NV2A_ENGINE_VP,											(1 <<  1)) \
	BITFLD(NV2A_ENGINE_FIXED,										(1 <<  2)) \
	METHOD(NV2A_SET_TRANSFORM_PROGRAM_CXT_WRITE_EN,				0x00001e98) \
	METHOD(NV2A_VP_UPLOAD_FROM_ID,								0x00001e9c) \
	METHOD(NV2A_VP_START_FROM_ID,								0x00001ea0) \
	METHOD(NV2A_VP_UPLOAD_CONST_ID,								0x00001ea4) \
	METHOD(NV2A_VIEWPORT_TRANSLATE_X,							0x00001f00) \
	METHOD(NV2A_VIEWPORT_TRANSLATE_Y,							0x00001f04) \
	METHOD(NV2A_VIEWPORT_TRANSLATE_Z,							0x00001f08) \
	METHOD(NV2A_VIEWPORT_TRANSLATE_W,							0x00001f0c) \
	RANGED(NV2A_DEBUG_INIT,										0x00001fc0, 0x4, 10)

#define ENUM_METHOD_Declare(Name, Method) \
constexpr DWORD Name = Method;

#define ENUM_RANGED_Declare(Name, Method, Pitch, Repeat) \
constexpr DWORD Name ## __0 = Method; \
constexpr DWORD Name ## __SIZE = Repeat; \
constexpr DWORD Name(int x) { return (Method+((x)*Pitch)); } // TODO : assert(x < Repeat);

#define ENUM_BITFLD_Declare(Name, Value) \
constexpr DWORD Name = Value;

#define ENUM_VALUE_Declare(Name, Value) \
constexpr DWORD Name = Value;

ENUM_NV2A(ENUM_METHOD_Declare, ENUM_RANGED_Declare, ENUM_BITFLD_Declare, ENUM_VALUE_Declare)

typedef enum _TXBType {
	xt_Unknown = 0, // Defined as zero, to coincide with default value of DxbxRenderStateInfo.T and DxbxTextureStageStateInfo.T

	xtBOOL,
	xtBYTE,
	xtD3DBLEND,
	xtD3DBLENDOP,
	xtD3DCLEAR,
	xtD3DCMPFUNC,
	xtD3DCOLOR,
	xtD3DCOLORWRITEENABLE,
	xtD3DCUBEMAP_FACES,
	xtD3DCULL,
	xtD3DDCC,
	xtD3DFILLMODE,
	xtD3DFOGMODE,
	xtD3DFORMAT,
	xtD3DFRONT,
	xtD3DLOGICOP,
	xtD3DMCS,
	xtD3DMULTISAMPLE_TYPE,
	xtD3DMULTISAMPLEMODE,
	xtD3DPRIMITIVETYPE,
	xtD3DRESOURCETYPE,
	xtD3DSAMPLEALPHA,
	xtD3DSHADEMODE,
	xtD3DSTENCILOP,
	xtD3DSWATH,
	xtD3DTEXTUREADDRESS, // Used for TextureStageState X_D3DTSS_ADDRESSU, X_D3DTSS_ADDRESSV and X_D3DTSS_ADDRESSW
	xtD3DTEXTUREFILTERTYPE, // Used for TextureStageState X_D3DTSS_MAGFILTER, X_D3DTSS_MINFILTER and X_D3DTSS_MIPFILTER
	xtD3DTEXTUREOP, // Used for TextureStageState X_D3DTSS_COLOROP and X_D3DTSS_ALPHAOP
	xtD3DTEXTURESTAGESTATETYPE,
	xtD3DTRANSFORMSTATETYPE,
	xtD3DVERTEXBLENDFLAGS, // Used for X_D3DRS_VERTEXBLEND
	xtD3DVSDE,
	xtD3DWRAP,
	xtDWORD,
	xtFloat,
	xtLONG
} TXBType;

typedef struct _RenderStateInfo {
	const char *S;   // String representation.
	WORD V;    // The XDK version since which a render state was introduced (using the 5911 declarations as a base).
	TXBType T = xt_Unknown; // The Xbox data type. Defaults to xt_Unknown.
	xbox::NV2AMETHOD M; // The related push buffer method. Not always a 1-to-1 mapping. Needs push-buffer interpretation & conversion code.
	D3DRENDERSTATETYPE PC = (D3DRENDERSTATETYPE)0; // Map XBox to PC render state
	const char *N;   // XDK notes. Defaults to ''.
	WORD R; // The XDK version since which a render state was removed
}
RenderStateInfo;

#define D3DRS_UNSUPPORTED ((D3DRENDERSTATETYPE)0)

extern const RenderStateInfo& GetDxbxRenderStateInfo(int State);

extern xbox::X_D3DFORMAT GetXboxPixelContainerFormat(const xbox::dword_xt XboxPixelContainer_Format);

extern xbox::X_D3DFORMAT GetXboxPixelContainerFormat(const xbox::X_D3DPixelContainer* pXboxPixelContainer);

extern bool ConvertD3DTextureToARGBBuffer(
	xbox::X_D3DFORMAT X_Format,
	uint8_t* pSrc,
	int SrcWidth, int SrcHeight, int SrcRowPitch, int SrcSlicePitch,
	uint8_t* pDst, int DstRowPitch, int DstSlicePitch,
	unsigned int uiDepth = 1,
	xbox::PVOID pPalleteData = xbox::zeroptr
);

extern void CxbxSetPixelContainerHeader
(
	xbox::X_D3DPixelContainer* pPixelContainer,
	DWORD                      Common,
	UINT                       Width,
	UINT                       Height,
	UINT                       Levels,
	xbox::X_D3DFORMAT          Format,
	UINT                       Dimensions,
	UINT                       Pitch
);

extern uint8_t* ConvertD3DTextureToARGB(
	xbox::X_D3DPixelContainer* pXboxPixelContainer,
	uint8_t* pSrc,
	int* pWidth, int* pHeight,
	xbox::PVOID pPalleteData = xbox::zeroptr
);

#endif
