package server

import (
	"context"
	"lark-vkm/internal/handlers"
	"lark-vkm/internal/initialization"
	"lark-vkm/pkg/openai"
	"log"

	"github.com/gin-gonic/gin"
	sdkginext "github.com/larksuite/oapi-sdk-gin"
	larkcard "github.com/larksuite/oapi-sdk-go/v3/card"
	"github.com/larksuite/oapi-sdk-go/v3/event/dispatcher"
	larkim "github.com/larksuite/oapi-sdk-go/v3/service/im/v1"
	"github.com/spf13/cobra"
)

var cmd = &cobra.Command{
	Use:   "server",
	Short: "run lark webhook server",
	Run: func(cmd *cobra.Command, args []string) {
		cfg, err := cmd.Flags().GetString("config")
		if err != nil {
			log.Println(err)
			return
		}

		initialization.InitRoleList()
		config := initialization.LoadConfig(cfg)
		initialization.LoadLarkClient(*config)
		gpt := openai.NewChatGPT(*config)
		handlers.InitHandlers(gpt, *config)

		eventHandler := dispatcher.NewEventDispatcher(
			config.FeishuAppVerificationToken, config.FeishuAppEncryptKey).
			OnP2MessageReceiveV1(handlers.Handler).
			OnP2MessageReadV1(func(ctx context.Context, event *larkim.P2MessageReadV1) error {
				return handlers.ReadHandler(ctx, event)
			})

		cardHandler := larkcard.NewCardActionHandler(
			config.FeishuAppVerificationToken, config.FeishuAppEncryptKey,
			handlers.CardHandler())

		r := gin.Default()
		r.GET("/ping", func(c *gin.Context) {
			c.JSON(200, gin.H{
				"message": "pong",
			})
		})
		r.POST("/webhook/event",
			sdkginext.NewEventHandlerFunc(eventHandler))
		r.POST("/webhook/card",
			sdkginext.NewCardActionHandlerFunc(
				cardHandler))

		err = initialization.StartServer(*config, r)
		if err != nil {
			log.Fatalf("failed to start server: %v", err)
		}
	},
}

func Register(rootCmd *cobra.Command) error {
	rootCmd.AddCommand(cmd)

	return nil
}
