#include <errno.h>
#include <string.h>

#include "dius/test/prelude.h"

namespace string_h {
[[gnu::noinline]] static auto do_strchr(char const* s, int ch) -> char const* {
    return strchr(s, ch);
}

[[gnu::noinline]] static auto do_strrchr(char const* s, int ch) -> char const* {
    return strrchr(s, ch);
}

[[gnu::noinline]] static auto do_strstr(char const* s, char const* t) -> char const* {
    return strstr(s, t);
}

[[gnu::noinline]] static auto do_strcmp(char const* s, char const* t) -> int {
    return strcmp(s, t);
}

[[gnu::noinline]] static auto do_strncmp(char const* s, char const* t, size_t count) -> int {
    return strncmp(s, t, count);
}

[[gnu::noinline]] static auto do_strcat(char* s, char const* t) -> char* {
    return strcat(s, t);
}

[[gnu::noinline]] static auto do_strncat(char* s, char const* t, size_t count) -> char* {
    return strncat(s, t, count);
}

[[gnu::noinline]] static auto do_strcpy(char* s, char const* t) -> char* {
    return strcpy(s, t);
}

[[gnu::noinline]] static auto do_strncpy(char* s, char const* t, size_t count) -> char* {
    return strncpy(s, t, count);
}

[[gnu::noinline]] static auto do_strlen(char const* s) -> usize {
    return strlen(s);
}

[[gnu::noinline]] static auto do_strcoll(char const* s, char const* t) -> int {
    return strcoll(s, t);
}

[[gnu::noinline]] static auto do_strerror(int errnum) -> char* {
    return strerror(errnum);
}

[[gnu::noinline]] static auto do_strxfrm(char* s, char const* t, size_t count) -> usize {
    return strxfrm(s, t, count);
}

[[gnu::noinline]] static auto do_memchr(byte const* s, int ch, usize n) -> void const* {
    return memchr(s, ch, di::black_box(n));
}

[[gnu::noinline]] static auto do_memcmp(byte* dest, byte const* src, usize n) -> int {
    return memcmp(dest, src, di::black_box(n));
}

[[gnu::noinline]] static auto do_memcpy(byte* dest, byte const* src, usize n) -> void* {
    return memcpy(dest, src, di::black_box(n));
}

[[gnu::noinline]] static auto do_memmove(byte* dest, byte const* src, usize n) -> void* {
    return memmove(dest, src, di::black_box(n));
}

[[gnu::noinline]] static auto do_memset(byte* dest, byte x, usize n) -> void* {
    return memset(dest, di::to_integer<int>(x), di::black_box(n));
}

static void strcat_() {
    char buffer[16] = {};
    auto const* a = di::black_box((char const*) "Hello");
    auto const* b = di::black_box((char const*) ", World");

    ASSERT_EQ(buffer, do_strcat(do_strcpy(buffer, a), b));
    ASSERT_EQ(do_strcmp(buffer, "Hello, World"), 0);
}

static void strncat_() {
    char buffer[10] = {};
    auto const* a = di::black_box((char const*) "Hello");
    auto const* b = di::black_box((char const*) ", World");

    ASSERT_EQ(buffer, do_strncat(do_strcpy(buffer, a), b, sizeof(buffer) - do_strlen(a) - 1));
    ASSERT_EQ(do_strcmp(buffer, "Hello, Wo"), 0);
}

static void strcmp_() {
    auto const* a = di::black_box((char const*) "Hello");
    auto const* b = di::black_box((char const*) "Hello");
    auto const* c = di::black_box((char const*) "HelloQ");
    auto const* d = di::black_box((char const*) "Hell");
    auto const* e = di::black_box((char const*) "Hellp");

    ASSERT_EQ(do_strcmp(a, b), 0);
    ASSERT_LT(do_strcmp(a, c), 0);
    ASSERT_GT(do_strcmp(a, d), 0);
    ASSERT_LT(do_strcmp(a, e), 0);
}

static void strncmp_() {
    auto const* a = di::black_box((char const*) "Hello");
    auto const* b = di::black_box((char const*) "Hello");
    auto const* c = di::black_box((char const*) "HelloQ");
    auto const* d = di::black_box((char const*) "Hell");
    auto const* e = di::black_box((char const*) "Hellp");

    ASSERT_EQ(do_strncmp(a, b, 5), 0);
    ASSERT_EQ(do_strncmp(a, b, 10), 0);
    ASSERT_EQ(do_strncmp(a, c, 5), 0);
    ASSERT_LT(do_strncmp(a, c, 6), 0);
    ASSERT_GT(do_strncmp(a, d, 5), 0);
    ASSERT_GT(do_strncmp(a, d, 10), 0);
    ASSERT_EQ(do_strncmp(a, d, 4), 0);
    ASSERT_LT(do_strncmp(a, e, 5), 0);
    ASSERT_LT(do_strncmp(a, e, 10), 0);
}

static void strcpy_() {
    char buffer[16] = {};
    di::fill(buffer, -1);
    auto const* s = di::black_box((char const*) "Hello");
    auto const* t = di::black_box((char const*) "");

    ASSERT_EQ(buffer, do_strcpy(buffer, s));
    ASSERT_EQ(do_strcmp(buffer, s), 0);

    ASSERT_EQ(buffer, do_strcpy(buffer, t));
    ASSERT_EQ(do_strcmp(buffer, t), 0);
}

static void strncpy_() {
    char buffer[8] = {};
    auto const* s = di::black_box((char const*) "Hello");
    auto const* t = di::black_box((char const*) "");
    auto const* q = di::black_box((char const*) "Hello, World");

    di::fill(buffer, -1);
    ASSERT_EQ(buffer, do_strncpy(buffer, s, 8));

    char expected1[] = { 'H', 'e', 'l', 'l', 'o', '\0', '\0', '\0' };
    ASSERT(di::container::equal(buffer, expected1));

    di::fill(buffer, -1);
    ASSERT_EQ(buffer, do_strncpy(buffer, t, 8));

    char expected2[] = { '\0', '\0', '\0', '\0', '\0', '\0', '\0', '\0' };
    ASSERT(di::container::equal(buffer, expected2));

    di::fill(buffer, -1);
    ASSERT_EQ(buffer, do_strncpy(buffer, q, 7));

    char expected3[] = { 'H', 'e', 'l', 'l', 'o', ',', ' ', -1 };
    ASSERT(di::container::equal(buffer, expected3));
}

static void strchr_() {
    auto const* s = di::black_box((char const*) "Hello");

    auto const* r1 = do_strchr(s, 'l');
    auto const* e1 = s + 2;
    ASSERT_EQ(r1, e1);

    auto const* r2 = do_strchr(s, '\0');
    auto const* e2 = s + 5;
    ASSERT_EQ(r2, e2);

    auto const* r3 = do_strchr(s, 'p');
    auto e3 = nullptr;
    ASSERT_EQ(r3, e3);

    auto const* t = di::black_box((char const*) "");
    auto const* r4 = do_strchr(t, 'a');
    auto e4 = nullptr;
    ASSERT_EQ(r4, e4);

    auto const* r5 = do_strchr(t, '\0');
    auto const* e5 = t;
    ASSERT_EQ(r5, e5);

    auto const* u = di::black_box((char const*) "Hello\xfe");
    auto const* r6 = do_strchr(u, '\xfe');
    auto const* e6 = u + 5;
    ASSERT_EQ(r6, e6);
}

static void strrchr_() {
    auto const* s = di::black_box((char const*) "Hello");

    auto const* r1 = do_strrchr(s, 'l');
    auto const* e1 = s + 3;
    ASSERT_EQ(r1, e1);

    auto const* r2 = do_strrchr(s, '\0');
    auto const* e2 = s + 5;
    ASSERT_EQ(r2, e2);

    auto const* r3 = do_strrchr(s, 'p');
    auto e3 = nullptr;
    ASSERT_EQ(r3, e3);

    auto const* t = di::black_box((char const*) "");
    auto const* r4 = do_strrchr(t, 'a');
    auto e4 = nullptr;
    ASSERT_EQ(r4, e4);

    auto const* r5 = do_strrchr(t, '\0');
    auto const* e5 = t;
    ASSERT_EQ(r5, e5);

    auto const* u = di::black_box((char const*) "Hello\xfe");
    auto const* r6 = do_strrchr(u, '\xfe');
    auto const* e6 = u + 5;
    ASSERT_EQ(r6, e6);
}

static void strstr_() {
    auto const* s = di::black_box((char const*) "Hello");
    auto const* t = di::black_box((char const*) "ell");
    auto const* v = di::black_box((char const*) "lll");
    auto const* u = di::black_box((char const*) "Helloo");
    auto const* e = di::black_box((char const*) "");

    auto const* r1 = do_strstr(s, t);
    auto const* e1 = s + 1;
    ASSERT_EQ(r1, e1);

    auto const* r2 = do_strstr(s, v);
    auto e2 = nullptr;
    ASSERT_EQ(r2, e2);

    auto const* r3 = do_strstr(s, u);
    auto e3 = nullptr;
    ASSERT_EQ(r3, e3);

    auto const* r4 = do_strstr(s, e);
    auto const* e4 = s;
    ASSERT_EQ(r4, e4);

    auto const* r5 = do_strstr(e, e);
    auto const* e5 = e;
    ASSERT_EQ(r5, e5);
}

static void strlen_() {
    auto const* s = di::black_box((char const*) "Hello");
    auto const* e = di::black_box((char const*) "");

    auto r1 = do_strlen(s);
    auto e1 = 5U;
    ASSERT_EQ(r1, e1);

    auto r2 = do_strlen(e);
    auto e2 = 0U;
    ASSERT_EQ(r2, e2);
}

static void memcpy_() {
    auto src = di::black_box(di::Array { 4_b, 5_b, 6_b, 7_b });
    auto dst = di::black_box(di::Array { 8_b, 9_b, 10_b, 11_b });

    ASSERT_EQ(dst.data() + 1, do_memcpy(dst.data() + 1, src.data(), 2));

    auto e1 = di::Array { 8_b, 4_b, 5_b, 11_b };
    ASSERT_EQ(dst, e1);
}

static void memmove_() {
    auto bytes = di::black_box(di::Array { 4_b, 5_b, 6_b, 7_b, 8_b });
    ASSERT_EQ(bytes.data(), do_memmove(bytes.data(), bytes.data() + 1, 2));

    auto e1 = di::Array { 5_b, 6_b, 6_b, 7_b, 8_b };
    ASSERT_EQ(bytes, e1);

    ASSERT_EQ(bytes.data() + 3, do_memmove(bytes.data() + 3, bytes.data() + 2, 2));

    auto e2 = di::Array { 5_b, 6_b, 6_b, 6_b, 7_b };
    ASSERT_EQ(bytes, e2);
}

static void memset_() {
    auto bytes = di::black_box(di::Array { 4_b, 5_b, 6_b, 7_b });
    ASSERT_EQ(bytes.data(), do_memset(bytes.data(), 9_b, 3));

    auto ex1 = di::Array { 9_b, 9_b, 9_b, 7_b };
    ASSERT_EQ(bytes, ex1);
}

static void memcmp_() {
    auto a = di::black_box(di::Array { 4_b, 5_b, 6_b, 7_b });
    auto b = di::black_box(di::Array { 4_b, 5_b, 6_b, 7_b });
    auto c = di::black_box(di::Array { 4_b, 5_b, 6_b, 8_b });

    ASSERT_EQ(do_memcmp(a.data(), b.data(), a.size()), 0);
    ASSERT_LT(do_memcmp(a.data(), c.data(), a.size()), 0);
    ASSERT_GT(do_memcmp(c.data(), a.data(), a.size()), 0);
}

static void memchr_() {
    auto bytes = di::black_box(di::Array { 4_b, 5_b, 6_b, 7_b });
    auto e = di::black_box(di::Array { 4_b, 5_b, 6_b, 7_b });

    auto const* r1 = do_memchr(bytes.data(), 5, bytes.size());
    ASSERT_EQ(r1, bytes.data() + 1);

    auto const* r2 = do_memchr(bytes.data(), 8, bytes.size());
    auto e2 = nullptr;
    ASSERT_EQ(r2, e2);

    auto const* r3 = do_memchr(e.data(), 8, e.size());
    auto e3 = nullptr;
    ASSERT_EQ(r3, e3);
}

static void strcoll_() {
    auto const* a = di::black_box((char const*) "Hello");
    auto const* b = di::black_box((char const*) "Hello");
    auto const* c = di::black_box((char const*) "HelloQ");
    auto const* d = di::black_box((char const*) "Hell");
    auto const* e = di::black_box((char const*) "Hellp");

    ASSERT_EQ(do_strcoll(a, b), 0);
    ASSERT_LT(do_strcoll(a, c), 0);
    ASSERT_GT(do_strcoll(a, d), 0);
    ASSERT_LT(do_strcoll(a, e), 0);
}

static void strxfrm_() {
    auto const* a = di::black_box((char const*) "Hello");
    auto buffer = di::Array<char, 6> {};

    auto r1 = do_strxfrm(buffer.data(), a, buffer.size());
    auto e1 = 5U;
    ASSERT_EQ(r1, e1);

    auto e2 = di::Array<char, 6> { 'H', 'e', 'l', 'l', 'o', '\0' };
    ASSERT_EQ(buffer, e2);

    buffer.fill(1);
    auto r3 = do_strxfrm(buffer.data(), a, 3);
    auto e3 = 5U;
    ASSERT_EQ(r3, e3);

    auto e4 = di::Array<char, 6> { 'H', 'e', 'l', '\x01', '\x01', '\x01' };
    ASSERT_EQ(buffer, e4);
}

static void strerror_() {
    auto e = di::black_box(ENOENT);
    auto* r = do_strerror(e);
    ASSERT_NOT_EQ(r, nullptr);
}

TEST(string_h, strcat_)
TEST(string_h, strncat_)
TEST(string_h, strcmp_)
TEST(string_h, strncmp_)
TEST(string_h, strcpy_)
TEST(string_h, strncpy_)
TEST(string_h, strchr_)
TEST(string_h, strrchr_)
TEST(string_h, strstr_)
TEST(string_h, strlen_)
TEST(string_h, memcpy_)
TEST(string_h, memmove_)
TEST(string_h, memset_)
TEST(string_h, memcmp_)
TEST(string_h, memchr_)
TEST(string_h, strcoll_)
TEST(string_h, strxfrm_)
TEST(string_h, strerror_)
}
