﻿namespace Catel.Tests.Logging
{
    using Catel.Logging;
    using NUnit.Framework;

    [TestFixture]
    public class CatelLogFacts
    {
        private CatelLog _logger;
        private CatelLog Logger
        {
            get { return _logger ?? (_logger = (CatelLog)LogManager.GetCatelLogger(typeof(CatelLogFacts), AlwaysLog)); }
            set { _logger = value; }
        }

        private bool AlwaysLog { get; set; }
        private ILogListener FakeListener { get; set; }

        /// <summary>
        /// Use Test_Initialize to run code before running each test.
        /// </summary>
        [SetUp]
        public void Test_Initialize()
        {
            AlwaysLog = true;
            FakeListener = LogManager.AddDebugListener();
            FakeListener.IgnoreCatelLogging = true;
        }

        /// <summary>
        /// Use Test_Cleanup to run code after each test has run.
        /// </summary>
        [TearDown]
        public void Test_Cleanup()
        {
            LogManager.RemoveLogger(typeof(CatelLogFacts).FullName);
            LogManager.RemoveListener(FakeListener);
            Logger = null;
        }

        [Test]
        public void Debug_AlwaysLogIsTrue_IgnoreCatelLogsIsTrue_WritesLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Debug;
            const string expectedMessage = "A message";

            var receivedLog = false;
            FakeListener.LogMessage += (sender, e) =>
                {
                    receivedLog = true;
                    Assert.That(e, Is.Not.Null);
                    Assert.That(e.LogEvent, Is.EqualTo(expectedLogEvent));
                    Assert.That(e.Message, Is.EqualTo(expectedMessage));
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.True);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            Assert.That(receivedLog, Is.True);
        }

        [Test]
        public void Debug_AlwaysLogIsFalse_IgnoreCatelLogsIsTrue_DoesNotWriteLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Debug;
            const string expectedMessage = "A message";

            AlwaysLog = false;

            FakeListener.LogMessage += (sender, e) =>
                {
                    Assert.Fail();
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.True);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            // did not log to listener
        }

        [Test]
        public void Debug_AlwaysLogIsFalse_IgnoreCatelLogsIsFalse_WritesLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Debug;
            const string expectedMessage = "A message";

            AlwaysLog = false;
            FakeListener.IgnoreCatelLogging = false;

            var receivedLog = false;
            FakeListener.LogMessage += (sender, e) =>
                {
                    receivedLog = true;
                    Assert.That(e, Is.Not.Null);
                    Assert.That(e.LogEvent, Is.EqualTo(expectedLogEvent));
                    Assert.That(e.Message, Is.EqualTo(expectedMessage));
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.False);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            Assert.That(receivedLog, Is.True);
        }

        [Test]
        public void Error_AlwaysLogIsTrue_IgnoreCatelLogsIsTrue_WritesLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Error;
            const string expectedMessage = "A message";

            var receivedLog = false;
            FakeListener.LogMessage += (sender, e) =>
                {
                    receivedLog = true;
                    Assert.That(e, Is.Not.Null);
                    Assert.That(e.LogEvent, Is.EqualTo(expectedLogEvent));
                    Assert.That(e.Message, Is.EqualTo(expectedMessage));
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.True);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            Assert.That(receivedLog, Is.True);
        }

        [Test]
        public void Error_AlwaysLogIsFalse_IgnoreCatelLogsIsTrue_DoesNotWriteLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Error;
            const string expectedMessage = "A message";

            AlwaysLog = false;

            FakeListener.LogMessage += (sender, e) =>
                {
                    Assert.Fail();
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.True);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            // did not log to listener
        }

        [Test]
        public void Error_AlwaysLogIsFalse_IgnoreCatelLogsIsFalse_WritesLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Error;
            const string expectedMessage = "A message";

            AlwaysLog = false;
            FakeListener.IgnoreCatelLogging = false;

            var receivedLog = false;
            FakeListener.LogMessage += (sender, e) =>
                {
                    receivedLog = true;
                    Assert.That(e, Is.Not.Null);
                    Assert.That(e.LogEvent, Is.EqualTo(expectedLogEvent));
                    Assert.That(e.Message, Is.EqualTo(expectedMessage));
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.False);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            Assert.That(receivedLog, Is.True);
        }

        [Test]
        public void Info_AlwaysLogIsTrue_IgnoreCatelLogsIsTrue_WritesLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Info;
            const string expectedMessage = "A message";

            var receivedLog = false;
            FakeListener.LogMessage += (sender, e) =>
                {
                    receivedLog = true;
                    Assert.That(e, Is.Not.Null);
                    Assert.That(e.LogEvent, Is.EqualTo(expectedLogEvent));
                    Assert.That(e.Message, Is.EqualTo(expectedMessage));
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.True);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            Assert.That(receivedLog, Is.True);
        }

        [Test]
        public void Info_AlwaysLogIsFalse_IgnoreCatelLogsIsTrue_DoesNotWriteLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Info;
            const string expectedMessage = "A message";

            AlwaysLog = false;

            FakeListener.LogMessage += (sender, e) =>
                {
                    Assert.Fail();
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.True);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            // did not log to listener
        }

        [Test]
        public void Info_AlwaysLogIsFalse_IgnoreCatelLogsIsFalse_WritesLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Info;
            const string expectedMessage = "A message";

            AlwaysLog = false;
            FakeListener.IgnoreCatelLogging = false;

            var receivedLog = false;
            FakeListener.LogMessage += (sender, e) =>
                {
                    receivedLog = true;
                    Assert.That(e, Is.Not.Null);
                    Assert.That(e.LogEvent, Is.EqualTo(expectedLogEvent));
                    Assert.That(e.Message, Is.EqualTo(expectedMessage));
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.False);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            Assert.That(receivedLog, Is.True);
        }

        [Test]
        public void Status_AlwaysLogIsTrue_IgnoreCatelLogsIsTrue_WritesLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Status;
            const string expectedMessage = "A message";

            var receivedLog = false;
            FakeListener.LogMessage += (sender, e) =>
                {
                    receivedLog = true;
                    Assert.That(e, Is.Not.Null);
                    Assert.That(e.LogEvent, Is.EqualTo(expectedLogEvent));
                    Assert.That(e.Message, Is.EqualTo(expectedMessage));
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.True);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            Assert.That(receivedLog, Is.True);
        }

        [Test]
        public void Status_AlwaysLogIsFalse_IgnoreCatelLogsIsTrue_DoesNotWriteWriteLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Status;
            const string expectedMessage = "A message";

            AlwaysLog = false;

            FakeListener.LogMessage += (sender, e) =>
                {
                    Assert.Fail();
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.True);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            // did not log to listener
        }

        [Test]
        public void Status_AlwaysLogIsFalse_IgnoreCatelLogsIsFalse_WritesLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Status;
            const string expectedMessage = "A message";

            AlwaysLog = false;
            FakeListener.IgnoreCatelLogging = false;

            var receivedLog = false;
            FakeListener.LogMessage += (sender, e) =>
                {
                    receivedLog = true;
                    Assert.That(e, Is.Not.Null);
                    Assert.That(e.LogEvent, Is.EqualTo(expectedLogEvent));
                    Assert.That(e.Message, Is.EqualTo(expectedMessage));
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.False);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            Assert.That(receivedLog, Is.True);
        }

        [Test]
        public void Warning_AlwaysLogIsTrue_IgnoreCatelLogsIsTrue_WritesLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Warning;
            const string expectedMessage = "A message";

            var receivedLog = false;
            FakeListener.LogMessage += (sender, e) =>
                {
                    receivedLog = true;
                    Assert.That(e, Is.Not.Null);
                    Assert.That(e.LogEvent, Is.EqualTo(expectedLogEvent));
                    Assert.That(e.Message, Is.EqualTo(expectedMessage));
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.True);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            Assert.That(receivedLog, Is.True);
        }

        [Test]
        public void Warning_AlwaysLogIsFalse_IgnoreCatelLogsIsTrue_DoesNotWriteLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Warning;
            const string expectedMessage = "A message";

            AlwaysLog = false;

            FakeListener.LogMessage += (sender, e) =>
                {
                    Assert.Fail();
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.True);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            // did not log to listener
        }

        [Test]
        public void Warning_AlwaysLogIsFalse_IgnoreCatelLogsIsFalse_WritesLog()
        {
            // ARRANGE
            const LogEvent expectedLogEvent = LogEvent.Warning;
            const string expectedMessage = "A message";

            AlwaysLog = false;
            FakeListener.IgnoreCatelLogging = false;

            var receivedLog = false;
            FakeListener.LogMessage += (sender, e) =>
                {
                    receivedLog = true;
                    Assert.That(e, Is.Not.Null);
                    Assert.That(e.LogEvent, Is.EqualTo(expectedLogEvent));
                    Assert.That(e.Message, Is.EqualTo(expectedMessage));
                };

            Assert.That(FakeListener.IgnoreCatelLogging, Is.False);

            // ACT
            Logger.Write(expectedLogEvent, expectedMessage);


            // ASSERT
            Assert.That(receivedLog, Is.True);
        }
    }
}
