﻿namespace Catel.Tests.Data
{
    using System;
    using System.Linq.Expressions;
    using Catel.Data;
    using Catel.Runtime.Serialization;

    /// <summary>
    /// IniEntry Data object class which fully supports serialization, property changed notifications,
    /// backwards compatibility and error checking.
    /// </summary>
    [Serializable]
    public class IniEntry : ComparableModelBase
    {
        [ExcludeFromSerialization]
        public int _onSerializingCalls;
        [ExcludeFromSerialization]
        public int _onSerializedCalls;
        [ExcludeFromSerialization]
        public int _onDeserializingCalls;
        [ExcludeFromSerialization]
        public int _onDeserializedCalls;

        public void ClearSerializationCounters()
        {
            _onSerializingCalls = 0;
            _onSerializedCalls = 0;
            _onDeserializingCalls = 0;
            _onDeserializedCalls = 0;
        }

        protected override void OnSerializing()
        {
            _onSerializingCalls++;

            base.OnSerializing();
        }

        protected override void OnSerialized()
        {
            _onSerializedCalls++;

            base.OnSerialized();
        }

        protected override void OnDeserializing()
        {
            _onDeserializingCalls++;

            base.OnDeserializing();
        }

        protected override void OnDeserialized()
        {
            _onDeserializedCalls++;

            base.OnDeserialized();
        }

        /// <summary>
        ///   Initializes a new object from scratch.
        /// </summary>
        public IniEntry()
        {
        }

        /// <summary>
        ///   Gets or sets the group.
        /// </summary>
        public string Group
        {
            get { return GetValue<string>(GroupProperty); }
            set { SetValue(GroupProperty, value); }
        }

        /// <summary>
        ///   Register the property so it is known in the class.
        /// </summary>
        public static readonly IPropertyData GroupProperty = RegisterProperty("Group", string.Empty);

        /// <summary>
        ///   Gets or sets the key.
        /// </summary>
        public string Key
        {
            get { return GetValue<string>(KeyProperty); }
            set { SetValue(KeyProperty, value); }
        }

        /// <summary>
        ///   Register the property so it is known in the class.
        /// </summary>
        public static readonly IPropertyData KeyProperty = RegisterProperty("Key", string.Empty);

        /// <summary>
        ///   Gets or sets the value.
        /// </summary>
        public string Value
        {
            get { return GetValue<string>(ValueProperty); }
            set { SetValue(ValueProperty, value); }
        }

        /// <summary>
        ///   Register the property so it is known in the class.
        /// </summary>
        public static readonly IPropertyData ValueProperty = RegisterProperty("Value", string.Empty);

        /// <summary>
        /// Gets or sets the ini entry type.
        /// </summary>
        public IniEntryType IniEntryType
        {
            get { return GetValue<IniEntryType>(IniEntryTypeProperty); }
            set { SetValue(IniEntryTypeProperty, value); }
        }

        /// <summary>
        /// Register the IniEntryType property so it is known in the class.
        /// </summary>
        public static readonly IPropertyData IniEntryTypeProperty = RegisterProperty("IniEntryType", IniEntryType.Public);

        /// <summary>
        ///   Allows a test to invoke the Notify Property Changed on an object.
        /// </summary>
        /// <typeparam name = "TProperty">The type of the property.</typeparam>
        /// <param name = "propertyExpression">The property expression.</param>
        public new void RaisePropertyChanged<TProperty>(Expression<Func<TProperty>> propertyExpression)
        {
            base.RaisePropertyChanged(propertyExpression);
        }

        /// <summary>
        ///   Allows a test to invoke the Notify Property Changed on an object.
        /// </summary>
        /// <param name = "propertyName">Name of the property.</param>
        public new void RaisePropertyChanged(string propertyName)
        {
            base.RaisePropertyChanged(propertyName);
        }

        /// <summary>
        ///   Sets whether this object is read-only or not.
        /// </summary>
        /// <param name = "isReadOnly">if set to <c>true</c>, the object will become read-only.</param>
        public void SetReadOnly(bool isReadOnly)
        {
            IsReadOnly = isReadOnly;
        }

        public void SetValue<TValue>(string propertyName, TValue value)
        {
            base.SetValue(propertyName, value);
        }

        public new TValue GetValue<TValue>(string propertyName)
        {
            return base.GetValue<TValue>(propertyName);
        }
    }
}
