"""
File: main.py

Author: Brandon Dalton
Organization: Swiftly Detecting

Description: Entry point for MODEL FORGE.
"""

import os
import sys
import yaml
import time
import signal
import argparse
from datetime import datetime
from pipeline.tasks import Tasks
from renderers import custom_markdown
from modelforge import ModelForge

# Python 3.9+ is required for MODEL FORGE
PY_MAJOR_REQUIRED = 3
PY_MINOR_REQUIRED = 10
PY_RT_MAJOR = sys.version_info.major
PY_RT_MINOR = sys.version_info.minor
RT_EXCEPTION_MESSAGE: str = (
    f"🚨 MODEL FORGE requires Python {PY_MAJOR_REQUIRED}.{PY_MINOR_REQUIRED}+"
)

# How many times a task is allowed to fail before moving on.
MAX_FAIL_LIMIT = 10


# Handle exit signal: CTRL+C
def signal_handler(sig, frame):
    print("Exiting and destroying resources...")
    ModelForge.destroy()
    sys.exit(0)


signal.signal(signal.SIGINT, signal_handler)


def get_results_filename(results_file_path=None):
    if results_file_path is None:
        current_datetime = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
        return os.path.expanduser(
            f"~/modelforge/modelforge_result_{current_datetime}.yaml"
        )
    return results_file_path


def ensure_directory_exists(filepath):
    directory = os.path.dirname(filepath)
    if not os.path.exists(directory):
        os.makedirs(directory)


def save_to_file(tasks, filename):
    with open(filename, "w") as file:
        yaml.dump([task.to_dict() for task in tasks], file)


def execute_task_iteration(task, iteration):
    # The number of times that a task has failed.
    invalid_responses = 0
    print(f"Executing iteration {iteration}/{task.run_count}")
    critique = None
    past_result = None
    while True:
        # Get the grade for the task which will be the `eval_result` and the `critique`
        grade = task.execute_and_validate(critique, past_result)
        validated = grade.get("eval_result", False)
        critique = grade.get("critique", "")
        if validated:
            critique = None
            past_result = None
            print(f"Task completed ==> {iteration/task.run_count*100}%")
            return task.positive_results[-1]

        invalid_responses += 1
        if invalid_responses >= MAX_FAIL_LIMIT:
            print("🚨 Reached maximum invalid responses. Moving to the next task!!")
            break
        if len(task.negative_results) > 0:
            past_result = task.negative_results[-1]
            print(
                f"❌ Not validated. Trying again... Here's what was completed:\n"
                f"---\n{past_result}\n"
                f"---\nEvaluation:\n{grade}\n---\n"
            )
        else:
            print("❌ Not validated. Trying again...\n")


def route_tasks(tasks, results_file_path=None, save_yaml=False, save_md=True):
    results_file = get_results_filename(results_file_path)
    ensure_directory_exists(results_file)

    for task in tasks:
        start_time = time.time()
        print(
            f"\n{task.name} with the {task.agent.base_model} model\n"
            # f"PROMPT: \n```\n{task.prompt}```"
        )
        for iteration in range(1, task.run_count + 1):
            iteration_start_time = time.time()
            positive_result = execute_task_iteration(task, iteration)
            print(
                f"✅ ===> {round(time.time() - iteration_start_time, 4)}s\n{positive_result}\n<===\n"
            )
            if save_yaml:
                save_to_file(tasks, results_file)

        average_time = (time.time() - start_time) / task.run_count
        print(f"\t⏰ Average time per iteration: {round(average_time, 4)}s")

    print("🤖 Done with all tasking!")
    for task in tasks:
        custom_markdown.render_and_write_code_list_as_markdown(
            task.positive_results, save=save_md
        )


def main():
    parser = argparse.ArgumentParser(description="Process LM tasking YAML file.")
    parser.add_argument(
        "config_file", type=str, help="Path to your task configuration YAML file"
    )
    parser.add_argument(
        "-o",
        "--output",
        type=str,
        default=f"./research_results_{datetime.now().strftime('%Y%m%d-%H%M%S')}.yaml",
        help="Path to save the results to. By default we'll save them as a timestamped YAML file in the current directory.",
    )

    parser.add_argument(
        "-sa",
        "--save-all",
        action="store_true",
        default=False,
        help="Should we save both the results YAML and the markdown generated by the LLMs?",
    )

    parser.add_argument(
        "-sm",
        "--save-markdown",
        action="store_true",
        default=True,
        help="Should save the markdown generated by the LLMs?",
    )

    parser.add_argument(
        "-sy",
        "--save-yaml",
        action="store_true",
        default=False,
        help="Should save the full YAML serialized results?",
    )

    args = parser.parse_args()
    tasks = Tasks(args.config_file).tasks
    if args.save_all:
        route_tasks(tasks, results_file_path=args.output, save_yaml=True, save_md=True)
    elif args.save_yaml:
        route_tasks(tasks, results_file_path=args.output, save_yaml=True)
    elif args.save_markdown:
        route_tasks(tasks, results_file_path=args.output, save_md=True)


if __name__ == "__main__":
    if PY_RT_MAJOR < PY_MAJOR_REQUIRED:
        raise Exception(RT_EXCEPTION_MESSAGE)
    elif PY_RT_MAJOR == PY_MAJOR_REQUIRED and PY_RT_MINOR < PY_MINOR_REQUIRED:
        raise Exception(RT_EXCEPTION_MESSAGE)
    main()
