// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

// <auto-generated/>

#nullable disable

using System;
using System.Collections.Generic;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using System.Xml.Linq;
using Azure.Core;
using Azure.Core.Pipeline;
using Azure.Data.Tables.Models;

namespace Azure.Data.Tables
{
    internal partial class TableRestClient
    {
        private readonly HttpPipeline _pipeline;
        private readonly string _url;
        private readonly string _version;

        /// <summary> The ClientDiagnostics is used to provide tracing support for the client library. </summary>
        internal ClientDiagnostics ClientDiagnostics { get; }

        /// <summary> Initializes a new instance of TableRestClient. </summary>
        /// <param name="clientDiagnostics"> The handler for diagnostic messaging in the client. </param>
        /// <param name="pipeline"> The HTTP pipeline for sending and receiving REST requests and responses. </param>
        /// <param name="url"> The URL of the service account or table that is the target of the desired operation. </param>
        /// <param name="version"> Specifies the version of the operation to use for this request. The default value is "2019-02-02". </param>
        /// <exception cref="ArgumentNullException"> <paramref name="clientDiagnostics"/>, <paramref name="pipeline"/>, <paramref name="url"/> or <paramref name="version"/> is null. </exception>
        public TableRestClient(ClientDiagnostics clientDiagnostics, HttpPipeline pipeline, string url, string version)
        {
            ClientDiagnostics = clientDiagnostics ?? throw new ArgumentNullException(nameof(clientDiagnostics));
            _pipeline = pipeline ?? throw new ArgumentNullException(nameof(pipeline));
            _url = url ?? throw new ArgumentNullException(nameof(url));
            _version = version ?? throw new ArgumentNullException(nameof(version));
        }

        internal HttpMessage CreateQueryRequest(string nextTableName, QueryOptions queryOptions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/Tables", false);
            if (queryOptions?.Format != null)
            {
                uri.AppendQuery("$format", queryOptions.Format.Value.ToString(), true);
            }
            if (queryOptions?.Top != null)
            {
                uri.AppendQuery("$top", queryOptions.Top.Value, true);
            }
            if (queryOptions?.Select != null)
            {
                uri.AppendQuery("$select", queryOptions.Select, true);
            }
            if (queryOptions?.Filter != null)
            {
                uri.AppendQuery("$filter", queryOptions.Filter, true);
            }
            if (nextTableName != null)
            {
                uri.AppendQuery("NextTableName", nextTableName, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("DataServiceVersion", "3.0");
            request.Headers.Add("Accept", "application/json;odata=minimalmetadata");
            return message;
        }

        /// <summary> Queries tables under the given account. </summary>
        /// <param name="nextTableName"> A table query continuation token from a previous call. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<ResponseWithHeaders<TableQueryResponse, TableQueryHeaders>> QueryAsync(string nextTableName = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateQueryRequest(nextTableName, queryOptions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new TableQueryHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        TableQueryResponse value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = TableQueryResponse.DeserializeTableQueryResponse(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Queries tables under the given account. </summary>
        /// <param name="nextTableName"> A table query continuation token from a previous call. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public ResponseWithHeaders<TableQueryResponse, TableQueryHeaders> Query(string nextTableName = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateQueryRequest(nextTableName, queryOptions);
            _pipeline.Send(message, cancellationToken);
            var headers = new TableQueryHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        TableQueryResponse value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = TableQueryResponse.DeserializeTableQueryResponse(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateCreateRequest(TableProperties tableProperties, ResponseFormat? responsePreference, QueryOptions queryOptions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/Tables", false);
            if (queryOptions?.Format != null)
            {
                uri.AppendQuery("$format", queryOptions.Format.Value.ToString(), true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("DataServiceVersion", "3.0");
            if (responsePreference != null)
            {
                request.Headers.Add("Prefer", responsePreference.Value.ToString());
            }
            request.Headers.Add("Accept", "application/json;odata=minimalmetadata");
            request.Headers.Add("Content-Type", "application/json;odata=nometadata");
            var content = new Utf8JsonRequestContent();
            content.JsonWriter.WriteObjectValue(tableProperties);
            request.Content = content;
            return message;
        }

        /// <summary> Creates a new table under the given account. </summary>
        /// <param name="tableProperties"> The Table properties. </param>
        /// <param name="responsePreference"> Specifies whether the response should include the inserted entity in the payload. Possible values are return-no-content and return-content. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="tableProperties"/> is null. </exception>
        public async Task<ResponseWithHeaders<TableResponse, TableCreateHeaders>> CreateAsync(TableProperties tableProperties, ResponseFormat? responsePreference = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (tableProperties == null)
            {
                throw new ArgumentNullException(nameof(tableProperties));
            }

            using var message = CreateCreateRequest(tableProperties, responsePreference, queryOptions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new TableCreateHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    {
                        TableResponse value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = TableResponse.DeserializeTableResponse(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                case 204:
                    return ResponseWithHeaders.FromValue((TableResponse)null, headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Creates a new table under the given account. </summary>
        /// <param name="tableProperties"> The Table properties. </param>
        /// <param name="responsePreference"> Specifies whether the response should include the inserted entity in the payload. Possible values are return-no-content and return-content. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="tableProperties"/> is null. </exception>
        public ResponseWithHeaders<TableResponse, TableCreateHeaders> Create(TableProperties tableProperties, ResponseFormat? responsePreference = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (tableProperties == null)
            {
                throw new ArgumentNullException(nameof(tableProperties));
            }

            using var message = CreateCreateRequest(tableProperties, responsePreference, queryOptions);
            _pipeline.Send(message, cancellationToken);
            var headers = new TableCreateHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    {
                        TableResponse value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = TableResponse.DeserializeTableResponse(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                case 204:
                    return ResponseWithHeaders.FromValue((TableResponse)null, headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateCreateRequest(RequestContent content, string format, string responsePreference, RequestContext context)
        {
            var message = _pipeline.CreateMessage(context, ResponseClassifier201204);
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/Tables", false);
            if (format != null)
            {
                uri.AppendQuery("$format", format, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("DataServiceVersion", "3.0");
            request.Headers.Add("Accept", "application/json;odata=minimalmetadata");
            if (responsePreference != null)
            {
                request.Headers.Add("Prefer", responsePreference);
            }
            request.Headers.Add("Content-Type", "application/json;odata=nometadata");
            request.Content = content;
            return message;
        }

        /// <summary>
        /// [Protocol Method] Creates a new table under the given account.
        /// <list type="bullet">
        /// <item>
        /// <description>
        /// This <see href="https://github.com/Azure/azure-sdk-for-net/blob/main/sdk/core/Azure.Core/samples/ProtocolMethods.md">protocol method</see> allows explicit creation of the request and processing of the response for advanced scenarios.
        /// </description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="content"> The content to send as the body of the request. </param>
        /// <param name="format"> Specifies the media type for the response. Allowed values: "application/json;odata=nometadata" | "application/json;odata=minimalmetadata" | "application/json;odata=fullmetadata". </param>
        /// <param name="responsePreference"> Specifies whether the response should include the inserted entity in the payload. Possible values are return-no-content and return-content. Allowed values: "return-no-content" | "return-content". </param>
        /// <param name="context"> The request context, which can override default behaviors of the client pipeline on a per-call basis. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="content"/> is null. </exception>
        /// <exception cref="RequestFailedException"> Service returned a non-success status code. </exception>
        /// <returns> The response returned from the service. </returns>
        public virtual async Task<Response> CreateAsync(RequestContent content, string format = null, string responsePreference = null, RequestContext context = null)
        {
            Argument.AssertNotNull(content, nameof(content));

            using var scope = ClientDiagnostics.CreateScope("Table.Create");
            scope.Start();
            try
            {
                using HttpMessage message = CreateCreateRequest(content, format, responsePreference, context);
                return await _pipeline.ProcessMessageAsync(message, context).ConfigureAwait(false);
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        /// <summary>
        /// [Protocol Method] Creates a new table under the given account.
        /// <list type="bullet">
        /// <item>
        /// <description>
        /// This <see href="https://github.com/Azure/azure-sdk-for-net/blob/main/sdk/core/Azure.Core/samples/ProtocolMethods.md">protocol method</see> allows explicit creation of the request and processing of the response for advanced scenarios.
        /// </description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="content"> The content to send as the body of the request. </param>
        /// <param name="format"> Specifies the media type for the response. Allowed values: "application/json;odata=nometadata" | "application/json;odata=minimalmetadata" | "application/json;odata=fullmetadata". </param>
        /// <param name="responsePreference"> Specifies whether the response should include the inserted entity in the payload. Possible values are return-no-content and return-content. Allowed values: "return-no-content" | "return-content". </param>
        /// <param name="context"> The request context, which can override default behaviors of the client pipeline on a per-call basis. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="content"/> is null. </exception>
        /// <exception cref="RequestFailedException"> Service returned a non-success status code. </exception>
        /// <returns> The response returned from the service. </returns>
        public virtual Response Create(RequestContent content, string format = null, string responsePreference = null, RequestContext context = null)
        {
            Argument.AssertNotNull(content, nameof(content));

            using var scope = ClientDiagnostics.CreateScope("Table.Create");
            scope.Start();
            try
            {
                using HttpMessage message = CreateCreateRequest(content, format, responsePreference, context);
                return _pipeline.ProcessMessage(message, context);
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        internal HttpMessage CreateDeleteRequest(string table)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/Tables('", false);
            uri.AppendPath(table, true);
            uri.AppendPath("')", false);
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Operation permanently deletes the specified table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/> is null. </exception>
        public async Task<ResponseWithHeaders<TableDeleteHeaders>> DeleteAsync(string table, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }

            using var message = CreateDeleteRequest(table);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new TableDeleteHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Operation permanently deletes the specified table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/> is null. </exception>
        public ResponseWithHeaders<TableDeleteHeaders> Delete(string table, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }

            using var message = CreateDeleteRequest(table);
            _pipeline.Send(message, cancellationToken);
            var headers = new TableDeleteHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateDeleteRequest(string table, RequestContext context)
        {
            var message = _pipeline.CreateMessage(context, ResponseClassifier204);
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/Tables('", false);
            uri.AppendPath(table, true);
            uri.AppendPath("')", false);
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary>
        /// [Protocol Method] Operation permanently deletes the specified table.
        /// <list type="bullet">
        /// <item>
        /// <description>
        /// This <see href="https://github.com/Azure/azure-sdk-for-net/blob/main/sdk/core/Azure.Core/samples/ProtocolMethods.md">protocol method</see> allows explicit creation of the request and processing of the response for advanced scenarios.
        /// </description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="context"> The request context, which can override default behaviors of the client pipeline on a per-call basis. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/> is null. </exception>
        /// <exception cref="ArgumentException"> <paramref name="table"/> is an empty string, and was expected to be non-empty. </exception>
        /// <exception cref="RequestFailedException"> Service returned a non-success status code. </exception>
        /// <returns> The response returned from the service. </returns>
        public virtual async Task<Response> DeleteAsync(string table, RequestContext context = null)
        {
            Argument.AssertNotNullOrEmpty(table, nameof(table));

            using var scope = ClientDiagnostics.CreateScope("Table.Delete");
            scope.Start();
            try
            {
                using HttpMessage message = CreateDeleteRequest(table, context);
                return await _pipeline.ProcessMessageAsync(message, context).ConfigureAwait(false);
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        /// <summary>
        /// [Protocol Method] Operation permanently deletes the specified table.
        /// <list type="bullet">
        /// <item>
        /// <description>
        /// This <see href="https://github.com/Azure/azure-sdk-for-net/blob/main/sdk/core/Azure.Core/samples/ProtocolMethods.md">protocol method</see> allows explicit creation of the request and processing of the response for advanced scenarios.
        /// </description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="context"> The request context, which can override default behaviors of the client pipeline on a per-call basis. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/> is null. </exception>
        /// <exception cref="ArgumentException"> <paramref name="table"/> is an empty string, and was expected to be non-empty. </exception>
        /// <exception cref="RequestFailedException"> Service returned a non-success status code. </exception>
        /// <returns> The response returned from the service. </returns>
        public virtual Response Delete(string table, RequestContext context = null)
        {
            Argument.AssertNotNullOrEmpty(table, nameof(table));

            using var scope = ClientDiagnostics.CreateScope("Table.Delete");
            scope.Start();
            try
            {
                using HttpMessage message = CreateDeleteRequest(table, context);
                return _pipeline.ProcessMessage(message, context);
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        internal HttpMessage CreateQueryEntitiesRequest(string table, int? timeout, string nextPartitionKey, string nextRowKey, QueryOptions queryOptions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendPath(table, true);
            uri.AppendPath("()", false);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            if (queryOptions?.Format != null)
            {
                uri.AppendQuery("$format", queryOptions.Format.Value.ToString(), true);
            }
            if (queryOptions?.Top != null)
            {
                uri.AppendQuery("$top", queryOptions.Top.Value, true);
            }
            if (queryOptions?.Select != null)
            {
                uri.AppendQuery("$select", queryOptions.Select, true);
            }
            if (queryOptions?.Filter != null)
            {
                uri.AppendQuery("$filter", queryOptions.Filter, true);
            }
            if (nextPartitionKey != null)
            {
                uri.AppendQuery("NextPartitionKey", nextPartitionKey, true);
            }
            if (nextRowKey != null)
            {
                uri.AppendQuery("NextRowKey", nextRowKey, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("DataServiceVersion", "3.0");
            request.Headers.Add("Accept", "application/json;odata=minimalmetadata");
            return message;
        }

        /// <summary> Queries entities in a table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="nextPartitionKey"> An entity query continuation token from a previous call. </param>
        /// <param name="nextRowKey"> An entity query continuation token from a previous call. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/> is null. </exception>
        public async Task<ResponseWithHeaders<TableEntityQueryResponse, TableQueryEntitiesHeaders>> QueryEntitiesAsync(string table, int? timeout = null, string nextPartitionKey = null, string nextRowKey = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }

            using var message = CreateQueryEntitiesRequest(table, timeout, nextPartitionKey, nextRowKey, queryOptions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new TableQueryEntitiesHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        TableEntityQueryResponse value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = TableEntityQueryResponse.DeserializeTableEntityQueryResponse(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Queries entities in a table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="nextPartitionKey"> An entity query continuation token from a previous call. </param>
        /// <param name="nextRowKey"> An entity query continuation token from a previous call. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/> is null. </exception>
        public ResponseWithHeaders<TableEntityQueryResponse, TableQueryEntitiesHeaders> QueryEntities(string table, int? timeout = null, string nextPartitionKey = null, string nextRowKey = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }

            using var message = CreateQueryEntitiesRequest(table, timeout, nextPartitionKey, nextRowKey, queryOptions);
            _pipeline.Send(message, cancellationToken);
            var headers = new TableQueryEntitiesHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        TableEntityQueryResponse value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = TableEntityQueryResponse.DeserializeTableEntityQueryResponse(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateQueryEntityWithPartitionAndRowKeyRequest(string table, string partitionKey, string rowKey, int? timeout, QueryOptions queryOptions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendPath(table, true);
            uri.AppendPath("(PartitionKey='", false);
            uri.AppendPath(partitionKey, true);
            uri.AppendPath("',RowKey='", false);
            uri.AppendPath(rowKey, true);
            uri.AppendPath("')", false);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            if (queryOptions?.Format != null)
            {
                uri.AppendQuery("$format", queryOptions.Format.Value.ToString(), true);
            }
            if (queryOptions?.Select != null)
            {
                uri.AppendQuery("$select", queryOptions.Select, true);
            }
            if (queryOptions?.Filter != null)
            {
                uri.AppendQuery("$filter", queryOptions.Filter, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("DataServiceVersion", "3.0");
            request.Headers.Add("Accept", "application/json;odata=minimalmetadata");
            return message;
        }

        /// <summary> Queries a single entity in a table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="partitionKey"> The partition key of the entity. </param>
        /// <param name="rowKey"> The row key of the entity. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/>, <paramref name="partitionKey"/> or <paramref name="rowKey"/> is null. </exception>
        public async Task<ResponseWithHeaders<IReadOnlyDictionary<string, object>, TableQueryEntityWithPartitionAndRowKeyHeaders>> QueryEntityWithPartitionAndRowKeyAsync(string table, string partitionKey, string rowKey, int? timeout = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }
            if (partitionKey == null)
            {
                throw new ArgumentNullException(nameof(partitionKey));
            }
            if (rowKey == null)
            {
                throw new ArgumentNullException(nameof(rowKey));
            }

            using var message = CreateQueryEntityWithPartitionAndRowKeyRequest(table, partitionKey, rowKey, timeout, queryOptions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new TableQueryEntityWithPartitionAndRowKeyHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        IReadOnlyDictionary<string, object> value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        Dictionary<string, object> dictionary = new Dictionary<string, object>();
                        foreach (var property in document.RootElement.EnumerateObject())
                        {
                            if (property.Value.ValueKind == JsonValueKind.Null)
                            {
                                dictionary.Add(property.Name, null);
                            }
                            else
                            {
                                dictionary.Add(property.Name, property.Value.GetObject());
                            }
                        }
                        value = dictionary;
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Queries a single entity in a table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="partitionKey"> The partition key of the entity. </param>
        /// <param name="rowKey"> The row key of the entity. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/>, <paramref name="partitionKey"/> or <paramref name="rowKey"/> is null. </exception>
        public ResponseWithHeaders<IReadOnlyDictionary<string, object>, TableQueryEntityWithPartitionAndRowKeyHeaders> QueryEntityWithPartitionAndRowKey(string table, string partitionKey, string rowKey, int? timeout = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }
            if (partitionKey == null)
            {
                throw new ArgumentNullException(nameof(partitionKey));
            }
            if (rowKey == null)
            {
                throw new ArgumentNullException(nameof(rowKey));
            }

            using var message = CreateQueryEntityWithPartitionAndRowKeyRequest(table, partitionKey, rowKey, timeout, queryOptions);
            _pipeline.Send(message, cancellationToken);
            var headers = new TableQueryEntityWithPartitionAndRowKeyHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        IReadOnlyDictionary<string, object> value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        Dictionary<string, object> dictionary = new Dictionary<string, object>();
                        foreach (var property in document.RootElement.EnumerateObject())
                        {
                            if (property.Value.ValueKind == JsonValueKind.Null)
                            {
                                dictionary.Add(property.Name, null);
                            }
                            else
                            {
                                dictionary.Add(property.Name, property.Value.GetObject());
                            }
                        }
                        value = dictionary;
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateQueryEntityWithPartitionAndRowKeyRequest(string table, string partitionKey, string rowKey, int? timeout, string format, string select, string filter, RequestContext context)
        {
            var message = _pipeline.CreateMessage(context, ResponseClassifier200);
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendPath(table, true);
            uri.AppendPath("(PartitionKey='", false);
            uri.AppendPath(partitionKey, true);
            uri.AppendPath("',RowKey='", false);
            uri.AppendPath(rowKey, true);
            uri.AppendPath("')", false);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            if (format != null)
            {
                uri.AppendQuery("$format", format, true);
            }
            if (select != null)
            {
                uri.AppendQuery("$select", select, true);
            }
            if (filter != null)
            {
                uri.AppendQuery("$filter", filter, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("DataServiceVersion", "3.0");
            request.Headers.Add("Accept", "application/json;odata=minimalmetadata");
            return message;
        }

        internal HttpMessage CreateUpdateEntityRequest(string table, string partitionKey, string rowKey, int? timeout, string ifMatch, IDictionary<string, object> tableEntityProperties, QueryOptions queryOptions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendPath(table, true);
            uri.AppendPath("(PartitionKey='", false);
            uri.AppendPath(partitionKey, true);
            uri.AppendPath("',RowKey='", false);
            uri.AppendPath(rowKey, true);
            uri.AppendPath("')", false);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            if (queryOptions?.Format != null)
            {
                uri.AppendQuery("$format", queryOptions.Format.Value.ToString(), true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("DataServiceVersion", "3.0");
            if (ifMatch != null)
            {
                request.Headers.Add("If-Match", ifMatch);
            }
            request.Headers.Add("Accept", "application/json");
            if (tableEntityProperties != null)
            {
                request.Headers.Add("Content-Type", "application/json");
                var content = new Utf8JsonRequestContent();
                content.JsonWriter.WriteStartObject();
                foreach (var item in tableEntityProperties)
                {
                    content.JsonWriter.WritePropertyName(item.Key);
                    if (item.Value == null)
                    {
                        content.JsonWriter.WriteNullValue();
                        continue;
                    }
                    content.JsonWriter.WriteObjectValue<object>(item.Value);
                }
                content.JsonWriter.WriteEndObject();
                request.Content = content;
            }
            return message;
        }

        /// <summary> Update entity in a table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="partitionKey"> The partition key of the entity. </param>
        /// <param name="rowKey"> The row key of the entity. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="ifMatch"> Match condition for an entity to be updated. If specified and a matching entity is not found, an error will be raised. To force an unconditional update, set to the wildcard character (*). If not specified, an insert will be performed when no existing entity is found to update and a replace will be performed if an existing entity is found. </param>
        /// <param name="tableEntityProperties"> The properties for the table entity. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/>, <paramref name="partitionKey"/> or <paramref name="rowKey"/> is null. </exception>
        public async Task<ResponseWithHeaders<TableUpdateEntityHeaders>> UpdateEntityAsync(string table, string partitionKey, string rowKey, int? timeout = null, string ifMatch = null, IDictionary<string, object> tableEntityProperties = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }
            if (partitionKey == null)
            {
                throw new ArgumentNullException(nameof(partitionKey));
            }
            if (rowKey == null)
            {
                throw new ArgumentNullException(nameof(rowKey));
            }

            using var message = CreateUpdateEntityRequest(table, partitionKey, rowKey, timeout, ifMatch, tableEntityProperties, queryOptions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new TableUpdateEntityHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Update entity in a table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="partitionKey"> The partition key of the entity. </param>
        /// <param name="rowKey"> The row key of the entity. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="ifMatch"> Match condition for an entity to be updated. If specified and a matching entity is not found, an error will be raised. To force an unconditional update, set to the wildcard character (*). If not specified, an insert will be performed when no existing entity is found to update and a replace will be performed if an existing entity is found. </param>
        /// <param name="tableEntityProperties"> The properties for the table entity. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/>, <paramref name="partitionKey"/> or <paramref name="rowKey"/> is null. </exception>
        public ResponseWithHeaders<TableUpdateEntityHeaders> UpdateEntity(string table, string partitionKey, string rowKey, int? timeout = null, string ifMatch = null, IDictionary<string, object> tableEntityProperties = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }
            if (partitionKey == null)
            {
                throw new ArgumentNullException(nameof(partitionKey));
            }
            if (rowKey == null)
            {
                throw new ArgumentNullException(nameof(rowKey));
            }

            using var message = CreateUpdateEntityRequest(table, partitionKey, rowKey, timeout, ifMatch, tableEntityProperties, queryOptions);
            _pipeline.Send(message, cancellationToken);
            var headers = new TableUpdateEntityHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateMergeEntityRequest(string table, string partitionKey, string rowKey, int? timeout, string ifMatch, IDictionary<string, object> tableEntityProperties, QueryOptions queryOptions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Patch;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendPath(table, true);
            uri.AppendPath("(PartitionKey='", false);
            uri.AppendPath(partitionKey, true);
            uri.AppendPath("',RowKey='", false);
            uri.AppendPath(rowKey, true);
            uri.AppendPath("')", false);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            if (queryOptions?.Format != null)
            {
                uri.AppendQuery("$format", queryOptions.Format.Value.ToString(), true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("DataServiceVersion", "3.0");
            if (ifMatch != null)
            {
                request.Headers.Add("If-Match", ifMatch);
            }
            request.Headers.Add("Accept", "application/json");
            if (tableEntityProperties != null)
            {
                request.Headers.Add("Content-Type", "application/json");
                var content = new Utf8JsonRequestContent();
                content.JsonWriter.WriteStartObject();
                foreach (var item in tableEntityProperties)
                {
                    content.JsonWriter.WritePropertyName(item.Key);
                    if (item.Value == null)
                    {
                        content.JsonWriter.WriteNullValue();
                        continue;
                    }
                    content.JsonWriter.WriteObjectValue<object>(item.Value);
                }
                content.JsonWriter.WriteEndObject();
                request.Content = content;
            }
            return message;
        }

        /// <summary> Merge entity in a table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="partitionKey"> The partition key of the entity. </param>
        /// <param name="rowKey"> The row key of the entity. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="ifMatch"> Match condition for an entity to be updated. If specified and a matching entity is not found, an error will be raised. To force an unconditional update, set to the wildcard character (*). If not specified, an insert will be performed when no existing entity is found to update and a merge will be performed if an existing entity is found. </param>
        /// <param name="tableEntityProperties"> The properties for the table entity. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/>, <paramref name="partitionKey"/> or <paramref name="rowKey"/> is null. </exception>
        public async Task<ResponseWithHeaders<TableMergeEntityHeaders>> MergeEntityAsync(string table, string partitionKey, string rowKey, int? timeout = null, string ifMatch = null, IDictionary<string, object> tableEntityProperties = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }
            if (partitionKey == null)
            {
                throw new ArgumentNullException(nameof(partitionKey));
            }
            if (rowKey == null)
            {
                throw new ArgumentNullException(nameof(rowKey));
            }

            using var message = CreateMergeEntityRequest(table, partitionKey, rowKey, timeout, ifMatch, tableEntityProperties, queryOptions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new TableMergeEntityHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Merge entity in a table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="partitionKey"> The partition key of the entity. </param>
        /// <param name="rowKey"> The row key of the entity. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="ifMatch"> Match condition for an entity to be updated. If specified and a matching entity is not found, an error will be raised. To force an unconditional update, set to the wildcard character (*). If not specified, an insert will be performed when no existing entity is found to update and a merge will be performed if an existing entity is found. </param>
        /// <param name="tableEntityProperties"> The properties for the table entity. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/>, <paramref name="partitionKey"/> or <paramref name="rowKey"/> is null. </exception>
        public ResponseWithHeaders<TableMergeEntityHeaders> MergeEntity(string table, string partitionKey, string rowKey, int? timeout = null, string ifMatch = null, IDictionary<string, object> tableEntityProperties = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }
            if (partitionKey == null)
            {
                throw new ArgumentNullException(nameof(partitionKey));
            }
            if (rowKey == null)
            {
                throw new ArgumentNullException(nameof(rowKey));
            }

            using var message = CreateMergeEntityRequest(table, partitionKey, rowKey, timeout, ifMatch, tableEntityProperties, queryOptions);
            _pipeline.Send(message, cancellationToken);
            var headers = new TableMergeEntityHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateDeleteEntityRequest(string table, string partitionKey, string rowKey, string ifMatch, int? timeout, QueryOptions queryOptions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendPath(table, true);
            uri.AppendPath("(PartitionKey='", false);
            uri.AppendPath(partitionKey, true);
            uri.AppendPath("',RowKey='", false);
            uri.AppendPath(rowKey, true);
            uri.AppendPath("')", false);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            if (queryOptions?.Format != null)
            {
                uri.AppendQuery("$format", queryOptions.Format.Value.ToString(), true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("DataServiceVersion", "3.0");
            request.Headers.Add("If-Match", ifMatch);
            request.Headers.Add("Accept", "application/json;odata=minimalmetadata");
            return message;
        }

        /// <summary> Deletes the specified entity in a table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="partitionKey"> The partition key of the entity. </param>
        /// <param name="rowKey"> The row key of the entity. </param>
        /// <param name="ifMatch"> Match condition for an entity to be deleted. If specified and a matching entity is not found, an error will be raised. To force an unconditional delete, set to the wildcard character (*). </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/>, <paramref name="partitionKey"/>, <paramref name="rowKey"/> or <paramref name="ifMatch"/> is null. </exception>
        public async Task<ResponseWithHeaders<TableDeleteEntityHeaders>> DeleteEntityAsync(string table, string partitionKey, string rowKey, string ifMatch, int? timeout = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }
            if (partitionKey == null)
            {
                throw new ArgumentNullException(nameof(partitionKey));
            }
            if (rowKey == null)
            {
                throw new ArgumentNullException(nameof(rowKey));
            }
            if (ifMatch == null)
            {
                throw new ArgumentNullException(nameof(ifMatch));
            }

            using var message = CreateDeleteEntityRequest(table, partitionKey, rowKey, ifMatch, timeout, queryOptions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new TableDeleteEntityHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Deletes the specified entity in a table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="partitionKey"> The partition key of the entity. </param>
        /// <param name="rowKey"> The row key of the entity. </param>
        /// <param name="ifMatch"> Match condition for an entity to be deleted. If specified and a matching entity is not found, an error will be raised. To force an unconditional delete, set to the wildcard character (*). </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/>, <paramref name="partitionKey"/>, <paramref name="rowKey"/> or <paramref name="ifMatch"/> is null. </exception>
        public ResponseWithHeaders<TableDeleteEntityHeaders> DeleteEntity(string table, string partitionKey, string rowKey, string ifMatch, int? timeout = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }
            if (partitionKey == null)
            {
                throw new ArgumentNullException(nameof(partitionKey));
            }
            if (rowKey == null)
            {
                throw new ArgumentNullException(nameof(rowKey));
            }
            if (ifMatch == null)
            {
                throw new ArgumentNullException(nameof(ifMatch));
            }

            using var message = CreateDeleteEntityRequest(table, partitionKey, rowKey, ifMatch, timeout, queryOptions);
            _pipeline.Send(message, cancellationToken);
            var headers = new TableDeleteEntityHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateInsertEntityRequest(string table, int? timeout, ResponseFormat? responsePreference, IDictionary<string, object> tableEntityProperties, QueryOptions queryOptions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendPath(table, true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            if (queryOptions?.Format != null)
            {
                uri.AppendQuery("$format", queryOptions.Format.Value.ToString(), true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("DataServiceVersion", "3.0");
            if (responsePreference != null)
            {
                request.Headers.Add("Prefer", responsePreference.Value.ToString());
            }
            request.Headers.Add("Accept", "application/json;odata=minimalmetadata");
            if (tableEntityProperties != null)
            {
                request.Headers.Add("Content-Type", "application/json;odata=nometadata");
                var content = new Utf8JsonRequestContent();
                content.JsonWriter.WriteStartObject();
                foreach (var item in tableEntityProperties)
                {
                    content.JsonWriter.WritePropertyName(item.Key);
                    if (item.Value == null)
                    {
                        content.JsonWriter.WriteNullValue();
                        continue;
                    }
                    content.JsonWriter.WriteObjectValue<object>(item.Value);
                }
                content.JsonWriter.WriteEndObject();
                request.Content = content;
            }
            return message;
        }

        /// <summary> Insert entity in a table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="responsePreference"> Specifies whether the response should include the inserted entity in the payload. Possible values are return-no-content and return-content. </param>
        /// <param name="tableEntityProperties"> The properties for the table entity. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/> is null. </exception>
        public async Task<ResponseWithHeaders<IReadOnlyDictionary<string, object>, TableInsertEntityHeaders>> InsertEntityAsync(string table, int? timeout = null, ResponseFormat? responsePreference = null, IDictionary<string, object> tableEntityProperties = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }

            using var message = CreateInsertEntityRequest(table, timeout, responsePreference, tableEntityProperties, queryOptions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new TableInsertEntityHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    {
                        IReadOnlyDictionary<string, object> value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        Dictionary<string, object> dictionary = new Dictionary<string, object>();
                        foreach (var property in document.RootElement.EnumerateObject())
                        {
                            if (property.Value.ValueKind == JsonValueKind.Null)
                            {
                                dictionary.Add(property.Name, null);
                            }
                            else
                            {
                                dictionary.Add(property.Name, property.Value.GetObject());
                            }
                        }
                        value = dictionary;
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                case 204:
                    return ResponseWithHeaders.FromValue((IReadOnlyDictionary<string, object>)null, headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Insert entity in a table. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="responsePreference"> Specifies whether the response should include the inserted entity in the payload. Possible values are return-no-content and return-content. </param>
        /// <param name="tableEntityProperties"> The properties for the table entity. </param>
        /// <param name="queryOptions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/> is null. </exception>
        public ResponseWithHeaders<IReadOnlyDictionary<string, object>, TableInsertEntityHeaders> InsertEntity(string table, int? timeout = null, ResponseFormat? responsePreference = null, IDictionary<string, object> tableEntityProperties = null, QueryOptions queryOptions = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }

            using var message = CreateInsertEntityRequest(table, timeout, responsePreference, tableEntityProperties, queryOptions);
            _pipeline.Send(message, cancellationToken);
            var headers = new TableInsertEntityHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    {
                        IReadOnlyDictionary<string, object> value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        Dictionary<string, object> dictionary = new Dictionary<string, object>();
                        foreach (var property in document.RootElement.EnumerateObject())
                        {
                            if (property.Value.ValueKind == JsonValueKind.Null)
                            {
                                dictionary.Add(property.Name, null);
                            }
                            else
                            {
                                dictionary.Add(property.Name, property.Value.GetObject());
                            }
                        }
                        value = dictionary;
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                case 204:
                    return ResponseWithHeaders.FromValue((IReadOnlyDictionary<string, object>)null, headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetAccessPolicyRequest(string table, int? timeout)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendPath(table, true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            uri.AppendQuery("comp", "acl", true);
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Retrieves details about any stored access policies specified on the table that may be used with Shared Access Signatures. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/> is null. </exception>
        public async Task<ResponseWithHeaders<IReadOnlyList<TableSignedIdentifier>, TableGetAccessPolicyHeaders>> GetAccessPolicyAsync(string table, int? timeout = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }

            using var message = CreateGetAccessPolicyRequest(table, timeout);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new TableGetAccessPolicyHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        IReadOnlyList<TableSignedIdentifier> value = default;
                        var document = XDocument.Load(message.Response.ContentStream, LoadOptions.PreserveWhitespace);
                        if (document.Element("SignedIdentifiers") is XElement signedIdentifiersElement)
                        {
                            var array = new List<TableSignedIdentifier>();
                            foreach (var e in signedIdentifiersElement.Elements("SignedIdentifier"))
                            {
                                array.Add(TableSignedIdentifier.DeserializeTableSignedIdentifier(e));
                            }
                            value = array;
                        }
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Retrieves details about any stored access policies specified on the table that may be used with Shared Access Signatures. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/> is null. </exception>
        public ResponseWithHeaders<IReadOnlyList<TableSignedIdentifier>, TableGetAccessPolicyHeaders> GetAccessPolicy(string table, int? timeout = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }

            using var message = CreateGetAccessPolicyRequest(table, timeout);
            _pipeline.Send(message, cancellationToken);
            var headers = new TableGetAccessPolicyHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        IReadOnlyList<TableSignedIdentifier> value = default;
                        var document = XDocument.Load(message.Response.ContentStream, LoadOptions.PreserveWhitespace);
                        if (document.Element("SignedIdentifiers") is XElement signedIdentifiersElement)
                        {
                            var array = new List<TableSignedIdentifier>();
                            foreach (var e in signedIdentifiersElement.Elements("SignedIdentifier"))
                            {
                                array.Add(TableSignedIdentifier.DeserializeTableSignedIdentifier(e));
                            }
                            value = array;
                        }
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateSetAccessPolicyRequest(string table, int? timeout, IEnumerable<TableSignedIdentifier> tableAcl)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendPath(table, true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            uri.AppendQuery("comp", "acl", true);
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("Accept", "application/xml");
            if (tableAcl != null)
            {
                request.Headers.Add("Content-Type", "application/xml");
                var content = new XmlWriterContent();
                content.XmlWriter.WriteStartElement("SignedIdentifiers");
                foreach (var item in tableAcl)
                {
                    content.XmlWriter.WriteObjectValue(item, "SignedIdentifier");
                }
                content.XmlWriter.WriteEndElement();
                request.Content = content;
            }
            return message;
        }

        /// <summary> Sets stored access policies for the table that may be used with Shared Access Signatures. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="tableAcl"> The acls for the table. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/> is null. </exception>
        public async Task<ResponseWithHeaders<TableSetAccessPolicyHeaders>> SetAccessPolicyAsync(string table, int? timeout = null, IEnumerable<TableSignedIdentifier> tableAcl = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }

            using var message = CreateSetAccessPolicyRequest(table, timeout, tableAcl);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new TableSetAccessPolicyHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Sets stored access policies for the table that may be used with Shared Access Signatures. </summary>
        /// <param name="table"> The name of the table. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. </param>
        /// <param name="tableAcl"> The acls for the table. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="table"/> is null. </exception>
        public ResponseWithHeaders<TableSetAccessPolicyHeaders> SetAccessPolicy(string table, int? timeout = null, IEnumerable<TableSignedIdentifier> tableAcl = null, CancellationToken cancellationToken = default)
        {
            if (table == null)
            {
                throw new ArgumentNullException(nameof(table));
            }

            using var message = CreateSetAccessPolicyRequest(table, timeout, tableAcl);
            _pipeline.Send(message, cancellationToken);
            var headers = new TableSetAccessPolicyHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        private static ResponseClassifier _responseClassifier201204;
        private static ResponseClassifier ResponseClassifier201204 => _responseClassifier201204 ??= new StatusCodeClassifier(stackalloc ushort[] { 201, 204 });
        private static ResponseClassifier _responseClassifier204;
        private static ResponseClassifier ResponseClassifier204 => _responseClassifier204 ??= new StatusCodeClassifier(stackalloc ushort[] { 204 });
        private static ResponseClassifier _responseClassifier200;
        private static ResponseClassifier ResponseClassifier200 => _responseClassifier200 ??= new StatusCodeClassifier(stackalloc ushort[] { 200 });
    }
}
