# Add and Delete Entities

This sample demonstrates how to add and delete entities. You will need to have previously [created a table](https://github.com/Azure/azure-sdk-for-net/blob/main/sdk/tables/Azure.Data.Tables/samples/Sample1CreateDeleteTables.md) in the service in order to query entities from it.To get started, you'll need access to either a Storage or Cosmos DB account.

## Create a `TableClient`

A `TableClient` is needed to perform table-level operations like inserting and deleting entities within the table, so it is ideal for dealing with only a specific table. There are two ways to get a `TableClient`:
- Call `GetTableClient` from the `TableServiceClient` with the table name.

```C# Snippet:TablesSample1GetTableClient
var tableClient2 = serviceClient.GetTableClient(tableName);
```

- Create a `TableClient` with a SAS URI, an endpoint and `TableSharedKeyCredential`, or a connection string.

```C# Snippet:TablesSample1CreateTableClient
var tableClient3 = new TableClient(
    new Uri(storageUri),
    tableName,
    new TableSharedKeyCredential(accountName, storageAccountKey));
```

If you are not familiar with creating tables, refer to the sample on [creating and deleting tables](https://github.com/Azure/azure-sdk-for-net/blob/main/sdk/tables/Azure.Data.Tables/samples/Sample1CreateDeleteTables.md).

## Create a local entity

An entity has a set of properties, and each property is a name, typed-value pair. Entities can be strongly-typed or in dictionary form using the `TableEntity` class.

### Strongly-typed entity

Define a strongly-typed entity class that extends from the `ITableEntity` interface, which imposes the following required properties: partition key, row key, ETag, and Timestamp. The last two properties are generated by the service and will not be set by the user.

```C# Snippet:TablesSample2DefineStronglyTypedEntity
// Define a strongly typed entity by implementing the ITableEntity interface.
public class OfficeSupplyEntity : ITableEntity
{
    public string Product { get; set; }
    public double Price { get; set; }
    public int Quantity { get; set; }
    public string PartitionKey { get; set; }
    public string RowKey { get; set; }
    public DateTimeOffset? Timestamp { get; set; }
    public ETag ETag { get; set; }
}
```

Once defined, create an entity with the class.

```C# Snippet:TablesSample2CreateStronglyTypedEntity
// Create an instance of the strongly-typed entity and set their properties.
var strongEntity = new OfficeSupplyEntity
{
    PartitionKey = partitionKey,
    RowKey = rowKeyStrong,
    Product = "Notebook",
    Price = 3.00,
    Quantity = 50
};

Console.WriteLine($"{tableEntity.RowKey}: {strongEntity.Product} costs ${strongEntity.Price}.");
```

### Dictionary entity

Create an entity with the `TableEntity` class. A partition key and row key is required to identify the entity, and additional properties can be added like a Dictionary.

Properties accessed using the indexer `[]` will be typed as an `object`, but `TableEntity` offers typed getters as shown below. Required properties can also be accessed as class properties.

```C# Snippet:TablesSample2CreateDictionaryEntity
// Make a dictionary entity by defining a <see cref="TableEntity">.
var tableEntity = new TableEntity(partitionKey, rowKey)
{
    { "Product", "Marker Set" },
    { "Price", 5.00 },
    { "Quantity", 21 }
};

Console.WriteLine($"{tableEntity.RowKey}: {tableEntity["Product"]} costs ${tableEntity.GetDouble("Price")}.");
```

## Add an entity

To add the entity to the table, invoke `AddEntity` and pass in the newly created entity.

```C# Snippet:TablesSample2AddEntity
// Add the newly created entity.
tableClient.AddEntity(tableEntity);
```

## Delete an entity

To delete an entity, invoke `DeleteEntity` and pass in its partition and row key.

```C# Snippet:TablesSample2DeleteEntity
// Delete the entity given the partition and row key.
tableClient.DeleteEntity(partitionKey, rowKey);
```

Alternatively, you can pass a `TableEntity`object.

```C# Snippet:TablesSample2DeleteEntityUsingObject
// Delete an entity given a TableEntity object
tableClient.DeleteEntity(tableEntity);
```
