// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

// <auto-generated/>

#nullable disable

using System;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using System.Threading.Tasks;
using System.Xml.Linq;
using Azure.Core;
using Azure.Core.Pipeline;
using Azure.Storage.Files.Shares.Models;

namespace Azure.Storage.Files.Shares
{
    internal partial class FileRestClient
    {
        private readonly HttpPipeline _pipeline;
        private readonly string _url;
        private readonly string _version;
        private readonly string _fileRangeWriteFromUrl;
        private readonly bool? _allowTrailingDot;
        private readonly ShareTokenIntent? _fileRequestIntent;
        private readonly bool? _allowSourceTrailingDot;

        /// <summary> The ClientDiagnostics is used to provide tracing support for the client library. </summary>
        internal ClientDiagnostics ClientDiagnostics { get; }

        /// <summary> Initializes a new instance of FileRestClient. </summary>
        /// <param name="clientDiagnostics"> The handler for diagnostic messaging in the client. </param>
        /// <param name="pipeline"> The HTTP pipeline for sending and receiving REST requests and responses. </param>
        /// <param name="url"> The URL of the service account, share, directory or file that is the target of the desired operation. </param>
        /// <param name="version"> Specifies the version of the operation to use for this request. The default value is "2025-01-05". </param>
        /// <param name="fileRangeWriteFromUrl"> Only update is supported: - Update: Writes the bytes downloaded from the source url into the specified range. The default value is "update". </param>
        /// <param name="allowTrailingDot"> If true, the trailing dot will not be trimmed from the target URI. </param>
        /// <param name="fileRequestIntent"> Valid value is backup. </param>
        /// <param name="allowSourceTrailingDot"> If true, the trailing dot will not be trimmed from the source URI. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="clientDiagnostics"/>, <paramref name="pipeline"/>, <paramref name="url"/>, <paramref name="version"/> or <paramref name="fileRangeWriteFromUrl"/> is null. </exception>
        public FileRestClient(ClientDiagnostics clientDiagnostics, HttpPipeline pipeline, string url, string version, string fileRangeWriteFromUrl, bool? allowTrailingDot = null, ShareTokenIntent? fileRequestIntent = null, bool? allowSourceTrailingDot = null)
        {
            ClientDiagnostics = clientDiagnostics ?? throw new ArgumentNullException(nameof(clientDiagnostics));
            _pipeline = pipeline ?? throw new ArgumentNullException(nameof(pipeline));
            _url = url ?? throw new ArgumentNullException(nameof(url));
            _version = version ?? throw new ArgumentNullException(nameof(version));
            _fileRangeWriteFromUrl = fileRangeWriteFromUrl ?? throw new ArgumentNullException(nameof(fileRangeWriteFromUrl));
            _allowTrailingDot = allowTrailingDot;
            _fileRequestIntent = fileRequestIntent;
            _allowSourceTrailingDot = allowSourceTrailingDot;
        }

        internal HttpMessage CreateCreateRequest(long fileContentLength, string fileAttributes, int? timeout, IDictionary<string, string> metadata, string filePermission, FilePermissionFormat? filePermissionFormat, string filePermissionKey, string fileCreationTime, string fileLastWriteTime, string fileChangeTime, FileHttpHeaders fileHttpHeaders, ShareFileRequestConditions shareFileRequestConditions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("x-ms-content-length", fileContentLength);
            request.Headers.Add("x-ms-type", "file");
            if (fileHttpHeaders?.FileContentType != null)
            {
                request.Headers.Add("x-ms-content-type", fileHttpHeaders.FileContentType);
            }
            if (fileHttpHeaders?.FileContentEncoding != null)
            {
                request.Headers.Add("x-ms-content-encoding", fileHttpHeaders.FileContentEncoding);
            }
            if (fileHttpHeaders?.FileContentLanguage != null)
            {
                request.Headers.Add("x-ms-content-language", fileHttpHeaders.FileContentLanguage);
            }
            if (fileHttpHeaders?.FileCacheControl != null)
            {
                request.Headers.Add("x-ms-cache-control", fileHttpHeaders.FileCacheControl);
            }
            if (fileHttpHeaders?.FileContentMD5 != null)
            {
                request.Headers.Add("x-ms-content-md5", fileHttpHeaders.FileContentMD5, "D");
            }
            if (fileHttpHeaders?.FileContentDisposition != null)
            {
                request.Headers.Add("x-ms-content-disposition", fileHttpHeaders.FileContentDisposition);
            }
            if (metadata != null)
            {
                request.Headers.Add("x-ms-meta-", metadata);
            }
            if (filePermission != null)
            {
                request.Headers.Add("x-ms-file-permission", filePermission);
            }
            if (filePermissionFormat != null)
            {
                request.Headers.Add("x-ms-file-permission-format", filePermissionFormat.Value.ToSerialString());
            }
            if (filePermissionKey != null)
            {
                request.Headers.Add("x-ms-file-permission-key", filePermissionKey);
            }
            request.Headers.Add("x-ms-file-attributes", fileAttributes);
            if (fileCreationTime != null)
            {
                request.Headers.Add("x-ms-file-creation-time", fileCreationTime);
            }
            if (fileLastWriteTime != null)
            {
                request.Headers.Add("x-ms-file-last-write-time", fileLastWriteTime);
            }
            if (fileChangeTime != null)
            {
                request.Headers.Add("x-ms-file-change-time", fileChangeTime);
            }
            if (shareFileRequestConditions?.LeaseId != null)
            {
                request.Headers.Add("x-ms-lease-id", shareFileRequestConditions.LeaseId);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Creates a new file or replaces a file. Note it only initializes the file with no content. </summary>
        /// <param name="fileContentLength"> Specifies the maximum size for the file, up to 4 TB. </param>
        /// <param name="fileAttributes"> If specified, the provided file attributes shall be set. Default value: ‘Archive’ for file and ‘Directory’ for directory. ‘None’ can also be specified as default. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="metadata"> A name-value pair to associate with a file storage object. </param>
        /// <param name="filePermission"> If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is &lt;= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="filePermissionFormat"> Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission. </param>
        /// <param name="filePermissionKey"> Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="fileCreationTime"> Creation time for the file/directory. Default value: Now. </param>
        /// <param name="fileLastWriteTime"> Last write time for the file/directory. Default value: Now. </param>
        /// <param name="fileChangeTime"> Change time for the file/directory. Default value: Now. </param>
        /// <param name="fileHttpHeaders"> Parameter group. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="fileAttributes"/> is null. </exception>
        public async Task<ResponseWithHeaders<FileCreateHeaders>> CreateAsync(long fileContentLength, string fileAttributes, int? timeout = null, IDictionary<string, string> metadata = null, string filePermission = null, FilePermissionFormat? filePermissionFormat = null, string filePermissionKey = null, string fileCreationTime = null, string fileLastWriteTime = null, string fileChangeTime = null, FileHttpHeaders fileHttpHeaders = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            if (fileAttributes == null)
            {
                throw new ArgumentNullException(nameof(fileAttributes));
            }

            using var message = CreateCreateRequest(fileContentLength, fileAttributes, timeout, metadata, filePermission, filePermissionFormat, filePermissionKey, fileCreationTime, fileLastWriteTime, fileChangeTime, fileHttpHeaders, shareFileRequestConditions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileCreateHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Creates a new file or replaces a file. Note it only initializes the file with no content. </summary>
        /// <param name="fileContentLength"> Specifies the maximum size for the file, up to 4 TB. </param>
        /// <param name="fileAttributes"> If specified, the provided file attributes shall be set. Default value: ‘Archive’ for file and ‘Directory’ for directory. ‘None’ can also be specified as default. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="metadata"> A name-value pair to associate with a file storage object. </param>
        /// <param name="filePermission"> If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is &lt;= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="filePermissionFormat"> Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission. </param>
        /// <param name="filePermissionKey"> Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="fileCreationTime"> Creation time for the file/directory. Default value: Now. </param>
        /// <param name="fileLastWriteTime"> Last write time for the file/directory. Default value: Now. </param>
        /// <param name="fileChangeTime"> Change time for the file/directory. Default value: Now. </param>
        /// <param name="fileHttpHeaders"> Parameter group. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="fileAttributes"/> is null. </exception>
        public ResponseWithHeaders<FileCreateHeaders> Create(long fileContentLength, string fileAttributes, int? timeout = null, IDictionary<string, string> metadata = null, string filePermission = null, FilePermissionFormat? filePermissionFormat = null, string filePermissionKey = null, string fileCreationTime = null, string fileLastWriteTime = null, string fileChangeTime = null, FileHttpHeaders fileHttpHeaders = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            if (fileAttributes == null)
            {
                throw new ArgumentNullException(nameof(fileAttributes));
            }

            using var message = CreateCreateRequest(fileContentLength, fileAttributes, timeout, metadata, filePermission, filePermissionFormat, filePermissionKey, fileCreationTime, fileLastWriteTime, fileChangeTime, fileHttpHeaders, shareFileRequestConditions);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileCreateHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateDownloadRequest(int? timeout, string range, bool? rangeGetContentMD5, string structuredBodyType, ShareFileRequestConditions shareFileRequestConditions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            message.BufferResponse = false;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            request.Headers.Add("x-ms-version", _version);
            if (range != null)
            {
                request.Headers.Add("x-ms-range", range);
            }
            if (rangeGetContentMD5 != null)
            {
                request.Headers.Add("x-ms-range-get-content-md5", rangeGetContentMD5.Value);
            }
            if (structuredBodyType != null)
            {
                request.Headers.Add("x-ms-structured-body", structuredBodyType);
            }
            if (shareFileRequestConditions?.LeaseId != null)
            {
                request.Headers.Add("x-ms-lease-id", shareFileRequestConditions.LeaseId);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Reads or downloads a file from the system, including its metadata and properties. </summary>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="range"> Return file data only from the specified byte range. </param>
        /// <param name="rangeGetContentMD5"> When this header is set to true and specified together with the Range header, the service returns the MD5 hash for the range, as long as the range is less than or equal to 4 MB in size. </param>
        /// <param name="structuredBodyType"> Specifies the response content should be returned as a structured message and specifies the message schema version and properties. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<ResponseWithHeaders<Stream, FileDownloadHeaders>> DownloadAsync(int? timeout = null, string range = null, bool? rangeGetContentMD5 = null, string structuredBodyType = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateDownloadRequest(timeout, range, rangeGetContentMD5, structuredBodyType, shareFileRequestConditions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileDownloadHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                case 206:
                    {
                        var value = message.ExtractResponseContent();
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Reads or downloads a file from the system, including its metadata and properties. </summary>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="range"> Return file data only from the specified byte range. </param>
        /// <param name="rangeGetContentMD5"> When this header is set to true and specified together with the Range header, the service returns the MD5 hash for the range, as long as the range is less than or equal to 4 MB in size. </param>
        /// <param name="structuredBodyType"> Specifies the response content should be returned as a structured message and specifies the message schema version and properties. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public ResponseWithHeaders<Stream, FileDownloadHeaders> Download(int? timeout = null, string range = null, bool? rangeGetContentMD5 = null, string structuredBodyType = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateDownloadRequest(timeout, range, rangeGetContentMD5, structuredBodyType, shareFileRequestConditions);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileDownloadHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                case 206:
                    {
                        var value = message.ExtractResponseContent();
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetPropertiesRequest(string sharesnapshot, int? timeout, ShareFileRequestConditions shareFileRequestConditions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Head;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            if (sharesnapshot != null)
            {
                uri.AppendQuery("sharesnapshot", sharesnapshot, true);
            }
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            request.Headers.Add("x-ms-version", _version);
            if (shareFileRequestConditions?.LeaseId != null)
            {
                request.Headers.Add("x-ms-lease-id", shareFileRequestConditions.LeaseId);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Returns all user-defined metadata, standard HTTP properties, and system properties for the file. It does not return the content of the file. </summary>
        /// <param name="sharesnapshot"> The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<ResponseWithHeaders<FileGetPropertiesHeaders>> GetPropertiesAsync(string sharesnapshot = null, int? timeout = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetPropertiesRequest(sharesnapshot, timeout, shareFileRequestConditions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileGetPropertiesHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Returns all user-defined metadata, standard HTTP properties, and system properties for the file. It does not return the content of the file. </summary>
        /// <param name="sharesnapshot"> The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public ResponseWithHeaders<FileGetPropertiesHeaders> GetProperties(string sharesnapshot = null, int? timeout = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetPropertiesRequest(sharesnapshot, timeout, shareFileRequestConditions);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileGetPropertiesHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateDeleteRequest(int? timeout, ShareFileRequestConditions shareFileRequestConditions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            request.Headers.Add("x-ms-version", _version);
            if (shareFileRequestConditions?.LeaseId != null)
            {
                request.Headers.Add("x-ms-lease-id", shareFileRequestConditions.LeaseId);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> removes the file from the storage account. </summary>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<ResponseWithHeaders<FileDeleteHeaders>> DeleteAsync(int? timeout = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateDeleteRequest(timeout, shareFileRequestConditions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileDeleteHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 202:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> removes the file from the storage account. </summary>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public ResponseWithHeaders<FileDeleteHeaders> Delete(int? timeout = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateDeleteRequest(timeout, shareFileRequestConditions);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileDeleteHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 202:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateSetHttpHeadersRequest(string fileAttributes, int? timeout, long? fileContentLength, string filePermission, FilePermissionFormat? filePermissionFormat, string filePermissionKey, string fileCreationTime, string fileLastWriteTime, string fileChangeTime, FileHttpHeaders fileHttpHeaders, ShareFileRequestConditions shareFileRequestConditions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "properties", true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            if (fileContentLength != null)
            {
                request.Headers.Add("x-ms-content-length", fileContentLength.Value);
            }
            if (fileHttpHeaders?.FileContentType != null)
            {
                request.Headers.Add("x-ms-content-type", fileHttpHeaders.FileContentType);
            }
            if (fileHttpHeaders?.FileContentEncoding != null)
            {
                request.Headers.Add("x-ms-content-encoding", fileHttpHeaders.FileContentEncoding);
            }
            if (fileHttpHeaders?.FileContentLanguage != null)
            {
                request.Headers.Add("x-ms-content-language", fileHttpHeaders.FileContentLanguage);
            }
            if (fileHttpHeaders?.FileCacheControl != null)
            {
                request.Headers.Add("x-ms-cache-control", fileHttpHeaders.FileCacheControl);
            }
            if (fileHttpHeaders?.FileContentMD5 != null)
            {
                request.Headers.Add("x-ms-content-md5", fileHttpHeaders.FileContentMD5, "D");
            }
            if (fileHttpHeaders?.FileContentDisposition != null)
            {
                request.Headers.Add("x-ms-content-disposition", fileHttpHeaders.FileContentDisposition);
            }
            if (filePermission != null)
            {
                request.Headers.Add("x-ms-file-permission", filePermission);
            }
            if (filePermissionFormat != null)
            {
                request.Headers.Add("x-ms-file-permission-format", filePermissionFormat.Value.ToSerialString());
            }
            if (filePermissionKey != null)
            {
                request.Headers.Add("x-ms-file-permission-key", filePermissionKey);
            }
            request.Headers.Add("x-ms-file-attributes", fileAttributes);
            if (fileCreationTime != null)
            {
                request.Headers.Add("x-ms-file-creation-time", fileCreationTime);
            }
            if (fileLastWriteTime != null)
            {
                request.Headers.Add("x-ms-file-last-write-time", fileLastWriteTime);
            }
            if (fileChangeTime != null)
            {
                request.Headers.Add("x-ms-file-change-time", fileChangeTime);
            }
            if (shareFileRequestConditions?.LeaseId != null)
            {
                request.Headers.Add("x-ms-lease-id", shareFileRequestConditions.LeaseId);
            }
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Sets HTTP headers on the file. </summary>
        /// <param name="fileAttributes"> If specified, the provided file attributes shall be set. Default value: ‘Archive’ for file and ‘Directory’ for directory. ‘None’ can also be specified as default. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="fileContentLength"> Resizes a file to the specified size. If the specified byte value is less than the current size of the file, then all ranges above the specified byte value are cleared. </param>
        /// <param name="filePermission"> If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is &lt;= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="filePermissionFormat"> Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission. </param>
        /// <param name="filePermissionKey"> Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="fileCreationTime"> Creation time for the file/directory. Default value: Now. </param>
        /// <param name="fileLastWriteTime"> Last write time for the file/directory. Default value: Now. </param>
        /// <param name="fileChangeTime"> Change time for the file/directory. Default value: Now. </param>
        /// <param name="fileHttpHeaders"> Parameter group. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="fileAttributes"/> is null. </exception>
        public async Task<ResponseWithHeaders<FileSetHttpHeadersHeaders>> SetHttpHeadersAsync(string fileAttributes, int? timeout = null, long? fileContentLength = null, string filePermission = null, FilePermissionFormat? filePermissionFormat = null, string filePermissionKey = null, string fileCreationTime = null, string fileLastWriteTime = null, string fileChangeTime = null, FileHttpHeaders fileHttpHeaders = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            if (fileAttributes == null)
            {
                throw new ArgumentNullException(nameof(fileAttributes));
            }

            using var message = CreateSetHttpHeadersRequest(fileAttributes, timeout, fileContentLength, filePermission, filePermissionFormat, filePermissionKey, fileCreationTime, fileLastWriteTime, fileChangeTime, fileHttpHeaders, shareFileRequestConditions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileSetHttpHeadersHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Sets HTTP headers on the file. </summary>
        /// <param name="fileAttributes"> If specified, the provided file attributes shall be set. Default value: ‘Archive’ for file and ‘Directory’ for directory. ‘None’ can also be specified as default. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="fileContentLength"> Resizes a file to the specified size. If the specified byte value is less than the current size of the file, then all ranges above the specified byte value are cleared. </param>
        /// <param name="filePermission"> If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is &lt;= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="filePermissionFormat"> Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission. </param>
        /// <param name="filePermissionKey"> Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="fileCreationTime"> Creation time for the file/directory. Default value: Now. </param>
        /// <param name="fileLastWriteTime"> Last write time for the file/directory. Default value: Now. </param>
        /// <param name="fileChangeTime"> Change time for the file/directory. Default value: Now. </param>
        /// <param name="fileHttpHeaders"> Parameter group. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="fileAttributes"/> is null. </exception>
        public ResponseWithHeaders<FileSetHttpHeadersHeaders> SetHttpHeaders(string fileAttributes, int? timeout = null, long? fileContentLength = null, string filePermission = null, FilePermissionFormat? filePermissionFormat = null, string filePermissionKey = null, string fileCreationTime = null, string fileLastWriteTime = null, string fileChangeTime = null, FileHttpHeaders fileHttpHeaders = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            if (fileAttributes == null)
            {
                throw new ArgumentNullException(nameof(fileAttributes));
            }

            using var message = CreateSetHttpHeadersRequest(fileAttributes, timeout, fileContentLength, filePermission, filePermissionFormat, filePermissionKey, fileCreationTime, fileLastWriteTime, fileChangeTime, fileHttpHeaders, shareFileRequestConditions);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileSetHttpHeadersHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateSetMetadataRequest(int? timeout, IDictionary<string, string> metadata, ShareFileRequestConditions shareFileRequestConditions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "metadata", true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            if (metadata != null)
            {
                request.Headers.Add("x-ms-meta-", metadata);
            }
            request.Headers.Add("x-ms-version", _version);
            if (shareFileRequestConditions?.LeaseId != null)
            {
                request.Headers.Add("x-ms-lease-id", shareFileRequestConditions.LeaseId);
            }
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Updates user-defined metadata for the specified file. </summary>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="metadata"> A name-value pair to associate with a file storage object. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<ResponseWithHeaders<FileSetMetadataHeaders>> SetMetadataAsync(int? timeout = null, IDictionary<string, string> metadata = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateSetMetadataRequest(timeout, metadata, shareFileRequestConditions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileSetMetadataHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Updates user-defined metadata for the specified file. </summary>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="metadata"> A name-value pair to associate with a file storage object. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public ResponseWithHeaders<FileSetMetadataHeaders> SetMetadata(int? timeout = null, IDictionary<string, string> metadata = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateSetMetadataRequest(timeout, metadata, shareFileRequestConditions);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileSetMetadataHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateAcquireLeaseRequest(int? timeout, int? duration, string proposedLeaseId)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "lease", true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-lease-action", "acquire");
            if (duration != null)
            {
                request.Headers.Add("x-ms-lease-duration", duration.Value);
            }
            if (proposedLeaseId != null)
            {
                request.Headers.Add("x-ms-proposed-lease-id", proposedLeaseId);
            }
            request.Headers.Add("x-ms-version", _version);
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> [Update] The Lease File operation establishes and manages a lock on a file for write and delete operations. </summary>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="duration"> Specifies the duration of the lease, in seconds, or negative one (-1) for a lease that never expires. A non-infinite lease can be between 15 and 60 seconds. A lease duration cannot be changed using renew or change. </param>
        /// <param name="proposedLeaseId"> Proposed lease ID, in a GUID string format. The File service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<ResponseWithHeaders<FileAcquireLeaseHeaders>> AcquireLeaseAsync(int? timeout = null, int? duration = null, string proposedLeaseId = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateAcquireLeaseRequest(timeout, duration, proposedLeaseId);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileAcquireLeaseHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> [Update] The Lease File operation establishes and manages a lock on a file for write and delete operations. </summary>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="duration"> Specifies the duration of the lease, in seconds, or negative one (-1) for a lease that never expires. A non-infinite lease can be between 15 and 60 seconds. A lease duration cannot be changed using renew or change. </param>
        /// <param name="proposedLeaseId"> Proposed lease ID, in a GUID string format. The File service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public ResponseWithHeaders<FileAcquireLeaseHeaders> AcquireLease(int? timeout = null, int? duration = null, string proposedLeaseId = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateAcquireLeaseRequest(timeout, duration, proposedLeaseId);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileAcquireLeaseHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateReleaseLeaseRequest(string leaseId, int? timeout)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "lease", true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-lease-action", "release");
            request.Headers.Add("x-ms-lease-id", leaseId);
            request.Headers.Add("x-ms-version", _version);
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> [Update] The Lease File operation establishes and manages a lock on a file for write and delete operations. </summary>
        /// <param name="leaseId"> Specifies the current lease ID on the resource. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="leaseId"/> is null. </exception>
        public async Task<ResponseWithHeaders<FileReleaseLeaseHeaders>> ReleaseLeaseAsync(string leaseId, int? timeout = null, CancellationToken cancellationToken = default)
        {
            if (leaseId == null)
            {
                throw new ArgumentNullException(nameof(leaseId));
            }

            using var message = CreateReleaseLeaseRequest(leaseId, timeout);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileReleaseLeaseHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> [Update] The Lease File operation establishes and manages a lock on a file for write and delete operations. </summary>
        /// <param name="leaseId"> Specifies the current lease ID on the resource. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="leaseId"/> is null. </exception>
        public ResponseWithHeaders<FileReleaseLeaseHeaders> ReleaseLease(string leaseId, int? timeout = null, CancellationToken cancellationToken = default)
        {
            if (leaseId == null)
            {
                throw new ArgumentNullException(nameof(leaseId));
            }

            using var message = CreateReleaseLeaseRequest(leaseId, timeout);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileReleaseLeaseHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateChangeLeaseRequest(string leaseId, int? timeout, string proposedLeaseId)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "lease", true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-lease-action", "change");
            request.Headers.Add("x-ms-lease-id", leaseId);
            if (proposedLeaseId != null)
            {
                request.Headers.Add("x-ms-proposed-lease-id", proposedLeaseId);
            }
            request.Headers.Add("x-ms-version", _version);
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> [Update] The Lease File operation establishes and manages a lock on a file for write and delete operations. </summary>
        /// <param name="leaseId"> Specifies the current lease ID on the resource. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="proposedLeaseId"> Proposed lease ID, in a GUID string format. The File service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="leaseId"/> is null. </exception>
        public async Task<ResponseWithHeaders<FileChangeLeaseHeaders>> ChangeLeaseAsync(string leaseId, int? timeout = null, string proposedLeaseId = null, CancellationToken cancellationToken = default)
        {
            if (leaseId == null)
            {
                throw new ArgumentNullException(nameof(leaseId));
            }

            using var message = CreateChangeLeaseRequest(leaseId, timeout, proposedLeaseId);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileChangeLeaseHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> [Update] The Lease File operation establishes and manages a lock on a file for write and delete operations. </summary>
        /// <param name="leaseId"> Specifies the current lease ID on the resource. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="proposedLeaseId"> Proposed lease ID, in a GUID string format. The File service returns 400 (Invalid request) if the proposed lease ID is not in the correct format. See Guid Constructor (String) for a list of valid GUID string formats. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="leaseId"/> is null. </exception>
        public ResponseWithHeaders<FileChangeLeaseHeaders> ChangeLease(string leaseId, int? timeout = null, string proposedLeaseId = null, CancellationToken cancellationToken = default)
        {
            if (leaseId == null)
            {
                throw new ArgumentNullException(nameof(leaseId));
            }

            using var message = CreateChangeLeaseRequest(leaseId, timeout, proposedLeaseId);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileChangeLeaseHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateBreakLeaseRequest(int? timeout, ShareFileRequestConditions shareFileRequestConditions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "lease", true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-lease-action", "break");
            if (shareFileRequestConditions?.LeaseId != null)
            {
                request.Headers.Add("x-ms-lease-id", shareFileRequestConditions.LeaseId);
            }
            request.Headers.Add("x-ms-version", _version);
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> [Update] The Lease File operation establishes and manages a lock on a file for write and delete operations. </summary>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<ResponseWithHeaders<FileBreakLeaseHeaders>> BreakLeaseAsync(int? timeout = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateBreakLeaseRequest(timeout, shareFileRequestConditions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileBreakLeaseHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 202:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> [Update] The Lease File operation establishes and manages a lock on a file for write and delete operations. </summary>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public ResponseWithHeaders<FileBreakLeaseHeaders> BreakLease(int? timeout = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateBreakLeaseRequest(timeout, shareFileRequestConditions);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileBreakLeaseHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 202:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateUploadRangeRequest(string range, ShareFileRangeWriteType fileRangeWrite, long contentLength, int? timeout, byte[] contentMD5, FileLastWrittenMode? fileLastWrittenMode, string structuredBodyType, long? structuredContentLength, Stream optionalbody, ShareFileRequestConditions shareFileRequestConditions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "range", true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-range", range);
            request.Headers.Add("x-ms-write", fileRangeWrite.ToSerialString());
            request.Headers.Add("x-ms-version", _version);
            if (shareFileRequestConditions?.LeaseId != null)
            {
                request.Headers.Add("x-ms-lease-id", shareFileRequestConditions.LeaseId);
            }
            if (fileLastWrittenMode != null)
            {
                request.Headers.Add("x-ms-file-last-write-time", fileLastWrittenMode.Value.ToSerialString());
            }
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            if (structuredBodyType != null)
            {
                request.Headers.Add("x-ms-structured-body", structuredBodyType);
            }
            if (structuredContentLength != null)
            {
                request.Headers.Add("x-ms-structured-content-length", structuredContentLength.Value);
            }
            request.Headers.Add("Accept", "application/xml");
            if (optionalbody != null)
            {
                request.Headers.Add("Content-Length", contentLength);
                if (contentMD5 != null)
                {
                    request.Headers.Add("Content-MD5", contentMD5, "D");
                }
                request.Headers.Add("Content-Type", "application/octet-stream");
                request.Content = RequestContent.Create(optionalbody);
            }
            return message;
        }

        /// <summary> Upload a range of bytes to a file. </summary>
        /// <param name="range"> Specifies the range of bytes to be written. Both the start and end of the range must be specified. For an update operation, the range can be up to 4 MB in size. For a clear operation, the range can be up to the value of the file's full size. The File service accepts only a single byte range for the Range and 'x-ms-range' headers, and the byte range must be specified in the following format: bytes=startByte-endByte. </param>
        /// <param name="fileRangeWrite"> Specify one of the following options: - Update: Writes the bytes specified by the request body into the specified range. The Range and Content-Length headers must match to perform the update. - Clear: Clears the specified range and releases the space used in storage for that range. To clear a range, set the Content-Length header to zero, and set the Range header to a value that indicates the range to clear, up to maximum file size. </param>
        /// <param name="contentLength"> Specifies the number of bytes being transmitted in the request body. When the x-ms-write header is set to clear, the value of this header must be set to zero. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="contentMD5"> An MD5 hash of the content. This hash is used to verify the integrity of the data during transport. When the Content-MD5 header is specified, the File service compares the hash of the content that has arrived with the header value that was sent. If the two hashes do not match, the operation will fail with error code 400 (Bad Request). </param>
        /// <param name="fileLastWrittenMode"> If the file last write time should be preserved or overwritten. </param>
        /// <param name="structuredBodyType"> Required if the request body is a structured message. Specifies the message schema version and properties. </param>
        /// <param name="structuredContentLength"> Required if the request body is a structured message. Specifies the length of the blob/file content inside the message body. Will always be smaller than Content-Length. </param>
        /// <param name="optionalbody"> Initial data. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="range"/> is null. </exception>
        public async Task<ResponseWithHeaders<FileUploadRangeHeaders>> UploadRangeAsync(string range, ShareFileRangeWriteType fileRangeWrite, long contentLength, int? timeout = null, byte[] contentMD5 = null, FileLastWrittenMode? fileLastWrittenMode = null, string structuredBodyType = null, long? structuredContentLength = null, Stream optionalbody = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            if (range == null)
            {
                throw new ArgumentNullException(nameof(range));
            }

            using var message = CreateUploadRangeRequest(range, fileRangeWrite, contentLength, timeout, contentMD5, fileLastWrittenMode, structuredBodyType, structuredContentLength, optionalbody, shareFileRequestConditions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileUploadRangeHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Upload a range of bytes to a file. </summary>
        /// <param name="range"> Specifies the range of bytes to be written. Both the start and end of the range must be specified. For an update operation, the range can be up to 4 MB in size. For a clear operation, the range can be up to the value of the file's full size. The File service accepts only a single byte range for the Range and 'x-ms-range' headers, and the byte range must be specified in the following format: bytes=startByte-endByte. </param>
        /// <param name="fileRangeWrite"> Specify one of the following options: - Update: Writes the bytes specified by the request body into the specified range. The Range and Content-Length headers must match to perform the update. - Clear: Clears the specified range and releases the space used in storage for that range. To clear a range, set the Content-Length header to zero, and set the Range header to a value that indicates the range to clear, up to maximum file size. </param>
        /// <param name="contentLength"> Specifies the number of bytes being transmitted in the request body. When the x-ms-write header is set to clear, the value of this header must be set to zero. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="contentMD5"> An MD5 hash of the content. This hash is used to verify the integrity of the data during transport. When the Content-MD5 header is specified, the File service compares the hash of the content that has arrived with the header value that was sent. If the two hashes do not match, the operation will fail with error code 400 (Bad Request). </param>
        /// <param name="fileLastWrittenMode"> If the file last write time should be preserved or overwritten. </param>
        /// <param name="structuredBodyType"> Required if the request body is a structured message. Specifies the message schema version and properties. </param>
        /// <param name="structuredContentLength"> Required if the request body is a structured message. Specifies the length of the blob/file content inside the message body. Will always be smaller than Content-Length. </param>
        /// <param name="optionalbody"> Initial data. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="range"/> is null. </exception>
        public ResponseWithHeaders<FileUploadRangeHeaders> UploadRange(string range, ShareFileRangeWriteType fileRangeWrite, long contentLength, int? timeout = null, byte[] contentMD5 = null, FileLastWrittenMode? fileLastWrittenMode = null, string structuredBodyType = null, long? structuredContentLength = null, Stream optionalbody = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            if (range == null)
            {
                throw new ArgumentNullException(nameof(range));
            }

            using var message = CreateUploadRangeRequest(range, fileRangeWrite, contentLength, timeout, contentMD5, fileLastWrittenMode, structuredBodyType, structuredContentLength, optionalbody, shareFileRequestConditions);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileUploadRangeHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateUploadRangeFromURLRequest(string range, string copySource, long contentLength, int? timeout, string sourceRange, byte[] sourceContentCrc64, string copySourceAuthorization, FileLastWrittenMode? fileLastWrittenMode, SourceModifiedAccessConditions sourceModifiedAccessConditions, ShareFileRequestConditions shareFileRequestConditions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "range", true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-range", range);
            request.Headers.Add("x-ms-copy-source", copySource);
            if (sourceRange != null)
            {
                request.Headers.Add("x-ms-source-range", sourceRange);
            }
            request.Headers.Add("x-ms-write", _fileRangeWriteFromUrl);
            if (sourceContentCrc64 != null)
            {
                request.Headers.Add("x-ms-source-content-crc64", sourceContentCrc64, "D");
            }
            if (sourceModifiedAccessConditions?.SourceIfMatchCrc64 != null)
            {
                request.Headers.Add("x-ms-source-if-match-crc64", sourceModifiedAccessConditions.SourceIfMatchCrc64, "D");
            }
            if (sourceModifiedAccessConditions?.SourceIfNoneMatchCrc64 != null)
            {
                request.Headers.Add("x-ms-source-if-none-match-crc64", sourceModifiedAccessConditions.SourceIfNoneMatchCrc64, "D");
            }
            request.Headers.Add("x-ms-version", _version);
            if (shareFileRequestConditions?.LeaseId != null)
            {
                request.Headers.Add("x-ms-lease-id", shareFileRequestConditions.LeaseId);
            }
            if (copySourceAuthorization != null)
            {
                request.Headers.Add("x-ms-copy-source-authorization", copySourceAuthorization);
            }
            if (fileLastWrittenMode != null)
            {
                request.Headers.Add("x-ms-file-last-write-time", fileLastWrittenMode.Value.ToSerialString());
            }
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_allowSourceTrailingDot != null)
            {
                request.Headers.Add("x-ms-source-allow-trailing-dot", _allowSourceTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Upload a range of bytes to a file where the contents are read from a URL. </summary>
        /// <param name="range"> Writes data to the specified byte range in the file. </param>
        /// <param name="copySource"> Specifies the URL of the source file or blob, up to 2 KB in length. To copy a file to another file within the same storage account, you may use Shared Key to authenticate the source file. If you are copying a file from another storage account, or if you are copying a blob from the same storage account or another storage account, then you must authenticate the source file or blob using a shared access signature. If the source is a public blob, no authentication is required to perform the copy operation. A file in a share snapshot can also be specified as a copy source. </param>
        /// <param name="contentLength"> Specifies the number of bytes being transmitted in the request body. When the x-ms-write header is set to clear, the value of this header must be set to zero. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="sourceRange"> Bytes of source data in the specified range. </param>
        /// <param name="sourceContentCrc64"> Specify the crc64 calculated for the range of bytes that must be read from the copy source. </param>
        /// <param name="copySourceAuthorization"> Only Bearer type is supported. Credentials should be a valid OAuth access token to copy source. </param>
        /// <param name="fileLastWrittenMode"> If the file last write time should be preserved or overwritten. </param>
        /// <param name="sourceModifiedAccessConditions"> Parameter group. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="range"/> or <paramref name="copySource"/> is null. </exception>
        public async Task<ResponseWithHeaders<FileUploadRangeFromURLHeaders>> UploadRangeFromURLAsync(string range, string copySource, long contentLength, int? timeout = null, string sourceRange = null, byte[] sourceContentCrc64 = null, string copySourceAuthorization = null, FileLastWrittenMode? fileLastWrittenMode = null, SourceModifiedAccessConditions sourceModifiedAccessConditions = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            if (range == null)
            {
                throw new ArgumentNullException(nameof(range));
            }
            if (copySource == null)
            {
                throw new ArgumentNullException(nameof(copySource));
            }

            using var message = CreateUploadRangeFromURLRequest(range, copySource, contentLength, timeout, sourceRange, sourceContentCrc64, copySourceAuthorization, fileLastWrittenMode, sourceModifiedAccessConditions, shareFileRequestConditions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileUploadRangeFromURLHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Upload a range of bytes to a file where the contents are read from a URL. </summary>
        /// <param name="range"> Writes data to the specified byte range in the file. </param>
        /// <param name="copySource"> Specifies the URL of the source file or blob, up to 2 KB in length. To copy a file to another file within the same storage account, you may use Shared Key to authenticate the source file. If you are copying a file from another storage account, or if you are copying a blob from the same storage account or another storage account, then you must authenticate the source file or blob using a shared access signature. If the source is a public blob, no authentication is required to perform the copy operation. A file in a share snapshot can also be specified as a copy source. </param>
        /// <param name="contentLength"> Specifies the number of bytes being transmitted in the request body. When the x-ms-write header is set to clear, the value of this header must be set to zero. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="sourceRange"> Bytes of source data in the specified range. </param>
        /// <param name="sourceContentCrc64"> Specify the crc64 calculated for the range of bytes that must be read from the copy source. </param>
        /// <param name="copySourceAuthorization"> Only Bearer type is supported. Credentials should be a valid OAuth access token to copy source. </param>
        /// <param name="fileLastWrittenMode"> If the file last write time should be preserved or overwritten. </param>
        /// <param name="sourceModifiedAccessConditions"> Parameter group. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="range"/> or <paramref name="copySource"/> is null. </exception>
        public ResponseWithHeaders<FileUploadRangeFromURLHeaders> UploadRangeFromURL(string range, string copySource, long contentLength, int? timeout = null, string sourceRange = null, byte[] sourceContentCrc64 = null, string copySourceAuthorization = null, FileLastWrittenMode? fileLastWrittenMode = null, SourceModifiedAccessConditions sourceModifiedAccessConditions = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            if (range == null)
            {
                throw new ArgumentNullException(nameof(range));
            }
            if (copySource == null)
            {
                throw new ArgumentNullException(nameof(copySource));
            }

            using var message = CreateUploadRangeFromURLRequest(range, copySource, contentLength, timeout, sourceRange, sourceContentCrc64, copySourceAuthorization, fileLastWrittenMode, sourceModifiedAccessConditions, shareFileRequestConditions);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileUploadRangeFromURLHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetRangeListRequest(string sharesnapshot, string prevsharesnapshot, int? timeout, string range, bool? supportRename, ShareFileRequestConditions shareFileRequestConditions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "rangelist", true);
            if (sharesnapshot != null)
            {
                uri.AppendQuery("sharesnapshot", sharesnapshot, true);
            }
            if (prevsharesnapshot != null)
            {
                uri.AppendQuery("prevsharesnapshot", prevsharesnapshot, true);
            }
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            if (range != null)
            {
                request.Headers.Add("x-ms-range", range);
            }
            if (shareFileRequestConditions?.LeaseId != null)
            {
                request.Headers.Add("x-ms-lease-id", shareFileRequestConditions.LeaseId);
            }
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            if (supportRename != null)
            {
                request.Headers.Add("x-ms-file-support-rename", supportRename.Value);
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Returns the list of valid ranges for a file. </summary>
        /// <param name="sharesnapshot"> The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. </param>
        /// <param name="prevsharesnapshot"> The previous snapshot parameter is an opaque DateTime value that, when present, specifies the previous snapshot. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="range"> Specifies the range of bytes over which to list ranges, inclusively. </param>
        /// <param name="supportRename"> This header is allowed only when PrevShareSnapshot query parameter is set. Determines whether the changed ranges for a file that has been renamed or moved between the target snapshot (or the live file) and the previous snapshot should be listed. If the value is true, the valid changed ranges for the file will be returned. If the value is false, the operation will result in a failure with 409 (Conflict) response. The default value is false. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<ResponseWithHeaders<ShareFileRangeList, FileGetRangeListHeaders>> GetRangeListAsync(string sharesnapshot = null, string prevsharesnapshot = null, int? timeout = null, string range = null, bool? supportRename = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetRangeListRequest(sharesnapshot, prevsharesnapshot, timeout, range, supportRename, shareFileRequestConditions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileGetRangeListHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ShareFileRangeList value = default;
                        var document = XDocument.Load(message.Response.ContentStream, LoadOptions.PreserveWhitespace);
                        if (document.Element("Ranges") is XElement rangesElement)
                        {
                            value = ShareFileRangeList.DeserializeShareFileRangeList(rangesElement);
                        }
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Returns the list of valid ranges for a file. </summary>
        /// <param name="sharesnapshot"> The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. </param>
        /// <param name="prevsharesnapshot"> The previous snapshot parameter is an opaque DateTime value that, when present, specifies the previous snapshot. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="range"> Specifies the range of bytes over which to list ranges, inclusively. </param>
        /// <param name="supportRename"> This header is allowed only when PrevShareSnapshot query parameter is set. Determines whether the changed ranges for a file that has been renamed or moved between the target snapshot (or the live file) and the previous snapshot should be listed. If the value is true, the valid changed ranges for the file will be returned. If the value is false, the operation will result in a failure with 409 (Conflict) response. The default value is false. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public ResponseWithHeaders<ShareFileRangeList, FileGetRangeListHeaders> GetRangeList(string sharesnapshot = null, string prevsharesnapshot = null, int? timeout = null, string range = null, bool? supportRename = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetRangeListRequest(sharesnapshot, prevsharesnapshot, timeout, range, supportRename, shareFileRequestConditions);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileGetRangeListHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ShareFileRangeList value = default;
                        var document = XDocument.Load(message.Response.ContentStream, LoadOptions.PreserveWhitespace);
                        if (document.Element("Ranges") is XElement rangesElement)
                        {
                            value = ShareFileRangeList.DeserializeShareFileRangeList(rangesElement);
                        }
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateStartCopyRequest(string copySource, int? timeout, IDictionary<string, string> metadata, string filePermission, FilePermissionFormat? filePermissionFormat, string filePermissionKey, CopyFileSmbInfo copyFileSmbInfo, ShareFileRequestConditions shareFileRequestConditions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            if (metadata != null)
            {
                request.Headers.Add("x-ms-meta-", metadata);
            }
            request.Headers.Add("x-ms-copy-source", copySource);
            if (filePermission != null)
            {
                request.Headers.Add("x-ms-file-permission", filePermission);
            }
            if (filePermissionFormat != null)
            {
                request.Headers.Add("x-ms-file-permission-format", filePermissionFormat.Value.ToSerialString());
            }
            if (filePermissionKey != null)
            {
                request.Headers.Add("x-ms-file-permission-key", filePermissionKey);
            }
            if (copyFileSmbInfo?.FilePermissionCopyMode != null)
            {
                request.Headers.Add("x-ms-file-permission-copy-mode", copyFileSmbInfo.FilePermissionCopyMode.Value.ToSerialString());
            }
            if (copyFileSmbInfo?.IgnoreReadOnly != null)
            {
                request.Headers.Add("x-ms-file-copy-ignore-readonly", copyFileSmbInfo.IgnoreReadOnly.Value);
            }
            if (copyFileSmbInfo?.FileAttributes != null)
            {
                request.Headers.Add("x-ms-file-attributes", copyFileSmbInfo.FileAttributes);
            }
            if (copyFileSmbInfo?.FileCreationTime != null)
            {
                request.Headers.Add("x-ms-file-creation-time", copyFileSmbInfo.FileCreationTime);
            }
            if (copyFileSmbInfo?.FileLastWriteTime != null)
            {
                request.Headers.Add("x-ms-file-last-write-time", copyFileSmbInfo.FileLastWriteTime);
            }
            if (copyFileSmbInfo?.FileChangeTime != null)
            {
                request.Headers.Add("x-ms-file-change-time", copyFileSmbInfo.FileChangeTime);
            }
            if (copyFileSmbInfo?.SetArchiveAttribute != null)
            {
                request.Headers.Add("x-ms-file-copy-set-archive", copyFileSmbInfo.SetArchiveAttribute.Value);
            }
            if (shareFileRequestConditions?.LeaseId != null)
            {
                request.Headers.Add("x-ms-lease-id", shareFileRequestConditions.LeaseId);
            }
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_allowSourceTrailingDot != null)
            {
                request.Headers.Add("x-ms-source-allow-trailing-dot", _allowSourceTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Copies a blob or file to a destination file within the storage account. </summary>
        /// <param name="copySource"> Specifies the URL of the source file or blob, up to 2 KB in length. To copy a file to another file within the same storage account, you may use Shared Key to authenticate the source file. If you are copying a file from another storage account, or if you are copying a blob from the same storage account or another storage account, then you must authenticate the source file or blob using a shared access signature. If the source is a public blob, no authentication is required to perform the copy operation. A file in a share snapshot can also be specified as a copy source. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="metadata"> A name-value pair to associate with a file storage object. </param>
        /// <param name="filePermission"> If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is &lt;= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="filePermissionFormat"> Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission. </param>
        /// <param name="filePermissionKey"> Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="copyFileSmbInfo"> Parameter group. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="copySource"/> is null. </exception>
        public async Task<ResponseWithHeaders<FileStartCopyHeaders>> StartCopyAsync(string copySource, int? timeout = null, IDictionary<string, string> metadata = null, string filePermission = null, FilePermissionFormat? filePermissionFormat = null, string filePermissionKey = null, CopyFileSmbInfo copyFileSmbInfo = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            if (copySource == null)
            {
                throw new ArgumentNullException(nameof(copySource));
            }

            using var message = CreateStartCopyRequest(copySource, timeout, metadata, filePermission, filePermissionFormat, filePermissionKey, copyFileSmbInfo, shareFileRequestConditions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileStartCopyHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 202:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Copies a blob or file to a destination file within the storage account. </summary>
        /// <param name="copySource"> Specifies the URL of the source file or blob, up to 2 KB in length. To copy a file to another file within the same storage account, you may use Shared Key to authenticate the source file. If you are copying a file from another storage account, or if you are copying a blob from the same storage account or another storage account, then you must authenticate the source file or blob using a shared access signature. If the source is a public blob, no authentication is required to perform the copy operation. A file in a share snapshot can also be specified as a copy source. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="metadata"> A name-value pair to associate with a file storage object. </param>
        /// <param name="filePermission"> If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is &lt;= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="filePermissionFormat"> Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission. </param>
        /// <param name="filePermissionKey"> Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="copyFileSmbInfo"> Parameter group. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="copySource"/> is null. </exception>
        public ResponseWithHeaders<FileStartCopyHeaders> StartCopy(string copySource, int? timeout = null, IDictionary<string, string> metadata = null, string filePermission = null, FilePermissionFormat? filePermissionFormat = null, string filePermissionKey = null, CopyFileSmbInfo copyFileSmbInfo = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            if (copySource == null)
            {
                throw new ArgumentNullException(nameof(copySource));
            }

            using var message = CreateStartCopyRequest(copySource, timeout, metadata, filePermission, filePermissionFormat, filePermissionKey, copyFileSmbInfo, shareFileRequestConditions);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileStartCopyHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 202:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateAbortCopyRequest(string copyId, int? timeout, ShareFileRequestConditions shareFileRequestConditions)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "copy", true);
            uri.AppendQuery("copyid", copyId, true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-copy-action", "abort");
            request.Headers.Add("x-ms-version", _version);
            if (shareFileRequestConditions?.LeaseId != null)
            {
                request.Headers.Add("x-ms-lease-id", shareFileRequestConditions.LeaseId);
            }
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Aborts a pending Copy File operation, and leaves a destination file with zero length and full metadata. </summary>
        /// <param name="copyId"> The copy identifier provided in the x-ms-copy-id header of the original Copy File operation. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="copyId"/> is null. </exception>
        public async Task<ResponseWithHeaders<FileAbortCopyHeaders>> AbortCopyAsync(string copyId, int? timeout = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            if (copyId == null)
            {
                throw new ArgumentNullException(nameof(copyId));
            }

            using var message = CreateAbortCopyRequest(copyId, timeout, shareFileRequestConditions);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileAbortCopyHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Aborts a pending Copy File operation, and leaves a destination file with zero length and full metadata. </summary>
        /// <param name="copyId"> The copy identifier provided in the x-ms-copy-id header of the original Copy File operation. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="shareFileRequestConditions"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="copyId"/> is null. </exception>
        public ResponseWithHeaders<FileAbortCopyHeaders> AbortCopy(string copyId, int? timeout = null, ShareFileRequestConditions shareFileRequestConditions = null, CancellationToken cancellationToken = default)
        {
            if (copyId == null)
            {
                throw new ArgumentNullException(nameof(copyId));
            }

            using var message = CreateAbortCopyRequest(copyId, timeout, shareFileRequestConditions);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileAbortCopyHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateListHandlesRequest(string marker, int? maxresults, int? timeout, string sharesnapshot)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "listhandles", true);
            if (marker != null)
            {
                uri.AppendQuery("marker", marker, true);
            }
            if (maxresults != null)
            {
                uri.AppendQuery("maxresults", maxresults.Value, true);
            }
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            if (sharesnapshot != null)
            {
                uri.AppendQuery("sharesnapshot", sharesnapshot, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Lists handles for file. </summary>
        /// <param name="marker"> A string value that identifies the portion of the list to be returned with the next list operation. The operation returns a marker value within the response body if the list returned was not complete. The marker value may then be used in a subsequent call to request the next set of list items. The marker value is opaque to the client. </param>
        /// <param name="maxresults"> Specifies the maximum number of entries to return. If the request does not specify maxresults, or specifies a value greater than 5,000, the server will return up to 5,000 items. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="sharesnapshot"> The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<ResponseWithHeaders<ListHandlesResponse, FileListHandlesHeaders>> ListHandlesAsync(string marker = null, int? maxresults = null, int? timeout = null, string sharesnapshot = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateListHandlesRequest(marker, maxresults, timeout, sharesnapshot);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileListHandlesHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ListHandlesResponse value = default;
                        var document = XDocument.Load(message.Response.ContentStream, LoadOptions.PreserveWhitespace);
                        if (document.Element("EnumerationResults") is XElement enumerationResultsElement)
                        {
                            value = ListHandlesResponse.DeserializeListHandlesResponse(enumerationResultsElement);
                        }
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Lists handles for file. </summary>
        /// <param name="marker"> A string value that identifies the portion of the list to be returned with the next list operation. The operation returns a marker value within the response body if the list returned was not complete. The marker value may then be used in a subsequent call to request the next set of list items. The marker value is opaque to the client. </param>
        /// <param name="maxresults"> Specifies the maximum number of entries to return. If the request does not specify maxresults, or specifies a value greater than 5,000, the server will return up to 5,000 items. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="sharesnapshot"> The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public ResponseWithHeaders<ListHandlesResponse, FileListHandlesHeaders> ListHandles(string marker = null, int? maxresults = null, int? timeout = null, string sharesnapshot = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateListHandlesRequest(marker, maxresults, timeout, sharesnapshot);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileListHandlesHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ListHandlesResponse value = default;
                        var document = XDocument.Load(message.Response.ContentStream, LoadOptions.PreserveWhitespace);
                        if (document.Element("EnumerationResults") is XElement enumerationResultsElement)
                        {
                            value = ListHandlesResponse.DeserializeListHandlesResponse(enumerationResultsElement);
                        }
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateForceCloseHandlesRequest(string handleId, int? timeout, string marker, string sharesnapshot)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "forceclosehandles", true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            if (marker != null)
            {
                uri.AppendQuery("marker", marker, true);
            }
            if (sharesnapshot != null)
            {
                uri.AppendQuery("sharesnapshot", sharesnapshot, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-handle-id", handleId);
            request.Headers.Add("x-ms-version", _version);
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Closes all handles open for given file. </summary>
        /// <param name="handleId"> Specifies handle ID opened on the file or directory to be closed. Asterisk (‘*’) is a wildcard that specifies all handles. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="marker"> A string value that identifies the portion of the list to be returned with the next list operation. The operation returns a marker value within the response body if the list returned was not complete. The marker value may then be used in a subsequent call to request the next set of list items. The marker value is opaque to the client. </param>
        /// <param name="sharesnapshot"> The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="handleId"/> is null. </exception>
        public async Task<ResponseWithHeaders<FileForceCloseHandlesHeaders>> ForceCloseHandlesAsync(string handleId, int? timeout = null, string marker = null, string sharesnapshot = null, CancellationToken cancellationToken = default)
        {
            if (handleId == null)
            {
                throw new ArgumentNullException(nameof(handleId));
            }

            using var message = CreateForceCloseHandlesRequest(handleId, timeout, marker, sharesnapshot);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileForceCloseHandlesHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Closes all handles open for given file. </summary>
        /// <param name="handleId"> Specifies handle ID opened on the file or directory to be closed. Asterisk (‘*’) is a wildcard that specifies all handles. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="marker"> A string value that identifies the portion of the list to be returned with the next list operation. The operation returns a marker value within the response body if the list returned was not complete. The marker value may then be used in a subsequent call to request the next set of list items. The marker value is opaque to the client. </param>
        /// <param name="sharesnapshot"> The snapshot parameter is an opaque DateTime value that, when present, specifies the share snapshot to query. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="handleId"/> is null. </exception>
        public ResponseWithHeaders<FileForceCloseHandlesHeaders> ForceCloseHandles(string handleId, int? timeout = null, string marker = null, string sharesnapshot = null, CancellationToken cancellationToken = default)
        {
            if (handleId == null)
            {
                throw new ArgumentNullException(nameof(handleId));
            }

            using var message = CreateForceCloseHandlesRequest(handleId, timeout, marker, sharesnapshot);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileForceCloseHandlesHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateRenameRequest(string renameSource, int? timeout, bool? replaceIfExists, bool? ignoreReadOnly, string sourceLeaseId, string destinationLeaseId, string filePermission, FilePermissionFormat? filePermissionFormat, string filePermissionKey, IDictionary<string, string> metadata, CopyFileSmbInfo copyFileSmbInfo, FileHttpHeaders fileHttpHeaders)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendQuery("comp", "rename", true);
            if (timeout != null)
            {
                uri.AppendQuery("timeout", timeout.Value, true);
            }
            request.Uri = uri;
            request.Headers.Add("x-ms-version", _version);
            request.Headers.Add("x-ms-file-rename-source", renameSource);
            if (replaceIfExists != null)
            {
                request.Headers.Add("x-ms-file-rename-replace-if-exists", replaceIfExists.Value);
            }
            if (ignoreReadOnly != null)
            {
                request.Headers.Add("x-ms-file-rename-ignore-readonly", ignoreReadOnly.Value);
            }
            if (sourceLeaseId != null)
            {
                request.Headers.Add("x-ms-source-lease-id", sourceLeaseId);
            }
            if (destinationLeaseId != null)
            {
                request.Headers.Add("x-ms-destination-lease-id", destinationLeaseId);
            }
            if (copyFileSmbInfo?.FileAttributes != null)
            {
                request.Headers.Add("x-ms-file-attributes", copyFileSmbInfo.FileAttributes);
            }
            if (copyFileSmbInfo?.FileCreationTime != null)
            {
                request.Headers.Add("x-ms-file-creation-time", copyFileSmbInfo.FileCreationTime);
            }
            if (copyFileSmbInfo?.FileLastWriteTime != null)
            {
                request.Headers.Add("x-ms-file-last-write-time", copyFileSmbInfo.FileLastWriteTime);
            }
            if (copyFileSmbInfo?.FileChangeTime != null)
            {
                request.Headers.Add("x-ms-file-change-time", copyFileSmbInfo.FileChangeTime);
            }
            if (filePermission != null)
            {
                request.Headers.Add("x-ms-file-permission", filePermission);
            }
            if (filePermissionFormat != null)
            {
                request.Headers.Add("x-ms-file-permission-format", filePermissionFormat.Value.ToSerialString());
            }
            if (filePermissionKey != null)
            {
                request.Headers.Add("x-ms-file-permission-key", filePermissionKey);
            }
            if (metadata != null)
            {
                request.Headers.Add("x-ms-meta-", metadata);
            }
            if (fileHttpHeaders?.FileContentType != null)
            {
                request.Headers.Add("x-ms-content-type", fileHttpHeaders.FileContentType);
            }
            if (_allowTrailingDot != null)
            {
                request.Headers.Add("x-ms-allow-trailing-dot", _allowTrailingDot.Value);
            }
            if (_allowSourceTrailingDot != null)
            {
                request.Headers.Add("x-ms-source-allow-trailing-dot", _allowSourceTrailingDot.Value);
            }
            if (_fileRequestIntent != null)
            {
                request.Headers.Add("x-ms-file-request-intent", _fileRequestIntent.Value.ToString());
            }
            request.Headers.Add("Accept", "application/xml");
            return message;
        }

        /// <summary> Renames a file. </summary>
        /// <param name="renameSource"> Required. Specifies the URI-style path of the source file, up to 2 KB in length. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="replaceIfExists"> Optional. A boolean value for if the destination file already exists, whether this request will overwrite the file or not. If true, the rename will succeed and will overwrite the destination file. If not provided or if false and the destination file does exist, the request will not overwrite the destination file. If provided and the destination file doesn’t exist, the rename will succeed. Note: This value does not override the x-ms-file-copy-ignore-read-only header value. </param>
        /// <param name="ignoreReadOnly"> Optional. A boolean value that specifies whether the ReadOnly attribute on a preexisting destination file should be respected. If true, the rename will succeed, otherwise, a previous file at the destination with the ReadOnly attribute set will cause the rename to fail. </param>
        /// <param name="sourceLeaseId"> Required if the source file has an active infinite lease. </param>
        /// <param name="destinationLeaseId"> Required if the destination file has an active infinite lease. The lease ID specified for this header must match the lease ID of the destination file. If the request does not include the lease ID or it is not valid, the operation fails with status code 412 (Precondition Failed). If this header is specified and the destination file does not currently have an active lease, the operation will also fail with status code 412 (Precondition Failed). </param>
        /// <param name="filePermission"> If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is &lt;= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="filePermissionFormat"> Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission. </param>
        /// <param name="filePermissionKey"> Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="metadata"> A name-value pair to associate with a file storage object. </param>
        /// <param name="copyFileSmbInfo"> Parameter group. </param>
        /// <param name="fileHttpHeaders"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="renameSource"/> is null. </exception>
        public async Task<ResponseWithHeaders<FileRenameHeaders>> RenameAsync(string renameSource, int? timeout = null, bool? replaceIfExists = null, bool? ignoreReadOnly = null, string sourceLeaseId = null, string destinationLeaseId = null, string filePermission = null, FilePermissionFormat? filePermissionFormat = null, string filePermissionKey = null, IDictionary<string, string> metadata = null, CopyFileSmbInfo copyFileSmbInfo = null, FileHttpHeaders fileHttpHeaders = null, CancellationToken cancellationToken = default)
        {
            if (renameSource == null)
            {
                throw new ArgumentNullException(nameof(renameSource));
            }

            using var message = CreateRenameRequest(renameSource, timeout, replaceIfExists, ignoreReadOnly, sourceLeaseId, destinationLeaseId, filePermission, filePermissionFormat, filePermissionKey, metadata, copyFileSmbInfo, fileHttpHeaders);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new FileRenameHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Renames a file. </summary>
        /// <param name="renameSource"> Required. Specifies the URI-style path of the source file, up to 2 KB in length. </param>
        /// <param name="timeout"> The timeout parameter is expressed in seconds. For more information, see &lt;a href="https://docs.microsoft.com/en-us/rest/api/storageservices/Setting-Timeouts-for-File-Service-Operations?redirectedfrom=MSDN"&gt;Setting Timeouts for File Service Operations.&lt;/a&gt;. </param>
        /// <param name="replaceIfExists"> Optional. A boolean value for if the destination file already exists, whether this request will overwrite the file or not. If true, the rename will succeed and will overwrite the destination file. If not provided or if false and the destination file does exist, the request will not overwrite the destination file. If provided and the destination file doesn’t exist, the rename will succeed. Note: This value does not override the x-ms-file-copy-ignore-read-only header value. </param>
        /// <param name="ignoreReadOnly"> Optional. A boolean value that specifies whether the ReadOnly attribute on a preexisting destination file should be respected. If true, the rename will succeed, otherwise, a previous file at the destination with the ReadOnly attribute set will cause the rename to fail. </param>
        /// <param name="sourceLeaseId"> Required if the source file has an active infinite lease. </param>
        /// <param name="destinationLeaseId"> Required if the destination file has an active infinite lease. The lease ID specified for this header must match the lease ID of the destination file. If the request does not include the lease ID or it is not valid, the operation fails with status code 412 (Precondition Failed). If this header is specified and the destination file does not currently have an active lease, the operation will also fail with status code 412 (Precondition Failed). </param>
        /// <param name="filePermission"> If specified the permission (security descriptor) shall be set for the directory/file. This header can be used if Permission size is &lt;= 8KB, else x-ms-file-permission-key header shall be used. Default value: Inherit. If SDDL is specified as input, it must have owner, group and dacl. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="filePermissionFormat"> Optional. Available for version 2023-06-01 and later. Specifies the format in which the permission is returned. Acceptable values are SDDL or binary. If x-ms-file-permission-format is unspecified or explicitly set to SDDL, the permission is returned in SDDL format. If x-ms-file-permission-format is explicitly set to binary, the permission is returned as a base64 string representing the binary encoding of the permission. </param>
        /// <param name="filePermissionKey"> Key of the permission to be set for the directory/file. Note: Only one of the x-ms-file-permission or x-ms-file-permission-key should be specified. </param>
        /// <param name="metadata"> A name-value pair to associate with a file storage object. </param>
        /// <param name="copyFileSmbInfo"> Parameter group. </param>
        /// <param name="fileHttpHeaders"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="renameSource"/> is null. </exception>
        public ResponseWithHeaders<FileRenameHeaders> Rename(string renameSource, int? timeout = null, bool? replaceIfExists = null, bool? ignoreReadOnly = null, string sourceLeaseId = null, string destinationLeaseId = null, string filePermission = null, FilePermissionFormat? filePermissionFormat = null, string filePermissionKey = null, IDictionary<string, string> metadata = null, CopyFileSmbInfo copyFileSmbInfo = null, FileHttpHeaders fileHttpHeaders = null, CancellationToken cancellationToken = default)
        {
            if (renameSource == null)
            {
                throw new ArgumentNullException(nameof(renameSource));
            }

            using var message = CreateRenameRequest(renameSource, timeout, replaceIfExists, ignoreReadOnly, sourceLeaseId, destinationLeaseId, filePermission, filePermissionFormat, filePermissionKey, metadata, copyFileSmbInfo, fileHttpHeaders);
            _pipeline.Send(message, cancellationToken);
            var headers = new FileRenameHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }
    }
}
