﻿// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

using System.Linq;
using System.Threading.Tasks;
using Azure.Core;
using Azure.Core.TestFramework;
using Azure.ResourceManager.Resources;
using Azure.ResourceManager.Resources.Models;
using NUnit.Framework;

namespace Azure.ResourceManager.Sql.Tests
{
    public class ManagedInstanceVulnerabilityAssessmentTests : SqlManagementTestBase
    {
        private ResourceGroupResource _resourceGroup;
        private ResourceIdentifier _resourceGroupIdentifier;
        public ManagedInstanceVulnerabilityAssessmentTests(bool isAsync)
            : base(isAsync)//, RecordedTestMode.Record)
        {
        }

        [OneTimeSetUp]
        public async Task GlobalSetUp()
        {
            var rgLro = await GlobalClient.GetDefaultSubscriptionAsync().Result.GetResourceGroups().CreateOrUpdateAsync(WaitUntil.Completed, SessionRecording.GenerateAssetName("Sql-RG-"), new ResourceGroupData(AzureLocation.WestUS2));
            ResourceGroupResource rg = rgLro.Value;
            _resourceGroupIdentifier = rg.Id;
            await StopSessionRecordingAsync();
        }

        [SetUp]
        public async Task TestSetUp()
        {
            var client = GetArmClient();
            _resourceGroup = await client.GetResourceGroupResource(_resourceGroupIdentifier).GetAsync();
        }

        [Test]
        [RecordedTest]
        public async Task ManagedInstanceVulnerabilityAssessmentApiTests()
        {
            // Create Managed Instance
            string managedInstanceName = Recording.GenerateAssetName("managed-instance-");
            string vnetName = Recording.GenerateAssetName("vnet-");
            var managedInstance = await CreateDefaultManagedInstance(managedInstanceName, vnetName, AzureLocation.WestUS2, _resourceGroup);
            Assert.IsNotNull(managedInstance.Data);

            string vulnerabilityAssessmentName = "Default";
            var collection = managedInstance.GetManagedInstanceVulnerabilityAssessments();

            //// 1.CreateOrUpdate - Ignore("need a valid blob storage container endpoint")
            //ManagedInstanceVulnerabilityAssessmentData data = new ManagedInstanceVulnerabilityAssessmentData()
            //{
            //    RecurringScans = new VulnerabilityAssessmentRecurringScansProperties()
            //    {
            //        IsEnabled = false,
            //        EmailSubscriptionAdmins = true,
            //    },
            //    StorageContainerPath = "SecureString",    //Value should be a valid blob storage container endpoint (e.g. https://MyAccount.blob.core.windows.net/containername)."
            //};
            //var vulnerabilityAssessment = await collection.CreateOrUpdateAsync(vulnerabilityAssessmentName, data);
            //Assert.IsNotNull(vulnerabilityAssessment.Value.Data);

            // 2.CheckIfExist
            Assert.IsTrue(await collection.ExistsAsync(vulnerabilityAssessmentName));

            // 3.Get
            var getvulnerabilityAssessment = await collection.GetAsync(vulnerabilityAssessmentName);
            Assert.IsNotNull(getvulnerabilityAssessment.Value.Data);
            Assert.AreEqual(vulnerabilityAssessmentName, getvulnerabilityAssessment.Value.Data.Name);

            // 4.GetAll
            var list = await collection.GetAllAsync().ToEnumerableAsync();
            Assert.IsNotEmpty(list);
            Assert.AreEqual(1, list.Count);
            Assert.AreEqual(vulnerabilityAssessmentName, list.FirstOrDefault().Data.Name);
        }
    }
}
