﻿// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Azure.Core;
using Azure.Core.TestFramework;
using Azure.ResourceManager.Resources;
using Azure.ResourceManager.SecurityCenter.Models;
using NUnit.Framework;

namespace Azure.ResourceManager.SecurityCenter.Tests
{
    internal class ServerVulnerabilityAssessmentTests : SecurityCenterManagementTestBase
    {
        private ResourceGroupResource _resourceGroup;
        private ServerVulnerabilityAssessmentCollection _serverVulnerabilityAssessmentCollection;
        public ServerVulnerabilityAssessmentTests(bool isAsync) : base(isAsync)//, RecordedTestMode.Record)
        {
        }

        [SetUp]
        public async Task TestSetUp()
        {
            DefaultLocation = AzureLocation.EastUS2;
            _resourceGroup = await CreateResourceGroup();
            string interfaceName = Recording.GenerateAssetName("networkInterface");
            string vmName = Recording.GenerateAssetName("vm");
            var networkInterface = await CreateNetworkInterface(_resourceGroup, interfaceName);
            var vm = await CreateVirtualMachine(_resourceGroup, networkInterface.Id, vmName);
            _serverVulnerabilityAssessmentCollection = _resourceGroup.GetServerVulnerabilityAssessments("Microsoft.Compute", "virtualMachines", vm.Id.Name);
        }

        [RecordedTest]
        public async Task ServerVulnerabilityAssessmentOperationE2ETest()
        {
            // CreateOrUpdate
            var serverVulnerabilityAssessment = await _serverVulnerabilityAssessmentCollection.CreateOrUpdateAsync(WaitUntil.Completed);
            ValidateServerVulnerabilityAssessment(serverVulnerabilityAssessment.Value);
            if (TestEnvironment.Mode == RecordedTestMode.Record)
            {
                Thread.Sleep(60000); // wait for auto deployment
            }

            // Exist
            bool flag = await _serverVulnerabilityAssessmentCollection.ExistsAsync();
            Assert.IsTrue(flag);

            // Get
            var getServerVulnerabilityAssessment = await _serverVulnerabilityAssessmentCollection.GetAsync();
            ValidateServerVulnerabilityAssessment(getServerVulnerabilityAssessment.Value);

            // GetAll
            var list = await _serverVulnerabilityAssessmentCollection.GetAllAsync().ToEnumerableAsync();
            Assert.IsNotEmpty(list);
            ValidateServerVulnerabilityAssessment(list.First());
        }

        [RecordedTest]
        [Ignore("Azure.RequestFailedException : The resource type could not be found in the namespace 'Microsoft.Security' for api version '2020-01-01'.")]
        public async Task Delete()
        {
            var serverVulnerabilityAssessment = await _serverVulnerabilityAssessmentCollection.CreateOrUpdateAsync(WaitUntil.Completed);
            await serverVulnerabilityAssessment.Value.DeleteAsync(WaitUntil.Completed);
            bool flag = await _serverVulnerabilityAssessmentCollection.ExistsAsync();
            Assert.IsFalse(flag);
        }

        private void ValidateServerVulnerabilityAssessment(ServerVulnerabilityAssessmentResource serverVulnerabilityAssessment)
        {
            Assert.IsNotNull(serverVulnerabilityAssessment);
            Assert.IsNotNull(serverVulnerabilityAssessment.Data.Id);
            Assert.AreEqual("Default", serverVulnerabilityAssessment.Data.Name);
            Assert.AreEqual("providers/Microsoft.Security/serverVulnerabilityAssessments", serverVulnerabilityAssessment.Data.ResourceType.ToString());
        }
    }
}
