// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

// <auto-generated/>

#nullable disable

using System;
using System.Globalization;
using System.Threading;
using System.Threading.Tasks;
using Autorest.CSharp.Core;
using Azure.Core;
using Azure.Core.Pipeline;
using Azure.ResourceManager.Models;

namespace Azure.ResourceManager.SecurityCenter
{
    /// <summary>
    /// A Class representing a DevOpsConfiguration along with the instance operations that can be performed on it.
    /// If you have a <see cref="ResourceIdentifier"/> you can construct a <see cref="DevOpsConfigurationResource"/>
    /// from an instance of <see cref="ArmClient"/> using the GetDevOpsConfigurationResource method.
    /// Otherwise you can get one from its parent resource <see cref="SecurityConnectorResource"/> using the GetDevOpsConfiguration method.
    /// </summary>
    public partial class DevOpsConfigurationResource : ArmResource
    {
        /// <summary> Generate the resource identifier of a <see cref="DevOpsConfigurationResource"/> instance. </summary>
        /// <param name="subscriptionId"> The subscriptionId. </param>
        /// <param name="resourceGroupName"> The resourceGroupName. </param>
        /// <param name="securityConnectorName"> The securityConnectorName. </param>
        public static ResourceIdentifier CreateResourceIdentifier(string subscriptionId, string resourceGroupName, string securityConnectorName)
        {
            var resourceId = $"/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default";
            return new ResourceIdentifier(resourceId);
        }

        private readonly ClientDiagnostics _devOpsConfigurationClientDiagnostics;
        private readonly DevOpsConfigurationsRestOperations _devOpsConfigurationRestClient;
        private readonly ClientDiagnostics _devOpsOrgAzureDevOpsOrgsClientDiagnostics;
        private readonly AzureDevOpsOrgsRestOperations _devOpsOrgAzureDevOpsOrgsRestClient;
        private readonly ClientDiagnostics _securityConnectorGitHubOwnerGitHubOwnersClientDiagnostics;
        private readonly GitHubOwnersRestOperations _securityConnectorGitHubOwnerGitHubOwnersRestClient;
        private readonly ClientDiagnostics _securityConnectorGitLabGroupGitLabGroupsClientDiagnostics;
        private readonly GitLabGroupsRestOperations _securityConnectorGitLabGroupGitLabGroupsRestClient;
        private readonly ClientDiagnostics _devOpsOperationResultsClientDiagnostics;
        private readonly DevOpsOperationResultsRestOperations _devOpsOperationResultsRestClient;
        private readonly DevOpsConfigurationData _data;

        /// <summary> Gets the resource type for the operations. </summary>
        public static readonly ResourceType ResourceType = "Microsoft.Security/securityConnectors/devops";

        /// <summary> Initializes a new instance of the <see cref="DevOpsConfigurationResource"/> class for mocking. </summary>
        protected DevOpsConfigurationResource()
        {
        }

        /// <summary> Initializes a new instance of the <see cref="DevOpsConfigurationResource"/> class. </summary>
        /// <param name="client"> The client parameters to use in these operations. </param>
        /// <param name="data"> The resource that is the target of operations. </param>
        internal DevOpsConfigurationResource(ArmClient client, DevOpsConfigurationData data) : this(client, data.Id)
        {
            HasData = true;
            _data = data;
        }

        /// <summary> Initializes a new instance of the <see cref="DevOpsConfigurationResource"/> class. </summary>
        /// <param name="client"> The client parameters to use in these operations. </param>
        /// <param name="id"> The identifier of the resource that is the target of operations. </param>
        internal DevOpsConfigurationResource(ArmClient client, ResourceIdentifier id) : base(client, id)
        {
            _devOpsConfigurationClientDiagnostics = new ClientDiagnostics("Azure.ResourceManager.SecurityCenter", ResourceType.Namespace, Diagnostics);
            TryGetApiVersion(ResourceType, out string devOpsConfigurationApiVersion);
            _devOpsConfigurationRestClient = new DevOpsConfigurationsRestOperations(Pipeline, Diagnostics.ApplicationId, Endpoint, devOpsConfigurationApiVersion);
            _devOpsOrgAzureDevOpsOrgsClientDiagnostics = new ClientDiagnostics("Azure.ResourceManager.SecurityCenter", DevOpsOrgResource.ResourceType.Namespace, Diagnostics);
            TryGetApiVersion(DevOpsOrgResource.ResourceType, out string devOpsOrgAzureDevOpsOrgsApiVersion);
            _devOpsOrgAzureDevOpsOrgsRestClient = new AzureDevOpsOrgsRestOperations(Pipeline, Diagnostics.ApplicationId, Endpoint, devOpsOrgAzureDevOpsOrgsApiVersion);
            _securityConnectorGitHubOwnerGitHubOwnersClientDiagnostics = new ClientDiagnostics("Azure.ResourceManager.SecurityCenter", SecurityConnectorGitHubOwnerResource.ResourceType.Namespace, Diagnostics);
            TryGetApiVersion(SecurityConnectorGitHubOwnerResource.ResourceType, out string securityConnectorGitHubOwnerGitHubOwnersApiVersion);
            _securityConnectorGitHubOwnerGitHubOwnersRestClient = new GitHubOwnersRestOperations(Pipeline, Diagnostics.ApplicationId, Endpoint, securityConnectorGitHubOwnerGitHubOwnersApiVersion);
            _securityConnectorGitLabGroupGitLabGroupsClientDiagnostics = new ClientDiagnostics("Azure.ResourceManager.SecurityCenter", SecurityConnectorGitLabGroupResource.ResourceType.Namespace, Diagnostics);
            TryGetApiVersion(SecurityConnectorGitLabGroupResource.ResourceType, out string securityConnectorGitLabGroupGitLabGroupsApiVersion);
            _securityConnectorGitLabGroupGitLabGroupsRestClient = new GitLabGroupsRestOperations(Pipeline, Diagnostics.ApplicationId, Endpoint, securityConnectorGitLabGroupGitLabGroupsApiVersion);
            _devOpsOperationResultsClientDiagnostics = new ClientDiagnostics("Azure.ResourceManager.SecurityCenter", ProviderConstants.DefaultProviderNamespace, Diagnostics);
            _devOpsOperationResultsRestClient = new DevOpsOperationResultsRestOperations(Pipeline, Diagnostics.ApplicationId, Endpoint);
#if DEBUG
			ValidateResourceId(Id);
#endif
        }

        /// <summary> Gets whether or not the current instance has data. </summary>
        public virtual bool HasData { get; }

        /// <summary> Gets the data representing this Feature. </summary>
        /// <exception cref="InvalidOperationException"> Throws if there is no data loaded in the current instance. </exception>
        public virtual DevOpsConfigurationData Data
        {
            get
            {
                if (!HasData)
                    throw new InvalidOperationException("The current instance does not have data, you must call Get first.");
                return _data;
            }
        }

        internal static void ValidateResourceId(ResourceIdentifier id)
        {
            if (id.ResourceType != ResourceType)
                throw new ArgumentException(string.Format(CultureInfo.CurrentCulture, "Invalid resource type {0} expected {1}", id.ResourceType, ResourceType), nameof(id));
        }

        /// <summary> Gets a collection of DevOpsOrgResources in the DevOpsConfiguration. </summary>
        /// <returns> An object representing collection of DevOpsOrgResources and their operations over a DevOpsOrgResource. </returns>
        public virtual DevOpsOrgCollection GetDevOpsOrgs()
        {
            return GetCachedClient(client => new DevOpsOrgCollection(client, Id));
        }

        /// <summary>
        /// Returns a monitored Azure DevOps organization resource.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/azureDevOpsOrgs/{orgName}</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>AzureDevOpsOrgs_Get</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="DevOpsOrgResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="orgName"> The Azure DevOps organization name. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="orgName"/> is null. </exception>
        /// <exception cref="ArgumentException"> <paramref name="orgName"/> is an empty string, and was expected to be non-empty. </exception>
        [ForwardsClientCalls]
        public virtual async Task<Response<DevOpsOrgResource>> GetDevOpsOrgAsync(string orgName, CancellationToken cancellationToken = default)
        {
            return await GetDevOpsOrgs().GetAsync(orgName, cancellationToken).ConfigureAwait(false);
        }

        /// <summary>
        /// Returns a monitored Azure DevOps organization resource.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/azureDevOpsOrgs/{orgName}</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>AzureDevOpsOrgs_Get</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="DevOpsOrgResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="orgName"> The Azure DevOps organization name. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="orgName"/> is null. </exception>
        /// <exception cref="ArgumentException"> <paramref name="orgName"/> is an empty string, and was expected to be non-empty. </exception>
        [ForwardsClientCalls]
        public virtual Response<DevOpsOrgResource> GetDevOpsOrg(string orgName, CancellationToken cancellationToken = default)
        {
            return GetDevOpsOrgs().Get(orgName, cancellationToken);
        }

        /// <summary> Gets a collection of SecurityConnectorGitHubOwnerResources in the DevOpsConfiguration. </summary>
        /// <returns> An object representing collection of SecurityConnectorGitHubOwnerResources and their operations over a SecurityConnectorGitHubOwnerResource. </returns>
        public virtual SecurityConnectorGitHubOwnerCollection GetSecurityConnectorGitHubOwners()
        {
            return GetCachedClient(client => new SecurityConnectorGitHubOwnerCollection(client, Id));
        }

        /// <summary>
        /// Returns a monitored GitHub owner.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/gitHubOwners/{ownerName}</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>GitHubOwners_Get</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="SecurityConnectorGitHubOwnerResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="ownerName"> The GitHub owner name. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="ownerName"/> is null. </exception>
        /// <exception cref="ArgumentException"> <paramref name="ownerName"/> is an empty string, and was expected to be non-empty. </exception>
        [ForwardsClientCalls]
        public virtual async Task<Response<SecurityConnectorGitHubOwnerResource>> GetSecurityConnectorGitHubOwnerAsync(string ownerName, CancellationToken cancellationToken = default)
        {
            return await GetSecurityConnectorGitHubOwners().GetAsync(ownerName, cancellationToken).ConfigureAwait(false);
        }

        /// <summary>
        /// Returns a monitored GitHub owner.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/gitHubOwners/{ownerName}</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>GitHubOwners_Get</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="SecurityConnectorGitHubOwnerResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="ownerName"> The GitHub owner name. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="ownerName"/> is null. </exception>
        /// <exception cref="ArgumentException"> <paramref name="ownerName"/> is an empty string, and was expected to be non-empty. </exception>
        [ForwardsClientCalls]
        public virtual Response<SecurityConnectorGitHubOwnerResource> GetSecurityConnectorGitHubOwner(string ownerName, CancellationToken cancellationToken = default)
        {
            return GetSecurityConnectorGitHubOwners().Get(ownerName, cancellationToken);
        }

        /// <summary> Gets a collection of SecurityConnectorGitLabGroupResources in the DevOpsConfiguration. </summary>
        /// <returns> An object representing collection of SecurityConnectorGitLabGroupResources and their operations over a SecurityConnectorGitLabGroupResource. </returns>
        public virtual SecurityConnectorGitLabGroupCollection GetSecurityConnectorGitLabGroups()
        {
            return GetCachedClient(client => new SecurityConnectorGitLabGroupCollection(client, Id));
        }

        /// <summary>
        /// Returns a monitored GitLab Group resource for a given fully-qualified name.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/gitLabGroups/{groupFQName}</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>GitLabGroups_Get</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="SecurityConnectorGitLabGroupResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="groupFQName"> The GitLab group fully-qualified name. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="groupFQName"/> is null. </exception>
        /// <exception cref="ArgumentException"> <paramref name="groupFQName"/> is an empty string, and was expected to be non-empty. </exception>
        [ForwardsClientCalls]
        public virtual async Task<Response<SecurityConnectorGitLabGroupResource>> GetSecurityConnectorGitLabGroupAsync(string groupFQName, CancellationToken cancellationToken = default)
        {
            return await GetSecurityConnectorGitLabGroups().GetAsync(groupFQName, cancellationToken).ConfigureAwait(false);
        }

        /// <summary>
        /// Returns a monitored GitLab Group resource for a given fully-qualified name.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/gitLabGroups/{groupFQName}</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>GitLabGroups_Get</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="SecurityConnectorGitLabGroupResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="groupFQName"> The GitLab group fully-qualified name. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="groupFQName"/> is null. </exception>
        /// <exception cref="ArgumentException"> <paramref name="groupFQName"/> is an empty string, and was expected to be non-empty. </exception>
        [ForwardsClientCalls]
        public virtual Response<SecurityConnectorGitLabGroupResource> GetSecurityConnectorGitLabGroup(string groupFQName, CancellationToken cancellationToken = default)
        {
            return GetSecurityConnectorGitLabGroups().Get(groupFQName, cancellationToken);
        }

        /// <summary>
        /// Gets a DevOps Configuration.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>DevOpsConfigurations_Get</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="DevOpsConfigurationResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public virtual async Task<Response<DevOpsConfigurationResource>> GetAsync(CancellationToken cancellationToken = default)
        {
            using var scope = _devOpsConfigurationClientDiagnostics.CreateScope("DevOpsConfigurationResource.Get");
            scope.Start();
            try
            {
                var response = await _devOpsConfigurationRestClient.GetAsync(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, cancellationToken).ConfigureAwait(false);
                if (response.Value == null)
                    throw new RequestFailedException(response.GetRawResponse());
                return Response.FromValue(new DevOpsConfigurationResource(Client, response.Value), response.GetRawResponse());
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        /// <summary>
        /// Gets a DevOps Configuration.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>DevOpsConfigurations_Get</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="DevOpsConfigurationResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public virtual Response<DevOpsConfigurationResource> Get(CancellationToken cancellationToken = default)
        {
            using var scope = _devOpsConfigurationClientDiagnostics.CreateScope("DevOpsConfigurationResource.Get");
            scope.Start();
            try
            {
                var response = _devOpsConfigurationRestClient.Get(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, cancellationToken);
                if (response.Value == null)
                    throw new RequestFailedException(response.GetRawResponse());
                return Response.FromValue(new DevOpsConfigurationResource(Client, response.Value), response.GetRawResponse());
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        /// <summary>
        /// Deletes a DevOps Connector.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>DevOpsConfigurations_Delete</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="DevOpsConfigurationResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="waitUntil"> <see cref="WaitUntil.Completed"/> if the method should wait to return until the long-running operation has completed on the service; <see cref="WaitUntil.Started"/> if it should return after starting the operation. For more information on long-running operations, please see <see href="https://github.com/Azure/azure-sdk-for-net/blob/main/sdk/core/Azure.Core/samples/LongRunningOperations.md"> Azure.Core Long-Running Operation samples</see>. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public virtual async Task<ArmOperation> DeleteAsync(WaitUntil waitUntil, CancellationToken cancellationToken = default)
        {
            using var scope = _devOpsConfigurationClientDiagnostics.CreateScope("DevOpsConfigurationResource.Delete");
            scope.Start();
            try
            {
                var response = await _devOpsConfigurationRestClient.DeleteAsync(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, cancellationToken).ConfigureAwait(false);
                var operation = new SecurityCenterArmOperation(_devOpsConfigurationClientDiagnostics, Pipeline, _devOpsConfigurationRestClient.CreateDeleteRequest(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name).Request, response, OperationFinalStateVia.Location);
                if (waitUntil == WaitUntil.Completed)
                    await operation.WaitForCompletionResponseAsync(cancellationToken).ConfigureAwait(false);
                return operation;
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        /// <summary>
        /// Deletes a DevOps Connector.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>DevOpsConfigurations_Delete</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="DevOpsConfigurationResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="waitUntil"> <see cref="WaitUntil.Completed"/> if the method should wait to return until the long-running operation has completed on the service; <see cref="WaitUntil.Started"/> if it should return after starting the operation. For more information on long-running operations, please see <see href="https://github.com/Azure/azure-sdk-for-net/blob/main/sdk/core/Azure.Core/samples/LongRunningOperations.md"> Azure.Core Long-Running Operation samples</see>. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public virtual ArmOperation Delete(WaitUntil waitUntil, CancellationToken cancellationToken = default)
        {
            using var scope = _devOpsConfigurationClientDiagnostics.CreateScope("DevOpsConfigurationResource.Delete");
            scope.Start();
            try
            {
                var response = _devOpsConfigurationRestClient.Delete(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, cancellationToken);
                var operation = new SecurityCenterArmOperation(_devOpsConfigurationClientDiagnostics, Pipeline, _devOpsConfigurationRestClient.CreateDeleteRequest(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name).Request, response, OperationFinalStateVia.Location);
                if (waitUntil == WaitUntil.Completed)
                    operation.WaitForCompletionResponse(cancellationToken);
                return operation;
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        /// <summary>
        /// Updates a DevOps Configuration.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>DevOpsConfigurations_Update</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="DevOpsConfigurationResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="waitUntil"> <see cref="WaitUntil.Completed"/> if the method should wait to return until the long-running operation has completed on the service; <see cref="WaitUntil.Started"/> if it should return after starting the operation. For more information on long-running operations, please see <see href="https://github.com/Azure/azure-sdk-for-net/blob/main/sdk/core/Azure.Core/samples/LongRunningOperations.md"> Azure.Core Long-Running Operation samples</see>. </param>
        /// <param name="data"> The DevOps configuration resource payload. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="data"/> is null. </exception>
        public virtual async Task<ArmOperation<DevOpsConfigurationResource>> UpdateAsync(WaitUntil waitUntil, DevOpsConfigurationData data, CancellationToken cancellationToken = default)
        {
            Argument.AssertNotNull(data, nameof(data));

            using var scope = _devOpsConfigurationClientDiagnostics.CreateScope("DevOpsConfigurationResource.Update");
            scope.Start();
            try
            {
                var response = await _devOpsConfigurationRestClient.UpdateAsync(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, data, cancellationToken).ConfigureAwait(false);
                var operation = new SecurityCenterArmOperation<DevOpsConfigurationResource>(new DevOpsConfigurationOperationSource(Client), _devOpsConfigurationClientDiagnostics, Pipeline, _devOpsConfigurationRestClient.CreateUpdateRequest(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, data).Request, response, OperationFinalStateVia.AzureAsyncOperation);
                if (waitUntil == WaitUntil.Completed)
                    await operation.WaitForCompletionAsync(cancellationToken).ConfigureAwait(false);
                return operation;
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        /// <summary>
        /// Updates a DevOps Configuration.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>DevOpsConfigurations_Update</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="DevOpsConfigurationResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="waitUntil"> <see cref="WaitUntil.Completed"/> if the method should wait to return until the long-running operation has completed on the service; <see cref="WaitUntil.Started"/> if it should return after starting the operation. For more information on long-running operations, please see <see href="https://github.com/Azure/azure-sdk-for-net/blob/main/sdk/core/Azure.Core/samples/LongRunningOperations.md"> Azure.Core Long-Running Operation samples</see>. </param>
        /// <param name="data"> The DevOps configuration resource payload. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="data"/> is null. </exception>
        public virtual ArmOperation<DevOpsConfigurationResource> Update(WaitUntil waitUntil, DevOpsConfigurationData data, CancellationToken cancellationToken = default)
        {
            Argument.AssertNotNull(data, nameof(data));

            using var scope = _devOpsConfigurationClientDiagnostics.CreateScope("DevOpsConfigurationResource.Update");
            scope.Start();
            try
            {
                var response = _devOpsConfigurationRestClient.Update(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, data, cancellationToken);
                var operation = new SecurityCenterArmOperation<DevOpsConfigurationResource>(new DevOpsConfigurationOperationSource(Client), _devOpsConfigurationClientDiagnostics, Pipeline, _devOpsConfigurationRestClient.CreateUpdateRequest(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, data).Request, response, OperationFinalStateVia.AzureAsyncOperation);
                if (waitUntil == WaitUntil.Completed)
                    operation.WaitForCompletion(cancellationToken);
                return operation;
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        /// <summary>
        /// Creates or updates a DevOps Configuration.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>DevOpsConfigurations_CreateOrUpdate</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="DevOpsConfigurationResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="waitUntil"> <see cref="WaitUntil.Completed"/> if the method should wait to return until the long-running operation has completed on the service; <see cref="WaitUntil.Started"/> if it should return after starting the operation. For more information on long-running operations, please see <see href="https://github.com/Azure/azure-sdk-for-net/blob/main/sdk/core/Azure.Core/samples/LongRunningOperations.md"> Azure.Core Long-Running Operation samples</see>. </param>
        /// <param name="data"> The DevOps configuration resource payload. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="data"/> is null. </exception>
        public virtual async Task<ArmOperation<DevOpsConfigurationResource>> CreateOrUpdateAsync(WaitUntil waitUntil, DevOpsConfigurationData data, CancellationToken cancellationToken = default)
        {
            Argument.AssertNotNull(data, nameof(data));

            using var scope = _devOpsConfigurationClientDiagnostics.CreateScope("DevOpsConfigurationResource.CreateOrUpdate");
            scope.Start();
            try
            {
                var response = await _devOpsConfigurationRestClient.CreateOrUpdateAsync(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, data, cancellationToken).ConfigureAwait(false);
                var operation = new SecurityCenterArmOperation<DevOpsConfigurationResource>(new DevOpsConfigurationOperationSource(Client), _devOpsConfigurationClientDiagnostics, Pipeline, _devOpsConfigurationRestClient.CreateCreateOrUpdateRequest(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, data).Request, response, OperationFinalStateVia.AzureAsyncOperation);
                if (waitUntil == WaitUntil.Completed)
                    await operation.WaitForCompletionAsync(cancellationToken).ConfigureAwait(false);
                return operation;
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        /// <summary>
        /// Creates or updates a DevOps Configuration.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>DevOpsConfigurations_CreateOrUpdate</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="DevOpsConfigurationResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="waitUntil"> <see cref="WaitUntil.Completed"/> if the method should wait to return until the long-running operation has completed on the service; <see cref="WaitUntil.Started"/> if it should return after starting the operation. For more information on long-running operations, please see <see href="https://github.com/Azure/azure-sdk-for-net/blob/main/sdk/core/Azure.Core/samples/LongRunningOperations.md"> Azure.Core Long-Running Operation samples</see>. </param>
        /// <param name="data"> The DevOps configuration resource payload. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="data"/> is null. </exception>
        public virtual ArmOperation<DevOpsConfigurationResource> CreateOrUpdate(WaitUntil waitUntil, DevOpsConfigurationData data, CancellationToken cancellationToken = default)
        {
            Argument.AssertNotNull(data, nameof(data));

            using var scope = _devOpsConfigurationClientDiagnostics.CreateScope("DevOpsConfigurationResource.CreateOrUpdate");
            scope.Start();
            try
            {
                var response = _devOpsConfigurationRestClient.CreateOrUpdate(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, data, cancellationToken);
                var operation = new SecurityCenterArmOperation<DevOpsConfigurationResource>(new DevOpsConfigurationOperationSource(Client), _devOpsConfigurationClientDiagnostics, Pipeline, _devOpsConfigurationRestClient.CreateCreateOrUpdateRequest(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, data).Request, response, OperationFinalStateVia.AzureAsyncOperation);
                if (waitUntil == WaitUntil.Completed)
                    operation.WaitForCompletion(cancellationToken);
                return operation;
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        /// <summary>
        /// Returns a list of all Azure DevOps organizations accessible by the user token consumed by the connector.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/listAvailableAzureDevOpsOrgs</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>AzureDevOpsOrgs_ListAvailable</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="DevOpsOrgResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <returns> An async collection of <see cref="DevOpsOrgResource"/> that may take multiple service requests to iterate over. </returns>
        public virtual AsyncPageable<DevOpsOrgResource> GetAvailableDevOpsOrgsAsync(CancellationToken cancellationToken = default)
        {
            HttpMessage FirstPageRequest(int? pageSizeHint) => _devOpsOrgAzureDevOpsOrgsRestClient.CreateListAvailableRequest(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name);
            return GeneratorPageableHelpers.CreateAsyncPageable(FirstPageRequest, null, e => new DevOpsOrgResource(Client, DevOpsOrgData.DeserializeDevOpsOrgData(e)), _devOpsOrgAzureDevOpsOrgsClientDiagnostics, Pipeline, "DevOpsConfigurationResource.GetAvailableDevOpsOrgs", "value", null, cancellationToken);
        }

        /// <summary>
        /// Returns a list of all Azure DevOps organizations accessible by the user token consumed by the connector.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/listAvailableAzureDevOpsOrgs</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>AzureDevOpsOrgs_ListAvailable</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="DevOpsOrgResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <returns> A collection of <see cref="DevOpsOrgResource"/> that may take multiple service requests to iterate over. </returns>
        public virtual Pageable<DevOpsOrgResource> GetAvailableDevOpsOrgs(CancellationToken cancellationToken = default)
        {
            HttpMessage FirstPageRequest(int? pageSizeHint) => _devOpsOrgAzureDevOpsOrgsRestClient.CreateListAvailableRequest(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name);
            return GeneratorPageableHelpers.CreatePageable(FirstPageRequest, null, e => new DevOpsOrgResource(Client, DevOpsOrgData.DeserializeDevOpsOrgData(e)), _devOpsOrgAzureDevOpsOrgsClientDiagnostics, Pipeline, "DevOpsConfigurationResource.GetAvailableDevOpsOrgs", "value", null, cancellationToken);
        }

        /// <summary>
        /// Returns a list of all GitHub owners accessible by the user token consumed by the connector.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/listAvailableGitHubOwners</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>GitHubOwners_ListAvailable</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="SecurityConnectorGitHubOwnerResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <returns> An async collection of <see cref="SecurityConnectorGitHubOwnerResource"/> that may take multiple service requests to iterate over. </returns>
        public virtual AsyncPageable<SecurityConnectorGitHubOwnerResource> GetAvailableGitHubOwnersAsync(CancellationToken cancellationToken = default)
        {
            HttpMessage FirstPageRequest(int? pageSizeHint) => _securityConnectorGitHubOwnerGitHubOwnersRestClient.CreateListAvailableRequest(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name);
            return GeneratorPageableHelpers.CreateAsyncPageable(FirstPageRequest, null, e => new SecurityConnectorGitHubOwnerResource(Client, SecurityConnectorGitHubOwnerData.DeserializeSecurityConnectorGitHubOwnerData(e)), _securityConnectorGitHubOwnerGitHubOwnersClientDiagnostics, Pipeline, "DevOpsConfigurationResource.GetAvailableGitHubOwners", "value", null, cancellationToken);
        }

        /// <summary>
        /// Returns a list of all GitHub owners accessible by the user token consumed by the connector.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/listAvailableGitHubOwners</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>GitHubOwners_ListAvailable</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="SecurityConnectorGitHubOwnerResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <returns> A collection of <see cref="SecurityConnectorGitHubOwnerResource"/> that may take multiple service requests to iterate over. </returns>
        public virtual Pageable<SecurityConnectorGitHubOwnerResource> GetAvailableGitHubOwners(CancellationToken cancellationToken = default)
        {
            HttpMessage FirstPageRequest(int? pageSizeHint) => _securityConnectorGitHubOwnerGitHubOwnersRestClient.CreateListAvailableRequest(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name);
            return GeneratorPageableHelpers.CreatePageable(FirstPageRequest, null, e => new SecurityConnectorGitHubOwnerResource(Client, SecurityConnectorGitHubOwnerData.DeserializeSecurityConnectorGitHubOwnerData(e)), _securityConnectorGitHubOwnerGitHubOwnersClientDiagnostics, Pipeline, "DevOpsConfigurationResource.GetAvailableGitHubOwners", "value", null, cancellationToken);
        }

        /// <summary>
        /// Returns a list of all GitLab groups accessible by the user token consumed by the connector.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/listAvailableGitLabGroups</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>GitLabGroups_ListAvailable</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="SecurityConnectorGitLabGroupResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <returns> An async collection of <see cref="SecurityConnectorGitLabGroupResource"/> that may take multiple service requests to iterate over. </returns>
        public virtual AsyncPageable<SecurityConnectorGitLabGroupResource> GetAvailableGitLabGroupsAsync(CancellationToken cancellationToken = default)
        {
            HttpMessage FirstPageRequest(int? pageSizeHint) => _securityConnectorGitLabGroupGitLabGroupsRestClient.CreateListAvailableRequest(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name);
            return GeneratorPageableHelpers.CreateAsyncPageable(FirstPageRequest, null, e => new SecurityConnectorGitLabGroupResource(Client, SecurityConnectorGitLabGroupData.DeserializeSecurityConnectorGitLabGroupData(e)), _securityConnectorGitLabGroupGitLabGroupsClientDiagnostics, Pipeline, "DevOpsConfigurationResource.GetAvailableGitLabGroups", "value", null, cancellationToken);
        }

        /// <summary>
        /// Returns a list of all GitLab groups accessible by the user token consumed by the connector.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/listAvailableGitLabGroups</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>GitLabGroups_ListAvailable</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// <item>
        /// <term>Resource</term>
        /// <description><see cref="SecurityConnectorGitLabGroupResource"/></description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <returns> A collection of <see cref="SecurityConnectorGitLabGroupResource"/> that may take multiple service requests to iterate over. </returns>
        public virtual Pageable<SecurityConnectorGitLabGroupResource> GetAvailableGitLabGroups(CancellationToken cancellationToken = default)
        {
            HttpMessage FirstPageRequest(int? pageSizeHint) => _securityConnectorGitLabGroupGitLabGroupsRestClient.CreateListAvailableRequest(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name);
            return GeneratorPageableHelpers.CreatePageable(FirstPageRequest, null, e => new SecurityConnectorGitLabGroupResource(Client, SecurityConnectorGitLabGroupData.DeserializeSecurityConnectorGitLabGroupData(e)), _securityConnectorGitLabGroupGitLabGroupsClientDiagnostics, Pipeline, "DevOpsConfigurationResource.GetAvailableGitLabGroups", "value", null, cancellationToken);
        }

        /// <summary>
        /// Get devops long running operation result.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/operationResults/{operationResultId}</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>DevOpsOperationResults_Get</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="operationResultId"> The operation result Id. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentException"> <paramref name="operationResultId"/> is an empty string, and was expected to be non-empty. </exception>
        /// <exception cref="ArgumentNullException"> <paramref name="operationResultId"/> is null. </exception>
        public virtual async Task<Response<OperationStatusResult>> GetDevOpsOperationResultAsync(string operationResultId, CancellationToken cancellationToken = default)
        {
            Argument.AssertNotNullOrEmpty(operationResultId, nameof(operationResultId));

            using var scope = _devOpsOperationResultsClientDiagnostics.CreateScope("DevOpsConfigurationResource.GetDevOpsOperationResult");
            scope.Start();
            try
            {
                var response = await _devOpsOperationResultsRestClient.GetAsync(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, operationResultId, cancellationToken).ConfigureAwait(false);
                return response;
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }

        /// <summary>
        /// Get devops long running operation result.
        /// <list type="bullet">
        /// <item>
        /// <term>Request Path</term>
        /// <description>/subscriptions/{subscriptionId}/resourceGroups/{resourceGroupName}/providers/Microsoft.Security/securityConnectors/{securityConnectorName}/devops/default/operationResults/{operationResultId}</description>
        /// </item>
        /// <item>
        /// <term>Operation Id</term>
        /// <description>DevOpsOperationResults_Get</description>
        /// </item>
        /// <item>
        /// <term>Default Api Version</term>
        /// <description>2023-09-01-preview</description>
        /// </item>
        /// </list>
        /// </summary>
        /// <param name="operationResultId"> The operation result Id. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentException"> <paramref name="operationResultId"/> is an empty string, and was expected to be non-empty. </exception>
        /// <exception cref="ArgumentNullException"> <paramref name="operationResultId"/> is null. </exception>
        public virtual Response<OperationStatusResult> GetDevOpsOperationResult(string operationResultId, CancellationToken cancellationToken = default)
        {
            Argument.AssertNotNullOrEmpty(operationResultId, nameof(operationResultId));

            using var scope = _devOpsOperationResultsClientDiagnostics.CreateScope("DevOpsConfigurationResource.GetDevOpsOperationResult");
            scope.Start();
            try
            {
                var response = _devOpsOperationResultsRestClient.Get(Id.SubscriptionId, Id.ResourceGroupName, Id.Parent.Name, operationResultId, cancellationToken);
                return response;
            }
            catch (Exception e)
            {
                scope.Failed(e);
                throw;
            }
        }
    }
}
