﻿// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

using System;
using System.ComponentModel;
using System.Globalization;
using Azure.Core;

namespace Azure.ResourceManager.Models
{
    /// <summary>
    /// A class representing SKU for resource.
    /// </summary>
    public sealed partial class ArmSku : IEquatable<ArmSku>
    {
        /// <summary>
        /// Compares this <see cref="ArmSku"/> instance with another object and determines if they are equals.
        /// </summary>
        /// <param name="other"> <see cref="ArmSku"/> object to compare. </param>
        /// <returns> True if they are equals, otherwise false. </returns>
        public bool Equals(ArmSku other)
        {
            if (other == null)
                return false;

            if (object.ReferenceEquals(this, other))
                return true;

            return string.Equals(Name, other.Name, StringComparison.InvariantCultureIgnoreCase) &&
                string.Equals(Family, other.Family, StringComparison.InvariantCultureIgnoreCase) &&
                string.Equals(Size, other.Size, StringComparison.InvariantCultureIgnoreCase) &&
                (Tier.HasValue ? Tier.Value.Equals(other.Tier) : !other.Tier.HasValue) &&
                long.Equals(Capacity, other.Capacity);
        }

        /// <inheritdoc/>
        [EditorBrowsable(EditorBrowsableState.Never)]
        public override bool Equals(object obj)
        {
            if (ReferenceEquals(this, obj))
            {
                return true;
            }

            if (ReferenceEquals(obj, null))
            {
                return false;
            }

            if (obj is not ArmSku other)
                return false;

            return Equals(other);
        }

        /// <inheritdoc/>
        [EditorBrowsable(EditorBrowsableState.Never)]
        public override int GetHashCode()
        {
            return HashCodeBuilder.Combine(
                Name?.ToLower(CultureInfo.InvariantCulture),
                Family?.ToLower(CultureInfo.InvariantCulture),
                Size?.ToLower(CultureInfo.InvariantCulture),
                Tier?.ToString().ToLower(CultureInfo.InvariantCulture),
                Capacity);
        }

        /// <summary>
        /// Compares this <see cref="ArmSku"/> instance with another object and determines if they are equals.
        /// </summary>
        /// <param name="left"> The sku on the left side of the operator. </param>
        /// <param name="right"> The sku on the right side of the operator. </param>
        /// <returns> True if they are equal, otherwise false. </returns>
        public static bool operator ==(ArmSku left, ArmSku right)
        {
            if (ReferenceEquals(left, null))
            {
                return ReferenceEquals(right, null);
            }

            return left.Equals(right);
        }

        /// <summary>
        /// Compares this <see cref="ArmSku"/> instance with another object and determines if they are not equal.
        /// </summary>
        /// <param name="left"> The sku on the left side of the operator. </param>
        /// <param name="right"> The sku on the right side of the operator. </param>
        /// <returns> True if they are not equal, otherwise false. </returns>
        public static bool operator !=(ArmSku left, ArmSku right)
        {
            return !(left == right);
        }
    }
}
