﻿// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

using System;
using System.Collections.Generic;
using System.Threading.Tasks;
using Azure.Core;
using Azure.Core.TestFramework;
using Azure.Identity;
using Azure.MixedReality.Authentication;
using Moq;
using NUnit.Framework;
using NUnit.Framework.Internal;

namespace Azure.MixedReality.ObjectAnchors.Conversion.Tests
{
    [TestFixture(true)]
    [TestFixture(false)]
    public class ObjectAnchorsConversionClientTests : ClientTestBase
    {
        private static string anyWorkingUriString = "https://sampleazurestorageurl.com";

        public ObjectAnchorsConversionClientTests(bool isAsync) : base(isAsync)
        {
        }

        public static IEnumerable<object[]> BadClientArgumentsTestData =>
            new List<object[]>
            {
                new object[] {
                    new Guid("43d00847-9b3d-4be4-8bcc-c6a8a3f2e45a"), "eastus2.azure.com",
                    new AccessToken("dummykey", new DateTimeOffset(new DateTime(3000,1,1))),
                    true
                },
                new object[] {
                    new Guid("43d00847-9b3d-4be4-8bcc-c6a8a3f2e45a"), null,
                    new AccessToken("dummykey", new DateTimeOffset(new DateTime(3000,1,1))),
                    false
                }
            };

        public static IEnumerable<object[]> BadFileTypeTestData =>
            new List<object[]>
            {
                new object[]
                {
                    AssetFileType.Obj,
                    true
                },
                new object[]
                {
                    new AssetFileType(".exe"),
                    false
                },
            };

        [Test]
        public void Constructor()
        {
            // Arrange
            Guid accountId = Guid.Parse("00000000-0000-0000-0000-000000000001");
            string accountDomain = "my.domain.com";
            AccessToken token = new AccessToken("dummykey", new DateTimeOffset(new DateTime(3000, 1, 1)));
            AzureKeyCredential keyCredential = new AzureKeyCredential("MyAccessKey");
            ObjectAnchorsConversionClientOptions options = new ObjectAnchorsConversionClientOptions();
            TokenCredential credential = new StaticAccessTokenCredential(token);

            // Act
            ObjectAnchorsConversionClient client = new ObjectAnchorsConversionClient(accountId, accountDomain, keyCredential);

            // Assert
            Assert.AreEqual(accountId, client.AccountId);
            Assert.AreEqual(accountDomain, client.AccountDomain);
            Assert.NotNull(client.SupportedAssetFileTypes);

            // Act and assert

            // new(Guid accountId, string accountDomain, AzureKeyCredential keyCredential)
            AssertArgumentException<ArgumentException>(nameof(accountId),
                () => new ObjectAnchorsConversionClient(default, accountDomain, keyCredential));
            AssertArgumentException<ArgumentNullException>(nameof(accountDomain),
                () => new ObjectAnchorsConversionClient(accountId, null!, keyCredential));
            AssertArgumentException<ArgumentException>(nameof(accountDomain),
                () => new ObjectAnchorsConversionClient(accountId, string.Empty, keyCredential));
            AssertArgumentException<ArgumentException>(nameof(accountDomain),
                () => new ObjectAnchorsConversionClient(accountId, " ", keyCredential));
            AssertArgumentException<ArgumentNullException>(nameof(keyCredential),
                () => new ObjectAnchorsConversionClient(accountId, accountDomain, null!));

            // new(Guid accountId, string accountDomain, AzureKeyCredential keyCredential, ObjectAnchorsConversionClientOptions options)
            AssertArgumentException<ArgumentException>(nameof(accountId),
                () => new ObjectAnchorsConversionClient(default, accountDomain, keyCredential, options));
            AssertArgumentException<ArgumentNullException>(nameof(accountDomain),
                () => new ObjectAnchorsConversionClient(accountId, null!, keyCredential, options));
            AssertArgumentException<ArgumentException>(nameof(accountDomain),
                () => new ObjectAnchorsConversionClient(accountId, string.Empty, keyCredential, options));
            AssertArgumentException<ArgumentException>(nameof(accountDomain),
                () => new ObjectAnchorsConversionClient(accountId, " ", keyCredential, options));
            AssertArgumentException<ArgumentNullException>(nameof(keyCredential),
                () => new ObjectAnchorsConversionClient(accountId, accountDomain, (AzureKeyCredential)null!, options));

            // new(Guid accountId, string accountDomain, AccessToken token, ObjectAnchorsConversionClientOptions options = null)
            AssertArgumentException<ArgumentException>(nameof(accountId),
                () => new ObjectAnchorsConversionClient(default, accountDomain, token, options));
            AssertArgumentException<ArgumentNullException>(nameof(accountDomain),
                () => new ObjectAnchorsConversionClient(accountId, null!, token, options));
            AssertArgumentException<ArgumentException>(nameof(accountDomain),
                () => new ObjectAnchorsConversionClient(accountId, string.Empty, token, options));
            AssertArgumentException<ArgumentException>(nameof(accountDomain),
                () => new ObjectAnchorsConversionClient(accountId, " ", token, options));

            // new(Guid accountId, string accountDomain, TokenCredential credential, ObjectAnchorsConversionClientOptions options = null)
            AssertArgumentException<ArgumentException>(nameof(accountId),
                () => new ObjectAnchorsConversionClient(default, accountDomain, credential, options));
            AssertArgumentException<ArgumentNullException>(nameof(accountDomain),
                () => new ObjectAnchorsConversionClient(accountId, null!, credential, options));
            AssertArgumentException<ArgumentException>(nameof(accountDomain),
                () => new ObjectAnchorsConversionClient(accountId, string.Empty, credential, options));
            AssertArgumentException<ArgumentException>(nameof(accountDomain),
                () => new ObjectAnchorsConversionClient(accountId, " ", credential, options));
            AssertArgumentException<ArgumentNullException>(nameof(credential),
                () => new ObjectAnchorsConversionClient(accountId, accountDomain, (TokenCredential)null!, options));
        }

        [Test]
        [TestCaseSource(nameof(BadClientArgumentsTestData))]
        public void BadClientArguments(Guid accountId, string accountDomain, AccessToken credential, bool shouldSucceed)
        {
            bool excepted = false;
            try
            {
                ObjectAnchorsConversionClient client = new ObjectAnchorsConversionClient(accountId, accountDomain, credential);
            }
            catch (ArgumentException)
            {
                excepted = true;
            }

            Assert.AreNotEqual(shouldSucceed, excepted);
        }

        [Test]
        [TestCaseSource(nameof(BadFileTypeTestData))]
        public async Task BadFileType(AssetFileType ft, bool passes)
        {
            bool exceptedWithUnsupportedFileType = !passes;
            try
            {
                ObjectAnchorsConversionClient client = new ObjectAnchorsConversionClient(Guid.NewGuid(), "eastus2.azure.com", new AccessToken("dummykey", new DateTimeOffset(new DateTime(3000, 1, 1))));
                await client.StartAssetConversionAsync(new AssetConversionOptions(new Uri(anyWorkingUriString), new AssetFileType(".exe"), new AssetConversionConfiguration(new System.Numerics.Vector3(0, 0, 1), 1, false)));
            }
            catch (AssetFileTypeNotSupportedException)
            {
                exceptedWithUnsupportedFileType = true;
            }
            catch (Exception)
            {
                // This is fine
            }

            Assert.True(exceptedWithUnsupportedFileType);
        }

        private static void AssertArgumentException<TException>(string argumentName, TestDelegate code)
            where TException : ArgumentException
        {
            TException exception = Assert.Throws<TException>(code);
            Assert.AreEqual(argumentName, exception.ParamName);
        }
    }
}
