﻿// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

using System;
using System.Threading.Tasks;
using Azure.AI.MetricsAdvisor.Administration;
using Azure.AI.MetricsAdvisor.Models;
using Azure.AI.MetricsAdvisor.Tests;
using NUnit.Framework;

namespace Azure.AI.MetricsAdvisor.Samples
{
    public partial class MetricsAdvisorSamples : MetricsAdvisorTestEnvironment
    {
        [Test]
        public async Task CreateAndDeleteHookAsync()
        {
            string endpoint = MetricsAdvisorUri;
            string subscriptionKey = MetricsAdvisorSubscriptionKey;
            string apiKey = MetricsAdvisorApiKey;
            var credential = new MetricsAdvisorKeyCredential(subscriptionKey, apiKey);

            var adminClient = new MetricsAdvisorAdministrationClient(new Uri(endpoint), credential);

            #region Snippet:CreateHookAsync
#if SNIPPET
            string hookName = "<hookName>";
#else
            string hookName = GetUniqueName();
#endif

            var emailHook = new EmailNotificationHook(hookName);

            emailHook.EmailsToAlert.Add("email1@sample.com");
            emailHook.EmailsToAlert.Add("email2@sample.com");

            Response<NotificationHook> response = await adminClient.CreateHookAsync(emailHook);

            NotificationHook createdHook = response.Value;

            Console.WriteLine($"Hook ID: {createdHook.Id}");
            #endregion

            // Delete the created hook to clean up the Metrics Advisor resource. Do not perform this
            // step if you intend to keep using the hook.

            await adminClient.DeleteHookAsync(createdHook.Id);
        }

        [Test]
        public async Task GetHookAsync()
        {
            string endpoint = MetricsAdvisorUri;
            string subscriptionKey = MetricsAdvisorSubscriptionKey;
            string apiKey = MetricsAdvisorApiKey;
            var credential = new MetricsAdvisorKeyCredential(subscriptionKey, apiKey);

            var adminClient = new MetricsAdvisorAdministrationClient(new Uri(endpoint), credential);

            string hookId = HookId;

            Response<NotificationHook> response = await adminClient.GetHookAsync(hookId);

            NotificationHook hook = response.Value;

            Console.WriteLine($"Hook name: {hook.Name}");
            Console.WriteLine($"Hook description: {hook.Description}");

            Console.WriteLine($"Hook administrators:");
            foreach (string admin in hook.Administrators)
            {
                Console.WriteLine($" - {admin}");
            }

            if (hook.HookKind == NotificationHookKind.Email)
            {
                EmailNotificationHook emailHook = hook as EmailNotificationHook;

                Console.WriteLine("Emails to alert:");
                foreach (string email in emailHook.EmailsToAlert)
                {
                    Console.WriteLine($" - {email}");
                }
            }
            else if (hook.HookKind == NotificationHookKind.Webhook)
            {
                WebNotificationHook webHook = hook as WebNotificationHook;

                Console.WriteLine($"Username: {webHook.Username}");
            }
        }

        [Test]
        public async Task UpdateHookAsync()
        {
            string endpoint = MetricsAdvisorUri;
            string subscriptionKey = MetricsAdvisorSubscriptionKey;
            string apiKey = MetricsAdvisorApiKey;
            var credential = new MetricsAdvisorKeyCredential(subscriptionKey, apiKey);

            var adminClient = new MetricsAdvisorAdministrationClient(new Uri(endpoint), credential);

            string hookId = HookId;

            Response<NotificationHook> response = await adminClient.GetHookAsync(hookId);
            NotificationHook hook = response.Value;

            string originalDescription = hook.Description;
            hook.Description = "This description was generated by a sample.";

            response = await adminClient.UpdateHookAsync(hook);
            NotificationHook updatedHook = response.Value;

            Console.WriteLine($"Updated description: {updatedHook.Description}");

            // Undo the changes to leave the hook unaltered. Skip this step if you intend to keep
            // the changes.

            hook.Description = originalDescription;
            await adminClient.UpdateHookAsync(hook);
        }

        [Test]
        public async Task GetHooksAsync()
        {
            string endpoint = MetricsAdvisorUri;
            string subscriptionKey = MetricsAdvisorSubscriptionKey;
            string apiKey = MetricsAdvisorApiKey;
            var credential = new MetricsAdvisorKeyCredential(subscriptionKey, apiKey);

            var adminClient = new MetricsAdvisorAdministrationClient(new Uri(endpoint), credential);

            var options = new GetHooksOptions()
            {
                MaxPageSize = 5
            };

            int hookCount = 0;

            await foreach (NotificationHook hook in adminClient.GetHooksAsync(options))
            {
                Console.WriteLine($"Hook ID: {hook.Id}");
                Console.WriteLine($"Name: {hook.Name}");
                Console.WriteLine($"Description: {hook.Description}");
                Console.WriteLine($"Type: {hook.HookKind}");
                Console.WriteLine();

                // Print at most 5 hooks.
                if (++hookCount >= 5)
                {
                    break;
                }
            }
        }
    }
}
