// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

// <auto-generated/>

#nullable disable

using System;
using System.Collections.Generic;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using Azure.Core;
using Azure.Core.Pipeline;
using Azure.Maps.Common;
using Azure.Maps.Weather.Models;

namespace Azure.Maps.Weather
{
    internal partial class WeatherRestClient
    {
        private readonly HttpPipeline _pipeline;
        private readonly Uri _endpoint;
        private readonly string _clientId;
        private readonly string _apiVersion;

        /// <summary> The ClientDiagnostics is used to provide tracing support for the client library. </summary>
        internal ClientDiagnostics ClientDiagnostics { get; }

        /// <summary> Initializes a new instance of WeatherRestClient. </summary>
        /// <param name="clientDiagnostics"> The handler for diagnostic messaging in the client. </param>
        /// <param name="pipeline"> The HTTP pipeline for sending and receiving REST requests and responses. </param>
        /// <param name="endpoint"> server parameter. </param>
        /// <param name="clientId"> Specifies which account is intended for usage in conjunction with the Microsoft Entra ID security model.  It represents a unique ID for the Azure Maps account and can be retrieved from the Azure Maps management  plane Account API. To use Microsoft Entra ID security in Azure Maps see the following [articles](https://aka.ms/amauthdetails) for guidance. </param>
        /// <param name="apiVersion"> Api Version. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="clientDiagnostics"/>, <paramref name="pipeline"/> or <paramref name="apiVersion"/> is null. </exception>
        public WeatherRestClient(ClientDiagnostics clientDiagnostics, HttpPipeline pipeline, Uri endpoint = null, string clientId = null, string apiVersion = "1.1")
        {
            ClientDiagnostics = clientDiagnostics ?? throw new ArgumentNullException(nameof(clientDiagnostics));
            _pipeline = pipeline ?? throw new ArgumentNullException(nameof(pipeline));
            _endpoint = endpoint ?? new Uri("https://atlas.microsoft.com");
            _clientId = clientId;
            _apiVersion = apiVersion ?? throw new ArgumentNullException(nameof(apiVersion));
        }

        internal HttpMessage CreateGetHourlyForecastRequest(JsonFormat format, IEnumerable<double> coordinates, WeatherDataUnit? unit, int? duration, string language)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/forecast/hourly/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            if (unit != null)
            {
                uri.AppendQuery("unit", unit.Value.ToString(), true);
            }
            if (duration != null)
            {
                uri.AppendQuery("duration", duration.Value, true);
            }
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get a detailed hourly weather forecast for up to 24 hours or a daily forecast for up to 10 days. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="duration">
        /// Time frame of the returned weather forecast. By default, the forecast data for next hour will be returned. Available values are
        ///   * `1` - Return forecast data for the next hour. Default value.
        ///   * `12` - Return hourly forecast for next 12 hours.
        ///   * `24` - Return hourly forecast for next 24 hours.
        ///   * `72` - Return hourly forecast for next 72 hours (3 days).
        ///   * `120` - Return hourly forecast for next 120 hours (5 days). Only available in S1 SKU.
        ///   * `240` - Return hourly forecast for next 240 hours (10 days). Only available in S1 SKU.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Hourly Forecast` API is an HTTP `GET` that Request detailed weather forecast by the hour for the next 1, 12, 24 (1 day), 72 (3 days), 120 (5 days), and 240 hours (10 days) for the given the given coordinate location.  The API returns details such as temperature, humidity, wind, precipitation, and ultraviolet (UV) index. For more information, see [Request hourly weather forecast data](/azure/azure-maps/how-to-request-weather-data#request-hourly-weather-forecast-data).
        ///
        /// If you are using the Gen1 S0 pricing tier, you can request hourly forecast for the next 1, 12, 24 hours (1 day), and 72 hours (3 days). If you are using Gen1 S1 or Gen2 pricing tier, you can also request hourly forecast for the next 120 (5 days) and 240 hours (10 days).
        /// </remarks>
        public async Task<Response<HourlyForecastResult>> GetHourlyForecastAsync(JsonFormat format, IEnumerable<double> coordinates, WeatherDataUnit? unit = null, int? duration = null, string language = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetHourlyForecastRequest(format, coordinates, unit, duration, language);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        HourlyForecastResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = HourlyForecastResult.DeserializeHourlyForecastResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get a detailed hourly weather forecast for up to 24 hours or a daily forecast for up to 10 days. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="duration">
        /// Time frame of the returned weather forecast. By default, the forecast data for next hour will be returned. Available values are
        ///   * `1` - Return forecast data for the next hour. Default value.
        ///   * `12` - Return hourly forecast for next 12 hours.
        ///   * `24` - Return hourly forecast for next 24 hours.
        ///   * `72` - Return hourly forecast for next 72 hours (3 days).
        ///   * `120` - Return hourly forecast for next 120 hours (5 days). Only available in S1 SKU.
        ///   * `240` - Return hourly forecast for next 240 hours (10 days). Only available in S1 SKU.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Hourly Forecast` API is an HTTP `GET` that Request detailed weather forecast by the hour for the next 1, 12, 24 (1 day), 72 (3 days), 120 (5 days), and 240 hours (10 days) for the given the given coordinate location.  The API returns details such as temperature, humidity, wind, precipitation, and ultraviolet (UV) index. For more information, see [Request hourly weather forecast data](/azure/azure-maps/how-to-request-weather-data#request-hourly-weather-forecast-data).
        ///
        /// If you are using the Gen1 S0 pricing tier, you can request hourly forecast for the next 1, 12, 24 hours (1 day), and 72 hours (3 days). If you are using Gen1 S1 or Gen2 pricing tier, you can also request hourly forecast for the next 120 (5 days) and 240 hours (10 days).
        /// </remarks>
        public Response<HourlyForecastResult> GetHourlyForecast(JsonFormat format, IEnumerable<double> coordinates, WeatherDataUnit? unit = null, int? duration = null, string language = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetHourlyForecastRequest(format, coordinates, unit, duration, language);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        HourlyForecastResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = HourlyForecastResult.DeserializeHourlyForecastResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetMinuteForecastRequest(JsonFormat format, IEnumerable<double> coordinates, int? interval, string language)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/forecast/minute/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            if (interval != null)
            {
                uri.AppendQuery("interval", interval.Value, true);
            }
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get a minute-by-minute forecast for the next 120 minutes in intervals of 1, 5 and 15 minutes. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="interval">
        /// Specifies time interval in minutes for the returned weather forecast. Supported values are
        ///   * `1` -  Retrieve forecast for 1-minute intervals. Returned by default.
        ///   * `5` - Retrieve forecasts for 5-minute intervals.
        ///   * `15` - Retrieve forecasts for 15-minute intervals.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Minute Forecast` API is an HTTP `GET` request that returns minute-by-minute forecasts for a given location for the next 120 minutes.  Users can request weather forecasts in intervals of 1, 5 and 15 minutes.  The response will include details such as the type of precipitation (including rain, snow, or a mixture of both), start time, and precipitation intensity value (dBZ). For more information, see [Request minute-by-minute weather forecast data](/azure/azure-maps/how-to-request-weather-data#request-minute-by-minute-weather-forecast-data).
        /// </remarks>
        public async Task<Response<MinuteForecastResult>> GetMinuteForecastAsync(JsonFormat format, IEnumerable<double> coordinates, int? interval = null, string language = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetMinuteForecastRequest(format, coordinates, interval, language);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        MinuteForecastResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = MinuteForecastResult.DeserializeMinuteForecastResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get a minute-by-minute forecast for the next 120 minutes in intervals of 1, 5 and 15 minutes. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="interval">
        /// Specifies time interval in minutes for the returned weather forecast. Supported values are
        ///   * `1` -  Retrieve forecast for 1-minute intervals. Returned by default.
        ///   * `5` - Retrieve forecasts for 5-minute intervals.
        ///   * `15` - Retrieve forecasts for 15-minute intervals.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Minute Forecast` API is an HTTP `GET` request that returns minute-by-minute forecasts for a given location for the next 120 minutes.  Users can request weather forecasts in intervals of 1, 5 and 15 minutes.  The response will include details such as the type of precipitation (including rain, snow, or a mixture of both), start time, and precipitation intensity value (dBZ). For more information, see [Request minute-by-minute weather forecast data](/azure/azure-maps/how-to-request-weather-data#request-minute-by-minute-weather-forecast-data).
        /// </remarks>
        public Response<MinuteForecastResult> GetMinuteForecast(JsonFormat format, IEnumerable<double> coordinates, int? interval = null, string language = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetMinuteForecastRequest(format, coordinates, interval, language);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        MinuteForecastResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = MinuteForecastResult.DeserializeMinuteForecastResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetQuarterDayForecastRequest(JsonFormat format, IEnumerable<double> coordinates, WeatherDataUnit? unit, int? duration, string language)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/forecast/quarterDay/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            if (unit != null)
            {
                uri.AppendQuery("unit", unit.Value.ToString(), true);
            }
            if (duration != null)
            {
                uri.AppendQuery("duration", duration.Value, true);
            }
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get a Quarter-Day Forecast for the next 1, 5, 10, or 15 days. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="duration">
        /// Specifies for how many days the quester-day forecast responses are returned. Supported values are:
        ///   * `1` - Return forecast data for the next day. Returned by default.
        ///   * `5` - Return forecast data for the next 5 days.
        ///   * `10` - Return forecast data for next 10 days.
        ///   * `15` - Return forecast data for the next 15 days.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Quarter-Day Forecast` API is an HTTP `GET` request that returns a detailed weather forecast by quarter-day for the next 1, 5, 10, or 15 days for a given location. Response data is presented by quarters of the day - morning, afternoon, evening, and overnight. Details such as temperature, humidity, wind, precipitation, and UV index are returned.
        /// </remarks>
        public async Task<Response<QuarterDayForecastResult>> GetQuarterDayForecastAsync(JsonFormat format, IEnumerable<double> coordinates, WeatherDataUnit? unit = null, int? duration = null, string language = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetQuarterDayForecastRequest(format, coordinates, unit, duration, language);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        QuarterDayForecastResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = QuarterDayForecastResult.DeserializeQuarterDayForecastResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get a Quarter-Day Forecast for the next 1, 5, 10, or 15 days. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="duration">
        /// Specifies for how many days the quester-day forecast responses are returned. Supported values are:
        ///   * `1` - Return forecast data for the next day. Returned by default.
        ///   * `5` - Return forecast data for the next 5 days.
        ///   * `10` - Return forecast data for next 10 days.
        ///   * `15` - Return forecast data for the next 15 days.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Quarter-Day Forecast` API is an HTTP `GET` request that returns a detailed weather forecast by quarter-day for the next 1, 5, 10, or 15 days for a given location. Response data is presented by quarters of the day - morning, afternoon, evening, and overnight. Details such as temperature, humidity, wind, precipitation, and UV index are returned.
        /// </remarks>
        public Response<QuarterDayForecastResult> GetQuarterDayForecast(JsonFormat format, IEnumerable<double> coordinates, WeatherDataUnit? unit = null, int? duration = null, string language = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetQuarterDayForecastRequest(format, coordinates, unit, duration, language);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        QuarterDayForecastResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = QuarterDayForecastResult.DeserializeQuarterDayForecastResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetCurrentConditionsRequest(JsonFormat format, IEnumerable<double> coordinates, WeatherDataUnit? unit, string details, int? duration, string language)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/currentConditions/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            if (unit != null)
            {
                uri.AppendQuery("unit", unit.Value.ToString(), true);
            }
            if (details != null)
            {
                uri.AppendQuery("details", details, true);
            }
            if (duration != null)
            {
                uri.AppendQuery("duration", duration.Value, true);
            }
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get current weather conditions. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="details">
        /// Return full details for the current conditions. Available values are
        ///   * `true` - Returns full details. By default all details are returned.
        ///   * `false` - Returns a truncated version of the current condition data, which includes observation date time, weather phrase, icon code, precipitation indicator flag, and temperature.
        /// </param>
        /// <param name="duration">
        /// Time frame of the returned weather conditions. By default, the most current weather conditions will be returned. Default value is 0. Supported values are:
        ///    * `0` - Return the most current weather conditions.
        ///    * `6` - Return weather conditions from past 6 hours.
        ///    * `24` - Return weather conditions from past 24 hours.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Current Conditions` API is an HTTP `GET` request that returns detailed current weather conditions such as precipitation, temperature and wind for a given coordinate location. Also, observations from the past 6 or 24 hours for a particular location can be retrieved. The basic information returned with The response includes details such as observation date and time, brief description of the weather conditions, weather icon, precipitation indicator flags, and temperature. Additional details such as RealFeel™ Temperature and UV index are also returned. For more information, see [Request real-time weather data](/azure/azure-maps/how-to-request-weather-data#request-real-time-weather-data)
        /// </remarks>
        public async Task<Response<CurrentConditionsResult>> GetCurrentConditionsAsync(JsonFormat format, IEnumerable<double> coordinates, WeatherDataUnit? unit = null, string details = null, int? duration = null, string language = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetCurrentConditionsRequest(format, coordinates, unit, details, duration, language);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        CurrentConditionsResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = CurrentConditionsResult.DeserializeCurrentConditionsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get current weather conditions. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="details">
        /// Return full details for the current conditions. Available values are
        ///   * `true` - Returns full details. By default all details are returned.
        ///   * `false` - Returns a truncated version of the current condition data, which includes observation date time, weather phrase, icon code, precipitation indicator flag, and temperature.
        /// </param>
        /// <param name="duration">
        /// Time frame of the returned weather conditions. By default, the most current weather conditions will be returned. Default value is 0. Supported values are:
        ///    * `0` - Return the most current weather conditions.
        ///    * `6` - Return weather conditions from past 6 hours.
        ///    * `24` - Return weather conditions from past 24 hours.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Current Conditions` API is an HTTP `GET` request that returns detailed current weather conditions such as precipitation, temperature and wind for a given coordinate location. Also, observations from the past 6 or 24 hours for a particular location can be retrieved. The basic information returned with The response includes details such as observation date and time, brief description of the weather conditions, weather icon, precipitation indicator flags, and temperature. Additional details such as RealFeel™ Temperature and UV index are also returned. For more information, see [Request real-time weather data](/azure/azure-maps/how-to-request-weather-data#request-real-time-weather-data)
        /// </remarks>
        public Response<CurrentConditionsResult> GetCurrentConditions(JsonFormat format, IEnumerable<double> coordinates, WeatherDataUnit? unit = null, string details = null, int? duration = null, string language = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetCurrentConditionsRequest(format, coordinates, unit, details, duration, language);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        CurrentConditionsResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = CurrentConditionsResult.DeserializeCurrentConditionsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetDailyForecastRequest(JsonFormat format, IEnumerable<double> coordinates, WeatherDataUnit? unit, int? duration, string language)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/forecast/daily/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            if (unit != null)
            {
                uri.AppendQuery("unit", unit.Value.ToString(), true);
            }
            if (duration != null)
            {
                uri.AppendQuery("duration", duration.Value, true);
            }
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get a daily detailed weather forecast for the next 1, 5, 10, 15, 25, or 45 days. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="duration">
        /// Specifies for how many days the daily forecast responses are returned. Available values are
        ///   * `1` - Return forecast data for the next day. Returned by default.
        ///   * `5` - Return forecast data for the next 5 days.
        ///   * `10` - Return forecast data for the next 10 days.
        ///   * `25` - Return forecast data for the next 25 days. Only available in S1 SKU.
        ///   * `45` - Return forecast data for the next 45 days. Only available in S1 SKU.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Daily Forecast` API is an HTTP `GET` request that returns detailed weather forecast such as temperature and wind by day for the next 1, 5, 10, 15, 25, or 45 days for a given coordinate location.  The response includes details such as temperature, wind, precipitation, air quality, and UV index. For more information, see [Request daily weather forecast data](/azure/azure-maps/how-to-request-weather-data#request-daily-weather-forecast-data).
        ///
        ///  If you are using the Gen1 S0 pricing tier, you can request daily forecast for the next 1, 5, 10, and 15 days. If you are using Gen1 S1 or Gen2 pricing tier, you can also request daily forecast for the next 25 days, and 45 days.
        /// </remarks>
        public async Task<Response<DailyForecastResult>> GetDailyForecastAsync(JsonFormat format, IEnumerable<double> coordinates, WeatherDataUnit? unit = null, int? duration = null, string language = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetDailyForecastRequest(format, coordinates, unit, duration, language);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DailyForecastResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DailyForecastResult.DeserializeDailyForecastResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get a daily detailed weather forecast for the next 1, 5, 10, 15, 25, or 45 days. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="duration">
        /// Specifies for how many days the daily forecast responses are returned. Available values are
        ///   * `1` - Return forecast data for the next day. Returned by default.
        ///   * `5` - Return forecast data for the next 5 days.
        ///   * `10` - Return forecast data for the next 10 days.
        ///   * `25` - Return forecast data for the next 25 days. Only available in S1 SKU.
        ///   * `45` - Return forecast data for the next 45 days. Only available in S1 SKU.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Daily Forecast` API is an HTTP `GET` request that returns detailed weather forecast such as temperature and wind by day for the next 1, 5, 10, 15, 25, or 45 days for a given coordinate location.  The response includes details such as temperature, wind, precipitation, air quality, and UV index. For more information, see [Request daily weather forecast data](/azure/azure-maps/how-to-request-weather-data#request-daily-weather-forecast-data).
        ///
        ///  If you are using the Gen1 S0 pricing tier, you can request daily forecast for the next 1, 5, 10, and 15 days. If you are using Gen1 S1 or Gen2 pricing tier, you can also request daily forecast for the next 25 days, and 45 days.
        /// </remarks>
        public Response<DailyForecastResult> GetDailyForecast(JsonFormat format, IEnumerable<double> coordinates, WeatherDataUnit? unit = null, int? duration = null, string language = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetDailyForecastRequest(format, coordinates, unit, duration, language);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DailyForecastResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DailyForecastResult.DeserializeDailyForecastResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetWeatherAlongRouteRequest(JsonFormat format, string query, string language)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/route/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            uri.AppendQuery("query", query, true);
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get a locationally precise, up-to-the-minute forecast that includes weather hazard assessments and notifications along a route. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="query">
        /// Coordinates through which the route is calculated, separated by colon (:) and entered in chronological order. A minimum of two waypoints is required. A single API call may contain up to 60 waypoints.
        /// A waypoint indicates location, ETA, and optional heading: latitude,longitude,ETA,heading, where
        ///   * `Latitude` - Latitude coordinate in decimal degrees.
        ///   * `Longitude` - Longitude coordinate in decimal degrees.
        ///   * `ETA (estimated time of arrival)` - The number of minutes from the present time that it will take for the vehicle to reach the waypoint. Allowed range is from 0.0 to 120.0 minutes.
        ///   * `Heading` - An optional value indicating the vehicle heading as it passes the waypoint. Expressed in clockwise degrees relative to true north. This is issued to calculate sun glare as a driving hazard. Allowed range is from 0.0 to 360.0 degrees. If not provided, a heading will automatically be derived based on the position of neighboring waypoints.
        ///
        /// It is recommended to stay within, or close to, the distance that can be traveled within 120-mins or shortly after. This way a more accurate assessment can be provided for the trip and prevent isolated events not being captured between waypoints.  Information can and should be updated along the route (especially for trips greater than 2 hours) to continuously pull new waypoints moving forward, but also to ensure that forecast information for content such as precipitation type and intensity is accurate as storms develop and dissipate over time.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="query"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Weather Along Route` API is an HTTP `GET` request that returns hyper local (one kilometer or less), up-to-the-minute weather nowcasts, weather hazard assessments, and notifications along a route described as a sequence of waypoints.
        ///  This includes a list of weather hazards affecting the waypoint or route, and the aggregated hazard index for each waypoint might be used to paint each portion of a route according to how safe it is for the driver. When submitting the waypoints, it is recommended to stay within, or close to, the distance that can be traveled within 120-mins or shortly after. Data is updated every five minutes.
        ///
        ///  The service supplements Azure Maps [Route Service](https://docs.microsoft.com/rest/api/maps/route) that allows you to first request a route between an origin and a destination and use that as an input for Weather Along Route endpoint.
        ///
        ///  In addition, the service supports scenarios to generate weather notifications for waypoints that experience an increase in intensity of a weather hazard. For example, if the vehicle is expected to begin experiencing heavy rain as it reaches a waypoint, a weather notification for heavy rain will be generated for that waypoint allowing the end product to display a heavy rain notification before the driver reaches that waypoint.
        ///  The trigger for when to display the notification for a waypoint could be based, for example, on a [geofence](https://docs.microsoft.com/azure/azure-maps/tutorial-iot-hub-maps), or selectable distance to the waypoint.
        ///
        ///  The API covers all regions of the planet except latitudes above Greenland and Antarctica.
        /// </remarks>
        public async Task<Response<WeatherAlongRouteResult>> GetWeatherAlongRouteAsync(JsonFormat format, string query, string language = null, CancellationToken cancellationToken = default)
        {
            if (query == null)
            {
                throw new ArgumentNullException(nameof(query));
            }

            using var message = CreateGetWeatherAlongRouteRequest(format, query, language);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        WeatherAlongRouteResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = WeatherAlongRouteResult.DeserializeWeatherAlongRouteResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get a locationally precise, up-to-the-minute forecast that includes weather hazard assessments and notifications along a route. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="query">
        /// Coordinates through which the route is calculated, separated by colon (:) and entered in chronological order. A minimum of two waypoints is required. A single API call may contain up to 60 waypoints.
        /// A waypoint indicates location, ETA, and optional heading: latitude,longitude,ETA,heading, where
        ///   * `Latitude` - Latitude coordinate in decimal degrees.
        ///   * `Longitude` - Longitude coordinate in decimal degrees.
        ///   * `ETA (estimated time of arrival)` - The number of minutes from the present time that it will take for the vehicle to reach the waypoint. Allowed range is from 0.0 to 120.0 minutes.
        ///   * `Heading` - An optional value indicating the vehicle heading as it passes the waypoint. Expressed in clockwise degrees relative to true north. This is issued to calculate sun glare as a driving hazard. Allowed range is from 0.0 to 360.0 degrees. If not provided, a heading will automatically be derived based on the position of neighboring waypoints.
        ///
        /// It is recommended to stay within, or close to, the distance that can be traveled within 120-mins or shortly after. This way a more accurate assessment can be provided for the trip and prevent isolated events not being captured between waypoints.  Information can and should be updated along the route (especially for trips greater than 2 hours) to continuously pull new waypoints moving forward, but also to ensure that forecast information for content such as precipitation type and intensity is accurate as storms develop and dissipate over time.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="query"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Weather Along Route` API is an HTTP `GET` request that returns hyper local (one kilometer or less), up-to-the-minute weather nowcasts, weather hazard assessments, and notifications along a route described as a sequence of waypoints.
        ///  This includes a list of weather hazards affecting the waypoint or route, and the aggregated hazard index for each waypoint might be used to paint each portion of a route according to how safe it is for the driver. When submitting the waypoints, it is recommended to stay within, or close to, the distance that can be traveled within 120-mins or shortly after. Data is updated every five minutes.
        ///
        ///  The service supplements Azure Maps [Route Service](https://docs.microsoft.com/rest/api/maps/route) that allows you to first request a route between an origin and a destination and use that as an input for Weather Along Route endpoint.
        ///
        ///  In addition, the service supports scenarios to generate weather notifications for waypoints that experience an increase in intensity of a weather hazard. For example, if the vehicle is expected to begin experiencing heavy rain as it reaches a waypoint, a weather notification for heavy rain will be generated for that waypoint allowing the end product to display a heavy rain notification before the driver reaches that waypoint.
        ///  The trigger for when to display the notification for a waypoint could be based, for example, on a [geofence](https://docs.microsoft.com/azure/azure-maps/tutorial-iot-hub-maps), or selectable distance to the waypoint.
        ///
        ///  The API covers all regions of the planet except latitudes above Greenland and Antarctica.
        /// </remarks>
        public Response<WeatherAlongRouteResult> GetWeatherAlongRoute(JsonFormat format, string query, string language = null, CancellationToken cancellationToken = default)
        {
            if (query == null)
            {
                throw new ArgumentNullException(nameof(query));
            }

            using var message = CreateGetWeatherAlongRouteRequest(format, query, language);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        WeatherAlongRouteResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = WeatherAlongRouteResult.DeserializeWeatherAlongRouteResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetSevereWeatherAlertsRequest(JsonFormat format, IEnumerable<double> coordinates, string language, string details)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/severe/alerts/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            if (details != null)
            {
                uri.AppendQuery("details", details, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get information about severe weather conditions such as hurricanes, thunderstorms, flooding, lightning, heat waves or forest fires for a given location. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="details">
        /// Return full details for the severe weather alerts. Available values are
        ///   * `true` - Returns full details. By default all details are returned.
        ///   * `false` - Returns a truncated version of the alerts data, which excludes the area-specific full description of alert details (`alertDetails`).
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// Severe weather phenomenon can significantly impact our everyday life and business operations. For example, severe weather conditions such as tropical storms, high winds or flooding can close roads and force logistics companies to reroute their fleet causing delays in reaching destinations and breaking the cold chain of refrigerated food products.
        ///
        ///  The `Get Severe Weather Alerts` API is an HTTP `GET` request that returns the severe weather alerts that are available worldwide from both official Government Meteorological Agencies and leading global to regional weather alert providers. The service can return details such as alert type, category, level and detailed description about the active severe alerts for the requested location, like hurricanes, thunderstorms, lightning, heat waves or forest fires. For more information, see [Request severe weather alerts](/azure-maps/how-to-request-weather-data#request-severe-weather-alerts)
        /// </remarks>
        public async Task<Response<SevereWeatherAlertsResult>> GetSevereWeatherAlertsAsync(JsonFormat format, IEnumerable<double> coordinates, string language = null, string details = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetSevereWeatherAlertsRequest(format, coordinates, language, details);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SevereWeatherAlertsResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = SevereWeatherAlertsResult.DeserializeSevereWeatherAlertsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get information about severe weather conditions such as hurricanes, thunderstorms, flooding, lightning, heat waves or forest fires for a given location. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="details">
        /// Return full details for the severe weather alerts. Available values are
        ///   * `true` - Returns full details. By default all details are returned.
        ///   * `false` - Returns a truncated version of the alerts data, which excludes the area-specific full description of alert details (`alertDetails`).
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// Severe weather phenomenon can significantly impact our everyday life and business operations. For example, severe weather conditions such as tropical storms, high winds or flooding can close roads and force logistics companies to reroute their fleet causing delays in reaching destinations and breaking the cold chain of refrigerated food products.
        ///
        ///  The `Get Severe Weather Alerts` API is an HTTP `GET` request that returns the severe weather alerts that are available worldwide from both official Government Meteorological Agencies and leading global to regional weather alert providers. The service can return details such as alert type, category, level and detailed description about the active severe alerts for the requested location, like hurricanes, thunderstorms, lightning, heat waves or forest fires. For more information, see [Request severe weather alerts](/azure-maps/how-to-request-weather-data#request-severe-weather-alerts)
        /// </remarks>
        public Response<SevereWeatherAlertsResult> GetSevereWeatherAlerts(JsonFormat format, IEnumerable<double> coordinates, string language = null, string details = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetSevereWeatherAlertsRequest(format, coordinates, language, details);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SevereWeatherAlertsResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = SevereWeatherAlertsResult.DeserializeSevereWeatherAlertsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetDailyIndicesRequest(JsonFormat format, IEnumerable<double> coordinates, string language, int? duration, int? indexId, int? indexGroupId)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/indices/daily/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            if (duration != null)
            {
                uri.AppendQuery("duration", duration.Value, true);
            }
            if (indexId != null)
            {
                uri.AppendQuery("indexId", indexId.Value, true);
            }
            if (indexGroupId != null)
            {
                uri.AppendQuery("indexGroupId", indexGroupId.Value, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use when you want to know if the weather conditions are optimal for a specific activity such as outdoor sporting activities, construction, or farming (results includes soil moisture information). </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="duration">
        /// Specifies for how many days the daily indices are returned. By default, the indices data for the current day will be returned. When requesting future indices data, the current day is included in the response as day 1. Available values are
        ///   * `1` - Return daily index data for the current day. Default value.
        ///   * `5` - Return 5 days of daily index data starting from the current day.
        ///   * `10` - Return 10 days of daily index data starting from the current day.
        ///   * `15` - Return 15 days of daily index data starting from the current day.
        /// </param>
        /// <param name="indexId"> Numeric index identifier that can be used for restricting returned results to the corresponding index type. Cannot be paired with `indexGroupId`. Please refer to [Weather services in Azure Maps](/azure/azure-maps/weather-services-concepts#index-ids-and-index-groups-ids) for details and to see the supported indices. </param>
        /// <param name="indexGroupId"> Numeric index group identifier that can be used for restricting returned results to the corresponding subset of indices (index group). Cannot be paired with `indexId`. Please refer to [Weather services in Azure Maps](/azure/azure-maps/weather-services-concepts#index-ids-and-index-groups-ids) for details and to see the supported index groups. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Daily Indices` API is an HTTP `GET` request returns index values that provide guidance to help when planning future activities. For example, a health mobile application can notify users that today is good weather for running or for other outdoors activities like playing golf or flying a kite. Retail stores can optimize their digital marketing campaigns based on predicted index values. The service returns in daily indices values for current and next 5, 10 and 15 days starting from current day.
        /// </remarks>
        public async Task<Response<DailyIndicesResult>> GetDailyIndicesAsync(JsonFormat format, IEnumerable<double> coordinates, string language = null, int? duration = null, int? indexId = null, int? indexGroupId = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetDailyIndicesRequest(format, coordinates, language, duration, indexId, indexGroupId);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DailyIndicesResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DailyIndicesResult.DeserializeDailyIndicesResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use when you want to know if the weather conditions are optimal for a specific activity such as outdoor sporting activities, construction, or farming (results includes soil moisture information). </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="duration">
        /// Specifies for how many days the daily indices are returned. By default, the indices data for the current day will be returned. When requesting future indices data, the current day is included in the response as day 1. Available values are
        ///   * `1` - Return daily index data for the current day. Default value.
        ///   * `5` - Return 5 days of daily index data starting from the current day.
        ///   * `10` - Return 10 days of daily index data starting from the current day.
        ///   * `15` - Return 15 days of daily index data starting from the current day.
        /// </param>
        /// <param name="indexId"> Numeric index identifier that can be used for restricting returned results to the corresponding index type. Cannot be paired with `indexGroupId`. Please refer to [Weather services in Azure Maps](/azure/azure-maps/weather-services-concepts#index-ids-and-index-groups-ids) for details and to see the supported indices. </param>
        /// <param name="indexGroupId"> Numeric index group identifier that can be used for restricting returned results to the corresponding subset of indices (index group). Cannot be paired with `indexId`. Please refer to [Weather services in Azure Maps](/azure/azure-maps/weather-services-concepts#index-ids-and-index-groups-ids) for details and to see the supported index groups. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Daily Indices` API is an HTTP `GET` request returns index values that provide guidance to help when planning future activities. For example, a health mobile application can notify users that today is good weather for running or for other outdoors activities like playing golf or flying a kite. Retail stores can optimize their digital marketing campaigns based on predicted index values. The service returns in daily indices values for current and next 5, 10 and 15 days starting from current day.
        /// </remarks>
        public Response<DailyIndicesResult> GetDailyIndices(JsonFormat format, IEnumerable<double> coordinates, string language = null, int? duration = null, int? indexId = null, int? indexGroupId = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetDailyIndicesRequest(format, coordinates, language, duration, indexId, indexGroupId);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DailyIndicesResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DailyIndicesResult.DeserializeDailyIndicesResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetTropicalStormActiveRequest(JsonFormat format)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/tropical/storms/active/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get a list of the active tropical storms issued by national weather forecasting agencies. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// The `Get Tropical Storm Active` API is an HTTP `GET` request that returns a list of all government-issued active tropical storms. Information about the tropical storms includes, government ID, basin ID, year of origin, name and if it is subtropical.
        /// </remarks>
        public async Task<Response<ActiveStormResult>> GetTropicalStormActiveAsync(JsonFormat format, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetTropicalStormActiveRequest(format);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ActiveStormResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ActiveStormResult.DeserializeActiveStormResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get a list of the active tropical storms issued by national weather forecasting agencies. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// The `Get Tropical Storm Active` API is an HTTP `GET` request that returns a list of all government-issued active tropical storms. Information about the tropical storms includes, government ID, basin ID, year of origin, name and if it is subtropical.
        /// </remarks>
        public Response<ActiveStormResult> GetTropicalStormActive(JsonFormat format, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetTropicalStormActiveRequest(format);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ActiveStormResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ActiveStormResult.DeserializeActiveStormResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateSearchTropicalStormRequest(JsonFormat format, int year, BasinId? basinId, int? governmentStormId)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/tropical/storms/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            uri.AppendQuery("year", year, true);
            if (basinId != null)
            {
                uri.AppendQuery("basinId", basinId.Value.ToString(), true);
            }
            if (governmentStormId != null)
            {
                uri.AppendQuery("govId", governmentStormId.Value, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get a list of storms issued by national weather forecasting agencies. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="year"> Year of the cyclone(s). </param>
        /// <param name="basinId"> Basin identifier. </param>
        /// <param name="governmentStormId"> Government storm Id. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// The `Get Tropical Storm Search` API is an HTTP `GET` request that returns a list of government-issued tropical storms by year, basin ID, and government ID. Information about the tropical storms includes, government ID, basin ID, status, year, name and if it is subtropical.
        /// </remarks>
        public async Task<Response<StormSearchResult>> SearchTropicalStormAsync(JsonFormat format, int year, BasinId? basinId = null, int? governmentStormId = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateSearchTropicalStormRequest(format, year, basinId, governmentStormId);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StormSearchResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = StormSearchResult.DeserializeStormSearchResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get a list of storms issued by national weather forecasting agencies. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="year"> Year of the cyclone(s). </param>
        /// <param name="basinId"> Basin identifier. </param>
        /// <param name="governmentStormId"> Government storm Id. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// The `Get Tropical Storm Search` API is an HTTP `GET` request that returns a list of government-issued tropical storms by year, basin ID, and government ID. Information about the tropical storms includes, government ID, basin ID, status, year, name and if it is subtropical.
        /// </remarks>
        public Response<StormSearchResult> SearchTropicalStorm(JsonFormat format, int year, BasinId? basinId = null, int? governmentStormId = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateSearchTropicalStormRequest(format, year, basinId, governmentStormId);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StormSearchResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = StormSearchResult.DeserializeStormSearchResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetTropicalStormForecastRequest(JsonFormat format, int year, BasinId basinId, int governmentStormId, WeatherDataUnit? unit, bool? includeDetails, bool? includeGeometricDetails, bool? includeWindowGeometry)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/tropical/storms/forecasts/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            uri.AppendQuery("year", year, true);
            uri.AppendQuery("basinId", basinId.ToString(), true);
            uri.AppendQuery("govId", governmentStormId, true);
            if (unit != null)
            {
                uri.AppendQuery("unit", unit.Value.ToString(), true);
            }
            if (includeDetails != null)
            {
                uri.AppendQuery("details", includeDetails.Value, true);
            }
            if (includeGeometricDetails != null)
            {
                uri.AppendQuery("radiiGeometry", includeGeometricDetails.Value, true);
            }
            if (includeWindowGeometry != null)
            {
                uri.AppendQuery("windowGeometry", includeWindowGeometry.Value, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get a list of tropical storms forecasted by national weather forecasting agencies. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="year"> Year of the cyclone(s). </param>
        /// <param name="basinId"> Basin identifier. </param>
        /// <param name="governmentStormId"> Government storm Id. </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="includeDetails"> When true, wind radii summary data is included in the response. </param>
        /// <param name="includeGeometricDetails"> When true, wind radii summary data and geoJSON details are included in the response. </param>
        /// <param name="includeWindowGeometry"> When true, window geometry data (geoJSON) is included in the response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// The `Get Tropical Storm Forecasts` API is an HTTP `GET` request that returns individual government-issued tropical storm forecasts. Information about the forecasted tropical storms includes, location, status, date the forecast was created, window, wind speed and wind radii.
        /// </remarks>
        public async Task<Response<StormForecastResult>> GetTropicalStormForecastAsync(JsonFormat format, int year, BasinId basinId, int governmentStormId, WeatherDataUnit? unit = null, bool? includeDetails = null, bool? includeGeometricDetails = null, bool? includeWindowGeometry = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetTropicalStormForecastRequest(format, year, basinId, governmentStormId, unit, includeDetails, includeGeometricDetails, includeWindowGeometry);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StormForecastResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = StormForecastResult.DeserializeStormForecastResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get a list of tropical storms forecasted by national weather forecasting agencies. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="year"> Year of the cyclone(s). </param>
        /// <param name="basinId"> Basin identifier. </param>
        /// <param name="governmentStormId"> Government storm Id. </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="includeDetails"> When true, wind radii summary data is included in the response. </param>
        /// <param name="includeGeometricDetails"> When true, wind radii summary data and geoJSON details are included in the response. </param>
        /// <param name="includeWindowGeometry"> When true, window geometry data (geoJSON) is included in the response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// The `Get Tropical Storm Forecasts` API is an HTTP `GET` request that returns individual government-issued tropical storm forecasts. Information about the forecasted tropical storms includes, location, status, date the forecast was created, window, wind speed and wind radii.
        /// </remarks>
        public Response<StormForecastResult> GetTropicalStormForecast(JsonFormat format, int year, BasinId basinId, int governmentStormId, WeatherDataUnit? unit = null, bool? includeDetails = null, bool? includeGeometricDetails = null, bool? includeWindowGeometry = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetTropicalStormForecastRequest(format, year, basinId, governmentStormId, unit, includeDetails, includeGeometricDetails, includeWindowGeometry);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StormForecastResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = StormForecastResult.DeserializeStormForecastResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetTropicalStormLocationsRequest(JsonFormat format, int year, BasinId basinId, int governmentStormId, bool? includeDetails, bool? includeGeometricDetails, WeatherDataUnit? unit, bool? includeCurrentStorm)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/tropical/storms/locations/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            uri.AppendQuery("year", year, true);
            uri.AppendQuery("basinId", basinId.ToString(), true);
            uri.AppendQuery("govId", governmentStormId, true);
            if (includeDetails != null)
            {
                uri.AppendQuery("details", includeDetails.Value, true);
            }
            if (includeGeometricDetails != null)
            {
                uri.AppendQuery("radiiGeometry", includeGeometricDetails.Value, true);
            }
            if (unit != null)
            {
                uri.AppendQuery("unit", unit.Value.ToString(), true);
            }
            if (includeCurrentStorm != null)
            {
                uri.AppendQuery("current", includeCurrentStorm.Value, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get the location of tropical storms from individual national weather forecasting agencies. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="year"> Year of the cyclone(s). </param>
        /// <param name="basinId"> Basin identifier. </param>
        /// <param name="governmentStormId"> Government storm Id. </param>
        /// <param name="includeDetails"> When true, wind radii summary data is included in the response. </param>
        /// <param name="includeGeometricDetails"> When true, wind radii summary data and geoJSON details are included in the response. </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="includeCurrentStorm"> When true, return the current storm location. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// The `Get Tropical Storm Locations` API is an HTTP `GET` request that returns the location of individual government-issued tropical storms. Information about the tropical storms includes, location coordinates, geometry, basin ID, date, wind details and wind radii.
        /// </remarks>
        public async Task<Response<StormLocationsResult>> GetTropicalStormLocationsAsync(JsonFormat format, int year, BasinId basinId, int governmentStormId, bool? includeDetails = null, bool? includeGeometricDetails = null, WeatherDataUnit? unit = null, bool? includeCurrentStorm = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetTropicalStormLocationsRequest(format, year, basinId, governmentStormId, includeDetails, includeGeometricDetails, unit, includeCurrentStorm);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StormLocationsResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = StormLocationsResult.DeserializeStormLocationsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get the location of tropical storms from individual national weather forecasting agencies. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="year"> Year of the cyclone(s). </param>
        /// <param name="basinId"> Basin identifier. </param>
        /// <param name="governmentStormId"> Government storm Id. </param>
        /// <param name="includeDetails"> When true, wind radii summary data is included in the response. </param>
        /// <param name="includeGeometricDetails"> When true, wind radii summary data and geoJSON details are included in the response. </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="includeCurrentStorm"> When true, return the current storm location. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// The `Get Tropical Storm Locations` API is an HTTP `GET` request that returns the location of individual government-issued tropical storms. Information about the tropical storms includes, location coordinates, geometry, basin ID, date, wind details and wind radii.
        /// </remarks>
        public Response<StormLocationsResult> GetTropicalStormLocations(JsonFormat format, int year, BasinId basinId, int governmentStormId, bool? includeDetails = null, bool? includeGeometricDetails = null, WeatherDataUnit? unit = null, bool? includeCurrentStorm = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetTropicalStormLocationsRequest(format, year, basinId, governmentStormId, includeDetails, includeGeometricDetails, unit, includeCurrentStorm);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StormLocationsResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = StormLocationsResult.DeserializeStormLocationsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetCurrentAirQualityRequest(JsonFormat format, IEnumerable<double> coordinates, string language, bool? includePollutantDetails)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/airQuality/current/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            if (includePollutantDetails != null)
            {
                uri.AppendQuery("pollutants", includePollutantDetails.Value, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get current air quality information that includes potential risks and suggested precautions. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="includePollutantDetails"> Boolean value that returns detailed information about each pollutant. By default is True. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Current Air Quality` API is an HTTP `GET` request that returns detailed information about the concentration of pollutants and overall status for current air quality, including pollution levels, air quality index values, the dominant pollutant, and a brief statement summarizing risk level and suggested precautions.
        /// </remarks>
        public async Task<Response<AirQualityResult>> GetCurrentAirQualityAsync(JsonFormat format, IEnumerable<double> coordinates, string language = null, bool? includePollutantDetails = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetCurrentAirQualityRequest(format, coordinates, language, includePollutantDetails);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        AirQualityResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = AirQualityResult.DeserializeAirQualityResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get current air quality information that includes potential risks and suggested precautions. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="includePollutantDetails"> Boolean value that returns detailed information about each pollutant. By default is True. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Current Air Quality` API is an HTTP `GET` request that returns detailed information about the concentration of pollutants and overall status for current air quality, including pollution levels, air quality index values, the dominant pollutant, and a brief statement summarizing risk level and suggested precautions.
        /// </remarks>
        public Response<AirQualityResult> GetCurrentAirQuality(JsonFormat format, IEnumerable<double> coordinates, string language = null, bool? includePollutantDetails = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetCurrentAirQualityRequest(format, coordinates, language, includePollutantDetails);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        AirQualityResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = AirQualityResult.DeserializeAirQualityResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetAirQualityDailyForecastsRequest(JsonFormat format, IEnumerable<double> coordinates, string language, DailyDuration? duration)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/airQuality/forecasts/daily/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            if (duration != null)
            {
                uri.AppendQuery("duration", duration.Value.ToSerialInt32(), true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get daily air quality forecasts for the next one to seven days that include pollutant levels, potential risks and suggested precautions. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="duration"> Specifies for how many days from now we would like to know about the air quality. Available values are 1, 2, 3, 4, 5, 6, and 7. Default value is 1. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Air Quality Daily Forecasts` API is an HTTP `GET` request that returns detailed information about the concentration of pollutants and overall status of forecasted daily air quality. The service can provide forecasted daily air quality information for the upcoming 1 to 7 days, including pollution levels, air quality index values, the dominant pollutant, and a brief statement summarizing risk level and suggested precautions.
        /// </remarks>
        public async Task<Response<DailyAirQualityForecastResult>> GetAirQualityDailyForecastsAsync(JsonFormat format, IEnumerable<double> coordinates, string language = null, DailyDuration? duration = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetAirQualityDailyForecastsRequest(format, coordinates, language, duration);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DailyAirQualityForecastResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DailyAirQualityForecastResult.DeserializeDailyAirQualityForecastResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get daily air quality forecasts for the next one to seven days that include pollutant levels, potential risks and suggested precautions. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="duration"> Specifies for how many days from now we would like to know about the air quality. Available values are 1, 2, 3, 4, 5, 6, and 7. Default value is 1. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Air Quality Daily Forecasts` API is an HTTP `GET` request that returns detailed information about the concentration of pollutants and overall status of forecasted daily air quality. The service can provide forecasted daily air quality information for the upcoming 1 to 7 days, including pollution levels, air quality index values, the dominant pollutant, and a brief statement summarizing risk level and suggested precautions.
        /// </remarks>
        public Response<DailyAirQualityForecastResult> GetAirQualityDailyForecasts(JsonFormat format, IEnumerable<double> coordinates, string language = null, DailyDuration? duration = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetAirQualityDailyForecastsRequest(format, coordinates, language, duration);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DailyAirQualityForecastResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DailyAirQualityForecastResult.DeserializeDailyAirQualityForecastResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetAirQualityHourlyForecastsRequest(JsonFormat format, IEnumerable<double> coordinates, string language, HourlyDuration? duration, bool? includePollutantDetails)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/airQuality/forecasts/hourly/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            if (duration != null)
            {
                uri.AppendQuery("duration", duration.Value.ToSerialInt32(), true);
            }
            if (includePollutantDetails != null)
            {
                uri.AppendQuery("pollutants", includePollutantDetails.Value, true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get hourly air quality forecasts for the next one to 96 hours that include pollutant levels, potential risks and suggested precautions. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="duration"> Specifies for how many hours from now we would like to know about the air quality. Available values are 1, 12, 24, 48, 72, 96. Default value is 1 hour. </param>
        /// <param name="includePollutantDetails"> Boolean value that returns detailed information about each pollutant. By default is True. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Air Quality Hourly Forecasts` API is an HTTP `GET` request that returns detailed information about the concentration of pollutants and overall status for forecasted upcoming hourly air quality. The service can provide forecasted hourly air quality information for the upcoming time spans of 1, 12, 24, 48, 72, and 96 hours, including pollution levels, air quality index values, the dominant pollutant, and a brief statement summarizing risk level and suggested precautions.
        /// </remarks>
        public async Task<Response<AirQualityResult>> GetAirQualityHourlyForecastsAsync(JsonFormat format, IEnumerable<double> coordinates, string language = null, HourlyDuration? duration = null, bool? includePollutantDetails = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetAirQualityHourlyForecastsRequest(format, coordinates, language, duration, includePollutantDetails);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        AirQualityResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = AirQualityResult.DeserializeAirQualityResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get hourly air quality forecasts for the next one to 96 hours that include pollutant levels, potential risks and suggested precautions. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="duration"> Specifies for how many hours from now we would like to know about the air quality. Available values are 1, 12, 24, 48, 72, 96. Default value is 1 hour. </param>
        /// <param name="includePollutantDetails"> Boolean value that returns detailed information about each pollutant. By default is True. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Air Quality Hourly Forecasts` API is an HTTP `GET` request that returns detailed information about the concentration of pollutants and overall status for forecasted upcoming hourly air quality. The service can provide forecasted hourly air quality information for the upcoming time spans of 1, 12, 24, 48, 72, and 96 hours, including pollution levels, air quality index values, the dominant pollutant, and a brief statement summarizing risk level and suggested precautions.
        /// </remarks>
        public Response<AirQualityResult> GetAirQualityHourlyForecasts(JsonFormat format, IEnumerable<double> coordinates, string language = null, HourlyDuration? duration = null, bool? includePollutantDetails = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetAirQualityHourlyForecastsRequest(format, coordinates, language, duration, includePollutantDetails);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        AirQualityResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = AirQualityResult.DeserializeAirQualityResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetDailyHistoricalActualsRequest(JsonFormat format, IEnumerable<double> coordinates, DateTimeOffset startDate, DateTimeOffset endDate, WeatherDataUnit? unit)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/historical/actuals/daily/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            uri.AppendQuery("startDate", startDate, "D", true);
            uri.AppendQuery("endDate", endDate, "D", true);
            if (unit != null)
            {
                uri.AppendQuery("unit", unit.Value.ToString(), true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get climatology data such as past daily actual observed temperatures, precipitation, snowfall and snow depth. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="startDate"> Start date in ISO 8601 format, for example, 2019-10-27. The date range supported is 1 to 31 calendar days, so be sure to specify a startDate and endDate that does not exceed a maximum of 31 days (i.e.: startDate=2012-01-01&amp;endDate=2012-01-31). </param>
        /// <param name="endDate"> End date in ISO 8601 format, for example, 2019-10-28. The date range supported is 1 to 31 calendar days, so be sure to specify a startDate and endDate that does not exceed a maximum of 31 days (i.e.: startDate=2012-01-01&amp;endDate=2012-01-31). </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Daily Historical Actuals` API is an HTTP `GET` request that returns climatology data such as past daily actual observed temperatures, precipitation, snowfall, snow depth and cooling/heating degree day information, for the day at a given coordinate location. The data is requested for a specified date range, up to 31 days in a single API request. Generally, historical data may be available as far back as the last 5 to 40+ years, depending on the location.
        /// </remarks>
        public async Task<Response<DailyHistoricalActualsResult>> GetDailyHistoricalActualsAsync(JsonFormat format, IEnumerable<double> coordinates, DateTimeOffset startDate, DateTimeOffset endDate, WeatherDataUnit? unit = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetDailyHistoricalActualsRequest(format, coordinates, startDate, endDate, unit);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DailyHistoricalActualsResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DailyHistoricalActualsResult.DeserializeDailyHistoricalActualsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get climatology data such as past daily actual observed temperatures, precipitation, snowfall and snow depth. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="startDate"> Start date in ISO 8601 format, for example, 2019-10-27. The date range supported is 1 to 31 calendar days, so be sure to specify a startDate and endDate that does not exceed a maximum of 31 days (i.e.: startDate=2012-01-01&amp;endDate=2012-01-31). </param>
        /// <param name="endDate"> End date in ISO 8601 format, for example, 2019-10-28. The date range supported is 1 to 31 calendar days, so be sure to specify a startDate and endDate that does not exceed a maximum of 31 days (i.e.: startDate=2012-01-01&amp;endDate=2012-01-31). </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Daily Historical Actuals` API is an HTTP `GET` request that returns climatology data such as past daily actual observed temperatures, precipitation, snowfall, snow depth and cooling/heating degree day information, for the day at a given coordinate location. The data is requested for a specified date range, up to 31 days in a single API request. Generally, historical data may be available as far back as the last 5 to 40+ years, depending on the location.
        /// </remarks>
        public Response<DailyHistoricalActualsResult> GetDailyHistoricalActuals(JsonFormat format, IEnumerable<double> coordinates, DateTimeOffset startDate, DateTimeOffset endDate, WeatherDataUnit? unit = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetDailyHistoricalActualsRequest(format, coordinates, startDate, endDate, unit);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DailyHistoricalActualsResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DailyHistoricalActualsResult.DeserializeDailyHistoricalActualsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetDailyHistoricalRecordsRequest(JsonFormat format, IEnumerable<double> coordinates, DateTimeOffset startDate, DateTimeOffset endDate, WeatherDataUnit? unit)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/historical/records/daily/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            uri.AppendQuery("startDate", startDate, "D", true);
            uri.AppendQuery("endDate", endDate, "D", true);
            if (unit != null)
            {
                uri.AppendQuery("unit", unit.Value.ToString(), true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get climatology data such as past daily record temperatures, precipitation and snowfall at a given location. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="startDate"> Start date in ISO 8601 format, for example, 2019-10-27. The date range supported is 1 to 31 calendar days, so be sure to specify a startDate and endDate that does not exceed a maximum of 31 days (i.e.: startDate=2012-01-01&amp;endDate=2012-01-31). </param>
        /// <param name="endDate"> End date in ISO 8601 format, for example, 2019-10-28. The date range supported is 1 to 31 calendar days, so be sure to specify a startDate and endDate that does not exceed a maximum of 31 days (i.e.: startDate=2012-01-01&amp;endDate=2012-01-31). </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Daily Historical Records` API is an HTTP `GET` request that returns climatology data such as past daily record temperatures, precipitation and snowfall at a given coordinate location. Availability of records data will vary by location. Generally, historical data may be available as far back as the last 5 to 40+ years, depending on the location.
        /// </remarks>
        public async Task<Response<DailyHistoricalRecordsResult>> GetDailyHistoricalRecordsAsync(JsonFormat format, IEnumerable<double> coordinates, DateTimeOffset startDate, DateTimeOffset endDate, WeatherDataUnit? unit = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetDailyHistoricalRecordsRequest(format, coordinates, startDate, endDate, unit);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DailyHistoricalRecordsResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DailyHistoricalRecordsResult.DeserializeDailyHistoricalRecordsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get climatology data such as past daily record temperatures, precipitation and snowfall at a given location. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="startDate"> Start date in ISO 8601 format, for example, 2019-10-27. The date range supported is 1 to 31 calendar days, so be sure to specify a startDate and endDate that does not exceed a maximum of 31 days (i.e.: startDate=2012-01-01&amp;endDate=2012-01-31). </param>
        /// <param name="endDate"> End date in ISO 8601 format, for example, 2019-10-28. The date range supported is 1 to 31 calendar days, so be sure to specify a startDate and endDate that does not exceed a maximum of 31 days (i.e.: startDate=2012-01-01&amp;endDate=2012-01-31). </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Daily Historical Records` API is an HTTP `GET` request that returns climatology data such as past daily record temperatures, precipitation and snowfall at a given coordinate location. Availability of records data will vary by location. Generally, historical data may be available as far back as the last 5 to 40+ years, depending on the location.
        /// </remarks>
        public Response<DailyHistoricalRecordsResult> GetDailyHistoricalRecords(JsonFormat format, IEnumerable<double> coordinates, DateTimeOffset startDate, DateTimeOffset endDate, WeatherDataUnit? unit = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetDailyHistoricalRecordsRequest(format, coordinates, startDate, endDate, unit);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DailyHistoricalRecordsResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DailyHistoricalRecordsResult.DeserializeDailyHistoricalRecordsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetDailyHistoricalNormalsRequest(JsonFormat format, IEnumerable<double> coordinates, DateTimeOffset startDate, DateTimeOffset endDate, WeatherDataUnit? unit)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/weather/historical/normals/daily/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (coordinates != null && !(coordinates is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("query", coordinates, ",", true);
            }
            uri.AppendQuery("startDate", startDate, "D", true);
            uri.AppendQuery("endDate", endDate, "D", true);
            if (unit != null)
            {
                uri.AppendQuery("unit", unit.Value.ToString(), true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get climatology data such as past daily normal temperatures, precipitation and cooling/heating degree day information for a given location. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="startDate"> Start date in ISO 8601 format, for example, 2019-10-27. The date range supported is 1 to 31 calendar days, so be sure to specify a startDate and endDate that does not exceed a maximum of 31 days (i.e.: startDate=2012-01-01&amp;endDate=2012-01-31). </param>
        /// <param name="endDate"> End date in ISO 8601 format, for example, 2019-10-28. The date range supported is 1 to 31 calendar days, so be sure to specify a startDate and endDate that does not exceed a maximum of 31 days (i.e.: startDate=2012-01-01&amp;endDate=2012-01-31). </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Daily Historical Normals` API is an HTTP `GET` request that returns climatology data such as past daily normal temperatures, precipitation and cooling/heating degree day information for the day at a given coordinate location. The historical normals are a 30-year average for temperatures and precipitation for a specific location. As is standard practice in climatology, the 30-year average covers years 1991-2020, this data will be used for one decade and then will reset in the year 2030. Generally, historical data may be available as far back as the last 5 to 40+ years, depending on the location.
        /// </remarks>
        public async Task<Response<DailyHistoricalNormalsResult>> GetDailyHistoricalNormalsAsync(JsonFormat format, IEnumerable<double> coordinates, DateTimeOffset startDate, DateTimeOffset endDate, WeatherDataUnit? unit = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetDailyHistoricalNormalsRequest(format, coordinates, startDate, endDate, unit);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DailyHistoricalNormalsResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DailyHistoricalNormalsResult.DeserializeDailyHistoricalNormalsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get climatology data such as past daily normal temperatures, precipitation and cooling/heating degree day information for a given location. </summary>
        /// <param name="format"> Desired format of the response. Only `json` format is supported. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="coordinates">
        /// The applicable query specified as a comma separated string composed by latitude followed by longitude e.g. "47.641268,-122.125679".
        ///
        /// Weather information is generally available for locations on land, bodies of water surrounded by land, and areas of the ocean that are within approximately 50 nautical miles of a coastline.
        /// </param>
        /// <param name="startDate"> Start date in ISO 8601 format, for example, 2019-10-27. The date range supported is 1 to 31 calendar days, so be sure to specify a startDate and endDate that does not exceed a maximum of 31 days (i.e.: startDate=2012-01-01&amp;endDate=2012-01-31). </param>
        /// <param name="endDate"> End date in ISO 8601 format, for example, 2019-10-28. The date range supported is 1 to 31 calendar days, so be sure to specify a startDate and endDate that does not exceed a maximum of 31 days (i.e.: startDate=2012-01-01&amp;endDate=2012-01-31). </param>
        /// <param name="unit"> Specifies to return the data in either metric units or imperial units. Default value is metric. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="coordinates"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Daily Historical Normals` API is an HTTP `GET` request that returns climatology data such as past daily normal temperatures, precipitation and cooling/heating degree day information for the day at a given coordinate location. The historical normals are a 30-year average for temperatures and precipitation for a specific location. As is standard practice in climatology, the 30-year average covers years 1991-2020, this data will be used for one decade and then will reset in the year 2030. Generally, historical data may be available as far back as the last 5 to 40+ years, depending on the location.
        /// </remarks>
        public Response<DailyHistoricalNormalsResult> GetDailyHistoricalNormals(JsonFormat format, IEnumerable<double> coordinates, DateTimeOffset startDate, DateTimeOffset endDate, WeatherDataUnit? unit = null, CancellationToken cancellationToken = default)
        {
            if (coordinates == null)
            {
                throw new ArgumentNullException(nameof(coordinates));
            }

            using var message = CreateGetDailyHistoricalNormalsRequest(format, coordinates, startDate, endDate, unit);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DailyHistoricalNormalsResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DailyHistoricalNormalsResult.DeserializeDailyHistoricalNormalsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }
    }
}
