// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

// <auto-generated/>

#nullable disable

using System;
using System.Collections.Generic;
using System.IO;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using Azure.Core;
using Azure.Core.Pipeline;
using Azure.Maps.Common;

namespace Azure.Maps.Rendering
{
    internal partial class RenderRestClient
    {
        private readonly HttpPipeline _pipeline;
        private readonly Uri _endpoint;
        private readonly string _clientId;
        private readonly string _apiVersion;
        private readonly MediaType? _accept;

        /// <summary> The ClientDiagnostics is used to provide tracing support for the client library. </summary>
        internal ClientDiagnostics ClientDiagnostics { get; }

        /// <summary> Initializes a new instance of RenderRestClient. </summary>
        /// <param name="clientDiagnostics"> The handler for diagnostic messaging in the client. </param>
        /// <param name="pipeline"> The HTTP pipeline for sending and receiving REST requests and responses. </param>
        /// <param name="endpoint"> server parameter. </param>
        /// <param name="clientId"> Specifies which account is intended for usage in conjunction with the Microsoft Entra ID security model.  It represents a unique ID for the Azure Maps account and can be retrieved from the Azure Maps management  plane Account API. To use Microsoft Entra ID security in Azure Maps see the following [articles](https://aka.ms/amauthdetails) for guidance. </param>
        /// <param name="apiVersion"> Api Version. </param>
        /// <param name="accept"> The Accept header field can be used to specify preferences regarding response media types. Allowed media types include image/jpeg and image/png. Return image in image/png if Accept header is not specified. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="clientDiagnostics"/>, <paramref name="pipeline"/> or <paramref name="apiVersion"/> is null. </exception>
        public RenderRestClient(ClientDiagnostics clientDiagnostics, HttpPipeline pipeline, Uri endpoint = null, string clientId = null, string apiVersion = "2024-04-01", MediaType? accept = null)
        {
            ClientDiagnostics = clientDiagnostics ?? throw new ArgumentNullException(nameof(clientDiagnostics));
            _pipeline = pipeline ?? throw new ArgumentNullException(nameof(pipeline));
            _endpoint = endpoint ?? new Uri("https://atlas.microsoft.com");
            _clientId = clientId;
            _apiVersion = apiVersion ?? throw new ArgumentNullException(nameof(apiVersion));
            _accept = accept;
        }

        internal HttpMessage CreateGetMapTileRequest(MapTileSetId tilesetId, MapTileIndex mapTileIndex, DateTimeOffset? timeStamp, MapTileSize? tileSize, string language, LocalizedMapView? localizedMapView)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/map/tile", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            uri.AppendQuery("tilesetId", tilesetId.ToString(), true);
            uri.AppendQuery("zoom", mapTileIndex.Z, true);
            uri.AppendQuery("x", mapTileIndex.X, true);
            uri.AppendQuery("y", mapTileIndex.Y, true);
            if (timeStamp != null)
            {
                uri.AppendQuery("timeStamp", timeStamp.Value, "O", true);
            }
            if (tileSize != null)
            {
                uri.AppendQuery("tileSize", tileSize.Value.ToString(), true);
            }
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            if (localizedMapView != null)
            {
                uri.AppendQuery("view", localizedMapView.Value.ToString(), true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json, image/jpeg, image/png, image/pbf, application/vnd.mapbox-vector-tile");
            return message;
        }

        /// <summary> Use to request map tiles in vector or raster format. </summary>
        /// <param name="tilesetId"> A tileset is a collection of raster or vector data broken up into a uniform grid of square tiles at preset  zoom levels. Every tileset has a **tilesetId** to use when making requests. The **tilesetId** for tilesets created using [Azure Maps Creator](https://aka.ms/amcreator) are generated through the  [Tileset Create API](https://docs.microsoft.com/rest/api/maps-creator/tileset). The ready-to-use tilesets supplied  by Azure Maps are listed below. For example, microsoft.base. </param>
        /// <param name="mapTileIndex"> Parameter group. </param>
        /// <param name="timeStamp">
        /// The desired date and time of the requested tile. This parameter must be specified in the standard date-time format (e.g. 2019-11-14T16:03:00-08:00), as defined by [ISO 8601](https://en.wikipedia.org/wiki/ISO_8601). This parameter is only supported when tilesetId parameter is set to one of the values below.
        ///
        /// * microsoft.weather.infrared.main: We provide tiles up to 3 hours in the past. Tiles are available in 10-minute intervals. We round the timeStamp value to the nearest 10-minute time frame.
        /// * microsoft.weather.radar.main: We provide tiles up to 1.5 hours in the past and up to 2 hours in the future. Tiles are available in 5-minute intervals. We round the timeStamp value to the nearest 5-minute time frame.
        /// </param>
        /// <param name="tileSize"> The size of the returned map tile in pixels. </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="localizedMapView">
        /// The View parameter (also called the "user region" parameter) allows you to show the correct maps for a certain country/region for geopolitically disputed regions. Different countries/regions have different views of such regions, and the View parameter allows your application to comply with the view required by the country/region your application will be serving. By default, the View parameter is set to “Unified” even if you haven’t defined it in  the request. It is your responsibility to determine the location of your users, and then set the View parameter correctly for that location. Alternatively, you have the option to set ‘View=Auto’, which will return the map data based on the IP  address of the request. The View parameter in Azure Maps must be used in compliance with applicable laws, including those  regarding mapping, of the country/region where maps, images and other data and third party content that you are authorized to  access via Azure Maps is made available. Example: view=IN.
        ///
        /// Please refer to [Supported Views](https://aka.ms/AzureMapsLocalizationViews) for details and to see the available Views.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="mapTileIndex"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Map Tiles` API in an HTTP GET request that allows users to request map tiles in vector or raster formats typically to be integrated  into a map control or SDK. Some example tiles that can be requested are Azure Maps road tiles, real-time  Weather Radar tiles or the map tiles created using [Azure Maps Creator](https://aka.ms/amcreator). By default,  Azure Maps uses vector tiles for its web map control ([Web SDK](/azure/azure-maps/about-azure-maps#web-sdk)) and [Android SDK](/azure/azure-maps/about-azure-maps#android-sdk).
        /// </remarks>
        public async Task<ResponseWithHeaders<Stream, RenderGetMapTileHeaders>> GetMapTileAsync(MapTileSetId tilesetId, MapTileIndex mapTileIndex, DateTimeOffset? timeStamp = null, MapTileSize? tileSize = null, string language = null, LocalizedMapView? localizedMapView = null, CancellationToken cancellationToken = default)
        {
            if (mapTileIndex == null)
            {
                throw new ArgumentNullException(nameof(mapTileIndex));
            }

            using var message = CreateGetMapTileRequest(tilesetId, mapTileIndex, timeStamp, tileSize, language, localizedMapView);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new RenderGetMapTileHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        var value = message.ExtractResponseContent();
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to request map tiles in vector or raster format. </summary>
        /// <param name="tilesetId"> A tileset is a collection of raster or vector data broken up into a uniform grid of square tiles at preset  zoom levels. Every tileset has a **tilesetId** to use when making requests. The **tilesetId** for tilesets created using [Azure Maps Creator](https://aka.ms/amcreator) are generated through the  [Tileset Create API](https://docs.microsoft.com/rest/api/maps-creator/tileset). The ready-to-use tilesets supplied  by Azure Maps are listed below. For example, microsoft.base. </param>
        /// <param name="mapTileIndex"> Parameter group. </param>
        /// <param name="timeStamp">
        /// The desired date and time of the requested tile. This parameter must be specified in the standard date-time format (e.g. 2019-11-14T16:03:00-08:00), as defined by [ISO 8601](https://en.wikipedia.org/wiki/ISO_8601). This parameter is only supported when tilesetId parameter is set to one of the values below.
        ///
        /// * microsoft.weather.infrared.main: We provide tiles up to 3 hours in the past. Tiles are available in 10-minute intervals. We round the timeStamp value to the nearest 10-minute time frame.
        /// * microsoft.weather.radar.main: We provide tiles up to 1.5 hours in the past and up to 2 hours in the future. Tiles are available in 5-minute intervals. We round the timeStamp value to the nearest 5-minute time frame.
        /// </param>
        /// <param name="tileSize"> The size of the returned map tile in pixels. </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="localizedMapView">
        /// The View parameter (also called the "user region" parameter) allows you to show the correct maps for a certain country/region for geopolitically disputed regions. Different countries/regions have different views of such regions, and the View parameter allows your application to comply with the view required by the country/region your application will be serving. By default, the View parameter is set to “Unified” even if you haven’t defined it in  the request. It is your responsibility to determine the location of your users, and then set the View parameter correctly for that location. Alternatively, you have the option to set ‘View=Auto’, which will return the map data based on the IP  address of the request. The View parameter in Azure Maps must be used in compliance with applicable laws, including those  regarding mapping, of the country/region where maps, images and other data and third party content that you are authorized to  access via Azure Maps is made available. Example: view=IN.
        ///
        /// Please refer to [Supported Views](https://aka.ms/AzureMapsLocalizationViews) for details and to see the available Views.
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="mapTileIndex"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Map Tiles` API in an HTTP GET request that allows users to request map tiles in vector or raster formats typically to be integrated  into a map control or SDK. Some example tiles that can be requested are Azure Maps road tiles, real-time  Weather Radar tiles or the map tiles created using [Azure Maps Creator](https://aka.ms/amcreator). By default,  Azure Maps uses vector tiles for its web map control ([Web SDK](/azure/azure-maps/about-azure-maps#web-sdk)) and [Android SDK](/azure/azure-maps/about-azure-maps#android-sdk).
        /// </remarks>
        public ResponseWithHeaders<Stream, RenderGetMapTileHeaders> GetMapTile(MapTileSetId tilesetId, MapTileIndex mapTileIndex, DateTimeOffset? timeStamp = null, MapTileSize? tileSize = null, string language = null, LocalizedMapView? localizedMapView = null, CancellationToken cancellationToken = default)
        {
            if (mapTileIndex == null)
            {
                throw new ArgumentNullException(nameof(mapTileIndex));
            }

            using var message = CreateGetMapTileRequest(tilesetId, mapTileIndex, timeStamp, tileSize, language, localizedMapView);
            _pipeline.Send(message, cancellationToken);
            var headers = new RenderGetMapTileHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        var value = message.ExtractResponseContent();
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetMapTilesetRequest(MapTileSetId tilesetId)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/map/tileset", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            uri.AppendQuery("tilesetId", tilesetId.ToString(), true);
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get metadata for a tileset. </summary>
        /// <param name="tilesetId"> A tileset is a collection of raster or vector data broken up into a uniform grid of square tiles at preset  zoom levels. Every tileset has a **tilesetId** to use when making requests. The **tilesetId** for tilesets created using [Azure Maps Creator](https://aka.ms/amcreator) are generated through the  [Tileset Create API](https://docs.microsoft.com/rest/api/maps-creator/tileset). The ready-to-use tilesets supplied  by Azure Maps are listed below. For example, microsoft.base. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// The Get Map Tileset API allows users to request metadata for a tileset.
        /// </remarks>
        public async Task<Response<MapTileSet>> GetMapTilesetAsync(MapTileSetId tilesetId, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetMapTilesetRequest(tilesetId);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        MapTileSet value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = MapTileSet.DeserializeMapTileSet(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get metadata for a tileset. </summary>
        /// <param name="tilesetId"> A tileset is a collection of raster or vector data broken up into a uniform grid of square tiles at preset  zoom levels. Every tileset has a **tilesetId** to use when making requests. The **tilesetId** for tilesets created using [Azure Maps Creator](https://aka.ms/amcreator) are generated through the  [Tileset Create API](https://docs.microsoft.com/rest/api/maps-creator/tileset). The ready-to-use tilesets supplied  by Azure Maps are listed below. For example, microsoft.base. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// The Get Map Tileset API allows users to request metadata for a tileset.
        /// </remarks>
        public Response<MapTileSet> GetMapTileset(MapTileSetId tilesetId, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetMapTilesetRequest(tilesetId);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        MapTileSet value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = MapTileSet.DeserializeMapTileSet(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetMapAttributionRequest(MapTileSetId tilesetId, int zoom, IEnumerable<double> bounds)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/map/attribution", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            uri.AppendQuery("tilesetId", tilesetId.ToString(), true);
            uri.AppendQuery("zoom", zoom, true);
            if (bounds != null && !(bounds is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("bounds", bounds, ",", true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get map copyright attribution information. </summary>
        /// <param name="tilesetId"> A tileset is a collection of raster or vector data broken up into a uniform grid of square tiles at preset  zoom levels. Every tileset has a **tilesetId** to use when making requests. The **tilesetId** for tilesets created using [Azure Maps Creator](https://aka.ms/amcreator) are generated through the  [Tileset Create API](https://docs.microsoft.com/rest/api/maps-creator/tileset). The ready-to-use tilesets supplied  by Azure Maps are listed below. For example, microsoft.base. </param>
        /// <param name="zoom"> Zoom level for the desired map attribution. </param>
        /// <param name="bounds"> The string that represents the rectangular area of a bounding box. The bounds parameter is defined by the 4 bounding box coordinates, with WGS84 longitude and latitude of the southwest corner followed by  WGS84 longitude and latitude of the northeast corner. The string is presented in the following  format: `[SouthwestCorner_Longitude, SouthwestCorner_Latitude, NortheastCorner_Longitude,  NortheastCorner_Latitude]`. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="bounds"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Map Attribution` API allows users to request map copyright attribution information for a section of a tileset.
        /// </remarks>
        public async Task<Response<MapAttribution>> GetMapAttributionAsync(MapTileSetId tilesetId, int zoom, IEnumerable<double> bounds, CancellationToken cancellationToken = default)
        {
            if (bounds == null)
            {
                throw new ArgumentNullException(nameof(bounds));
            }

            using var message = CreateGetMapAttributionRequest(tilesetId, zoom, bounds);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        MapAttribution value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = MapAttribution.DeserializeMapAttribution(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get map copyright attribution information. </summary>
        /// <param name="tilesetId"> A tileset is a collection of raster or vector data broken up into a uniform grid of square tiles at preset  zoom levels. Every tileset has a **tilesetId** to use when making requests. The **tilesetId** for tilesets created using [Azure Maps Creator](https://aka.ms/amcreator) are generated through the  [Tileset Create API](https://docs.microsoft.com/rest/api/maps-creator/tileset). The ready-to-use tilesets supplied  by Azure Maps are listed below. For example, microsoft.base. </param>
        /// <param name="zoom"> Zoom level for the desired map attribution. </param>
        /// <param name="bounds"> The string that represents the rectangular area of a bounding box. The bounds parameter is defined by the 4 bounding box coordinates, with WGS84 longitude and latitude of the southwest corner followed by  WGS84 longitude and latitude of the northeast corner. The string is presented in the following  format: `[SouthwestCorner_Longitude, SouthwestCorner_Latitude, NortheastCorner_Longitude,  NortheastCorner_Latitude]`. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="bounds"/> is null. </exception>
        /// <remarks>
        ///
        /// The `Get Map Attribution` API allows users to request map copyright attribution information for a section of a tileset.
        /// </remarks>
        public Response<MapAttribution> GetMapAttribution(MapTileSetId tilesetId, int zoom, IEnumerable<double> bounds, CancellationToken cancellationToken = default)
        {
            if (bounds == null)
            {
                throw new ArgumentNullException(nameof(bounds));
            }

            using var message = CreateGetMapAttributionRequest(tilesetId, zoom, bounds);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        MapAttribution value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = MapAttribution.DeserializeMapAttribution(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetMapStateTileRequest(string statesetId, MapTileIndex mapTileIndex)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/map/statetile", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            uri.AppendQuery("zoom", mapTileIndex.Z, true);
            uri.AppendQuery("x", mapTileIndex.X, true);
            uri.AppendQuery("y", mapTileIndex.Y, true);
            uri.AppendQuery("statesetId", statesetId, true);
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/vnd.mapbox-vector-tile, application/json");
            return message;
        }

        /// <summary> Use to get state tiles in vector format that can then be used to display feature state information in an indoor map. </summary>
        /// <param name="statesetId"> The stateset id. </param>
        /// <param name="mapTileIndex"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="statesetId"/> or <paramref name="mapTileIndex"/> is null. </exception>
        /// <remarks>
        ///
        /// Fetches state tiles in vector format typically to be integrated into indoor maps module of map control or SDK. The map control will call this API after user turns on dynamic styling. For more information, see [Zoom Levels and Tile Grid](/azure/location-based-services/zoom-levels-and-tile-grid).
        /// </remarks>
        public async Task<ResponseWithHeaders<Stream, RenderGetMapStateTileHeaders>> GetMapStateTileAsync(string statesetId, MapTileIndex mapTileIndex, CancellationToken cancellationToken = default)
        {
            if (statesetId == null)
            {
                throw new ArgumentNullException(nameof(statesetId));
            }
            if (mapTileIndex == null)
            {
                throw new ArgumentNullException(nameof(mapTileIndex));
            }

            using var message = CreateGetMapStateTileRequest(statesetId, mapTileIndex);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new RenderGetMapStateTileHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        var value = message.ExtractResponseContent();
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get state tiles in vector format that can then be used to display feature state information in an indoor map. </summary>
        /// <param name="statesetId"> The stateset id. </param>
        /// <param name="mapTileIndex"> Parameter group. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="statesetId"/> or <paramref name="mapTileIndex"/> is null. </exception>
        /// <remarks>
        ///
        /// Fetches state tiles in vector format typically to be integrated into indoor maps module of map control or SDK. The map control will call this API after user turns on dynamic styling. For more information, see [Zoom Levels and Tile Grid](/azure/location-based-services/zoom-levels-and-tile-grid).
        /// </remarks>
        public ResponseWithHeaders<Stream, RenderGetMapStateTileHeaders> GetMapStateTile(string statesetId, MapTileIndex mapTileIndex, CancellationToken cancellationToken = default)
        {
            if (statesetId == null)
            {
                throw new ArgumentNullException(nameof(statesetId));
            }
            if (mapTileIndex == null)
            {
                throw new ArgumentNullException(nameof(mapTileIndex));
            }

            using var message = CreateGetMapStateTileRequest(statesetId, mapTileIndex);
            _pipeline.Send(message, cancellationToken);
            var headers = new RenderGetMapStateTileHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        var value = message.ExtractResponseContent();
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetCopyrightCaptionRequest(ResponseFormat format)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/map/copyright/caption/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get copyright information to use when rendering a tile. </summary>
        /// <param name="format"> Desired format of the response. Value can be either _json_ or _xml_. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// The `Get Copyright Caption` API is an HTTP GET request designed to serve copyright information to be used with tiles requested from the Render service. In addition to a basic copyright for the whole map, it can serve specific groups of copyrights for some countries/regions.
        ///
        /// As an alternative to copyrights for map request, it can also return captions for displaying provider information on the map.
        /// </remarks>
        public async Task<Response<CopyrightCaption>> GetCopyrightCaptionAsync(ResponseFormat format, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetCopyrightCaptionRequest(format);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        CopyrightCaption value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = CopyrightCaption.DeserializeCopyrightCaption(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get copyright information to use when rendering a tile. </summary>
        /// <param name="format"> Desired format of the response. Value can be either _json_ or _xml_. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// The `Get Copyright Caption` API is an HTTP GET request designed to serve copyright information to be used with tiles requested from the Render service. In addition to a basic copyright for the whole map, it can serve specific groups of copyrights for some countries/regions.
        ///
        /// As an alternative to copyrights for map request, it can also return captions for displaying provider information on the map.
        /// </remarks>
        public Response<CopyrightCaption> GetCopyrightCaption(ResponseFormat format, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetCopyrightCaptionRequest(format);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        CopyrightCaption value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = CopyrightCaption.DeserializeCopyrightCaption(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetMapStaticImageRequest(MapTileSetId? tilesetId, TrafficTilesetId? trafficLayer, int? zoom, IEnumerable<double> center, IEnumerable<double> boundingBoxPrivate, int? height, int? width, string language, LocalizedMapView? localizedMapView, IEnumerable<string> pins, IEnumerable<string> path)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/map/static", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (tilesetId != null)
            {
                uri.AppendQuery("tilesetId", tilesetId.Value.ToString(), true);
            }
            if (trafficLayer != null)
            {
                uri.AppendQuery("trafficLayer", trafficLayer.Value.ToString(), true);
            }
            if (zoom != null)
            {
                uri.AppendQuery("zoom", zoom.Value, true);
            }
            if (center != null && !(center is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("center", center, ",", true);
            }
            if (boundingBoxPrivate != null && !(boundingBoxPrivate is Common.ChangeTrackingList<double> changeTrackingList0 && changeTrackingList0.IsUndefined))
            {
                uri.AppendQueryDelimited("bbox", boundingBoxPrivate, ",", true);
            }
            if (height != null)
            {
                uri.AppendQuery("height", height.Value, true);
            }
            if (width != null)
            {
                uri.AppendQuery("width", width.Value, true);
            }
            if (language != null)
            {
                uri.AppendQuery("language", language, true);
            }
            if (localizedMapView != null)
            {
                uri.AppendQuery("view", localizedMapView.Value.ToString(), true);
            }
            if (pins != null && !(pins is Common.ChangeTrackingList<string> changeTrackingList1 && changeTrackingList1.IsUndefined))
            {
                foreach (var param in pins)
                {
                    uri.AppendQuery("pins", param, true);
                }
            }
            if (path != null && !(path is Common.ChangeTrackingList<string> changeTrackingList2 && changeTrackingList2.IsUndefined))
            {
                foreach (var param in path)
                {
                    uri.AppendQuery("path", param, true);
                }
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            if (_accept != null)
            {
                request.Headers.Add("Accept", _accept.Value.ToString());
            }
            return message;
        }

        /// <summary>
        /// This rendering API produces static, rasterized map views of a user-defined area. It's suitable for lightweight web applications, when the desired user experience doesn't require interactive map controls, or when bandwidth is limited. This API is also useful for embedding maps in applications outside of the browser, in backend services, report generation, or desktop applications.
        ///
        ///  This API includes parameters for basic data visualization:
        ///
        /// - Labeled pushpins in multiple styles.
        /// - Render circle, path, and polygon geometry types.
        ///
        /// For more information and detailed examples, see [Render custom data on a raster map](/azure/azure-maps/how-to-render-custom-data).
        /// &lt;br&gt;&lt;br&gt;
        /// The dimensions of the bbox parameter are constrained, depending on the zoom level. This ensures the resulting image has an appropriate level of detail.
        /// &lt;br&gt;&lt;br&gt;
        ///
        ///   |Zoom Level | Min Lon Range   | Max Lon Range   | Min Lat Range| Max Lat Range|
        ///   |:----------|:----------------|:----------------|:----------------|:-------------|
        ///   |0          | 56.25     | 360.0       | 30.1105585173    | 180.0        |
        ///   |1          | 28.125         | 360.0       | 14.87468995    | 180.0        |
        ///   |2         | 14.063       | 351.5625      |  7.4130741851    | 137.9576312246       |
        ///   |3     | 7.03125     | 175.78125    |  3.7034501005    |   73.6354071932     |
        ///   |4     | 3.515625    | 87.890625  | 1.8513375155  | 35.4776115315  |
        ///   |5          | 1.7578125  | 43.9453125  | 0.925620264 | 17.4589959239  |
        ///   |6       | 0.87890625 | 21.97265625 | 0.4628040687  | 8.6907788223  |
        ///   |7      | 0.439453125 |  10.986328125 | 0.2314012764  | 4.3404320789 |
        ///   |8       | 0.2197265625 | 5.4931640625 | 0.1157005434  | 2.1695927024  |
        ///   |9      | 0.1098632812 | 2.7465820312  |  0.0578502599  | 1.0847183194  |
        ///   |10    | 0.0549316406  | 1.3732910156 | 0.0289251285  | 0.5423494021  |
        ///   |11     | 0.0274658203 | 0.6866455078 | 0.014462564 | 0.2711734813 |
        ///   |12      |  0.0137329102  | 0.3433227539 | 0.007231282 | 0.1355865882  |
        ///   |13    | 0.0068664551 | 0.171661377 | 0.003615641 | 0.067793275 |
        ///   |14     |  0.0034332275  | 0.0858306885 | 0.0018078205 | 0.0338966351 |
        ///   |15     |  0.0017166138 | 0.0429153442 | 0.0009039102 | 0.0169483173 |
        ///   |16   |  0.0008583069  | 0.0214576721  |  0.0004519551  | 0.0084741586 |
        ///   |17  | 0.0004291534 |  0.0107288361  |  0.0002259776 | 0.0042370793 |
        ///   |18    | 0.0002145767 | 0.005364418 | 0.0001129888 | 0.0021185396 |
        ///   |19    | 0.0001072884  |  0.002682209  | 5.64944E-05 | 0.0010592698 |
        ///   |20    |  5.36442E-05  | 0.0013411045 | 2.82472E-05  | 0.0005296349 |
        ///
        /// _Note_ : Either **center** or **bbox** parameter must be supplied to the API.
        /// </summary>
        /// <param name="tilesetId"> Map style to be returned. Possible values are microsoft.base.road, microsoft.base.darkgrey, and microsoft.imagery.  Default value is set to be microsoft.base.road. For more information, see [Render TilesetId](https://learn.microsoft.com/en-us/rest/api/maps/render/get-map-tileset?view=rest-maps-2023-06-01&amp;tabs=HTTP#tilesetid). </param>
        /// <param name="trafficLayer"> Optional Value, indicating no traffic flow overlaid on the image result. Possible values are microsoft.traffic.relative.main and none. Default value is none, indicating no traffic flow returned. If traffic related tilesetId is provided, will return map image with corresponding traffic layer. For more information, see [Render TilesetId](https://learn.microsoft.com/en-us/rest/api/maps/render/get-map-tileset?view=rest-maps-2023-06-01&amp;tabs=HTTP#tilesetid). </param>
        /// <param name="zoom"> Desired zoom level of the map. Support zoom value range from 0-20 (inclusive) for tilesetId being microsoft.base.road or microsoft.base.darkgrey. Support zoom value range from 0-19 (inclusive) for tilesetId being microsoft.imagery. Default value is 12.&lt;br&gt;&lt;br&gt;For more information, see [Zoom Levels and Tile Grid](https://docs.microsoft.com/azure/location-based-services/zoom-levels-and-tile-grid). </param>
        /// <param name="center">
        /// Coordinates of the center point in double. Format: 'lon,lat'. Longitude range: -180 to 180. Latitude range: -90 to 90.
        ///
        /// Note: Either center or bbox are required parameters. They are
        /// mutually exclusive.
        /// </param>
        /// <param name="boundingBoxPrivate">
        /// A bounding box is defined by two latitudes and two longitudes that represent the four sides of a rectangular area on the Earth. Format : 'minLon, minLat,
        /// maxLon, maxLat' (in double).
        ///
        /// Note: Either bbox or center are required
        /// parameters. They are mutually exclusive. bbox shouldn’t be used with
        /// height or width.
        ///
        /// The maximum and minimum allowed ranges for Lat and Lon are defined for each zoom level
        /// in the table at the top of this page.
        /// </param>
        /// <param name="height">
        /// Height of the resulting image in pixels. Range from 80 to 1500. Default
        /// is 512. It shouldn’t be used with bbox.
        /// </param>
        /// <param name="width"> Width of the resulting image in pixels. Range from 80 to 2000. Default is 512. It should not be used with bbox. </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="localizedMapView">
        /// The View parameter (also called the "user region" parameter) allows you to show the correct maps for a certain country/region for geopolitically disputed regions. Different countries/regions have different views of such regions, and the View parameter allows your application to comply with the view required by the country/region your application will be serving. By default, the View parameter is set to “Unified” even if you haven’t defined it in  the request. It is your responsibility to determine the location of your users, and then set the View parameter correctly for that location. Alternatively, you have the option to set ‘View=Auto’, which will return the map data based on the IP  address of the request. The View parameter in Azure Maps must be used in compliance with applicable laws, including those  regarding mapping, of the country/region where maps, images and other data and third party content that you are authorized to  access via Azure Maps is made available. Example: view=IN.
        ///
        /// Please refer to [Supported Views](https://aka.ms/AzureMapsLocalizationViews) for details and to see the available Views.
        /// </param>
        /// <param name="pins">
        /// Pushpin style and instances. Use this parameter to optionally add pushpins to the image.
        /// The pushpin style describes the appearance of the pushpins, and the instances specify
        /// the coordinates of the pushpins (in double) and optional labels for each pin. (Be sure to properly URL-encode values of this
        /// parameter since it will contain reserved characters such as pipes and punctuation.)
        ///
        /// The Azure Maps account S0 SKU only supports a single instance of the pins parameter. Other SKUs
        /// allow multiple instances of the pins parameter to specify multiple pin styles.
        ///
        /// To render a pushpin at latitude 45°N and longitude 122°W using the default built-in pushpin style, add the
        /// querystring parameter
        ///
        /// `pins=default||-122 45`
        ///
        /// Note that the longitude comes before the latitude.
        /// After URL encoding this will look like
        ///
        /// `pins=default%7C%7C-122+45`
        ///
        /// All of the examples here show the pins
        /// parameter without URL encoding, for clarity.
        ///
        /// To render a pin at multiple locations, separate each location with a pipe character. For example, use
        ///
        /// `pins=default||-122 45|-119.5 43.2|-121.67 47.12`
        ///
        /// The S0 Azure Maps account SKU only allows five pushpins. Other account SKUs do not have this limitation.
        ///
        /// ### Style Modifiers
        ///
        /// You can modify the appearance of the pins by adding style modifiers. These are added after the style but before
        /// the locations and labels. Style modifiers each have a two-letter name. These abbreviated names are used to help
        /// reduce the length of the URL.
        ///
        /// To change the color of the pushpin, use the 'co' style modifier and specify the color using the HTML/CSS RGB color
        /// format which is a six-digit hexadecimal number (the three-digit form is not supported). For example, to use
        /// a deep pink color which you would specify as #FF1493 in CSS, use
        ///
        /// `pins=default|coFF1493||-122 45`
        ///
        /// ### Pushpin Labels
        ///
        /// To add a label to the pins, put the label in single quotes just before the coordinates. Avoid using special character such as `|` or `||` in label. For example, to label
        /// three pins with the values '1', '2', and '3', use
        ///
        /// `pins=default||'1'-122 45|'2'-119.5 43.2|'3'-121.67 47.12`
        ///
        /// There is a built-in pushpin style called 'none' that does not display a pushpin image. You can use this if
        /// you want to display labels without any pin image. For example,
        ///
        /// `pins=none||'A'-122 45|'B'-119.5 43.2`
        ///
        /// To change the color of the pushpin labels, use the 'lc' label color style modifier. For example, to use pink
        /// pushpins with black labels, use
        ///
        /// `pins=default|coFF1493|lc000000||-122 45`
        ///
        /// To change the size of the labels, use the 'ls' label size style modifier. The label size represents the approximate
        /// height of the label text in pixels. For example, to increase the label size to 12, use
        ///
        /// `pins=default|ls12||'A'-122 45|'B'-119 43`
        ///
        /// The labels are centered at the pushpin 'label anchor.' The anchor location is predefined for built-in pushpins and
        /// is at the top center of custom pushpins (see below). To override the label anchor, using the 'la' style modifier
        /// and provide X and Y pixel coordinates for the anchor. These coordinates are relative to the top left corner of the
        /// pushpin image. Positive X values move the anchor to the right, and positive Y values move the anchor down. For example,
        /// to position the label anchor 10 pixels right and 4 pixels above the top left corner of the pushpin image,
        /// use
        ///
        /// `pins=default|la10 -4||'A'-122 45|'B'-119 43`
        ///
        /// ### Custom Pushpins
        ///
        /// To use a custom pushpin image, use the word 'custom' as the pin style name, and then specify a URL after the
        /// location and label information. The maximum allowed size for a customized label image is 65,536 pixels. Use two pipe characters to indicate that you're done specifying locations and are
        /// starting the URL. For example,
        ///
        /// `pins=custom||-122 45||http://contoso.com/pushpins/red.png`
        ///
        /// After URL encoding, this would look like
        ///
        /// `pins=custom%7C%7C-122+45%7C%7Chttp%3A%2F%2Fcontoso.com%2Fpushpins%2Fred.png`
        ///
        /// By default, custom pushpin images are drawn centered at the pin coordinates. This usually isn't ideal as it obscures
        /// the location that you're trying to highlight. To override the anchor location of the pin image, use the 'an'
        /// style modifier. This uses the same format as the 'la' label anchor style modifier. For example, if your custom
        /// pin image has the tip of the pin at the top left corner of the image, you can set the anchor to that spot by
        /// using
        ///
        /// `pins=custom|an0 0||-122 45||http://contoso.com/pushpins/red.png`
        ///
        /// Note: If you use the 'co' color modifier with a custom pushpin image, the specified color will replace the RGB
        /// channels of the pixels in the image but will leave the alpha (opacity) channel unchanged. This would usually
        /// only be done with a solid-color custom image.
        ///
        /// ### Scale, Rotation, and Opacity
        ///
        /// You can make pushpins and their labels larger or smaller by using the 'sc' scale style modifier. This is a
        /// value greater than zero. A value of 1 is the standard scale. Values larger than 1 will make the pins larger, and
        /// values smaller than 1 will make them smaller. For example, to draw the pushpins 50% larger than normal, use
        ///
        /// `pins=default|sc1.5||-122 45`
        ///
        /// You can rotate pushpins and their labels by using the 'ro' rotation style modifier. This is a number of degrees
        /// of clockwise rotation. Use a negative number to rotate counter-clockwise. For example, to rotate the pushpins
        /// 90 degrees clockwise and double their size, use
        ///
        /// `pins=default|ro90|sc2||-122 45`
        ///
        /// You can make pushpins and their labels partially transparent by specifying the 'al' alpha style modifier.
        /// This is a number between 0 and 1 indicating the opacity of the pushpins. Zero makes them completely transparent
        /// (and not visible) and 1 makes them completely opaque (which is the default). For example, to make pushpins
        /// and their labels only 67% opaque, use
        ///
        /// `pins=default|al.67||-122 45`
        ///
        /// ### Style Modifier Summary
        ///
        /// Modifier  | Description    | Type    | Range
        /// :--------:|---------------|--------|----------
        /// al        | Alpha (opacity) |  float | 0 to 1
        /// an        | Pin anchor    | &lt;int32, int32&gt;  | *
        /// co        | Pin color      | string | 000000 to FFFFFF
        /// la        | Label anchor   | &lt;int32, int32&gt; | *
        /// lc        | Label color   | string  | 000000 to FFFFFF
        /// ls        | Label size      | float | Greater than 0
        /// ro        | Rotation    | float    | -360 to 360
        /// sc        | Scale         | float  | Greater than 0
        ///
        /// * X and Y coordinates can be anywhere within pin image or a margin around it.
        /// The margin size is the minimum of the pin width and height.
        /// </param>
        /// <param name="path">
        /// Path style and locations (in double). Use this parameter to optionally add lines, polygons or circles to the image.
        /// The path style describes the appearance of the line and fill. (Be sure to properly URL-encode values of this
        /// parameter since it will contain reserved characters such as pipes and punctuation.)
        ///
        /// Path parameter is supported in Azure Maps account SKU starting with S1. Multiple instances of the path parameter
        /// allow to specify multiple geometries with their styles. Number of parameters per request is limited to 10 and
        /// number of locations is limited to 100 per path.
        ///
        /// To render a circle with radius 100 meters and center point at latitude 45°N and longitude 122°W using the default style, add the
        /// querystring parameter
        ///
        /// `path=ra100||-122 45`
        ///
        /// Note that the longitude comes before the latitude.
        /// After URL encoding this will look like
        ///
        /// `path=ra100%7C%7C-122+45`
        ///
        /// All of the examples here show the path parameter without URL encoding, for clarity.
        ///
        /// To render a line, separate each location with a pipe character. For example, use
        ///
        /// `path=||-122 45|-119.5 43.2|-121.67 47.12`
        ///
        /// A polygon is specified with a closed path, where the first and last points are equal. For example, use
        ///
        /// `path=||-122 45|-119.5 43.2|-121.67 47.12|-122 45`
        ///
        /// Longitude value for locations of lines and polygons can be in the range from -360 to 360 to allow for rendering of geometries crossing the anti-meridian.
        ///
        /// ### Style Modifiers
        ///
        /// You can modify the appearance of the path by adding style modifiers. These are added before the locations.
        /// Style modifiers each have a two-letter name. These abbreviated names are used to help reduce the length
        /// of the URL.
        ///
        /// To change the color of the outline, use the 'lc' style modifier and specify the color using the HTML/CSS RGB color
        /// format which is a six-digit hexadecimal number (the three-digit form is not supported). For example, to use
        /// a deep pink color which you would specify as #FF1493 in CSS, use
        ///
        /// `path=lcFF1493||-122 45|-119.5 43.2`
        ///
        /// Multiple style modifiers may be combined to create a more complex visual style.
        ///
        /// `lc0000FF|lw3|la0.60|fa0.50||-122.2 47.6|-122.2 47.7|-122.3 47.7|-122.3 47.6|-122.2 47.6`
        ///
        /// ### Style Modifier Summary
        ///
        /// Modifier  | Description       | Type     | Range
        /// :--------:|---------------|--------|----------
        /// lc        | Line color      | string       | 000000 to FFFFFF
        /// fc        | Fill color        | string       | 000000 to FFFFFF
        /// la    | Line alpha (opacity)     |  float     | 0 to 1
        /// fa  | Fill alpha (opacity)   |     float       | 0 to 1
        /// lw   | Line width     |int32        | (0, 50]
        /// ra        | Circle radius (meters) |   float  | Greater than 0
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<ResponseWithHeaders<Stream, RenderGetMapStaticImageHeaders>> GetMapStaticImageAsync(MapTileSetId? tilesetId = null, TrafficTilesetId? trafficLayer = null, int? zoom = null, IEnumerable<double> center = null, IEnumerable<double> boundingBoxPrivate = null, int? height = null, int? width = null, string language = null, LocalizedMapView? localizedMapView = null, IEnumerable<string> pins = null, IEnumerable<string> path = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetMapStaticImageRequest(tilesetId, trafficLayer, zoom, center, boundingBoxPrivate, height, width, language, localizedMapView, pins, path);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new RenderGetMapStaticImageHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        var value = message.ExtractResponseContent();
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary>
        /// This rendering API produces static, rasterized map views of a user-defined area. It's suitable for lightweight web applications, when the desired user experience doesn't require interactive map controls, or when bandwidth is limited. This API is also useful for embedding maps in applications outside of the browser, in backend services, report generation, or desktop applications.
        ///
        ///  This API includes parameters for basic data visualization:
        ///
        /// - Labeled pushpins in multiple styles.
        /// - Render circle, path, and polygon geometry types.
        ///
        /// For more information and detailed examples, see [Render custom data on a raster map](/azure/azure-maps/how-to-render-custom-data).
        /// &lt;br&gt;&lt;br&gt;
        /// The dimensions of the bbox parameter are constrained, depending on the zoom level. This ensures the resulting image has an appropriate level of detail.
        /// &lt;br&gt;&lt;br&gt;
        ///
        ///   |Zoom Level | Min Lon Range   | Max Lon Range   | Min Lat Range| Max Lat Range|
        ///   |:----------|:----------------|:----------------|:----------------|:-------------|
        ///   |0          | 56.25     | 360.0       | 30.1105585173    | 180.0        |
        ///   |1          | 28.125         | 360.0       | 14.87468995    | 180.0        |
        ///   |2         | 14.063       | 351.5625      |  7.4130741851    | 137.9576312246       |
        ///   |3     | 7.03125     | 175.78125    |  3.7034501005    |   73.6354071932     |
        ///   |4     | 3.515625    | 87.890625  | 1.8513375155  | 35.4776115315  |
        ///   |5          | 1.7578125  | 43.9453125  | 0.925620264 | 17.4589959239  |
        ///   |6       | 0.87890625 | 21.97265625 | 0.4628040687  | 8.6907788223  |
        ///   |7      | 0.439453125 |  10.986328125 | 0.2314012764  | 4.3404320789 |
        ///   |8       | 0.2197265625 | 5.4931640625 | 0.1157005434  | 2.1695927024  |
        ///   |9      | 0.1098632812 | 2.7465820312  |  0.0578502599  | 1.0847183194  |
        ///   |10    | 0.0549316406  | 1.3732910156 | 0.0289251285  | 0.5423494021  |
        ///   |11     | 0.0274658203 | 0.6866455078 | 0.014462564 | 0.2711734813 |
        ///   |12      |  0.0137329102  | 0.3433227539 | 0.007231282 | 0.1355865882  |
        ///   |13    | 0.0068664551 | 0.171661377 | 0.003615641 | 0.067793275 |
        ///   |14     |  0.0034332275  | 0.0858306885 | 0.0018078205 | 0.0338966351 |
        ///   |15     |  0.0017166138 | 0.0429153442 | 0.0009039102 | 0.0169483173 |
        ///   |16   |  0.0008583069  | 0.0214576721  |  0.0004519551  | 0.0084741586 |
        ///   |17  | 0.0004291534 |  0.0107288361  |  0.0002259776 | 0.0042370793 |
        ///   |18    | 0.0002145767 | 0.005364418 | 0.0001129888 | 0.0021185396 |
        ///   |19    | 0.0001072884  |  0.002682209  | 5.64944E-05 | 0.0010592698 |
        ///   |20    |  5.36442E-05  | 0.0013411045 | 2.82472E-05  | 0.0005296349 |
        ///
        /// _Note_ : Either **center** or **bbox** parameter must be supplied to the API.
        /// </summary>
        /// <param name="tilesetId"> Map style to be returned. Possible values are microsoft.base.road, microsoft.base.darkgrey, and microsoft.imagery.  Default value is set to be microsoft.base.road. For more information, see [Render TilesetId](https://learn.microsoft.com/en-us/rest/api/maps/render/get-map-tileset?view=rest-maps-2023-06-01&amp;tabs=HTTP#tilesetid). </param>
        /// <param name="trafficLayer"> Optional Value, indicating no traffic flow overlaid on the image result. Possible values are microsoft.traffic.relative.main and none. Default value is none, indicating no traffic flow returned. If traffic related tilesetId is provided, will return map image with corresponding traffic layer. For more information, see [Render TilesetId](https://learn.microsoft.com/en-us/rest/api/maps/render/get-map-tileset?view=rest-maps-2023-06-01&amp;tabs=HTTP#tilesetid). </param>
        /// <param name="zoom"> Desired zoom level of the map. Support zoom value range from 0-20 (inclusive) for tilesetId being microsoft.base.road or microsoft.base.darkgrey. Support zoom value range from 0-19 (inclusive) for tilesetId being microsoft.imagery. Default value is 12.&lt;br&gt;&lt;br&gt;For more information, see [Zoom Levels and Tile Grid](https://docs.microsoft.com/azure/location-based-services/zoom-levels-and-tile-grid). </param>
        /// <param name="center">
        /// Coordinates of the center point in double. Format: 'lon,lat'. Longitude range: -180 to 180. Latitude range: -90 to 90.
        ///
        /// Note: Either center or bbox are required parameters. They are
        /// mutually exclusive.
        /// </param>
        /// <param name="boundingBoxPrivate">
        /// A bounding box is defined by two latitudes and two longitudes that represent the four sides of a rectangular area on the Earth. Format : 'minLon, minLat,
        /// maxLon, maxLat' (in double).
        ///
        /// Note: Either bbox or center are required
        /// parameters. They are mutually exclusive. bbox shouldn’t be used with
        /// height or width.
        ///
        /// The maximum and minimum allowed ranges for Lat and Lon are defined for each zoom level
        /// in the table at the top of this page.
        /// </param>
        /// <param name="height">
        /// Height of the resulting image in pixels. Range from 80 to 1500. Default
        /// is 512. It shouldn’t be used with bbox.
        /// </param>
        /// <param name="width"> Width of the resulting image in pixels. Range from 80 to 2000. Default is 512. It should not be used with bbox. </param>
        /// <param name="language">
        /// Language in which search results should be returned. Should be one of supported IETF language tags, case insensitive. When data in specified language is not available for a specific field, default language is used.
        ///
        /// Please refer to [Supported Languages](https://docs.microsoft.com/azure/azure-maps/supported-languages) for details.
        /// </param>
        /// <param name="localizedMapView">
        /// The View parameter (also called the "user region" parameter) allows you to show the correct maps for a certain country/region for geopolitically disputed regions. Different countries/regions have different views of such regions, and the View parameter allows your application to comply with the view required by the country/region your application will be serving. By default, the View parameter is set to “Unified” even if you haven’t defined it in  the request. It is your responsibility to determine the location of your users, and then set the View parameter correctly for that location. Alternatively, you have the option to set ‘View=Auto’, which will return the map data based on the IP  address of the request. The View parameter in Azure Maps must be used in compliance with applicable laws, including those  regarding mapping, of the country/region where maps, images and other data and third party content that you are authorized to  access via Azure Maps is made available. Example: view=IN.
        ///
        /// Please refer to [Supported Views](https://aka.ms/AzureMapsLocalizationViews) for details and to see the available Views.
        /// </param>
        /// <param name="pins">
        /// Pushpin style and instances. Use this parameter to optionally add pushpins to the image.
        /// The pushpin style describes the appearance of the pushpins, and the instances specify
        /// the coordinates of the pushpins (in double) and optional labels for each pin. (Be sure to properly URL-encode values of this
        /// parameter since it will contain reserved characters such as pipes and punctuation.)
        ///
        /// The Azure Maps account S0 SKU only supports a single instance of the pins parameter. Other SKUs
        /// allow multiple instances of the pins parameter to specify multiple pin styles.
        ///
        /// To render a pushpin at latitude 45°N and longitude 122°W using the default built-in pushpin style, add the
        /// querystring parameter
        ///
        /// `pins=default||-122 45`
        ///
        /// Note that the longitude comes before the latitude.
        /// After URL encoding this will look like
        ///
        /// `pins=default%7C%7C-122+45`
        ///
        /// All of the examples here show the pins
        /// parameter without URL encoding, for clarity.
        ///
        /// To render a pin at multiple locations, separate each location with a pipe character. For example, use
        ///
        /// `pins=default||-122 45|-119.5 43.2|-121.67 47.12`
        ///
        /// The S0 Azure Maps account SKU only allows five pushpins. Other account SKUs do not have this limitation.
        ///
        /// ### Style Modifiers
        ///
        /// You can modify the appearance of the pins by adding style modifiers. These are added after the style but before
        /// the locations and labels. Style modifiers each have a two-letter name. These abbreviated names are used to help
        /// reduce the length of the URL.
        ///
        /// To change the color of the pushpin, use the 'co' style modifier and specify the color using the HTML/CSS RGB color
        /// format which is a six-digit hexadecimal number (the three-digit form is not supported). For example, to use
        /// a deep pink color which you would specify as #FF1493 in CSS, use
        ///
        /// `pins=default|coFF1493||-122 45`
        ///
        /// ### Pushpin Labels
        ///
        /// To add a label to the pins, put the label in single quotes just before the coordinates. Avoid using special character such as `|` or `||` in label. For example, to label
        /// three pins with the values '1', '2', and '3', use
        ///
        /// `pins=default||'1'-122 45|'2'-119.5 43.2|'3'-121.67 47.12`
        ///
        /// There is a built-in pushpin style called 'none' that does not display a pushpin image. You can use this if
        /// you want to display labels without any pin image. For example,
        ///
        /// `pins=none||'A'-122 45|'B'-119.5 43.2`
        ///
        /// To change the color of the pushpin labels, use the 'lc' label color style modifier. For example, to use pink
        /// pushpins with black labels, use
        ///
        /// `pins=default|coFF1493|lc000000||-122 45`
        ///
        /// To change the size of the labels, use the 'ls' label size style modifier. The label size represents the approximate
        /// height of the label text in pixels. For example, to increase the label size to 12, use
        ///
        /// `pins=default|ls12||'A'-122 45|'B'-119 43`
        ///
        /// The labels are centered at the pushpin 'label anchor.' The anchor location is predefined for built-in pushpins and
        /// is at the top center of custom pushpins (see below). To override the label anchor, using the 'la' style modifier
        /// and provide X and Y pixel coordinates for the anchor. These coordinates are relative to the top left corner of the
        /// pushpin image. Positive X values move the anchor to the right, and positive Y values move the anchor down. For example,
        /// to position the label anchor 10 pixels right and 4 pixels above the top left corner of the pushpin image,
        /// use
        ///
        /// `pins=default|la10 -4||'A'-122 45|'B'-119 43`
        ///
        /// ### Custom Pushpins
        ///
        /// To use a custom pushpin image, use the word 'custom' as the pin style name, and then specify a URL after the
        /// location and label information. The maximum allowed size for a customized label image is 65,536 pixels. Use two pipe characters to indicate that you're done specifying locations and are
        /// starting the URL. For example,
        ///
        /// `pins=custom||-122 45||http://contoso.com/pushpins/red.png`
        ///
        /// After URL encoding, this would look like
        ///
        /// `pins=custom%7C%7C-122+45%7C%7Chttp%3A%2F%2Fcontoso.com%2Fpushpins%2Fred.png`
        ///
        /// By default, custom pushpin images are drawn centered at the pin coordinates. This usually isn't ideal as it obscures
        /// the location that you're trying to highlight. To override the anchor location of the pin image, use the 'an'
        /// style modifier. This uses the same format as the 'la' label anchor style modifier. For example, if your custom
        /// pin image has the tip of the pin at the top left corner of the image, you can set the anchor to that spot by
        /// using
        ///
        /// `pins=custom|an0 0||-122 45||http://contoso.com/pushpins/red.png`
        ///
        /// Note: If you use the 'co' color modifier with a custom pushpin image, the specified color will replace the RGB
        /// channels of the pixels in the image but will leave the alpha (opacity) channel unchanged. This would usually
        /// only be done with a solid-color custom image.
        ///
        /// ### Scale, Rotation, and Opacity
        ///
        /// You can make pushpins and their labels larger or smaller by using the 'sc' scale style modifier. This is a
        /// value greater than zero. A value of 1 is the standard scale. Values larger than 1 will make the pins larger, and
        /// values smaller than 1 will make them smaller. For example, to draw the pushpins 50% larger than normal, use
        ///
        /// `pins=default|sc1.5||-122 45`
        ///
        /// You can rotate pushpins and their labels by using the 'ro' rotation style modifier. This is a number of degrees
        /// of clockwise rotation. Use a negative number to rotate counter-clockwise. For example, to rotate the pushpins
        /// 90 degrees clockwise and double their size, use
        ///
        /// `pins=default|ro90|sc2||-122 45`
        ///
        /// You can make pushpins and their labels partially transparent by specifying the 'al' alpha style modifier.
        /// This is a number between 0 and 1 indicating the opacity of the pushpins. Zero makes them completely transparent
        /// (and not visible) and 1 makes them completely opaque (which is the default). For example, to make pushpins
        /// and their labels only 67% opaque, use
        ///
        /// `pins=default|al.67||-122 45`
        ///
        /// ### Style Modifier Summary
        ///
        /// Modifier  | Description    | Type    | Range
        /// :--------:|---------------|--------|----------
        /// al        | Alpha (opacity) |  float | 0 to 1
        /// an        | Pin anchor    | &lt;int32, int32&gt;  | *
        /// co        | Pin color      | string | 000000 to FFFFFF
        /// la        | Label anchor   | &lt;int32, int32&gt; | *
        /// lc        | Label color   | string  | 000000 to FFFFFF
        /// ls        | Label size      | float | Greater than 0
        /// ro        | Rotation    | float    | -360 to 360
        /// sc        | Scale         | float  | Greater than 0
        ///
        /// * X and Y coordinates can be anywhere within pin image or a margin around it.
        /// The margin size is the minimum of the pin width and height.
        /// </param>
        /// <param name="path">
        /// Path style and locations (in double). Use this parameter to optionally add lines, polygons or circles to the image.
        /// The path style describes the appearance of the line and fill. (Be sure to properly URL-encode values of this
        /// parameter since it will contain reserved characters such as pipes and punctuation.)
        ///
        /// Path parameter is supported in Azure Maps account SKU starting with S1. Multiple instances of the path parameter
        /// allow to specify multiple geometries with their styles. Number of parameters per request is limited to 10 and
        /// number of locations is limited to 100 per path.
        ///
        /// To render a circle with radius 100 meters and center point at latitude 45°N and longitude 122°W using the default style, add the
        /// querystring parameter
        ///
        /// `path=ra100||-122 45`
        ///
        /// Note that the longitude comes before the latitude.
        /// After URL encoding this will look like
        ///
        /// `path=ra100%7C%7C-122+45`
        ///
        /// All of the examples here show the path parameter without URL encoding, for clarity.
        ///
        /// To render a line, separate each location with a pipe character. For example, use
        ///
        /// `path=||-122 45|-119.5 43.2|-121.67 47.12`
        ///
        /// A polygon is specified with a closed path, where the first and last points are equal. For example, use
        ///
        /// `path=||-122 45|-119.5 43.2|-121.67 47.12|-122 45`
        ///
        /// Longitude value for locations of lines and polygons can be in the range from -360 to 360 to allow for rendering of geometries crossing the anti-meridian.
        ///
        /// ### Style Modifiers
        ///
        /// You can modify the appearance of the path by adding style modifiers. These are added before the locations.
        /// Style modifiers each have a two-letter name. These abbreviated names are used to help reduce the length
        /// of the URL.
        ///
        /// To change the color of the outline, use the 'lc' style modifier and specify the color using the HTML/CSS RGB color
        /// format which is a six-digit hexadecimal number (the three-digit form is not supported). For example, to use
        /// a deep pink color which you would specify as #FF1493 in CSS, use
        ///
        /// `path=lcFF1493||-122 45|-119.5 43.2`
        ///
        /// Multiple style modifiers may be combined to create a more complex visual style.
        ///
        /// `lc0000FF|lw3|la0.60|fa0.50||-122.2 47.6|-122.2 47.7|-122.3 47.7|-122.3 47.6|-122.2 47.6`
        ///
        /// ### Style Modifier Summary
        ///
        /// Modifier  | Description       | Type     | Range
        /// :--------:|---------------|--------|----------
        /// lc        | Line color      | string       | 000000 to FFFFFF
        /// fc        | Fill color        | string       | 000000 to FFFFFF
        /// la    | Line alpha (opacity)     |  float     | 0 to 1
        /// fa  | Fill alpha (opacity)   |     float       | 0 to 1
        /// lw   | Line width     |int32        | (0, 50]
        /// ra        | Circle radius (meters) |   float  | Greater than 0
        /// </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public ResponseWithHeaders<Stream, RenderGetMapStaticImageHeaders> GetMapStaticImage(MapTileSetId? tilesetId = null, TrafficTilesetId? trafficLayer = null, int? zoom = null, IEnumerable<double> center = null, IEnumerable<double> boundingBoxPrivate = null, int? height = null, int? width = null, string language = null, LocalizedMapView? localizedMapView = null, IEnumerable<string> pins = null, IEnumerable<string> path = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetMapStaticImageRequest(tilesetId, trafficLayer, zoom, center, boundingBoxPrivate, height, width, language, localizedMapView, pins, path);
            _pipeline.Send(message, cancellationToken);
            var headers = new RenderGetMapStaticImageHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        var value = message.ExtractResponseContent();
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetCopyrightFromBoundingBoxRequest(ResponseFormat format, BoundingBox boundingBox, IncludeText? includeText)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/map/copyright/bounding/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (boundingBox?.SouthWest != null && !(boundingBox?.SouthWest is Common.ChangeTrackingList<double> changeTrackingList && changeTrackingList.IsUndefined))
            {
                uri.AppendQueryDelimited("mincoordinates", boundingBox.SouthWest, ",", true);
            }
            if (boundingBox?.NorthEast != null && !(boundingBox?.NorthEast is Common.ChangeTrackingList<double> changeTrackingList0 && changeTrackingList0.IsUndefined))
            {
                uri.AppendQueryDelimited("maxcoordinates", boundingBox.NorthEast, ",", true);
            }
            if (includeText != null)
            {
                uri.AppendQuery("text", includeText.Value.ToString(), true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get copyright information for the specified bounding box. </summary>
        /// <param name="format"> Desired format of the response. Value can be either _json_ or _xml_. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="boundingBox"> Parameter group. </param>
        /// <param name="includeText"> Yes/no value to exclude textual data from response. Only images and country/region names will be in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="boundingBox"/> is null. </exception>
        /// <remarks>
        ///
        /// Returns copyright information for a given bounding box. Bounding-box requests should specify the minimum and maximum longitude and latitude (EPSG-3857) coordinates
        /// </remarks>
        public async Task<Response<RenderCopyright>> GetCopyrightFromBoundingBoxAsync(ResponseFormat format, BoundingBox boundingBox, IncludeText? includeText = null, CancellationToken cancellationToken = default)
        {
            if (boundingBox == null)
            {
                throw new ArgumentNullException(nameof(boundingBox));
            }

            using var message = CreateGetCopyrightFromBoundingBoxRequest(format, boundingBox, includeText);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        RenderCopyright value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = RenderCopyright.DeserializeRenderCopyright(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get copyright information for the specified bounding box. </summary>
        /// <param name="format"> Desired format of the response. Value can be either _json_ or _xml_. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="boundingBox"> Parameter group. </param>
        /// <param name="includeText"> Yes/no value to exclude textual data from response. Only images and country/region names will be in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="boundingBox"/> is null. </exception>
        /// <remarks>
        ///
        /// Returns copyright information for a given bounding box. Bounding-box requests should specify the minimum and maximum longitude and latitude (EPSG-3857) coordinates
        /// </remarks>
        public Response<RenderCopyright> GetCopyrightFromBoundingBox(ResponseFormat format, BoundingBox boundingBox, IncludeText? includeText = null, CancellationToken cancellationToken = default)
        {
            if (boundingBox == null)
            {
                throw new ArgumentNullException(nameof(boundingBox));
            }

            using var message = CreateGetCopyrightFromBoundingBoxRequest(format, boundingBox, includeText);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        RenderCopyright value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = RenderCopyright.DeserializeRenderCopyright(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetCopyrightForTileRequest(ResponseFormat format, MapTileIndex mapTileIndex, IncludeText? includeText)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/map/copyright/tile/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            uri.AppendQuery("zoom", mapTileIndex.Z, true);
            uri.AppendQuery("x", mapTileIndex.X, true);
            uri.AppendQuery("y", mapTileIndex.Y, true);
            if (includeText != null)
            {
                uri.AppendQuery("text", includeText.Value.ToString(), true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get copyright information. </summary>
        /// <param name="format"> Desired format of the response. Value can be either _json_ or _xml_. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="mapTileIndex"> Parameter group. </param>
        /// <param name="includeText"> Yes/no value to exclude textual data from response. Only images and country/region names will be in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="mapTileIndex"/> is null. </exception>
        /// <remarks>
        ///
        /// To obtain the copyright information for a particular tile, the request should specify the tile's zoom level and x and y coordinates. For more information, see [Zoom Levels and Tile Grid](/azure/azure-maps/zoom-levels-and-tile-grid).
        ///
        /// Copyrights API is designed to serve copyright information for Render service. In addition to basic copyright for the whole map, API is serving specific groups of copyrights for some countries/regions.
        /// </remarks>
        public async Task<Response<RenderCopyright>> GetCopyrightForTileAsync(ResponseFormat format, MapTileIndex mapTileIndex, IncludeText? includeText = null, CancellationToken cancellationToken = default)
        {
            if (mapTileIndex == null)
            {
                throw new ArgumentNullException(nameof(mapTileIndex));
            }

            using var message = CreateGetCopyrightForTileRequest(format, mapTileIndex, includeText);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        RenderCopyright value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = RenderCopyright.DeserializeRenderCopyright(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get copyright information. </summary>
        /// <param name="format"> Desired format of the response. Value can be either _json_ or _xml_. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="mapTileIndex"> Parameter group. </param>
        /// <param name="includeText"> Yes/no value to exclude textual data from response. Only images and country/region names will be in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="mapTileIndex"/> is null. </exception>
        /// <remarks>
        ///
        /// To obtain the copyright information for a particular tile, the request should specify the tile's zoom level and x and y coordinates. For more information, see [Zoom Levels and Tile Grid](/azure/azure-maps/zoom-levels-and-tile-grid).
        ///
        /// Copyrights API is designed to serve copyright information for Render service. In addition to basic copyright for the whole map, API is serving specific groups of copyrights for some countries/regions.
        /// </remarks>
        public Response<RenderCopyright> GetCopyrightForTile(ResponseFormat format, MapTileIndex mapTileIndex, IncludeText? includeText = null, CancellationToken cancellationToken = default)
        {
            if (mapTileIndex == null)
            {
                throw new ArgumentNullException(nameof(mapTileIndex));
            }

            using var message = CreateGetCopyrightForTileRequest(format, mapTileIndex, includeText);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        RenderCopyright value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = RenderCopyright.DeserializeRenderCopyright(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetCopyrightForWorldRequest(ResponseFormat format, IncludeText? includeText)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.Reset(_endpoint);
            uri.AppendPath("/map/copyright/world/", false);
            uri.AppendPath(format.ToString(), true);
            uri.AppendQuery("api-version", _apiVersion, true);
            if (includeText != null)
            {
                uri.AppendQuery("text", includeText.Value.ToString(), true);
            }
            request.Uri = uri;
            if (_clientId != null)
            {
                request.Headers.Add("x-ms-client-id", _clientId);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Use to get copyright information for for the world. </summary>
        /// <param name="format"> Desired format of the response. Value can be either _json_ or _xml_. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="includeText"> Yes/no value to exclude textual data from response. Only images and country/region names will be in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// Returns the copyright information for the world. To obtain the default copyright information for the whole world, don't specify a tile or bounding box.
        ///
        /// Copyrights API is designed to serve copyright information for Render service. In addition to basic copyright for the whole map, API is serving specific groups of copyrights for some countries/regions.
        /// </remarks>
        public async Task<Response<RenderCopyright>> GetCopyrightForWorldAsync(ResponseFormat format, IncludeText? includeText = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetCopyrightForWorldRequest(format, includeText);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        RenderCopyright value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = RenderCopyright.DeserializeRenderCopyright(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Use to get copyright information for for the world. </summary>
        /// <param name="format"> Desired format of the response. Value can be either _json_ or _xml_. The default value is AutoRest.CSharp.Output.Models.Types.EnumTypeValue. </param>
        /// <param name="includeText"> Yes/no value to exclude textual data from response. Only images and country/region names will be in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks>
        ///
        /// Returns the copyright information for the world. To obtain the default copyright information for the whole world, don't specify a tile or bounding box.
        ///
        /// Copyrights API is designed to serve copyright information for Render service. In addition to basic copyright for the whole map, API is serving specific groups of copyrights for some countries/regions.
        /// </remarks>
        public Response<RenderCopyright> GetCopyrightForWorld(ResponseFormat format, IncludeText? includeText = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetCopyrightForWorldRequest(format, includeText);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        RenderCopyright value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = RenderCopyright.DeserializeRenderCopyright(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }
    }
}
