// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

// <auto-generated/>

#nullable disable

using System;
using System.Collections.Generic;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using Azure.Core;
using Azure.Core.Pipeline;
using Azure.Security.KeyVault.Storage.Models;

namespace Azure.Security.KeyVault.Storage
{
    internal partial class ManagedStorageRestClient
    {
        private readonly HttpPipeline _pipeline;
        private readonly string _vaultBaseUrl;
        private readonly string _apiVersion;

        /// <summary> The ClientDiagnostics is used to provide tracing support for the client library. </summary>
        internal ClientDiagnostics ClientDiagnostics { get; }

        /// <summary> Initializes a new instance of ManagedStorageRestClient. </summary>
        /// <param name="clientDiagnostics"> The handler for diagnostic messaging in the client. </param>
        /// <param name="pipeline"> The HTTP pipeline for sending and receiving REST requests and responses. </param>
        /// <param name="vaultBaseUrl"> The vault name, for example https://myvault.vault.azure.net. </param>
        /// <param name="apiVersion"> Api Version. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="clientDiagnostics"/>, <paramref name="pipeline"/>, <paramref name="vaultBaseUrl"/> or <paramref name="apiVersion"/> is null. </exception>
        public ManagedStorageRestClient(ClientDiagnostics clientDiagnostics, HttpPipeline pipeline, string vaultBaseUrl, string apiVersion = "7.1")
        {
            ClientDiagnostics = clientDiagnostics ?? throw new ArgumentNullException(nameof(clientDiagnostics));
            _pipeline = pipeline ?? throw new ArgumentNullException(nameof(pipeline));
            _vaultBaseUrl = vaultBaseUrl ?? throw new ArgumentNullException(nameof(vaultBaseUrl));
            _apiVersion = apiVersion ?? throw new ArgumentNullException(nameof(apiVersion));
        }

        internal HttpMessage CreateGetStorageAccountsRequest(int? maxresults)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage", false);
            if (maxresults != null)
            {
                uri.AppendQuery("maxresults", maxresults.Value, true);
            }
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> List storage accounts managed by the specified key vault. This operation requires the storage/list permission. </summary>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<Response<StorageListResult>> GetStorageAccountsAsync(int? maxresults = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetStorageAccountsRequest(maxresults);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageListResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = StorageListResult.DeserializeStorageListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> List storage accounts managed by the specified key vault. This operation requires the storage/list permission. </summary>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public Response<StorageListResult> GetStorageAccounts(int? maxresults = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetStorageAccountsRequest(maxresults);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageListResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = StorageListResult.DeserializeStorageListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetDeletedStorageAccountsRequest(int? maxresults)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/deletedstorage", false);
            if (maxresults != null)
            {
                uri.AppendQuery("maxresults", maxresults.Value, true);
            }
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Lists deleted storage accounts for the specified vault. </summary>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks> The Get Deleted Storage Accounts operation returns the storage accounts that have been deleted for a vault enabled for soft-delete. This operation requires the storage/list permission. </remarks>
        public async Task<Response<DeletedStorageListResult>> GetDeletedStorageAccountsAsync(int? maxresults = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetDeletedStorageAccountsRequest(maxresults);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedStorageListResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DeletedStorageListResult.DeserializeDeletedStorageListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Lists deleted storage accounts for the specified vault. </summary>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <remarks> The Get Deleted Storage Accounts operation returns the storage accounts that have been deleted for a vault enabled for soft-delete. This operation requires the storage/list permission. </remarks>
        public Response<DeletedStorageListResult> GetDeletedStorageAccounts(int? maxresults = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetDeletedStorageAccountsRequest(maxresults);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedStorageListResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DeletedStorageListResult.DeserializeDeletedStorageListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetDeletedStorageAccountRequest(string storageAccountName)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/deletedstorage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Gets the specified deleted storage account. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        /// <remarks> The Get Deleted Storage Account operation returns the specified deleted storage account along with its attributes. This operation requires the storage/get permission. </remarks>
        public async Task<Response<DeletedStorageBundle>> GetDeletedStorageAccountAsync(string storageAccountName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateGetDeletedStorageAccountRequest(storageAccountName);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedStorageBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DeletedStorageBundle.DeserializeDeletedStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Gets the specified deleted storage account. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        /// <remarks> The Get Deleted Storage Account operation returns the specified deleted storage account along with its attributes. This operation requires the storage/get permission. </remarks>
        public Response<DeletedStorageBundle> GetDeletedStorageAccount(string storageAccountName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateGetDeletedStorageAccountRequest(storageAccountName);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedStorageBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DeletedStorageBundle.DeserializeDeletedStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreatePurgeDeletedStorageAccountRequest(string storageAccountName)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/deletedstorage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Permanently deletes the specified storage account. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        /// <remarks> The purge deleted storage account operation removes the secret permanently, without the possibility of recovery. This operation can only be performed on a soft-delete enabled vault. This operation requires the storage/purge permission. </remarks>
        public async Task<Response> PurgeDeletedStorageAccountAsync(string storageAccountName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreatePurgeDeletedStorageAccountRequest(storageAccountName);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 204:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Permanently deletes the specified storage account. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        /// <remarks> The purge deleted storage account operation removes the secret permanently, without the possibility of recovery. This operation can only be performed on a soft-delete enabled vault. This operation requires the storage/purge permission. </remarks>
        public Response PurgeDeletedStorageAccount(string storageAccountName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreatePurgeDeletedStorageAccountRequest(storageAccountName);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 204:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateRecoverDeletedStorageAccountRequest(string storageAccountName)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/deletedstorage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendPath("/recover", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Recovers the deleted storage account. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        /// <remarks> Recovers the deleted storage account in the specified vault. This operation can only be performed on a soft-delete enabled vault. This operation requires the storage/recover permission. </remarks>
        public async Task<Response<StorageBundle>> RecoverDeletedStorageAccountAsync(string storageAccountName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateRecoverDeletedStorageAccountRequest(storageAccountName);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = StorageBundle.DeserializeStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Recovers the deleted storage account. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        /// <remarks> Recovers the deleted storage account in the specified vault. This operation can only be performed on a soft-delete enabled vault. This operation requires the storage/recover permission. </remarks>
        public Response<StorageBundle> RecoverDeletedStorageAccount(string storageAccountName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateRecoverDeletedStorageAccountRequest(storageAccountName);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = StorageBundle.DeserializeStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateBackupStorageAccountRequest(string storageAccountName)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendPath("/backup", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Backs up the specified storage account. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        /// <remarks> Requests that a backup of the specified storage account be downloaded to the client. This operation requires the storage/backup permission. </remarks>
        public async Task<Response<BackupStorageResult>> BackupStorageAccountAsync(string storageAccountName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateBackupStorageAccountRequest(storageAccountName);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        BackupStorageResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = BackupStorageResult.DeserializeBackupStorageResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Backs up the specified storage account. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        /// <remarks> Requests that a backup of the specified storage account be downloaded to the client. This operation requires the storage/backup permission. </remarks>
        public Response<BackupStorageResult> BackupStorageAccount(string storageAccountName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateBackupStorageAccountRequest(storageAccountName);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        BackupStorageResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = BackupStorageResult.DeserializeBackupStorageResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateRestoreStorageAccountRequest(byte[] storageBundleBackup)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage/restore", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/json");
            var model = new StorageRestoreParameters(storageBundleBackup);
            var content = new Utf8JsonRequestContent();
            content.JsonWriter.WriteObjectValue(model);
            request.Content = content;
            return message;
        }

        /// <summary> Restores a backed up storage account to a vault. </summary>
        /// <param name="storageBundleBackup"> The backup blob associated with a storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageBundleBackup"/> is null. </exception>
        /// <remarks> Restores a backed up storage account to a vault. This operation requires the storage/restore permission. </remarks>
        public async Task<Response<StorageBundle>> RestoreStorageAccountAsync(byte[] storageBundleBackup, CancellationToken cancellationToken = default)
        {
            if (storageBundleBackup == null)
            {
                throw new ArgumentNullException(nameof(storageBundleBackup));
            }

            using var message = CreateRestoreStorageAccountRequest(storageBundleBackup);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = StorageBundle.DeserializeStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Restores a backed up storage account to a vault. </summary>
        /// <param name="storageBundleBackup"> The backup blob associated with a storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageBundleBackup"/> is null. </exception>
        /// <remarks> Restores a backed up storage account to a vault. This operation requires the storage/restore permission. </remarks>
        public Response<StorageBundle> RestoreStorageAccount(byte[] storageBundleBackup, CancellationToken cancellationToken = default)
        {
            if (storageBundleBackup == null)
            {
                throw new ArgumentNullException(nameof(storageBundleBackup));
            }

            using var message = CreateRestoreStorageAccountRequest(storageBundleBackup);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = StorageBundle.DeserializeStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateDeleteStorageAccountRequest(string storageAccountName)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Deletes a storage account. This operation requires the storage/delete permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        public async Task<Response<DeletedStorageBundle>> DeleteStorageAccountAsync(string storageAccountName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateDeleteStorageAccountRequest(storageAccountName);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedStorageBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DeletedStorageBundle.DeserializeDeletedStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Deletes a storage account. This operation requires the storage/delete permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        public Response<DeletedStorageBundle> DeleteStorageAccount(string storageAccountName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateDeleteStorageAccountRequest(storageAccountName);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedStorageBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DeletedStorageBundle.DeserializeDeletedStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetStorageAccountRequest(string storageAccountName)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Gets information about a specified storage account. This operation requires the storage/get permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        public async Task<Response<StorageBundle>> GetStorageAccountAsync(string storageAccountName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateGetStorageAccountRequest(storageAccountName);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = StorageBundle.DeserializeStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Gets information about a specified storage account. This operation requires the storage/get permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        public Response<StorageBundle> GetStorageAccount(string storageAccountName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateGetStorageAccountRequest(storageAccountName);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = StorageBundle.DeserializeStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateSetStorageAccountRequest(string storageAccountName, string resourceId, string activeKeyName, bool autoRegenerateKey, string regenerationPeriod, StorageAccountAttributes storageAccountAttributes, IDictionary<string, string> tags)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/json");
            StorageAccountCreateParameters storageAccountCreateParameters = new StorageAccountCreateParameters(resourceId, activeKeyName, autoRegenerateKey)
            {
                RegenerationPeriod = regenerationPeriod,
                StorageAccountAttributes = storageAccountAttributes
            };
            if (tags != null)
            {
                foreach (var value in tags)
                {
                    storageAccountCreateParameters.Tags.Add(value);
                }
            }
            var model = storageAccountCreateParameters;
            var content = new Utf8JsonRequestContent();
            content.JsonWriter.WriteObjectValue(model);
            request.Content = content;
            return message;
        }

        /// <summary> Creates or updates a new storage account. This operation requires the storage/set permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="resourceId"> Storage account resource id. </param>
        /// <param name="activeKeyName"> Current active storage account key name. </param>
        /// <param name="autoRegenerateKey"> whether keyvault should manage the storage account for the user. </param>
        /// <param name="regenerationPeriod"> The key regeneration time duration specified in ISO-8601 format. </param>
        /// <param name="storageAccountAttributes"> The attributes of the storage account. </param>
        /// <param name="tags"> Application specific metadata in the form of key-value pairs. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/>, <paramref name="resourceId"/> or <paramref name="activeKeyName"/> is null. </exception>
        public async Task<Response<StorageBundle>> SetStorageAccountAsync(string storageAccountName, string resourceId, string activeKeyName, bool autoRegenerateKey, string regenerationPeriod = null, StorageAccountAttributes storageAccountAttributes = null, IDictionary<string, string> tags = null, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (resourceId == null)
            {
                throw new ArgumentNullException(nameof(resourceId));
            }
            if (activeKeyName == null)
            {
                throw new ArgumentNullException(nameof(activeKeyName));
            }

            using var message = CreateSetStorageAccountRequest(storageAccountName, resourceId, activeKeyName, autoRegenerateKey, regenerationPeriod, storageAccountAttributes, tags);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = StorageBundle.DeserializeStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Creates or updates a new storage account. This operation requires the storage/set permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="resourceId"> Storage account resource id. </param>
        /// <param name="activeKeyName"> Current active storage account key name. </param>
        /// <param name="autoRegenerateKey"> whether keyvault should manage the storage account for the user. </param>
        /// <param name="regenerationPeriod"> The key regeneration time duration specified in ISO-8601 format. </param>
        /// <param name="storageAccountAttributes"> The attributes of the storage account. </param>
        /// <param name="tags"> Application specific metadata in the form of key-value pairs. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/>, <paramref name="resourceId"/> or <paramref name="activeKeyName"/> is null. </exception>
        public Response<StorageBundle> SetStorageAccount(string storageAccountName, string resourceId, string activeKeyName, bool autoRegenerateKey, string regenerationPeriod = null, StorageAccountAttributes storageAccountAttributes = null, IDictionary<string, string> tags = null, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (resourceId == null)
            {
                throw new ArgumentNullException(nameof(resourceId));
            }
            if (activeKeyName == null)
            {
                throw new ArgumentNullException(nameof(activeKeyName));
            }

            using var message = CreateSetStorageAccountRequest(storageAccountName, resourceId, activeKeyName, autoRegenerateKey, regenerationPeriod, storageAccountAttributes, tags);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = StorageBundle.DeserializeStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateUpdateStorageAccountRequest(string storageAccountName, string activeKeyName, bool? autoRegenerateKey, string regenerationPeriod, StorageAccountAttributes storageAccountAttributes, IDictionary<string, string> tags)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Patch;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/json");
            StorageAccountUpdateParameters storageAccountUpdateParameters = new StorageAccountUpdateParameters()
            {
                ActiveKeyName = activeKeyName,
                AutoRegenerateKey = autoRegenerateKey,
                RegenerationPeriod = regenerationPeriod,
                StorageAccountAttributes = storageAccountAttributes
            };
            if (tags != null)
            {
                foreach (var value in tags)
                {
                    storageAccountUpdateParameters.Tags.Add(value);
                }
            }
            var model = storageAccountUpdateParameters;
            var content = new Utf8JsonRequestContent();
            content.JsonWriter.WriteObjectValue(model);
            request.Content = content;
            return message;
        }

        /// <summary> Updates the specified attributes associated with the given storage account. This operation requires the storage/set/update permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="activeKeyName"> The current active storage account key name. </param>
        /// <param name="autoRegenerateKey"> whether keyvault should manage the storage account for the user. </param>
        /// <param name="regenerationPeriod"> The key regeneration time duration specified in ISO-8601 format. </param>
        /// <param name="storageAccountAttributes"> The attributes of the storage account. </param>
        /// <param name="tags"> Application specific metadata in the form of key-value pairs. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        public async Task<Response<StorageBundle>> UpdateStorageAccountAsync(string storageAccountName, string activeKeyName = null, bool? autoRegenerateKey = null, string regenerationPeriod = null, StorageAccountAttributes storageAccountAttributes = null, IDictionary<string, string> tags = null, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateUpdateStorageAccountRequest(storageAccountName, activeKeyName, autoRegenerateKey, regenerationPeriod, storageAccountAttributes, tags);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = StorageBundle.DeserializeStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Updates the specified attributes associated with the given storage account. This operation requires the storage/set/update permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="activeKeyName"> The current active storage account key name. </param>
        /// <param name="autoRegenerateKey"> whether keyvault should manage the storage account for the user. </param>
        /// <param name="regenerationPeriod"> The key regeneration time duration specified in ISO-8601 format. </param>
        /// <param name="storageAccountAttributes"> The attributes of the storage account. </param>
        /// <param name="tags"> Application specific metadata in the form of key-value pairs. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        public Response<StorageBundle> UpdateStorageAccount(string storageAccountName, string activeKeyName = null, bool? autoRegenerateKey = null, string regenerationPeriod = null, StorageAccountAttributes storageAccountAttributes = null, IDictionary<string, string> tags = null, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateUpdateStorageAccountRequest(storageAccountName, activeKeyName, autoRegenerateKey, regenerationPeriod, storageAccountAttributes, tags);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = StorageBundle.DeserializeStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateRegenerateStorageAccountKeyRequest(string storageAccountName, string keyName)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendPath("/regeneratekey", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/json");
            var model = new StorageAccountRegenerteKeyParameters(keyName);
            var content = new Utf8JsonRequestContent();
            content.JsonWriter.WriteObjectValue(model);
            request.Content = content;
            return message;
        }

        /// <summary> Regenerates the specified key value for the given storage account. This operation requires the storage/regeneratekey permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="keyName"> The storage account key name. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> or <paramref name="keyName"/> is null. </exception>
        public async Task<Response<StorageBundle>> RegenerateStorageAccountKeyAsync(string storageAccountName, string keyName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (keyName == null)
            {
                throw new ArgumentNullException(nameof(keyName));
            }

            using var message = CreateRegenerateStorageAccountKeyRequest(storageAccountName, keyName);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = StorageBundle.DeserializeStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Regenerates the specified key value for the given storage account. This operation requires the storage/regeneratekey permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="keyName"> The storage account key name. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> or <paramref name="keyName"/> is null. </exception>
        public Response<StorageBundle> RegenerateStorageAccountKey(string storageAccountName, string keyName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (keyName == null)
            {
                throw new ArgumentNullException(nameof(keyName));
            }

            using var message = CreateRegenerateStorageAccountKeyRequest(storageAccountName, keyName);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = StorageBundle.DeserializeStorageBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetSasDefinitionsRequest(string storageAccountName, int? maxresults)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendPath("/sas", false);
            if (maxresults != null)
            {
                uri.AppendQuery("maxresults", maxresults.Value, true);
            }
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> List storage SAS definitions for the given storage account. This operation requires the storage/listsas permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        public async Task<Response<SasDefinitionListResult>> GetSasDefinitionsAsync(string storageAccountName, int? maxresults = null, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateGetSasDefinitionsRequest(storageAccountName, maxresults);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SasDefinitionListResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = SasDefinitionListResult.DeserializeSasDefinitionListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> List storage SAS definitions for the given storage account. This operation requires the storage/listsas permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        public Response<SasDefinitionListResult> GetSasDefinitions(string storageAccountName, int? maxresults = null, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateGetSasDefinitionsRequest(storageAccountName, maxresults);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SasDefinitionListResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = SasDefinitionListResult.DeserializeSasDefinitionListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetDeletedSasDefinitionsRequest(string storageAccountName, int? maxresults)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/deletedstorage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendPath("/sas", false);
            if (maxresults != null)
            {
                uri.AppendQuery("maxresults", maxresults.Value, true);
            }
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Lists deleted SAS definitions for the specified vault and storage account. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        /// <remarks> The Get Deleted Sas Definitions operation returns the SAS definitions that have been deleted for a vault enabled for soft-delete. This operation requires the storage/listsas permission. </remarks>
        public async Task<Response<DeletedSasDefinitionListResult>> GetDeletedSasDefinitionsAsync(string storageAccountName, int? maxresults = null, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateGetDeletedSasDefinitionsRequest(storageAccountName, maxresults);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedSasDefinitionListResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DeletedSasDefinitionListResult.DeserializeDeletedSasDefinitionListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Lists deleted SAS definitions for the specified vault and storage account. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> is null. </exception>
        /// <remarks> The Get Deleted Sas Definitions operation returns the SAS definitions that have been deleted for a vault enabled for soft-delete. This operation requires the storage/listsas permission. </remarks>
        public Response<DeletedSasDefinitionListResult> GetDeletedSasDefinitions(string storageAccountName, int? maxresults = null, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateGetDeletedSasDefinitionsRequest(storageAccountName, maxresults);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedSasDefinitionListResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DeletedSasDefinitionListResult.DeserializeDeletedSasDefinitionListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetDeletedSasDefinitionRequest(string storageAccountName, string sasDefinitionName)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/deletedstorage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendPath("/sas/", false);
            uri.AppendPath(sasDefinitionName, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Gets the specified deleted sas definition. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="sasDefinitionName"> The name of the SAS definition. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> or <paramref name="sasDefinitionName"/> is null. </exception>
        /// <remarks> The Get Deleted SAS Definition operation returns the specified deleted SAS definition along with its attributes. This operation requires the storage/getsas permission. </remarks>
        public async Task<Response<DeletedSasDefinitionBundle>> GetDeletedSasDefinitionAsync(string storageAccountName, string sasDefinitionName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (sasDefinitionName == null)
            {
                throw new ArgumentNullException(nameof(sasDefinitionName));
            }

            using var message = CreateGetDeletedSasDefinitionRequest(storageAccountName, sasDefinitionName);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedSasDefinitionBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DeletedSasDefinitionBundle.DeserializeDeletedSasDefinitionBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Gets the specified deleted sas definition. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="sasDefinitionName"> The name of the SAS definition. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> or <paramref name="sasDefinitionName"/> is null. </exception>
        /// <remarks> The Get Deleted SAS Definition operation returns the specified deleted SAS definition along with its attributes. This operation requires the storage/getsas permission. </remarks>
        public Response<DeletedSasDefinitionBundle> GetDeletedSasDefinition(string storageAccountName, string sasDefinitionName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (sasDefinitionName == null)
            {
                throw new ArgumentNullException(nameof(sasDefinitionName));
            }

            using var message = CreateGetDeletedSasDefinitionRequest(storageAccountName, sasDefinitionName);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedSasDefinitionBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DeletedSasDefinitionBundle.DeserializeDeletedSasDefinitionBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateRecoverDeletedSasDefinitionRequest(string storageAccountName, string sasDefinitionName)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/deletedstorage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendPath("/sas/", false);
            uri.AppendPath(sasDefinitionName, true);
            uri.AppendPath("/recover", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Recovers the deleted SAS definition. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="sasDefinitionName"> The name of the SAS definition. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> or <paramref name="sasDefinitionName"/> is null. </exception>
        /// <remarks> Recovers the deleted SAS definition for the specified storage account. This operation can only be performed on a soft-delete enabled vault. This operation requires the storage/recover permission. </remarks>
        public async Task<Response<SasDefinitionBundle>> RecoverDeletedSasDefinitionAsync(string storageAccountName, string sasDefinitionName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (sasDefinitionName == null)
            {
                throw new ArgumentNullException(nameof(sasDefinitionName));
            }

            using var message = CreateRecoverDeletedSasDefinitionRequest(storageAccountName, sasDefinitionName);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SasDefinitionBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = SasDefinitionBundle.DeserializeSasDefinitionBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Recovers the deleted SAS definition. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="sasDefinitionName"> The name of the SAS definition. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> or <paramref name="sasDefinitionName"/> is null. </exception>
        /// <remarks> Recovers the deleted SAS definition for the specified storage account. This operation can only be performed on a soft-delete enabled vault. This operation requires the storage/recover permission. </remarks>
        public Response<SasDefinitionBundle> RecoverDeletedSasDefinition(string storageAccountName, string sasDefinitionName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (sasDefinitionName == null)
            {
                throw new ArgumentNullException(nameof(sasDefinitionName));
            }

            using var message = CreateRecoverDeletedSasDefinitionRequest(storageAccountName, sasDefinitionName);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SasDefinitionBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = SasDefinitionBundle.DeserializeSasDefinitionBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateDeleteSasDefinitionRequest(string storageAccountName, string sasDefinitionName)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendPath("/sas/", false);
            uri.AppendPath(sasDefinitionName, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Deletes a SAS definition from a specified storage account. This operation requires the storage/deletesas permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="sasDefinitionName"> The name of the SAS definition. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> or <paramref name="sasDefinitionName"/> is null. </exception>
        public async Task<Response<DeletedSasDefinitionBundle>> DeleteSasDefinitionAsync(string storageAccountName, string sasDefinitionName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (sasDefinitionName == null)
            {
                throw new ArgumentNullException(nameof(sasDefinitionName));
            }

            using var message = CreateDeleteSasDefinitionRequest(storageAccountName, sasDefinitionName);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedSasDefinitionBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DeletedSasDefinitionBundle.DeserializeDeletedSasDefinitionBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Deletes a SAS definition from a specified storage account. This operation requires the storage/deletesas permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="sasDefinitionName"> The name of the SAS definition. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> or <paramref name="sasDefinitionName"/> is null. </exception>
        public Response<DeletedSasDefinitionBundle> DeleteSasDefinition(string storageAccountName, string sasDefinitionName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (sasDefinitionName == null)
            {
                throw new ArgumentNullException(nameof(sasDefinitionName));
            }

            using var message = CreateDeleteSasDefinitionRequest(storageAccountName, sasDefinitionName);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedSasDefinitionBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DeletedSasDefinitionBundle.DeserializeDeletedSasDefinitionBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetSasDefinitionRequest(string storageAccountName, string sasDefinitionName)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendPath("/sas/", false);
            uri.AppendPath(sasDefinitionName, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Gets information about a SAS definition for the specified storage account. This operation requires the storage/getsas permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="sasDefinitionName"> The name of the SAS definition. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> or <paramref name="sasDefinitionName"/> is null. </exception>
        public async Task<Response<SasDefinitionBundle>> GetSasDefinitionAsync(string storageAccountName, string sasDefinitionName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (sasDefinitionName == null)
            {
                throw new ArgumentNullException(nameof(sasDefinitionName));
            }

            using var message = CreateGetSasDefinitionRequest(storageAccountName, sasDefinitionName);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SasDefinitionBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = SasDefinitionBundle.DeserializeSasDefinitionBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Gets information about a SAS definition for the specified storage account. This operation requires the storage/getsas permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="sasDefinitionName"> The name of the SAS definition. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> or <paramref name="sasDefinitionName"/> is null. </exception>
        public Response<SasDefinitionBundle> GetSasDefinition(string storageAccountName, string sasDefinitionName, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (sasDefinitionName == null)
            {
                throw new ArgumentNullException(nameof(sasDefinitionName));
            }

            using var message = CreateGetSasDefinitionRequest(storageAccountName, sasDefinitionName);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SasDefinitionBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = SasDefinitionBundle.DeserializeSasDefinitionBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateSetSasDefinitionRequest(string storageAccountName, string sasDefinitionName, string templateUri, SasTokenType sasType, string validityPeriod, SasDefinitionAttributes sasDefinitionAttributes, IDictionary<string, string> tags)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendPath("/sas/", false);
            uri.AppendPath(sasDefinitionName, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/json");
            SasDefinitionCreateParameters sasDefinitionCreateParameters = new SasDefinitionCreateParameters(templateUri, sasType, validityPeriod)
            {
                SasDefinitionAttributes = sasDefinitionAttributes
            };
            if (tags != null)
            {
                foreach (var value in tags)
                {
                    sasDefinitionCreateParameters.Tags.Add(value);
                }
            }
            var model = sasDefinitionCreateParameters;
            var content = new Utf8JsonRequestContent();
            content.JsonWriter.WriteObjectValue(model);
            request.Content = content;
            return message;
        }

        /// <summary> Creates or updates a new SAS definition for the specified storage account. This operation requires the storage/setsas permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="sasDefinitionName"> The name of the SAS definition. </param>
        /// <param name="templateUri"> The SAS definition token template signed with an arbitrary key.  Tokens created according to the SAS definition will have the same properties as the template. </param>
        /// <param name="sasType"> The type of SAS token the SAS definition will create. </param>
        /// <param name="validityPeriod"> The validity period of SAS tokens created according to the SAS definition. </param>
        /// <param name="sasDefinitionAttributes"> The attributes of the SAS definition. </param>
        /// <param name="tags"> Application specific metadata in the form of key-value pairs. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/>, <paramref name="sasDefinitionName"/>, <paramref name="templateUri"/> or <paramref name="validityPeriod"/> is null. </exception>
        public async Task<Response<SasDefinitionBundle>> SetSasDefinitionAsync(string storageAccountName, string sasDefinitionName, string templateUri, SasTokenType sasType, string validityPeriod, SasDefinitionAttributes sasDefinitionAttributes = null, IDictionary<string, string> tags = null, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (sasDefinitionName == null)
            {
                throw new ArgumentNullException(nameof(sasDefinitionName));
            }
            if (templateUri == null)
            {
                throw new ArgumentNullException(nameof(templateUri));
            }
            if (validityPeriod == null)
            {
                throw new ArgumentNullException(nameof(validityPeriod));
            }

            using var message = CreateSetSasDefinitionRequest(storageAccountName, sasDefinitionName, templateUri, sasType, validityPeriod, sasDefinitionAttributes, tags);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SasDefinitionBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = SasDefinitionBundle.DeserializeSasDefinitionBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Creates or updates a new SAS definition for the specified storage account. This operation requires the storage/setsas permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="sasDefinitionName"> The name of the SAS definition. </param>
        /// <param name="templateUri"> The SAS definition token template signed with an arbitrary key.  Tokens created according to the SAS definition will have the same properties as the template. </param>
        /// <param name="sasType"> The type of SAS token the SAS definition will create. </param>
        /// <param name="validityPeriod"> The validity period of SAS tokens created according to the SAS definition. </param>
        /// <param name="sasDefinitionAttributes"> The attributes of the SAS definition. </param>
        /// <param name="tags"> Application specific metadata in the form of key-value pairs. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/>, <paramref name="sasDefinitionName"/>, <paramref name="templateUri"/> or <paramref name="validityPeriod"/> is null. </exception>
        public Response<SasDefinitionBundle> SetSasDefinition(string storageAccountName, string sasDefinitionName, string templateUri, SasTokenType sasType, string validityPeriod, SasDefinitionAttributes sasDefinitionAttributes = null, IDictionary<string, string> tags = null, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (sasDefinitionName == null)
            {
                throw new ArgumentNullException(nameof(sasDefinitionName));
            }
            if (templateUri == null)
            {
                throw new ArgumentNullException(nameof(templateUri));
            }
            if (validityPeriod == null)
            {
                throw new ArgumentNullException(nameof(validityPeriod));
            }

            using var message = CreateSetSasDefinitionRequest(storageAccountName, sasDefinitionName, templateUri, sasType, validityPeriod, sasDefinitionAttributes, tags);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SasDefinitionBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = SasDefinitionBundle.DeserializeSasDefinitionBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateUpdateSasDefinitionRequest(string storageAccountName, string sasDefinitionName, string templateUri, SasTokenType? sasType, string validityPeriod, SasDefinitionAttributes sasDefinitionAttributes, IDictionary<string, string> tags)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Patch;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendPath("/storage/", false);
            uri.AppendPath(storageAccountName, true);
            uri.AppendPath("/sas/", false);
            uri.AppendPath(sasDefinitionName, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/json");
            SasDefinitionUpdateParameters sasDefinitionUpdateParameters = new SasDefinitionUpdateParameters()
            {
                TemplateUri = templateUri,
                SasType = sasType,
                ValidityPeriod = validityPeriod,
                SasDefinitionAttributes = sasDefinitionAttributes
            };
            if (tags != null)
            {
                foreach (var value in tags)
                {
                    sasDefinitionUpdateParameters.Tags.Add(value);
                }
            }
            var model = sasDefinitionUpdateParameters;
            var content = new Utf8JsonRequestContent();
            content.JsonWriter.WriteObjectValue(model);
            request.Content = content;
            return message;
        }

        /// <summary> Updates the specified attributes associated with the given SAS definition. This operation requires the storage/setsas permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="sasDefinitionName"> The name of the SAS definition. </param>
        /// <param name="templateUri"> The SAS definition token template signed with an arbitrary key.  Tokens created according to the SAS definition will have the same properties as the template. </param>
        /// <param name="sasType"> The type of SAS token the SAS definition will create. </param>
        /// <param name="validityPeriod"> The validity period of SAS tokens created according to the SAS definition. </param>
        /// <param name="sasDefinitionAttributes"> The attributes of the SAS definition. </param>
        /// <param name="tags"> Application specific metadata in the form of key-value pairs. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> or <paramref name="sasDefinitionName"/> is null. </exception>
        public async Task<Response<SasDefinitionBundle>> UpdateSasDefinitionAsync(string storageAccountName, string sasDefinitionName, string templateUri = null, SasTokenType? sasType = null, string validityPeriod = null, SasDefinitionAttributes sasDefinitionAttributes = null, IDictionary<string, string> tags = null, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (sasDefinitionName == null)
            {
                throw new ArgumentNullException(nameof(sasDefinitionName));
            }

            using var message = CreateUpdateSasDefinitionRequest(storageAccountName, sasDefinitionName, templateUri, sasType, validityPeriod, sasDefinitionAttributes, tags);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SasDefinitionBundle value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = SasDefinitionBundle.DeserializeSasDefinitionBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Updates the specified attributes associated with the given SAS definition. This operation requires the storage/setsas permission. </summary>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="sasDefinitionName"> The name of the SAS definition. </param>
        /// <param name="templateUri"> The SAS definition token template signed with an arbitrary key.  Tokens created according to the SAS definition will have the same properties as the template. </param>
        /// <param name="sasType"> The type of SAS token the SAS definition will create. </param>
        /// <param name="validityPeriod"> The validity period of SAS tokens created according to the SAS definition. </param>
        /// <param name="sasDefinitionAttributes"> The attributes of the SAS definition. </param>
        /// <param name="tags"> Application specific metadata in the form of key-value pairs. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="storageAccountName"/> or <paramref name="sasDefinitionName"/> is null. </exception>
        public Response<SasDefinitionBundle> UpdateSasDefinition(string storageAccountName, string sasDefinitionName, string templateUri = null, SasTokenType? sasType = null, string validityPeriod = null, SasDefinitionAttributes sasDefinitionAttributes = null, IDictionary<string, string> tags = null, CancellationToken cancellationToken = default)
        {
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }
            if (sasDefinitionName == null)
            {
                throw new ArgumentNullException(nameof(sasDefinitionName));
            }

            using var message = CreateUpdateSasDefinitionRequest(storageAccountName, sasDefinitionName, templateUri, sasType, validityPeriod, sasDefinitionAttributes, tags);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SasDefinitionBundle value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = SasDefinitionBundle.DeserializeSasDefinitionBundle(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetStorageAccountsNextPageRequest(string nextLink, int? maxresults)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> List storage accounts managed by the specified key vault. This operation requires the storage/list permission. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> is null. </exception>
        public async Task<Response<StorageListResult>> GetStorageAccountsNextPageAsync(string nextLink, int? maxresults = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }

            using var message = CreateGetStorageAccountsNextPageRequest(nextLink, maxresults);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageListResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = StorageListResult.DeserializeStorageListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> List storage accounts managed by the specified key vault. This operation requires the storage/list permission. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> is null. </exception>
        public Response<StorageListResult> GetStorageAccountsNextPage(string nextLink, int? maxresults = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }

            using var message = CreateGetStorageAccountsNextPageRequest(nextLink, maxresults);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        StorageListResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = StorageListResult.DeserializeStorageListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetDeletedStorageAccountsNextPageRequest(string nextLink, int? maxresults)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Lists deleted storage accounts for the specified vault. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> is null. </exception>
        /// <remarks> The Get Deleted Storage Accounts operation returns the storage accounts that have been deleted for a vault enabled for soft-delete. This operation requires the storage/list permission. </remarks>
        public async Task<Response<DeletedStorageListResult>> GetDeletedStorageAccountsNextPageAsync(string nextLink, int? maxresults = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }

            using var message = CreateGetDeletedStorageAccountsNextPageRequest(nextLink, maxresults);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedStorageListResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DeletedStorageListResult.DeserializeDeletedStorageListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Lists deleted storage accounts for the specified vault. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> is null. </exception>
        /// <remarks> The Get Deleted Storage Accounts operation returns the storage accounts that have been deleted for a vault enabled for soft-delete. This operation requires the storage/list permission. </remarks>
        public Response<DeletedStorageListResult> GetDeletedStorageAccountsNextPage(string nextLink, int? maxresults = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }

            using var message = CreateGetDeletedStorageAccountsNextPageRequest(nextLink, maxresults);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedStorageListResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DeletedStorageListResult.DeserializeDeletedStorageListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetSasDefinitionsNextPageRequest(string nextLink, string storageAccountName, int? maxresults)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> List storage SAS definitions for the given storage account. This operation requires the storage/listsas permission. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="storageAccountName"/> is null. </exception>
        public async Task<Response<SasDefinitionListResult>> GetSasDefinitionsNextPageAsync(string nextLink, string storageAccountName, int? maxresults = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateGetSasDefinitionsNextPageRequest(nextLink, storageAccountName, maxresults);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SasDefinitionListResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = SasDefinitionListResult.DeserializeSasDefinitionListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> List storage SAS definitions for the given storage account. This operation requires the storage/listsas permission. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="storageAccountName"/> is null. </exception>
        public Response<SasDefinitionListResult> GetSasDefinitionsNextPage(string nextLink, string storageAccountName, int? maxresults = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateGetSasDefinitionsNextPageRequest(nextLink, storageAccountName, maxresults);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        SasDefinitionListResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = SasDefinitionListResult.DeserializeSasDefinitionListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetDeletedSasDefinitionsNextPageRequest(string nextLink, string storageAccountName, int? maxresults)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_vaultBaseUrl, false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Lists deleted SAS definitions for the specified vault and storage account. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="storageAccountName"/> is null. </exception>
        /// <remarks> The Get Deleted Sas Definitions operation returns the SAS definitions that have been deleted for a vault enabled for soft-delete. This operation requires the storage/listsas permission. </remarks>
        public async Task<Response<DeletedSasDefinitionListResult>> GetDeletedSasDefinitionsNextPageAsync(string nextLink, string storageAccountName, int? maxresults = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateGetDeletedSasDefinitionsNextPageRequest(nextLink, storageAccountName, maxresults);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedSasDefinitionListResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = DeletedSasDefinitionListResult.DeserializeDeletedSasDefinitionListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Lists deleted SAS definitions for the specified vault and storage account. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="storageAccountName"> The name of the storage account. </param>
        /// <param name="maxresults"> Maximum number of results to return in a page. If not specified the service will return up to 25 results. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="storageAccountName"/> is null. </exception>
        /// <remarks> The Get Deleted Sas Definitions operation returns the SAS definitions that have been deleted for a vault enabled for soft-delete. This operation requires the storage/listsas permission. </remarks>
        public Response<DeletedSasDefinitionListResult> GetDeletedSasDefinitionsNextPage(string nextLink, string storageAccountName, int? maxresults = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (storageAccountName == null)
            {
                throw new ArgumentNullException(nameof(storageAccountName));
            }

            using var message = CreateGetDeletedSasDefinitionsNextPageRequest(nextLink, storageAccountName, maxresults);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        DeletedSasDefinitionListResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = DeletedSasDefinitionListResult.DeserializeDeletedSasDefinitionListResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }
    }
}
