﻿// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

using System;
using System.IO;
using System.Threading.Tasks;
using Azure.Core;
using Azure.Security.KeyVault.Secrets;
using Azure.Storage.Blobs;

#pragma warning disable CS8892 // Method 'UserAuthenticationSnippets.Main()' will not be used as an entry point because a synchronous entry point 'AutoGeneratedProgram.Main(string[])' was found.

namespace Azure.Identity.Samples
{
    public class UserAuthenticationSnippets
    {
        #region Snippet:Identity_ClientSideUserAuthentication_Persist_TokenCache_AuthRecordPath

        private const string AUTH_RECORD_PATH = "./tokencache.bin";

        #endregion

        public void Identity_ClientSideUserAuthentication_SimpleInteractiveBrowser()
        {
            #region Snippet:Identity_ClientSideUserAuthentication_SimpleInteractiveBrowser

            var client = new SecretClient(
                new Uri("https://myvault.vault.azure.net/"),
                new InteractiveBrowserCredential()
            );

            #endregion
        }

        public void Identity_ClientSideUserAuthentication_SimpleDeviceCode()
        {
            #region Snippet:Identity_ClientSideUserAuthentication_SimpleDeviceCode

            var credential = new DeviceCodeCredential();

            var client = new BlobClient(
                new Uri("https://myaccount.blob.core.windows.net/mycontainer/myblob"),
                credential
            );

            #endregion
        }

        public async Task Identity_ClientSideUserAuthentication_DisableAutomaticAuthentication()
        {
            #region Snippet:Identity_ClientSideUserAuthentication_DisableAutomaticAuthentication

            var credential = new InteractiveBrowserCredential(
                new InteractiveBrowserCredentialOptions { DisableAutomaticAuthentication = true });

            await credential.AuthenticateAsync();

            var client = new SecretClient(new Uri("https://myvault.vault.azure.net/"), credential);

            #endregion

            #region Snippet:Identity_ClientSideUserAuthentication_DisableAutomaticAuthentication_ExHandling

            try
            {
                client.GetSecret("secret");
            }
            catch (AuthenticationRequiredException e)
            {
                await EnsureAnimationCompleteAsync();

                await credential.AuthenticateAsync(e.TokenRequestContext);

                client.GetSecret("secret");
            }

            #endregion
        }

        private Task EnsureAnimationCompleteAsync() => Task.CompletedTask;

        public static async Task<TokenCredential> GetUserCredentialAsync()
        {
            if (!File.Exists(AUTH_RECORD_PATH))
            {
                #region Snippet:Identity_ClientSideUserAuthentication_Persist_TokenCache

                var credential = new InteractiveBrowserCredential(
                    new InteractiveBrowserCredentialOptions { TokenCachePersistenceOptions = new TokenCachePersistenceOptions() });

                #endregion

                #region Snippet:Identity_ClientSideUserAuthentication_Persist_AuthRecord

                AuthenticationRecord authRecord = await credential.AuthenticateAsync();

                using (var authRecordStream = new FileStream(AUTH_RECORD_PATH, FileMode.Create, FileAccess.Write))
                {
                    await authRecord.SerializeAsync(authRecordStream);
                }

                #endregion

                return credential;
            }
            else
            {
                #region Snippet:Identity_ClientSideUserAuthentication_Persist_SilentAuth

                AuthenticationRecord authRecord;

                using (var authRecordStream = new FileStream(AUTH_RECORD_PATH, FileMode.Open, FileAccess.Read))
                {
                    authRecord = await AuthenticationRecord.DeserializeAsync(authRecordStream);
                }

                var credential = new InteractiveBrowserCredential(
                    new InteractiveBrowserCredentialOptions
                    {
                        TokenCachePersistenceOptions = new TokenCachePersistenceOptions(),
                        AuthenticationRecord = authRecord
                    });

                #endregion

                return credential;
            }
        }

        public static async Task Main()
        {
            #region Snippet:AuthenticationRecord_TokenCachePersistenceOptions

            const string TOKEN_CACHE_NAME = "MyTokenCache";
            InteractiveBrowserCredential credential;
            AuthenticationRecord authRecord;

            // Check if an AuthenticationRecord exists on disk.
            // If it does not exist, get one and serialize it to disk.
            // If it does exist, load it from disk and deserialize it.
            if (!File.Exists(AUTH_RECORD_PATH))
            {
                // Construct a credential with TokenCachePersistenceOptions specified to ensure that the token cache is persisted to disk.
                // We can also optionally specify a name for the cache to avoid having it cleared by other applications.
                credential = new InteractiveBrowserCredential(
                    new InteractiveBrowserCredentialOptions { TokenCachePersistenceOptions = new TokenCachePersistenceOptions { Name = TOKEN_CACHE_NAME } });

                // Call AuthenticateAsync to fetch a new AuthenticationRecord.
                authRecord = await credential.AuthenticateAsync();

                // Serialize the AuthenticationRecord to disk so that it can be re-used across executions of this initialization code.
                using var authRecordStream = new FileStream(AUTH_RECORD_PATH, FileMode.Create, FileAccess.Write);
                await authRecord.SerializeAsync(authRecordStream);
            }
            else
            {
                // Load the previously serialized AuthenticationRecord from disk and deserialize it.
                using var authRecordStream = new FileStream(AUTH_RECORD_PATH, FileMode.Open, FileAccess.Read);
                authRecord = await AuthenticationRecord.DeserializeAsync(authRecordStream);

                // Construct a new client with our TokenCachePersistenceOptions with the addition of the AuthenticationRecord property.
                // This tells the credential to use the same token cache in addition to which account to try and fetch from cache when GetToken is called.
                credential = new InteractiveBrowserCredential(
                    new InteractiveBrowserCredentialOptions
                    {
                        TokenCachePersistenceOptions = new TokenCachePersistenceOptions { Name = TOKEN_CACHE_NAME },
                        AuthenticationRecord = authRecord
                    });
            }

            // Construct our client with the credential which is connected to the token cache
            // with the capability of silent authentication for the account specified in the AuthenticationRecord.
            var client = new SecretClient(new Uri("https://myvault.vault.azure.net/"), credential);

            #endregion
        }
    }
}
