﻿// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

using System.Diagnostics;
using System.Linq;
using System.Text;
using Microsoft.CodeAnalysis;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.Text;

namespace EventGridSourceGenerator
{
    /// <summary>
    /// This class generates the SystemEventNames constant values as well as the SystemEventExtensions which contains a mapping
    /// from constant values to deserialization method for each system event.
    /// </summary>
    [Generator]
    internal class EventGridSourceGenerator : ISourceGenerator
    {
        private SourceVisitor _visitor;
        private bool _isSystemEventsLibrary;
        private const string Indent = "    ";

        public void Execute(GeneratorExecutionContext context)
        {
            _visitor = new SourceVisitor();
            _isSystemEventsLibrary = context.Compilation.AssemblyName == "Azure.Messaging.EventGrid.SystemEvents";
            var root = context.Compilation.GetSymbolsWithName(
                "SystemEvents",
                SymbolFilter.Namespace)
                .Single();
            _visitor.Visit(root);

            context.AddSource("SystemEventNames.cs", SourceText.From(ConstructSystemEventNames(), Encoding.UTF8));
            context.AddSource("SystemEventExtensions.cs", SourceText.From(ConstructSystemEventExtensions(), Encoding.UTF8));
        }

        public void Initialize(GeneratorInitializationContext context)
        {
            // Uncomment to debug
            //if (!Debugger.IsAttached)
            //{
            //    Debugger.Launch();
            //}
        }

        private string ConstructSystemEventNames()
        {
            string ns = _isSystemEventsLibrary ? "Azure.Messaging.EventGrid.SystemEvents" : "Azure.Messaging.EventGrid";
            var sourceBuilder = new StringBuilder(
$@"// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

// <auto-generated/>

using Azure.Messaging.EventGrid.SystemEvents;

namespace {ns}
{{
    /// <summary>
    ///  Represents the names of the various event types for the system events published to
    ///  Azure Event Grid.
    /// </summary>
    public static class SystemEventNames
    {{
");
            for (int i = 0; i < _visitor.SystemEvents.Count; i++)
            {
                if (i > 0)
                {
                    sourceBuilder.AppendLine();
                }
                SystemEventNode sysEvent = _visitor.SystemEvents[i];

                // Add the ref docs for each constant
                sourceBuilder.AppendLine($"{Indent}{Indent}/// <summary>");
                sourceBuilder.AppendLine(
                    !_isSystemEventsLibrary
                        ? $"{Indent}{Indent}/// The value of the Event Type stored in <see cref=\"EventGridEvent.EventType\"/> and <see cref=\"CloudEvent.Type\"/> "
                        : $"{Indent}{Indent}/// The value of the Event Type stored in <see cref=\"CloudEvent.Type\"/> ");

                sourceBuilder.AppendLine($"{Indent}{Indent}/// for the <see cref=\"{sysEvent.EventName}\"/> system event.");
                sourceBuilder.AppendLine($"{Indent}{Indent}/// </summary>");

                // Add the constant
                sourceBuilder.AppendLine($"{Indent}{Indent}public const string {sysEvent.EventConstantName} = {sysEvent.EventType};");
            }

            sourceBuilder.Append($@"{Indent}}}
}}");
            return sourceBuilder.ToString();
        }

        private string ConstructSystemEventExtensions()
        {
            var sourceBuilder = new StringBuilder(
                $@"// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

// <auto-generated/>

using System;
using System.Collections.Generic;
using System.Text.Json;
using Azure.Messaging.EventGrid.SystemEvents;
{(_isSystemEventsLibrary ? "using System.ClientModel.Primitives;" : string.Empty)}

namespace Azure.Messaging.EventGrid
{{
    internal class SystemEventExtensions
    {{
        public static object AsSystemEventData(string eventType, JsonElement data)
        {{
            if (s_systemEventDeserializers.TryGetValue(eventType, out Func<JsonElement,{(_isSystemEventsLibrary ? " ModelReaderWriterOptions," : string.Empty)} object> systemDeserializationFunction))
            {{
                return systemDeserializationFunction(data{(_isSystemEventsLibrary ? ", null" : string.Empty)});
            }}
            else
            {{
                return null;
            }}
        }}

        internal static readonly IReadOnlyDictionary<string, Func<JsonElement,{(_isSystemEventsLibrary ? " ModelReaderWriterOptions," : string.Empty)} object>> s_systemEventDeserializers = new Dictionary<string, Func<JsonElement,{(_isSystemEventsLibrary ? " ModelReaderWriterOptions," : string.Empty)} object>>(StringComparer.OrdinalIgnoreCase)
        {{
");
            foreach (SystemEventNode sysEvent in _visitor.SystemEvents)
            {
                // Add each an entry for each system event to the dictionary containing a mapping from constant name to deserialization method.
                sourceBuilder.AppendLine(
                    $"{Indent}{Indent}{Indent}{{ SystemEventNames.{sysEvent.EventConstantName}, {sysEvent.EventName}.{sysEvent.DeserializeMethod} }},");
            }
            sourceBuilder.Append($@"{Indent}{Indent}}};
{Indent}}}
}}");
            return sourceBuilder.ToString();
        }
    }
}
