// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

// <auto-generated/>

#nullable disable

using System;
using System.IO;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using Azure.Core;
using Azure.Core.Pipeline;

namespace Azure.Containers.ContainerRegistry
{
    internal partial class ContainerRegistryRestClient
    {
        private readonly HttpPipeline _pipeline;
        private readonly string _url;
        private readonly string _apiVersion;

        /// <summary> The ClientDiagnostics is used to provide tracing support for the client library. </summary>
        internal ClientDiagnostics ClientDiagnostics { get; }

        /// <summary> Initializes a new instance of ContainerRegistryRestClient. </summary>
        /// <param name="clientDiagnostics"> The handler for diagnostic messaging in the client. </param>
        /// <param name="pipeline"> The HTTP pipeline for sending and receiving REST requests and responses. </param>
        /// <param name="url"> Registry login URL. </param>
        /// <param name="apiVersion"> Api Version. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="clientDiagnostics"/>, <paramref name="pipeline"/>, <paramref name="url"/> or <paramref name="apiVersion"/> is null. </exception>
        public ContainerRegistryRestClient(ClientDiagnostics clientDiagnostics, HttpPipeline pipeline, string url, string apiVersion = "2021-07-01")
        {
            ClientDiagnostics = clientDiagnostics ?? throw new ArgumentNullException(nameof(clientDiagnostics));
            _pipeline = pipeline ?? throw new ArgumentNullException(nameof(pipeline));
            _url = url ?? throw new ArgumentNullException(nameof(url));
            _apiVersion = apiVersion ?? throw new ArgumentNullException(nameof(apiVersion));
        }

        internal HttpMessage CreateCheckDockerV2SupportRequest()
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/v2/", false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Tells whether this Docker Registry instance supports Docker Registry HTTP API v2. </summary>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<Response> CheckDockerV2SupportAsync(CancellationToken cancellationToken = default)
        {
            using var message = CreateCheckDockerV2SupportRequest();
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Tells whether this Docker Registry instance supports Docker Registry HTTP API v2. </summary>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public Response CheckDockerV2Support(CancellationToken cancellationToken = default)
        {
            using var message = CreateCheckDockerV2SupportRequest();
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetManifestRequest(string name, string reference, string accept)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/v2/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/manifests/", false);
            uri.AppendPath(reference, true);
            request.Uri = uri;
            if (accept != null)
            {
                request.Headers.Add("accept", accept);
            }
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Get the manifest identified by `name` and `reference` where `reference` can be a tag or digest. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="reference"> A tag or a digest, pointing to a specific image. </param>
        /// <param name="accept"> Accept header string delimited by comma. For example, application/vnd.docker.distribution.manifest.v2+json. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="reference"/> is null. </exception>
        public async Task<Response<ManifestWrapper>> GetManifestAsync(string name, string reference, string accept = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (reference == null)
            {
                throw new ArgumentNullException(nameof(reference));
            }

            using var message = CreateGetManifestRequest(name, reference, accept);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ManifestWrapper value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ManifestWrapper.DeserializeManifestWrapper(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Get the manifest identified by `name` and `reference` where `reference` can be a tag or digest. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="reference"> A tag or a digest, pointing to a specific image. </param>
        /// <param name="accept"> Accept header string delimited by comma. For example, application/vnd.docker.distribution.manifest.v2+json. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="reference"/> is null. </exception>
        public Response<ManifestWrapper> GetManifest(string name, string reference, string accept = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (reference == null)
            {
                throw new ArgumentNullException(nameof(reference));
            }

            using var message = CreateGetManifestRequest(name, reference, accept);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ManifestWrapper value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ManifestWrapper.DeserializeManifestWrapper(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateCreateManifestRequest(string name, string reference, Stream payload, string contentType)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/v2/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/manifests/", false);
            uri.AppendPath(reference, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            if (contentType != null)
            {
                request.Headers.Add("Content-Type", contentType);
            }
            request.Content = RequestContent.Create(payload);
            return message;
        }

        /// <summary> Put the manifest identified by `name` and `reference` where `reference` can be a tag or digest. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="reference"> A tag or a digest, pointing to a specific image. </param>
        /// <param name="payload"> Manifest body, can take v1 or v2 values depending on accept header. </param>
        /// <param name="contentType"> The manifest's Content-Type. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/>, <paramref name="reference"/> or <paramref name="payload"/> is null. </exception>
        public async Task<ResponseWithHeaders<ContainerRegistryCreateManifestHeaders>> CreateManifestAsync(string name, string reference, Stream payload, string contentType = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (reference == null)
            {
                throw new ArgumentNullException(nameof(reference));
            }
            if (payload == null)
            {
                throw new ArgumentNullException(nameof(payload));
            }

            using var message = CreateCreateManifestRequest(name, reference, payload, contentType);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryCreateManifestHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Put the manifest identified by `name` and `reference` where `reference` can be a tag or digest. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="reference"> A tag or a digest, pointing to a specific image. </param>
        /// <param name="payload"> Manifest body, can take v1 or v2 values depending on accept header. </param>
        /// <param name="contentType"> The manifest's Content-Type. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/>, <paramref name="reference"/> or <paramref name="payload"/> is null. </exception>
        public ResponseWithHeaders<ContainerRegistryCreateManifestHeaders> CreateManifest(string name, string reference, Stream payload, string contentType = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (reference == null)
            {
                throw new ArgumentNullException(nameof(reference));
            }
            if (payload == null)
            {
                throw new ArgumentNullException(nameof(payload));
            }

            using var message = CreateCreateManifestRequest(name, reference, payload, contentType);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryCreateManifestHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateDeleteManifestRequest(string name, string reference)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/v2/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/manifests/", false);
            uri.AppendPath(reference, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Delete the manifest identified by `name` and `reference`. Note that a manifest can _only_ be deleted by `digest`. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="reference"> Digest of a BLOB. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="reference"/> is null. </exception>
        public async Task<Response> DeleteManifestAsync(string name, string reference, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (reference == null)
            {
                throw new ArgumentNullException(nameof(reference));
            }

            using var message = CreateDeleteManifestRequest(name, reference);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 202:
                case 404:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Delete the manifest identified by `name` and `reference`. Note that a manifest can _only_ be deleted by `digest`. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="reference"> Digest of a BLOB. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="reference"/> is null. </exception>
        public Response DeleteManifest(string name, string reference, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (reference == null)
            {
                throw new ArgumentNullException(nameof(reference));
            }

            using var message = CreateDeleteManifestRequest(name, reference);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 202:
                case 404:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetRepositoriesRequest(string last, int? n)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/acr/v1/_catalog", false);
            if (last != null)
            {
                uri.AppendQuery("last", last, true);
            }
            if (n != null)
            {
                uri.AppendQuery("n", n.Value, true);
            }
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> List repositories. </summary>
        /// <param name="last"> Query parameter for the last item in previous query. Result set will include values lexically after last. </param>
        /// <param name="n"> query parameter for max number of items. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public async Task<ResponseWithHeaders<Repositories, ContainerRegistryGetRepositoriesHeaders>> GetRepositoriesAsync(string last = null, int? n = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetRepositoriesRequest(last, n);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryGetRepositoriesHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        Repositories value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = Repositories.DeserializeRepositories(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> List repositories. </summary>
        /// <param name="last"> Query parameter for the last item in previous query. Result set will include values lexically after last. </param>
        /// <param name="n"> query parameter for max number of items. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        public ResponseWithHeaders<Repositories, ContainerRegistryGetRepositoriesHeaders> GetRepositories(string last = null, int? n = null, CancellationToken cancellationToken = default)
        {
            using var message = CreateGetRepositoriesRequest(last, n);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryGetRepositoriesHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        Repositories value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = Repositories.DeserializeRepositories(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetPropertiesRequest(string name)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/acr/v1/", false);
            uri.AppendPath(name, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Get repository attributes. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> is null. </exception>
        public async Task<Response<ContainerRepositoryProperties>> GetPropertiesAsync(string name, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateGetPropertiesRequest(name);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ContainerRepositoryProperties value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ContainerRepositoryProperties.DeserializeContainerRepositoryProperties(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Get repository attributes. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> is null. </exception>
        public Response<ContainerRepositoryProperties> GetProperties(string name, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateGetPropertiesRequest(name);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ContainerRepositoryProperties value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ContainerRepositoryProperties.DeserializeContainerRepositoryProperties(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateDeleteRepositoryRequest(string name)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/acr/v1/", false);
            uri.AppendPath(name, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Delete the repository identified by `name`. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> is null. </exception>
        public async Task<Response> DeleteRepositoryAsync(string name, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateDeleteRepositoryRequest(name);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 202:
                case 404:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Delete the repository identified by `name`. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> is null. </exception>
        public Response DeleteRepository(string name, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateDeleteRepositoryRequest(name);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 202:
                case 404:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateUpdatePropertiesRequest(string name, RepositoryWriteableProperties value)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Patch;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/acr/v1/", false);
            uri.AppendPath(name, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            if (value != null)
            {
                request.Headers.Add("Content-Type", "application/json");
                var content = new Utf8JsonRequestContent();
                content.JsonWriter.WriteObjectValue(value);
                request.Content = content;
            }
            return message;
        }

        /// <summary> Update the attribute identified by `name` where `reference` is the name of the repository. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="value"> Repository attribute value. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> is null. </exception>
        public async Task<Response<ContainerRepositoryProperties>> UpdatePropertiesAsync(string name, RepositoryWriteableProperties value = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateUpdatePropertiesRequest(name, value);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ContainerRepositoryProperties value0 = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value0 = ContainerRepositoryProperties.DeserializeContainerRepositoryProperties(document.RootElement);
                        return Response.FromValue(value0, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Update the attribute identified by `name` where `reference` is the name of the repository. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="value"> Repository attribute value. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> is null. </exception>
        public Response<ContainerRepositoryProperties> UpdateProperties(string name, RepositoryWriteableProperties value = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateUpdatePropertiesRequest(name, value);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ContainerRepositoryProperties value0 = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value0 = ContainerRepositoryProperties.DeserializeContainerRepositoryProperties(document.RootElement);
                        return Response.FromValue(value0, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetTagsRequest(string name, string last, int? n, string orderby, string digest)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/acr/v1/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/_tags", false);
            if (last != null)
            {
                uri.AppendQuery("last", last, true);
            }
            if (n != null)
            {
                uri.AppendQuery("n", n.Value, true);
            }
            if (orderby != null)
            {
                uri.AppendQuery("orderby", orderby, true);
            }
            if (digest != null)
            {
                uri.AppendQuery("digest", digest, true);
            }
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> List tags of a repository. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="last"> Query parameter for the last item in previous query. Result set will include values lexically after last. </param>
        /// <param name="n"> query parameter for max number of items. </param>
        /// <param name="orderby"> orderby query parameter. </param>
        /// <param name="digest"> filter by digest. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> is null. </exception>
        public async Task<ResponseWithHeaders<TagList, ContainerRegistryGetTagsHeaders>> GetTagsAsync(string name, string last = null, int? n = null, string orderby = null, string digest = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateGetTagsRequest(name, last, n, orderby, digest);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryGetTagsHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        TagList value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = TagList.DeserializeTagList(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> List tags of a repository. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="last"> Query parameter for the last item in previous query. Result set will include values lexically after last. </param>
        /// <param name="n"> query parameter for max number of items. </param>
        /// <param name="orderby"> orderby query parameter. </param>
        /// <param name="digest"> filter by digest. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> is null. </exception>
        public ResponseWithHeaders<TagList, ContainerRegistryGetTagsHeaders> GetTags(string name, string last = null, int? n = null, string orderby = null, string digest = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateGetTagsRequest(name, last, n, orderby, digest);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryGetTagsHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        TagList value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = TagList.DeserializeTagList(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetTagPropertiesRequest(string name, string reference)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/acr/v1/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/_tags/", false);
            uri.AppendPath(reference, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Get tag attributes by tag. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="reference"> Tag name. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="reference"/> is null. </exception>
        public async Task<Response<ArtifactTagProperties>> GetTagPropertiesAsync(string name, string reference, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (reference == null)
            {
                throw new ArgumentNullException(nameof(reference));
            }

            using var message = CreateGetTagPropertiesRequest(name, reference);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ArtifactTagProperties value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ArtifactTagProperties.DeserializeArtifactTagProperties(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Get tag attributes by tag. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="reference"> Tag name. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="reference"/> is null. </exception>
        public Response<ArtifactTagProperties> GetTagProperties(string name, string reference, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (reference == null)
            {
                throw new ArgumentNullException(nameof(reference));
            }

            using var message = CreateGetTagPropertiesRequest(name, reference);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ArtifactTagProperties value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ArtifactTagProperties.DeserializeArtifactTagProperties(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateUpdateTagAttributesRequest(string name, string reference, TagWriteableProperties value)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Patch;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/acr/v1/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/_tags/", false);
            uri.AppendPath(reference, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            if (value != null)
            {
                request.Headers.Add("Content-Type", "application/json");
                var content = new Utf8JsonRequestContent();
                content.JsonWriter.WriteObjectValue(value);
                request.Content = content;
            }
            return message;
        }

        /// <summary> Update tag attributes. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="reference"> Tag name. </param>
        /// <param name="value"> Tag attribute value. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="reference"/> is null. </exception>
        public async Task<Response<ArtifactTagProperties>> UpdateTagAttributesAsync(string name, string reference, TagWriteableProperties value = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (reference == null)
            {
                throw new ArgumentNullException(nameof(reference));
            }

            using var message = CreateUpdateTagAttributesRequest(name, reference, value);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ArtifactTagProperties value0 = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value0 = ArtifactTagProperties.DeserializeArtifactTagProperties(document.RootElement);
                        return Response.FromValue(value0, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Update tag attributes. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="reference"> Tag name. </param>
        /// <param name="value"> Tag attribute value. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="reference"/> is null. </exception>
        public Response<ArtifactTagProperties> UpdateTagAttributes(string name, string reference, TagWriteableProperties value = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (reference == null)
            {
                throw new ArgumentNullException(nameof(reference));
            }

            using var message = CreateUpdateTagAttributesRequest(name, reference, value);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ArtifactTagProperties value0 = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value0 = ArtifactTagProperties.DeserializeArtifactTagProperties(document.RootElement);
                        return Response.FromValue(value0, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateDeleteTagRequest(string name, string reference)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/acr/v1/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/_tags/", false);
            uri.AppendPath(reference, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Delete tag. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="reference"> Tag name. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="reference"/> is null. </exception>
        public async Task<Response> DeleteTagAsync(string name, string reference, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (reference == null)
            {
                throw new ArgumentNullException(nameof(reference));
            }

            using var message = CreateDeleteTagRequest(name, reference);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 202:
                case 404:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Delete tag. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="reference"> Tag name. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="reference"/> is null. </exception>
        public Response DeleteTag(string name, string reference, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (reference == null)
            {
                throw new ArgumentNullException(nameof(reference));
            }

            using var message = CreateDeleteTagRequest(name, reference);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 202:
                case 404:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetManifestsRequest(string name, string last, int? n, string orderby)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/acr/v1/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/_manifests", false);
            if (last != null)
            {
                uri.AppendQuery("last", last, true);
            }
            if (n != null)
            {
                uri.AppendQuery("n", n.Value, true);
            }
            if (orderby != null)
            {
                uri.AppendQuery("orderby", orderby, true);
            }
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> List manifests of a repository. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="last"> Query parameter for the last item in previous query. Result set will include values lexically after last. </param>
        /// <param name="n"> query parameter for max number of items. </param>
        /// <param name="orderby"> orderby query parameter. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> is null. </exception>
        public async Task<ResponseWithHeaders<AcrManifests, ContainerRegistryGetManifestsHeaders>> GetManifestsAsync(string name, string last = null, int? n = null, string orderby = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateGetManifestsRequest(name, last, n, orderby);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryGetManifestsHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        AcrManifests value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = AcrManifests.DeserializeAcrManifests(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> List manifests of a repository. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="last"> Query parameter for the last item in previous query. Result set will include values lexically after last. </param>
        /// <param name="n"> query parameter for max number of items. </param>
        /// <param name="orderby"> orderby query parameter. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> is null. </exception>
        public ResponseWithHeaders<AcrManifests, ContainerRegistryGetManifestsHeaders> GetManifests(string name, string last = null, int? n = null, string orderby = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateGetManifestsRequest(name, last, n, orderby);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryGetManifestsHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        AcrManifests value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = AcrManifests.DeserializeAcrManifests(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetManifestPropertiesRequest(string name, string digest)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/acr/v1/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/_manifests/", false);
            uri.AppendPath(digest, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Get manifest attributes. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="digest"/> is null. </exception>
        public async Task<Response<ArtifactManifestProperties>> GetManifestPropertiesAsync(string name, string digest, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }

            using var message = CreateGetManifestPropertiesRequest(name, digest);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ArtifactManifestProperties value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ArtifactManifestProperties.DeserializeArtifactManifestProperties(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Get manifest attributes. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="digest"/> is null. </exception>
        public Response<ArtifactManifestProperties> GetManifestProperties(string name, string digest, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }

            using var message = CreateGetManifestPropertiesRequest(name, digest);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ArtifactManifestProperties value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ArtifactManifestProperties.DeserializeArtifactManifestProperties(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateUpdateManifestPropertiesRequest(string name, string digest, ManifestWriteableProperties value)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Patch;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/acr/v1/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/_manifests/", false);
            uri.AppendPath(digest, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            if (value != null)
            {
                request.Headers.Add("Content-Type", "application/json");
                var content = new Utf8JsonRequestContent();
                content.JsonWriter.WriteObjectValue(value);
                request.Content = content;
            }
            return message;
        }

        /// <summary> Update properties of a manifest. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="value"> Manifest attribute value. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="digest"/> is null. </exception>
        public async Task<Response<ArtifactManifestProperties>> UpdateManifestPropertiesAsync(string name, string digest, ManifestWriteableProperties value = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }

            using var message = CreateUpdateManifestPropertiesRequest(name, digest, value);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ArtifactManifestProperties value0 = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value0 = ArtifactManifestProperties.DeserializeArtifactManifestProperties(document.RootElement);
                        return Response.FromValue(value0, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Update properties of a manifest. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="value"> Manifest attribute value. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="digest"/> is null. </exception>
        public Response<ArtifactManifestProperties> UpdateManifestProperties(string name, string digest, ManifestWriteableProperties value = null, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }

            using var message = CreateUpdateManifestPropertiesRequest(name, digest, value);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ArtifactManifestProperties value0 = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value0 = ArtifactManifestProperties.DeserializeArtifactManifestProperties(document.RootElement);
                        return Response.FromValue(value0, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetRepositoriesNextPageRequest(string nextLink, string last, int? n)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> List repositories. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="last"> Query parameter for the last item in previous query. Result set will include values lexically after last. </param>
        /// <param name="n"> query parameter for max number of items. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> is null. </exception>
        public async Task<ResponseWithHeaders<Repositories, ContainerRegistryGetRepositoriesHeaders>> GetRepositoriesNextPageAsync(string nextLink, string last = null, int? n = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }

            using var message = CreateGetRepositoriesNextPageRequest(nextLink, last, n);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryGetRepositoriesHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        Repositories value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = Repositories.DeserializeRepositories(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> List repositories. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="last"> Query parameter for the last item in previous query. Result set will include values lexically after last. </param>
        /// <param name="n"> query parameter for max number of items. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> is null. </exception>
        public ResponseWithHeaders<Repositories, ContainerRegistryGetRepositoriesHeaders> GetRepositoriesNextPage(string nextLink, string last = null, int? n = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }

            using var message = CreateGetRepositoriesNextPageRequest(nextLink, last, n);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryGetRepositoriesHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        Repositories value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = Repositories.DeserializeRepositories(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetTagsNextPageRequest(string nextLink, string name, string last, int? n, string orderby, string digest)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> List tags of a repository. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="last"> Query parameter for the last item in previous query. Result set will include values lexically after last. </param>
        /// <param name="n"> query parameter for max number of items. </param>
        /// <param name="orderby"> orderby query parameter. </param>
        /// <param name="digest"> filter by digest. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="name"/> is null. </exception>
        public async Task<ResponseWithHeaders<TagList, ContainerRegistryGetTagsHeaders>> GetTagsNextPageAsync(string nextLink, string name, string last = null, int? n = null, string orderby = null, string digest = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateGetTagsNextPageRequest(nextLink, name, last, n, orderby, digest);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryGetTagsHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        TagList value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = TagList.DeserializeTagList(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> List tags of a repository. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="last"> Query parameter for the last item in previous query. Result set will include values lexically after last. </param>
        /// <param name="n"> query parameter for max number of items. </param>
        /// <param name="orderby"> orderby query parameter. </param>
        /// <param name="digest"> filter by digest. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="name"/> is null. </exception>
        public ResponseWithHeaders<TagList, ContainerRegistryGetTagsHeaders> GetTagsNextPage(string nextLink, string name, string last = null, int? n = null, string orderby = null, string digest = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateGetTagsNextPageRequest(nextLink, name, last, n, orderby, digest);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryGetTagsHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        TagList value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = TagList.DeserializeTagList(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetManifestsNextPageRequest(string nextLink, string name, string last, int? n, string orderby)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> List manifests of a repository. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="last"> Query parameter for the last item in previous query. Result set will include values lexically after last. </param>
        /// <param name="n"> query parameter for max number of items. </param>
        /// <param name="orderby"> orderby query parameter. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="name"/> is null. </exception>
        public async Task<ResponseWithHeaders<AcrManifests, ContainerRegistryGetManifestsHeaders>> GetManifestsNextPageAsync(string nextLink, string name, string last = null, int? n = null, string orderby = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateGetManifestsNextPageRequest(nextLink, name, last, n, orderby);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryGetManifestsHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        AcrManifests value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = AcrManifests.DeserializeAcrManifests(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> List manifests of a repository. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="last"> Query parameter for the last item in previous query. Result set will include values lexically after last. </param>
        /// <param name="n"> query parameter for max number of items. </param>
        /// <param name="orderby"> orderby query parameter. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="name"/> is null. </exception>
        public ResponseWithHeaders<AcrManifests, ContainerRegistryGetManifestsHeaders> GetManifestsNextPage(string nextLink, string name, string last = null, int? n = null, string orderby = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateGetManifestsNextPageRequest(nextLink, name, last, n, orderby);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryGetManifestsHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        AcrManifests value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = AcrManifests.DeserializeAcrManifests(document.RootElement);
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }
    }
}
