// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

// <auto-generated/>

#nullable disable

using System;
using System.IO;
using System.Threading;
using System.Threading.Tasks;
using Azure.Core;
using Azure.Core.Pipeline;

namespace Azure.Containers.ContainerRegistry
{
    internal partial class ContainerRegistryBlobRestClient
    {
        private readonly HttpPipeline _pipeline;
        private readonly string _url;

        /// <summary> The ClientDiagnostics is used to provide tracing support for the client library. </summary>
        internal ClientDiagnostics ClientDiagnostics { get; }

        /// <summary> Initializes a new instance of ContainerRegistryBlobRestClient. </summary>
        /// <param name="clientDiagnostics"> The handler for diagnostic messaging in the client. </param>
        /// <param name="pipeline"> The HTTP pipeline for sending and receiving REST requests and responses. </param>
        /// <param name="url"> Registry login URL. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="clientDiagnostics"/>, <paramref name="pipeline"/> or <paramref name="url"/> is null. </exception>
        public ContainerRegistryBlobRestClient(ClientDiagnostics clientDiagnostics, HttpPipeline pipeline, string url)
        {
            ClientDiagnostics = clientDiagnostics ?? throw new ArgumentNullException(nameof(clientDiagnostics));
            _pipeline = pipeline ?? throw new ArgumentNullException(nameof(pipeline));
            _url = url ?? throw new ArgumentNullException(nameof(url));
        }

        internal HttpMessage CreateGetBlobRequest(string name, string digest)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            message.BufferResponse = false;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/v2/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/blobs/", false);
            uri.AppendPath(digest, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/octet-stream");
            return message;
        }

        /// <summary> Retrieve the blob from the registry identified by digest. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="digest"/> is null. </exception>
        public async Task<ResponseWithHeaders<Stream, ContainerRegistryBlobGetBlobHeaders>> GetBlobAsync(string name, string digest, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }

            using var message = CreateGetBlobRequest(name, digest);
            RedirectPolicy.SetAllowAutoRedirect(message, true);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryBlobGetBlobHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        var value = message.ExtractResponseContent();
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                case 307:
                    return ResponseWithHeaders.FromValue((Stream)null, headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Retrieve the blob from the registry identified by digest. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="digest"/> is null. </exception>
        public ResponseWithHeaders<Stream, ContainerRegistryBlobGetBlobHeaders> GetBlob(string name, string digest, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }

            using var message = CreateGetBlobRequest(name, digest);
            RedirectPolicy.SetAllowAutoRedirect(message, true);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryBlobGetBlobHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        var value = message.ExtractResponseContent();
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                case 307:
                    return ResponseWithHeaders.FromValue((Stream)null, headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateCheckBlobExistsRequest(string name, string digest)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Head;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/v2/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/blobs/", false);
            uri.AppendPath(digest, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Same as GET, except only the headers are returned. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="digest"/> is null. </exception>
        public async Task<ResponseWithHeaders<ContainerRegistryBlobCheckBlobExistsHeaders>> CheckBlobExistsAsync(string name, string digest, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }

            using var message = CreateCheckBlobExistsRequest(name, digest);
            RedirectPolicy.SetAllowAutoRedirect(message, true);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryBlobCheckBlobExistsHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                case 307:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Same as GET, except only the headers are returned. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="digest"/> is null. </exception>
        public ResponseWithHeaders<ContainerRegistryBlobCheckBlobExistsHeaders> CheckBlobExists(string name, string digest, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }

            using var message = CreateCheckBlobExistsRequest(name, digest);
            RedirectPolicy.SetAllowAutoRedirect(message, true);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryBlobCheckBlobExistsHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                case 307:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateDeleteBlobRequest(string name, string digest)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/v2/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/blobs/", false);
            uri.AppendPath(digest, true);
            request.Uri = uri;
            return message;
        }

        /// <summary> Removes an already uploaded blob. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="digest"/> is null. </exception>
        public async Task<ResponseWithHeaders<ContainerRegistryBlobDeleteBlobHeaders>> DeleteBlobAsync(string name, string digest, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }

            using var message = CreateDeleteBlobRequest(name, digest);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryBlobDeleteBlobHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 202:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Removes an already uploaded blob. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> or <paramref name="digest"/> is null. </exception>
        public ResponseWithHeaders<ContainerRegistryBlobDeleteBlobHeaders> DeleteBlob(string name, string digest, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }

            using var message = CreateDeleteBlobRequest(name, digest);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryBlobDeleteBlobHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 202:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateMountBlobRequest(string name, string @from, string mount)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/v2/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/blobs/uploads/", false);
            uri.AppendQuery("from", @from, true);
            uri.AppendQuery("mount", mount, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Mount a blob identified by the `mount` parameter from another repository. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="from"> Name of the source repository. </param>
        /// <param name="mount"> Digest of blob to mount from the source repository. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/>, <paramref name="from"/> or <paramref name="mount"/> is null. </exception>
        public async Task<ResponseWithHeaders<ContainerRegistryBlobMountBlobHeaders>> MountBlobAsync(string name, string @from, string mount, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (@from == null)
            {
                throw new ArgumentNullException(nameof(@from));
            }
            if (mount == null)
            {
                throw new ArgumentNullException(nameof(mount));
            }

            using var message = CreateMountBlobRequest(name, @from, mount);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryBlobMountBlobHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Mount a blob identified by the `mount` parameter from another repository. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="from"> Name of the source repository. </param>
        /// <param name="mount"> Digest of blob to mount from the source repository. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/>, <paramref name="from"/> or <paramref name="mount"/> is null. </exception>
        public ResponseWithHeaders<ContainerRegistryBlobMountBlobHeaders> MountBlob(string name, string @from, string mount, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (@from == null)
            {
                throw new ArgumentNullException(nameof(@from));
            }
            if (mount == null)
            {
                throw new ArgumentNullException(nameof(mount));
            }

            using var message = CreateMountBlobRequest(name, @from, mount);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryBlobMountBlobHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetUploadStatusRequest(string nextLink)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Retrieve status of upload identified by uuid. The primary purpose of this endpoint is to resolve the current status of a resumable upload. </summary>
        /// <param name="nextLink"> Link acquired from upload start or previous chunk. Note, do not include initial / (must do substring(1) ). </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> is null. </exception>
        public async Task<ResponseWithHeaders<ContainerRegistryBlobGetUploadStatusHeaders>> GetUploadStatusAsync(string nextLink, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }

            using var message = CreateGetUploadStatusRequest(nextLink);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryBlobGetUploadStatusHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Retrieve status of upload identified by uuid. The primary purpose of this endpoint is to resolve the current status of a resumable upload. </summary>
        /// <param name="nextLink"> Link acquired from upload start or previous chunk. Note, do not include initial / (must do substring(1) ). </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> is null. </exception>
        public ResponseWithHeaders<ContainerRegistryBlobGetUploadStatusHeaders> GetUploadStatus(string nextLink, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }

            using var message = CreateGetUploadStatusRequest(nextLink);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryBlobGetUploadStatusHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 204:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateUploadChunkRequest(string nextLink, Stream value, string contentRange, string contentLength)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Patch;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            if (contentRange != null)
            {
                request.Headers.Add("Content-Range", contentRange);
            }
            if (contentLength != null)
            {
                request.Headers.Add("Content-Length", contentLength);
            }
            request.Headers.Add("Content-Type", "application/octet-stream");
            request.Content = RequestContent.Create(value);
            return message;
        }

        /// <summary> Upload a stream of data without completing the upload. </summary>
        /// <param name="nextLink"> Link acquired from upload start or previous chunk. Note, do not include initial / (must do substring(1) ). </param>
        /// <param name="value"> Raw data of blob. </param>
        /// <param name="contentRange"> Range of bytes identifying the desired block of content represented by the body. Start must the end offset retrieved via status check plus one. Note that this is a non-standard use of the Content-Range header. </param>
        /// <param name="contentLength"> Length of the chunk being uploaded, corresponding the length of the request body. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="value"/> is null. </exception>
        public async Task<ResponseWithHeaders<ContainerRegistryBlobUploadChunkHeaders>> UploadChunkAsync(string nextLink, Stream value, string contentRange = null, string contentLength = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (value == null)
            {
                throw new ArgumentNullException(nameof(value));
            }

            using var message = CreateUploadChunkRequest(nextLink, value, contentRange, contentLength);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryBlobUploadChunkHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 202:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Upload a stream of data without completing the upload. </summary>
        /// <param name="nextLink"> Link acquired from upload start or previous chunk. Note, do not include initial / (must do substring(1) ). </param>
        /// <param name="value"> Raw data of blob. </param>
        /// <param name="contentRange"> Range of bytes identifying the desired block of content represented by the body. Start must the end offset retrieved via status check plus one. Note that this is a non-standard use of the Content-Range header. </param>
        /// <param name="contentLength"> Length of the chunk being uploaded, corresponding the length of the request body. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="value"/> is null. </exception>
        public ResponseWithHeaders<ContainerRegistryBlobUploadChunkHeaders> UploadChunk(string nextLink, Stream value, string contentRange = null, string contentLength = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (value == null)
            {
                throw new ArgumentNullException(nameof(value));
            }

            using var message = CreateUploadChunkRequest(nextLink, value, contentRange, contentLength);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryBlobUploadChunkHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 202:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateCompleteUploadRequest(string digest, string nextLink, Stream value)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Put;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendRawNextLink(nextLink, false);
            uri.AppendQuery("digest", digest, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            if (value != null)
            {
                request.Headers.Add("Content-Type", "application/octet-stream");
                request.Content = RequestContent.Create(value);
            }
            return message;
        }

        /// <summary> Complete the upload, providing all the data in the body, if necessary. A request without a body will just complete the upload with previously uploaded content. </summary>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="nextLink"> Link acquired from upload start or previous chunk. Note, do not include initial / (must do substring(1) ). </param>
        /// <param name="value"> Optional raw data of blob. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="digest"/> or <paramref name="nextLink"/> is null. </exception>
        public async Task<ResponseWithHeaders<ContainerRegistryBlobCompleteUploadHeaders>> CompleteUploadAsync(string digest, string nextLink, Stream value = null, CancellationToken cancellationToken = default)
        {
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }

            using var message = CreateCompleteUploadRequest(digest, nextLink, value);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryBlobCompleteUploadHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Complete the upload, providing all the data in the body, if necessary. A request without a body will just complete the upload with previously uploaded content. </summary>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="nextLink"> Link acquired from upload start or previous chunk. Note, do not include initial / (must do substring(1) ). </param>
        /// <param name="value"> Optional raw data of blob. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="digest"/> or <paramref name="nextLink"/> is null. </exception>
        public ResponseWithHeaders<ContainerRegistryBlobCompleteUploadHeaders> CompleteUpload(string digest, string nextLink, Stream value = null, CancellationToken cancellationToken = default)
        {
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }

            using var message = CreateCompleteUploadRequest(digest, nextLink, value);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryBlobCompleteUploadHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 201:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateCancelUploadRequest(string nextLink)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/", false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Cancel outstanding upload processes, releasing associated resources. If this is not called, the unfinished uploads will eventually timeout. </summary>
        /// <param name="nextLink"> Link acquired from upload start or previous chunk. Note, do not include initial / (must do substring(1) ). </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> is null. </exception>
        public async Task<Response> CancelUploadAsync(string nextLink, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }

            using var message = CreateCancelUploadRequest(nextLink);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 204:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Cancel outstanding upload processes, releasing associated resources. If this is not called, the unfinished uploads will eventually timeout. </summary>
        /// <param name="nextLink"> Link acquired from upload start or previous chunk. Note, do not include initial / (must do substring(1) ). </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> is null. </exception>
        public Response CancelUpload(string nextLink, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }

            using var message = CreateCancelUploadRequest(nextLink);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 204:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateStartUploadRequest(string name)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/v2/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/blobs/uploads/", false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Initiate a resumable blob upload with an empty request body. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> is null. </exception>
        public async Task<ResponseWithHeaders<ContainerRegistryBlobStartUploadHeaders>> StartUploadAsync(string name, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateStartUploadRequest(name);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryBlobStartUploadHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 202:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Initiate a resumable blob upload with an empty request body. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/> is null. </exception>
        public ResponseWithHeaders<ContainerRegistryBlobStartUploadHeaders> StartUpload(string name, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }

            using var message = CreateStartUploadRequest(name);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryBlobStartUploadHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 202:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetChunkRequest(string name, string digest, string range)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            message.BufferResponse = false;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/v2/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/blobs/", false);
            uri.AppendPath(digest, true);
            request.Uri = uri;
            request.Headers.Add("Range", range);
            request.Headers.Add("Accept", "application/octet-stream");
            return message;
        }

        /// <summary> Retrieve the blob from the registry identified by `digest`. This endpoint may also support RFC7233 compliant range requests. Support can be detected by issuing a HEAD request. If the header `Accept-Range: bytes` is returned, range requests can be used to fetch partial content. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="range"> Format : bytes=&lt;start&gt;-&lt;end&gt;,  HTTP Range header specifying blob chunk. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/>, <paramref name="digest"/> or <paramref name="range"/> is null. </exception>
        public async Task<ResponseWithHeaders<Stream, ContainerRegistryBlobGetChunkHeaders>> GetChunkAsync(string name, string digest, string range, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }
            if (range == null)
            {
                throw new ArgumentNullException(nameof(range));
            }

            using var message = CreateGetChunkRequest(name, digest, range);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryBlobGetChunkHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 206:
                    {
                        var value = message.ExtractResponseContent();
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Retrieve the blob from the registry identified by `digest`. This endpoint may also support RFC7233 compliant range requests. Support can be detected by issuing a HEAD request. If the header `Accept-Range: bytes` is returned, range requests can be used to fetch partial content. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="range"> Format : bytes=&lt;start&gt;-&lt;end&gt;,  HTTP Range header specifying blob chunk. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/>, <paramref name="digest"/> or <paramref name="range"/> is null. </exception>
        public ResponseWithHeaders<Stream, ContainerRegistryBlobGetChunkHeaders> GetChunk(string name, string digest, string range, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }
            if (range == null)
            {
                throw new ArgumentNullException(nameof(range));
            }

            using var message = CreateGetChunkRequest(name, digest, range);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryBlobGetChunkHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 206:
                    {
                        var value = message.ExtractResponseContent();
                        return ResponseWithHeaders.FromValue(value, headers, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateCheckChunkExistsRequest(string name, string digest, string range)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Head;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_url, false);
            uri.AppendPath("/v2/", false);
            uri.AppendPath(name, true);
            uri.AppendPath("/blobs/", false);
            uri.AppendPath(digest, true);
            request.Uri = uri;
            request.Headers.Add("Range", range);
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Same as GET, except only the headers are returned. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="range"> Format : bytes=&lt;start&gt;-&lt;end&gt;,  HTTP Range header specifying blob chunk. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/>, <paramref name="digest"/> or <paramref name="range"/> is null. </exception>
        public async Task<ResponseWithHeaders<ContainerRegistryBlobCheckChunkExistsHeaders>> CheckChunkExistsAsync(string name, string digest, string range, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }
            if (range == null)
            {
                throw new ArgumentNullException(nameof(range));
            }

            using var message = CreateCheckChunkExistsRequest(name, digest, range);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            var headers = new ContainerRegistryBlobCheckChunkExistsHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Same as GET, except only the headers are returned. </summary>
        /// <param name="name"> Name of the image (including the namespace). </param>
        /// <param name="digest"> Digest of a BLOB. </param>
        /// <param name="range"> Format : bytes=&lt;start&gt;-&lt;end&gt;,  HTTP Range header specifying blob chunk. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="name"/>, <paramref name="digest"/> or <paramref name="range"/> is null. </exception>
        public ResponseWithHeaders<ContainerRegistryBlobCheckChunkExistsHeaders> CheckChunkExists(string name, string digest, string range, CancellationToken cancellationToken = default)
        {
            if (name == null)
            {
                throw new ArgumentNullException(nameof(name));
            }
            if (digest == null)
            {
                throw new ArgumentNullException(nameof(digest));
            }
            if (range == null)
            {
                throw new ArgumentNullException(nameof(range));
            }

            using var message = CreateCheckChunkExistsRequest(name, digest, range);
            _pipeline.Send(message, cancellationToken);
            var headers = new ContainerRegistryBlobCheckChunkExistsHeaders(message.Response);
            switch (message.Response.Status)
            {
                case 200:
                    return ResponseWithHeaders.FromValue(headers, message.Response);
                default:
                    throw new RequestFailedException(message.Response);
            }
        }
    }
}
