# The Certificate Service Client 

By default, the `ConfidentialLedgerClient` will automatically configure itself to trust the root TLS certificate advertised by the `ConfidentialLedgerCertificateClient` `GetLedgerIdentity` operation.
However, if specific customizations are needed to the client option's `Transport`, this automatic configuration needs to be done manually.

Because Confidential Ledgers use self-signed certificates securely generated and stored in an SGX enclave, the certificate for each Confidential Ledger must first be retrieved from the Confidential Ledger Identity Service.

```C# Snippet:GetIdentity
Uri identityServiceEndpoint = new("https://identity.confidential-ledger.core.azure.com"); // The hostname from the identityServiceUri
var identityClient = new ConfidentialLedgerCertificateClient(identityServiceEndpoint);

// Get the ledger's  TLS certificate for our ledger.
string ledgerId = "<the ledger id>"; // ex. "my-ledger" from "https://my-ledger.eastus.cloudapp.azure.com"
Response response = identityClient.GetLedgerIdentity(ledgerId);
X509Certificate2 ledgerTlsCert = ConfidentialLedgerCertificateClient.ParseCertificate(response);

X509Chain certificateChain = new();
// Revocation is not required by CCF. Hence revocation checks must be skipped to avoid validation failing unnecessarily.
certificateChain.ChainPolicy.RevocationMode = X509RevocationMode.NoCheck;
// Add the ledger identity TLS certificate to the ExtraStore.
certificateChain.ChainPolicy.ExtraStore.Add(ledgerTlsCert);
// AllowUnknownCertificateAuthority will NOT allow validation of all unknown self-signed certificates.
// It extends trust to the ExtraStore, which in this case contains the trusted ledger identity TLS certificate.
// This makes it possible for validation of certificate chains terminating in the ledger identity TLS certificate to pass.
// Note: .NET 5 introduced `CustomTrustStore` but we cannot use that here as we must support older versions of .NET.
certificateChain.ChainPolicy.VerificationFlags = X509VerificationFlags.AllowUnknownCertificateAuthority;
certificateChain.ChainPolicy.VerificationTime = DateTime.Now;

// Define a validation function to ensure that certificates presented to the client only pass validation if
// they are trusted by the ledger identity TLS certificate.
bool CertValidationCheck(HttpRequestMessage httpRequestMessage, X509Certificate2 cert, X509Chain x509Chain, SslPolicyErrors sslPolicyErrors)
{
    // Validate the presented certificate chain, using the ChainPolicy defined above.
    // Note: this check will allow certificates signed by standard CAs as well as those signed by the ledger identity TLS certificate.
    bool isChainValid = certificateChain.Build(cert);
    if (!isChainValid)
        return false;

    // Ensure that the presented certificate chain passes validation only if it is rooted in the the ledger identity TLS certificate.
    var rootCert = certificateChain.ChainElements[certificateChain.ChainElements.Count - 1].Certificate;
    var isChainRootedInTheTlsCert = rootCert.RawData.SequenceEqual(ledgerTlsCert.RawData);
    return isChainRootedInTheTlsCert;
}

// Create an HttpClientHandler to use our certValidationCheck function.
var httpHandler = new HttpClientHandler();
httpHandler.ServerCertificateCustomValidationCallback = CertValidationCheck;

// Create the ledger client using a transport that uses our custom ServerCertificateCustomValidationCallback.
var options = new ConfidentialLedgerClientOptions { Transport = new HttpClientTransport(httpHandler) };
var ledgerClient = new ConfidentialLedgerClient(TestEnvironment.ConfidentialLedgerUrl, new DefaultAzureCredential(), options);
```
