# Obtain user tokens from Azure Communication Services

This sample demonstrates how to obtain user tokens from Azure Communication Services. You can use this token to authenticate your users to use Azure Communication Services offerings.

To get started you'll need an Azure Communication Services resource. See the README for prerequisites and instructions.

<!-- TODO: Update ReadMe -->

## Create a `CommunicationIdentityClient`

To create a new `CommunicationIdentityClient` you need a connection string to the Azure Communication Services resource that you can get from the Azure Portal once you have created the resource.

You can set `connectionString` based on an environment variable, a configuration setting, or any way that works for your application.

```C# Snippet:CreateCommunicationIdentityClientAsync
// Get a connection string to our Azure Communication resource.
var connectionString = "<connection_string>";
var client = new CommunicationIdentityClient(connectionString);
```

Or alternatively using the endpoint and access key acquired from an Azure Communication Resources in the [Azure Portal][azure_portal].

```C# Snippet:CreateCommunicationIdentityFromAccessKey
var endpoint = new Uri("https://my-resource.communication.azure.com");
var accessKey = "<access_key>";
var client = new CommunicationIdentityClient(endpoint, new AzureKeyCredential(accessKey));
```

Clients also have the option to authenticate using a valid Active Directory token.

```C# Snippet:CreateCommunicationIdentityFromToken
var endpoint = new Uri("https://my-resource.communication.azure.com");
TokenCredential tokenCredential = TestEnvironment.Credential;
var client = new CommunicationIdentityClient(endpoint, tokenCredential);
```

## Create a user

The `CommunicationIdentityClient` can be used to create users and get tokens.

```C# Snippet:CreateCommunicationUserAsync
Response<CommunicationUserIdentifier> userResponse = await client.CreateUserAsync();
CommunicationUserIdentifier user = userResponse.Value;
Console.WriteLine($"User id: {user.Id}");
```

You will need to store the `identity` that is returned by Azure Communication Services. This is necessary for creating tokens or refreshing them in the future and to map your user identities with Azure Communication Services identities. The `identity` value should be treated as a secret.

## Generate a user token

 <!---: TODO: Update the website address for explaining the scopes -->

In the example code snippet below, it is assumed that the token is generated for a user on your application. During token generation you should also pass list of scopes to Azure Communication Services,
so the token generated by Azure Communication Services is authorized only to do things determined by `scopes`.

<!-- You can see the full list of scopes TODO: [here](https://github.com/mikben/azure-docs-pr/blob/release-project-spool/articles/project-spool/concepts/identity-model.md).-->

Every token generated has an expiry date stamped on it. Once the token is expired, you can renew the token by calling the same method.

```C# Snippet:CreateCommunicationTokenAsync
Response<AccessToken> tokenResponse = await client.GetTokenAsync(user, scopes: new[] { CommunicationTokenScope.Chat });
string token = tokenResponse.Value.Token;
DateTimeOffset expiresOn = tokenResponse.Value.ExpiresOn;
Console.WriteLine($"Token: {token}");
Console.WriteLine($"Expires On: {expiresOn}");
```

It's also possible to create a Communication Identity access token by customizing the expiration time. Validity period of the token must be within [1,24] hours range. If not provided, the default value of 24 hours will be used.

```C# Snippet:CreateCommunicationTokenAsyncWithCustomExpiration
TimeSpan tokenExpiresIn = TimeSpan.FromHours(1);
Response<AccessToken> tokenResponse = await client.GetTokenAsync(user, scopes: new[] { CommunicationTokenScope.Chat }, tokenExpiresIn);
string token = tokenResponse.Value.Token;
DateTimeOffset expiresOn = tokenResponse.Value.ExpiresOn;
Console.WriteLine($"Token: {token}");
Console.WriteLine($"Expires On: {expiresOn}");
```

## Creating a user and a token with custom expiration in the same request

You can create user and token in the same request. You can specify expiration time for the token. The token can be configured to expire in as little as one hour or as long as 24 hours. The default expiration time is 24 hours.

```C# Snippet:CreateCommunicationUserAndTokenWithCustomExpirationAsync
TimeSpan tokenExpiresIn = TimeSpan.FromHours(1);
Response<CommunicationUserIdentifierAndToken> response = await client.CreateUserAndTokenAsync(scopes: new[] { CommunicationTokenScope.Chat }, tokenExpiresIn);
var (user, token) = response.Value;
Console.WriteLine($"User id: {user.Id}");
Console.WriteLine($"Token: {token.Token}");
```

## Exchange an Azure AD access token of a Teams User for a Communication Identity access token

The `CommunicationIdentityClient` can be used to exchange an Azure AD access token of a Teams user for a new Communication Identity access token with a matching expiration time.

The `GetTokenForTeamsUser` function accepts the following parameters wrapped into the `GetTokenForTeamsUserOptions` option bag:
- `teamsUserAadToken` Azure Active Directory access token of a Teams user
- `clientId` Client ID of an Azure AD application to be verified against the appId claim in the Azure AD access token
- `userObjectId` Object ID of an Azure AD user (Teams User) to be verified against the OID claim in the Azure AD access token

```C# Snippet:GetTokenForTeamsUserAsync
Response<AccessToken> tokenResponse = await client.GetTokenForTeamsUserAsync(new GetTokenForTeamsUserOptions(teamsUserAadToken, clientId, userObjectId));
string token = tokenResponse.Value.Token;
Console.WriteLine($"Token: {token}");
```

<!-- LINKS -->
<!--
[ReadMe](https://github.com/Azure/azure-sdk-for-net/tree/main/sdk/communication/Azure.Communication.Administration/samples/ReadMe.md)
-->
