// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

// <auto-generated/>

#nullable disable

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.Json;
using System.Threading;
using System.Threading.Tasks;
using Azure.Core;
using Azure.Core.Pipeline;

namespace Azure.Communication.Chat
{
    internal partial class ChatThreadRestClient
    {
        private readonly HttpPipeline _pipeline;
        private readonly string _endpoint;
        private readonly string _apiVersion;

        /// <summary> The ClientDiagnostics is used to provide tracing support for the client library. </summary>
        internal ClientDiagnostics ClientDiagnostics { get; }

        /// <summary> Initializes a new instance of ChatThreadRestClient. </summary>
        /// <param name="clientDiagnostics"> The handler for diagnostic messaging in the client. </param>
        /// <param name="pipeline"> The HTTP pipeline for sending and receiving REST requests and responses. </param>
        /// <param name="endpoint"> The endpoint of the Azure Communication resource. </param>
        /// <param name="apiVersion"> Api Version. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="clientDiagnostics"/>, <paramref name="pipeline"/>, <paramref name="endpoint"/> or <paramref name="apiVersion"/> is null. </exception>
        public ChatThreadRestClient(ClientDiagnostics clientDiagnostics, HttpPipeline pipeline, string endpoint, string apiVersion = "2024-03-07")
        {
            ClientDiagnostics = clientDiagnostics ?? throw new ArgumentNullException(nameof(clientDiagnostics));
            _pipeline = pipeline ?? throw new ArgumentNullException(nameof(pipeline));
            _endpoint = endpoint ?? throw new ArgumentNullException(nameof(endpoint));
            _apiVersion = apiVersion ?? throw new ArgumentNullException(nameof(apiVersion));
        }

        internal HttpMessage CreateListChatReadReceiptsRequest(string chatThreadId, int? maxPageSize, int? skip)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendPath("/readReceipts", false);
            if (maxPageSize != null)
            {
                uri.AppendQuery("maxPageSize", maxPageSize.Value, true);
            }
            if (skip != null)
            {
                uri.AppendQuery("skip", skip.Value, true);
            }
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Gets chat message read receipts for a thread. </summary>
        /// <param name="chatThreadId"> Thread id to get the chat message read receipts for. </param>
        /// <param name="maxPageSize"> The maximum number of chat message read receipts to be returned per page. </param>
        /// <param name="skip"> Skips chat message read receipts up to a specified position in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public async Task<Response<ChatMessageReadReceiptsCollection>> ListChatReadReceiptsAsync(string chatThreadId, int? maxPageSize = null, int? skip = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateListChatReadReceiptsRequest(chatThreadId, maxPageSize, skip);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatMessageReadReceiptsCollection value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ChatMessageReadReceiptsCollection.DeserializeChatMessageReadReceiptsCollection(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Gets chat message read receipts for a thread. </summary>
        /// <param name="chatThreadId"> Thread id to get the chat message read receipts for. </param>
        /// <param name="maxPageSize"> The maximum number of chat message read receipts to be returned per page. </param>
        /// <param name="skip"> Skips chat message read receipts up to a specified position in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public Response<ChatMessageReadReceiptsCollection> ListChatReadReceipts(string chatThreadId, int? maxPageSize = null, int? skip = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateListChatReadReceiptsRequest(chatThreadId, maxPageSize, skip);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatMessageReadReceiptsCollection value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ChatMessageReadReceiptsCollection.DeserializeChatMessageReadReceiptsCollection(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateSendChatReadReceiptRequest(string chatThreadId, string chatMessageId)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendPath("/readReceipts", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/json");
            var model = new SendReadReceiptRequest(chatMessageId);
            var content = new Utf8JsonRequestContent();
            content.JsonWriter.WriteObjectValue(model);
            request.Content = content;
            return message;
        }

        /// <summary> Sends a read receipt event to a thread, on behalf of a user. </summary>
        /// <param name="chatThreadId"> Thread id to send the read receipt event to. </param>
        /// <param name="chatMessageId"> Id of the latest chat message read by the user. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> or <paramref name="chatMessageId"/> is null. </exception>
        public async Task<Response> SendChatReadReceiptAsync(string chatThreadId, string chatMessageId, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }
            if (chatMessageId == null)
            {
                throw new ArgumentNullException(nameof(chatMessageId));
            }

            using var message = CreateSendChatReadReceiptRequest(chatThreadId, chatMessageId);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Sends a read receipt event to a thread, on behalf of a user. </summary>
        /// <param name="chatThreadId"> Thread id to send the read receipt event to. </param>
        /// <param name="chatMessageId"> Id of the latest chat message read by the user. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> or <paramref name="chatMessageId"/> is null. </exception>
        public Response SendChatReadReceipt(string chatThreadId, string chatMessageId, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }
            if (chatMessageId == null)
            {
                throw new ArgumentNullException(nameof(chatMessageId));
            }

            using var message = CreateSendChatReadReceiptRequest(chatThreadId, chatMessageId);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateSendChatMessageRequest(string chatThreadId, string content, string senderDisplayName, ChatMessageType? type, IDictionary<string, string> metadata)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendPath("/messages", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/json");
            SendChatMessageRequest sendChatMessageRequest = new SendChatMessageRequest(content)
            {
                SenderDisplayName = senderDisplayName,
                Type = type
            };
            if (metadata != null)
            {
                foreach (var value in metadata)
                {
                    sendChatMessageRequest.Metadata.Add(value);
                }
            }
            var model = sendChatMessageRequest;
            var content0 = new Utf8JsonRequestContent();
            content0.JsonWriter.WriteObjectValue(model);
            request.Content = content0;
            return message;
        }

        /// <summary> Sends a message to a thread. </summary>
        /// <param name="chatThreadId"> The thread id to send the message to. </param>
        /// <param name="content"> Chat message content. </param>
        /// <param name="senderDisplayName"> The display name of the chat message sender. This property is used to populate sender name for push notifications. </param>
        /// <param name="type"> The chat message type. </param>
        /// <param name="metadata"> Message metadata. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> or <paramref name="content"/> is null. </exception>
        public async Task<Response<SendChatMessageResultInternal>> SendChatMessageAsync(string chatThreadId, string content, string senderDisplayName = null, ChatMessageType? type = null, IDictionary<string, string> metadata = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }
            if (content == null)
            {
                throw new ArgumentNullException(nameof(content));
            }

            using var message = CreateSendChatMessageRequest(chatThreadId, content, senderDisplayName, type, metadata);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 201:
                    {
                        SendChatMessageResultInternal value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = SendChatMessageResultInternal.DeserializeSendChatMessageResultInternal(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Sends a message to a thread. </summary>
        /// <param name="chatThreadId"> The thread id to send the message to. </param>
        /// <param name="content"> Chat message content. </param>
        /// <param name="senderDisplayName"> The display name of the chat message sender. This property is used to populate sender name for push notifications. </param>
        /// <param name="type"> The chat message type. </param>
        /// <param name="metadata"> Message metadata. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> or <paramref name="content"/> is null. </exception>
        public Response<SendChatMessageResultInternal> SendChatMessage(string chatThreadId, string content, string senderDisplayName = null, ChatMessageType? type = null, IDictionary<string, string> metadata = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }
            if (content == null)
            {
                throw new ArgumentNullException(nameof(content));
            }

            using var message = CreateSendChatMessageRequest(chatThreadId, content, senderDisplayName, type, metadata);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 201:
                    {
                        SendChatMessageResultInternal value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = SendChatMessageResultInternal.DeserializeSendChatMessageResultInternal(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateListChatMessagesRequest(string chatThreadId, int? maxPageSize, DateTimeOffset? startTime)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendPath("/messages", false);
            if (maxPageSize != null)
            {
                uri.AppendQuery("maxPageSize", maxPageSize.Value, true);
            }
            if (startTime != null)
            {
                uri.AppendQuery("startTime", startTime.Value, "O", true);
            }
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Gets a list of messages from a thread. </summary>
        /// <param name="chatThreadId"> The thread id of the message. </param>
        /// <param name="maxPageSize"> The maximum number of messages to be returned per page. </param>
        /// <param name="startTime"> The earliest point in time to get messages up to. The timestamp should be in RFC3339 format: `yyyy-MM-ddTHH:mm:ssZ`. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public async Task<Response<ChatMessagesCollection>> ListChatMessagesAsync(string chatThreadId, int? maxPageSize = null, DateTimeOffset? startTime = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateListChatMessagesRequest(chatThreadId, maxPageSize, startTime);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatMessagesCollection value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ChatMessagesCollection.DeserializeChatMessagesCollection(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Gets a list of messages from a thread. </summary>
        /// <param name="chatThreadId"> The thread id of the message. </param>
        /// <param name="maxPageSize"> The maximum number of messages to be returned per page. </param>
        /// <param name="startTime"> The earliest point in time to get messages up to. The timestamp should be in RFC3339 format: `yyyy-MM-ddTHH:mm:ssZ`. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public Response<ChatMessagesCollection> ListChatMessages(string chatThreadId, int? maxPageSize = null, DateTimeOffset? startTime = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateListChatMessagesRequest(chatThreadId, maxPageSize, startTime);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatMessagesCollection value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ChatMessagesCollection.DeserializeChatMessagesCollection(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetChatMessageRequest(string chatThreadId, string chatMessageId)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendPath("/messages/", false);
            uri.AppendPath(chatMessageId, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Gets a message by id. </summary>
        /// <param name="chatThreadId"> The thread id to which the message was sent. </param>
        /// <param name="chatMessageId"> The message id. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> or <paramref name="chatMessageId"/> is null. </exception>
        public async Task<Response<ChatMessageInternal>> GetChatMessageAsync(string chatThreadId, string chatMessageId, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }
            if (chatMessageId == null)
            {
                throw new ArgumentNullException(nameof(chatMessageId));
            }

            using var message = CreateGetChatMessageRequest(chatThreadId, chatMessageId);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatMessageInternal value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ChatMessageInternal.DeserializeChatMessageInternal(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Gets a message by id. </summary>
        /// <param name="chatThreadId"> The thread id to which the message was sent. </param>
        /// <param name="chatMessageId"> The message id. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> or <paramref name="chatMessageId"/> is null. </exception>
        public Response<ChatMessageInternal> GetChatMessage(string chatThreadId, string chatMessageId, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }
            if (chatMessageId == null)
            {
                throw new ArgumentNullException(nameof(chatMessageId));
            }

            using var message = CreateGetChatMessageRequest(chatThreadId, chatMessageId);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatMessageInternal value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ChatMessageInternal.DeserializeChatMessageInternal(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateUpdateChatMessageRequest(string chatThreadId, string chatMessageId, string content, IDictionary<string, string> metadata)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Patch;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendPath("/messages/", false);
            uri.AppendPath(chatMessageId, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/merge-patch+json");
            UpdateChatMessageRequest updateChatMessageRequest = new UpdateChatMessageRequest()
            {
                Content = content
            };
            if (metadata != null)
            {
                foreach (var value in metadata)
                {
                    updateChatMessageRequest.Metadata.Add(value);
                }
            }
            var model = updateChatMessageRequest;
            var content0 = new Utf8JsonRequestContent();
            content0.JsonWriter.WriteObjectValue(model);
            request.Content = content0;
            return message;
        }

        /// <summary> Updates a message. </summary>
        /// <param name="chatThreadId"> The thread id to which the message was sent. </param>
        /// <param name="chatMessageId"> The message id. </param>
        /// <param name="content"> Chat message content. </param>
        /// <param name="metadata"> Message metadata. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> or <paramref name="chatMessageId"/> is null. </exception>
        public async Task<Response> UpdateChatMessageAsync(string chatThreadId, string chatMessageId, string content = null, IDictionary<string, string> metadata = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }
            if (chatMessageId == null)
            {
                throw new ArgumentNullException(nameof(chatMessageId));
            }

            using var message = CreateUpdateChatMessageRequest(chatThreadId, chatMessageId, content, metadata);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 204:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Updates a message. </summary>
        /// <param name="chatThreadId"> The thread id to which the message was sent. </param>
        /// <param name="chatMessageId"> The message id. </param>
        /// <param name="content"> Chat message content. </param>
        /// <param name="metadata"> Message metadata. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> or <paramref name="chatMessageId"/> is null. </exception>
        public Response UpdateChatMessage(string chatThreadId, string chatMessageId, string content = null, IDictionary<string, string> metadata = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }
            if (chatMessageId == null)
            {
                throw new ArgumentNullException(nameof(chatMessageId));
            }

            using var message = CreateUpdateChatMessageRequest(chatThreadId, chatMessageId, content, metadata);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 204:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateDeleteChatMessageRequest(string chatThreadId, string chatMessageId)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Delete;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendPath("/messages/", false);
            uri.AppendPath(chatMessageId, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Deletes a message. </summary>
        /// <param name="chatThreadId"> The thread id to which the message was sent. </param>
        /// <param name="chatMessageId"> The message id. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> or <paramref name="chatMessageId"/> is null. </exception>
        public async Task<Response> DeleteChatMessageAsync(string chatThreadId, string chatMessageId, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }
            if (chatMessageId == null)
            {
                throw new ArgumentNullException(nameof(chatMessageId));
            }

            using var message = CreateDeleteChatMessageRequest(chatThreadId, chatMessageId);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 204:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Deletes a message. </summary>
        /// <param name="chatThreadId"> The thread id to which the message was sent. </param>
        /// <param name="chatMessageId"> The message id. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> or <paramref name="chatMessageId"/> is null. </exception>
        public Response DeleteChatMessage(string chatThreadId, string chatMessageId, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }
            if (chatMessageId == null)
            {
                throw new ArgumentNullException(nameof(chatMessageId));
            }

            using var message = CreateDeleteChatMessageRequest(chatThreadId, chatMessageId);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 204:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateListChatParticipantsRequest(string chatThreadId, int? maxPageSize, int? skip)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendPath("/participants", false);
            if (maxPageSize != null)
            {
                uri.AppendQuery("maxPageSize", maxPageSize.Value, true);
            }
            if (skip != null)
            {
                uri.AppendQuery("skip", skip.Value, true);
            }
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Gets the participants of a thread. </summary>
        /// <param name="chatThreadId"> Thread id to get participants for. </param>
        /// <param name="maxPageSize"> The maximum number of participants to be returned per page. </param>
        /// <param name="skip"> Skips participants up to a specified position in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public async Task<Response<ChatParticipantsCollection>> ListChatParticipantsAsync(string chatThreadId, int? maxPageSize = null, int? skip = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateListChatParticipantsRequest(chatThreadId, maxPageSize, skip);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatParticipantsCollection value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ChatParticipantsCollection.DeserializeChatParticipantsCollection(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Gets the participants of a thread. </summary>
        /// <param name="chatThreadId"> Thread id to get participants for. </param>
        /// <param name="maxPageSize"> The maximum number of participants to be returned per page. </param>
        /// <param name="skip"> Skips participants up to a specified position in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public Response<ChatParticipantsCollection> ListChatParticipants(string chatThreadId, int? maxPageSize = null, int? skip = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateListChatParticipantsRequest(chatThreadId, maxPageSize, skip);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatParticipantsCollection value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ChatParticipantsCollection.DeserializeChatParticipantsCollection(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateRemoveChatParticipantRequest(string chatThreadId, CommunicationIdentifierModelKind? kind, string rawId, CommunicationUserIdentifierModel communicationUser, PhoneNumberIdentifierModel phoneNumber, MicrosoftTeamsUserIdentifierModel microsoftTeamsUser, MicrosoftTeamsAppIdentifierModel microsoftTeamsApp)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendPath("/participants/:remove", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/json");
            var model = new CommunicationIdentifierModel()
            {
                Kind = kind,
                RawId = rawId,
                CommunicationUser = communicationUser,
                PhoneNumber = phoneNumber,
                MicrosoftTeamsUser = microsoftTeamsUser,
                MicrosoftTeamsApp = microsoftTeamsApp
            };
            var content = new Utf8JsonRequestContent();
            content.JsonWriter.WriteObjectValue(model);
            request.Content = content;
            return message;
        }

        /// <summary> Remove a participant from a thread. </summary>
        /// <param name="chatThreadId"> Thread id to remove the participant from. </param>
        /// <param name="kind"> The identifier kind. Only required in responses. </param>
        /// <param name="rawId"> Raw Id of the identifier. Optional in requests, required in responses. </param>
        /// <param name="communicationUser"> The communication user. </param>
        /// <param name="phoneNumber"> The phone number. </param>
        /// <param name="microsoftTeamsUser"> The Microsoft Teams user. </param>
        /// <param name="microsoftTeamsApp"> The Microsoft Teams application. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public async Task<Response> RemoveChatParticipantAsync(string chatThreadId, CommunicationIdentifierModelKind? kind = null, string rawId = null, CommunicationUserIdentifierModel communicationUser = null, PhoneNumberIdentifierModel phoneNumber = null, MicrosoftTeamsUserIdentifierModel microsoftTeamsUser = null, MicrosoftTeamsAppIdentifierModel microsoftTeamsApp = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateRemoveChatParticipantRequest(chatThreadId, kind, rawId, communicationUser, phoneNumber, microsoftTeamsUser, microsoftTeamsApp);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 204:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Remove a participant from a thread. </summary>
        /// <param name="chatThreadId"> Thread id to remove the participant from. </param>
        /// <param name="kind"> The identifier kind. Only required in responses. </param>
        /// <param name="rawId"> Raw Id of the identifier. Optional in requests, required in responses. </param>
        /// <param name="communicationUser"> The communication user. </param>
        /// <param name="phoneNumber"> The phone number. </param>
        /// <param name="microsoftTeamsUser"> The Microsoft Teams user. </param>
        /// <param name="microsoftTeamsApp"> The Microsoft Teams application. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public Response RemoveChatParticipant(string chatThreadId, CommunicationIdentifierModelKind? kind = null, string rawId = null, CommunicationUserIdentifierModel communicationUser = null, PhoneNumberIdentifierModel phoneNumber = null, MicrosoftTeamsUserIdentifierModel microsoftTeamsUser = null, MicrosoftTeamsAppIdentifierModel microsoftTeamsApp = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateRemoveChatParticipantRequest(chatThreadId, kind, rawId, communicationUser, phoneNumber, microsoftTeamsUser, microsoftTeamsApp);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 204:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateAddChatParticipantsRequest(string chatThreadId, IEnumerable<ChatParticipantInternal> participants)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendPath("/participants/:add", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/json");
            var model = new AddChatParticipantsRequest(participants.ToList());
            var content = new Utf8JsonRequestContent();
            content.JsonWriter.WriteObjectValue(model);
            request.Content = content;
            return message;
        }

        /// <summary> Adds thread participants to a thread. If participants already exist, no change occurs. </summary>
        /// <param name="chatThreadId"> Id of the thread to add participants to. </param>
        /// <param name="participants"> Participants to add to a chat thread. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> or <paramref name="participants"/> is null. </exception>
        public async Task<Response<AddChatParticipantsResult>> AddChatParticipantsAsync(string chatThreadId, IEnumerable<ChatParticipantInternal> participants, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }
            if (participants == null)
            {
                throw new ArgumentNullException(nameof(participants));
            }

            using var message = CreateAddChatParticipantsRequest(chatThreadId, participants);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 201:
                    {
                        AddChatParticipantsResult value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = AddChatParticipantsResult.DeserializeAddChatParticipantsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Adds thread participants to a thread. If participants already exist, no change occurs. </summary>
        /// <param name="chatThreadId"> Id of the thread to add participants to. </param>
        /// <param name="participants"> Participants to add to a chat thread. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> or <paramref name="participants"/> is null. </exception>
        public Response<AddChatParticipantsResult> AddChatParticipants(string chatThreadId, IEnumerable<ChatParticipantInternal> participants, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }
            if (participants == null)
            {
                throw new ArgumentNullException(nameof(participants));
            }

            using var message = CreateAddChatParticipantsRequest(chatThreadId, participants);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 201:
                    {
                        AddChatParticipantsResult value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = AddChatParticipantsResult.DeserializeAddChatParticipantsResult(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateUpdateChatThreadPropertiesRequest(string chatThreadId, string topic)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Patch;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/merge-patch+json");
            var model = new UpdateChatThreadRequest()
            {
                Topic = topic
            };
            var content = new Utf8JsonRequestContent();
            content.JsonWriter.WriteObjectValue(model);
            request.Content = content;
            return message;
        }

        /// <summary> Updates a thread's properties. </summary>
        /// <param name="chatThreadId"> The id of the thread to update. </param>
        /// <param name="topic"> Chat thread topic. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public async Task<Response> UpdateChatThreadPropertiesAsync(string chatThreadId, string topic = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateUpdateChatThreadPropertiesRequest(chatThreadId, topic);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 204:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Updates a thread's properties. </summary>
        /// <param name="chatThreadId"> The id of the thread to update. </param>
        /// <param name="topic"> Chat thread topic. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public Response UpdateChatThreadProperties(string chatThreadId, string topic = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateUpdateChatThreadPropertiesRequest(chatThreadId, topic);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 204:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateGetChatThreadPropertiesRequest(string chatThreadId)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Gets a chat thread's properties. </summary>
        /// <param name="chatThreadId"> Id of the thread. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public async Task<Response<ChatThreadPropertiesInternal>> GetChatThreadPropertiesAsync(string chatThreadId, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateGetChatThreadPropertiesRequest(chatThreadId);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatThreadPropertiesInternal value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ChatThreadPropertiesInternal.DeserializeChatThreadPropertiesInternal(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Gets a chat thread's properties. </summary>
        /// <param name="chatThreadId"> Id of the thread. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public Response<ChatThreadPropertiesInternal> GetChatThreadProperties(string chatThreadId, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateGetChatThreadPropertiesRequest(chatThreadId);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatThreadPropertiesInternal value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ChatThreadPropertiesInternal.DeserializeChatThreadPropertiesInternal(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateSendTypingNotificationRequest(string chatThreadId, string senderDisplayName)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Post;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendPath("/chat/threads/", false);
            uri.AppendPath(chatThreadId, true);
            uri.AppendPath("/typing", false);
            uri.AppendQuery("api-version", _apiVersion, true);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            request.Headers.Add("Content-Type", "application/json");
            var model = new SendTypingNotificationRequest()
            {
                SenderDisplayName = senderDisplayName
            };
            var content = new Utf8JsonRequestContent();
            content.JsonWriter.WriteObjectValue(model);
            request.Content = content;
            return message;
        }

        /// <summary> Posts a typing event to a thread, on behalf of a user. </summary>
        /// <param name="chatThreadId"> Id of the thread. </param>
        /// <param name="senderDisplayName"> The display name of the typing notification sender. This property is used to populate sender name for push notifications. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public async Task<Response> SendTypingNotificationAsync(string chatThreadId, string senderDisplayName = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateSendTypingNotificationRequest(chatThreadId, senderDisplayName);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Posts a typing event to a thread, on behalf of a user. </summary>
        /// <param name="chatThreadId"> Id of the thread. </param>
        /// <param name="senderDisplayName"> The display name of the typing notification sender. This property is used to populate sender name for push notifications. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="chatThreadId"/> is null. </exception>
        public Response SendTypingNotification(string chatThreadId, string senderDisplayName = null, CancellationToken cancellationToken = default)
        {
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateSendTypingNotificationRequest(chatThreadId, senderDisplayName);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    return message.Response;
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateListChatReadReceiptsNextPageRequest(string nextLink, string chatThreadId, int? maxPageSize, int? skip)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Gets chat message read receipts for a thread. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="chatThreadId"> Thread id to get the chat message read receipts for. </param>
        /// <param name="maxPageSize"> The maximum number of chat message read receipts to be returned per page. </param>
        /// <param name="skip"> Skips chat message read receipts up to a specified position in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="chatThreadId"/> is null. </exception>
        public async Task<Response<ChatMessageReadReceiptsCollection>> ListChatReadReceiptsNextPageAsync(string nextLink, string chatThreadId, int? maxPageSize = null, int? skip = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateListChatReadReceiptsNextPageRequest(nextLink, chatThreadId, maxPageSize, skip);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatMessageReadReceiptsCollection value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ChatMessageReadReceiptsCollection.DeserializeChatMessageReadReceiptsCollection(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Gets chat message read receipts for a thread. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="chatThreadId"> Thread id to get the chat message read receipts for. </param>
        /// <param name="maxPageSize"> The maximum number of chat message read receipts to be returned per page. </param>
        /// <param name="skip"> Skips chat message read receipts up to a specified position in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="chatThreadId"/> is null. </exception>
        public Response<ChatMessageReadReceiptsCollection> ListChatReadReceiptsNextPage(string nextLink, string chatThreadId, int? maxPageSize = null, int? skip = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateListChatReadReceiptsNextPageRequest(nextLink, chatThreadId, maxPageSize, skip);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatMessageReadReceiptsCollection value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ChatMessageReadReceiptsCollection.DeserializeChatMessageReadReceiptsCollection(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateListChatMessagesNextPageRequest(string nextLink, string chatThreadId, int? maxPageSize, DateTimeOffset? startTime)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Gets a list of messages from a thread. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="chatThreadId"> The thread id of the message. </param>
        /// <param name="maxPageSize"> The maximum number of messages to be returned per page. </param>
        /// <param name="startTime"> The earliest point in time to get messages up to. The timestamp should be in RFC3339 format: `yyyy-MM-ddTHH:mm:ssZ`. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="chatThreadId"/> is null. </exception>
        public async Task<Response<ChatMessagesCollection>> ListChatMessagesNextPageAsync(string nextLink, string chatThreadId, int? maxPageSize = null, DateTimeOffset? startTime = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateListChatMessagesNextPageRequest(nextLink, chatThreadId, maxPageSize, startTime);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatMessagesCollection value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ChatMessagesCollection.DeserializeChatMessagesCollection(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Gets a list of messages from a thread. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="chatThreadId"> The thread id of the message. </param>
        /// <param name="maxPageSize"> The maximum number of messages to be returned per page. </param>
        /// <param name="startTime"> The earliest point in time to get messages up to. The timestamp should be in RFC3339 format: `yyyy-MM-ddTHH:mm:ssZ`. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="chatThreadId"/> is null. </exception>
        public Response<ChatMessagesCollection> ListChatMessagesNextPage(string nextLink, string chatThreadId, int? maxPageSize = null, DateTimeOffset? startTime = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateListChatMessagesNextPageRequest(nextLink, chatThreadId, maxPageSize, startTime);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatMessagesCollection value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ChatMessagesCollection.DeserializeChatMessagesCollection(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        internal HttpMessage CreateListChatParticipantsNextPageRequest(string nextLink, string chatThreadId, int? maxPageSize, int? skip)
        {
            var message = _pipeline.CreateMessage();
            var request = message.Request;
            request.Method = RequestMethod.Get;
            var uri = new RawRequestUriBuilder();
            uri.AppendRaw(_endpoint, false);
            uri.AppendRawNextLink(nextLink, false);
            request.Uri = uri;
            request.Headers.Add("Accept", "application/json");
            return message;
        }

        /// <summary> Gets the participants of a thread. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="chatThreadId"> Thread id to get participants for. </param>
        /// <param name="maxPageSize"> The maximum number of participants to be returned per page. </param>
        /// <param name="skip"> Skips participants up to a specified position in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="chatThreadId"/> is null. </exception>
        public async Task<Response<ChatParticipantsCollection>> ListChatParticipantsNextPageAsync(string nextLink, string chatThreadId, int? maxPageSize = null, int? skip = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateListChatParticipantsNextPageRequest(nextLink, chatThreadId, maxPageSize, skip);
            await _pipeline.SendAsync(message, cancellationToken).ConfigureAwait(false);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatParticipantsCollection value = default;
                        using var document = await JsonDocument.ParseAsync(message.Response.ContentStream, default, cancellationToken).ConfigureAwait(false);
                        value = ChatParticipantsCollection.DeserializeChatParticipantsCollection(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }

        /// <summary> Gets the participants of a thread. </summary>
        /// <param name="nextLink"> The URL to the next page of results. </param>
        /// <param name="chatThreadId"> Thread id to get participants for. </param>
        /// <param name="maxPageSize"> The maximum number of participants to be returned per page. </param>
        /// <param name="skip"> Skips participants up to a specified position in response. </param>
        /// <param name="cancellationToken"> The cancellation token to use. </param>
        /// <exception cref="ArgumentNullException"> <paramref name="nextLink"/> or <paramref name="chatThreadId"/> is null. </exception>
        public Response<ChatParticipantsCollection> ListChatParticipantsNextPage(string nextLink, string chatThreadId, int? maxPageSize = null, int? skip = null, CancellationToken cancellationToken = default)
        {
            if (nextLink == null)
            {
                throw new ArgumentNullException(nameof(nextLink));
            }
            if (chatThreadId == null)
            {
                throw new ArgumentNullException(nameof(chatThreadId));
            }

            using var message = CreateListChatParticipantsNextPageRequest(nextLink, chatThreadId, maxPageSize, skip);
            _pipeline.Send(message, cancellationToken);
            switch (message.Response.Status)
            {
                case 200:
                    {
                        ChatParticipantsCollection value = default;
                        using var document = JsonDocument.Parse(message.Response.ContentStream);
                        value = ChatParticipantsCollection.DeserializeChatParticipantsCollection(document.RootElement);
                        return Response.FromValue(value, message.Response);
                    }
                default:
                    throw new RequestFailedException(message.Response);
            }
        }
    }
}
