﻿// Copyright (c) Microsoft Corporation. All rights reserved.
// Licensed under the MIT License.

using System;

namespace Azure.AI.Inference
{
    public partial class ChatCompletionsToolCall
    {
        // CUSTOM CODE NOTE:
        //   This code allows the concrete tool call type to directly pass through use of its underlying function
        //   rather than having a separate layer of indirection.

        /// <inheritdoc cref="FunctionCall.Name"/>
        public string Name
        {
            get => Function.Name;
            set => Function.Name = value;
        }

        /// <inheritdoc cref="FunctionCall.Arguments"/>
        public string Arguments
        {
            get => Function.Arguments;
            set => Function.Arguments = value;
        }

        /// <summary> Creates a new <see cref="ChatCompletionsToolCall"/> representing a function call made by the model. </summary>
        /// <param name="toolCallId"> The ID of the tool call. </param>
        /// <param name="functionName"> The name of the function that model is calling. </param>
        /// <param name="functionArguments">
        ///     The arguments that model is calling the function with, which are generated by the model in JSON format.
        ///     Note that the model does not always generate valid JSON, and may hallucinate parameters not defined by your
        ///     function schema. Validate the arguments in your code before calling your function.
        /// </param>
        /// <exception cref="ArgumentNullException"> <paramref name="toolCallId"/>, <paramref name="functionName"/> or <paramref name="functionArguments"/> is null. </exception>
        /// <exception cref="ArgumentException"> <paramref name="toolCallId"/>, <paramref name="functionName"/> or <paramref name="functionArguments"/> is an empty string, and was expected to be non-empty. </exception>
        public static ChatCompletionsToolCall CreateFunctionToolCall(string toolCallId, string functionName, string functionArguments)
        {
            Argument.AssertNotNullOrEmpty(toolCallId, nameof(toolCallId));
            Argument.AssertNotNullOrEmpty(functionName, nameof(functionName));
            Argument.AssertNotNullOrEmpty(functionArguments, nameof(functionArguments));

            FunctionCall function = new(functionName, functionArguments);

            return new(
                id: toolCallId,
                type: ChatCompletionsToolCallType.Function,
                function: function,
                serializedAdditionalRawData: null);
        }
    }
}
