<?php

declare(strict_types=1);

namespace App\Controller\Api\Stations;

use App\Entity\StationHlsStream;
use App\Exception\ValidationException;
use App\Http\ServerRequest;
use App\OpenApi;
use OpenApi\Attributes as OA;

/** @extends AbstractStationApiCrudController<StationHlsStream> */
#[
    OA\Get(
        path: '/station/{station_id}/hls_streams',
        operationId: 'getHlsStreams',
        description: 'List all current HLS streams.',
        security: OpenApi::API_KEY_SECURITY,
        tags: ['Stations: HLS Streams'],
        parameters: [
            new OA\Parameter(ref: OpenApi::REF_STATION_ID_REQUIRED),
        ],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Success',
                content: new OA\JsonContent(
                    type: 'array',
                    items: new OA\Items(ref: '#/components/schemas/StationMount')
                )
            ),
            new OA\Response(ref: OpenApi::REF_RESPONSE_ACCESS_DENIED, response: 403),
            new OA\Response(ref: OpenApi::REF_RESPONSE_GENERIC_ERROR, response: 500),
        ]
    ),
    OA\Post(
        path: '/station/{station_id}/hls_streams',
        operationId: 'addHlsStream',
        description: 'Create a new HLS stream.',
        security: OpenApi::API_KEY_SECURITY,
        requestBody: new OA\RequestBody(
            content: new OA\JsonContent(ref: '#/components/schemas/StationMount')
        ),
        tags: ['Stations: HLS Streams'],
        parameters: [
            new OA\Parameter(ref: OpenApi::REF_STATION_ID_REQUIRED),
        ],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Success',
                content: new OA\JsonContent(ref: '#/components/schemas/StationMount')
            ),
            new OA\Response(ref: OpenApi::REF_RESPONSE_ACCESS_DENIED, response: 403),
            new OA\Response(ref: OpenApi::REF_RESPONSE_GENERIC_ERROR, response: 500),
        ]
    ),
    OA\Get(
        path: '/station/{station_id}/hls_stream/{id}',
        operationId: 'getHlsStream',
        description: 'Retrieve details for a single HLS stream.',
        security: OpenApi::API_KEY_SECURITY,
        tags: ['Stations: HLS Streams'],
        parameters: [
            new OA\Parameter(ref: OpenApi::REF_STATION_ID_REQUIRED),
            new OA\Parameter(
                name: 'id',
                description: 'HLS Stream ID',
                in: 'path',
                required: true,
                schema: new OA\Schema(type: 'integer', format: 'int64')
            ),
        ],
        responses: [
            new OA\Response(
                response: 200,
                description: 'Success',
                content: new OA\JsonContent(ref: '#/components/schemas/StationMount')
            ),
            new OA\Response(ref: OpenApi::REF_RESPONSE_ACCESS_DENIED, response: 403),
            new OA\Response(ref: OpenApi::REF_RESPONSE_NOT_FOUND, response: 404),
            new OA\Response(ref: OpenApi::REF_RESPONSE_GENERIC_ERROR, response: 500),
        ]
    ),
    OA\Put(
        path: '/station/{station_id}/hls_stream/{id}',
        operationId: 'editHlsStream',
        description: 'Update details of a single HLS stream.',
        security: OpenApi::API_KEY_SECURITY,
        requestBody: new OA\RequestBody(
            content: new OA\JsonContent(ref: '#/components/schemas/StationMount')
        ),
        tags: ['Stations: HLS Streams'],
        parameters: [
            new OA\Parameter(ref: OpenApi::REF_STATION_ID_REQUIRED),
            new OA\Parameter(
                name: 'id',
                description: 'HLS Stream ID',
                in: 'path',
                required: true,
                schema: new OA\Schema(type: 'integer', format: 'int64')
            ),
        ],
        responses: [
            new OA\Response(ref: OpenApi::REF_RESPONSE_SUCCESS, response: 200),
            new OA\Response(ref: OpenApi::REF_RESPONSE_ACCESS_DENIED, response: 403),
            new OA\Response(ref: OpenApi::REF_RESPONSE_NOT_FOUND, response: 404),
            new OA\Response(ref: OpenApi::REF_RESPONSE_GENERIC_ERROR, response: 500),
        ]
    ),
    OA\Delete(
        path: '/station/{station_id}/hls_stream/{id}',
        operationId: 'deleteHlsStream',
        description: 'Delete a single HLS stream.',
        security: OpenApi::API_KEY_SECURITY,
        tags: ['Stations: HLS Streams'],
        parameters: [
            new OA\Parameter(ref: OpenApi::REF_STATION_ID_REQUIRED),
            new OA\Parameter(
                name: 'id',
                description: 'HLS Stream ID',
                in: 'path',
                required: true,
                schema: new OA\Schema(type: 'integer', format: 'int64')
            ),
        ],
        responses: [
            new OA\Response(ref: OpenApi::REF_RESPONSE_SUCCESS, response: 200),
            new OA\Response(ref: OpenApi::REF_RESPONSE_ACCESS_DENIED, response: 403),
            new OA\Response(ref: OpenApi::REF_RESPONSE_NOT_FOUND, response: 404),
            new OA\Response(ref: OpenApi::REF_RESPONSE_GENERIC_ERROR, response: 500),
        ]
    )
]
final class HlsStreamsController extends AbstractStationApiCrudController
{
    protected string $entityClass = StationHlsStream::class;
    protected string $resourceRouteName = 'api:stations:hls_stream';

    protected function createRecord(ServerRequest $request, array $data): object
    {
        $station = $request->getStation();
        if ($station->getMaxHlsStreams() !== 0 && $station->getMaxHlsStreams() <= $station->getHlsStreams()->count()) {
            throw new ValidationException(
                __('Unable to create a new stream, station\'s maximum HLS streams reached.')
            );
        }

        return parent::createRecord($request, $data);
    }
}
