---
lang-ref: ch.13-3
title: 图形卷积网络 III
lecturer: Alfredo Canziani
authors: Go Inoue, Muhammad Osama Khan, Muhammad Shujaat Mirza, Muhammad Muneeb Afzal
date: 28 Apr 2020
lang: zh
translation-date: 17 Sep 2020
translator: Jonathan Sum(😊🍩📙)
---


## [图形卷积网络(英文简称GCN)](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=47s)

图形卷积网络是一种使用数据结构的架构。在说得更深入一点前，让我们快速地回顾一下自我注意力机制，如图形卷积网络和自我注意力机制都是埋论上一样的。


### 回顾一下: 自我注意力机制

<!-- - In self-attention, we have a set of input $\lbrace\boldsymbol{x}\_{i}\rbrace^{t}_{i=1}$.
Unlike a sequence, it does not have an order.
- Hidden vector $\boldsymbol{h}$ is given by linear combination of the vectors in the set.
- We can express this as $\boldsymbol{X}\boldsymbol{a}$ using matrix vector multiplication, where $\boldsymbol{a}$ contains coefficients that scale the input vector $\boldsymbol{x}_{i}$. -->
在自我注意力机制中，我们有一个输入集$\lbrace\boldsymbol{x}\_{i}\rbrace^{t}\_{i=1}$。不像序列那样，它没有顺序的。
隐藏向量$\boldsymbol{h}$是由集之中的向量的线性组合得出来的。
我们可以用矩阵向量乘法来以$\boldsymbol{X}\boldsymbol{a}$去表达这个东西，这里$\boldsymbol{a}$包含一些会缩放向量$\boldsymbol{x}_{i}$的系数。


<!-- *For a detailed explanation, refer to the notes of [Week 12]({{site.baseurl}}/en/week12/12-3/).* -->
*如果想要更多深入的解释，请看[第12个星期的笔记]({{site.baseurl}}/en/week12/12-3/)。*


<!-- ### Notation -->
### 符号

<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure1.png" height="400px" /><br>
<b>图 1</b>: 图形卷积网络
</center>

<!-- In Figure 1, vertex $v$ is comprised of two vectors: input $\boldsymbol{x}$ and its hidden representation $\boldsymbol{h}$.
We also have multiple vertices $v_{j}$, which is comprised of $\boldsymbol{x}\_j$ and $\boldsymbol{h}\_j$.
In this graph, vertices are connected with directed edges. -->
在图1，顶点$v$包含两个向量:输入$\boldsymbol{x}$和它的隐蔽表示$\boldsymbol{h}$。我们也有多个顶点$v_{j}$，它們包含$\boldsymbol{x}\_j$ 和 $\boldsymbol{h}\_j$。在这个图形，顶点都是以有向边来连接起来。

<!-- We represent these directed edges with adjacency vector $\boldsymbol{a}$, where each element $\alpha_{j}$ is set to $1$ if there is a directed edge from $v_{j}$ to $v$. -->
我们以邻接向量$\boldsymbol{a}$来代表这些有向边，这里如果是一个有向边由$v_{j}$到$v$的话，那每一个元件$\alpha_{j}$都设定为$1$。 

$$
\alpha_{j} \stackrel{\tiny \downarrow}{=} 1 \Leftrightarrow v_{j} \rightarrow v
\tag{方程式1}
$$

<!-- The degree (number of incoming edges) $d$ is defined as the norm of this adjacency vector, *i.e.* $\Vert\boldsymbol{a}\Vert_{1} $, which is the number of ones in the vector $\boldsymbol{a}$. -->
度数$d$，也就是连接入来的边的数量是多少来决定度数，而它是以一个邻接向量的范数来定义出来，*比如*$\Vert\boldsymbol{a}\Vert_{1} $，也就是在向量$\boldsymbol{a}$中有多少个一。

$$
    d = \Vert\boldsymbol{a}\Vert_{1}
\tag{方程式2}
$$

<!-- The hidden vector $\boldsymbol{h}$ is given by the following expression: -->
而隐藏向量$\boldsymbol{h}$是以这个表达式来表逹:

$$
    \boldsymbol{h}=f(\boldsymbol{U}\boldsymbol{x} + \boldsymbol{V}\boldsymbol{X}\boldsymbol{a}d^{-1})
\tag{方程式3}
$$

<!-- where $f(\cdot)$ is a non-linear function such as ReLU $(\cdot)^{+}$, Sigmoid $\sigma(\cdot)$, and hyperbolic tangent $\tanh(\cdot)$. -->
这里$f(\cdot)$是一个非线性函数，比如修正线性单元英文为ReLU，符号为$(\cdot)^{+}$，而S形函数的符号为$\sigma(\cdot)$，而双曲正切的符号为$\tanh(\cdot)$。

<!-- The $\boldsymbol{U}\boldsymbol{x}$ term takes into account the vertex $v$ itself, by applying rotation $\boldsymbol{U}$ to the input $v$. -->
这个$\boldsymbol{U}\boldsymbol{x}$项是要用在顶点$v$上，就是对输入$v$用上这个旋转$\boldsymbol{U}$。

<!-- Remember that in self-attention, the hidden vector $\boldsymbol{h}$ is computed by $\boldsymbol{X}\boldsymbol{a}$, which means that the columns in $\boldsymbol{X}$ is scaled by the factors in $\boldsymbol{a}$.
In the context of GCN, this means that if we have multiple incoming edges,*i.e.* multiple ones in adjacency vector $\boldsymbol{a}$, $\boldsymbol{X}\boldsymbol{a}$ gets larger.
On the other hand, if we have only one incoming edge, this value gets smaller.
To remedy this issue of the value being proportionate to the number of incoming edges, we divide it by the number of incoming edges $d$.
We then apply rotation $\boldsymbol{V}$ to $\boldsymbol{X}\boldsymbol{a}d^{-1}$. -->
记得在自我注意力机制中，隐藏向量$\boldsymbol{h}$是以$\boldsymbol{X}\boldsymbol{a}$来计算出来，
也就是说在$\boldsymbol{X}$中的行是以在$\boldsymbol{a}$中的因素缩放出来的。而图形卷积网络的内容中，
这意思着这样，如果我们有多个由外传入来的边，比如在邻接向量$\boldsymbol{a}$中有多个1，$\_boldsymbol{X}\_boldsymbol{a}$就会变得很大。
在另一面，如果我们只有一个传入来的边，这个值就会变得很细。
去补救这个数值与由外传入来的边是成比例的问题，我们对它除以一个数，
这个数就是由外传入来的边有多少的数量，这个数叫$d$。我们之后对$\_boldsymbol{X}\_boldsymbol{a}d^{-1}$使用一个旋转$\_boldsymbol{V}$。


<!-- We can represent this hidden representation $\boldsymbol{h}$ for the entire set of inputs $\boldsymbol{x}$ using the following matrix notation: -->
我們可以以這個隱藏表示$\boldsymbol{h}$來以下方的矩陣符號來表達整個輸入集$\boldsymbol{x}$

$$
\{\boldsymbol{x}_{i}\}^{t}_{i=1}\rightsquigarrow \boldsymbol{H}=f(\boldsymbol{UX}+ \boldsymbol{VXAD}^{-1})
\tag{方程式4}
$$

where $\vect{D} = \text{diag}(d_{i})$.


<!-- ## [Residual Gated GCN Theory and Code](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=992s) -->
## [残差门控式图形卷积网络理论和代码](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=992s)

<!-- Residual Gated Graph Convolutional Network is a type of GCN that can be represented as shown in Figure 2: -->
残差门控式图形卷积网络是一种图形卷积网络，也可以以图2来表示出来:

<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure2.png" height="300px" /><br>
<!-- <b>Fig. 2</b>: Residual Gated Graph Convolutional Network -->
<b>图 2</b>: 残差门控式图形卷积网络
</center>

<!-- As with the standard GCN, the vertex $v$ consists of two vectors: input $\boldsymbol{x}$ and its hidden representation $\boldsymbol{h}$. However, in this case, the edges also have a feature representation, where $\boldsymbol{e_{j}^{x}}$ represents the input edge representation and $\boldsymbol{e_{j}^{h}}$ represents the hidden edge representation. -->
以一個標準的圖形卷積网絡，頂點$v$由這兩個向量組成:輸入$\boldsymbol{x}$和它的隱藏表示$\boldsymbol{h}$。相反，在這個例子，邊都有一個特徵表示，而這裡的$\boldsymbol{e_{j}^{x}}$ 代表著一個輸入邊表示和$\boldsymbol{e_{j}^{h}}$代表一個隱藏邊表示。


<!-- The hidden representation $\boldsymbol{h}$ of the vertex $v$ is computed by the following equation: -->
就如基本型图形卷积网络，顶点$v$由两个向量组成:输入x和它的隐藏表示$\boldsymbol{h}$ 。

$$
    \boldsymbol{h}=\boldsymbol{x} + \bigg(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}\bigg)^{+}
\tag{Eq. 5}
$$

<!-- where $\boldsymbol{x}$ is the input representation, $\boldsymbol{Ax}$ represents a rotation applied to the input $\boldsymbol{x}$ and $\sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}$ denotes the summation of elementwise multiplication of rotated incoming features $\boldsymbol{Bx_{j}}$ and a gate $\eta(\boldsymbol{e_{j}})$. In contrast to the standard GCN above where we average the incoming representations, the gate term is critical to the implementation of Residual Gated GCN since it allows us to modulate the incoming representations based on the edge representations. -->
这里$\boldsymbol{x}$是一个输入表示，而$\boldsymbol{Ax}$代表对输入$\boldsymbol{x}$用上了一个旋转，而且$\sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}$表示一个已旋转了的传入来的特征$\boldsymbol{Bx_{j}}$和门$\eta(\boldsymbol{e_{j}})$的逐元素乘法，然后最后就来一个总结。和标准的图形卷积网络比较，上方的就会对传入来的表示进行平均值运算，而门控项是对实作残差门控式图形卷积网络来说是最重要的，因为它容许我们去调节传入来的基于边的表示。

<!-- Note that the summation is only over vertices ${v_j}$ that have incoming edges to vertex ${v}$. The term residual (in Residual Gated GCN) comes from the fact that in order to calculate the hidden representation $\boldsymbol{h}$, we add the input representation $\boldsymbol{x}$. The gate term $\eta(\boldsymbol{e_{j}})$ is calculated as shown below: -->
注意到的是总和只是总和所有顶点${v_j}$，而这些顶点都有传入来的边连到顶点${v}$。而在残差门控式图形卷积网络中的项:「残差」，这个项事实上来自一个做法，如果要计算出隐藏$\boldsymbol{h}$，我们就要加那个输入表示$\boldsymbol{x}$。而那个门项$\eta(\boldsymbol{e_{j}})$是以下方那样来计算:

$$
    \eta(\boldsymbol{e_{j}})=\sigma(\boldsymbol{e_{j}})\bigg(\sum_{v_k→v}\sigma(\boldsymbol{e_{k}})\bigg)^{-1}
\tag{方程式6}
$$

<!-- The gate value $\eta(\boldsymbol{e_{j}})$ is a normalized sigmoid obtained by dividing the sigmoid of the incoming edge representation by the sum of sigmoids of all incoming edge representations. Note that in order to calculate the gate term, we need the representation of the edge $\boldsymbol{e_{j}}$, which can be computed using the equations below: -->
而门数值$\eta(\boldsymbol{e_{j}})$是归一化后的S型函数，取得它的方法是这样的:先把所有传入来的边的表示每一个进行S型函数，然后加起来，最后就是对所有传入来的边的表示进行了S型函数后除以刚刚加好的数。注意的是，为了计算门项，我们要边$\boldsymbol{e_{j}}$的表示，也就可能会被这样计算出来，以以下的方程式:

$$
    \boldsymbol{e_{j}} = \boldsymbol{Ce_{j}^{x}} + \boldsymbol{Dx_{j}} + \boldsymbol{Ex}
\tag{方程式7}
$$

$$
    \boldsymbol{e_{j}^{h}}=\boldsymbol{e_{j}^{x}}+(\boldsymbol{e_{j}})^{+}
\tag{方程式8}
$$

<!-- The hidden edge representation $\boldsymbol{e_{j}^{h}}$ is obtained by the summation of the initial edge representation $\boldsymbol{e_{j}^{x}}$ and $\texttt{ReLU}(\cdot)$ applied to $\boldsymbol{e_{j}}$ where $\boldsymbol{e_{j}}$ is in turn given by the summation of a rotation applied to $\boldsymbol{e_{j}^{x}}$, a rotation applied to the input representation $\boldsymbol{x_{j}}$ of the vertex $v_{j}$ and a rotation applied to the input representation $\boldsymbol{x}$ of the vertex $v$. -->
这个隐藏的边表示$\boldsymbol{e_{j}^{h}}$是由总和初始边缘表示$\boldsymbol{e_{j}^{x}}$和$\texttt{ReLU}(\cdot)$所得到，这里的修正线性单元$\texttt{ReLU}(\cdot)$，就是用在$\boldsymbol{e_{j}}$。而$\boldsymbol{e_{j}}$就是由总和所有这些东西们: 1. 用在$\boldsymbol{x_{j}}$上的旋转﹑2.一个用在顶点$v_{j}$的输入表示xj的旋转﹑3.一个用在顶点$v$的输入表示 $\boldsymbol{x}$的旋转。

<!-- *Note: In order to calculate hidden representations downstream (*e.g.* $2^\text{nd}$ layer hidden representations), we can simply replace the input feature representations by the $1^\text{st}$ layer feature representations in the equations above.* -->
*注意: 为了计算出低层的隐藏表示(比如说$2^\text{nd}$层的隐藏表示)，我们可以在上方的方程式简单地更换输入特征表示为$1^\text{st}$层特征表示。*


<!-- ### Graph Classification and Residual Gated GCN Layer -->
### 图形分类和残差门控式图形卷积网络的层

<!-- In this section, we introduce the problem of graph classification and code up a Residual Gated GCN layer. In addition to the usual import statements, we add the following: -->
在这个部份，我们介绍了图形分类问题和写残差门控式图形卷积网络的代码的层。不单止一般也要的汇入们，我们也加这些:

```python
os.environ['DGLBACKEND'] = 'pytorch'
import dgl
from dgl import DGLGraph
from dgl.data import MiniGCDataset
import networkx as nx
```

<!-- The first line tells DGL to use PyTorch as the backend. Deep Graph Library ([DGL](https://www.dgl.ai/)) provides various functionalities on graphs whereas networkx allows us to visualise the graphs. -->
第一行令DGL去用Pytorch作为后端。深度图形函式库([DGL](https://www.dgl.ai/))在图形们上提供各式各样的功能，而networkx容许我们去视觉地看明明这些图形。

<!-- In this notebook, the task is to classify a given graph structure into one of 8 graph types. The dataset obtained from `dgl.data.MiniGCDataset` yields some number of graphs (`num_graphs`) with nodes between `min_num_v` and `max_num_v`. Hence, all the graphs obtained do not have the same number of nodes/vertices. -->
在这个笔记本，任务是去分类出一个给予的图形结构为8种不同的图形的其中之一。而利用`min_num_v`和`max_num_v`之间的节点去在`dgl.data.MiniGCDataset`得到的数据集，就会给出一些图形的数字(`num_graphs`)。所以，所有得到的图形不会有同样的节点或顶点的数字。


<!-- *Note: In order to familiarize yourself with the basics of `DGLGraphs`, it is recommended to go through the short tutorial [here](https://docs.dgl.ai/api/python/graph.html).* -->
*为了使您熟悉DGLGraphs的基础知识，建议您阅读[這个](https://docs.dgl.ai/api/python/graph.html)简短的教程*

<!-- Having created the graphs, the next task is to add some signal to the domain. Features can be applied to nodes and edges of a `DGLGraph`. The features are represented as a dictionary of names (strings) and tensors (**fields**). `ndata` and `edata` are syntax sugar to access the feature data of all nodes and edges. -->
创造这个图形后，下一步就是在域中加一些信号。特征可以被加到去节点和`DGLGraph`的边。而特征是以名字（字符串）和张量（**字段，英文field**）。 `ndata`和`edata`都是语法糖来访问所有节点和边的特征数据。

<!-- The following code snippet shows how the features are generated. Each node is assigned a value equal to the number of incident edges, whereas each edge is assigned the value 1. -->
而接下来的代码片段显示出特征如何生成出来。而每一个节点都设有一个数值，这数值等于有多少条和别的节点连起来的连接边(incident edges)，这里每一条边的数值设为1。
```python
def create_artificial_features(dataset):
    for (graph, _) in dataset:
        graph.ndata['feat'] = graph.in_degrees().view(-1, 1).float()
        graph.edata['feat'] = torch.ones(graph.number_of_edges(), 1)
    return dataset
```

<!-- Training and testing datasets are created and features are assigned as shown below: -->
训练用的数据集和测试用的数据集都被创造出来，而特征都被设为下方那样:

```python
trainset = MiniGCDataset(350, 10, 20)
testset = MiniGCDataset(100, 10, 20)

trainset = create_artificial_features(trainset)
testset = create_artificial_features(testset)
```

<!-- A sample graph from the trainset has the following representation. Here, we observe that the graph has 15 nodes and 45 edges and both the nodes and edges have a feature representation of shape `(1,)` as expected. Furthermore, the `0` signifies that this graph belongs to class 0. -->
一个来的训练集的图形就有以下的设定。这里，我们看到图形有15个节点和45条边，而且两个节点和边都有如预期那样大小为`(1, )`的特征表示。而且，这个0表示这个图形是属于第`0`类。
```python
(DGLGraph(num_nodes=15, num_edges=45,
         ndata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}
         edata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}), 0)
```


<!-- ### Note about DGL Message and Reduce Functions -->
### 有关DGL信息的笔记和降低函数

<!-- >In DGL, the *message functions* are expressed as **Edge UDF**s (User Defined Functions). Edge UDFs take in a single argument `edges`. It has three members `src`, `dst`, and `data` for accessing source node features, destination node features, and edge features respectively.
The *reduce functions* are **Node UDF**s. Node UDFs have a single argument `nodes`, which has two members `data` and `mailbox`. `data` contains the node features and `mailbox` contains all incoming message features, stacked along the second dimension (hence the `dim=1` argument).
`update_all(message_func, reduce_func)` sends messages through all edges and updates all nodes. -->
>在DGL，*信息函数*们都被表达为**边UDFs** (使用者定义出来的函数)。边UDFs会接收一个单一的实参`edges`。它有3个东西:`src`和`dst`﹑`data`。这三个东西都对应地用来访问源头节点的特征和目的地节点的特征﹑边的特征。这个`降低函数`都是``节点UDFs``。节点UDFs有一个实参node(`节点`)，也就是有两个东西`data`和`mailbox`。对应着第二个维度来堆起来(所以这里是`dim=1`)，那`data`就包含的节点特征和`mailbox`就包含所有由外入来的信息特征。而这个函数`update_all(message_func, reduce_func)`会发送所有信息连过所有边，然后更新所有节点。


<!-- ### [Gated Residual GCN Layer Implementation](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=2098s) -->
### [门控式残剩图形卷积网络的层的实作](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=2098s)

<!-- A Gated Residual GCN layer is implemented as shown in the code snippets below. -->
一个门控式残剩图形卷积网络的层的实行就如以下的代码所示。

<!-- Firstly, all the rotations of the input features $\boldsymbol{Ax}$, $\boldsymbol{Bx_{j}}$, $\boldsymbol{Ce_{j}^{x}}$, $\boldsymbol{Dx_{j}}$ and $\boldsymbol{Ex}$ are computed by defining `nn.Linear` layers inside the `__init__` function and then forward propagating the input representations `h` and `e` through the linear layers inside the `forward` function. -->
第一，所有的输入特征的旋转$\boldsymbol{Ax}$和$\boldsymbol{Bx_{j}}$﹑$\boldsymbol{Ce_{j}^{x}}$﹑$\boldsymbol{Dx_{j}}$﹑$\boldsymbol{Ex}$都是以在`__init__`定义出来的`nn.Linear`的层们计算出来的，之后就是对输入表示`h`和`e`由在`forward` 函數中的线性层(英文linear layers)中进行前向传播。

```python
class GatedGCN_layer(nn.Module):

    def __init__(self, input_dim, output_dim):
        super().__init__()
        self.A = nn.Linear(input_dim, output_dim)
        self.B = nn.Linear(input_dim, output_dim)
        self.C = nn.Linear(input_dim, output_dim)
        self.D = nn.Linear(input_dim, output_dim)
        self.E = nn.Linear(input_dim, output_dim)
        self.bn_node_h = nn.BatchNorm1d(output_dim)
        self.bn_node_e = nn.BatchNorm1d(output_dim)
```

<!-- Secondly, we compute the edge representations. This is done inside the `message_func` function, which iterates over all the edges and computes the edge representations. Specifically, the line `e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']` computes *(Eq. 7)* from above. The `message_func` function ships `Bh_j` (which is $\boldsymbol{Bx_{j}}$ from (Eq. 5)) and `e_ij` (Eq. 7) through the edge into the destination node's mailbox. -->
第二，我们计算这个边表示。这是在`message_func`函数中完成的，也就对所有边进行迭代，同时去计算边的特征。特别地，这条线`e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']`在上方那里计算出来*(方程式7)*。而这个`message_func`函数送这个`Bh_j`(也就是方程式5的$\boldsymbol{Bx_{j}}$ )和这个`e_ij`(方程式7)，通过边，送到目的地节点的邮箱。

```python
def message_func(self, edges):
    Bh_j = edges.src['Bh']
    # e_ij = Ce_ij + Dhi + Ehj
    e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']
    edges.data['e'] = e_ij
    return {'Bh_j' : Bh_j, 'e_ij' : e_ij}
```

<!-- Thirdly, the `reduce_func` function collects the shipped messages by the `message_func` function. After collecting the node data `Ah` and shipped data `Bh_j` and `e_ij` from the `mailbox`, the line `h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)` computes the hidden representation of each node as indicated in (Eq. 5). Note however, that this only represents the term $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ without the $\texttt{ReLU}(\cdot)$ and residual connection. -->
第3，这个`reduce_func`函数以`message_func`函数来收集送出去的信息。之后收集节点数据`Ah`，然后由`mailbox`送出两个数据`Bh_j`和`e_ij`。而这条线`h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)`计算每一个节点的隐藏表示，方程式5显示出来。注意一点，但是是这样的，这个项$(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$是没有$\texttt{ReLU}(\cdot)$和残剩连接来表示出来的。

```python
def reduce_func(self, nodes):
    Ah_i = nodes.data['Ah']
    Bh_j = nodes.mailbox['Bh_j']
    e = nodes.mailbox['e_ij']
    # sigma_ij = sigmoid(e_ij)
    sigma_ij = torch.sigmoid(e)
    # hi = Ahi + sum_j eta_ij * Bhj
    h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)
    return {'h' : h}
```

<!-- Inside the `forward` function, having called `g.update_all`, we obtain the results of graph convolution `h` and `e`, which represent the terms $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ from (Eq.5) and $\boldsymbol{e_{j}}$ from (Eq. 7) respectively. Then, we normalize `h` and `e` with respect to the graph node size and graph edge size respectively. Batch normalization is then applied so that we can train the network effectively. Finally, we apply $\texttt{ReLU}(\cdot)$ and add the residual connections to obtain the hidden representations for the nodes and edges, which are then returned by the `forward` function. -->
在`forward`函数中，叫`g.update_all`的话，我们就得到图形卷积的结果`h`和`e`，也就这一个项对应地表示出这个来自方程式5的项$(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$和来自方程式7的$\boldsymbol{e_{j}}$。之后我们就以「对着(respect)」图形的节点大小和图形边的大小，这个两个分别对应这两个`h`和`e`来归一化。而批量标准化就是之后用上的，所以我们就可以有效地训练网络。最后，我们用上一个$\texttt{ReLU}(\cdot)$和加一个残剩连接来为这些节点们和边们取得一些隐藏表示，也就之后以`forward`函数作为回传值回传出来。

```python
def forward(self, g, h, e, snorm_n, snorm_e):

    h_in = h # 残剩连接
    e_in = e # 残剩连接

    g.ndata['h']  = h
    g.ndata['Ah'] = self.A(h)
    g.ndata['Bh'] = self.B(h)
    g.ndata['Dh'] = self.D(h)
    g.ndata['Eh'] = self.E(h)
    g.edata['e']  = e
    g.edata['Ce'] = self.C(e)

    g.update_all(self.message_func, self.reduce_func)

    h = g.ndata['h'] # 图形卷积的结果
    e = g.edata['e'] # 图形卷积的结果

    h = h * snorm_n # 以「对着(respect)」激活来归一化
    e = e * snorm_e # 以「对着(respect)」激活来归一化

    h = self.bn_node_h(h) # 批量标准化
    e = self.bn_node_e(e) # 批量标准化

    h = torch.relu(h) # 非线性激活
    e = torch.relu(e) # 非线性激活

    h = h_in + h # 残剩连接
    e = e_in + e # 残剩连接

    return h, e
```

<!-- Next, we define the `MLP_Layer` module which contains several fully connected layers (FCN). We create a list of fully connected layers and forward through the network. -->
下一步，我们定义`MLP_Layer`模组为一个包含多个完全性连接层(英文:fully connected layers (FCN))。我们创造一列连接层们，然后向前式通过这个网络。

<!-- Finally, we define our `GatedGCN` model which comprises of the previously defined classes: `GatedGCN_layer` and `MLP_layer`. The definition of our model (`GatedGCN`) is shown below. -->
最后我们定义我们的门控式图形卷积网络模型，英文代表为`GatedGCN`，也就包含之前定义好的类: `GatedGCN_layer`和`MLP_layer`。这个我们的模型的定义(GatedGCN)就在下方所示。

 ```python
 class GatedGCN(nn.Module):
    def __init__(self, input_dim, hidden_dim, output_dim, L):
        super().__init__()
        self.embedding_h = nn.Linear(input_dim, hidden_dim)
        self.embedding_e = nn.Linear(1, hidden_dim)
        self.GatedGCN_layers = nn.ModuleList([
            GatedGCN_layer(hidden_dim, hidden_dim) for _ in range(L)
        ])
        self.MLP_layer = MLP_layer(hidden_dim, output_dim)
    def forward(self, g, h, e, snorm_n, snorm_e):
        # 输入嵌入
        h = self.embedding_h(h)
        e = self.embedding_e(e)
        # 图形卷积层们
        for GGCN_layer in self.GatedGCN_layers:
            h, e = GGCN_layer(g, h, e, snorm_n, snorm_e)
        # MLP分类器
        g.ndata['h'] = h
        y = dgl.mean_nodes(g,'h')
        y = self.MLP_layer(y)
        return y
 ```

<!-- In our constructor, we define the embeddings for `e` and `h` (`self.embedding_e ` and `self.embedding_h`), `self.GatedGCN_layers` which is list (of size $L$) of our previously defined model: `GatedGCN_layer`, and finally `self.MLP_layer` which was also defined before. Next, using these initializations, we simply foward through the model and output `y`. -->
在我們的建構子，我們定義好用在`e`和`h`的嵌入(`self.embedding_e`和`self.embedding_h`)，而`self.GatedGCN_layers`也就是一列大小為$L$的，而且是我們之前定義好的模型: `GatedGCN_layer`，而且最後就是這個`self.MLP_layer`，也都是之前定義好的。下一步，用這些初始化的話，我們就簡單地向前式通過這個模型，然後就得出輸出`y`。

<!-- To better understand the model, we initiate an object of the model and print it for better visualization: -->
為了更好的明白模型，我們初始一個模型的物体，我們也把它顯示出來更好地用視覺來看清楚這個模型:

```python
# 實例化這個網絡
model = GatedGCN(input_dim=1, hidden_dim=100, output_dim=8, L=2)
print(model)
```

<!-- The main structure of the model is shown below: -->
實例化這個網絡

```python
GatedGCN(
  (embedding_h): Linear(in_features=1, out_features=100, bias=True)
  (embedding_e): Linear(in_features=1, out_features=100, bias=True)
  (GatedGCN_layers): ModuleList(
    (0): GatedGCN_layer(...)
    (1): GatedGCN_layer(... ))
  (MLP_layer): MLP_layer(
    (FC_layers): ModuleList(...))
```

<!-- Not surprisingly, we have two layers of `GatedGCN_layer` (since `L=2`) followed by a `MLP_layer` which finally yields an output of 8 values. -->
不出所料，我們有兩層來自`GatedGCN_layer`的層，那是我們在`MLP_layer`後設定了`L=2`，也就會輸出8個數值。

<!-- Moving on, we define our `train` and `evaluate` functions. In our `train` function, we have our generic code which takes samples from `dataloader`.  Next, `batch_graphs`, `batch_x`, `batch_e`, `batch_snorm_n` and `batch_snorm_e` are fed into our model which returns `batch_scores` (of size 8). The predicted scores are compared with the ground truth in our loss function: `loss(batch_scores, batch_labels)`. Then, we zero out gradients (`optimizer.zero_grad()`), perform backprop (`J.backward()`) and updates our weights (`optimizer.step()`). Finally, loss for epoch and training accuracy is calculated. Furthermore, we use a similar code for our `evaluate` function. -->
繼續說下去吧，我們定義出我們的`train`和`evaluate`函數。在我們的`train`函數，我們有我們的通用性的代碼，也就是由`dataloader`中取得一些樣本。下一步，`batch_graphs`和 `batch_x`﹑ `batch_e`﹑ `batch_snorm_n` ﹑ `batch_snorm_e`都被輸入到我們的模型中，也就是會回傳出`batch_scores`(大小為8)。而預測分數就會在損失函數中被用來與實際正確答案(ground truth)來做比較:`loss(batch_scores, batch_labels)`。之後，我們把梯度設回零(`optimizer.zero_grad()`)，進行回傳(`J.backward()`)和更新我們的權重(`optimizer.step()`)。最後，每一個周期的損失和訓練準確度就會被計算出來了。加上，我們在`evaluate`函數中使用相似的代碼。

<!-- Finally, we are ready to train! We found that after 40 epochs of training, our model has learned to classify the graphs with a test accuracy of $87$%. -->
最後了，我們就準備好去訓練了!我們看到40個周期訓練，我們的模型就學到如何去分類出圖形，同時測試準確度為$87$%。
