---
lang-ref: ch.08-3
title: Generative Models - Variational Autoencoders
lecturer: Alfredo Canziani
authors: Richard Pang, Aja Klevs, Hsin-Rung Chou, Mrinal Jain
date: 24 March 2020
lang: zh
translation-date: 20 Jun 2020
translator: Elizabeth Zhao
---

### <!--[Recap: Auto-encoder (AE)](https://www.youtube.com/watch?v=7Rb4s9wNOmc&t=155s)-->

### [回顾: 自编码器（Auto-encoder，AE）](https://www.youtube.com/watch?v=7Rb4s9wNOmc&t=155s)

<!--To summarize at a high level, a very simple form of AE is as follows:-->

总结一下AE的最简形式：

- <!--First, the autoencoder takes in an input and maps it to a hidden state through an affine transformation $\boldsymbol{h} = f(\boldsymbol{W}_h \boldsymbol{x} + \boldsymbol{b}_h)$, where $f$ is an (element-wise) activation function. This is the **encoder** stage. Note that $\boldsymbol{h}$ is also called the **code**.-->首先，AE通过一个仿射变换（affine transformation）将输入映射到隐藏状态：$\boldsymbol{h} = f(\boldsymbol{W}_h \boldsymbol{x} + \boldsymbol{b}_h)$，其中$f$是一个一一对应的激活函数。这是**编码（encoder）**阶段。$\boldsymbol{h}$也称作**代码（code）**。
- <!--Next, $\hat{\boldsymbol{x}} = g(\boldsymbol{W}_x \boldsymbol{h} + \boldsymbol{b}_x)$, where $g$ is an activation function. This is the **decoder** stage.-->其次，$\hat{\boldsymbol{x}} = g(\boldsymbol{W}_x \boldsymbol{h} + \boldsymbol{b}_x)$，其中$g$是激活函数。这是**解码（decoder）**阶段。

<!--*For a detailed explaination, refer to the notes of [Week 7]({{site.baseurl}}/en/week07/07-3/).*-->

* 详细解释请参考笔记：[第七周]({{site.baseurl}}/zh/week07/07-3/).*

### <!--Intuition behind VAE and a comparison with classic autoencoders--> 

### VAE的直观理解与经典AE的比较

<!--Next, we introduce Variational Autoencoders (or VAE), a type of generative models. But why do we even care about generative models? To answer the question, discriminative models learn to make predictions given some observations, but generative models aim to simulate the data generation process. One effect is that generative models can better understand the underlying causal relations which leads to better generalization.-->

接下来，我们介绍了变分自编码器（Variational Autoencoders，VAE），它是一种生成模型（generative model）。为什么我们要关心生成模型呢？首先我们要了解，判别模型（discriminative model）是学习从给出的观察中做出预测，而生成模型要模拟数据生成的过程。所以生成模型可以更好地理解因果关系，从而有更好的泛化（generalization）。

<!--Note that although VAE has "Autoencoders" (AE) in its name (because of structural or architectural similarity to auto-encoders), the formulations between VAEs and AEs are very different. See *Figure 1* below.-->

注意，虽然VAE名字里包含了AE（因为架构相似），但是VAE和AE的构想有很多不同，如图1所示。

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_1.png" height="400px" /><br>
<b>图1</b>: VAE vs. 经典AE
</center>

<!--**What's the difference between variational auto-encoder (VAE) and classic auto-encoder (AE)?**-->

**VAE与经典AE之间的区别** 

<!--For VAE:--> 

VAE：

- <!--First, the encoder stage: we pass the input $\boldsymbol{x}$ to the encoder. Instead of generating a hidden representation $\boldsymbol{h}$ (the code) in AE, the code in VAE comprises two things: $\mathbb{E}(\boldsymbol{z})$ and $\mathbb{V}(\boldsymbol{z})$ where $\boldsymbol{z}$ is the latent random variable following a Gaussian distribution with mean $\mathbb{E}(\boldsymbol{z})$ and variance $\mathbb{V}(\boldsymbol{z})$. Note that people use Gaussian distributions as the encoded distribution in practice, but other distributions can be used as well.--> 第一步，编码阶段：将输入$\boldsymbol{x}$传送到编码器。VAE的代码由两种东西构成：一种是$\mathbb{E}(\boldsymbol{z})$，另一种是 $\mathbb{V}(\boldsymbol{z})$。$\boldsymbol{z}$是一个隐随机变量，它遵循高斯分布，其平均数为$\mathbb{E}(\boldsymbol{z})$、方差为$\mathbb{V}(\boldsymbol{z})$。注意，在实际应用中，人们用高斯分布作为编码分布，但是也可以用其他的分布。
    -  <!--The encoder will be a function from $\mathcal{X}$ to $\mathbb{R}^{2d}$: $\boldsymbol{x} \mapsto \boldsymbol{h}$ (here we use $\boldsymbol{h}$ to represent the concatenation of $\mathbb{E}(\boldsymbol{z})$ and $\mathbb{V}(\boldsymbol{z})$).-->编码器是一个从$\mathcal{X}$映射到$\mathbb{R}^{2d}$的函数：$\boldsymbol{x} \mapsto \boldsymbol{h}$（此处我们用$\boldsymbol{h}$表示$\mathbb{E}(\boldsymbol{z})$ 和$\mathbb{V}(\boldsymbol{z})$之间的联系）。
- <!--Next, we will sample $\boldsymbol{z}$ from the above distribution parametrized by the encoder; specifically, $\mathbb{E}(\boldsymbol{z})$ and $\mathbb{V}(\boldsymbol{z})$ are passed into a **sampler** to generate the latent variable $\boldsymbol{z}$.-->其次，我们通过上述被编码器参数化的分布对$\boldsymbol{z}$采样。具体讲就是$\mathbb{E}(\boldsymbol{z})$和$\mathbb{V}(\boldsymbol{z})$被传送进一个**采样器（sampler）**生成隐函数$\boldsymbol{z}$。
- <!--Next, $\boldsymbol{z}$ is passed into the decoder to generate $\hat{\boldsymbol{x}}$.-->然后，$\boldsymbol{z}$被传送进解码器生成$\hat{\boldsymbol{x}}$。
    - <!--The decoder will be a function from $\mathcal{Z}$ to $\mathbb{R}^{n}$: $\boldsymbol{z} \mapsto \boldsymbol{\hat{x}}$.-->解码器是一个从$\mathcal{Z}$映射到$\mathbb{R}^{n}$的函数：$\boldsymbol{z} \mapsto \boldsymbol{\hat{x}}$。

<!--In fact, for classic autoencoder, we can think of $\boldsymbol{h}$ as just the vector $\E(\boldsymbol{z})$ in the VAE formulation. In short, the main difference between VAEs and AEs is that VAEs have a good latent space that enables generative process.-->

其实，对于经典AE，我们可以将$\boldsymbol{h}$看作VAE形成里的向量$\E(\boldsymbol{z})$。简单来讲，VAE和AE之间的主要区别是VAE有一个潜在空间可以启用生成过程。

### <!--The VAE objective (loss) function-->

### VAE的目标（损失）函数

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_2.png" /><br>
<b>图2</b>: 输入空间到潜在空间到映射
</center>

<!--*See Figure 2 above. For now, ignore the top-right corner (which is the reparameterisation trick explained in the next section).*-->

*见图2。暂时忽略右上角（重参数技巧，详见下一节）*

<!--First, we encode from input space (left) to latent space (right), through encoder and noise. Next, we decode from latent space (right) to output space (left). To go from the latent to input space (the generative process) will need to either learn the distribution (of the latent code) or enforce some structure. In our case, VAE enforces some structure to the latent space.-->

首先，我们通过编码器和噪声将输入空间（图左）编译至潜在空间（图右）。然后，我们从潜在空间（图右）解码到输出空间（图左）。从潜在空间到输入空间（生成过程）需要学习数据（或潜在代码的）分布，或者强迫形成某种结构。这里，VAE迫使潜在空间形成某种结构。

<!--As usual, to train VAE, we minimize a loss function. The loss function is therefore composed of a reconstruction term as well as a regularization term.-->

和通常一样，我们最小化损失函数来训练VAE。损失函数由一个重构因子（Reconstruction Term）和一个正则化因子（Regularization Term）组成。

- <!--The reconstruction term is on the final layer (left side of the figure). This corresponds to $l(\boldsymbol{x}, \hat{\boldsymbol{x}})$ in the figure.--> 最后一层的重建因子（图左），对应的是图中的$l(\boldsymbol{x}, \hat{\boldsymbol{x}})$ 。
- <!--The regularization term is on the latent layer, to enforce some specific Gaussian structure on the latent space (right side of the figure). We do so by using a penalty term $l_{KL}(\boldsymbol{z}, \mathcal{N}(\boldsymbol{0}, \boldsymbol{I}_d))$. Without this term, VAE will act like a classic autoencoder, which may lead to overfitting, and we won't have the generative properties that we desire.-->正则化因子在隐藏层中，它是为了迫使隐藏空间（图右）形成某种高斯分布结构。我们用一个惩罚因子 $l_{KL}(\boldsymbol{z}, \mathcal{N}(\boldsymbol{0}, \boldsymbol{I}_d))$来完成。没有了这一项，VAE就像一个经典AE一样，会造成过拟合，而且会失去我们想要的生成性质。

### <!--Discussion on sampling $\boldsymbol{z}$ (**reparameterisation trick**)-->

### $\boldsymbol{z}$采样的讨论（重参数技巧（Reparameterization Trick））

<!--How do we sample from the distribution returned by the encoder in VAE? According to above, we sample from the Gaussian distribution, in order to obtain $\boldsymbol{z}$. However, this is problematic, because when we do gradient descent to train the VAE model, we don't know how to do backpropagation through the sampling module.-->

我们如何对VAE中被编码的分布进行采样？如上所述，为了取得$\boldsymbol{z}$，我们从高斯分布中采样。然而，这么做的问题是，当我们用梯度下降训练VAE模型时，我们不知道该如何通过采样部分做向后传播。

<!--Instead, we use the **reparameterization trick** to "sample" $\boldsymbol{z}$. We use $\boldsymbol{z} = \mathbb{E}(\boldsymbol{z}) + \boldsymbol{\epsilon} \odot \sqrt{\mathbb{V}(\boldsymbol{z})}$ where $\epsilon\sim \mathcal{N}(\boldsymbol{0}, \boldsymbol{I}_d)$. In this case, backpropagation in training is possible. Specifically, the gradients will go through the (element-wise) multiplication and addition in the above equation.-->我们用**重参数技巧**来“采样”$\boldsymbol{z}$。我们用$\boldsymbol{z} = \mathbb{E}(\boldsymbol{z}) + \boldsymbol{\epsilon} \odot \sqrt{\mathbb{V}(\boldsymbol{z})}$，其中$\epsilon\sim \mathcal{N}(\boldsymbol{0}, \boldsymbol{I}_d)$。这种情况下就可以在训练中使用向后传播。具体来讲，梯度会经过上述式子里的一一对应相乘和相加。

## <!--[Breaking apart the VAE Loss Function](https://www.youtube.com/watch?v=7Rb4s9wNOmc&t=997s)-->

## [详解VAE损失函数](https://www.youtube.com/watch?v=7Rb4s9wNOmc&t=997s)



### <!--Visualizing Latent Variable Estimates and Reconstruction Loss--> 

### 可视化预测隐变量（Latent Variable）与重构损失函数

 <!--As stated above, the loss function for the VAE contains two parts: a reconstruction term and a regularization term.  We can write this as-->

如上所述，VAE的损失函数有两个部分：一个重构因子和一个正则化因子。我们可以将其写作：
$$
l(\boldsymbol{x}, \hat{\boldsymbol{x}}) = l_{reconstruction} + \beta l_{\text{KL}}(\boldsymbol{z},\mathcal{N}(\textbf{0}, \boldsymbol{I}_d))
$$

<!--To visualize the purpose of each term in the loss function, we can think of each estimated $\boldsymbol{z}$ value as a circle in $2d$ space, where the centre of the circle is $\mathbb{E}(\boldsymbol{z})$ and the surrounding area are the possible values of $\boldsymbol{z}$ determined by $\mathbb{V}(\boldsymbol{z}).$--> 要想象损失函数里的每一项的用意，我们可以将每一个估计的$\boldsymbol{z}$想成一个二维空间里的圆圈，圆心是$\mathbb{E}(\boldsymbol{z})$。圆内周围的区域是$\boldsymbol{z}$的可能的值，它取决于$\mathbb{V}(\boldsymbol{z})。

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_3.png" height="350px" /><br>
<b>图3</b>: 隐藏层里的向量z可视化成的泡泡
</center>

<!--*In Figure 3 above, each bubble represents an estimated region of $\boldsymbol{z}$, and the arrows represent how the reconstruction term pushes each estimated value away from the others, which is explained more below.*-->

*如图三所示，每个泡泡代表一个$\boldsymbol{z}$的估计区域。箭头代表了重构因子将每个估计值从彼此之间推开，下面是详细解释*。 

<!--If there is overlap between any two estimates of $z$, (visually, if two bubbles overlap) this creates ambiguity for reconstruction because the points in the overlap can be mapped to both original inputs.  Therefore the reconstruction loss will push the points away from one another.-->

如果任意两个的$z$估值之间有重叠（两个泡泡相互重叠），这会对重构造成干扰，因为重叠的点回映射到同一个原始输入。所以重构损失函数会将点和点推开。

<!--However, if we use just the reconstruction loss, the estimates will continue to be pushed away from each other and the system could blow up.  This is where the penalty term comes in.-->

然而，如果我们只用重构损失函数，估测值会持续将彼此推远，系统就会爆炸。这时就需要用到惩罚因子。

<!--**Note:** for binary inputs the reconstruction loss is-->

**注：**二元输入（Binary Input）的重构损失函数为：
$$
l(\boldsymbol{x}, \hat{\boldsymbol{x}}) = - \sum\limits_{i=1}^n [x_i \log{(\hat{x_i})} + (1 - x_i)\log{(1-\hat{x_i})}]
$$

<!--and for real valued inputs the reconstruction loss is-->

实输入（Real-valued Input）的重构损失函数为：
$$
l(\boldsymbol{x}, \hat{\boldsymbol{x}}) = \frac{1}{2} \Vert\boldsymbol{x} - \hat{\boldsymbol{x}} \Vert^2
$$

### <!--The penalty term-->

### 惩罚因子（Penalty Term）



<!--The second term is the relative entropy (a measure of the distance between two distributions) between $\boldsymbol{z}$ which comes from a Gaussian with mean $\mathbb{E}(\boldsymbol{z})$, variance $\mathbb{V}(\boldsymbol{z})$ and the standard normal distribution. If we expand this second term in the VAE loss function we get:-->

第二项是$\boldsymbol{z}$之间的相对熵（Relative Entropy，是两个分布之间的距离的测量），其来源于高斯分布，平均值是$\mathbb{E}(\boldsymbol{z})$，方差是 $\mathbb{V}(\boldsymbol{z})$。我们将VAE损失函数的第二项展开可得：
$$
\beta l_{\text{KL}}(\boldsymbol{z},\mathcal{N}(\textbf{0}, \boldsymbol{I}_d)) = \frac{\beta}{2} \sum\limits_{i=1}^d(\mathbb{V}(z_i) - \log{[\mathbb{V}(z_i)]} - 1 + \mathbb{E}(z_i)^2)
$$

<!--Where each expression in the summation has four terms. Below we write out and graph the first three terms in *图4*.-->

其中，和的每一步有四项。我们在下方写出第一项，并在图4画出后面三项。
$$
v_i = \mathbb{V}(z_i) - \log{[\mathbb{V}(z_i)]} - 1
$$

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_4.png" /><br>
<b>图4</b>: 图示相对熵如何迫使“泡泡”的方差为1
</center>

<!--So we can see that this expression is minimized when $z_i$ has variance 1.  Therefore our penalty loss will keep the variance of our estimated latent variables at around 1.  Visually, this means our "bubbles" from above will have a radius of around 1.-->

可以看到，当方差为1时，表达式到达最小值。所以我们的惩罚损失函数会将估测的隐藏变量的方差保留到约1。想象一下的话，这就意味着上图的“泡泡”的半径约等于1.

<!--The last term, $\mathbb{E}(z_i)^2$, minimizes the distance between the $z_i$ and therefore prevents the "exploding" encouraged by the reconstruction term.-->

最后一项，$\mathbb{E}(z_i)^2$会最小化$z_i$之间的距离，因此预防了重构因子引发的“爆炸”。

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_5.png" height="400px"/><br>
<b>图5</b>: 解释VAE：“泡泡”中的“泡泡”
</center>

<!--*Figure 5 above shows how VAE loss pushed the estimated latent variables as close together as possible without any overlap while keeping the estimated variance of each point around one.*-->

*图5显示了VAE损失函数把估测的隐藏变量之间挤压而不造成重叠，且同时使每个点的估测方差维持在1。*

<!--**Note:** The $\beta$ in the VAE loss function is a hyperparameter that dictates how to weight the reconstruction and penalty terms.-->

**注：**VAE损失函数中的$\beta$是一个超参数（Hyperparameter），它用来决定重构因子和惩罚因子之间的权重。

## <!--[Implementation of Variational Autoencoder (VAE)](https://www.youtube.com/watch?v=7Rb4s9wNOmc&t=1893s)-->

## [VAE的实现](https://www.youtube.com/watch?v=7Rb4s9wNOmc&t=1893s)

<!--The Jupyter notebook can be found [here](https://github.com/Atcold/pytorch-Deep-Learning/blob/master/11-VAE.ipynb).-->

Jupyter Notebook在[这里](https://github.com/Atcold/pytorch-Deep-Learning/blob/master/11-VAE.ipynb)。

<!--In this notebook, we implement a VAE and train it on the MNIST dataset. Then we sample $\boldsymbol{z}$ from a normal distribution and feed to the decoder and compare the result. Finally, we look at how $\boldsymbol{z}$ changes in 2D projection.-->

在这个笔记里，我们执行了VAE并在MNIST数据集对它进行训练。然后我们从一个正态分布里对$\boldsymbol{z}$进行采样，并且将它输送到解码器里比较结果。最后，我们观察$\boldsymbol{z}$在二维投影里如何变化。

 <!--**Note:** In the MNIST dataset used, the pixel values have been normalized to be in range $[0, 1]$.-->

**注：** 在我们用到的MNIST数据集中，像素值已经被归一化到$[0, 1]$的区间。

### <!--The Encoder and the Decoder--> 

### 编码与解码

1. <!--We define the encoder and decoder in our `VAE` module.-->我们在VAE模块中定义编码器和解码器。
2. <!--For the last linear layer of encoder, we define the output to be of size $2d$, of which the first $d$ values are the means and the remaining $d$ values are the variances. We sample $\boldsymbol{z} \in R^d$ using these means and variances as explained in the reparameterisation trick before.-->在编码器的最后一个线性层，我们定义输的大小为$2d$，前一半的值为平均值，剩下一半的值为方差。我们用这些平均值和方差对$\boldsymbol{z} \in R^d$采样，用以解释之前的重参数技巧。
3. <!--For the last linear layer in the decoder, we use the sigmoid activation so that we can have output in range $[0, 1]$, similar to the input data.-->对于编码器的最后一个线性层，我们用Sigmoid激活函数，便于使输出集中在$[0, 1]$之间，和输入数据类似。

```python
class VAE(nn.Module):
    def __init__(self):
        super().__init__()

        self.encoder = nn.Sequential(
            nn.Linear(784, d ** 2),
            nn.ReLU(),
            nn.Linear(d ** 2, d * 2)
        )

        self.decoder = nn.Sequential(
            nn.Linear(d, d ** 2),
            nn.ReLU(),
            nn.Linear(d ** 2, 784),
            nn.Sigmoid(),
        )
```

### <!--Reparameterisation and the `forward` function-->

### 重参数与`forward`函数

<!--For the `reparameterise` function, if the model is in training mode, we compute the standard deviation (`std`) from log variance (`logvar`). We use log variance instead of variance because we want to make sure the variance is non-negative, and taking the log of it ensures that we have the full range of the variance, which makes the training more stable.-->

对于`reparameterise`函数，如果模型在训练模式下，我们从log方差（`logvar`）中计算标准偏差（`std`）。我们用log方差而不是普通的方差，因为我们想要使方差为非负的，对其求log可以保证我们有方差的全部值域，这样可以试训练更加稳定。

 <!--During training, the `reparameterise` function will do the reparameterisation trick so that we can do backpropagation in training. To connect to the concept of a yellow bubble, as explained in the lecture, every time this function is called, we draw a point `eps = std.data.new(std.size()).normal_()`, so if we do 100 times, we will get 100 points which roughly forms a sphere because it's normal distribution, and the line `eps.mul(std).add_(mu)` will make this sphere centred in `mu` with radius equal to `std`.--> 在训练过程中，`reparameterise`方程做出重参数技巧以便于我们可以在训练中做向后传播。如讲座中所解释，为了与黄色泡泡的概念连接起来，每次这个方程式被召唤后，我们会画一个点`eps = std.data.new(std.size()).normal_()`。如果我们重复100次，我们可以得到100个点，这样会大致形成一个球，因为这是一个正态分布。而且`eps.mul(std).add_(mu)`使这个球以`mu`为圆心、以`std`为半径。

<!--For the `forward` function, we first compute the `mu` (first half) and `logvar` (second half) from the encoder, then we compute the $\boldsymbol{z}$ via the `reparamterise` function. Finally, we return the output of the decoder.-->

对`forward`方程，我们先从编码器中计算`mu`（前一半）和`logvar`（后一半），然后我们通过`reparamterise`方程计算$\boldsymbol{z}$。最后，我们return解码器的输出。

```python
def reparameterise(self, mu, logvar):
    if self.training:
        std = logvar.mul(0.5).exp_()
        eps = std.data.new(std.size()).normal_()
        return eps.mul(std).add_(mu)
    else:
        return mu

def forward(self, x):
    mu_logvar = self.encoder(x.view(-1, 784)).view(-1, 2, d)
    mu = mu_logvar[:, 0, :]
    logvar = mu_logvar[:, 1, :]
    z = self.reparameterise(mu, logvar)
    return self.decoder(z), mu, logvar
```

### <!--Loss function for the VAE--> 

### VAE的损失函数

<!--Here we define the reconstruction loss (binary cross entropy) and the relative entropy (KL divergence penalty).-->

这里我们定义重构损失函数（二元交叉熵，binary cross entropy）和相对熵（KL散度熵，KL Divergence Penalty）。

```python
def loss_function(x_hat, x, mu, logvar):
    BCE = nn.functional.binary_cross_entropy(
        x_hat, x.view(-1, 784), reduction='sum'
    )
    KLD = 0.5 * torch.sum(logvar.exp() - logvar - 1 + mu.pow(2))

    return BCE + KLD
```

### <!--Generating new samples-->

### 生成新的样本



<!--After we train our model, we can sample a random $z$ from the normal distribution and feed it to our decoder. We can observe in *Figure 6* that some of the results are not good because our decoder has not "covered" the whole latent space. This can be improved if we train the model for more epochs.-->

在我们训练完模型之后，我们可以从正态分布里抽取一个随机的$z$并把它输送到解码器。我们可以观察图6，其中一些结果不太理想，因为我们的解码器没有“覆盖”整个隐藏空间。我们可以通过增加训练周期来提升结果。

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_6.png" /><br>
<b>图6</b>: 在隐藏层里随机移动
</center>

<!--We can look at how one digit morphs into another, which would not have been possible if we used an autoencoder. We can see that when we walk in the latent space, the output of the decoder still looks legit. *Figure 7* below shows how we morph the digit $3$ to $8$.-->

我们可以观察一个数字如何变形成另一个，而这用AE是不能可完成的。可以看到当我们在隐藏空间里漫步时，解码器的输出看起来任然完好。图7展示了我们如何是数字$3$变形至数字$8$。

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_7.png" /><br>
<b>图7</b>: 数字3到数字8的变形
</center>



### <!--Projection of means-->

### 平均数的投影

<!--Finally, let's take a look at how the latent space changes during/after training. The following charts in *Figure 8* are the means from the output of the encoder, projected on 2D space, where each colour represents a digit. We can see that from epoch 0, the classes are spreading everywhere, with only little concentration. As the model is trained, the latent space becomes more well-defined and the classes (digits) starts to form clusters.-->

最后，我们来观察隐藏层在训练中/训练后是如何变化的。图8中的图表是从编码器输出的平均值，投射在一个二维空间上，每个颜色代表一个数字。可以看到从周期0开始，数字的类别分布得很散，几乎没有各自的聚集。当模型开始被训练后，隐藏空间逐渐变得定义明确（well-defined），并且数字开始形成聚类。

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_8.png" /><br>
<b>Fig. 8</b>: 在隐藏空间中的平均值$\E(\vect{z})$的二维投影
</center>

