---
lang-ref: ch.13-3
lang: ko
title: 그래프 합성곱 신경망 III
lecturer: Alfredo Canziani
authors: Go Inoue, Muhammad Osama Khan, Muhammad Shujaat Mirza, Muhammad Muneeb Afzal
date: 28 Apr 2020
translation-date: Sep 1 2020
translator: Seok Hoan (Kevin) Choi
---

<!--
## [Introduction to Graph Convolutional Network (GCN)](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=47s)
Graph Convolutional Network (GCN) is one type of architecture that utilizes the structure of data.
Before going into details, let's have a quick recap on self-attention, as GCN and self-attention are conceptually relevant.
-->

## [그래프 합성곱 신경망 개요 <sup>Introduction to Graph Convolutional Network (GCN)</sup>](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=47s)

그래프  합성곱  네트워크는  데이터  구조를  활용하는  아키텍쳐  유형  중  하나이다. 세부  사항으로  들어가기  전에, GCN과  셀프-어텐션<sup>self-attention</sup>은  개념적으로  관련이  있으므로  셀프-어텐션에  대해  간단히  복습해보자.

<!--

### Recap: Self-attention

- In self-attention, we have a set of input $\lbrace\boldsymbol{x}\_{i}\rbrace^{t}_{i=1}$.
Unlike a sequence, it does not have an order.
- Hidden vector $\boldsymbol{h}$ is given by linear combination of the vectors in the set.
- We can express this as $\boldsymbol{X}\boldsymbol{a}$ using matrix vector multiplication, where $\boldsymbol{a}$ contains coefficients that scale the input vector $\boldsymbol{x}_{i}$.
*For a detailed explanation, refer to the notes of [Week 12]({{site.baseurl}}/ko/week12/12-3/).*

-->

### 복습: 셀프-어텐션

- 셀프-어텐션에서는  입력값  집합 $\lbrace\boldsymbol{x}\_{i}\rbrace^{t}_{i=1}$ 을  갖는다. 시퀀스<sup></sup>와  다르게  셀프-어텐션은  순서가  없다.
- 집합  안에  벡터들의  선형결합<sup></sup>에  의해  은닉  벡터<sup></sup> $\boldsymbol{h}$가  주어진다.
- 이를  행렬  벡터  곱셈을  사용해 $\boldsymbol{X}\boldsymbol{a}$로  표현이  가능하며, $\boldsymbol{a}$는  입력벡터 $\boldsymbol{x}_{i}$에  상응하는  차수<sup>coefficient</sup>를  갖는다.

*더  자세한  설명은, [Week 12]({{site.baseurl}}/ko/week12/12-3/)에서  참조  가능하다.*

<!--

### Notation

<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure1.png" height="400px" /><br>
<b>Fig. 1</b>: Graph Convolutional Network
</center>

In Figure 1, vertex $v$ is comprised of two vectors: input $\boldsymbol{x}$ and its hidden representation $\boldsymbol{h}$.
We also have multiple vertices $v_{j}$, which is comprised of $\boldsymbol{x}\_j$ and $\boldsymbol{h}\_j$.
In this graph, vertices are connected with directed edges.

-->

### 표기법

<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure1.png" height="400px" /><br>
<b>Fig. 1</b>: 그래프 합성곱 그래프
</center>

Figure 1에  보여지는  것과  같이, 정점 $v$는  두  개의  벡터 (입력값 $\boldsymbol{x}$ 와  은닉  표현 $\boldsymbol{h}$) 로  이루어져있다. 다수의  정점 $v_{j}$ 또한 $\boldsymbol{x}_j$ 와 $\boldsymbol{h}_j$로  이루어져있고, 그래프에는  정점들이  유향  엣지<sup>directed edge: 간선</sup>들로  연결되어있다.

<!--
We represent these directed edges with adjacency vector $\boldsymbol{a}$, where each element $\alpha_{j}$ is set to $1$ if there is a directed edge from $v_{j}$ to $v$.

$$
\alpha_{j} \stackrel{\tiny \downarrow}{=} 1 \Leftrightarrow v_{j} \rightarrow v
\tag{Eq. 1}
$$
-->

이러한  유향 엣지는  인접  벡터<sup>adjacency vector</sup> $\boldsymbol{a}$로  표현이  가능하며, 유향 엣지가 $v_{j}$에서 $v$로  향한다면  각  요소 $\alpha_{j}$는 $1$이  된다.

$$
\alpha_{j} \stackrel{\tiny \downarrow}{=} 1 \Leftrightarrow v_{j} \rightarrow v
\tag{Eq. 1}
$$

<!--
The degree (number of incoming edges) $d$ is defined as the norm of this adjacency vector, *i.e.* $\Vert\boldsymbol{a}\Vert_{1} $, which is the number of ones in the vector $\boldsymbol{a}$.

$$
d = \Vert\boldsymbol{a}\Vert_{1}
\tag{Eq. 2}
$$

-->

차수 (들어오는  엣지들의  수) $d$는  인접  벡터의  노름 <sup>norm</sup>*i.e.*  $\Vert\boldsymbol{a}\Vert_{1}$ 으로  정의되며, 이는  벡터 $\boldsymbol{a}$의 1의  수와  같다.

<!--
The hidden vector $\boldsymbol{h}$ is given by the following expression:

$$
\boldsymbol{h}=f(\boldsymbol{U}\boldsymbol{x} + \boldsymbol{V}\boldsymbol{X}\boldsymbol{a}d^{-1})
\tag{Eq. 3}
$$

where $f(\cdot)$ is a non-linear function such as ReLU $(\cdot)^{+}$, Sigmoid $\sigma(\cdot)$, and hyperbolic tangent $\tanh(\cdot)$.
-->

은닉  벡터 $\boldsymbol{h}$ 는  다음과  같이  주어진다:

$$
\boldsymbol{h}=f(\boldsymbol{U}\boldsymbol{x} + \boldsymbol{V}\boldsymbol{X}\boldsymbol{a}d^{-1})
\tag{Eq. 3}
$$

$f(\cdot)$는 ReLU $(\cdot)^{+}$, 시그모이드 Sigmoid $\sigma(\cdot)$, 쌍곡탄젠트 $\tanh(\cdot)$와  같은  비선형  함수<sup>non-linear function</sup>이다.

<!--
The $\boldsymbol{U}\boldsymbol{x}$ term takes into account the vertex $v$ itself, by applying rotation $\boldsymbol{U}$ to the input $v$.
Remember that in self-attention, the hidden vector $\boldsymbol{h}$ is computed by $\boldsymbol{X}\boldsymbol{a}$, which means that the columns in $\boldsymbol{X}$ is scaled by the factors in $\boldsymbol{a}$.
In the context of GCN, this means that if we have multiple incoming edges,*i.e.* multiple ones in adjacency vector $\boldsymbol{a}$, $\boldsymbol{X}\boldsymbol{a}$ gets larger.
-->

$\boldsymbol{U}\boldsymbol{x}$는  회전 $\boldsymbol{U}$를  입력값 $v$에  적용함으로  정점<sup>vertex </sup> $v$를  다룬다.
셀프-어텐션에서  기억해야  할  것은  은닉  벡터 $\boldsymbol{h}$는 $\boldsymbol{X}\boldsymbol{a}$에서  계산되고, 이는 $\boldsymbol{X}$에서의  열은 $\boldsymbol{a}$의  각  원소들에  의해  스케일링<sup>scaled</sup>된다.
GCN 맥락에서  이는  여러개의  수신  엣지들이  있을  때( 인접  벡터 $\boldsymbol{a}$ 안에  다수의 1들)  $\boldsymbol{X}\boldsymbol{a}$값이  더욱  커지게  된다는  것을  뜻한다.

<!--
On the other hand, if we have only one incoming edge, this value gets smaller.
To remedy this issue of the value being proportionate to the number of incoming edges, we divide it by the number of incoming edges $d$.
We then apply rotation $\boldsymbol{V}$ to $\boldsymbol{X}\boldsymbol{a}d^{-1}$.
-->

반면에, 수신  엣지가  하나라면, 위  값은  작아진다.
수신  엣지의  수에  비례하는  값의  문제를  수신  엣지의  수 $d$로  나누는  것으로  해결  가능하다.
그런 다음, $\boldsymbol{X}\boldsymbol{a}d^{-1}$에 회전 $\boldsymbol{V}$를 적용한다.

<!--

We can represent this hidden representation $\boldsymbol{h}$ for the entire set of inputs $\boldsymbol{x}$ using the following matrix notation:

$$
\{\boldsymbol{x}_{i}\}^{t}_{i=1}\rightsquigarrow \boldsymbol{H}=f(\boldsymbol{UX}+ \boldsymbol{VXAD}^{-1})
\tag{Eq. 4}
$$

where $\vect{D} = \text{diag}(d_{i})$.
-->

전체  입력  집합$\boldsymbol{x}$에  대해여  은닉표현 $\boldsymbol{h}$을  다음과  같은  행렬  표기법을  사용하여  표현  가능하다:

$$
\{\boldsymbol{x}_{i}\}^{t}_{i=1}\rightsquigarrow \boldsymbol{H}=f(\boldsymbol{UX}+ \boldsymbol{VXAD}^{-1})
\tag{Eq. 4}
$$

여기서 $\vect{D} = \text{diag}(d_{i})$.

<!--
## [Residual Gated GCN Theory and Code](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=992s)

Residual Gated Graph Convolutional Network is a type of GCN that can be represented as shown in Figure 2:

<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure2.png" height="300px" /><br>
<b>Fig. 2</b>: Residual Gated Graph Convolutional Network
</center>
-->

## [Residual Gated GCN 이론과 코드](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=992s)

Residual Gated 그래프 합성곱 신경망은 Figure 2처럼 표현되는 GCN의 한 종류이다.

<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure2.png" height="300px" /><br>
<b>Fig. 2</b>: Residual Gated Graph Convolutional Network
</center>

<!--
As with the standard GCN, the vertex $v$ consists of two vectors: input $\boldsymbol{x}$ and its hidden representation $\boldsymbol{h}$. However, in this case, the edges also have a feature representation, where $\boldsymbol{e_{j}^{x}}$ represents the input edge representation and $\boldsymbol{e_{j}^{h}}$ represents the hidden edge representation.
-->

기본 GCN처럼, 정첨 $v$는 2개의  벡터  입력값 $\boldsymbol{x}$와  은닉  표현 $\boldsymbol{h}$ 갖는다. 하지만  이런  경우에는  엣지들  또한  특징  표현을  갖는다. 여기서 $\boldsymbol{e_{j}^{x}}$는  입력  엣지  표현을  나타내고, $\boldsymbol{e_{j}^{h}}$는  은닉  엣지  표현을  나타낸다.

<!--
The hidden representation $\boldsymbol{h}$ of the vertex $v$ is computed by the following equation:

$$
\boldsymbol{h}=\boldsymbol{x} + \bigg(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}\bigg)^{+}
\tag{Eq. 5}
$$
-->

정점 $v$의  은닉  표현 $\boldsymbol{h}$ 는  다음  방정식으로  계산된다:

$$
\boldsymbol{h}=\boldsymbol{x} + \bigg(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}\bigg)^{+}
\tag{Eq. 5}
$$

<!--
where $\boldsymbol{x}$ is the input representation, $\boldsymbol{Ax}$ represents a rotation applied to the input $\boldsymbol{x}$ and $\sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}$ denotes the summation of elementwise multiplication of rotated incoming features $\boldsymbol{Bx_{j}}$ and a gate $\eta(\boldsymbol{e_{j}})$. In contrast to the standard GCN above where we average the incoming representations, the gate term is critical to the implementation of Residual Gated GCN since it allows us to modulate the incoming representations based on the edge representations.
-->

여기서 $\boldsymbol{x}$는  입력  표현이고, $\boldsymbol{Ax}$ 는  입력값 $\boldsymbol{x}$에  적용된  회전<sup>rotation</sup>을  나타내고, $\sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}$는  회전된  수신  특징<sup>incoming features</sup> $\boldsymbol{Bx_{j}}$와  게이트 $\eta(\boldsymbol{e_{j}})$의  원소별  곱셈<sup>elementwise multiplication</sup>의  합이다. 위에서  다룬  기본 GCN에서의  수신  표현들을  평균낸  것과  다르게, 게이트  항<sup>term</sup>은  수신  표현들을  엣지  표현  기반으로  조절할  수  있게  하기에  게이트  항은 Residual Gated GCN을  구현하는데  매우  중요한  역활을  한다.

<!--
Note that the summation is only over vertices ${v_j}$ that have incoming edges to vertex ${v}$. The term residual (in Residual Gated GCN) comes from the fact that in order to calculate the hidden representation $\boldsymbol{h}$, we add the input representation $\boldsymbol{x}$. The gate term $\eta(\boldsymbol{e_{j}})$ is calculated as shown below:

$$
\eta(\boldsymbol{e_{j}})=\sigma(\boldsymbol{e_{j}})\bigg(\sum_{v_k→v}\sigma(\boldsymbol{e_{k}})\bigg)^{-1}
\tag{Eq. 6}
$$
-->

주의할  점은  합이  정점 $v$로  향하는  수신  엣지를  갖는  정점들 ${v_j}$만으로  이루어져  있다는  것이다. Residual Gated GCN에서의  단어 Residual은  은닉  표현 $\boldsymbol{h}$ 을  계산하기  위해  입력  표현 $\boldsymbol{x}$ 을  더했다는  것에서  비롯된다. 게이트  항 $\eta(\boldsymbol{e_{j}})$ 은  다음과  같이  계산된다:

$$
\eta(\boldsymbol{e_{j}})=\sigma(\boldsymbol{e_{j}})\bigg(\sum_{v_k→v}\sigma(\boldsymbol{e_{k}})\bigg)^{-1}
\tag{Eq. 6}
$$

<!--
The gate value $\eta(\boldsymbol{e_{j}})$ is a normalized sigmoid obtained by dividing the sigmoid of the incoming edge representation by the sum of sigmoids of all incoming edge representations. Note that in order to calculate the gate term, we need the representation of the edge $\boldsymbol{e_{j}}$, which can be computed using the equations below:

$$
\boldsymbol{e_{j}} = \boldsymbol{Ce_{j}^{x}} + \boldsymbol{Dx_{j}} + \boldsymbol{Ex}
\tag{Eq. 7}
$$

$$
\boldsymbol{e_{j}^{h}}=\boldsymbol{e_{j}^{x}}+(\boldsymbol{e_{j}})^{+}
\tag{Eq. 8}
$$
-->

게이트  값 $\eta(\boldsymbol{e_{j}})$는  수신  엣지표현들의  시그모이드를  모든  수신  엣지들의  시그모이드의  합을  나눈  것으로  얻어지는  정규화된  시그모이드이다. 주의해야  할  건, 게이트  항을  계산하기  위해서는  다음과  같이  계산되는  엣지 $\boldsymbol{e_{j}}$의  표현이  필요하다:

$$
\boldsymbol{e_{j}} = \boldsymbol{Ce_{j}^{x}} + \boldsymbol{Dx_{j}} + \boldsymbol{Ex}
\tag{Eq. 7}
$$

$$
\boldsymbol{e_{j}^{h}}=\boldsymbol{e_{j}^{x}}+(\boldsymbol{e_{j}})^{+}
\tag{Eq. 8}
$$

<!--
The hidden edge representation $\boldsymbol{e_{j}^{h}}$ is obtained by the summation of the initial edge representation $\boldsymbol{e_{j}^{x}}$ and $\texttt{ReLU}(\cdot)$ applied to $\boldsymbol{e_{j}}$ where $\boldsymbol{e_{j}}$ is in turn given by the summation of a rotation applied to $\boldsymbol{e_{j}^{x}}$, a rotation applied to the input representation $\boldsymbol{x_{j}}$ of the vertex $v_{j}$ and a rotation applied to the input representation $\boldsymbol{x}$ of the vertex $v$.
-->

은닉  엣지  표현 $\boldsymbol{e_{j}^{h}}$는  초기  엣지  표현<sup></sup> $\boldsymbol{e_{j}^{x}}$와 $\texttt{ReLU}(\cdot)$ 가  적용된 $\boldsymbol{e_{j}}$의  합이다. 여기서 $\boldsymbol{e_{j}}$는 $\boldsymbol{e_{j}^{x}}$, 정점 $v_{j}$의  입력  표현 $\boldsymbol{e_{j}^{x}}$, 그리고  정점 $v$의  입력  표현 $\boldsymbol{x}$에  각  회전이  적용되진  값들의  합에서  주어진다.

<!--
*Note: In order to calculate hidden representations downstream (*e.g.* $2^\text{nd}$ layer hidden representations), we can simply replace the input feature representations by the $1^\text{st}$ layer feature representations in the equations above.*
-->

*참고: 밑에  은닉  표현 (*e.g.* $2^\text{nd}$ 레이어의  은닉  표현)을  계산하기  위해, 위  방정식에서의  입력  특징  표현을 $1^\text{st}$ 레이어  특징  표현으로  바꾸는  것이  가능하다.*

<!--
### Graph Classification and Residual Gated GCN Layer

In this section, we introduce the problem of graph classification and code up a Residual Gated GCN layer. In addition to the usual import statements, we add the following:
-->

### 그래프  분류와 Residual Gated GCN 레이어

이번  섹션에서는, 그래프  분류  문제와 Residual Gated GCN 레이어의  코드에  대해  알아볼  것이다. import 문과  더불어  다음  코드들을  더한다:

<!--
```python
os.environ['DGLBACKEND'] = 'pytorch'
import dgl
from dgl import DGLGraph
from dgl.data import MiniGCDataset
import networkx as nx
```

The first line tells DGL to use PyTorch as the backend. Deep Graph Library ([DGL](https://www.dgl.ai/)) provides various functionalities on graphs whereas networkx allows us to visualise the graphs.
-->

```python
os.environ['DGLBACKEND'] = 'pytorch'
import dgl
from dgl import DGLGraph
from dgl.data import MiniGCDataset
import networkx as nx
```

첫줄은 DGL에  백엔드에 PyTorch를  사용하라  명령한다.

Deep Graph Library ([DGL](https://www.dgl.ai/))는  그래프에  관해  다양한  기능들을  제공하고,  networkx는  그래프의  시각화를  도와준다.

<!--
In this notebook, the task is to classify a given graph structure into one of 8 graph types. The dataset obtained from `dgl.data.MiniGCDataset` yields some number of graphs (`num_graphs`) with nodes between `min_num_v` and `max_num_v`. Hence, all the graphs obtained do not have the same number of nodes/vertices.

*Note: In order to familiarize yourself with the basics of `DGLGraphs`, it is recommended to go through the short tutorial [here](https://docs.dgl.ai/api/python/graph.html).*
-->

이  노트북에서의  목표는  주어진  그래프의  구조를 8가지의  그래프  타입중  하나로  분류하는  것이다. 데이터셋은 `dgl.data.MiniGCDataset`에서  주어지며  그래프의  숫자(`num_graphs`) 를 `min_num_v`와 `max_num_v`사이의  노드들을  보여준다. 따라서  얻은  모든  그래프들은  서로  같은  수의  노드과  엣지를  갖지  않는다.

*참고: `DGLGraphs`에  익숙해지기  위해서  짧은  튜토리얼을  참조하는  것을  추천한다 [참조](https://docs.dgl.ai/api/python/graph.html).*

<!--
Having created the graphs, the next task is to add some signal to the domain. Features can be applied to nodes and edges of a `DGLGraph`. The features are represented as a dictionary of names (strings) and tensors (**fields**). `ndata` and `edata` are syntax sugar to access the feature data of all nodes and edges.
-->

그래프를  만든  후  다음  작업은  도메인에  신호를  추가하는  것이다. 특징들은`DGLGraph`의  노드와  엣지에  적용될  수  있다. 특징들은  이름 (문자열)과  텐서 (**필드**)의  딕셔너리로  표현된다. `ndata`과 `edata`는  모든  노드과  엣지들의  특징  데이터를  접근하기  위한  문법적  설탕<sup>syntax sugar</sup>이다.

<!--
The following code snippet shows how the features are generated. Each node is assigned a value equal to the number of incident edges, whereas each edge is assigned the value 1.

```python
def create_artificial_features(dataset):
for (graph, _) in dataset:
graph.ndata['feat'] = graph.in_degrees().view(-1, 1).float()
graph.edata['feat'] = torch.ones(graph.number_of_edges(), 1)
return dataset
```

-->

아래의  코드는  특징들이  어떻게  생성되는지  보여준다. 각  노드는  부속  엣지  수와  동일한  값이  할당되고  각  엣지는 1이  할당됩니다.

```python
def create_artificial_features(dataset):
    for (graph, _) in dataset:
        graph.ndata['feat'] = graph.in_degrees().view(-1, 1).float()
        graph.edata['feat'] = torch.ones(graph.number_of_edges(), 1)
    return dataset
```

<!--

Training and testing datasets are created and features are assigned as shown below:

```python
trainset = MiniGCDataset(350, 10, 20)
testset = MiniGCDataset(100, 10, 20)

trainset = create_artificial_features(trainset)
testset = create_artificial_features(testset)
```

-->

아래와  같이  학습  데이터와  테스트  데이터는  생성되고  특징은  할당된다:

```python
trainset = MiniGCDataset(350, 10, 20)
testset = MiniGCDataset(100, 10, 20)

trainset = create_artificial_features(trainset)
testset = create_artificial_features(testset)
```

<!--

A sample graph from the trainset has the following representation. Here, we observe that the graph has 15 nodes and 45 edges and both the nodes and edges have a feature representation of shape `(1,)` as expected. Furthermore, the `0` signifies that this graph belongs to class 0.

```python
(DGLGraph(num_nodes=15, num_edges=45,
ndata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}
edata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}), 0)
```

-->

테스트  데이터에서의  샘플  그래프는  다음과  같은  표현을  갖는다. 여기서  그래프가 15개의  노드와 45개의  엣지를  갖고  노드과  엣지 모두  예상대로 `(1,)`모양의  특징  표현을  갖는  것을  확인할  수  있다. 또한,  `0` 은  이  그래프가  클래스 0에  속함을  의미한다.

```python
(DGLGraph(num_nodes=15, num_edges=45,
         ndata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}
         edata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}), 0)
```

<!--

### Note about DGL Message and Reduce Functions

>In DGL, the *message functions* are expressed as **Edge UDF**s (User Defined Functions). Edge UDFs take in a single argument `edges`. It has three members `src`, `dst`, and `data` for accessing source node features, destination node features, and edge features respectively.

The *reduce functions* are **Node UDF**s. Node UDFs have a single argument `nodes`, which has two members `data` and `mailbox`. `data` contains the node features and `mailbox` contains all incoming message features, stacked along the second dimension (hence the `dim=1` argument).

`update_all(message_func, reduce_func)` sends messages through all edges and updates all nodes.
-->

### DGL 메시지와 Reduce 함수

>DGL에서는 *message functions* 는 **엣지 UDF** (사용자  정의  함수)로  표현된다. 엣지 UDF는  단일  인수 `edges`를  받고  소스  노드  특징, 목적지  노드 특징, 그리고  엣지  특징에  각각  액세스  하기  위한 3개의  멤버 `src`, `dst`, `data` 가  있다.

*reduce function* 은 **노드 UDF** 이고  노드 UDF에는 `data` 와 `mailbox` 두  개의  멤버가  있는  단일  인수<sup>single argument </sup> `nodes`가  있다. `data`에는  노드  특징을  갖고 `mailbox`는  두  번째  차원을  따라  쌓인  모든  수신  메시지  특징을 갖는다 (그래서 `dim=1` 인수를 갖는다).

`update_all(message_func, reduce_func)`는 모든 엣지들에 메세지를 보내고 모든 노드들을 업데이트한다. 

<!--
### [Gated Residual GCN Layer Implementation](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=2098s)

A Gated Residual GCN layer is implemented as shown in the code snippets below.

Firstly, all the rotations of the input features $\boldsymbol{Ax}$, $\boldsymbol{Bx_{j}}$, $\boldsymbol{Ce_{j}^{x}}$, $\boldsymbol{Dx_{j}}$ and $\boldsymbol{Ex}$ are computed by defining `nn.Linear` layers inside the `__init__` function and then forward propagating the input representations `h` and `e` through the linear layers inside the `forward` function.
-->

### [Gated Residual GCN 레이어 구현](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=2098s)

Gated Residual GCN 레이어는 아래 코드 처럼 구현된다.

먼저, 입력 특징 $\boldsymbol{Ax}$, $\boldsymbol{Bx_{j}}$, $\boldsymbol{Ce_{j}^{x}}$, $\boldsymbol{Dx_{j}}$ and $\boldsymbol{Ex}$의 모든 회전은 `__init__`함수 안의 `nn.Linear`레이어를 정의함과 `forward`함수 안의 선형 레이어를 통한 입력 표현 `h`와 `e`의 순전파<sup>forward propagating</sup>로 계산이 가능하다.

```python
class GatedGCN_layer(nn.Module):

    def __init__(self, input_dim, output_dim):
        super().__init__()
        self.A = nn.Linear(input_dim, output_dim)
        self.B = nn.Linear(input_dim, output_dim)
        self.C = nn.Linear(input_dim, output_dim)
        self.D = nn.Linear(input_dim, output_dim)
        self.E = nn.Linear(input_dim, output_dim)
        self.bn_node_h = nn.BatchNorm1d(output_dim)
        self.bn_node_e = nn.BatchNorm1d(output_dim)
```

<!--
Secondly, we compute the edge representations. This is done inside the `message_func` function, which iterates over all the edges and computes the edge representations. Specifically, the line `e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']` computes *(Eq. 7)* from above. The `message_func` function ships `Bh_j` (which is $\boldsymbol{Bx_{j}}$ from (Eq. 5)) and `e_ij` (Eq. 7) through the edge into the destination node's mailbox.
-->

그 다음으로는, 엣지 표현을 계산한다. 이는 `message_func`함수 안에서 이루어지며 모든 엣지에 반복하여 엣지 표현을 계산한다. 특히, `e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']` 라인은 위의 *(Eq. 7)*를 계산하며, `message_func` 함수는 `Bh_j` ((Eq. 5)에서의 $\boldsymbol{Bx_{j}}$) 와 `e_ij` (Eq. 7) 를  목적 노드의 `mailbox`에 보낸다.

```python
def message_func(self, edges):
    Bh_j = edges.src['Bh']
    # e_ij = Ce_ij + Dhi + Ehj
    e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']
    edges.data['e'] = e_ij
    return {'Bh_j' : Bh_j, 'e_ij' : e_ij}
```

<!--
Thirdly, the `reduce_func` function collects the shipped messages by the `message_func` function. After collecting the node data `Ah` and shipped data `Bh_j` and `e_ij` from the `mailbox`, the line `h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)` computes the hidden representation of each node as indicated in (Eq. 5). Note however, that this only represents the term $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ without the $\texttt{ReLU}(\cdot)$ and residual connection.
-->

세번째로, `reduce_func` 함수는 보내진 메시지들을 `message_func`함수로 모은다. 노드 데이터 `Ah`와  보내진 데이터 `Bh_j` 그리고 `mailbox`에서의 `e_ij` 를 모은 후, `h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)` 라인은 (Eq.5) 에서 나타나는 각 노드들의 은닉 표현을 계산한다. 하지만 참고해야할 건, 오직 이는 $\texttt{ReLU}(\cdot)$ 와 잔차 연결을 뺀 $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ 항만을 표현한다.


```python
def reduce_func(self, nodes):
    Ah_i = nodes.data['Ah']
    Bh_j = nodes.mailbox['Bh_j']
    e = nodes.mailbox['e_ij']
    # sigma_ij = sigmoid(e_ij)
    sigma_ij = torch.sigmoid(e)
    # hi = Ahi + sum_j eta_ij * Bhj
    h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)
    return {'h' : h}
```

<!--
Inside the `forward` function, having called `g.update_all`, we obtain the results of graph convolution `h` and `e`, which represent the terms $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ from (Eq.5) and $\boldsymbol{e_{j}}$ from (Eq. 7) respectively. Then, we normalize `h` and `e` with respect to the graph node size and graph edge size respectively. Batch normalization is then applied so that we can train the network effectively. Finally, we apply $\texttt{ReLU}(\cdot)$ and add the residual connections to obtain the hidden representations for the nodes and edges, which are then returned by the `forward` function.
-->

`g.update_all`을 호출한 `forward` 함수 안에는  (Eq.5) 에서의 $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$와 (Eq. 7) 에서의  $\boldsymbol{e_{j}}$를 나타내는 그래프 합성곱의 결과 `h`와 `e`를 얻는다. 그런 다음 그래프 노드 크기와 그래프 엣지 크기에 대해 각각 `h`와 `e`를 정규화하면 배치 정규화가 적용되어 네트워크를 효과적으로 학습시킬 수 있게된다. 마지막으로,  `forward`함수에 의해 반환되는  노드와 엣지에 대한 은닉 표현을 얻기 위해 $\texttt{ReLU}(\cdot)$를 적용하고 잔차 연결을 추가한다.

```python
def forward(self, g, h, e, snorm_n, snorm_e):

    h_in = h # residual connection
    e_in = e # residual connection

    g.ndata['h']  = h
    g.ndata['Ah'] = self.A(h)
    g.ndata['Bh'] = self.B(h)
    g.ndata['Dh'] = self.D(h)
    g.ndata['Eh'] = self.E(h)
    g.edata['e']  = e
    g.edata['Ce'] = self.C(e)

    g.update_all(self.message_func, self.reduce_func)

    h = g.ndata['h'] # result of graph convolution
    e = g.edata['e'] # result of graph convolution

    h = h * snorm_n # normalize activation w.r.t. graph node size
    e = e * snorm_e # normalize activation w.r.t. graph edge size

    h = self.bn_node_h(h) # batch normalization
    e = self.bn_node_e(e) # batch normalization

    h = torch.relu(h) # non-linear activation
    e = torch.relu(e) # non-linear activation

    h = h_in + h # residual connection
    e = e_in + e # residual connection

    return h, e
```

<!--
Next, we define the `MLP_Layer` module which contains several fully connected layers (FCN). We create a list of fully connected layers and forward through the network.

Finally, we define our `GatedGCN` model which comprises of the previously defined classes: `GatedGCN_layer` and `MLP_layer`. The definition of our model (`GatedGCN`) is shown below.
-->

다음으로 여러개의 완전 연결 레이어 (FCN)을 포함하는`MLP_Layer` 모듈을 정의하고 완전 연결 레이어들의 리시트를 만들어 네트워크를 따라 전달한다.

마지막으로, 이전에 정의된 클래스 `GatedGCN_layer` and `MLP_layer`으로 구성된 `GatedGCN`를 정의한다. `GatedGCN`의 정의는 아래와 같다.

 ```python
 class GatedGCN(nn.Module):
    def __init__(self, input_dim, hidden_dim, output_dim, L):
        super().__init__()
        self.embedding_h = nn.Linear(input_dim, hidden_dim)
        self.embedding_e = nn.Linear(1, hidden_dim)
        self.GatedGCN_layers = nn.ModuleList([
            GatedGCN_layer(hidden_dim, hidden_dim) for _ in range(L)
        ])
        self.MLP_layer = MLP_layer(hidden_dim, output_dim)
    def forward(self, g, h, e, snorm_n, snorm_e):
        # input embedding
        h = self.embedding_h(h)
        e = self.embedding_e(e)
        # graph convnet layers
        for GGCN_layer in self.GatedGCN_layers:
            h, e = GGCN_layer(g, h, e, snorm_n, snorm_e)
        # MLP classifier
        g.ndata['h'] = h
        y = dgl.mean_nodes(g,'h')
        y = self.MLP_layer(y)
        return y
 ```

<!--
In our constructor, we define the embeddings for `e` and `h` (`self.embedding_e ` and `self.embedding_h`), `self.GatedGCN_layers` which is list (of size $L$) of our previously defined model: `GatedGCN_layer`, and finally `self.MLP_layer` which was also defined before. Next, using these initializations, we simply foward through the model and output `y`.

To better understand the model, we initiate an object of the model and print it for better visualization:
-->

생성자<sup>constructor</sup> 안에서 전에 정의된 모델 `GatedGCN_layer`와 `self.MLP_layer`의 ($L$크기를 가진) 리시트인 `e`와`h` (`self.embedding_e ` and `self.embedding_h`)의 임베딩을 정의한다. 그 다음, 이러한 초기화를 사용하여 모델을 진행하고 `y`를 출력한다.

```python
# instantiate network
model = GatedGCN(input_dim=1, hidden_dim=100, output_dim=8, L=2)
print(model)
```

<!--
The main structure of the model is shown below:
-->

모델의 주요 구조는 다음과 같다:

```python
GatedGCN(
  (embedding_h): Linear(in_features=1, out_features=100, bias=True)
  (embedding_e): Linear(in_features=1, out_features=100, bias=True)
  (GatedGCN_layers): ModuleList(
    (0): GatedGCN_layer(...)
    (1): GatedGCN_layer(... ))
  (MLP_layer): MLP_layer(
    (FC_layers): ModuleList(...))
```

<!--
Not surprisingly, we have two layers of `GatedGCN_layer` (since `L=2`) followed by a `MLP_layer` which finally yields an output of 8 values.

Moving on, we define our `train` and `evaluate` functions. In our `train` function, we have our generic code which takes samples from `dataloader`.  Next, `batch_graphs`, `batch_x`, `batch_e`, `batch_snorm_n` and `batch_snorm_e` are fed into our model which returns `batch_scores` (of size 8). The predicted scores are compared with the ground truth in our loss function: `loss(batch_scores, batch_labels)`. Then, we zero out gradients (`optimizer.zero_grad()`), perform backprop (`J.backward()`) and updates our weights (`optimizer.step()`). Finally, loss for epoch and training accuracy is calculated. Furthermore, we use a similar code for our `evaluate` function.
-->

당연하게도 (`L=2` 이므로) 두개의 `GatedGCN_layer` 레이어와  `MLP_layer` 가 있으며 최종적으로 8개의 값을 출력한다.

나아가 `train`과 `evaluate`함수를 정의하고 `train` 함수에서는 `dataloader` 에서 샘플을 가져오는 제네릭 코드<sup>generic code</sup>를 가지며, `batch_graphs`, `batch_x`, `batch_e`, `batch_snorm_n`, `batch_snorm_e`는 모델에 입력되어 (크기 8의) `batch_scores`을 출력하며 예측된 점수들은 손실함수의 실측 값 과 비교된다: `loss(batch_scores, batch_labels)`.  그 다음으로 경사도를 영처리(`optimizer.zero_grad()`), 를 하며 역전파 (`J.backward()`)와 가중치를 업데이트(`optimizer.step()`)한다.  마지막으로, 에폭<sup>epoch</sup>과 학습 정확도의 손실이 계산되며, 이는 `evaluate`에서 또한 비슷한 코드를 사용한다.

<!--
Finally, we are ready to train! We found that after 40 epochs of training, our model has learned to classify the graphs with a test accuracy of $87$%.
-->

드디어 학습할 모든 준비를 마쳤다! 40 에폭의 학습의 결과, 모델은 $87$%의 그래프 분류 정확도를 가졌다.
