---
lang-ref: ch.13-3
title: Graph Convolutional Network
lecturer: Alfredo Canziani
authors: Go Inoue, Muhammad Osama Khan, Muhammad Shujaat Mirza, Muhammad Muneeb Afzal
date: 28 Apr 2020
lang: ja
translation-date: 6 Dec 2020
translator: Shiro Takagi
---


<!-- ## Introduction to Graph Convolutional Network (GCN) -->
## Graph Convolutional Network (GCN)の紹介

<!-- Graph Convolutional Network (GCN) is one type of architecture that utilizes the structure of data.
Before going into details, let's have a quick recap on self-attention, as GCN and self-attention are conceptually relevant. -->

GCN（Graph Convolutional Network）は、データの構造を利用したアーキテクチャの一種です。
GCNとself-attentionは概念的に関係があるので、詳細に入る前に、self-attentionについて簡単に復習しておきましょう。

<!-- ### Recap: Self-attention -->
### Self-attentionのおさらい

<!-- - In self-attention, we have a set of input $\lbrace\boldsymbol{x}\_{i}\rbrace^{t}_{i=1}$.
Unlike a sequence, it does not have an order.
- Hidden vector $\boldsymbol{h}$ is given by linear combination of the vectors in the set.
- We can express this as $\boldsymbol{X}\boldsymbol{a}$ using matrix vector multiplication, where $\boldsymbol{a}$ contains coefficients that scale the input vector $\boldsymbol{x}_{i}$. -->

- Self-attentionでは、入力の集合$\lbrace\boldsymbol{x}_{i}\rbrace^{t}_{i=1}$があります。
系列データと違って、順番がありません。
- 隠れベクトル $\boldsymbol{h}$ は、この集合の中のベクトルの線形結合で与えられます。
- これを行列ベクトルの掛け算を使って $\boldsymbol{X}\boldsymbol{a}$ と表現します。ここで$\boldsymbol{a}$ には入力ベクトル $\boldsymbol{x}_{i}$ をスケーリングする係数が含まれています。

<!-- *For a detailed explanation, refer to the notes of [Week 12]({{site.baseurl}}/en/week12/12-3/).* -->
*詳しい説明は、第12週のノート({{site.baseurl}}/ja/week12/12-3/)を参照してください。*

<!-- ### Notation -->
### ノーテーション

<!-- <center>
<img src="{{site.baseurl}}/images/week13/13-3/figure1.png" height="400px" /><br>
<b>Fig. 1</b>: Graph Convolutional Network
</center> -->
<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure1.png" height="400px" /><br>
<b>図1</b>: グラフ畳み込みニューラルネット
</center>

<!-- In Figure 1, vertex $v$ is comprised of two vectors: input $\boldsymbol{x}$ and its hidden representation $\boldsymbol{h}$.
We also have multiple vertices $v_{j}$, which is comprised of $\boldsymbol{x}_{j}$ and $\boldsymbol{h}_{j}$.
In this graph, vertices are connected with directed edges. -->
図１において、頂点 $v$ は、入力 $\boldsymbol{x}$ とその隠れ表現 $\boldsymbol{h}$ の２つのベクトルで構成されています。
また、複数の頂点 $v_{j}$ があり、これは $\boldsymbol{x}_j$ と $\boldsymbol{h}_j$ で構成されています。
このグラフでは、頂点は有向辺で結ばれています。


<!-- We represent these directed edges with adjacency vector $\boldsymbol{a}$, where each element $\alpha_{j}$ is set to $1$ if there is a directed edge from $v_{j}$ to $v$. -->
この有向辺を、隣接ベクトル $\boldsymbol{a}$ で表現し、各要素 $\alpha_{j}$ は、$v_{j}$ から $v$ への有向辺があれば $1$ とします。
<!--  -->
$$
\alpha_{j} \stackrel{\tiny \downarrow}{=} 1 \Leftrightarrow v_{j} \rightarrow v
\tag{Eq. 1}
$$
<!-- 
The degree (number of incoming edges) $d$ is defined as the norm of this adjacency vector, i.e. $\Vert\boldsymbol{a}\Vert_{1} $, which is the number of ones in the vector $\boldsymbol{a}$. -->
次数（入ってくる辺の数）$d$は、この隣接関係ベクトルのノルム、すなわち、$\boldsymbol{a}$の中の1の数である$\Vert\boldsymbol{a}\Vert_{1}$として定義されます。

$$
    d = \Vert\boldsymbol{a}\Vert_{1}
\tag{Eq. 2}
$$

<!-- The hidden vector $\boldsymbol{h}$ is given by the following expression: -->
隠れベクトル $\boldsymbol{h}$ は次の式で与えられます。

$$
    \boldsymbol{h}=f(\boldsymbol{U}\boldsymbol{x} + \boldsymbol{V}\boldsymbol{X}\boldsymbol{a}d^{-1})
\tag{Eq. 3}
$$

<!-- where $f(\cdot)$ is a non-linear function such as ReLU $(\cdot)^{+}$, Sigmoid $\sigma(\cdot)$, and hyperbolic tangent $\tanh(\cdot)$.

The $\boldsymbol{U}\boldsymbol{x}$ term takes into account the vertex $v$ itself, by applying rotation $\boldsymbol{U}$ to the input $v$. -->

ここで、$f(\cdot)$は、ReLU $(\cdot)^{+}$, Sigmoid $\sigma(\cdot)$, hyperbolic tangent $\tanh(\cdot)$などの非線形関数です。

この$\boldsymbol{U}\boldsymbol{x}$という項は、入力$v$に回転$\boldsymbol{U}$を適用することで、頂点$v$自体を考慮しています。

<!-- Remember that in self-attention, the hidden vector $\boldsymbol{h}$ is computed by $\boldsymbol{X}\boldsymbol{a}$, which means that the columns in $\boldsymbol{X}$ is scaled by the factors in $\boldsymbol{a}$.
In the context of GCN, this means that if we have multiple incoming edges, i.e., multiple ones in adjacency vector $\boldsymbol{a}$, $\boldsymbol{X}\boldsymbol{a}$ gets larger.
On the other hand, if we have only one incoming edge, this value gets smaller.
To remedy this issue of the value being proportionate to the number of incoming edges, we divide it by the number of incoming edges $d$.
We then apply rotation $\boldsymbol{V}$ to $\boldsymbol{X}\boldsymbol{a}d^{-1}$.

We can represent this hidden representation $\boldsymbol{h}$ for the entire set of inputs $\boldsymbol{x}$ using the following matrix notation: -->

Self-attentionでは、隠れベクトル $\boldsymbol{h}$ は、$\boldsymbol{X}\boldsymbol{a}$ で計算され、$\boldsymbol{X}$ の列は、$\boldsymbol{a}$の要素でスケーリングされることを覚えておいてください。
GCNの文脈では、これは、隣接ベクトルの中に複数の辺が入ってくると（例えば隣接行列$\boldsymbol{a}$のなかの複数の要素）$\boldsymbol{X}\boldsymbol{a}$が大きくなることを意味します。
一方、入ってくる辺が1つしかない場合は、この値は小さくなります。
この値が流入する辺の数に比例するという問題を解決するために、流入する辺の数$d$で割ってみましょう。
そして、$\boldsymbol{X}\boldsymbol{a}d^{-1}$に回転$\boldsymbol{V}$をかけます。

この隠れ表現$\boldsymbol{h}$を、入力の全集合 $\boldsymbol{x}$に対して、次のような行列記法で表現することができます。

$$
\{\boldsymbol{x}_{i}\}^{t}_{i=1}\rightsquigarrow \boldsymbol{H}=f(\boldsymbol{UX}+ \boldsymbol{VXAD}^{-1})
\tag{Eq. 4}
$$

<!-- where $\vect{D} = \text{diag}(d_{i})$. -->
ただし$\vect{D} = \text{diag}(d_{i})$です。


<!-- ## Residual Gated GCN Theory and Code -->
## Residual Gated GCNの理論とコード

<!-- Residual Gated Graph Convolutional Network is a type of GCN that can be represented as shown in Figure 2: -->
Residual Gatedグラフ畳み込みニューラルネットは、図２に示すようなGCNの一種です。

<!-- <center>
<img src="{{site.baseurl}}/images/week13/13-3/figure2.png" height="300px" /><br>
<b>Fig. 2</b>: Residual Gated Graph Convolutional Network
</center> -->
<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure2.png" height="300px" /><br>
<b>図2</b>: Residual Gatedグラフ畳み込みニューラルネット
</center>

<!-- As with the standard GCN, the vertex $v$ consists of two vectors: input $\boldsymbol{x}$ and its hidden representation $\boldsymbol{h}$. However, in this case, the edges also have a feature representation, where $\boldsymbol{e_{j}^{x}}$ represents the input edge representation and $\boldsymbol{e_{j}^{h}}$ represents the hidden edge representation.

The hidden representation $\boldsymbol{h}$ of the vertex $v$ is computed by the following equation: -->

標準的な GCN と同様に、頂点 $v$ は、入力 $\boldsymbol{x}$ とその隠れ表現 $\boldsymbol{h}$ の 2つのベクトルで構成されています。ただし、この場合、エッジにも特徴表現があり、入力辺の表現を $\boldsymbol{e_{j}^{x}}$、隠れ辺の表現を $\boldsymbol{e_{j}^{h}}$ とします。

頂点 $v$ の隠れ表現 $\boldsymbol{h}$ は、次式で求められます。

$$
    \boldsymbol{h}=\boldsymbol{x} + \bigg(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}\bigg)^{+}
\tag{Eq. 5}
$$

<!-- where $\boldsymbol{x}$ is the input representation, $\boldsymbol{Ax}$ represents a rotation applied to the input $\boldsymbol{x}$ and $\sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}$ denotes the summation of elementwise multiplication of rotated incoming features $\boldsymbol{Bx_{j}}$ and a gate $\eta(\boldsymbol{e_{j}})$. In contrast to the standard GCN above where we average the incoming representations, the gate term is critical to the implementation of Residual Gated GCN since it allows us to modulate the incoming representations based on the edge representations.

Note that the summation is only over vertices ${v_j}$ that have incoming edges to vertex ${v}$. The term residual (in Residual Gated GCN) comes from the fact that in order to calculate the hidden representation $\boldsymbol{h}$, we add the input representation $\boldsymbol{x}$. The gate term $\eta(\boldsymbol{e_{j}})$ is calculated as shown below: -->

ここで、$\boldsymbol{x}$  は入力表現です。$\boldsymbol{Ax}$ は、入力$\boldsymbol{x}$の回転を表し、$\sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}$ は、回転させた入力特徴量$\boldsymbol{Bx_{j}}$とゲート$\eta(\boldsymbol{e_{j}})$との要素毎の積の和を表します。上で紹介した標準的なGCNでは入力表現を平均化するのに対し、ゲート項は、エッジ表現に基づいて入力表現を変化させることができるので、Residual GatedGCNの実装には重要です。

なお、和は、頂点${v}$へ入力されるエッジを持つ頂点${v_j}$のみを対象としていることに注意してください。Residual Gated GCNにおけるresidualという用語は、隠れ表現 $\boldsymbol{h}$ を計算するために、入力表現 $\boldsymbol{x}$ を加えることに由来します。ゲート項$\eta(\boldsymbol{e_{j}})$は，次のように計算されます。

$$
    \eta(\boldsymbol{e_{j}})=\sigma(\boldsymbol{e_{j}})\bigg(\sum_{v_k→v}\sigma(\boldsymbol{e_{k}})\bigg)^{-1}
\tag{Eq. 6}
$$

<!-- The gate value $\eta(\boldsymbol{e_{j}})$ is a normalized sigmoid obtained by dividing the sigmoid of the incoming edge representation by the sum of sigmoids of all incoming edge representations. Note that in order to calculate the gate term, we need the representation of the edge $\boldsymbol{e_{j}}$, which can be computed using the equations below: -->
ゲート値 $\eta(\boldsymbol{e_{j}})$ は、入ってくる辺のシグモイドをすべての入ってくる辺のシグモイドの和で割った正規化シグモイドです。ゲート項を計算するためには、次の式を用いて計算できるエッジの表現 $\boldsymbol{e_{j}}$ が必要です。

$$
    \boldsymbol{e_{j}} = \boldsymbol{Ce_{j}^{x}} + \boldsymbol{Dx_{j}} + \boldsymbol{Ex}
\tag{Eq. 7}
$$

$$
    \boldsymbol{e_{j}^{h}}=\boldsymbol{e_{j}^{x}}+(\boldsymbol{e_{j}})^{+}
\tag{Eq. 8}
$$

<!-- The hidden edge representation $\boldsymbol{e_{j}^{h}}$ is obtained by the summation of the initial edge representation $\boldsymbol{e_{j}^{x}}$ and $\texttt{ReLU}(\cdot)$ applied to $\boldsymbol{e_{j}}$ where $\boldsymbol{e_{j}}$ is in turn given by the summation of a rotation applied to $\boldsymbol{e_{j}^{x}}$, a rotation applied to the input representation $\boldsymbol{x_{j}}$ of the vertex $v_{j}$ and a rotation applied to the input representation $\boldsymbol{x}$ of the vertex $v$.

*Note: In order to calculate hidden representations downstream (e.g. $2^\text{nd}$ layer hidden representations), we can simply replace the input feature representations by the $1^\text{st}$ layer feature representations in the equations above.* -->

辺の隠れ表現 $\boldsymbol{e_{j}^{h}}$ は、辺の表現の初期値  $\boldsymbol{e_{j}^{x}}$ と$\boldsymbol{e_{j}}$に適用された$\texttt{ReLU}(\cdot)$の和です。ここで、$\boldsymbol{e_{j}}$は、$\boldsymbol{e_{j}^{x}}$に適用される回転の和で与えられます。この回転は頂点 $v_{j}$ の入力表現 $\boldsymbol{x_{j}}$に対する回転と、頂点 $v$ の入力表現 $\boldsymbol{x}$ に対する回転です。

*注：下流の隠れ表現（例えば $2^\text{nd}$ 層の隠れ表現）を計算するには，入力特徴表現を上式の $1^\text{st}$ 層の特徴表現に置き換えれば大丈夫です。


<!-- ### Graph Classification and Residual Gated GCN Layer -->
### グラフ分類とResidual Gated GCN Layer

<!-- In this section, we introduce the problem of graph classification and code up a Residual Gated GCN layer. In addition to the usual import statements, we add the following: -->
ここでは、グラフ分類の問題点を紹介し、Residual Gated GCN層をコードに落としこみます。通常のimport文に加えて、以下を追加します。

```python
os.environ['DGLBACKEND'] = 'pytorch'
import dgl
from dgl import DGLGraph
from dgl.data import MiniGCDataset
import networkx as nx
```

<!-- The first line tells DGL to use PyTorch as the backend. Deep Graph Library ([DGL](https://www.dgl.ai/)) provides various functionalities on graphs whereas networkx allows us to visualise the graphs. -->

<!-- In this notebook, the task is to classify a given graph structure into one of 8 graph types. The dataset obtained from `dgl.data.MiniGCDataset` yields some number of graphs (`num_graphs`) with nodes between `min_num_v` and `max_num_v`. Hence, all the graphs obtained do not have the same number of nodes/vertices.

*Note: In order to familiarize yourself with the basics of `DGLGraphs`, it is recommended to go through the short tutorial [here](https://docs.dgl.ai/api/python/graph.html).*

Having created the graphs, the next task is to add some signal to the domain. Features can be applied to nodes and edges of a `DGLGraph`. The features are represented as a dictionary of names (strings) and tensors (**fields**). `ndata` and `edata` are syntax sugar to access the feature data of all nodes and edges. -->

最初の行では、DGLにPyTorchをバックエンドとして使うように指示しています。Deep Graph Library ([DGL](https://www.dgl.ai/)) はグラフに関する様々な機能を提供しています。

このnotebookでは、与えられたグラフ構造を8つのグラフタイプのうちの1つに分類することを課題としています。`dgl.data.MiniGCDataset` から得られるデータセットには、`min_num_v` から `max_num_v` の間にノードを持つグラフ (`num_graphs`) がいくつか含まれています。したがって、得られたグラフはすべて同じ数のノード/頂点を持つわけではありません。

注: `DGLGraphs` の基本を理解するためには、[こちら](https://docs.dgl.ai/api/python/graph.html)のチュートリアルを参照することをお勧めします。

グラフを作成したら、次の作業はドメインにシグナルを追加することです。特徴表現は、名前（文字列）とテンソル（**fields**）の辞書として表現されます。`ndata` と `edata` は、すべてのノードとエッジの特徴データにアクセスするための糖衣構文です。

<!-- The following code snippet shows how the features are generated. Each node is assigned a value equal to the number of incident edges, whereas each edge is assigned the value 1. -->
以下のコードスニペットは、特徴量がどのように生成されるかを示しています。各ノードには入射エッジの数に等しい値が割り当てられ、各エッジには値1が割り当てられます。

```python
def create_artificial_features(dataset):
    for (graph, _) in dataset:
        graph.ndata['feat'] = graph.in_degrees().view(-1, 1).float()
        graph.edata['feat'] = torch.ones(graph.number_of_edges(), 1)
    return dataset
```

<!-- Training and testing datasets are created and features are assigned as shown below: -->
訓練データとテストデータが作成され、特徴量が次のように割り当てられます。

```python
trainset = MiniGCDataset(350, 10, 20)
testset = MiniGCDataset(100, 10, 20)

trainset = create_artificial_features(trainset)
testset = create_artificial_features(testset)
```

<!-- A sample graph from the trainset has the following representation. Here, we observe that the graph has 15 nodes and 45 edges and both the nodes and edges have a feature representation of shape `(1,)` as expected. Furthermore, the `0` signifies that this graph belongs to class 0. -->
訓練データ集合のサンプルグラフは以下のような表現です。ここで、このグラフは15個のノードと45個のエッジを持ち、ノードもエッジも期待通りの形状 `(1,)` の特徴表現を持っていることがわかります。さらに、`0`はこのグラフがクラス0に属していることを示しています。

```python
(DGLGraph(num_nodes=15, num_edges=45,
         ndata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}
         edata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}), 0)
```


<!-- ### Note about DGL Message and Reduce Functions -->
### DGLのMessage関数とReduce関数に関する注意

<!-- >In DGL, the *message functions* are expressed as **Edge UDF**s (User Defined Functions). Edge UDFs take in a single argument `edges`. It has three members `src`, `dst`, and `data` for accessing source node features, destination node features, and edge features respectively.
The *reduce functions* are **Node UDF**s. Node UDFs have a single argument `nodes`, which has two members `data` and `mailbox`. `data` contains the node features and `mailbox` contains all incoming message features, stacked along the second dimension (hence the `dim=1` argument).
`update_all(message_func, reduce_func)` sends messages through all edges and updates all nodes. -->

>DGLでは、*メッセージ関数*は**Edge UDF**(ユーザー定義関数)として表現されます。Edge UDFは単一の引数 `edges` を持ちます。エッジUDFは、ソースノードの特徴、到達ノードの特徴、エッジの特徴、のそれぞれアクセスするために、`src`, `dst`, `data` の3つのメンバを持ちます。
**Node UDF**は、*reduce関数*です。ノードUDFは単一の引数 `nodes` を持ち、その引数には2つのメンバ `data` と `mailbox` を持ちます。`data`にはノードの特徴が含まれ、`mailbox`にはすべての受信メッセージの特徴が含まれ、2番目の次元に沿って積み上げられています（そのため、`dim=1`引数が指定されています←次元は0から始まるので）。
`update_all(message_func, reduce_func)` は、すべてのエッジを経由してメッセージを送信し、すべてのノードを更新します。


<!-- ### Gated Residual GCN Layer Implementation -->
### Gated Residual GCN層の実装

<!-- A Gated Residual GCN layer is implemented as shown in the code snippets below.

Firstly, all the rotations of the input features $\boldsymbol{Ax}$, $\boldsymbol{Bx_{j}}$, $\boldsymbol{Ce_{j}^{x}}$, $\boldsymbol{Dx_{j}}$ and $\boldsymbol{Ex}$ are computed by defining `nn.Linear` layers inside the `__init__` function and then forward propagating the input representations `h` and `e` through the linear layers inside the `forward` function. -->

Gated Residual GCN層は、以下のコードスニペットのように実装されています。

まず、`__init__`関数内に、`nn.Linear`層を定義して、入力表現 `h`, `e` を `forward` 関数内の線形層を介して順伝搬させることで、入力特徴 $\boldsymbol{Ax}$, $\boldsymbol{Bx_{j}}$, $\boldsymbol{Ce_{j}^{x}}$, $\boldsymbol{Dx_{j}}$, $\boldsymbol{Ex}$ のすべての回転を計算します。

```python
class GatedGCN_layer(nn.Module):

    def __init__(self, input_dim, output_dim):
        super().__init__()
        self.A = nn.Linear(input_dim, output_dim)
        self.B = nn.Linear(input_dim, output_dim)
        self.C = nn.Linear(input_dim, output_dim)
        self.D = nn.Linear(input_dim, output_dim)
        self.E = nn.Linear(input_dim, output_dim)
        self.bn_node_h = nn.BatchNorm1d(output_dim)
        self.bn_node_e = nn.BatchNorm1d(output_dim)
```

<!-- Secondly, we compute the edge representations. This is done inside the `message_func` function, which iterates over all the edges and computes the edge representations. Specifically, the line `e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']` computes *(Eq. 7)* from above. The `message_func` function ships `Bh_j` (which is $\boldsymbol{Bx_{j}}$ from (Eq. 5)) and `e_ij` (Eq. 7) through the edge into the destination node's mailbox. -->
次に、エッジの表現を計算します。これは `message_func` 関数の中で行われ、すべての辺を反復処理して辺の表現を計算します。具体的には、`e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']` という行で *(Eq. 7)* を計算します。関数 `message_func` は、`Bh_j` (これは (Eq. 5)の$\boldsymbol{Bx_{j}}$  です) と `e_ij` (Eq. 7) を、エッジを経由して宛先ノードのメールボックスに送信します。

```python
def message_func(self, edges):
    Bh_j = edges.src['Bh']
    # e_ij = Ce_ij + Dhi + Ehj
    e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']
    edges.data['e'] = e_ij
    return {'Bh_j' : Bh_j, 'e_ij' : e_ij}
```

<!-- Thirdly, the `reduce_func` function collects the shipped messages by the `message_func` function. After collecting the node data `Ah` and shipped data `Bh_j` and `e_ij` from the `mailbox`, the line `h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)` computes the hidden representation of each node as indicated in (Eq. 5). Note however, that this only represents the term $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ without the $\texttt{ReLU}(\cdot)$ and residual connection. -->
第三に、`reduce_func`関数は、`message_func`関数によって配送されたメッセージを収集します。メールボックスからノードデータ `Ah` と配送されたデータ `Bh_j`, `e_ij` を収集した後、`h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)` 行により、(Eq. 5)で示されるように、各ノードの隠れ表現を計算します。ただし、これは$\texttt{ReLU}(\cdot)$とresidual connectionなしで $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$項だけを表しています。

```python
def reduce_func(self, nodes):
    Ah_i = nodes.data['Ah']
    Bh_j = nodes.mailbox['Bh_j']
    e = nodes.mailbox['e_ij']
    # sigma_ij = sigmoid(e_ij)
    sigma_ij = torch.sigmoid(e)
    # hi = Ahi + sum_j eta_ij * Bhj
    h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)
    return {'h' : h}
```

<!-- Inside the `forward` function, having called `g.update_all`, we obtain the results of graph convolution `h` and `e`, which represent the terms $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ from (Eq.5) and $\boldsymbol{e_{j}}$ from (Eq. 7) respectively. Then, we normalize `h` and `e` with respect to the graph node size and graph edge size respectively. Batch normalization is then applied so that we can train the network effectively. Finally, we apply $\texttt{ReLU}(\cdot)$ and add the residual connections to obtain the hidden representations for the nodes and edges, which are then returned by the `forward` function. -->
関数 `forward` の中で `g.update_all` を呼び、グラフの畳み込みの結果 `h` と `e` を得ます。これは (Eq.5)より、$(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$を、(Eq.7)より$\boldsymbol{e_{j}}$を表しています。そして、グラフのノードサイズとグラフのエッジサイズを基準にして `h` と `e` を正規化します。その後、ネットワークを効率的に学習できるように、バッチ正規化を行います。最後に、$\texttt{ReLU}(\cdot)$を適用し、residual connectionを加えて、ノードとエッジの隠れ表現を得ます。

```python
def forward(self, g, h, e, snorm_n, snorm_e):

    h_in = h # residual connection
    e_in = e # residual connection

    g.ndata['h']  = h
    g.ndata['Ah'] = self.A(h)
    g.ndata['Bh'] = self.B(h)
    g.ndata['Dh'] = self.D(h)
    g.ndata['Eh'] = self.E(h)
    g.edata['e']  = e
    g.edata['Ce'] = self.C(e)

    g.update_all(self.message_func, self.reduce_func)

    h = g.ndata['h'] # result of graph convolution
    e = g.edata['e'] # result of graph convolution

    h = h * snorm_n # normalize activation w.r.t. graph node size
    e = e * snorm_e # normalize activation w.r.t. graph edge size

    h = self.bn_node_h(h) # batch normalization
    e = self.bn_node_e(e) # batch normalization

    h = torch.relu(h) # non-linear activation
    e = torch.relu(e) # non-linear activation

    h = h_in + h # residual connection
    e = e_in + e # residual connection

    return h, e
```

<!-- Next, we define the `MLP_Layer` module which contains several fully connected layers (FCN). We create a list of fully connected layers and forward through the network.

Finally, we define our `GatedGCN` model which comprises of the previously defined classes: `GatedGCN_layer` and `MLP_layer`. The definition of our model (`GatedGCN`) is shown below. -->

次に、複数の全結合層(FCN)を含む `MLP_Layer` モジュールを定義します。全結合層のリストを作成し、順伝播を定義します。

最後に、`GatedGCN_layer` と `MLP_layer` という先に定義したクラスからなる `GatedGCN` モデルを定義します。`GatedGCN`モデルの定義を以下に示します。

 ```python
 class GatedGCN(nn.Module):
    def __init__(self, input_dim, hidden_dim, output_dim, L):
        super().__init__()
        self.embedding_h = nn.Linear(input_dim, hidden_dim)
        self.embedding_e = nn.Linear(1, hidden_dim)
        self.GatedGCN_layers = nn.ModuleList([
            GatedGCN_layer(hidden_dim, hidden_dim) for _ in range(L)
        ])
        self.MLP_layer = MLP_layer(hidden_dim, output_dim)
    def forward(self, g, h, e, snorm_n, snorm_e):
        # input embedding
        h = self.embedding_h(h)
        e = self.embedding_e(e)
        # graph convnet layers
        for GGCN_layer in self.GatedGCN_layers:
            h, e = GGCN_layer(g, h, e, snorm_n, snorm_e)
        # MLP classifier
        g.ndata['h'] = h
        y = dgl.mean_nodes(g,'h')
        y = self.MLP_layer(y)
        return y
 ```

<!-- In our constructor, we define the embeddings for `e` and `h` (`self.embedding_e ` and `self.embedding_h`), `self.GatedGCN_layers` which is list (of size $L$) of our previously defined model: `GatedGCN_layer`, and finally `self.MLP_layer` which was also defined before. Next, using these initializations, we simply foward through the model and output `y`.

To better understand the model, we initiate an object of the model and print it for better visualization: -->

私たちのコンストラクタでは、`e`と`h`の埋め込み(`self.embedding_e ` と `self.embedding_h`)、`self.GatedGCN_layers`を定義しています。これは以前定義したモデル`GatedGCN_layer`からなるサイズ$L$のリストです。また、`self.MLP_layer`も定義します。これも以前定義したものです。次に、これらの初期化を使って、順伝播して `y` を出力します。

モデルをより理解しやすくするために、モデルのオブジェクトを初期化してして`print`します。

```python
# instantiate network
model = GatedGCN(input_dim=1, hidden_dim=100, output_dim=8, L=2)
print(model)
```

<!-- The main structure of the model is shown below: -->
モデルの主な構造をいかに示します。

```python
GatedGCN(
  (embedding_h): Linear(in_features=1, out_features=100, bias=True)
  (embedding_e): Linear(in_features=1, out_features=100, bias=True)
  (GatedGCN_layers): ModuleList(
    (0): GatedGCN_layer(...)
    (1): GatedGCN_layer(... ))
  (MLP_layer): MLP_layer(
    (FC_layers): ModuleList(...))
```

<!-- Not surprisingly, we have two layers of `GatedGCN_layer` (since `L=2`) followed by a `MLP_layer` which finally yields an output of 8 values. -->

<!-- Moving on, we define our `train` and `evaluate` functions. In our `train` function, we have our generic code which takes samples from `dataloader`.  Next, `batch_graphs`, `batch_x`, `batch_e`, `batch_snorm_n` and `batch_snorm_e` are fed into our model which returns `batch_scores` (of size 8). The predicted scores are compared with the ground truth in our loss function: `loss(batch_scores, batch_labels)`. Then, we zero out gradients (`optimizer.zero_grad()`), perform backprop (`J.backward()`) and updates our weights (`optimizer.step()`). Finally, loss for epoch and training accuracy is calculated. Furthermore, we use a similar code for our `evaluate` function.

Finally, we are ready to train! We found that after 40 epochs of training, our model has learned to classify the graphs with a test accuracy of $87$%. -->


驚くことではありませんが、`GatedGCN_layer`の2つの層（`L=2`なので）と、`MLP_layer`の2つの層があり、最終的には8つの値が出力されます。

次に、`train` と `evaluate` 関数を定義します。`train` 関数では、`dataloader` からサンプルを取得する汎用コードを定義しています。 次に、`batch_graphs`, `batch_x`, `batch_e`, `batch_snorm_n`, `batch_snorm_e` をモデルに入力し、`batch_scores` (サイズ8) を返します。予測されたスコアは、損失関数 `loss(batch_scores, batch_labels)` を用いて真の値と比較されます。次に、勾配をゼロにし (`optimizer.zero_grad()`)、逆伝播を行い (`J.backward()`)、重みを更新します (`optimizer.step()`)。最後に、各エポックの損失と精度を計算します。さらに、同様のコードを `evaluate` 関数にも用います。

最後に、訓練の準備ができました！40エポックの学習を行った結果、我々のモデルはテスト精度$87$%でグラフを分類することを学習したことがわかりました。
