---
lang-ref: ch.08-3
title: Generative Models - Variational Autoencoders
lecturer: Alfredo Canziani
authors: Richard Pang, Aja Klevs, Hsin-Rung Chou, Mrinal Jain
date: 24 March 2020
lang: ja
translation-date: 28 Nov 2020
translator: Jesmer Wong
---


### [復習: オートエンコーダー (AE)](https://www.youtube.com/watch?v=7Rb4s9wNOmc&t=155s)

<!-- To summarize at a high level, a very simple form of AE is as follows:

- First, the autoencoder takes in an input and maps it to a hidden state through an affine transformation$\boldsymbol{h} = f(\boldsymbol{W}_h \boldsymbol{x} + \boldsymbol{b}_h)$, where$f$is an (element-wise) activation function. This is the **encoder** stage. Note that$\boldsymbol{h}$is also called the **code**.
- Next,$\hat{\boldsymbol{x}} = g(\boldsymbol{W}_x \boldsymbol{h} + \boldsymbol{b}_x)$, where$g$is an activation function. This is the **decoder** stage.

*For a detailed explaination, refer to the notes of [Week 7]({{site.baseurl}}/en/week07/07-3/).*　-->

大まかに要約すると、非常にシンプルなオートエンコーダー(AE) は次のとおりです。

- 最初に、オートエンコーダーは入力を受け取り、アフィン変換を介して隠れ状態に写像します$\boldsymbol{h} = f(\boldsymbol{W}_h \boldsymbol{x} + \boldsymbol{b}_h)$。ここで、$f$は（要素ごとの）活性化関数です。 これは **エンコーダー** のステージです。$\boldsymbol{h}$は　**コード**　とも呼ばれます。
- 次に、$\hat{\boldsymbol{x}} = g（\boldsymbol {W} _x \boldsymbol {h} + \boldsymbol {b} _x）$です。ここで$g$は活性化関数です。 これは**デコーダー**ステージです。

*詳細な説明については、[第7週]({{site.baseurl}} / en / week07 / 07-3 /)のメモを参照してください。*


<!-- ### Intuition behind VAE and a comparison with classic autoencoders

Next, we introduce Variational Autoencoders (or VAE), a type of generative models. But why do we even care about generative models? To answer the question, discriminative models learn to make predictions given some observations, but generative models aim to simulate the data generation process. One effect is that generative models can better understand the underlying causal relations which leads to better generalization.

Note that although VAE has "Autoencoders" (AE) in its name (because of structural or architectural similarity to auto-encoders), the formulations between VAEs and AEs are very different. See *Figure 1* below.　-->

### VAEの背後にある直感と従来のオートエンコーダーとの比較

次に、生成モデルの一種である変分オートエンコーダー（またはVAE）を紹介します。なぜ生成モデルまで気にするのかというと、識別モデルはいくつかの観察を与えられて結果を予測することについて学習しますが、生成モデルはデータ生成プロセスをシミュレーションすることを目的としているからです。 よって、生成モデルは背後の因果関係をよりよく理解することができ、そのおかげでよりよく汎化することができます。

VAEの名前には「オートエンコーダー」（AE）が含まれていますが（オートエンコーダーと構造的またはアーキテクチャ的に類似しているため）、定式化の上では大きく異なります。以下の*図1*を参照してください。

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_1.png" height="400px" /><br>
<b>図1</b>: VAE *vs.* 通常の AE
</center>

<!-- **What's the difference between variational auto-encoder (VAE) and classic auto-encoder (AE)?**

For VAE:
- First, the encoder stage: we pass the input$\boldsymbol{x}$to the encoder. Instead of generating a hidden representation$\boldsymbol{h}$(the code) in AE, the code in VAE comprises two things:$\mathbb{E}(\boldsymbol{z})$and$\mathbb{V}(\boldsymbol{z})$where$\boldsymbol{z}$is the latent random variable following a Gaussian distribution with mean$\mathbb{E}(\boldsymbol{z})$and variance$\mathbb{V}(\boldsymbol{z})$. Note that people use Gaussian distributions as the encoded distribution in practice, but other distributions can be used as well.
- The encoder will be a function from$\mathcal{X}$to$\mathbb{R}^{2d}$:$\boldsymbol{x} \mapsto \boldsymbol{h}$(here we use$\boldsymbol{h}$to represent the concatenation of$\mathbb{E}(\boldsymbol{z})$and$\mathbb{V}(\boldsymbol{z})$).
- Next, we will sample$\boldsymbol{z}$from the above distribution parametrized by the encoder; specifically,$\mathbb{E}(\boldsymbol{z})$and$\mathbb{V}(\boldsymbol{z})$are passed into a **sampler** to generate the latent variable$\boldsymbol{z}$.
- Next,$\boldsymbol{z}$is passed into the decoder to generate$\hat{\boldsymbol{x}}$.
    - The decoder will be a function from$\mathcal{Z}$to$\mathbb{R}^{n}$:$\boldsymbol{z} \mapsto \boldsymbol{\hat{x}}$.

In fact, for classic autoencoder, we can think of$\boldsymbol{h}$as just the vector$\E(\boldsymbol{z})$in the VAE formulation. In short, the main difference between VAEs and AEs is that VAEs have a good latent space that enables generative process.　-->

**変分オートエンコーダー（VAE）と通常のオートエンコーダー（AE）の違いは何ですか？**

VAEの場合：
- 最初に、エンコーダーステージについてです：入力$\boldsymbol{x}$をエンコーダーに渡します。 AEで隠れ表現$\boldsymbol{h}$（コード）を生成する代わりに、VAEのコードは$\mathbb{E}(\boldsymbol{z})$と$\mathbb{V}(\boldsymbol{z})$の２種類があります。ここで、$\boldsymbol{z}$は、平均$\mathbb{E}(\boldsymbol{z})$と分散$\mathbb{V}(\boldsymbol{z})$を持つガウス分布に従う潜在確率変数です。実際には、エンコードされた分布としてはガウス分布ではない分布も使用できます。
- エンコーダーは$\mathcal{X}$から$\mathbb{R}^{2d}$への関数$\boldsymbol{x} \mapsto \boldsymbol{h}$です。ただし、ここでは$\boldsymbol{h}$で、$\mathbb {E}(\boldsymbol {z})$と$\mathbb{V}(\boldsymbol {z})$の連結を表しています。
- 次に、エンコーダーによってパラメトライズされた上記の分布から$\boldsymbol{z}$をサンプリングします。具体的には、$\mathbb{E}(\boldsymbol{z})$と$\mathbb{V}(\boldsymbol {z})$が**サンプラー**に渡され、潜在変数$\boldsymbol{z}$が生成されます。
- 次に、$\boldsymbol{z}$がデコーダーに渡され、$\hat{\boldsymbol{x}}$が生成されます。
    - デコーダーは$\mathcal{Z}$から$\mathbb{R}^{n}$への関数$\boldsymbol{z} \mapsto \boldsymbol{\hat{x}}$です。

実際、通常のオートエンコーダの場合、$\boldsymbol{h}$はVAEの定式化におけるベクトル$\E(\boldsymbol{z})$とみなすことができます。要するに、VAEとAEの主な違いは、VAEは生成プロセスを可能にする優れた潜在空間を有しているということです。


<!-- ### The VAE objective (loss) function

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_2.png" /><br>
<b>Fig. 2</b>: Mapping from input space to latent space
</center>

*See Figure 2 above. For now, ignore the top-right corner (which is the reparameterisation trick explained in the next section).*

First, we encode from input space (left) to latent space (right), through encoder and noise. Next, we decode from latent space (right) to output space (left). To go from the latent to input space (the generative process) we will need to either learn the distribution (of the latent code) or enforce some structure. In our case, VAE enforces some structure to the latent space.


As usual, to train VAE, we minimize a loss function. The loss function is therefore composed of a reconstruction term as well as a regularization term.
- The reconstruction term is on the final layer (left side of the figure). This corresponds to$l(\boldsymbol{x}, \hat{\boldsymbol{x}})$in the figure.
- The regularization term is on the latent layer, to enforce some specific Gaussian structure on the latent space (right side of the figure). We do so by using a penalty term$l_{KL}(\boldsymbol{z}, \mathcal{N}(\boldsymbol{0}, \boldsymbol{I}_d))$. Without this term, VAE will act like a classic autoencoder, which may lead to overfitting, and we won't have the generative properties that we desire.　-->

### VAEの目的（損失）関数

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_2.png" /><br>
<b>図2</b>: 入力空間から潜在空間への写像
</center>


*上記の図2を参照してください。とりあえずは右上のもの（次のセクションで説明するreparametrization trick）は無視してください。*

まず、エンコーダーとノイズを使用して、入力空間（左）から潜在空間（右）にエンコードします。次に、潜在空間（右）から出力空間（左）にデコードします。潜在空間から入力空間（生成プロセス）に移行するには、（潜在コードの）分布を学習するか、何らかの構造を強制する必要があります。今回の場合、VAEは潜在空間に何らかの構造を強制するようにします。


いつものように、VAEを訓練するために、損失関数を最小化します。損失関数は、再構成項と正則化項で構成されます。
- 再構成項は最終層（図の左側）にあります。これは、図の$l(\boldsymbol{x}, \hat{\boldsymbol{x}})$に対応します。
- 正則化項は潜在層にあり、潜在空間（図の右側）に特定のガウス構造を適用します。これを行うには、ペナルティ項$l_{KL}(\boldsymbol{z}, \mathcal{N}(\boldsymbol{0}, \boldsymbol{I}_d))$を使用します。これがないと、VAEは普通のオートエンコーダーのように機能し、オーバーフィッティングしてしまう可能性があり、私たちが必要としている、生成する能力を得ることができません。


<!-- ### Discussion on sampling$\boldsymbol{z}$(**reparameterisation trick**)

How do we sample from the distribution returned by the encoder in VAE? According to above, we sample from the Gaussian distribution, in order to obtain$\boldsymbol{z}$. However, this is problematic, because when we do gradient descent to train the VAE model, we don't know how to do backpropagation through the sampling module.

Instead, we use the **reparameterization trick** to "sample"$\boldsymbol{z}$. We use$\boldsymbol{z} = \mathbb{E}(\boldsymbol{z}) + \boldsymbol{\epsilon} \odot \sqrt{\mathbb{V}(\boldsymbol{z})}$where$\epsilon\sim \mathcal{N}(\boldsymbol{0}, \boldsymbol{I}_d)$. In this case, backpropagation in training is possible. Specifically, the gradients will go through the (element-wise) multiplication and addition in the above equation.　-->

### サンプリング$\boldsymbol{z}$に関する議論（**reparametrization trick**）

VAEのエンコーダーによって返される分布からどうやってサンプリングをするのでしょうか？ 上記によると、$\boldsymbol{z}$は、ガウス分布からサンプリングします。 ただし、VAEモデルを訓練するために勾配降下法をも用いる場合、サンプリング・モジュールを介して逆伝播を実行する方法がわからないため、これは問題になります。

代わりに、**reparametrization trick**を使用して$\boldsymbol{z}$を「サンプリング」します。$\boldsymbol{z} = \mathbb{E}(\boldsymbol{z}) + \boldsymbol{\epsilon} \odot \sqrt{\mathbb{V}(\boldsymbol{z})}$を使用します。ここで$\epsilon\sim \mathcal{N}(\boldsymbol{0}, \boldsymbol{I}_d)$です。この場合、訓練時に逆伝播をすることが可能です。 具体的には、勾配は上記の式の（要素ごとの）積と和を通過します。


<!-- ## [Breaking apart the VAE Loss Function](https://www.youtube.com/watch?v=7Rb4s9wNOmc&t=997s)


### Visualizing Latent Variable Estimates and Reconstruction Loss

As stated above, the loss function for the VAE contains two parts: a reconstruction term and a regularization term.  We can write this as

$$
l(\boldsymbol{x}, \hat{\boldsymbol{x}}) = l_{reconstruction} + \beta l_{\text{KL}}(\boldsymbol{z},\mathcal{N}(\textbf{0}, \boldsymbol{I}_d))
$$

To visualize the purpose of each term in the loss function, we can think of each estimated$\boldsymbol{z}$value as a circle in$2d$space, where the centre of the circle is$\mathbb{E}(\boldsymbol{z})$and the surrounding area are the possible values of$\boldsymbol{z}$determined by$\mathbb{V}(\boldsymbol{z}).$-->

## [VAEの損失関数の分析](https://www.youtube.com/watch?v=7Rb4s9wNOmc&t=997s)

### 潜在変数の推定値と再構成損失の可視化

上記のように、VAEの損失関数には、再構成項と正則化項の2つの部分が含まれます。 これは次のように書くことができます

$$
l(\boldsymbol{x}, \hat{\boldsymbol{x}}) = l_{reconstruction} + \beta l_{\text{KL}}(\boldsymbol{z},\mathcal{N}(\textbf{0}, \boldsymbol{I}_d))
$$

損失関数の各項を可視化するために、推定された各$\boldsymbol{z}$の値を、円の中心が$\mathbb{E}(\boldsymbol{z})$で、その周辺の領域が$\mathbb{V}(\boldsymbol{z})$によって定まる$\boldsymbol{z}$のとりうる値であるような、$2$次元空間の円と考えることができます。

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_3.png" height="350px" /><br>
<b>図3</b>: 潜在空間における円としてのベクトル$z$の可視化
</center>

<!-- *In Figure 3 above, each bubble represents an estimated region of$\boldsymbol{z}$, and the arrows represent how the reconstruction term pushes each estimated value away from the others, which is explained more below.*

If there is overlap between any two estimates of$z$, (visually, if two bubbles overlap) this creates ambiguity for reconstruction because the points in the overlap can be mapped to both original inputs. Therefore the reconstruction loss will push the points away from one another.

However, if we use just the reconstruction loss, the estimates will continue to be pushed away from each other and the system could blow up.  This is where the penalty term comes in. 

**Note:** for binary inputs the reconstruction loss is

-->

*上記の図3では、各円は$\boldsymbol{z}$の推定領域を表し、矢印は再構成項が各推定値を他の推定値から遠ざける方法を表しています。これについては以下で詳しく説明します。*

$z$の任意の2つの推定値の間に重なりがある場合（視覚的には、2つの円が重なっている場合）、重なっている領域の中の点を両方の元の入力にマッピングできるため、再構成にあいまいさが生じます。 したがって、再構成損失は、点を互いに遠ざけます。

ただし、再構成損失のみを使用すると、推定値は互いに離れたままになり、システムが発散する可能性があります。これがペナルティ項を考える理由です。

**注：**バイナリ入力の場合、再構成損失は

$$
l(\boldsymbol{x}, \hat{\boldsymbol{x}}) = - \sum\limits_{i=1}^n [x_i \log{(\hat{x_i})} + (1 - x_i)\log{(1-\hat{x_i})}]
$$

<!-- and for real valued inputs the reconstruction loss is -->
実数値の入力の場合、再構成損失は

$$
l(\boldsymbol{x}, \hat{\boldsymbol{x}}) = \frac{1}{2} \Vert\boldsymbol{x} - \hat{\boldsymbol{x}} \Vert^2
$$

<!-- 
### The penalty term

The second term is the relative entropy (a measure of the distance between two distributions) between$\boldsymbol{z}$which comes from a Gaussian with mean$\mathbb{E}(\boldsymbol{z})$, variance$\mathbb{V}(\boldsymbol{z})$and the standard normal distribution. If we expand this second term in the VAE loss function we get:

$$
\beta l_{\text{KL}}(\boldsymbol{z},\mathcal{N}(\textbf{0}, \boldsymbol{I}_d)) = \frac{\beta}{2} \sum\limits_{i=1}^d(\mathbb{V}(z_i) - \log{[\mathbb{V}(z_i)]} - 1 + \mathbb{E}(z_i)^2)
$$

Where each expression in the summation has four terms. Below we write out and graph the first three terms in *Figure 4*.　-->

### ペナルティ項

2番目の項は、平均$\mathbb{E}(\boldsymbol {z})$、分散$\mathbb{V}(\boldsymbol{z})$のガウス分布に由来する$\boldsymbol{z}$と標準正規分布の間の相対エントロピー（2つの分布の間の距離）です。 この第2項をVAE損失関数で拡張すると、次のようになります。

$$
\beta l_{\text{KL}}(\boldsymbol{z},\mathcal{N}(\textbf{0}, \boldsymbol{I}_d)) = \frac{\beta}{2} \sum\limits_{i=1}^d(\mathbb{V}(z_i) - \log{[\mathbb{V}(z_i)]} - 1 + \mathbb{E}(z_i)^2)
$$

各式には合計すると4つの項があります。 以下に、*図4*の最初の3つの項を書き出してグラフ化します。

$$
v_i = \mathbb{V}(z_i) - \log{[\mathbb{V}(z_i)]} - 1
$$

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_4.png" /><br>
<b>図4</b>: 相対エントロピーがどのようにして円の分散が1となるように強制するかを示したプロット
</center>

<!-- So we can see that this expression is minimized when$z_i$has variance 1.  Therefore our penalty loss will keep the variance of our estimated latent variables at around 1.  Visually, this means our "bubbles" from above will have a radius of around 1.

The last term,$\mathbb{E}(z_i)^2$, minimizes the distance between the$z_i$and therefore prevents the "exploding" encouraged by the reconstruction term.　-->

この計算をみると、$z_i$の分散が1の場合、この式が最小化されることがわかります。したがって、ペナルティ損失により、推定された潜在変数の分散が約1に保たれます。視覚的には、上の「バブル（円）」の半径は約1になります。

最後の項$\mathbb{E}(z_i)^2$は、$z_i$間の距離を最小化するため、再構成項によって促進される発散を防ぎます。

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_5.png" height="400px"/><br>
<b>図5</b>: 「バブルのバブル」としてのんVAEの解釈
</center>

<!-- *Figure 5 above shows how VAE loss pushed the estimated latent variables as close together as possible without any overlap while keeping the estimated variance of each point around one.*

**Note:** The$\beta$in the VAE loss function is a hyperparameter that dictates how to weight the reconstruction and penalty terms. -->

*上の図5は、VAE損失が、どのようにして各点の推定分散を1前後に保ちながら、推定された潜在変数を重なり合わせずにできる限り近づけたかを示しています。*

**注：** VAE損失関数の$\beta$は、再構成項とペナルティの項を重みでバランスするハイパーパラメーターです。

<!-- ## [Implementation of Variational Autoencoder (VAE)](https://www.youtube.com/watch?v=7Rb4s9wNOmc&t=1893s)

The Jupyter notebook can be found [here](https://github.com/Atcold/pytorch-Deep-Learning/blob/master/11-VAE.ipynb).

In this notebook, we implement a VAE and train it on the MNIST dataset.
Then we sample$\boldsymbol{z}$from a normal distribution and feed to the decoder and compare the result. Finally, we look at how$\boldsymbol{z}$changes in 2D projection.

**Note:** In the MNIST dataset used, the pixel values have been normalized to be in range$[0, 1]$. -->

## [Variational Autoencoder（VAE）の実装](https://www.youtube.com/watch?v=7Rb4s9wNOmc&t=1893s)
このノートブックでは、VAEを実装し、MNISTデータセットで訓練します。
次に、正規分布から$\boldsymbol{z}$をサンプリングしてデコーダーにフィードし、結果を比較します。 最後に、2次元射影で$\boldsymbol{z}$がどのように変化するかを見ていきます。

**注：**使用されているMNISTデータセットでは、ピクセル値は$[0、1]$の範囲で正規化されています。

<!-- ### The Encoder and the Decoder

1. We define the encoder and decoder in our `VAE` module.
2. For the last linear layer of encoder, we define the output to be of size$2d$, of which the first$d$values are the means and the remaining$d$values are the variances. We sample$\boldsymbol{z} \in R^d$using these means and variances as explained in the reparameterisation trick before.
3. For the last linear layer in the decoder, we use the sigmoid activation so that we can have output in range$[0, 1]$, similar to the input data. -->

### エンコーダーとデコーダー

1. 「VAE」モジュールでエンコーダーとデコーダーを定義します。
2. エンコーダーの最後の線形層について、出力をサイズを$2d$とします。前半の$d$は平均であり、後半の$d$は分散です。前のreparameterisation trickで説明したように、これらの平均と分散を使用して$\boldsymbol{z} \in R^d$をサンプリングします。
3. デコーダーの最後の線形レイヤーでは、入力データと同様に、範囲$[0、1]$で出力できるように、シグモイド活性層を使用します。

```python
class VAE(nn.Module):
    def __init__(self):
        super().__init__()

        self.encoder = nn.Sequential(
            nn.Linear(784, d ** 2),
            nn.ReLU(),
            nn.Linear(d ** 2, d * 2)
        )

        self.decoder = nn.Sequential(
            nn.Linear(d, d ** 2),
            nn.ReLU(),
            nn.Linear(d ** 2, 784),
            nn.Sigmoid(),
        )
```


<!-- ### Reparameterisation and the `forward` function

For the `reparameterise` function, if the model is in training mode, we compute the standard deviation (`std`) from log variance (`logvar`). We use log variance instead of variance because we want to make sure the variance is non-negative, and taking the log of it ensures that we have the full range of the variance, which makes the training more stable.

During training, the `reparameterise` function will do the reparameterisation trick so that we can do backpropagation in training. To connect to the concept of a yellow bubble, as explained in the lecture, every time this function is called, we draw a point `eps = std.data.new(std.size()).normal_()`, so if we do 100 times, we will get 100 points which roughly forms a sphere because it's normal distribution, and the line `eps.mul(std).add_(mu)` will make this sphere centred in `mu` with radius equal to `std`.

For the `forward` function, we first compute the `mu` (first half) and `logvar` (second half) from the encoder, then we compute the$\boldsymbol{z}$via the `reparamterise` function. Finally, we return the output of the decoder.　-->

###　Reparameterisationと `forward`関数

`reparameterise`関数については、モデルを訓練するとき、対数分散（` logvar`）から標準偏差（ `std`）を計算します。分散が負ではないことを保証するために対数分散を使用します。その対数を取ると、分散の全範囲が確保され、訓練がより安定します。

訓練中、`reparameterise`関数がreparametrization trickを実行するため、逆伝播を実行することができます。講義で説明したように、上記に書いて黄色いバブルの概念に接続するために、この関数が呼び出されるたびに、`eps = std.data.new(std.size()).normal_()`で点を描画します。 正規分布であるため、100回実行すると100個の点で結成された大きい球が得られ、この線 `eps.mul（std）.add_（mu）`により、球は `mu`を中心とし、半径は`std`になります。

`forward`では、最初にエンコーダーから`mu`（前半）と `logvar`（後半）を計算し、次に`reparameterise`関数を介して$\boldsymbol{z}$を計算します。最後に、デコーダーの出力を返します。

```python
def reparameterise(self, mu, logvar):
    if self.training:
        std = logvar.mul(0.5).exp_()
        eps = std.data.new(std.size()).normal_()
        return eps.mul(std).add_(mu)
    else:
        return mu

def forward(self, x):
    mu_logvar = self.encoder(x.view(-1, 784)).view(-1, 2, d)
    mu = mu_logvar[:, 0, :]
    logvar = mu_logvar[:, 1, :]
    z = self.reparameterise(mu, logvar)
    return self.decoder(z), mu, logvar
```


<!-- ### Loss function for the VAE

Here we define the reconstruction loss (binary cross entropy) and the relative entropy (KL divergence penalty). -->

### VAEの損失関数

ここでは、再構成損失（バイナリ・クロス・エントロピー）と相対エントロピー（KLダイバージェンスペナルティ）を定義します。

```python
def loss_function(x_hat, x, mu, logvar):
    BCE = nn.functional.binary_cross_entropy(
        x_hat, x.view(-1, 784), reduction='sum'
    )
    KLD = 0.5 * torch.sum(logvar.exp() - logvar - 1 + mu.pow(2))

    return BCE + KLD
```


<!-- ### Generating new samples

After we train our model, we can sample a random$z$from the normal distribution and feed it to our decoder. We can observe in *Figure 6* that some of the results are not good because our decoder has not "covered" the whole latent space. This can be improved if we train the model for more epochs. -->

### 新しいサンプルの生成

モデルを訓練した後、正規分布からランダムな$z$をサンプリングし、それをデコーダーに供給します。 *図6*から、一部の結果が良くないことがわかります。それは、デコーダーが潜在空間全体を「カバー」していないためです。より多くのエポックでモデルを訓練すると、これを改善できます。

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_6.png" /><br>
<b>図6</b>: 潜在空間をランダムに移動する
</center>

<!-- We can look at how one digit morphs into another, which would not have been possible if we used an autoencoder. We can see that when we walk in the latent space, the output of the decoder still looks legit. *Figure 7* below shows how we morph the digit$3$to$8$. -->

ある数字が別の数字にどのように変化するか（オートエンコーダを使っていたたらこれはできなかったことです）について見てみましょう。潜在空間をみてみると、デコーダーの出力はまだちゃんとしていることがわかります。 以下の*図7*は、数字$3$を$8$に変換する方法を示しています。

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_7.png" /><br>
<b>図7</b>: 数字の3を8に変換する
</center>


<!-- ### Projection of means

Finally, let's take a look at how the latent space changes during/after training. The following charts in *Figure 8* are the means from the output of the encoder, projected on 2D space, where each colour represents a digit. We can see that from epoch 0, the classes are spreading everywhere, with only little concentration. As the model is trained, the latent space becomes more well-defined and the classes (digits) starts to form clusters. -->

### 平均の投射

最後に、訓練中/訓練後に潜在空間がどのように変化するかを見てみましょう。 *図8*の次のグラフは、2次元空間に射影されたエンコーダーの出力からの平均であり、各色は数字を表しています。エポック0では、クラスはほとんど集中せずにばらついていることがわかります。モデルが訓練されると、潜在空間がより明確になり、クラス（数字）がクラスターを形成し始めます。

<center>
<img src="{{site.baseurl}}/images/week08/08-3/fig_8.png" /><br>
<b>図8</b>: 潜在空間での平均$\E(\vect{z})$の2次元射影
</center>
