---
lang: fr
lang-ref: ch.13-3
title: Réseau convolutif pour graphe III
lecturer: Alfredo Canziani
authors: Go Inoue, Muhammad Osama Khan, Muhammad Shujaat Mirza, Muhammad Muneeb Afzal
date: 28 Apr 2020
translation-date: 24 Aug 2020
translator: Loïck Bourdois
---


<!--
## [Introduction to Graph Convolutional Network (GCN)](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=47s)

Graph Convolutional Network (GCN) is one type of architecture that utilizes the structure of data.
Before going into details, let's have a quick recap on self-attention, as GCN and self-attention are conceptually relevant.
-->

## [Introduction au réseau convolutif pour graphe](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=47s)

Le réseau convolutif pour graphe (GCN pour *Graph Convolutional Network*) est un type d'architecture qui utilise la structure des données.
Avant d'entrer dans les détails, faisons un rapide rappel sur l'auto-attention car le GCN et l'auto-attention sont conceptuellement pertinents.

<!--
### Recap: Self-attention

- In self-attention, we have a set of input $\lbrace\boldsymbol{x}\_{i}\rbrace^{t}_{i=1}$.
Unlike a sequence, it does not have an order.
- Hidden vector $\boldsymbol{h}$ is given by linear combination of the vectors in the set.
- We can express this as $\boldsymbol{X}\boldsymbol{a}$ using matrix vector multiplication, where $\boldsymbol{a}$ contains coefficients that scale the input vector $\boldsymbol{x}_{i}$.

*For a detailed explanation, refer to the notes of [Week 12]({{site.baseurl}}/en/week12/12-3/).*
-->

### Récapitulatif sur l'auto-attention

- Dans l'auto-attention, nous avons un ensemble d'entrées $\lbrace\boldsymbol{x}\_{i}\rbrace^{t}_{i=1}$.
Contrairement à une séquence, elle n'a pas d'ordre.
- Le vecteur caché $\boldsymbol{h}$ est donné par une combinaison linéaire des vecteurs de l'ensemble.
- Nous pouvons l'exprimer sous la forme $\boldsymbol{X}\boldsymbol{a}$ en utilisant une multiplication matricielle des vecteurs, où $\boldsymbol{a}$ contient des coefficients qui mettent à l'échelle le vecteur d'entrée $\boldsymbol{x}_{i}$.

Pour une explication détaillée, voir les notes de la [semaine 12]({{site.baseurl}}/fr/semaine12/12-3/).


<!--
### Notation

<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure1.png" height="400px" /><br>
<b>Fig. 1</b>: Graph Convolutional Network
</center>

In Figure 1, vertex $v$ is comprised of two vectors: input $\boldsymbol{x}$ and its hidden representation $\boldsymbol{h}$.
We also have multiple vertices $v_{j}$, which is comprised of $\boldsymbol{x}\_j$ and $\boldsymbol{h}\_j$.
In this graph, vertices are connected with directed edges.

We represent these directed edges with adjacency vector $\boldsymbol{a}$, where each element $\alpha_{j}$ is set to $1$ if there is a directed edge from $v_{j}$ to $v$.

$$
\alpha_{j} \stackrel{\tiny \downarrow}{=} 1 \Leftrightarrow v_{j} \rightarrow v
\tag{Eq. 1}
$$

The degree (number of incoming edges) $d$ is defined as the norm of this adjacency vector, *i.e.* $\Vert\boldsymbol{a}\Vert_{1} $, which is the number of ones in the vector $\boldsymbol{a}$.

$$
    d = \Vert\boldsymbol{a}\Vert_{1}
\tag{Eq. 2}
$$

The hidden vector $\boldsymbol{h}$ is given by the following expression:

$$
    \boldsymbol{h}=f(\boldsymbol{U}\boldsymbol{x} + \boldsymbol{V}\boldsymbol{X}\boldsymbol{a}d^{-1})
\tag{Eq. 3}
$$

where $f(\cdot)$ is a non-linear function such as ReLU $(\cdot)^{+}$, Sigmoid $\sigma(\cdot)$, and hyperbolic tangent $\tanh(\cdot)$.

The $\boldsymbol{U}\boldsymbol{x}$ term takes into account the vertex $v$ itself, by applying rotation $\boldsymbol{U}$ to the input $v$.

Remember that in self-attention, the hidden vector $\boldsymbol{h}$ is computed by $\boldsymbol{X}\boldsymbol{a}$, which means that the columns in $\boldsymbol{X}$ is scaled by the factors in $\boldsymbol{a}$.
In the context of GCN, this means that if we have multiple incoming edges,*i.e.* multiple ones in adjacency vector $\boldsymbol{a}$, $\boldsymbol{X}\boldsymbol{a}$ gets larger.
On the other hand, if we have only one incoming edge, this value gets smaller.
To remedy this issue of the value being proportionate to the number of incoming edges, we divide it by the number of incoming edges $d$.
We then apply rotation $\boldsymbol{V}$ to $\boldsymbol{X}\boldsymbol{a}d^{-1}$.

We can represent this hidden representation $\boldsymbol{h}$ for the entire set of inputs $\boldsymbol{x}$ using the following matrix notation:

$$
\{\boldsymbol{x}_{i}\}^{t}_{i=1}\rightsquigarrow \boldsymbol{H}=f(\boldsymbol{UX}+ \boldsymbol{VXAD}^{-1})
\tag{Eq. 4}
$$

where $\vect{D} = \text{diag}(d_{i})$.
-->


### Notation

<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure1.png" height="400px" /><br>
<b>Figure 1 :</b> ConvNet pour graphe
</center>

Dans la figure 1, le sommet $v$ est composé de deux vecteurs : l'entrée $\boldsymbol{x}$ et sa représentation cachée $\boldsymbol{h}$.
Nous avons également plusieurs sommets $v_{j}$, qui se composent de $\boldsymbol{x}\_j$ et de $\boldsymbol{h}\_j$.
Dans ce graphe, les sommets sont reliés par des arêtes dirigées.

Nous représentons ces arêtes dirigées avec le vecteur d'adjacence $\boldsymbol{a}$, où chaque élément $\alpha_{j}$ est fixé à $1$ s'il y a une arête dirigée de $v_{j}$ à $v$.


$$
\alpha_{j} \stackrel{\tiny \downarrow}{=} 1 \Leftrightarrow v_{j} \rightarrow v
\tag{Eq. 1}
$$

Le degré (nombre d'arêtes entrantes) $d$ est défini comme la norme de ce vecteur de'adjacence, c'est-à-dire $\Vert\boldsymbol{a}\Vert_{1} $, qui est le nombre de 1 dans le vecteur $\boldsymbol{a}$.

$$
    d = \Vert\boldsymbol{a}\Vert_{1}
\tag{Eq. 2}
$$

Le vecteur caché $\boldsymbol{h}$ est donné par l'expression suivante :

$$
    \boldsymbol{h}=f(\boldsymbol{U}\boldsymbol{x} + \boldsymbol{V}\boldsymbol{X}\boldsymbol{a}d^{-1})
\tag{Eq. 3}
$$


où $f(\cdot)$ est une fonction non linéaire telle que ReLU $(\cdot)^{+}$, Sigmoïde $\sigma(\cdot)$ et tangente hyperbolique $\tanh(\cdot)$.

Le terme $\boldsymbol{U}\boldsymbol{x}$ prend en compte le sommet $v$ lui-même en appliquant la rotation $\boldsymbol{U}$ à l'entrée $v$.

Dans l'auto-attention, le vecteur caché $\boldsymbol{h}$ est calculé par $\boldsymbol{X}\boldsymbol{a}$, ce qui signifie que les colonnes dans $\boldsymbol{X}$ sont mises à l'échelle par les facteurs dans $\boldsymbol{a}$.
Dans le contexte du GCN, cela signifie que si nous avons plusieurs arêtes entrantes, c'est-à-dire plusieurs dans le vecteur d'adjacence $\boldsymbol{a}$, $\boldsymbol{X}\boldsymbol{a}$ s'agrandit.
En revanche, si nous n'avons qu'une seule arête entrante, cette valeur devient plus petite.
Pour remédier à ce problème de proportionnalité de la valeur par rapport au nombre d’arêtes entrantes, nous la divisons par le nombre d’arêtes entrantes $d$.
Nous appliquons ensuite la rotation $\boldsymbol{V}$ à $\boldsymbol{X}\boldsymbol{a}d^{-1}$.

Nous pouvons représenter cette représentation cachée $\boldsymbol{h}$ pour l'ensemble des entrées $\boldsymbol{x}$ en utilisant la notation matricielle suivante :


$$
\{\boldsymbol{x}_{i}\}^{t}_{i=1}\rightsquigarrow \boldsymbol{H}=f(\boldsymbol{UX}+ \boldsymbol{VXAD}^{-1})
\tag{Eq. 4}
$$

où $\vect{D} = \text{diag}(d_{i})$.



<!--
## [Residual Gated GCN Theory and Code](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=992s)

Residual Gated Graph Convolutional Network is a type of GCN that can be represented as shown in Figure 2:

<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure2.png" height="300px" /><br>
<b>Fig. 2</b>: Residual Gated Graph Convolutional Network
</center>

As with the standard GCN, the vertex $v$ consists of two vectors: input $\boldsymbol{x}$ and its hidden representation $\boldsymbol{h}$. However, in this case, the edges also have a feature representation, where $\boldsymbol{e_{j}^{x}}$ represents the input edge representation and $\boldsymbol{e_{j}^{h}}$ represents the hidden edge representation.

The hidden representation $\boldsymbol{h}$ of the vertex $v$ is computed by the following equation:

$$
    \boldsymbol{h}=\boldsymbol{x} + \bigg(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}\bigg)^{+}
\tag{Eq. 5}
$$

where $\boldsymbol{x}$ is the input representation, $\boldsymbol{Ax}$ represents a rotation applied to the input $\boldsymbol{x}$ and $\sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}$ denotes the summation of elementwise multiplication of rotated incoming features $\boldsymbol{Bx_{j}}$ and a gate $\eta(\boldsymbol{e_{j}})$. In contrast to the standard GCN above where we average the incoming representations, the gate term is critical to the implementation of Residual Gated GCN since it allows us to modulate the incoming representations based on the edge representations.

Note that the summation is only over vertices ${v_j}$ that have incoming edges to vertex ${v}$. The term residual (in Residual Gated GCN) comes from the fact that in order to calculate the hidden representation $\boldsymbol{h}$, we add the input representation $\boldsymbol{x}$. The gate term $\eta(\boldsymbol{e_{j}})$ is calculated as shown below:

$$
    \eta(\boldsymbol{e_{j}})=\sigma(\boldsymbol{e_{j}})\bigg(\sum_{v_k→v}\sigma(\boldsymbol{e_{k}})\bigg)^{-1}
\tag{Eq. 6}
$$

The gate value $\eta(\boldsymbol{e_{j}})$ is a normalized sigmoid obtained by dividing the sigmoid of the incoming edge representation by the sum of sigmoids of all incoming edge representations. Note that in order to calculate the gate term, we need the representation of the edge $\boldsymbol{e_{j}}$, which can be computed using the equations below:

$$
    \boldsymbol{e_{j}} = \boldsymbol{Ce_{j}^{x}} + \boldsymbol{Dx_{j}} + \boldsymbol{Ex}
\tag{Eq. 7}
$$

$$
    \boldsymbol{e_{j}^{h}}=\boldsymbol{e_{j}^{x}}+(\boldsymbol{e_{j}})^{+}
\tag{Eq. 8}
$$

The hidden edge representation $\boldsymbol{e_{j}^{h}}$ is obtained by the summation of the initial edge representation $\boldsymbol{e_{j}^{x}}$ and $\texttt{ReLU}(\cdot)$ applied to $\boldsymbol{e_{j}}$ where $\boldsymbol{e_{j}}$ is in turn given by the summation of a rotation applied to $\boldsymbol{e_{j}^{x}}$, a rotation applied to the input representation $\boldsymbol{x_{j}}$ of the vertex $v_{j}$ and a rotation applied to the input representation $\boldsymbol{x}$ of the vertex $v$.

*Note: In order to calculate hidden representations downstream (*e.g.* $2^\text{nd}$ layer hidden representations), we can simply replace the input feature representations by the $1^\text{st}$ layer feature representations in the equations above.*
-->


## [Théorie et code du GCN résiduel à porte](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=992s)

Le GCN résiduel à porte (RG CGN pour *Residual Gated GCN*) est un type de GCN qui peut être représenté comme le montre la figure 2 :

<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure2.png" height="300px" /><br>
<b>Figure 2 :</b> GCN résiduel à porte </center>

Comme pour le GCN standard, le sommet $v$ est constitué de deux vecteurs : l'entrée $\boldsymbol{x}$ et sa représentation cachée $\boldsymbol{h}$. Toutefois, dans ce cas, les arêtes ont également une représentation de caractéristique, où $\boldsymbol{e_{j}^{x}}$ représente la représentation de l'arête d'entrée et $\boldsymbol{e_{j}^{h}}$ représente la représentation de l'arête cachée.

La représentation cachée $\boldsymbol{h}$ du sommet $v$ est calculée par l'équation suivante :

$$
    \boldsymbol{h}=\boldsymbol{x} + \bigg(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}\bigg)^{+}
\tag{Eq. 5}
$$

où $\boldsymbol{x}$ est la représentation de l'entrée, $\boldsymbol{Ax}$ représente une rotation appliquée à l'entrée $\boldsymbol{x}$ et $\sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}$ indique la somme des multiplications par éléments des éléments entrants ayant subi une rotation $\boldsymbol{Bx_{j}}$ et d'une porte $\eta(\boldsymbol{e_{j}})$. Contrairement au GCN standard ci-dessus où nous faisons la moyenne des représentations entrantes, le terme de porte est essentiel à la mise en œuvre du RG CGN car il nous permet de moduler les représentations des entrées en fonction des représentations des arêtes.

A noter que la sommation se fait uniquement sur les sommets ${v_j}$ qui ont des arêtes entrantes au sommet ${v}$. Le terme résiduel (dans *Residual Gated GCN*) vient du fait que pour calculer la représentation cachée $\boldsymbol{h}$, on ajoute la représentation d'entrée $\boldsymbol{x}$. Le terme de porte $\eta(\boldsymbol{e_{j}})$ est calculé comme indiqué ci-dessous :

$$
    \eta(\boldsymbol{e_{j}})=\sigma(\boldsymbol{e_{j}})\bigg(\sum_{v_k→v}\sigma(\boldsymbol{e_{k}})\bigg)^{-1}
\tag{Eq. 6}
$$

La valeur de porte $\eta(\boldsymbol{e_{j}})$ est une sigmoïde normalisée obtenue en divisant la sigmoïde de la représentation de l’arête entrante par la somme des sigmoïdes de toutes les représentations de l’arête entrante. A noter que pour calculer le terme de porte, nous avons besoin de la représentation de l’arête $\boldsymbol{e_{j}}$, qui peut être calculée à l'aide des équations ci-dessous :

$$
    \boldsymbol{e_{j}} = \boldsymbol{Ce_{j}^{x}} + \boldsymbol{Dx_{j}} + \boldsymbol{Ex}
\tag{Eq. 7}
$$

$$
    \boldsymbol{e_{j}^{h}}=\boldsymbol{e_{j}^{x}}+(\boldsymbol{e_{j}})^{+}
\tag{Eq. 8}
$$

La représentation des arêtes cachées $\boldsymbol{e_{j}^{h}}$ est obtenue par la sommation de la représentation initiale des arêtes $\boldsymbol{e_{j}^{x}}$ et $\texttt{ReLU}(\cdot)$ appliqué à $\boldsymbol{e_{j}}$ où $\boldsymbol{e_{j}}$ est à son tour donné par la somme d'une rotation appliquée à $\boldsymbol{e_{j}^{x}}$, une rotation appliquée à la représentation en entrée $\boldsymbol{x_{j}}$ du sommet $v_{j}$ et une rotation appliquée à la représentation en entrée $\boldsymbol{x}$ du sommet $v$.

Note : afin de calculer les représentations cachées en aval (par exemple la deuxième couche de représentations cachées), nous pouvons simplement remplacer les représentations des caractéristiques d'entrée par les représentations des caractéristiques de la couche première dans les équations au-dessus.


<!--
### Graph Classification and Residual Gated GCN Layer

In this section, we introduce the problem of graph classification and code up a Residual Gated GCN layer. In addition to the usual import statements, we add the following:

```python
os.environ['DGLBACKEND'] = 'pytorch'
import dgl
from dgl import DGLGraph
from dgl.data import MiniGCDataset
import networkx as nx
```

The first line tells DGL to use PyTorch as the backend. Deep Graph Library ([DGL](https://www.dgl.ai/)) provides various functionalities on graphs whereas networkx allows us to visualise the graphs.

In this notebook, the task is to classify a given graph structure into one of 8 graph types. The dataset obtained from `dgl.data.MiniGCDataset` yields some number of graphs (`num_graphs`) with nodes between `min_num_v` and `max_num_v`. Hence, all the graphs obtained do not have the same number of nodes/vertices.

*Note: In order to familiarize yourself with the basics of `DGLGraphs`, it is recommended to go through the short tutorial [here](https://docs.dgl.ai/api/python/graph.html).*

Having created the graphs, the next task is to add some signal to the domain. Features can be applied to nodes and edges of a `DGLGraph`. The features are represented as a dictionary of names (strings) and tensors (**fields**). `ndata` and `edata` are syntax sugar to access the feature data of all nodes and edges.

The following code snippet shows how the features are generated. Each node is assigned a value equal to the number of incident edges, whereas each edge is assigned the value 1.
```python
def create_artificial_features(dataset):
    for (graph, _) in dataset:
        graph.ndata['feat'] = graph.in_degrees().view(-1, 1).float()
        graph.edata['feat'] = torch.ones(graph.number_of_edges(), 1)
    return dataset
```

Training and testing datasets are created and features are assigned as shown below:

```python
trainset = MiniGCDataset(350, 10, 20)
testset = MiniGCDataset(100, 10, 20)

trainset = create_artificial_features(trainset)
testset = create_artificial_features(testset)
```

A sample graph from the trainset has the following representation. Here, we observe that the graph has 15 nodes and 45 edges and both the nodes and edges have a feature representation of shape `(1,)` as expected. Furthermore, the `0` signifies that this graph belongs to class 0.
```python
(DGLGraph(num_nodes=15, num_edges=45,
         ndata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}
         edata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}), 0)
```
-->


### Graph Classification et couche GCN résiduelle

Pour cette section et dans la suite, les codes se réfèrent au *notebook* Jupyter dont la version anglaise est disponible [ici](https://github.com/Atcold/pytorch-Deep-Learning/blob/master/16-gated_GCN.ipynb) et la version française [ici](https://github.com/lbourdois/pytorch-Deep-Learning-Notebooks-in-French/blob/master/16-gated_GCN.ipynb).

Dans cette section, nous introduisons le problème de la classification des graphes et codons une couche GCN résiduelle. En plus des déclarations d'importation habituelles, nous ajoutons ce qui suit :

```python
os.environ['DGLBACKEND'] = 'pytorch'
import dgl
from dgl import DGLGraph
from dgl.data import MiniGCDataset
import networkx as nx
```

La première ligne indique à Deep Graph Library ([DGL](https://www.dgl.ai/)) d'utiliser PyTorch comme backend. *DGL* fournit diverses fonctionnalités sur les graphes alors que *networkx* nous permet de visualiser les graphes.

Dans ce *notebook*, la tâche consiste à classer une structure de graphe donnée dans l'un des 8 types de graphes. Le jeu de données obtenu à partir de `dgl.data.MiniGCDataset` donne un certain nombre de graphes (`num_graphs`) avec des nœuds entre `min_num_v` et `max_num_v`. Par conséquent, tous les graphes obtenus n'ont pas le même nombre de nœuds/sommets.

Note : afin de vous familiariser avec les bases des `DGLGraphs`, il est recommandé de suivre le court tutoriel (en anglais) disponible [ici](https://docs.dgl.ai/api/python/graph.html).


Après avoir créé les graphes, la tâche suivante consiste à ajouter un signal au domaine. Des caractéristiques peuvent être appliquées aux nœuds et aux arêtes d'un `DGLGraph`. Les caractéristiques sont représentées par un dictionnaire de noms (chaînes de caractères) et de tenseurs (**champs**). Les `ndata` et `edata` sont des sucres syntaxiques permettant d'accéder aux données des caractéristiques de tous les nœuds et arêtes.

L'extrait de code suivant montre comment les caractéristiques sont générées. Chaque nœud se voit attribuer une valeur égale au nombre d’arêtes incidentes, tandis que chaque bord se voit attribuer la valeur 1.

```python
def create_artificial_features(dataset):
    for (graph, _) in dataset:
        graph.ndata['feat'] = graph.in_degrees().view(-1, 1).float()
        graph.edata['feat'] = torch.ones(graph.number_of_edges(), 1)
    return dataset
```

Des jeux de données d’entraînement et de test sont créés et des caractéristiques sont attribuées comme indiqué ci-dessous :

```python
trainset = MiniGCDataset(350, 10, 20)
testset = MiniGCDataset(100, 10, 20)

trainset = create_artificial_features(trainset)
testset = create_artificial_features(testset)
```

Un exemple de graphe issu du jeu d’entraînement a la représentation suivante. Ici, nous observons que le graphe a 15 nœuds et 45 arêtes et que les nœuds et les arêtes ont une représentation caractéristique de la forme `(1,)` comme prévu. De plus, le `0` signifie que ce graphe appartient à la classe 0.


```python
(DGLGraph(num_nodes=15, num_edges=45,
         ndata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}
         edata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}), 0)
```


<!--
### Note about DGL Message and Reduce Functions

>In DGL, the *message functions* are expressed as **Edge UDF**s (User Defined Functions). Edge UDFs take in a single argument `edges`. It has three members `src`, `dst`, and `data` for accessing source node features, destination node features, and edge features respectively.
The *reduce functions* are **Node UDF**s. Node UDFs have a single argument `nodes`, which has two members `data` and `mailbox`. `data` contains the node features and `mailbox` contains all incoming message features, stacked along the second dimension (hence the `dim=1` argument).
`update_all(message_func, reduce_func)` sends messages through all edges and updates all nodes.
-->

### Note sur le message DGL et les fonctions de réduction

Dans DGL, les *fonctions de message* sont exprimées sous la forme de **Edge UDF**s (*User Defined Functions* : fonctions définies par l'utilisateur). Les **Edge UDF**s prennent en compte un seul argument `edges` (arêtes). Il a trois membres `src`, `dst` et `data` pour accéder respectivement aux fonctions du nœud source, du nœud de destination et des fonctions des arêtes.
Les fonctions de réduction sont des **Node UDF**s. Les node UDFs ont un seul argument `nodes`, qui a deux membres `data` et `mailbox`. `data` contient les caractéristiques du nœud et `mailbox` contient toutes les caractéristiques des messages entrants, empilées le long de la seconde dimension (d'où l'argument `dim=1`).
`update_all(message_func, reduce_func)` envoie les messages par toutes les arêtes et met à jour tous les nœuds.


<!--
### [Gated Residual GCN Layer Implementation](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=2098s)

A Gated Residual GCN layer is implemented as shown in the code snippets below.

Firstly, all the rotations of the input features $\boldsymbol{Ax}$, $\boldsymbol{Bx_{j}}$, $\boldsymbol{Ce_{j}^{x}}$, $\boldsymbol{Dx_{j}}$ and $\boldsymbol{Ex}$ are computed by defining `nn.Linear` layers inside the `__init__` function and then forward propagating the input representations `h` and `e` through the linear layers inside the `forward` function.

```python
class GatedGCN_layer(nn.Module):

    def __init__(self, input_dim, output_dim):
        super().__init__()
        self.A = nn.Linear(input_dim, output_dim)
        self.B = nn.Linear(input_dim, output_dim)
        self.C = nn.Linear(input_dim, output_dim)
        self.D = nn.Linear(input_dim, output_dim)
        self.E = nn.Linear(input_dim, output_dim)
        self.bn_node_h = nn.BatchNorm1d(output_dim)
        self.bn_node_e = nn.BatchNorm1d(output_dim)
```

Secondly, we compute the edge representations. This is done inside the `message_func` function, which iterates over all the edges and computes the edge representations. Specifically, the line `e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']` computes *(Eq. 7)* from above. The `message_func` function ships `Bh_j` (which is $\boldsymbol{Bx_{j}}$ from (Eq. 5)) and `e_ij` (Eq. 7) through the edge into the destination node's mailbox.

```python
def message_func(self, edges):
    Bh_j = edges.src['Bh']
    # e_ij = Ce_ij + Dhi + Ehj
    e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']
    edges.data['e'] = e_ij
    return {'Bh_j' : Bh_j, 'e_ij' : e_ij}
```

Thirdly, the `reduce_func` function collects the shipped messages by the `message_func` function. After collecting the node data `Ah` and shipped data `Bh_j` and `e_ij` from the `mailbox`, the line `h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)` computes the hidden representation of each node as indicated in (Eq. 5). Note however, that this only represents the term $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ without the $\texttt{ReLU}(\cdot)$ and residual connection.

```python
def reduce_func(self, nodes):
    Ah_i = nodes.data['Ah']
    Bh_j = nodes.mailbox['Bh_j']
    e = nodes.mailbox['e_ij']
    # sigma_ij = sigmoid(e_ij)
    sigma_ij = torch.sigmoid(e)
    # hi = Ahi + sum_j eta_ij * Bhj
    h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)
    return {'h' : h}
```

Inside the `forward` function, having called `g.update_all`, we obtain the results of graph convolution `h` and `e`, which represent the terms $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ from (Eq.5) and $\boldsymbol{e_{j}}$ from (Eq. 7) respectively. Then, we normalize `h` and `e` with respect to the graph node size and graph edge size respectively. Batch normalization is then applied so that we can train the network effectively. Finally, we apply $\texttt{ReLU}(\cdot)$ and add the residual connections to obtain the hidden representations for the nodes and edges, which are then returned by the `forward` function.

```python
def forward(self, g, h, e, snorm_n, snorm_e):

    h_in = h # residual connection
    e_in = e # residual connection

    g.ndata['h']  = h
    g.ndata['Ah'] = self.A(h)
    g.ndata['Bh'] = self.B(h)
    g.ndata['Dh'] = self.D(h)
    g.ndata['Eh'] = self.E(h)
    g.edata['e']  = e
    g.edata['Ce'] = self.C(e)

    g.update_all(self.message_func, self.reduce_func)

    h = g.ndata['h'] # result of graph convolution
    e = g.edata['e'] # result of graph convolution

    h = h * snorm_n # normalize activation w.r.t. graph node size
    e = e * snorm_e # normalize activation w.r.t. graph edge size

    h = self.bn_node_h(h) # batch normalization
    e = self.bn_node_e(e) # batch normalization

    h = torch.relu(h) # non-linear activation
    e = torch.relu(e) # non-linear activation

    h = h_in + h # residual connection
    e = e_in + e # residual connection

    return h, e
```


Next, we define the `MLP_Layer` module which contains several fully connected layers (FCN). We create a list of fully connected layers and forward through the network.

Finally, we define our `GatedGCN` model which comprises of the previously defined classes: `GatedGCN_layer` and `MLP_layer`. The definition of our model (`GatedGCN`) is shown below.

 ```python
 class GatedGCN(nn.Module):
    def __init__(self, input_dim, hidden_dim, output_dim, L):
        super().__init__()
        self.embedding_h = nn.Linear(input_dim, hidden_dim)
        self.embedding_e = nn.Linear(1, hidden_dim)
        self.GatedGCN_layers = nn.ModuleList([
            GatedGCN_layer(hidden_dim, hidden_dim) for _ in range(L)
        ])
        self.MLP_layer = MLP_layer(hidden_dim, output_dim)
    def forward(self, g, h, e, snorm_n, snorm_e):
        # input embedding
        h = self.embedding_h(h)
        e = self.embedding_e(e)
        # graph convnet layers
        for GGCN_layer in self.GatedGCN_layers:
            h, e = GGCN_layer(g, h, e, snorm_n, snorm_e)
        # MLP classifier
        g.ndata['h'] = h
        y = dgl.mean_nodes(g,'h')
        y = self.MLP_layer(y)
        return y
 ```

In our constructor, we define the embeddings for `e` and `h` (`self.embedding_e ` and `self.embedding_h`), `self.GatedGCN_layers` which is list (of size $L$) of our previously defined model: `GatedGCN_layer`, and finally `self.MLP_layer` which was also defined before. Next, using these initializations, we simply foward through the model and output `y`.

To better understand the model, we initiate an object of the model and print it for better visualization:

```python
# instantiate network
model = GatedGCN(input_dim=1, hidden_dim=100, output_dim=8, L=2)
print(model)
```

The main structure of the model is shown below:

```python
GatedGCN(
  (embedding_h): Linear(in_features=1, out_features=100, bias=True)
  (embedding_e): Linear(in_features=1, out_features=100, bias=True)
  (GatedGCN_layers): ModuleList(
    (0): GatedGCN_layer(...)
    (1): GatedGCN_layer(... ))
  (MLP_layer): MLP_layer(
    (FC_layers): ModuleList(...))
```

Not surprisingly, we have two layers of `GatedGCN_layer` (since `L=2`) followed by a `MLP_layer` which finally yields an output of 8 values.

Moving on, we define our `train` and `evaluate` functions. In our `train` function, we have our generic code which takes samples from `dataloader`.  Next, `batch_graphs`, `batch_x`, `batch_e`, `batch_snorm_n` and `batch_snorm_e` are fed into our model which returns `batch_scores` (of size 8). The predicted scores are compared with the ground truth in our loss function: `loss(batch_scores, batch_labels)`. Then, we zero out gradients (`optimizer.zero_grad()`), perform backprop (`J.backward()`) and updates our weights (`optimizer.step()`). Finally, loss for epoch and training accuracy is calculated. Furthermore, we use a similar code for our `evaluate` function.

Finally, we are ready to train! We found that after 40 epochs of training, our model has learned to classify the graphs with a test accuracy of $87$%.
-->



### [Implémentation de la couche GR GCN](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=2098s)

Une couche GR GCN est implémentée comme indiqué dans les extraits de code ci-dessous.

Premièrement, toutes les rotations des caractéristiques d'entrée $\boldsymbol{Ax}$, $\boldsymbol{Bx_{j}}$, $\boldsymbol{Ce_{j}^{x}}$, $\boldsymbol{Dx_{j}}$ et $\boldsymbol{Ex}$ sont calculées en définissant `nn.Linear` couches à l'intérieur de la fonction `__init__` puis en propageant les représentations d'entrée `h` et `e` à travers les couches linéaires à l'intérieur de la fonction `forward`.

```python
class GatedGCN_layer(nn.Module):

    def __init__(self, input_dim, output_dim):
        super().__init__()
        self.A = nn.Linear(input_dim, output_dim)
        self.B = nn.Linear(input_dim, output_dim)
        self.C = nn.Linear(input_dim, output_dim)
        self.D = nn.Linear(input_dim, output_dim)
        self.E = nn.Linear(input_dim, output_dim)
        self.bn_node_h = nn.BatchNorm1d(output_dim)
        self.bn_node_e = nn.BatchNorm1d(output_dim)
```

Deuxièmement, nous calculons les représentations de l’arête. Ceci est fait à l'intérieur de la fonction `message_func`, qui itère sur toutes les arêtes et calcule leurs représentations. Plus précisément, la ligne `e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']` calcule *(Eq. 7)* à partir du haut. La fonction `message_func` envoie `Bh_j` (qui est $\boldsymbol{Bx_{j}}$ de (Eq. 5)) et `e_ij` (Eq. 7) par le bord dans la boîte aux lettres du noeud de destination.


```python
def message_func(self, edges):
    Bh_j = edges.src['Bh']
    # e_ij = Ce_ij + Dhi + Ehj
    e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']
    edges.data['e'] = e_ij
    return {'Bh_j' : Bh_j, 'e_ij' : e_ij}
```

Troisièmement, la fonction `reduce_func` collecte les messages envoyés par la fonction `message_func`. Après avoir collecté les données du nœud `Ah` et les données expédiées `Bh_j` et `e_ij` de la `mailbox`, la ligne `h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)` calcule la représentation cachée de chaque nœud comme indiqué dans (Eq. 5). A noter cependant que cela ne représente que le terme $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ sans le $\texttt{ReLU}(\cdot)$ et la connexion résiduelle.

```python
def reduce_func(self, nodes) :
    Ah_i = nodes.data [Ah]
    Bh_j = nodes.mailbox['Bh_j']
    e = nœuds.boîte aux lettres [e_ij]
    # sigma_ij = sigmoid(e_ij)
    sigma_ij = torche.sigmoid(e)
    # hi = Ahi + sum_j eta_ij * Bhj
    h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)
    return {'h' : h}
```

A l'intérieur de la fonction `forward`, ayant appelé `g.update_all`, nous obtenons les résultats de la convolution des graphes `h` et `e`, qui représentent les termes $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ de (Eq.5) et $\boldsymbol{e_{j}}$ de (Eq. 7) respectivement. Ensuite, nous normalisons `h` et `e` par rapport à la taille du nœud et de l’arête du graphe respectivement. La *batch normalisation* est ensuite appliquée afin que nous puissions entraîner le réseau efficacement. Enfin, nous appliquons $\texttt{ReLU}(\cdot)$ et ajoutons les connexions résiduelles pour obtenir les représentations cachées des nœuds et des arêtes, qui sont ensuite renvoyées par la fonction `forward`.

```python
def forward(self, g, h, e, snorm_n, snorm_e) :

    h_in = h # connexion résiduelle
    e_in = e # connexion résiduelle

    g.ndata [h] = h
    g.ndata ["Ah"] = self.A(h)
    g.ndata [Bh] = self.B(h)
    g.ndata [Dh] = self.D(h)
    g.ndata ["Eh"] = self.E(h)
    g.edata ["e"] = e
    g.edata ["Ce"] = self.C(e)

    g.update_all(self.message_func, self.reduce_func)

    h = g.ndata [h] # résultat de la convolution du graphe
    e = g.edata['e'] # résultat de la convolution du graphe

    h = h * snorm_n # normalisation de l'activation par rapport à la taille des noeuds du graphe
    e = e * snorm_e # normalisation de l'activation par rapport à la taille des arêtes du graphe

    h = self.bn_node_h(h) # batch normalisation
    e = self.bn_node_e(e) # batch normalisation

    h = torch.relu(h) # activation non linéaire
    e = torch.relu(e) # activation non linéaire

    h = h_in + h # connexion résiduelle
    e = e_in + e # connexion résiduelle

     return h, e
```

Ensuite, nous définissons le module `MLP_Layer` qui contient plusieurs couches entièrement connectées. Nous créons une liste de couches entièrement connectées et nous les transmettons à travers le réseau.

Enfin, nous définissons notre modèle `GatedGCN` qui comprend les classes définies précédemment : `GatedGCN_layer` et `MLP_layer`. La définition de notre modèle (`GatedGCN`) est présentée ci-dessous.

```python
 class GatedGCN(nn.Module):
    def __init__(self, input_dim, hidden_dim, output_dim, L):
        super().__init__()
        self.embedding_h = nn.Linear(input_dim, hidden_dim)
        self.embedding_e = nn.Linear(1, hidden_dim)
        self.GatedGCN_layers = nn.ModuleList([
            GatedGCN_layer(hidden_dim, hidden_dim) for _ in range(L)
        ])
        self.MLP_layer = MLP_layer(hidden_dim, output_dim)
    def forward(self, g, h, e, snorm_n, snorm_e):
        # enchâssement en entrée
        h = self.embedding_h(h)
        e = self.embedding_e(e)
        # couche GCN
        for GGCN_layer in self.GatedGCN_layers:
            h, e = GGCN_layer(g, h, e, snorm_n, snorm_e)
        # classifieur MLP
        g.ndata['h'] = h
        y = dgl.mean_nodes(g,'h')
        y = self.MLP_layer(y)
        return y
 ```

Dans notre constructeur, nous définissons les enchâssements pour `e` et `h` (`self.embedding_e` et `self.embedding_h`), `self.GatedGCN_layers` qui est la liste (de taille $L$) de notre modèle précédemment défini : `GatedGCN_layer`, et enfin `self.MLP_layer` qui a également été défini auparavant. Ensuite, en utilisant ces initialisations, nous faisons simplement avancer le modèle et nous obtenons un `y`.

Pour mieux comprendre le modèle, nous initialisons un objet du modèle et l’affichons pour une meilleure visualisation :


```python
model = GatedGCN(input_dim=1, hidden_dim=100, output_dim=8, L=2)
print(model)
```
La structure principale du modèle est présentée ci-dessous :

```python
GatedGCN(
  (embedding_h): Linear(in_features=1, out_features=100, bias=True)
  (embedding_e): Linear(in_features=1, out_features=100, bias=True)
  (GatedGCN_layers): ModuleList(
    (0): GatedGCN_layer(...)
    (1): GatedGCN_layer(... ))
  (MLP_layer): MLP_layer(
    (FC_layers): ModuleList(...))
```


Sans surprise, nous avons deux couches de `GatedGCN_layer` (puisque `L=2`) suivies d'une `MLP_layer` qui donne finalement une sortie de 8 valeurs.

Ensuite, nous définissons nos fonctions `train` et `evaluate`. Dans notre fonction `train`, nous avons notre code générique qui prend des échantillons dans le `dataloader`.  Ensuite, les `batch_graphs`, `batch_x`, `batch_e`, `batch_snorm_n` et `batch_snorm_e` sont introduits dans notre modèle qui retourne des `batch_scores` (de taille 8). Les scores prédits sont comparés à la vérité de base dans notre fonction de perte : `loss(batch_scores, batch_labels)`. Ensuite, nous mettons à zéro les gradients (`optimizer.zero_grad()`), nous effectuons une rétropropagation (`J.backward()`) et nous mettons à jour nos poids (`optimizer.step()`). Enfin, la perte pour l'époque et la précision de l'entraînement est calculée. Nous utilisons un code similaire pour notre fonction `evaluate`.

Enfin, nous sommes prêts à pour l’entraînement ! Nous pouvons constater qu'après $40$ époques d'entraînement, notre modèle a appris à classer les graphes avec une précision de test de $87$%.
