---
lang: fr
lang-ref: ch.12-3
title: Attention et Transformer
lecturer: Alfredo Canziani
authors: Francesca Guiso, Annika Brundyn, Noah Kasmanoff, and Luke Martin
date: 21 Apr 2020
translation-date: 12 Aug 2020
translator: Loïck Bourdois
---


<!--
## [Attention](https://www.youtube.com/watch?v=f01J0Dri-6k&t=69s)

We introduce the concept of attention before talking about the Transformer architecture. There are two main types of attention: self attention *vs.* cross attention, within those categories, we can have hard *vs.* soft attention.

As we will later see, transformers are made up of attention modules, which are mappings between sets, rather than sequences, which means we do not impose an ordering to our inputs/outputs.
-->

## [Attention](https://www.youtube.com/watch?v=f01J0Dri-6k&t=69s)

Nous introduisons le concept d'attention avant de parler de l'architecture du Transformer. Il existe deux types principaux d'attention : l'auto-attention vs l'attention croisée. Dans ces catégories, nous pouvons avoir une attention dure vs une attention douce.

Comme nous le verrons plus tard, les *transformers* sont constitués de modules d'attention, qui sont des associations entre des ensembles (plutôt que des séquences), ce qui signifie que nous n'imposons pas d'ordre à nos entrées/sorties.



<!--
### Self Attention (I)

Consider a set of $t$ input $\boldsymbol{x}$'s:

$$
\lbrace\boldsymbol{x}_i\rbrace_{i=1}^t = \lbrace\boldsymbol{x}_1,\cdots,\boldsymbol{x}_t\rbrace
$$

where each $\boldsymbol{x}_i$ is an $n$-dimensional vector. Since the set has $t$ elements, each of which belongs to $\mathbb{R}^n$, we can represent the set as a matrix $\boldsymbol{X}\in\mathbb{R}^{n \times t}$.

With self-attention, the hidden representation $h$ is a linear combination of the inputs:

$$
\boldsymbol{h} = \alpha_1 \boldsymbol{x}_1 + \alpha_2 \boldsymbol{x}_2 + \cdots +  \alpha_t \boldsymbol{x}_t
$$

Using the matrix representation described above, we can write the hidden layer as the matrix product:

$$
\boldsymbol{h} = \boldsymbol{X} \boldsymbol{a}
$$

where $\boldsymbol{a} \in \mathbb{R}^t$ is a column vector with components $\alpha_i$.

Note that this differs from the hidden representation we have seen so far, where the inputs are multiplied by a matrix of weights.

Depending on the constraints we impose on the vector $\vect{a}$, we can achieve hard or soft attention.
-->

### Auto-attention (I)

Considérons un ensemble de $t$ d'entrée $\boldsymbol{x}$'s :

$$
\lbrace\boldsymbol{x}_i\rbrace_{i=1}^t = \lbrace\boldsymbol{x}_1,\cdots,\boldsymbol{x}_t\rbrace
$$

où chaque symbole $\boldsymbol{x}_i$ est un vecteur dimensionnel $n$. Comme l'ensemble comporte des éléments $t$, dont chacun appartient à $\mathbb{R}^n$, nous pouvons représenter l'ensemble comme une matrice $\boldsymbol{X}\in\mathbb{R}^{n \times t}$.

Avec l'auto-attention, la représentation cachée $h$ est une combinaison linéaire des entrées :

$$
\boldsymbol{h} = \alpha_1 \boldsymbol{x}_1 + \alpha_2 \boldsymbol{x}_2 + \cdots + \alpha_t \boldsymbol{x}_t
$$

En utilisant la représentation matricielle décrite ci-dessus, nous pouvons écrire la couche cachée comme le produit matriciel :

$$
\boldsymbol{h} = \boldsymbol{X} \boldsymbol{a}
$$

où $\boldsymbol{a} \in \mathbb{R}^t$ est un vecteur colonne avec les composantes $\alpha_i$.

A noter que cela diffère de la représentation cachée que nous avons vue jusqu'à présent, où les entrées sont multipliées par une matrice de poids.

Selon les contraintes que nous imposons au vecteur $\vect{a}$, nous pouvons obtenir une attention dure ou douce.


<!--
#### Hard Attention

With hard-attention, we impose the following constraint on the alphas: $\Vert\vect{a}\Vert_0 = 1$. This means $\vect{a}$ is a one-hot vector. Therefore, all but one of the coefficients in the linear combination of the inputs equals zero, and the hidden representation reduces to the input $\boldsymbol{x}_i$ corresponding to the element $\alpha_i=1$.
-->

#### Attention dure

Avec beaucoup d'attention, nous imposons aux alphas la contrainte suivante : $\Vert\vect{a}\Vert_0 = 1$. Cela signifie que $\vect{a}$ est un vecteur *one-hot*. Par conséquent, tous les coefficients de la combinaison linéaire des entrées sauf un sont égaux à $0$ et la représentation cachée se réduit à l'entrée $\boldsymbol{x}_i$ correspondant à l'élément $\alpha_i=1$.

<!--
#### Soft Attention

With soft attention, we impose that $\Vert\vect{a}\Vert_1 = 1$. The hidden representations is a linear combination of the inputs where the coefficients sum up to 1.
-->

#### Attention douce

Avec une attention douce, nous imposons que $\Vert\vect{a}\Vert_1 = 1$. La représentation cachée est une combinaison linéaire des entrées où les coefficients s'additionnent à 1.

<!--
### Self Attention (II)

Where do the $\alpha_i$ come from?

We obtain the vector $\vect{a} \in \mathbb{R}^t$ in the following way:

$$
\vect{a} = \text{[soft](arg)max}_{\beta} (\boldsymbol{X}^{\top}\boldsymbol{x})
$$

Where $\beta$ represents the inverse temperature parameter of the $\text{soft(arg)max}(\cdot)$. $\boldsymbol{X}^{\top}\in\mathbb{R}^{t \times n}$ is the transposed matrix representation of the set $\lbrace\boldsymbol{x}_i \rbrace\_{i=1}^t$, and $\boldsymbol{x}$ represents a generic $\boldsymbol{x}_i$ from the set. Note that the $j$ème row of $X^{\top}$ corresponds to an element $\boldsymbol{x}_j\in\mathbb{R}^n$, so the $j$ème row of $\boldsymbol{X}^{\top}\boldsymbol{x}$ is the scalar product of $\boldsymbol{x}_j$ with each $\boldsymbol{x}_i$ in $\lbrace \boldsymbol{x}_i \rbrace\_{i=1}^t$.

The components of the vector $\vect{a}$ are also called "scores" because the scalar product between two vectors tells us how aligned or similar two vectors are. Therefore, the elements of $\vect{a}$ provide information about the similarity of the overall set to a particular $\boldsymbol{x}_i$.

The square brackets represent an optional argument. Note that if $\arg\max(\cdot)$ is used, we get a one-hot vector of alphas, resulting in hard attention. On the other hand, $\text{soft(arg)max}(\cdot)$ leads to soft attention. In each case, the components of the resulting vector $\vect{a}$ sum to 1.

Generating $\vect{a}$ this way gives a set of them, one for each $\boldsymbol{x}_i$. Moreover, each $\vect{a}_i \in \mathbb{R}^t$ so we can stack the alphas in a matrix $\boldsymbol{A}\in \mathbb{R}^{t \times t}$.

Since each hidden state is a linear combination of the inputs $\boldsymbol{X}$ and a vector $\vect{a}$, we obtain a set of $t$ hidden states, which we can stack into a matrix $\boldsymbol{H}\in \mathbb{R}^{n \times t}$.

$$
\boldsymbol{H}=\boldsymbol{XA}
$$
-->

### Auto-attention (II)

D'où viennent les $\alpha_i$ ?

Nous obtenons le vecteur $\vect{a} \in \mathbb{R}^t$ de la manière suivante :

$$
\vect{a} = \text{[soft](arg)max}_{\beta} (\boldsymbol{X}^{\top}\boldsymbol{x})
$$

où $\beta$ représente le paramètre de température inverse du $\text{soft(arg)max}(\cdot)$. $\boldsymbol{X}^{\top}\in\mathbb{R}^{t \times n}$ est la représentation matricielle transposée de l'ensemble $\lbrace\boldsymbol{x}_i \rbrace\_{i=1}^t$, et $\boldsymbol{x}$ représente un $\boldsymbol{x}_i$ générique de l'ensemble. Notez que la ligne $j$ème de $X^{\top}$ correspond à un élément $\boldsymbol{x}_j\in\mathbb{R}^n$, Ainsi, la $j$ème rangée de $\boldsymbol{X}^{\top}\boldsymbol{x}$ est le produit scalaire de $\boldsymbol{x}_j$ avec chaque $\boldsymbol{x}_i$ dans $\lbrace \boldsymbol{x}_i \rbrace\_{i=1}^t$.

Les composantes du vecteur $\vect{a}$ sont également appelées *scores* car le produit scalaire entre deux vecteurs nous indique dans quelle mesure deux vecteurs sont alignés ou similaires. Par conséquent, les éléments de $\vect{a}$ fournissent des informations sur la similarité de l'ensemble global avec un symbole particulier $\boldsymbol{x}_i$.

Les crochets représentent un argument facultatif. A noter que si $\arg\max(\cdot)$ est utilisé, nous obtenons un vecteur *one-hot* d'alphas, ce qui entraîne une attention soutenue. En revanche, $\text{soft(arg)max}(\cdot)$ conduit à une attention douce. Dans chaque cas, les composantes du vecteur résultant $\vect{a}$ s'additionnent à 1.

En générant $\vect{a}$ de cette façon, on obtient un ensemble d'entre elles, une pour chaque symbole $\boldsymbol{x}_i$. De plus, chaque $\vect{a}_i \in \mathbb{R}^t$ permet d'empiler les alphas dans une matrice $\boldsymbol{A}\in \mathbb{R}^{t \times t}$.

Comme chaque état caché est une combinaison linéaire des entrées $\boldsymbol{X}$ et d'un vecteur $\vect{a}$, nous obtenons un ensemble d'états cachés $t$, que nous pouvons empiler dans une matrice $\boldsymbol{H}\in \mathbb{R}^{n \times t}$.

$$
\boldsymbol{H}=\boldsymbol{XA}
$$

<!--
## [Key-value store](https://www.youtube.com/watch?v=f01J0Dri-6k&t=1056s)

A key-value store is a paradigm designed for storing (saving), retrieving (querying) and managing associative arrays (dictionaries / hash tables).

For example, say we wanted to find a recipe to make lasagne. We have a recipe book and search for "lasagne" - this is the query. This query is checked against all possible keys in your dataset - in this case, this could be the titles of all the recipes in the book. We check how aligned the query is with each title to find the maximum matching score between the query and all the respective keys. If our output is the argmax function - we retrieve the single recipe with the highest score. Otherwise, if we use a soft argmax function, we would get a probability distribution and can retrieve in order from the most similar content to less and less relevant recipes matching the query.

Basically, the query is the question. Given one query, we check this query against every key and retrieve all matching content.
-->

## [Key-value store](https://www.youtube.com/watch?v=f01J0Dri-6k&t=1056s)

Un *key-value store* est un paradigme conçu pour le stockage (sauvegarde), la récupération (interrogation) et la gestion de tableaux associatifs (dictionnaires / tables de hachage).

Par exemple, disons que nous voulons trouver une recette pour faire des lasagnes. Nous avons un livre de recettes et nous cherchons « lasagne » : c'est la requête. Cette requête est comparée à toutes les clés possibles de votre ensemble de données : dans ce cas, il peut s'agir des titres de toutes les recettes du livre. Nous vérifions l'alignement de la requête avec chaque titre pour trouver le score maximum de correspondance entre la requête et toutes les clés respectives. Si notre résultat est la fonction argmax, nous récupérons la recette unique ayant obtenu le score le plus élevé. Sinon, si nous utilisons une fonction argmax souple, nous obtenons une distribution de probabilité et pouvons récupérer dans l'ordre, à partir du contenu le plus similaire, les recettes de moins en moins pertinentes qui correspondent à la requête.

Fondamentalement, la requête est la question. Pour une même question, nous comparons cette requête à chaque clé et nous récupérons tout le contenu correspondant.

<!--
### Queries, keys and values

$$
\begin{aligned}
\vect{q} &= \vect{W_q x} \\
\vect{k} &= \vect{W_k x} \\
\vect{v} &= \vect{W_v x}
\end{aligned}
$$

Each of the vectors $\vect{q}, \vect{k}, \vect{v}$ can simply be viewed as rotations of the specific input $\vect{x}$. Where $\vect{q}$ is just $\vect{x}$ rotated by $\vect{W_q}$, $\vect{k}$ is just $\vect{x}$ rotated by $\vect{W_k}$ and similarly for $\vect{v}$. Note that this is the first time we are introducing "learnable" parameters. We also do not include any non-linearities since attention is completely based on orientation.

In order to compare the query against all possible keys, $\vect{q}$ and $\vect{k}$ must have the same dimensionality, *i.e.* $\vect{q}, \vect{k} \in \mathbb{R}^{d'}$.

However, $\vect{v}$ can be of any dimension. If we continue with our lasagne recipe example - we need the query to have the dimension as the keys, *i.e.* the titles of the different recipes that we're searching through. The dimension of the corresponding recipe retrieved, $\vect{v}$, can be arbitrarily long though. So we have that $\vect{v} \in \mathbb{R}^{d''}$.

For simplicity, here we will make the assumption that everything has dimension $d$, i.e.

$$
d' = d'' = d
$$

So now we have a set of $\vect{x}$'s, a set of queries, a set of keys and a set of values. We can stack these sets into matrices each with $t$ columns since we stacked $t$ vectors; each vector has height $d$.

$$
\{ \vect{x}_i \}_{i=1}^t \rightsquigarrow \{ \vect{q}_i \}_{i=1}^t, \, \{ \vect{k}_i \}_{i=1}^t, \, \, \{ \vect{v}_i \}_{i=1}^t \rightsquigarrow \vect{Q}, \vect{K}, \vect{V} \in \mathbb{R}^{d \times t}
$$

We compare one query $\vect{q}$ against the matrix of all keys $\vect{K}$:

$$
\vect{a} = \text{[soft](arg)max}_{\beta} (\vect{K}^{\top} \vect{q}) \in \mathbb{R}^t
$$

Then the hidden layer is going to be the linear combination of the columns of $\vect{V}$ weighted by the coefficients in $\vect{a}$:

$$
\vect{h} = \vect{V} \vect{a} \in \mathbb{R}^d
$$

Since we have $t$ queries, we'll get $t$ corresponding $\vect{a}$ weights and therefore a matrix $\vect{A}$ of dimension $t \times t$.

$$
\{ \vect{q}_i \}_{i=1}^t \rightsquigarrow \{ \vect{a}_i \}_{i=1}^t, \rightsquigarrow \vect{A} \in \mathbb{R}^{t \times t}
$$

Therefore in matrix notation we have:

$$
\vect{H} = \vect{VA} \in \mathbb{R}^{d \times t}
$$

As an aside, we typically set $\beta$ to:

$$
\beta = \frac{1}{\sqrt{d}}
$$

This is done to keep the temperature constant across different choices of dimension $d$ and so we divide by the square root of the number of dimensions $d$. (Think what is the length of the vector $\vect{1} \in \R^d$.)

For implementation, we can speed up computation by stacking all the $\vect{W}$'s into one tall $\vect{W}$ and then calculate $\vect{q}, \vect{k}, \vect{v}$ in one go:

$$
\begin{bmatrix}
\vect{q} \\
\vect{k} \\
\vect{v}
\end{bmatrix} =
\begin{bmatrix}
\vect{W_q} \\
\vect{W_k} \\
\vect{W_v}
\end{bmatrix} \vect{x} \in \mathbb{R}^{3d}
$$

There is also the concept of "heads". Above we have seen an example with one head but we could have multiple heads. For example, say we have $h$ heads, then we have $h$ $\vect{q}$'s, $h$ $\vect{k}$'s and $h$ $\vect{v}$'s and we end up with a vector in $\mathbb{R}^{3hd}$:

$$
\begin{bmatrix}
\vect{q}^1 \\
\vect{q}^2 \\
\vdots \\
\vect{q}^h \\
\vect{k}^1 \\
\vect{k}^2 \\
\vdots \\
\vect{k}^h \\
\vect{v}^1 \\
\vect{v}^2 \\
\vdots \\
\vect{v}^h
\end{bmatrix} =
\begin{bmatrix}
\vect{W_q}^1 \\
\vect{W_q}^2 \\
\vdots \\
\vect{W_q}^h \\
\vect{W_k}^1 \\
\vect{W_k}^2 \\
\vdots \\
\vect{W_k}^h \\
\vect{W_v}^1 \\
\vect{W_v}^2 \\
\vdots \\
\vect{W_v}^h
\end{bmatrix} \vect{x} \in \R^{3hd}
$$

However, we can still transform the multi-headed values to have the original dimension $\R^d$ by using a $\vect{W_h} \in \mathbb{R}^{d \times hd}$. This is just one possible way to implement the key-value store.
-->

### Requêtes (q), clés (k) et valeurs (v)

$$
\begin{aligned}
\vect{q} &= \vect{W_q x} \\
\vect{k} &= \vect{W_k x} \\
\vect{v} &= \vect{W_v x}
\end{aligned}
$$

Chacun des vecteurs $\vect{q}, \vect{k}, \vect{v}$ peut être simplement considéré comme une rotation de l'entrée spécifique $\vect{x}$. Où $\vect{q}$ est juste $\vect{x}$ tourné par $\vect{W_q}$, $\vect{k}$ est juste $\vect{x}$ tourné par $\vect{W_k}$ et de même pour $\vect{v}$. Notons que c'est la première fois que nous introduisons des paramètres apprenables. Nous n'incluons pas non plus de non-linéarités puisque l'attention est entièrement basée sur l'orientation.

Afin de comparer la requête à toutes les clés possibles, $\vect{q}$ et $\vect{k}$ doivent avoir la même dimensionnalité, c'est-à-dire $\vect{q}, \vect{k} \in \mathbb{R}^{d'}$.

Cependant, $\vect{v}$ peut être de n'importe quelle dimension. Si nous continuons avec notre exemple de recette de lasagnes, nous avons besoin que la requête ait la dimension comme clés, c'est-à-dire les titres des différentes recettes que nous recherchons. La dimension de la recette correspondante $\vect{v}$, peut cependant être arbitrairement longue. Nous avons donc ce $\vect{v} \in \mathbb{R}^{d''}$.

Par souci de simplicité, nous supposerons ici que tout est de dimension $d$, c'est-à-dire :

$$
d' = d'' = d
$$

Nous avons donc maintenant un ensemble de $\vect{x}$, un ensemble de requêtes, un ensemble de clés et un ensemble de valeurs. Nous pouvons empiler ces ensembles dans des matrices dont chacune comporte des colonnes $t$ puisque nous avons empilé $t$ vecteurs. Chaque vecteur a une hauteur $d$.

$$
\{ \vect{x}_i \}_{i=1}^t \rightsquigarrow \{ \vect{q}_i \}_{i=1}^t, \, \{ \vect{k}_i \}_{i=1}^t, \, \, \{ \vect{v}_i \}_{i=1}^t \rightsquigarrow \vect{Q}, \vect{K}, \vect{V} \in \mathbb{R}^{d \times t}
$$

Nous comparons une requête $\vect{q}$ à la matrice de toutes les clés $\vect{K}$ :

$$
\vect{a} = \text{[soft](arg)max}_{\beta} (\vect{K}^{\top} \vect{q}) \in \mathbb{R}^t
$$

La couche cachée est alors la combinaison linéaire des colonnes de $\vect{V}$ pondérées par les coefficients $\vect{a}$ :

$$
\vect{h} = \vect{V} \vect{a} \in \mathbb{R}^d
$$

Puisque nous avons des requêtes $t$, nous obtenons $t$ correspondant aux $\vect{a}$ poids et donc une matrice $\vect{A}$ de dimension $t \times t$.

$$
\{ \vect{q}_i \}_{i=1}^t \rightsquigarrow \{ \vect{a}_i \}_{i=1}^t, \rightsquigarrow \vect{A} \in \mathbb{R}^{t \times t}
$$


C'est pourquoi, en notation matricielle, nous avons :

$$
\vect{H} = \vect{VA} \in \mathbb{R}^{d \times t}
$$
  

A part, nous mettons généralement $\beta$ à :

$$
\beta = \frac{1}{\sqrt{d}}
$$

Ceci est fait pour maintenir la température constante parmi les différents choix de dimension $d$ et nous divisons donc par la racine carrée du nombre de dimensions $d$.

Pour l'implémentation, nous pouvons accélérer le calcul en empilant tous les $\vect{W}$ en un seul grand $\vect{W}$ et calculer ensuite $\vect{q}, \vect{k}, \vect{v}$ en une seule fois :

$$
\begin{bmatrix}
\vect{q} \\
\vect{k} \\
\vect{v}
\end{bmatrix} =
\begin{bmatrix}
\vect{W_q} \\
\vect{W_k} \\
\vect{W_v}
\end{bmatrix} \vect{x} \in \mathbb{R}^{3d}
$$

Il y a aussi le concept de *têtes*. Nous avons vu ci-dessus un exemple avec une seule tête, mais nous pouvons en avoir plusieurs. Par exemple, disons que nous avons $h$ têtes puis nous avons des $h$ $\vect{q}$, $h$ $\vect{k}$ et $h$ $\vect{v}$ et nous nous retrouvons avec un vecteur en $\mathbb{R}^{3hd}$ :

$$
\begin{bmatrix}
\vect{q}^1 \\
\vect{q}^2 \\
\vdots \\
\vect{q}^h \\
\vect{k}^1 \\
\vect{k}^2 \\
\vdots \\
\vect{k}^h \\
\vect{v}^1 \\
\vect{v}^2 \\
\vdots \\
\vect{v}^h
\end{bmatrix} =
\begin{bmatrix}
\vect{W_q}^1 \\
\vect{W_q}^2 \\
\vdots \\
\vect{W_q}^h \\
\vect{W_k}^1 \\
\vect{W_k}^2 \\
\vdots \\
\vect{W_k}^h \\
\vect{W_v}^1 \\
\vect{W_v}^2 \\
\vdots \\
\vect{W_v}^h
\end{bmatrix} \vect{x} \in \R^{3hd}
$$

Cependant, nous pouvons toujours transformer les valeurs des têtes pour obtenir la dimension originale $\R^d$ en utilisant un $\vect{W_h} \in \mathbb{R}^{d \times hd}$. Ce n'est qu'une des façons possibles de mettre en œuvre le *key-value store*.

<!--
## [The Transformer](https://www.youtube.com/watch?v=f01J0Dri-6k&t=2114s)

Expanding on our knowledge of attention in particular, we now interpret the fundamental building blocks of the transformer. In particular, we will take a forward pass through a basic transformer, and see how attention is used in the standard encoder-decoder paradigm and compares to the sequential architectures of RNNs.
-->

## [Le Transformer](https://www.youtube.com/watch?v=f01J0Dri-6k&t=2114s)

En élargissant notre connaissance de l'attention, nous interprétons maintenant les éléments fondamentaux du transformer. En particulier, nous allons faire une passe en avant à travers un transformer de base et voir comment l'attention est utilisée dans le paradigme standard de l'encodeur-décodeur. Nous allons également effectuer une comparaison avec les architectures séquentielles des RNNs.

<!--
### Encoder-Decoder Architecture

We should be familiar with this terminology. It is shown most prominently during autoencoder demonstrations, and is prerequisite understanding up to this point. To summarize, an input is fed through an encoder and decoder which impose some sort of bottleneck on the data, forcing only the most important information through. This information is stored in the output of the encoder block, and can be used for a variety of unrelated tasks.

<center>
<img src="{{site.baseurl}}/images/week12/12-3/figure1.png" style="zoom: 60%; background-color:#DCDCDC;" /><br>
<b>Figure 1:</b> Two example diagrams of an autoencoder. The model on the left shows how an autoencoder can be design with two affine transformations + activations, where the image on the right replaces this single "layer" with an arbitrary module of operations.
</center>

Our "attention" is drawn to the autoencoder layout as shown in the model on the right and will now take a look inside, in the context of transformers.
-->

### Architecture encodeur-décodeur

Nous devons nous familiariser avec cette terminologie. Elle est surtout mise en évidence lors des démonstrations d'auto-encodeurs et constitue jusqu'à présent un préalable à la compréhension. Pour résumer, une entrée passe par un encodeur et un décodeur qui imposent une sorte de goulot d'étranglement aux données, ne laissant passer que les informations les plus importantes. Ces informations sont stockées dans la sortie du bloc de l'encodeur et peuvent être utilisées pour toute une série de tâches sans rapport les unes avec les autres.

<center>
<img src="{{site.baseurl}}/images/week12/12-3/figure1.png" style="zoom : 60% ; background-color:#DCDCDC;" /><br>
<b>Figure 1 :</b> Deux exemples de diagrammes d'un auto-encodeur. Le modèle de gauche montre comment un auto-encodeur peut être conçu avec deux transformations affines + activations. L'image de droite remplace cette couche unique par un module d'opérations arbitraire.
</center>

Notre attention est attirée sur la disposition de l'auto-encodeur comme indiqué dans le modèle de droite et nous allons maintenant jeter un coup d'œil à l'intérieur, dans le contexte des *transformers*.

<!--
### Encoder Module

<center>
<img src="{{site.baseurl}}/images/week12/12-3/figure2.png" style="zoom: 60%; background-color:#DCDCDC;" /><br>
<b>Figure 2:</b> The transformer encoder, which accepts at set of inputs $\vect{x}$, and outputs a set of hidden representations $\vect{h}^\text{Enc}$.
</center>

The encoder module accepts a set of inputs, which are simultaneously fed through the self attention block and bypasses it to reach the `Add, Norm` block. At which point, they are again simultaneously passed through the 1D-Convolution and another `Add, Norm` block, and consequently outputted as the set of hidden representation. This set of hidden representation is then either sent through an arbitrary number of encoder modules *i.e.* more layers), or to the decoder. We shall now discuss these blocks in more detail.
-->

### Module Encodeur

<center>
<img src="{{site.baseurl}}/images/week12/12-3/figure2.png" style="zoom : 60% ; background-color:#DCDCDC;" /><br>
<b>Figure 2 :</b> L’encodeur du transformer prend un ensemble d'entrées $\vect{x}$ et renvoie un ensemble de représentations cachées $\vect{h}^\text{Enc}$
</center>

Le module d'encodage accepte un ensemble d'entrées, qui sont simultanément passées par le bloc d'auto-attention et le contourne pour atteindre le bloc `Add, Norm`. À ce stade, elles sont à nouveau passées simultanément par la convolution 1D et un autre bloc `Add, Norm`. Elles en ressortent en sortie comme l'ensemble de la représentation cachée. Cet ensemble de représentation cachée est alors envoyé soit par un nombre arbitraire de modules d'encodage (c'est-à-dire plus de couches), soit au décodeur. Nous allons examiner ces blocs plus en détail.

<!--
### Self-attention

The self-attention model is a normal attention model. The query, key, and value are generated from the same item of the sequential input. In tasks that try to model sequential data, positional encodings are added prior to this input. The output of this block is the attention-weighted values. The self-attention block accepts a set of inputs, from $1, \cdots , t$, and outputs $1, \cdots, t$ attention weighted values which are fed through the rest of the encoder.

<center>
<img src="{{site.baseurl}}/images/week12/12-3/figure3.png" style="zoom: 60%; background-color:#DCDCDC;" /><br>
<b>Figure 3:</b> The self-attention block. The sequence of inputs is shown as a set along the 3rd dimension, and concatenated.
</center>
-->

### Auto-attention

Le modèle d'auto-attention est un modèle d'attention normale. La requête, la clé et la valeur sont générées à partir du même élément de l'entrée séquentielle. Dans les tâches qui tentent de modéliser des données séquentielles, des encodeurs de position sont ajoutés avant cette entrée. La sortie de ce bloc est constituée des valeurs pondérées de l'attention. Le bloc auto-attention accepte un ensemble d'entrées, de $1, \cdots, t$, et produit des valeurs pondérées par l'attention de $1, \cdots, t$ qui sont transmises au reste de l'encodeur.

<center>
<img src="{{site.baseurl}}/images/week12/12-3/figure3.png" style="zoom : 60% ; background-color:#DCDCDC;" /><br>
<b>Figure 3 :</b> Le bloc auto-attention. La séquence d'entrées est représentée comme un ensemble le long de la 3ème dimension et concaténée
</center>


<!--
#### Add, Norm

The add norm block has two components. First is the add block, which is a residual connection, and layer normalization.
-->

#### Add, Norm

Le bloc `Add&Norm` a deux composantes. Le premier est le bloc `Add`, qui est une connexion résiduelle et le deuxième `Norm` qui est la normalisation des couches.

<!--
#### 1D-convolution

Following this step, a 1D-convolution (aka a position-wise feed forward network) is applied. This block consists of two dense layers. Depending on what values are set, this block allows you to adjust the dimensions of the output $\vect{h}^\text{Enc}$.
-->

#### Convolution 1D

Après cette étape, une convolution 1D est appliquée. Ce bloc est constitué de deux couches denses. En fonction des valeurs définies, ce bloc permet d'ajuster les dimensions de la sortie $\vect{h}^\text{Enc}$.


<!--
### Decoder Module

The transformer decoder follows a similar procedure as the encoder. However, there is one additional sub-block to take into account. Additionally, the inputs to this module are different.

<center>
<img src="{{site.baseurl}}/images/week12/12-3/figure5.png" style="zoom: 60%; background-color:#DCDCDC;" /><br>
<b>Figure 4:</b> A friendlier explanation of the decoder.
</center>
-->

### Module décodeur

Le transformer-décodeur suit une procédure similaire à celle de l'encodeur. Cependant, il y a un sous-bloc supplémentaire à prendre en compte. De plus, les entrées de ce module sont différentes.

<center>
<img src="{{site.baseurl}}/images/week12/12-3/figure5.png" style="zoom : 60% ; background-color:#DCDCDC;" /><br>
<b>Figure 4 :</b> Une explication plus amicale du décodeur
</center>


<!--
#### Cross-attention

The cross attention follows the query, key, and value setup used for the self-attention blocks.  However, the inputs are a little more complicated. The input to the decoder is a data point $\vect{y}\_i$, which is then passed through the self attention and add norm blocks, and finally ends up at the cross-attention block. This serves as the query for cross-attention, where the key and value pairs are the output $\vect{h}^\text{Enc}$, where this output was calculated with all past inputs $\vect{x}\_1, \cdots, \vect{x}\_{t}$.
-->

#### Attention croisée

L'attention croisée suit la configuration des requêtes, des clés et des valeurs utilisées pour les blocs d'auto-attention. Cependant, les entrées sont un peu plus compliquées. L'entrée dans le décodeur est un point de données $\vect{y}\_i$, qui est ensuite passé par l'auto-attention et les blocs *Add Norm*, et qui se termine finalement par le bloc d'auto-attention. Celui-ci sert de requête pour l'attention croisée, où les paires de clés et de valeurs sont la sortie $\vect{h}^\text{Enc}$, où cette sortie a été calculée avec toutes les entrées passées $\vect{x}\_1, \cdots, \vect{x}\_{t}$.


<!--
## Summary

A set, $\vect{x}\_1$ to $\vect{x}\_{t}$ is fed through the encoder. Using self-attention and some more blocks, an output representation, $\lbrace\vect{h}^\text{Enc}\rbrace_{i=1}^t$ is obtained, which is fed to the decoder. After applying self-attention to it, cross attention is applied. In this block, the query corresponds to a representation of a symbol in the target language $\vect{y}\_i$, and the key and values are from the source language sentence ($\vect{x}\_1$ to $\vect{x}\_{t}$). Intuitively, cross attention finds which values in the input sequence are most relevant to constructing $\vect{y}\_t$, and therefore deserve the highest attention coefficients. The output of this cross attention is then fed through another 1D-convolution sub-block, and we have $\vect{h}^\text{Dec}$. For the specified target language, it is straightforward from here to see how training will commence, by comparing $\lbrace\vect{h}^\text{Dec}\rbrace_{i=1}^t$ to some target data.
-->

## Résumé

Un ensemble, $\vect{x}\_1$ à $\vect{x}\_{t}$ est transmis par l'encodeur. En utilisant l'auto-attention et quelques autres blocs, on obtient une représentation de sortie, $\lbrace\vect{h}^\text{Enc}\rbrace_{i=1}^t$, qui est envoyée au décodeur. Après lui avoir appliqué l'auto-attention, l'attention croisée est appliquée. Dans ce bloc, la requête correspond à une représentation d'un symbole dans la langue cible $\vect{y}\_i$, et la clé et les valeurs proviennent de la phrase de la langue source ($\vect{x}\_1$ à $\vect{x}\_{t}$). Intuitivement, l'attention croisée permet de trouver quelles valeurs de la séquence d'entrée sont les plus pertinentes pour construire $\vect{y}\_t$, et méritent donc les coefficients d'attention les plus élevés. La sortie de cette attention croisée est ensuite acheminée à travers un autre sous-bloc à convolution 1D et nous obtenons $\vect{h}^\text{Dec}$. Pour la langue cible spécifiée, il est facile de voir comment l’entraînement va commencer, en comparant $\lbrace\vect{h}^\text{Dec}\rbrace_{i=1}^t$ à certaines données cibles.


<!--
### Word Language Models

There are a few important facts we left out before to explain the most important modules of a transformer, but will need to discuss them now to understand how transformers can achieve state-of-art results in language tasks.
-->

### Modèles du monde en linguistique

Abordons quelques éléments laissés de côté lors de l’explication des modules les plus importants d'un transformer pour comprendre comment cette architecture peut obtenir des résultats de pointe dans les tâches linguistiques.

<!--
#### Positional encoding

Attention mechanisms allow us to parallelize the operations and greatly accelerate a model's training time,  but loses sequential information. The positional encoding feature enables allows us to capture this context.
-->

#### L’encodage positionnel

Les mécanismes d'attention nous permettent de mettre en parallèle les opérations et d'accélérer considérablement le temps d'entraînement d'un modèle mais perdent des informations séquentielles. L’encodage positionnel nous permet de saisir ce contexte.

<!--
#### Semantic Representations

Throughout the training of a transformer, many hidden representations are generated. To create an embedding space similar to the one used by the word-language model example in PyTorch, the output of the cross-attention, will provide a semantic representation of the word $x_i$, at which point further experimentation can be performed over this dataset.
-->

#### Représentations sémantiques

Tout au long de l’entraînement d'un transformer, de nombreuses représentations cachées sont générées. Pour créer un espace d’enchâssement similaire à celui utilisé dans PyTorch, la sortie de l'attention croisée fournit une représentation sémantique du mot $x_i$ pouvant servir à d'autres expériences (tâches) sur ce jeu de données.

<!--
### Code Summary

We will now see the blocks of transformers discussed above in a far more understandable format, code!

The first module we will look at the multi-headed attention block. Depenending on query, key, and values entered into this block, it can either be used for self or cross attention.


```python
class MultiHeadAttention(nn.Module):
    def __init__(self, d_model, num_heads, p, d_input=None):
        super().__init__()
        self.num_heads = num_heads
        self.d_model = d_model
        if d_input is None:
            d_xq = d_xk = d_xv = d_model
        else:
            d_xq, d_xk, d_xv = d_input
        # Embedding dimension of model is a multiple of number of heads
        assert d_model % self.num_heads == 0
        self.d_k = d_model // self.num_heads
        # These are still of dimension d_model. To split into number of heads
        self.W_q = nn.Linear(d_xq, d_model, bias=False)
        self.W_k = nn.Linear(d_xk, d_model, bias=False)
        self.W_v = nn.Linear(d_xv, d_model, bias=False)
        # Outputs of all sub-layers need to be of dimension d_model
        self.W_h = nn.Linear(d_model, d_model)
```


Initialization of multi-headed attention class. If a `d_input` is provided, this becomes cross attention. Otherwise, self-attention. The query, key, value setup is constructed as a linear transformation of the input `d_model`.


```python
def scaled_dot_product_attention(self, Q, K, V):
    batch_size = Q.size(0)
    k_length = K.size(-2)

    # Scaling by d_k so that the soft(arg)max doesnt saturate
    Q = Q / np.sqrt(self.d_k)  # (bs, n_heads, q_length, dim_per_head)
    scores = torch.matmul(Q, K.transpose(2,3))  # (bs, n_heads, q_length, k_length)

    A = nn_Softargmax(dim=-1)(scores)  # (bs, n_heads, q_length, k_length)

    # Get the weighted average of the values
    H = torch.matmul(A, V)  # (bs, n_heads, q_length, dim_per_head)

    return H, A
```

Return hidden layer corresponding to encodings of values after scaled by the attention vector. For book-keeping purposes (which values in the sequence were masked out by attention?) A is also returned.

```python
def split_heads(self, x, batch_size):
    return x.view(batch_size, -1, self.num_heads, self.d_k).transpose(1, 2)
```

Split the last dimension into (`heads` × `depth`). Return after transpose to put in shape (`batch_size` × `num_heads` × `seq_length` × `d_k`)

```python
def group_heads(self, x, batch_size):
    return x.transpose(1, 2).contiguous().
        view(batch_size, -1, self.num_heads * self.d_k)
```

Combines the attention heads together, to get correct shape consistent with batch size and sequence length.

```python
def forward(self, X_q, X_k, X_v):
    batch_size, seq_length, dim = X_q.size()
    # After transforming, split into num_heads
    Q = self.split_heads(self.W_q(X_q), batch_size)
    K = self.split_heads(self.W_k(X_k), batch_size)
    V = self.split_heads(self.W_v(X_v), batch_size)
    # Calculate the attention weights for each of the heads
    H_cat, A = self.scaled_dot_product_attention(Q, K, V)
    # Put all the heads back together by concat
    H_cat = self.group_heads(H_cat, batch_size)  # (bs, q_length, dim)
    # Final linear layer
    H = self.W_h(H_cat)  # (bs, q_length, dim)
    return H, A
```

The forward pass of multi headed attention.

Given an input is split into q, k, and v, at which point these values are fed through a scaled dot product attention mechanism, concatenated and fed through a final linear layer. The last output of the attention block is the attention found, and the hidden representation that is passed through the remaining blocks.

Although the next block shown in the transformer/encoder's is the Add,Norm, which is a function already built into PyTorch. As such, it is an extremely simple implementation, and does not need it's own class. Next is the 1-D convolution block. Please refer to previous sections for more details.

Now that we have all of our main classes built (or built for us), we now turn to an encoder module.

```python
class EncoderLayer(nn.Module):
    def __init__(self, d_model, num_heads, conv_hidden_dim, p=0.1):
        self.mha = MultiHeadAttention(d_model, num_heads, p)
        self.layernorm1 = nn.LayerNorm(normalized_shape=d_model, eps=1e-6)
        self.layernorm2 = nn.LayerNorm(normalized_shape=d_model, eps=1e-6)

    def forward(self, x):
        attn_output, _ = self.mha(x, x, x)
        out1 = self.layernorm1(x + attn_output)
        cnn_output = self.cnn(out1)
        out2 = self.layernorm2(out1 + cnn_output)
        return out2
```

In the most powerful transformers, an arbitarily large number of these encoders are stacked on top of one another.

Recall that self attention by itself does not have any recurrence or convolutions, but that's what allows it to run so quickly. To make it sensitive to position we provide positional encodings. These are calculated as follows:


$$
\begin{aligned}
E(p, 2)    &= \sin(p / 10000^{2i / d}) \\
E(p, 2i+1) &= \cos(p / 10000^{2i / d})
\end{aligned}
$$


As to not take up too much room on the finer details, we will point you to https://github.com/Atcold/pytorch-Deep-Learning/blob/master/15-transformer.ipynb for the full code used here.


An entire encoder, with N stacked encoder layers, as well as position embeddings, is written out as


```python
class Encoder(nn.Module):
    def __init__(self, num_layers, d_model, num_heads, ff_hidden_dim,
            input_vocab_size, maximum_position_encoding, p=0.1):
        self.embedding = Embeddings(d_model, input_vocab_size,
                                    maximum_position_encoding, p)
        self.enc_layers = nn.ModuleList()
        for _ in range(num_layers):
            self.enc_layers.append(EncoderLayer(d_model, num_heads,
                                                ff_hidden_dim, p))
    def forward(self, x):
        x = self.embedding(x) # Transform to (batch_size, input_seq_length, d_model)
        for i in range(self.num_layers):
            x = self.enc_layers[i](x)
        return x  # (batch_size, input_seq_len, d_model)
```
-->

### Résumé du code

Voyons les blocs du transformer dont nous avons parlé plus haut dans un format beaucoup plus compréhensible : le code !

Dans le premier module, nous examinons le bloc d'attention mullti-têtes. En fonction de la requête, de la clé et des valeurs entrées dans ce bloc, il peut être utilisé pour l'auto-attention ou l'attention croisée.

```python
class MultiHeadAttention(nn.Module):
    def __init__(self, d_model, num_heads, p, d_input=None):
        super().__init__()
        self.num_heads = num_heads
        self.d_model = d_model
        if d_input is None:
            d_xq = d_xk = d_xv = d_model
        else:
            d_xq, d_xk, d_xv = d_input
        # L'enchâssement de la dimension du modele est un multiple du nombre de têtes
        assert d_model % self.num_heads == 0
        self.d_k = d_model // self.num_heads
        # Ceux-ci sont toujours de dimension d_model. Pour diviser en nombre de têtes
        self.W_q = nn.Linear(d_xq, d_model, bias=False)
        self.W_k = nn.Linear(d_xk, d_model, bias=False)
        self.W_v = nn.Linear(d_xv, d_model, bias=False)
        # Les sorties de toutes les sous-couches doivent être de dimension d_model.
        self.W_h = nn.Linear(d_model, d_model)
```

Initialisation de la classe d'attention à plusieurs têtes. Si une `d_input` est fournie, cela devient une attention croisée. Sinon, c’est l’auto-attention. La configuration de la requête, de la clé, de la valeur est construite comme une transformation linéaire du `d_model` d'entrée.

```python
def scaled_dot_product_attention(self, Q, K, V):
    batch_size = Q.size(0)
    k_length = K.size(-2)

     Mise à l'échelle par d_k pour que le soft(arg)max ne sature pas
    Q = Q / np.sqrt(self.d_k)  # (bs, n_heads, q_length, dim_per_head)
    scores = torch.matmul(Q, K.transpose(2,3))  # (bs, n_heads, q_length, k_length)

    A = nn_Softargmax(dim=-1)(scores)  # (bs, n_heads, q_length, k_length)

    # Donne la moyenne pondérée des valeurs
    H = torch.matmul(A, V)  # (bs, n_heads, q_length, dim_per_head)

    return H, A
```

Retourne la couche cachée correspondant aux encodeurs des valeurs après mise à l'échelle par le vecteur d'attention. Pour des raisons comptables (quelles valeurs de la séquence ont été masquées par l'attention ?), $A$ est également renvoyé.

```python
def split_heads(self, x, batch_size):
    return x.view(batch_size, -1, self.num_heads, self.d_k).transpose(1, 2)
```

Divise la dernière dimension en (`heads` × `depth`). On renvoie après transposition pour mettre en forme (`batch_size` × `num_heads` × `seq_length` × `d_k`)

```python
def group_heads(self, x, batch_size):
    return x.transpose(1, 2).contiguous().
        view(batch_size, -1, self.num_heads * self.d_k)
```

Combine les têtes d'attention ensemble, pour obtenir une forme correcte conforme à la taille du batch et à la longueur de la séquence.

```python
def forward(self, X_q, X_k, X_v):
    batch_size, seq_length, dim = X_q.size()
    # Après la transformation, divise en num_heads
    Q = self.split_heads(self.W_q(X_q), batch_size)
    K = self.split_heads(self.W_k(X_k), batch_size)
    V = self.split_heads(self.W_v(X_v), batch_size)
    # Calcule les poids d'attention pour chacune des têtes
    H_cat, A = self.scaled_dot_product_attention(Q, K, V)
    # Remettre toutes les têtes ensemble par concaténation
    H_cat = self.group_heads(H_cat, batch_size)  # (bs, q_length, dim)
    # Couche linéaire finale
    H = self.W_h(H_cat)  # (bs, q_length, dim)
    return H, A
```

Le passage en avant de l'attention à plusieurs têtes.

Une entrée donnée est divisée en q, k et v, et ces valeurs sont alors transmises par un mécanisme d'attention par produit scalaire, puis concaténées et transmises par une couche linéaire finale. La dernière sortie du bloc d'attention est l'attention trouvée et la représentation cachée qui est passée à travers les blocs restants.

Le bloc *Add,Norm* est une fonction déjà intégrée dans PyTorch. En tant que telle, c'est une implémentation extrêmement simple et n'a pas besoin de sa propre classe. Le bloc suivant est le bloc de convolution 1D. 

Maintenant que toutes nos classes principales sont construites (ou construites pour nous), nous nous tournons vers un module encodeur.

```python
class EncoderLayer(nn.Module):
    def __init__(self, d_model, num_heads, conv_hidden_dim, p=0.1):
        self.mha = MultiHeadAttention(d_model, num_heads, p)
        self.layernorm1 = nn.LayerNorm(normalized_shape=d_model, eps=1e-6)
        self.layernorm2 = nn.LayerNorm(normalized_shape=d_model, eps=1e-6)

    def forward(self, x):
        attn_output, _ = self.mha(x, x, x)
        out1 = self.layernorm1(x + attn_output)
        cnn_output = self.cnn(out1)
        out2 = self.layernorm2(out1 + cnn_output)
        return out2
```

Dans les *transformers* les plus puissants, un nombre important de ces encodeurs sont empilés les uns sur les autres.

Rappelons que l'auto-attention n'a en soi ni récurrence ni convolutions, mais que c'est ce qui lui permet de fonctionner si rapidement. Pour être sensible à la position, nous fournissons des encodeurs de position. Ceux-ci sont calculés comme suit :

$$
\begin{aligned}
E(p, 2)    &= \sin(p / 10000^{2i / d}) \\
E(p, 2i+1) &= \cos(p / 10000^{2i / d})
\end{aligned}
$$

Pour ne pas trop s'attarder sur les détails, nous vous renvoyons au *notebook* pour le code complet utilisé. Il est disponible en anglais [ici](https://github.com/Atcold/pytorch-Deep-Learning/blob/master/15-transformer.ipynb) et en français [ici](https://github.com/lbourdois/pytorch-Deep-Learning-Notebooks-in-French/blob/master/15-transformer.ipynb).


Un encodeur complet, avec $N$ couches d'encodeur empilées, ainsi que des enchâssements de position, s'écrit :

```python
class Encoder(nn.Module):
    def __init__(self, num_layers, d_model, num_heads, ff_hidden_dim,
            input_vocab_size, maximum_position_encoding, p=0.1):
        self.embedding = Embeddings(d_model, input_vocab_size,
                                    maximum_position_encoding, p)
        self.enc_layers = nn.ModuleList()
        for _ in range(num_layers):
            self.enc_layers.append(EncoderLayer(d_model, num_heads,
                                                ff_hidden_dim, p))
    def forward(self, x):
        x = self.embedding(x) # Transforme en (batch_size, input_seq_length, d_model)
        for i in range(self.num_layers):
            x = self.enc_layers[i](x)
        return x  # (batch_size, input_seq_len, d_model)
```

<!--
## Example Use

There is a lot of tasks you can use just an Encoder for. In the accompanying notebook, we see how an encoder can be used for sentiment analysis.

Using the imdb review dataset, we can output from the encoder a latent representation of the a sequence of text, and train this encoding process with binary cross entropy, corresponding to a positive or negative movie review.

Again we leave out the nuts and bolts, and direct you to the notebook, but here is the most important architectural components used in the transformer:



```python
class TransformerClassifier(nn.Module):
    def forward(self, x):
        x = Encoder()(x)
        x = nn.Linear(d_model, num_answers)(x)
        return torch.max(x, dim=1)

model = TransformerClassifier(num_layers=1, d_model=32, num_heads=2,
                         conv_hidden_dim=128, input_vocab_size=50002, num_answers=2)
```
Where this model is trained in typical fashion.
-->

## Exemple d'utilisation

Il y a beaucoup de tâches pour lesquelles vous pouvez utiliser un simple encodeur. Dans le *notebook*, nous voyons comment un encodeur peut être utilisé pour l'analyse des sentiments.

En utilisant le jeu de données d'IMDB (l’équivalent anglophone d’AlloCiné), nous pouvons sortir de l'encodeur une représentation latente d'une séquence de texte et entraîner ce processus d'encodage avec une entropie croisée binaire, correspondant à une critique de film positive ou négative.
Là encore, nous laissons de côté les dessous techniques et nous vous dirigeons vers le *notebook*. Voici les composants architecturaux les plus importants utilisés dans le transformer :

```python
class TransformerClassifier(nn.Module):
    def forward(self, x):
        x = Encoder()(x)
        x = nn.Linear(d_model, num_answers)(x)
        return torch.max(x, dim=1)

model = TransformerClassifier(num_layers=1, d_model=32, num_heads=2,
                         conv_hidden_dim=128, input_vocab_size=50002, num_answers=2)
```
Où ce modèle est entraîné de façon typique.
