---
lang-ref: ch.05-3
title: Comprendre les convolutions et autograd
lecturer: Alfredo Canziani
authors: Leyi Zhu, Siqi Wang, Tao Wang, Anqi Zhang
date: 25 Feb 2020
lang: fr
translation-date: 05 Aug 2020
translator: Loïck Bourdois
---

<!--
## [Understanding 1D convolution](https://www.youtube.com/watch?v=eEzCZnOFU1w&t=140s)

In this part we will discuss convolution, since we would like to explore the sparsity, stationarity, compositionality of the data.

Instead of using the matrix $A$ discussed in the [previous week]({{site.baseurl}}/en/week04/04-1), we will change the matrix width to the kernel size $k$. Therefore, each row of the matrix is a kernel. We can use the kernels by stacking and shifting (see Fig 1). Then we can have $m$ layers of height $n-k+1$.

<center>
<img src="{{site.baseurl}}/images/week05/05-3/Illustration_1D_Conv.png" alt="1" style="zoom:40%;" /><br>
<b>Fig 1</b>: Illustration of 1D Convolution
</center>

The output is $m$ (thickness) vectors of size $n-k+1$.

<center>
<img src="{{site.baseurl}}/images/week05/05-3/Result_1D_Conv.png" alt="2" style="zoom:40%;" /><br>
<b>Fig 2</b>: Result of 1D Convolution
</center>

Furthermore, a single input vector can viewed as a monophonic signal.

<center>
<img src="{{site.baseurl}}/images/week05/05-3/Monophonic_Signal.png" alt="3" style="zoom:40%;" /><br>
<b>Fig 3</b>: Monophonic Signal
</center>

Now, the input $x$ is a mapping

$$
x:\Omega\rightarrow\mathbb{R}^{c}
$$

where $\Omega = \lbrace 1, 2, 3, \cdots \rbrace \subset \mathbb{N}^1$ (since this is $1$ dimensional signal / it has a $1$ dimensional domain) and in this case the channel number $c$ is $1$. When $c = 2$ this becomes a stereophonic signal.

For the 1D convolution, we can just compute the scalar product, kernel by kernel (see Fig 4).

<center>
<img src="{{site.baseurl}}/images/week05/05-3/Layer_by_layer_scalar_product.png" alt="4" style="zoom:40%;" /><br>
<b>Fig 4</b>: Layer-by-layer Scalar Product of 1D Convolution
</center>
-->


## [Comprendre la convolution 1D](https://www.youtube.com/watch?v=eEzCZnOFU1w&t=140s)

<!--
Dans cette partie, nous parlerons de la convolution, puisque nous souhaitons explorer l’éparsité, la stationnarité, la compositionnalité des données.
-->

Au lieu d'utiliser la matrice $A$ de la [semaine précédente]({{site.baseurl}}/fr/week04/04-1), nous allons changer la largeur de la matrice pour la taille du noyau $k$. Par conséquent, chaque ligne de la matrice est un noyau. Nous pouvons utiliser les noyaux en les empilant et en les déplaçant (voir la figure 1). Nous pouvons alors avoir $m$ couches de hauteur $n-k+1$.

<center>
<img src="{{site.baseurl}}/images/week05/05-3/Illustration_1D_Conv.png" alt="1" style="zoom:40% ;" /><br>
<b>Figure 1 :</b> Illustration de la convolution 1D
</center>

La sortie est $m$ vecteurs de taille $n-k+1$.

<center>
<img src="{{site.baseurl}}/images/week05/05-3/Result_1D_Conv.png" alt="2" style="zoom:40% ;" /><br>
<b>Figure 2</b> : Résultat de la convolution 1D
</center>

De plus, un seul vecteur d'entrée peut être considéré comme un signal monophonique.

<center>
<img src="{{site.baseurl}}/images/week05/05-3/Monophonic_Signal.png" alt="3" style="zoom:40% ;" /><br>
<b>Figure 3</b> : Signal monophonique
</center>

Maintenant, l'entrée $ x:\Omega\rightarrow\mathbb{R}^{c}$ avec $\Omega = \lbrace 1, 2, 3, \cdots \rbrace \subset \mathbb{N}^1$ (puisque c'est un signal de dimension $1$ / il a un domaine de dimension $1$) et dans ce cas le numéro de canal $c$ est $1$. Lorsque $c = 2$, cela devient un signal stéréophonique.

Pour la convolution 1D, nous pouvons simplement calculer le produit scalaire, noyau par noyau (voir figure 4).

<center>
<img src="{{site.baseurl}}/images/week05/05-3/Layer_by_layer_scalar_product.png" alt="4" style="zoom:40% ;" /><br>
<b>Figure 4</b> : Produit scalaire couche par couche de la convolution 1D
</center>


<!--
## [Dimension of kernels and output width in PyTorch](https://www.youtube.com/watch?v=eEzCZnOFU1w&t=1095s)

Tips: We can use ***question mark*** in IPython to get access to the documents of functions. For example,

```python
Init signature:
nn.Conv1d(
	in_channels,           # number of channels in the input image
	out_channels,          # number of channels produced by the convolution
	kernel_size,           # size of the convolving kernel
	stride=1,              # stride of the convolution
	padding=0,             # zero-padding added to both sides of the input
	dilation=1,            # spacing between kernel elements
	groups=1,              # nb of blocked connections from input to output
	bias=True,             # if `True`, adds a learnable bias to the output
	padding_mode='zeros',  # accepted values `zeros` and `circular`
)
```
-->

## [Dimension des noyaux et largeur des sorties dans PyTorch](https://www.youtube.com/watch?v=eEzCZnOFU1w&t=1095s)

Astuce : on peut utiliser le ***question mark*** dans IPython pour avoir accès aux documents de fonctions. Par exemple,

```python
Init signature :
nn.Conv1d(
	in_channels, # nombre de canaux dans l'image d'entrée
	out_channels, # nombre de canaux produits par la convolution
	kernel_size, # taille du noyau en convolution
	stride=1, # pas de la convolution
	padding=0, # rembourage de 0 ajouté aux deux côtés de l'entrée
	dilatation=1, # espacement entre les éléments du noyau
	groups=1, # nombre de connexions bloquées de l'entrée à la sortie
	bias=True, # si `True`, ajoute un biais appris à la sortie
	padding_mode='zeros', # valeurs acceptées `zeros` et `circular`.
)
```


<!--
### 1D convolution

We have $1$ dimensional convolution going from $2$ channels (stereophonic signal) to $16$ channels ($16$ kernels) with kernel size of $3$ and stride of $1$. We then have $16$ kernels with thickness $2$ and length $3$. Let's assume that the input signal has a batch of size $1$ (one signal), $2$ channels and $64$ samples. The resulting output layer has $1$ signal, $16$ channels and the length of the signal is $62$ ($=64-3+1$). Also, if we output the bias size, we'll find the bias size is $16$, since we have one bias per weight.

```python
conv = nn.Conv1d(2, 16, 3)  # 2 channels (stereo signal), 16 kernels of size 3
conv.weight.size()          # output: torch.Size([16, 2, 3])
conv.bias.size()            # output: torch.Size([16])

x = torch.rand(1, 2, 64)    # batch of size 1, 2 channels, 64 samples
conv(x).size()              # output: torch.Size([1, 16, 62])

conv = nn.Conv1d(2, 16, 5)  # 2 channels, 16 kernels of size 5
conv(x).size()              # output: torch.Size([1, 16, 60])
```
-->

### Convolution 1D

Nous avons une convolution de dimension $1$ allant de $2$ canaux (signal stéréophonique) à $16$ canaux ($16$ noyaux) avec une taille de noyau de $3$ et un pas de $1$. Nous avons ensuite $16$ noyaux avec une épaisseur de $2$ et une longueur de $3$. Supposons que le signal d'entrée ait un batch de taille $1$ (un signal), de $2$ canaux et $64$ échantillons. La couche de sortie résultante a $1$ signal, des $16$ canaux et la longueur du signal est de $62$ ($=64-3+1$). De plus, si nous affichons la taille du biais, nous constaterons qu’elle de $16$, puisque nous avons un biais par poids.


```python
conv = nn.Conv1d(2, 16, 3) # 2 canaux (signal stéréo), 16 noyaux de taille 3
conv.weight.size() # sortie : torch.Size([16, 2, 3])
conv.bias.size() # sortie : torch.Size([16])

x = torche.rand(1, 2, 64) # batch de taille 1, 2 canaux, 64 échantillons
conv(x).size() # sortie : torch.Size([1, 16, 62])

conv = nn.Conv1d(2, 16, 5) # 2 canaux, 16 noyaux de taille 5
conv(x).size() # sortie : torch.Size([1, 16, 60])
```


<!--
### 2D convolution

We first define the input data as $1$ sample, $20$ channels (say, we're using an hyperspectral image) with height $64$ and width $128$. The 2D convolution has $20$ channels from input and $16$ kernels with size of $3 \times 5$. After the convolution, the output data has $1$ sample, $16$ channels with height $62$ ($=64-3+1$) and width $124$ ($=128-5+1$).

```python
x = torch.rand(1, 20, 64, 128)    # 1 sample, 20 channels, height 64, and width 128
conv = nn.Conv2d(20, 16, (3, 5))  # 20 channels, 16 kernels, kernel size is 3 x 5
conv.weight.size()                # output: torch.Size([16, 20, 3, 5])
conv(x).size()                    # output: torch.Size([1, 16, 62, 124])
```

If we want to achieve the same dimensionality, we can have paddings. Continuing the code above, we can add new parameters to the convolution function: `stride=1` and `padding=(1, 2)`, which means $1$ on $y$ direction ($1$ at the top and $1$ at the bottom) and $2$ on $x$ direction. Then the output signal is in the same size compared to the input signal. The number of dimensions that is required to store the collection of kernels when you perform 2D convolution is $4$.

```python
# 20 channels, 16 kernels of size 3 x 5, stride is 1, padding of 1 and 2
conv = nn.Conv2d(20, 16, (3, 5), 1, (1, 2))
conv(x).size()  # output: torch.Size([1, 16, 64, 128])
```
-->

### Convolution 2D

Nous définissons d'abord les données d'entrée comme un $1$ échantillon, $20$ canaux (disons que nous utilisons une image hyperspectrale) avec une hauteur de $64$ et une largeur de $128$. La convolution 2D a $20$ canaux en entrée et $16$ noyaux de taille de $3\times5$. Après la convolution, la donnée en sortie a un $1$ échantillon, $16$ canaux avec une hauteur de $62$ ($=64-3+1$) et une largeur de $124$ ($=128-5+1$).

```python
x = torch.rand(1, 20, 64, 128) # 1 échantillon, 20 canaux, hauteur 64, et largeur 128
conv = nn.Conv2d(20, 16, (3, 5))  # 20 canaux, 16 noyaux, la taille des noyaux est de 3 x 5
conv.weight.size() # sortie : torch.Size([16, 20, 3, 5])
conv(x).size() # sortie : torch.Size([1, 16, 62, 124])
```

Si nous voulons atteindre la même dimensionnalité, nous pouvons rembourrer. En continuant le code ci-dessus, nous pouvons ajouter de nouveaux paramètres à la fonction de convolution : `stride=1` et `padding=(1, 2)`, ce qui signifie $1$ dans la direction $y$ ($1$ en haut et $1$ en bas) et $2$ dans la direction $x$. Le signal de sortie est alors de la même taille que le signal d'entrée. Le nombre de dimensions nécessaires pour stocker la collection de noyaux lorsque vous effectuez une convolution 2D est de $4$.

```python
# 20 canaux, 16 noyaux de taille 3 x 5, pas de 1, rembourrage 1 et 2
conv = nn.Conv2d(20, 16, (3, 5), 1, (1, 2))
conv(x).size() # sortie : torch.Size([1, 16, 64, 128])
```

<!--
## [How automatic gradient works?](https://www.youtube.com/watch?v=eEzCZnOFU1w&t=1634s)

In this section we're going to ask torch to check all the computation over the tensors so that we can perform the computation of partial derivatives.

- Create a $2\times2$ tensor $\boldsymbol{x}$ with gradient-accumulation capabilities;
- Deduct $2$ from all elements of $\boldsymbol{x}$ and get $\boldsymbol{y}$; (If we print `y.grad_fn`, we will get `<SubBackward0 object at 0x12904b290>`, which means that `y` is generated by the module of subtraction $\boldsymbol{x}-2$. Also we can use `y.grad_fn.next_functions[0][0].variable` to derive the original tensor.)
- Do more operations: $\boldsymbol{z} = 3\boldsymbol{y}^2$;
- Calculate the mean of $\boldsymbol{z}$.

<center>
<img src="{{site.baseurl}}/images/week05/05-3/Flow_Chart.png" alt="5" style="zoom:60%;" /><br>
<b>Fig 5</b>: Flow Chart of the Auto-gradient Example
</center>

Back propagation is used for computing the gradients. In this example, the process of back propagation can be viewed as computing the gradient $\frac{d\boldsymbol{a}}{d\boldsymbol{x}}$. After computing $\frac{d\boldsymbol{a}}{d\boldsymbol{x}}$ by hand as a validation, we can find that the execution of `a.backward()` gives us the same value of *x.grad* as our computation.

Here is the process of computing back propagation by hand:

$$
\begin{aligned}
a &= \frac{1}{4} (z_1 + z_2 + z_3 + z_4) \\
z_i &= 3y_i^2 = 3(x_i-2)^2 \\
\frac{da}{dx_i} &= \frac{1}{4}\times3\times2(x_i-2) = \frac{3}{2}x_i-3 \\
x &= \begin{pmatrix} 1&2\\3&4\end{pmatrix} \\
\left(\frac{da}{dx_i}\right)^\top &= \begin{pmatrix} 1.5-3&3-3\\[2mm]4.5-3&6-3\end{pmatrix}=\begin{pmatrix} -1.5&0\\[2mm]1.5&3\end{pmatrix}
\end{aligned}
$$

Whenever you use partial derivative in PyTorch, you get the same shape of the original data. But the correct Jacobian thing should be the transpose.
-->

## [Comment fonctionne le gradient automatique ?](https://www.youtube.com/watch?v=eEzCZnOFU1w&t=1634s)

Dans cette section, nous allons demander à Pytorch de vérifier tous les calculs sur les tenseurs afin que nous puissions effectuer le calcul des dérivés partielles.

- Créer un tenseur $2\times2$ $\boldsymbol{x}$ avec des capacités d'accumulation de gradients
- Enlever $2$ à tous les éléments de $\boldsymbol{x}$ et obtenir $\boldsymbol{y}$  
Si nous affichons `y.grad_fn`, nous obtenons l'objet `<SubBackward0 à 0x12904b290>`, ce qui signifie que `y` est généré par le module de soustraction $\boldsymbol{x}-2$. On peut aussi utiliser `y.grad_fn.next_functions[0][0].variable` pour dériver le tenseur original.
- Faire d'autres opérations : $\boldsymbol{z} = 3\boldsymbol{y}^2$
- Calculer la moyenne de $\boldsymbol{z}$

<center>
<img src="{{site.baseurl}}/images/week05/05-3/Flow_Chart.png" alt="5" style="zoom:60% ;" /><br>
<b>Figure 5</b> : Exemple d’organigramme d'auto-gradient
</center>

La rétropropagation est utilisée pour calculer les gradients. Dans cet exemple, le processus de rétropropagation peut être considéré comme le calcul du gradient $\frac{d\boldsymbol{a}}{d\boldsymbol{x}}$. Après avoir calculé manuellement $\frac{d\boldsymbol{a}}{d\boldsymbol{x}}$ comme validation, nous pouvons constater que l'exécution de `a.backward()` nous donne la même valeur que *x.grad*.

Voici le processus de calcul de la rétropropagation à la main :

$$
\begin{aligned}
a &= \frac{1}{4} (z_1 + z_2 + z_3 + z_4) \\
z_i &= 3y_i^2 = 3(x_i-2)^2 \\
\frac{da}{dx_i} &= \frac{1}{4}\times3\times2(x_i-2) = \frac{3}{2}x_i-3 \\
x &= \begin{pmatrix} 1&2\\3&4\end{pmatrix} \\
\left(\frac{da}{dx_i}\right)^\top &= \begin{pmatrix} 1.5-3&3-3\\[2mm]4.5-3&6-3\end{pmatrix}=\begin{pmatrix} -1.5&0\\[2mm]1.5&3\end{pmatrix}
\end{aligned}
$$


Chaque fois que nous utilisons une dérivée partielle dans PyTorch, nous obtenons la même forme que les données originales. Mais la bonne jacobienne devrait être la transposée.


<!--
### From basic to more crazy

Now we have a $1\times3$ vector $x$, assign $y$ to the double $x$ and keep doubling $y$ until its norm is smaller than $1000$. Due to the randomness we have for $x$, we cannot directly know the number of iterations when the procedure terminates.

```python
x = torch.randn(3, requires_grad=True)

y = x * 2
i = 0
while y.data.norm() < 1000:
    y = y * 2
    i += 1
```

However, we can infer it easily by knowing the gradients we have.

```python
gradients = torch.FloatTensor([0.1, 1.0, 0.0001])
y.backward(gradients)

print(x.grad)
tensor([1.0240e+02, 1.0240e+03, 1.0240e-01])
print(i)
9
```

As for the inference, we can use `requires_grad=True` to label that we want to track the gradient accumulation as shown below. If we omit `requires_grad=True` in either $x$ or $w$'s declaration and call `backward()` on $z$, there will be runtime error due to we do not have gradient accumulation on $x$ or $w$.

```python
# Both x and w that allows gradient accumulation
x = torch.arange(1., n + 1, requires_grad=True)
w = torch.ones(n, requires_grad=True)
z = w @ x
z.backward()
print(x.grad, w.grad, sep='\n')
```

And, we can have `with torch.no_grad()` to omit the gradient accumulation.

```python
x = torch.arange(1., n + 1)
w = torch.ones(n, requires_grad=True)

# All torch tensors will not have gradient accumulation
with torch.no_grad():
    z = w @ x

try:
    z.backward()  # PyTorch will throw an error here, since z has no grad accum.
except RuntimeError as e:
    print('RuntimeError!!! >:[')
    print(e)
```
-->


### Vers plus de folie

Maintenant que nous avons un vecteur $1\times3$ $x$, assignons $2x$ à $y$ et doublons $y$ jusqu'à ce que sa norme soit inférieure à $1000$. En raison du caractère aléatoire que nous avons pour $x$, nous ne pouvons pas connaître directement le nombre d'itérations lorsque la procédure se termine.

```python
x = torch.randn(3, requires_grad=True)

y = x * 2
i = 0
while y.data.norm() < 1000:
    y = y * 2
    i += 1
```

Cependant, nous pouvons facilement le déduire en connaissant les gradients dont nous disposons.

```python
gradients = torch.FloatTensor([0.1, 1.0, 0.0001])
y.backward(gradients)

print(x.grad)
tensor([1.0240e+02, 1.0240e+03, 1.0240e-01])
print(i)
9
```

Quant à l'inférence, nous pouvons utiliser `requires_grad=True` pour indiquer que nous voulons suivre l'accumulation du gradient comme indiqué ci-dessous. Si nous omettons `requires_grad=True` dans la déclaration de $x$ ou $w$ et appelons `backward()` sur $z$, il y aura une erreur d'exécution due au fait que nous n'avons pas d'accumulation de gradient sur $x$ ou $w$.

```python
# x et w qui permettent l'accumulation de gradients
x = torch.arange(1., n + 1, requires_grad=True)
w = torch.ones(n, requires_grad=True)
z = w @ x
z.backward()
print(x.grad, w.grad, sep='\n')
```

Nous pouvons avoir `with torch.no_grad()` pour omettre l'accumulation de gradient.

```python
x = torch.arange(1., n + 1)
w = torch.ones(n, requires_grad=True)

# Tous les tenseurs n'auront pas l'accumulation de gradient
with torch.no_grad():
    z = w @ x

try:
    z.backward()# PyTorch va retourner une erreur ici, puisque z n'a pas de grad accum.
except RuntimeError as e:
    print('RuntimeError!!! >:[')
    print(e)
```

<!--
## More stuff -- custom gradients

Also, instead of basic numerical operations, we can generate our own self-defined modules / functions, which can be plugged into the neural graph. The Jupyter Notebook can be found [here](https://github.com/Atcold/pytorch-Deep-Learning/blob/master/extra/b-custom_grads.ipynb).

To do so, we need to inherit `torch.autograd.Function` and override `forward()` and `backward()` functions. For example, if we want to training nets, we need to get the forward pass and know the partial derivatives of the input respect to the output, such that we can use this module in any kind of point in the code. Then, by using back-propagation (chain rule), we can plug the thing anywhere in the chain of operations, as long as we know the partial derivatives of the input respect to the output.

In this case, there are three examples of ***custom modules*** in the *notebook*, the `add`, `split`, and `max` modules. For example, the custom addition module:

```python
# Module personnalisé supplémentaire
class MyAdd(torch.autograd.Function):

    @staticmethod
    def forward(ctx, x1, x2):
        # ctx est un contexte où nous pouvons sauvegarder les calculs pour la rétropropagation
        ctx.save_for_backward(x1, x2)
        return x1 + x2

    @staticmethod
    def backward(ctx, grad_output):
        x1, x2 = ctx.saved_tensors
        grad_x1 = grad_output * torch.ones_like(x1)
        grad_x2 = grad_output * torch.ones_like(x2)
        # on a besoin de retourner les gradients pour la phase avant
        return grad_x1, grad_x2
```

If we have addition of two things and get an output, we need to overwrite the forward function like this. And when we go down to do back propagation, the gradients copied over both sides. So we overwrite the backward function by copying.

For `split` and `max`, see the code of how we overwrite forward and backward functions in the *notebook*. If we come from the same thing and **Split**, when go down doing gradients, we should add / sum them. For `argmax`, it selects the index of the highest thing, so the index of the highest should be $1$ while others being $0$. Remember, according to different custom modules, we need to overwrite its own forward pass and how they do gradients in backward function.
-->

## Plus de choses : des gradients personnalisés
Au lieu d'opérations numériques de base, nous pouvons générer nos propres modules / fonctions qui peuvent être branchés sur le graphe neural. Le *notebook* Jupyter se trouve [ici](https://github.com/Atcold/pytorch-Deep-Learning/blob/master/extra/b-custom_grads.ipynb).  
Pour ce faire, nous devons partir de la fonction `torch.autograd.Function` et remplacer les fonctions `forward()` et `backward()`. Par exemple, si nous voulons entraîner des réseaux, nous devons obtenir la propagation en avant dans le réseau et connaître les dérivées partielles de l'entrée par rapport à la sortie, de sorte que nous puissions utiliser ce module en tout point du code. Ensuite, en utilisant la rétropropagation (règle de la chaîne), nous pouvons insérer la chose n'importe où dans la chaîne d'opérations, à condition de connaître les dérivées partielles de l'entrée par rapport à la sortie.

Dans ce cas, il y a trois exemples de ***modules personnalisés*** dans le *notebook*, les modules `add`, `split`, et `max`. Par exemple, le module d'ajout personnalisé :
```python
# Module personnalisé supplémentaire
class MyAdd(torch.autograd.Function):

    @staticmethod
    def forward(ctx, x1, x2):
        # ctx est un contexte où nous pouvons sauvegarder les calculs pour la rétropropagation
        ctx.save_for_backward(x1, x2)
        return x1 + x2

    @staticmethod
    def backward(ctx, grad_output):
        x1, x2 = ctx.saved_tensors
        grad_x1 = grad_output * torch.ones_like(x1)
        grad_x2 = grad_output * torch.ones_like(x2)
        # on a besoin de retourner les gradients pour la phase avant
        return grad_x1, grad_x2
```
Si nous avons l'addition de deux choses et que nous obtenons un résultat, nous devons écraser la fonction *forward* comme ceci. Et lorsque nous rétropropagation, les gradients sont copiés sur les deux côtés. Nous écrasons donc la fonction *backward* en copiant.

Pour les fonctions `split` et `max`, consultez le code pour la façon dont nous écrasons les fonctions *forward* et *backward* dans le *notebook*. Pour `argmax`, cela sélectionne l'indice de la chose la plus élevée. Ainsi l'indice de la plus élevée devrait être de $1$ et $0$ pour les autres.
