---
lang: es
lang-ref: ch.13-3
title: Redes Convolucionales en Grafos III
lecturer: Alfredo Canziani
authors: Go Inoue, Muhammad Osama Khan, Muhammad Shujaat Mirza, Muhammad Muneeb Afzal
date: 28 Apr 2020
translation-date: 16 Sep 2020
translator: Joaquim Castilla (xcastilla)
---

<!-- ## [Introduction to Graph Convolutional Network (GCN)](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=47s) -->
## [Introducción a las Redes Convolucionales de Grafos(GCN)](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=47s)

<!-- Graph Convolutional Network (GCN) is one type of architecture that utilizes the structure of data.
Before going into details, let's have a quick recap on self-attention, as GCN and self-attention are conceptually relevant. -->
Las Redes Convolucionales de Grafos (GCN) son un tipo de arquitectura que utilizan la estructura de los datos.
Antes de entrar en detalles, hagamos una revisión rápida sobre la auto-atención, dado que las GCN y la auto-atención son conceptualmente relevantes.

<!-- ### Recap: Self-attention -->
### Revisión: Auto-atención

<!-- - In self-attention, we have a set of input $\lbrace\boldsymbol{x}\_{i}\rbrace^{t}_{i=1}$.
Unlike a sequence, it does not have an order.
- Hidden vector $\boldsymbol{h}$ is given by linear combination of the vectors in the set.
- We can express this as $\boldsymbol{X}\boldsymbol{a}$ using matrix vector multiplication, where $\boldsymbol{a}$ contains coefficients that scale the input vector $\boldsymbol{x}_{i}$.

*For a detailed explanation, refer to the notes of [Week 12]({{site.baseurl}}/en/week12/12-3/).* -->
- En auto-atención, tenemos un conjunto de entradas $\lbrace\boldsymbol{x}\_{i}\rbrace^{t}_{i=1}$.
A diferencia de una secuencia, este conjunto no tiene un orden.
- El vector oculto $\boldsymbol{h}$ viene dado por una combinación linear de los vectores en el conjunto.
- Esto lo podemos expresar como $\boldsymbol{X}\boldsymbol{a}$ utilizando una multiplicación matriz vector, donde $\boldsymbol{a}$ contiene los coeficientes que escalan al vector de entrada $\boldsymbol{x}_{i}$.

*Para una explicación detallada, consulta las notas de la [Semana 12]({{site.baseurl}}/es/week12/12-3/).*

<!-- ### Notation -->
### Notación

<!-- <center>
<img src="{{site.baseurl}}/images/week13/13-3/figure1.png" height="400px" /><br>
<b>Fig. 1</b>: Graph Convolutional Network
</center> -->
<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure1.png" height="400px" /><br>
<b>Fig. 1</b>: Red Convolucional de Grafos
</center>

<!-- 
In Figure 1, vertex $v$ is comprised of two vectors: input $\boldsymbol{x}$ and its hidden representation $\boldsymbol{h}$.
We also have multiple vertices $v_{j}$, which is comprised of $\boldsymbol{x}_{j}$ and $\boldsymbol{h}_{j}$.
In this graph, vertices are connected with directed edges. -->
En la Figura 1, el vértice $v$ está formado por dos vectores:
el vector de entrada $\boldsymbol{x}$ y su representación oculta $\boldsymbol{h}$.
Tenemos también múltiples vértices $v_{j}$, que están formados por $\boldsymbol{x}\_j$ y $\boldsymbol{h}\_j$.
En este grafo, los vértices están conectados por ejes dirigidos.

<!-- We represent these directed edges with adjacecy vector $\boldsymbol{a}$, where each element $\alpha_{j}$ is set to $1$ if there is a directed edge from $v_{j}$ to $v$. -->
Representamos estos ejes dirigidos con un vector de adyacencia $\boldsymbol{a}$, donde cada elemento $\alpha_{j}$ tiene asignado un $1$ si hay eje de $v_{j}$ a $v$.

$$
\alpha_{j} \stackrel{\tiny \downarrow}{=} 1 \Leftrightarrow v_{j} \rightarrow v
\tag{Ec. 1}
$$

<!-- The degree (number of incoming edges) $d$ is defined as the norm of this adjacency vector, *i.e.* $\Vert\boldsymbol{a}\Vert_{1} $, which is the number of ones in the vector $\boldsymbol{a}$. -->
El grado (número de ejes entrantes) $d$ se define como la norma de este vector de adyacencia, es decir
$\Vert\boldsymbol{a}\Vert_{1} $, que es el número de unos en el vector $\boldsymbol{a}$.

$$
    d = \Vert\boldsymbol{a}\Vert_{1}
\tag{Ec. 2}
$$

<!-- The hidden vector $\boldsymbol{h}$ is given by the following expression: -->
El vector oculto $\boldsymbol{h}$ se determina mediante la siguiente expresión:

$$
    \boldsymbol{h}=f(\boldsymbol{U}\boldsymbol{x} + \boldsymbol{V}\boldsymbol{X}\boldsymbol{a}d^{-1})
\tag{Ec. 3}
$$

<!-- where $f(\cdot)$ is a non-linear function such as ReLU $(\cdot)^{+}$, Sigmoid $\sigma(\cdot)$, and hyperbolic tangent $\tanh(\cdot)$. -->
donde $f(\cdot)$ es una función no lineal como ReLU $(\cdot)^{+}$, Sigmoidal $\sigma(\cdot)$ y la tangente hiperbólica $\tanh(\cdot)$.

<!-- The $\boldsymbol{U}\boldsymbol{x}$ term takes into account the vertex $v$ itself, by applying rotation $\boldsymbol{U}$ to the input $v$. -->
El término $\boldsymbol{U}\boldsymbol{x}$ tiene en cuenta al vértice $v$ en sí mismo, aplicando la rotación $\boldsymbol{U}$ a la entrada $v$.

<!-- Remember that in self-attention, the hidden vector $\boldsymbol{h}$ is computed by $\boldsymbol{X}\boldsymbol{a}$, which means that the columns in $\boldsymbol{X}$ is scaled by the factors in $\boldsymbol{a}$.
In the context of GCN, this means that if we have multiple incoming edges,*i.e.* multiple ones in adjacency vector $\boldsymbol{a}$, $\boldsymbol{X}\boldsymbol{a}$ gets larger.
On the other hand, if we have only one incoming edge, this value gets smaller.
To remedy this issue of the value being proportionate to the number of incoming edges, we divide it by the number of incoming edges $d$.
We then apply rotation $\boldsymbol{V}$ to $\boldsymbol{X}\boldsymbol{a}d^{-1}$. -->
Recuerda que en auto-atención, el vector oculto $\boldsymbol{h}$ se calcula como $\boldsymbol{X}\boldsymbol{a}$, lo que significa que se hace un escalado de las columnas de $\boldsymbol{X}$ con los factores en $\boldsymbol{a}$.
En el contexto de las GCN, esto significa que si tenemos múltiples ejes entrantes, es decir, múltiples unos en el vector de adyacencia $\boldsymbol{a}$, $\boldsymbol{X}\boldsymbol{a}$ aumentará.
Por otra parte, si solo tenemos un eje entrante, este valor disminuye.
Para solucionar el problema de que el valor sea proporcional al número de ejes entrantes, lo dividimos por el número total de ejes entrantes $d$.
Después aplicamos la rotación $\boldsymbol{V}$ a $\boldsymbol{X}\boldsymbol{a}d^{-1}$.

<!-- We can represent this hidden representation $\boldsymbol{h}$ for the entire set of inputs $\boldsymbol{x}$ using the following matrix notation: -->
Podemos representar está representación oculta $\boldsymbol{h}$ para el conjunto entero de entradas $\boldsymbol{x}$ utilizando la siguiente notación matricial:

$$
\{\boldsymbol{x}_{i}\}^{t}_{i=1}\rightsquigarrow \boldsymbol{H}=f(\boldsymbol{UX}+ \boldsymbol{VXAD}^{-1})
\tag{Ec. 4}
$$

<!-- where $\vect{D} = \text{diag}(d_{i})$. -->
donde $\vect{D} = \text{diag}(d_{i})$.

<!-- ## [Residual Gated GCN Theory and Code](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=992s) -->
## [Teoría de Residual Gated GCN con código](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=992s)

<!-- Residual Gated Graph Convolutional Network is a type of GCN that can be represented as shown in Figure 2:
 -->
Las Redes convolucionales de Grafos "Residual Gated" son un tipo de GCN que se pueden representar como se muestra en la Figura 2:

<!-- <center>
<img src="{{site.baseurl}}/images/week13/13-3/figure2.png" height="300px" /><br>
<b>Fig. 2</b>: Residual Gated Graph Convolutional Network
</center> -->
<center>
<img src="{{site.baseurl}}/images/week13/13-3/figure2.png" height="300px" /><br>
<b>Fig. 2</b>: Red Convolucional de Grafos Residual Gated
</center>

<!-- As with the standard GCN, the vertex $v$ consists of two vectors: input $\boldsymbol{x}$ and its hidden representation $\boldsymbol{h}$. However, in this case, the edges also have a feature representation, where $\boldsymbol{e_{j}^{x}}$ represents the input edge representation and $\boldsymbol{e_{j}^{h}}$ represents the hidden edge representation. -->
Al igual que con las GCN estándar, el vértice $v$ consiste en dos vectores: la entrada $\boldsymbol{x}$ y su representación oculta $\boldsymbol{h}$. Sin embargo, en este caso, los ejes también tienen una representación de características, donde $\boldsymbol{e_{j}^{x}}$ representa la representación del eje de entrada y $\boldsymbol{e_{j}^{h}}$ representa la representación del eje oculto.

<!-- The hidden representation $\boldsymbol{h}$ of the vertex $v$ is computed by the following equation: -->
La representación $\boldsymbol{h}$ del vértice $v$ se calcula mediante la siguiente ecuación:

$$
    \boldsymbol{h}=\boldsymbol{x} + \bigg(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}\bigg)^{+}
\tag{Ec. 5}
$$

<!-- where $\boldsymbol{x}$ is the input representation, $\boldsymbol{Ax}$ represents a rotation applied to the input $\boldsymbol{x}$ and $\sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}$ denotes the summation of elementwise multiplication of rotated incoming features $\boldsymbol{Bx_{j}}$ and a gate $\eta(\boldsymbol{e_{j}})$. In contrast to the standard GCN above where we average the incoming representations, the gate term is critical to the implementation of Residual Gated GCN since it allows us to modulate the incoming representations based on the edge representations. -->
donde $\boldsymbol{x}$ es la representación de la entrada, $\boldsymbol{Ax}$ representa una rotación aplicada a la entrada $\boldsymbol{x}$ y $\sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}}$ denota el sumatorio de la multiplicación elemento a elemento de las características de entrada rotadas y una compuerta (gate) $\eta(\boldsymbol{e_{j}})$. A diferencia de la antedicha GCN, donde promediamos las representaciones entrantes, el término compuerta es crítico para la implementación de las Residual Gated GCN, ya que nos permite modular las representaciones de entrada en base a las representaciones de los ejes.

<!-- Note that the summation is only over vertices ${v_j}$ that have incoming edges to vertex ${v}$. The term residual (in Residual Gated GCN) comes from the fact that in order to calculate the hidden representation $\boldsymbol{h}$, we add the input representation $\boldsymbol{x}$. The gate term $\eta(\boldsymbol{e_{j}})$ is calculated as shown below: -->
Nótese que la suma es solo sobre los vértices ${v_j}$ que tienen ejes entrantes en el vértice ${v}$. El término residual (en las Residual Gated GCN) viene del hecho de que para calcular la representación oculta $\boldsymbol{h}$, sumamos la representación de entrada $\boldsymbol{x}$. El término compuerta $\eta(\boldsymbol{e_{j}})$ se calcula como se muestra a continuación:

$$
    \eta(\boldsymbol{e_{j}})=\sigma(\boldsymbol{e_{j}})\bigg(\sum_{v_k→v}\sigma(\boldsymbol{e_{k}})\bigg)^{-1}
\tag{Ec. 6}
$$

<!-- The gate value $\eta(\boldsymbol{e_{j}})$ is a normalized sigmoid obtained by dividing the sigmoid of the incoming edge representation by the sum of sigmoids of all incoming edge representations. Note that in order to calculate the gate term, we need the representation of the edge $\boldsymbol{e_{j}}$, which can be computed using the equations below: -->
El valor de la compuerta $\eta(\boldsymbol{e_{j}})$ es una sigmoidal normalizada que se obtiene dividiendo la sigmoidal de la representación del eje entrante por la suma de las sigmoidales de las representaciones de todos los ejes entrantes. Nótese que para calcular el término compuerta, necesitamos la representación del eje $\boldsymbol{e_{j}}$, que podemos calcular utilizando las siguientes ecuaciones:

$$
    \boldsymbol{e_{j}} = \boldsymbol{Ce_{j}^{x}} + \boldsymbol{Dx_{j}} + \boldsymbol{Ex}
\tag{Ec. 7}
$$

$$
    \boldsymbol{e_{j}^{h}}=\boldsymbol{e_{j}^{x}}+(\boldsymbol{e_{j}})^{+}
\tag{Ec. 8}
$$

<!-- The hidden edge representation $\boldsymbol{e_{j}^{h}}$ is obtained by the summation of the initial edge representation $\boldsymbol{e_{j}^{x}}$ and $\texttt{ReLU}(\cdot)$ applied to $\boldsymbol{e_{j}}$ where $\boldsymbol{e_{j}}$ is in turn given by the summation of a rotation applied to $\boldsymbol{e_{j}^{x}}$, a rotation applied to the input representation $\boldsymbol{x_{j}}$ of the vertex $v_{j}$ and a rotation applied to the input representation $\boldsymbol{x}$ of the vertex $v$. -->
La representación del eje oculto $\boldsymbol{e_{j}^{h}}$ se obtiene mediante el sumatorio de la representación del eje inicial $\boldsymbol{e_{j}^{x}}$ y $\texttt{ReLU}(\cdot)$ aplicada a $\boldsymbol{e_{j}}$, donde $\boldsymbol{e_{j}}$ por su parte viene dada por por la suma de una rotación aplicada a $\boldsymbol{e_{j}^{x}}$, una rotación aplicada a la representación de entrada $\boldsymbol{x_{j}}$ del vértice $v_{j}$ y una rotación aplicada a la representación de entrada $\boldsymbol{x}$ del vértice $v$.

<!-- *Note: In order to calculate hidden representations downstream (*e.g.* $2^\text{nd}$ layer hidden representations), we can simply replace the input feature representations by the $1^\text{st}$ layer feature representations in the equations above.* -->
*Nota: Para calcular las representaciones ocultas descendientes (*por ejemplo* la representación oculta de la $2^\text{ª}$ capa), podemos reemplazar las representaciones de entrada por la representación de características de la $1^\text{ª}$ capa en las ecuaciones arriba mencionadas.*

<!-- ### Graph Classification and Residual Gated GCN Layer -->
### Clasificación de grafos y Capa Residual Gated GCN

<!-- In this section, we introduce the problem of graph classification and code up a Residual Gated GCN layer. In addition to the usual import statements, we add the following: -->
En esta sección, introducimos el problema de clasificación de grafos y escribiremos código para una capa Residual Gated GCN.
Además de utilizar las instrucciones de import habituales, añadiremos las siguientes:

```python
os.environ['DGLBACKEND'] = 'pytorch'
import dgl
from dgl import DGLGraph
from dgl.data import MiniGCDataset
import networkx as nx
```

<!-- The first line tells DGL to use PyTorch as the backend. Deep Graph Library ([DGL](https://www.dgl.ai/)) provides various functionalities on graphs whereas networkx allows us to visualise the graphs. -->
La primera linea instruye a DGL para que utilice PyTorch como backend. Deep Graph Library ([DGL](https://www.dgl.ai/)) proporciona diversas funcionalidades sobre grafos, mientras que networkx nos permite visualizar estos grafos.

<!-- In this notebook, the task is to classify a given graph structure into one of 8 graph types. The dataset obtained from `dgl.data.MiniGCDataset` yields some number of graphs (`num_graphs`) with nodes between `min_num_v` and `max_num_v`. Hence, all the graphs obtained do not have the same number of nodes/vertices. -->
En este notebook, la tarea es clasificar una estructura de grafo dada en uno de 8 tipos de grafos. La base de datos obtenida de `dgl.data.MiniGCDataset` nos proporciona un número de grafos (`num_graphs`) con un número de nodos entre `min_num_v` y `max_num_v`. Por ello, no todos los grafos obtenidos tendrán el mismo número de nodos/verices.

<!-- *Note: In order to familiarize yourself with the basics of `DGLGraphs`, it is recommended to go through the short tutorial [here](https://docs.dgl.ai/api/python/graph.html).* -->
*Nota: Para familiarizarte de forma básica con `DGLGraphs`, se recomienda leer [este]((https://docs.dgl.ai/api/python/graph.html)) tutorial corto.*

<!-- Having created the graphs, the next task is to add some signal to the domain. Features can be applied to nodes and edges of a `DGLGraph`. The features are represented as a dictionary of names (strings) and tensors (**fields**). `ndata` and `edata` are syntax sugar to access the feature data of all nodes and edges. -->
Habiendo creado los grafos, la siguiente tarea es añadir alguna señal al dominio. Se pueden aplicar características a nodos y ejes de un `DGLGraph`. Las características vienen representadas por un diccionario de nombres (strings) y tensores (**fields**). `ndata` y `edata` son azúcar sintáctico para acceder a las características de todos los nodos y ejes.

<!-- The following code snippet shows how the features are generated. Each node is assigned a value equal to the number of incident edges, whereas each edge is assigned the value 1. -->
El siguiente fragmento de código muestra como se generan las características. A cada nodo se le asigna un valor igual al número de ejes entrantes, mientras que a cada eje se le asigna el valor 1.
```python
def create_artificial_features(dataset):
    for (graph, _) in dataset:
        graph.ndata['feat'] = graph.in_degrees().view(-1, 1).float()
        graph.edata['feat'] = torch.ones(graph.number_of_edges(), 1)
    return dataset
```

<!-- Training and testing datasets are created and features are assigned as shown below: -->
A continuación se muestra como se crean los conjuntos de entrenamiento y test y se asignan las características:

```python
trainset = MiniGCDataset(350, 10, 20)
testset = MiniGCDataset(100, 10, 20)

trainset = create_artificial_features(trainset)
testset = create_artificial_features(testset)
```

<!-- A sample graph from the trainset has the following representation. Here, we observe that the graph has 15 nodes and 45 edges and both the nodes and edges have a feature representation of shape `(1,)` as expected. Furthermore, the `0` signifies that this graph belongs to class 0. -->
Una muestra de un grafo del set de entrenamiento tiene la siguiente representación. Aquí, observamos que el grafo tiene 15 nodos y 45 ejes; y que tanto los nodos como los ejes tienen una representación de características de forma `(1,)`, como es de esperar.
Además, el  `0` significa que éste grafo pertenece a la case 0.
```python
(DGLGraph(num_nodes=15, num_edges=45,
         ndata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}
         edata_schemes={'feat': Scheme(shape=(1,), dtype=torch.float32)}), 0)
```

<!-- ### Note about DGL Message and Reduce Functions -->
### Nota sobre las funciones Message y Reduce de DGL

<!-- >>In DGL, the *message functions* are expressed as **Edge UDF**s (User Defined Functions). Edge UDFs take in a single argument `edges`. It has three members `src`, `dst`, and `data` for accessing source node features, destination node features, and edge features respectively.
The *reduce functions* are **Node UDF**s. Node UDFs have a single argument `nodes`, which has two members `data` and `mailbox`. `data` contains the node features and `mailbox` contains all incoming message features, stacked along the second dimension (hence the `dim=1` argument).
`update_all(message_func, reduce_func)` sends messages through all edges and updates all nodes. -->
> En DGL, las *funciones mensaje* son expresadas como **Edge UDF** (User Defined Funcions, funciones definidas por el usuario). Las funciones Edge UDF toman un solo argumento `edges`. Tiene tres miembros `src`, `dst`, y `data` para acceder a las características de los nodos fuente, las características de los nodos destino y las características de los ejes, respectivamente.
Las *funciones reduce* son **Node UDF**. Los Node UDF tienen un solo argumento `nodes`, que tiene dos miembros  `data` y `mailbox`. `data` contiene las características del nodo y `mailbox` contiene todas las características de mensajes entrantes, apiladas en la segunda dimensión (por eso el argumento `dim=1`).
`update_all(message_func, reduce_func)` envía mensajes a través de todos los ejes y actualiza todos los nodos.

<!-- ### [Gated Residual GCN Layer Implementation](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=2098s) -->
### [Implementación de la capa Gated Residual GCN](https://www.youtube.com/watch?v=2aKXWqkbpWg&list=PLLHTzKZzVU9eaEyErdV26ikyolxOsz6mq&index=25&t=2098s)

<!-- A Gated Residual GCN layer is implemented as shown in the code snippets below.-->
Una capa Gated Residual GCN se implementa como se muestra en el fragmento de código que sigue.

<!-- Firstly, all the rotations of the input features $\boldsymbol{Ax}$, $\boldsymbol{Bx_{j}}$, $\boldsymbol{Ce_{j}^{x}}$, $\boldsymbol{Dx_{j}}$ and $\boldsymbol{Ex}$ are computed by defining `nn.Linear` layers inside the `__init__` function and then forward propagating the input representations `h` and `e` through the linear layers inside the `forward` function. -->
Primero, todas las rotaciones de características entrantes $\boldsymbol{Ax}$, $\boldsymbol{Bx_{j}}$, $\boldsymbol{Ce_{j}^{x}}$, $\boldsymbol{Dx_{j}}$ y $\boldsymbol{Ex}$ se computan definiendo capas `nn.Linear` dentro de la función `__init__` y propagando hacia delante la representación de entrada `h` y `e` a través de las capas lineales dentro de la función `forward`.


```python
class GatedGCN_layer(nn.Module):

    def __init__(self, input_dim, output_dim):
        super().__init__()
        self.A = nn.Linear(input_dim, output_dim)
        self.B = nn.Linear(input_dim, output_dim)
        self.C = nn.Linear(input_dim, output_dim)
        self.D = nn.Linear(input_dim, output_dim)
        self.E = nn.Linear(input_dim, output_dim)
        self.bn_node_h = nn.BatchNorm1d(output_dim)
        self.bn_node_e = nn.BatchNorm1d(output_dim)
```

<!-- Secondly, we compute the edge representations. This is done inside the `message_func` function, which iterates over all the edges and computes the edge representations. Specifically, the line `e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']` computes *(Eq. 7)* from above. The `message_func` function ships `Bh_j` (which is $\boldsymbol{Bx_{j}}$ from (Eq. 5)) and `e_ij` (Eq. 7) through the edge into the destination node's mailbox. -->
Segundo, computamos las representaciones de los ejes. Esto se hace desde la función `message_func`, que itera sobre todos los ejes y computa sus representaciones. Especificamente, la linea `e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']` computa la *(Eq. 7)* vista anteriormente. La función `message_func` devuelve `Bh_j` (que es $\boldsymbol{Bx_{j}}$ de la (Ec. 5)) y `e_ij` (Ec. 7) a través del eje hasta el mailbox del nodo vecino.

```python
def message_func(self, edges):
    Bh_j = edges.src['Bh']
    # e_ij = Ce_ij + Dhi + Ehj
    e_ij = edges.data['Ce'] + edges.src['Dh'] + edges.dst['Eh']
    edges.data['e'] = e_ij
    return {'Bh_j' : Bh_j, 'e_ij' : e_ij}
```

<!-- Thirdly, the `reduce_func` function collects the shipped messages by the `message_func` function. After collecting the node data `Ah` and shipped data `Bh_j` and `e_ij` from the `mailbox`, the line `h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)` computes the hidden representation of each node as indicated in (Eq. 5). Note however, that this only represents the term $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ without the $\texttt{ReLU}(\cdot)$ and residual connection. -->
Tercero, la función `reduce_func` recopila los mensajes enviados por la funcion `message_func`. Después de recopilar los datos del nodo `Ah` y los datos enviados `Bh_j` e `e_ij` del `mailbox`, la linea `h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)` computa la representación oculta de cada nodo como se indica en (Ec. 5). Nótese que ésto solamente representa el término $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ sin la conexión residual $\texttt{ReLU}(\cdot)$.

```python
def reduce_func(self, nodes):
    Ah_i = nodes.data['Ah']
    Bh_j = nodes.mailbox['Bh_j']
    e = nodes.mailbox['e_ij']
    # sigma_ij = sigmoid(e_ij)
    sigma_ij = torch.sigmoid(e)
    # hi = Ahi + sum_j eta_ij * Bhj
    h = Ah_i + torch.sum(sigma_ij * Bh_j, dim=1) / torch.sum(sigma_ij, dim=1)
    return {'h' : h}
```
<!-- Inside the `forward` function, having called `g.update_all`, we obtain the results of graph convolution `h` and `e`, which represent the terms $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ from (Eq.5) and $\boldsymbol{e_{j}}$ from (Eq. 7) respectively. Then, we normalize `h` and `e` with respect to the graph node size and graph edge size respectively. Batch normalization is then applied so that we can train the network effectively. Finally, we apply $\texttt{ReLU}(\cdot)$ and add the residual connections to obtain the hidden representations for the nodes and edges, which are then returned by the `forward` function. -->
Dentro de la función `forward`, habiendo llamado `g.update_all`, obtenemos los resultados de la convolución de grafos `h` and `e`, que representa los términos $(\boldsymbol{Ax} + \sum_{v_j→v}{\eta(\boldsymbol{e_{j}})\odot \boldsymbol{Bx_{j}}})$ de la (Eq.5) y $\boldsymbol{e_{j}}$ de la (Ec. 7) respectivamente. Después se aplica normalización por batch para poder entrenar la red de manera efectiva. Finalmente, aplicamos $\texttt{ReLU}(\cdot)$ y sumamos las conexiones residuales para obtener las representaciones ocultas de los nodos y ejes, que a la postre son retornadas por la función `forward`.

```python
def forward(self, g, h, e, snorm_n, snorm_e):

    h_in = h # conexión residual
    e_in = e # conexión residual

    g.ndata['h']  = h
    g.ndata['Ah'] = self.A(h)
    g.ndata['Bh'] = self.B(h)
    g.ndata['Dh'] = self.D(h)
    g.ndata['Eh'] = self.E(h)
    g.edata['e']  = e
    g.edata['Ce'] = self.C(e)

    g.update_all(self.message_func, self.reduce_func)

    h = g.ndata['h'] # resultado de la convolución del grafo
    e = g.edata['e'] # resultado de la convolución del grafo

    h = h * snorm_n # normalizar activación con respecto al número de nodos del grafo
    e = e * snorm_e # normalizar activación con respecto al número de ejes del grafo

    h = self.bn_node_h(h) # normalización por batch
    e = self.bn_node_e(e) # normalización por batch

    h = torch.relu(h) # activación no linear
    e = torch.relu(e) # activación no linear

    h = h_in + h # conexión residual
    e = e_in + e # conexión residual

    return h, e
```


<!-- Next, we define the `MLP_Layer` module which contains several fully connected layers (FCN). We create a list of fully connected layers and forward through the network. -->
A continuación, definimos el módulo `MLP_Layer`, que contiene varias capas totalmente conectadas (FCN). Creamos una lista de capas totalmente conectadas y la mandamos hacia delante a través de la red.

<!-- Finally, we define our `GatedGCN` model which comprises of the previously defined classes: `GatedGCN_layer` and `MLP_layer`. The definition of our model (`GatedGCN`) is shown below. -->
Finalmente, definimos nuestro modelo `GatedGCN` que contiene las clases anteriormente mencionadas: `GatedGCN_layer` y `MLP_layer`. La definición de nuestro modelo (`GatedGCN`) se muestra a continuación.


 ```python
 class GatedGCN(nn.Module):
    def __init__(self, input_dim, hidden_dim, output_dim, L):
        super().__init__()
        self.embedding_h = nn.Linear(input_dim, hidden_dim)
        self.embedding_e = nn.Linear(1, hidden_dim)
        self.GatedGCN_layers = nn.ModuleList([
            GatedGCN_layer(hidden_dim, hidden_dim) for _ in range(L)
        ])
        self.MLP_layer = MLP_layer(hidden_dim, output_dim)
    def forward(self, g, h, e, snorm_n, snorm_e):
        # embedding de entrada
        h = self.embedding_h(h)
        e = self.embedding_e(e)
        # capas convolucionales del grafo
        for GGCN_layer in self.GatedGCN_layers:
            h, e = GGCN_layer(g, h, e, snorm_n, snorm_e)
        # clasificador MLP
        g.ndata['h'] = h
        y = dgl.mean_nodes(g,'h')
        y = self.MLP_layer(y)
        return y
 ```

<!-- In our constructor, we define the embeddings for `e` and `h` (`self.embedding_e ` and `self.embedding_h`), `self.GatedGCN_layers` which is list (of size $L$) of our previously defined model: `GatedGCN_layer`, and finally `self.MLP_layer` which was also defined before. Next, using these initializations, we simply foward through the model and output `y`. -->
En nuestro constructor, definimos los embeddings para `e` y `h` (`self.embedding_e ` y `self.embedding_h`), `self.GatedGCN_layers`, que es una lista (de tamaño $L$) del modelo definido anteriormente: `GatedGCN_layer`, y finalmente `self.MLP_layer`, que también se ha definido anteriormente.
A continuación, utilizamos estas inicializaciones y simplemente enviamos hacia delante a través del modelo generamos la salida `y`.

<!-- To better understand the model, we initiate an object of the model and print it for better visualization: -->
Para entender mejor el modelo, inicializamos un objecto del modelo e imprimimos su contenido para una mejor visualización:
```python
# instanciamos la red
model = GatedGCN(input_dim=1, hidden_dim=100, output_dim=8, L=2)
print(model)
```

<!-- The main structure of the model is shown below: -->
La estructura principal del modelo se muestra a continuación:

```python
GatedGCN(
  (embedding_h): Linear(in_features=1, out_features=100, bias=True)
  (embedding_e): Linear(in_features=1, out_features=100, bias=True)
  (GatedGCN_layers): ModuleList(
    (0): GatedGCN_layer(...)
    (1): GatedGCN_layer(... ))
  (MLP_layer): MLP_layer(
    (FC_layers): ModuleList(...))
```

<!--Not surprisingly, we have two layers of `GatedGCN_layer` (since `L=2`) followed by a `MLP_layer` which finally yields an output of 8 values. -->
Como es de esperar, tenemos dos capas de `GatedGCN_layer` (ya que `L=2`), seguidas por una capa `MLP_layer` que finalmente nos proporciona una salida de 8 valores.

<!-- Moving on, we define our `train` and `evaluate` functions. In our `train` function, we have our generic code which takes samples from `dataloader`.  Next, `batch_graphs`, `batch_x`, `batch_e`, `batch_snorm_n` and `batch_snorm_e` are fed into our model which returns `batch_scores` (of size 8). The predicted scores are compared with the ground truth in our loss function: `loss(batch_scores, batch_labels)`. Then, we zero out gradients (`optimizer.zero_grad()`), perform backprop (`J.backward()`) and updates our weights (`optimizer.step()`). Finally, loss for epoch and training accuracy is calculated. Furthermore, we use a similar code for our `evaluate` function. -->
Seguidamente, definimos nuestras funciones  `train` y `evaluate`. En nuestra función `train`, tenemos nuestro código genérico que toma muestras de `dataloader`. A continuación, `batch_graphs`, `batch_x`, `batch_e`, `batch_snorm_n` y `batch_snorm_e` son alimentadas a nuestro modelo, que retorna `batch_scores` (de tamaño 8). Las puntuaciones predecidas son comparadas con los valores "ground truth" en nuestra función de pérdida: `loss(batch_scores, batch_labels)`. Después, asignamos el valor cero a los gradientes (`optimizer.zero_grad()`), hacemos retropropagación (`J.backward()`) y actualizamos nuestros pesos (`optimizer.step()`). Finalmente, calculamos la pérdida por época y la precisión de entrenamiento. Utilizamos un código similar para nuestra función `evaluate`.

<!-- Finally, we are ready to train! We found that after 40 epochs of training, our model has learned to classify the graphs with a test accuracy of $87$%. -->
¡Finalmente estamos preparados para entrenar! Observamos que después de 40 épocas de entrenamiento, nuestro modelo ha aprendido a clasificar los grafos con una precisión de test del $87$%.

