/*
 * Copyright © 2021-present Arcade Data Ltd (info@arcadedata.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * SPDX-FileCopyrightText: 2021-present Arcade Data Ltd (info@arcadedata.com)
 * SPDX-License-Identifier: Apache-2.0
 */
package com.arcadedb.schema;

import com.arcadedb.TestHelper;
import com.arcadedb.database.MutableDocument;
import com.arcadedb.engine.Bucket;
import com.arcadedb.exception.SchemaException;

import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;

import java.util.*;

import static org.assertj.core.api.Assertions.*;

public class DropTypeTest extends TestHelper {
  private static final int    TOT        = 10;
  private static final String TYPE_NAME  = "V";
  private static final String TYPE_NAME2 = "V2";
  private static final String TYPE_NAME3 = "V3";

  @Test
  public void testDropAndRecreateType() {
    database.transaction(() -> {
      assertThat(database.getSchema().existsType(TYPE_NAME)).isFalse();

      final DocumentType type = database.getSchema().buildDocumentType().withName(TYPE_NAME).withTotalBuckets(3).create();

      final DocumentType type2 = database.getSchema().buildDocumentType().withName(TYPE_NAME2).withTotalBuckets(3).create();
      type2.addSuperType(type);

      final DocumentType type3 = database.getSchema().buildDocumentType().withName(TYPE_NAME3).withTotalBuckets(3).create();
      type3.addSuperType(type2);

      type.createProperty("id", Integer.class);
      type.createProperty("name", String.class);

      for (int i = 0; i < TOT; ++i) {
        final MutableDocument v = database.newDocument(TYPE_NAME2);
        v.set("id", i);
        v.set("name", "Jay");
        v.set("surname", "Miner");
        v.save();
      }

      final MutableDocument v = database.newDocument(TYPE_NAME);
      v.set("id", TOT);
      v.save();

      database.commit();

      final List<Bucket> buckets = type2.getBuckets(false);

      database.getSchema().dropType(TYPE_NAME2);

      // CHECK ALL THE BUCKETS ARE REMOVED
      for (final Bucket b : buckets) {
        try {
          database.getSchema().getBucketById(b.getFileId());
          fail();
        } catch (final SchemaException e) {
        }

        try {
          database.getSchema().getBucketByName(b.getName());
          fail();
        } catch (final SchemaException e) {
        }

        try {
          database.getSchema().getFileById(b.getFileId());
          fail();
        } catch (final SchemaException e) {
        }
      }

      // CHECK TYPE HAS BEEN REMOVED FROM INHERITANCE
      for (final DocumentType parent : type2.getSuperTypes())
        assertThat(parent.getSubTypes().contains(type2)).isFalse();

      for (final DocumentType sub : type2.getSubTypes()) {
        assertThat(sub.getSuperTypes().contains(type2)).isFalse();
        assertThat(sub.getSuperTypes().contains(type)).isTrue();
      }

      // CHECK INHERITANCE CHAIN IS CONSISTENT
      assertThat(type.getSuperTypes().isEmpty()).isTrue();

      for (final DocumentType sub : type.getSubTypes())
        assertThat(sub.getSuperTypes().contains(type)).isTrue();

      assertThat(database.countType(TYPE_NAME, true)).isEqualTo(1);

      final DocumentType newType = database.getSchema().getOrCreateDocumentType(TYPE_NAME2);
      assertThat(database.countType(TYPE_NAME, true)).isEqualTo(1);
      assertThat(database.countType(TYPE_NAME2, true)).isEqualTo(0);
      assertThat(database.countType(TYPE_NAME2, false)).isEqualTo(0);

      newType.addSuperType(TYPE_NAME);

      // CHECK INHERITANCE CHAIN IS CONSISTENT AGAIN
      for (final DocumentType parent : newType.getSuperTypes())
        assertThat(parent.getSubTypes().contains(newType)).isTrue();

      for (final DocumentType sub : newType.getSubTypes())
        assertThat(sub.getSuperTypes().contains(newType)).isTrue();

      assertThat(database.countType(TYPE_NAME, true)).isEqualTo(1);
      assertThat(database.countType(TYPE_NAME2, true)).isEqualTo(0);
      assertThat(database.countType(TYPE_NAME2, false)).isEqualTo(0);

      database.begin();

      for (int i = 0; i < TOT; ++i) {
        final MutableDocument v2 = database.newDocument(TYPE_NAME2);
        v2.set("id", TOT * 2 + i);
        v2.set("name", "Jay");
        v2.set("surname", "Miner");
        v2.save();
      }

      final MutableDocument v3 = database.newDocument(TYPE_NAME);
      v3.set("id", TOT);
      v3.save();

      assertThat(database.countType(TYPE_NAME, true)).isEqualTo(TOT + 2);
      assertThat(database.countType(TYPE_NAME2, false)).isEqualTo(TOT);
      assertThat(database.countType(TYPE_NAME, false)).isEqualTo(2);
    });
  }
}
